//Please read license.txt for licensing and copyright information

#include "headers.h"
#include "db.h"
#include "net.h"
#include <boost/filesystem.hpp>
#include <boost/filesystem/fstream.hpp>

using namespace std;
using namespace boost;


unsigned int nWalletDBUpdated;
uint64 nAccountingEntryNumber = 0;


// CDB
static CCriticalSection cs_db;
static bool fDbEnvInit = false;
DbEnv dbenv(0);
static map<string, int> mapFileUseCount;
static map<string, Db*> mapDb;

class CDBInit
{
public:
    CDBInit()
    {
    }
    ~CDBInit()
    {
        if (fDbEnvInit)
        {
            dbenv.close(0);
            fDbEnvInit = false;
        }
    }
}
instance_of_cdbinit;


CDB::CDB(const char* pszFile, const char* pszMode) : pdb(NULL)
{
    int ret;
    if (pszFile == NULL)
        return;

    fReadOnly = (!strchr(pszMode, '+') && !strchr(pszMode, 'w'));
    bool fCreate = strchr(pszMode, 'c');
    unsigned int nFlags = DB_THREAD;
    if (fCreate)
        nFlags |= DB_CREATE;

    MUTEX_LOCK(cs_db);

    if (!fDbEnvInit)
    {
        if (fShutdown)
            return;
        string strDataDir = GetDataDir();
        string strLogDir = strDataDir + "/database";
        filesystem::create_directory(strLogDir.c_str());
        string strErrorFile = strDataDir + "/db.log";
        debugprintf(INFO, "dbenv.open strLogDir=%s strErrorFile=%s\n", strLogDir.c_str(), strErrorFile.c_str());

        dbenv.set_lg_dir(strLogDir.c_str());
        dbenv.set_lg_max(10000000);
        dbenv.set_lk_max_locks(10000);
        dbenv.set_lk_max_objects(10000);
        dbenv.set_errfile(fopenSOLID(strErrorFile.c_str(), "a")); /// debug
        dbenv.set_flags(DB_AUTO_COMMIT, 1);
        //dbenv.set_cachesize(0,1*1024*1024,0);
        ret = dbenv.open(strDataDir.c_str(),
                         DB_CREATE     |
                         DB_INIT_LOCK  |
                         DB_INIT_LOG   |
                         DB_INIT_MPOOL |
                         DB_INIT_TXN   |
                         DB_THREAD     |
                         DB_RECOVER,
                         S_IRUSR | S_IWUSR);
        if (ret > 0)
            throw runtime_error(strprintf("CDB() : error %d opening database environment", ret));
        fDbEnvInit = true;
    }

    strFile = pszFile;
    ++mapFileUseCount[strFile];
    pdb = mapDb[strFile];
    if (pdb == NULL)
    {
        pdb = new Db(&dbenv, 0);

        ret = pdb->open(NULL,      // Txn pointer
                        pszFile,   // Filename
                        "main",    // Logical db name
                        DB_BTREE,  // Database type
                        nFlags,    // Flags
                        0);

        if (ret > 0)
        {
            delete pdb;
            pdb = NULL;
            {
                MUTEX_LOCK(cs_db);
                --mapFileUseCount[strFile];
            }
            strFile = "";
            throw runtime_error(strprintf("CDB() : can't open database file %s, error %d", pszFile, ret));
        }

        if (fCreate && !Exists(string("version")))
        {
            bool fTmp = fReadOnly;
            fReadOnly = false;
            WriteVersion(VERSION);
            fReadOnly = fTmp;
        }

        mapDb[strFile] = pdb;
    }
}

void CDB::Close()
{
    if (!pdb)           return;
    if (!vTxn.empty())  vTxn.front()->abort();
    vTxn.clear();
    pdb = NULL;

    // Flush database activity from memory pool to disk log
    unsigned int nMinutes = 0;
    if (fReadOnly)              nMinutes = 1;
    if (strFile == "addr.dat")  nMinutes = 2;
    if (strFile == "blkindex.dat" && Block_IsInitialDownload() && g_qBlockBestHeight % 500 != 0)    nMinutes = 1;
    dbenv.txn_checkpoint(0, nMinutes, 0);

    {
        MUTEX_LOCK(cs_db);
        --mapFileUseCount[strFile];
    }
}

void static CloseDb(const string& strFile)
{
    MUTEX_LOCK(cs_db);
    if (mapDb[strFile] != NULL)
    {
        // Close the database handle
        Db* pdb = mapDb[strFile];
        pdb->close(0);
        delete pdb;
        mapDb[strFile] = NULL;
    }
}

void DBFlush(bool fShutdown)
{
    // Flush log data to the actual data file
    //  on all files that are not in use
    debugprintf(INFO, "DBFlush(%s)%s\n", fShutdown ? "true" : "false", fDbEnvInit ? "" : " db not started");
    if (!fDbEnvInit)
        return;

    MUTEX_LOCK(cs_db);

    map<string, int>::iterator mi = mapFileUseCount.begin();
    while (mi != mapFileUseCount.end())
    {
        string strFile = (*mi).first;
        int nRefCount = (*mi).second;
        debugprintf(INFO, "%s refcount=%d\n", strFile.c_str(), nRefCount);
        if (nRefCount == 0)
        {
            // Move log data to the dat file
            CloseDb(strFile);
            dbenv.txn_checkpoint(0, 0, 0);
            debugprintf(INFO, "%s flush\n", strFile.c_str());
            dbenv.lsn_reset(strFile.c_str(), 0);
            mapFileUseCount.erase(mi++);
        }
        else
            mi++;
    }
    if (fShutdown)
    {
        char** listp;
        if (mapFileUseCount.empty())
            dbenv.log_archive(&listp, DB_ARCH_REMOVE);
        dbenv.close(0);
        fDbEnvInit = false;
    }
}



void CTxDB::ShowTX()
{
    Dbc* pcursor = GetCursor();
    if (!pcursor)   return;


    map<string, uint64> totaladdr;
    map<uint64,string> totaladdrrev;

    // Load mapBlockIndex
    unsigned int fFlags = DB_SET_RANGE;
    loop
    {
        // Read next record
        CDataStream ssKey;
        if (fFlags == DB_SET_RANGE)
            ssKey << make_pair(string("tx"), uint256(0));
        CDataStream ssValue;
        int ret = ReadAtCursor(pcursor, ssKey, ssValue, fFlags);
        fFlags = DB_NEXT;
        if (ret == DB_NOTFOUND)
            break;
        else if (ret != 0)
            return;

        // Unserialize
        string strType;
        ssKey >> strType;
        CTxDB txdb;

        if (strType == "tx")
        {
            //CDiskTxPos txpos;
            //ssValue >> txpos;
            CTxIndex txpos;
            ssValue >> txpos;
            if(txpos.GetDepthInMainChain()<120) continue;
            //txpos.pos.print();
            CTransaction tx;
            tx.ReadFromDisk(txpos.pos);



            //for(int x=0;x<txpos.vSpent.size();x++)
            {
                //if(txpos.vSpent[x].IsNull())
                {
                    BOOST_FOREACH(const CTxOut& txout, tx.vout)
                    {
                        totaladdr[txout.ToString()] += txout.nValue;
                    }

                    BOOST_FOREACH(const CTxIn& txin, tx.vin)
                    {
                        COutPoint prevout = txin.prevout;
                        CTxIndex txindex;
                        if(!txdb.ReadTxIndex(prevout.hash, txindex)) continue;
                        CTransaction txPrev;
                        if (!txPrev.ReadFromDisk(txindex.pos)) continue;

                        totaladdr[txPrev.vout[prevout.n].ToString()] -= txPrev.vout[prevout.n].nValue;
                    }
                }
            }

            //tx.print();

        }
        else
        {
                //break;
        }
    }


    pcursor->close();
}




//
// CTxDB
//

bool CTxDB::ReadTxIndex(uint256 hash, CTxIndex& txindex)
{
    assert(!fClient);
    txindex.SetNull();
    return Read(make_pair(string("tx"), hash), txindex);
}

bool CTxDB::UpdateTxIndex(uint256 hash, const CTxIndex& txindex)
{
    assert(!fClient);
    return Write(make_pair(string("tx"), hash), txindex);
}

bool CTxDB::AddTxIndex(const CTransaction& tx, const CDiskTxPos& pos, int nHeight)
{
    assert(!fClient);

    // Add to tx index
    uint256 hash = tx.GetHash();
    CTxIndex txindex(pos, tx.vout.size());
    return Write(make_pair(string("tx"), hash), txindex);
}

bool CTxDB::EraseTxIndex(const CTransaction& tx)
{
    assert(!fClient);
    uint256 hash = tx.GetHash();

    return Erase(make_pair(string("tx"), hash));
}

bool CTxDB::ContainsTx(uint256 hash)
{
    assert(!fClient);
    return Exists(make_pair(string("tx"), hash));
}

bool CTxDB::ReadOwnerTxes(uint160 hash160, int nMinHeight, vector<CTransaction>& vtx)
{
    assert(!fClient);
    vtx.clear();

    // Get cursor
    Dbc* pcursor = GetCursor();
    if (!pcursor)
        return false;

    unsigned int fFlags = DB_SET_RANGE;
    loop
    {
        // Read next record
        CDataStream ssKey;
        if (fFlags == DB_SET_RANGE)
            ssKey << string("owner") << hash160 << CDiskTxPos(0, 0, 0);
        CDataStream ssValue;
        int ret = ReadAtCursor(pcursor, ssKey, ssValue, fFlags);
        fFlags = DB_NEXT;
        if (ret == DB_NOTFOUND)
            break;
        else if (ret != 0)
        {
            pcursor->close();
            return false;
        }

        // Unserialize
        string strType;
        uint160 hashItem;
        CDiskTxPos pos;
        ssKey >> strType >> hashItem >> pos;
        int nItemHeight;
        ssValue >> nItemHeight;

        // Read transaction
        if (strType != "owner" || hashItem != hash160)
            break;
        if (nItemHeight >= nMinHeight)
        {
            vtx.resize(vtx.size()+1);
            if (!vtx.back().ReadFromDisk(pos))
            {
                pcursor->close();
                return false;
            }
        }
    }

    pcursor->close();
    return true;
}

bool CTxDB::ReadDiskTx(uint256 hash, CTransaction& tx, CTxIndex& txindex)
{
    assert(!fClient);
    tx.SetNull();
    if (!ReadTxIndex(hash, txindex))
        return false;
    return (tx.ReadFromDisk(txindex.pos));
}

bool CTxDB::ReadDiskTx(uint256 hash, CTransaction& tx)
{
    CTxIndex txindex;
    return ReadDiskTx(hash, tx, txindex);
}

bool CTxDB::ReadDiskTx(COutPoint outpoint, CTransaction& tx, CTxIndex& txindex)
{
    return ReadDiskTx(outpoint.hash, tx, txindex);
}

bool CTxDB::ReadDiskTx(COutPoint outpoint, CTransaction& tx)
{
    CTxIndex txindex;
    return ReadDiskTx(outpoint.hash, tx, txindex);
}

bool CTxDB::WriteBlockIndex(const CDiskBlockIndex& blockindex)
{
    return Write(make_pair(string("blockindex"), blockindex.GetBlockHash()), blockindex);
}

bool CTxDB::EraseBlockIndex(uint256 hash)
{
    return Erase(make_pair(string("blockindex"), hash));
}

bool CTxDB::ReadHashBestChain(uint256& hashBestChain)
{
    return Read(string("hashBestChain"), hashBestChain);
}

bool CTxDB::WriteHashBestChain(uint256 hashBestChain)
{
    return Write(string("hashBestChain"), hashBestChain);
}

bool CTxDB::ReadBestInvalidWork(CBigNum& bnBestInvalidWork)
{
    return Read(string("bnBestInvalidWork"), bnBestInvalidWork);
}

bool CTxDB::WriteBestInvalidWork(CBigNum bnBestInvalidWork)
{
    return Write(string("bnBestInvalidWork"), bnBestInvalidWork);
}


bool BlockIndexHeightSort (CBlockIndex *i,CBlockIndex *j) { return (i->blk.nBlockNum<j->blk.nBlockNum); }

bool CTxDB::LoadBlockIndex()
{
    Dbc* pcursor = GetCursor();     // Get database cursor
    if (!pcursor)   return false;

    unsigned int fFlags = DB_SET_RANGE;

    std::list<CDiskBlockIndex> disklist;
    CDiskBlockIndex diskindex;
    loop
    {
        // Read next record
        CDataStream ssKey;
        if (fFlags == DB_SET_RANGE) ssKey << make_pair(string("blockindex"), uint256(0));
        CDataStream ssValue;
        int ret = ReadAtCursor(pcursor, ssKey, ssValue, fFlags);
        fFlags = DB_NEXT;
        if (ret == DB_NOTFOUND) break;
        else if (ret != 0)      return false;

        // Unserialize
        string strType;
        ssKey >> strType;
        if (strType != "blockindex")    break;


        ssValue >> diskindex;
        disklist.push_back(diskindex);
    }
    pcursor->close();

    BOOST_FOREACH(const CDiskBlockIndex item, disklist)
    {
        uint256 hash = item.GetBlockHash();
        CBlockIndex* pindexNew = Block_InsertBlockIndex(hash);
        pindexNew->pprev          = Block_InsertBlockIndex(item.m_hashPrev);
        pindexNew->pnext          = Block_InsertBlockIndex(item.m_hashNext);
        pindexNew->nFile          = item.nFile;
        pindexNew->nBlockPos      = item.nBlockPos;
        pindexNew->blk = item.blk;
        if (!pindexNew->CheckIndex())   return error("LoadBlockIndex() : CheckIndex failed at %"PRI64d"", pindexNew->blk.nBlockNum);
    }

    //find genesis index
    map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(g_BlockGenesisHash);
    if (mi != g_BlockIndexMap.end())    g_pBlockGenesisIndex=(*mi).second;


    // Calculate bnChainWork
    vector<CBlockIndex*> vSortedByHeight;
    vSortedByHeight.reserve(g_BlockIndexMap.size());
    BOOST_FOREACH(const PAIRTYPE(uint256, CBlockIndex*)& item, g_BlockIndexMap)
    {
        vSortedByHeight.push_back(item.second);
    }
    sort(vSortedByHeight.begin(), vSortedByHeight.end(),BlockIndexHeightSort);
    BOOST_FOREACH(CBlockIndex* pindex, vSortedByHeight)
    {
        if(pindex->blk.nBlockNum && (pindex->blk.nBlockNum%2)==0 && pindex->pprev)
        {
            //assume each "old" trusted block had a million in the trust tx to avoid needing to read all blocks on disk for actual amount
            pindex->bnChainWork = pindex->pprev->bnChainWork + GetBlockWork(pindex->pprev->blk.dwBits);
        }
        else
        {
            pindex->bnChainWork = (pindex->pprev ? pindex->pprev->bnChainWork : 0) + GetBlockWork(pindex->blk.dwBits);
        }
    }

    // Load g_BlockBestChainHash pointer to end of best chain
    if (!ReadHashBestChain(g_BlockBestChainHash))
    {
        if (g_pBlockGenesisIndex == NULL)
            return true;
        return error("CTxDB::LoadBlockIndex() : g_BlockBestChainHash not loaded");
    }
    if (!g_BlockIndexMap.count(g_BlockBestChainHash))
        return error("CTxDB::LoadBlockIndex() : g_BlockBestChainHash not found in the block index");
    g_pBlockBestIndex = g_BlockIndexMap[g_BlockBestChainHash];
    g_qBlockBestHeight = g_pBlockBestIndex->blk.nBlockNum;
    g_bnBlockBestChainWork = g_pBlockBestIndex->bnChainWork;
    debugprintf(INFO, "LoadBlockIndex(): g_BlockBestChainHash=%s  height=%"PRI64d"\n", g_BlockBestChainHash.ToString().substr(0,20).c_str(), g_qBlockBestHeight);

    // Load bnBestInvalidWork, OK if it doesn't exist
    ReadBestInvalidWork(g_bnBlockBestInvalidWork);

    // Verify last 6 blocks in the best chain
    CBlockIndex* pindexFork = NULL;
    for (CBlockIndex* pindex = g_pBlockBestIndex; pindex && pindex->pprev; pindex = pindex->pprev)
    {
        if (pindex->blk.nBlockNum < g_qBlockBestHeight-6)  break;
        CBlock block;
        if (!block.ReadFromDisk(pindex))    return error("LoadBlockIndex() : block.ReadFromDisk failed");
        if (!block.VerifyBlock())
        {
            debugprintf(ERR, "LoadBlockIndex() : *** found bad block at %"PRI64d", hash=%s\n", pindex->blk.nBlockNum, pindex->GetBlockHash().ToString().c_str());
            pindexFork = pindex->pprev;
        }
        //for these last few blocks which could be involved in a reorg we actually want to test the trusted amount
        block.CalculateChainWork(pindex);
    }
    if (pindexFork)
    {
        // Reorg back to the fork
        debugprintf(WARN, "LoadBlockIndex() : *** moving best chain pointer back to block %"PRI64d"\n", pindexFork->blk.nBlockNum);
        CBlock block;
        if (!block.ReadFromDisk(pindexFork))
            return error("LoadBlockIndex() : block.ReadFromDisk failed");
        CTxDB txdb;
        block.SetBestChain(txdb, pindexFork);
    }

    return true;
}





//
// CAddrDB
//

bool CAddrDB::WriteAddress(const CAddress& addr)
{
    return Write(make_pair(string("addr"), addr.GetKey()), addr);
}

bool CAddrDB::EraseAddress(const CAddress& addr)
{
    return Erase(make_pair(string("addr"), addr.GetKey()));
}

bool CAddrDB::LoadAddresses()
{
    MUTEX_LOCK(cs_mapAddresses);

    // Load user provided addresses
    CAutoFile filein = fopenSOLID((GetDataDir() + "/addr.txt").c_str(), "rt");
    if (filein)
    {
        try
        {
            char psz[1000];
            while (fgets(psz, sizeof(psz), filein))
            {
                CAddress addr(psz, false, NODE_NETWORK);
                addr.nTime = 0; // so it won't relay unless successfully connected
                if (addr.IsValid())
                    AddAddress(addr);
            }
        }
        catch (...) { }
    }

    // Get cursor
    Dbc* pcursor = GetCursor();
    if (!pcursor)   return false;

    loop
    {
        // Read next record
        CDataStream ssKey;
        CDataStream ssValue;
        int ret = ReadAtCursor(pcursor, ssKey, ssValue);
        if (ret == DB_NOTFOUND)
            break;
        else if (ret != 0)
            return false;

        // Unserialize
        string strType;
        ssKey >> strType;
        if (strType == "addr")
        {
            CAddress addr;
            ssValue >> addr;
            mapAddresses.insert(make_pair(addr.GetKey(), addr));
        }
    }
    pcursor->close();
    debugprintf(INFO, "Loaded %d addresses\n", mapAddresses.size());

    return true;
}

bool LoadAddresses()
{
    return CAddrDB("cr+").LoadAddresses();
}




//
// CWalletDB
//

bool CWalletDB::WriteName(const string& strAddress, const string& strName)
{
    nWalletDBUpdated++;
    return Write(make_pair(string("name"), strAddress), strName);
}

bool CWalletDB::EraseName(const string& strAddress)
{
    // This should only be used for sending addresses, never for receiving addresses,
    // receiving addresses must always have an address book entry if they're not change return.
    nWalletDBUpdated++;
    return Erase(make_pair(string("name"), strAddress));
}

bool CWalletDB::ReadAccount(const string& strAccount, CAccount& account)
{
    account.SetNull();
    return Read(make_pair(string("acc"), strAccount), account);
}

bool CWalletDB::WriteAccount(const string& strAccount, const CAccount& account)
{
    return Write(make_pair(string("acc"), strAccount), account);
}

bool CWalletDB::WriteAccountingEntry(const CAccountingEntry& acentry)
{
    return Write(boost::make_tuple(string("acentry"), acentry.strAccount, ++nAccountingEntryNumber), acentry);
}

int64 CWalletDB::GetAccountCreditDebit(const string& strAccount)
{
    list<CAccountingEntry> entries;
    ListAccountCreditDebit(strAccount, entries);

    int64 nCreditDebit = 0;
    BOOST_FOREACH (const CAccountingEntry& entry, entries)
        nCreditDebit += entry.nCreditDebit;

    return nCreditDebit;
}

void CWalletDB::ListAccountCreditDebit(const string& strAccount, list<CAccountingEntry>& entries)
{
    bool fAllAccounts = (strAccount == "*");

    Dbc* pcursor = GetCursor();
    if (!pcursor)
        throw runtime_error("CWalletDB::ListAccountCreditDebit() : cannot create DB cursor");
    unsigned int fFlags = DB_SET_RANGE;
    loop
    {
        // Read next record
        CDataStream ssKey;
        if (fFlags == DB_SET_RANGE)
            ssKey << boost::make_tuple(string("acentry"), (fAllAccounts? string("") : strAccount), uint64(0));
        CDataStream ssValue;
        int ret = ReadAtCursor(pcursor, ssKey, ssValue, fFlags);
        fFlags = DB_NEXT;
        if (ret == DB_NOTFOUND)
            break;
        else if (ret != 0)
        {
            pcursor->close();
            throw runtime_error("CWalletDB::ListAccountCreditDebit() : error scanning DB");
        }

        // Unserialize
        string strType;
        ssKey >> strType;
        if (strType != "acentry")
            break;
        CAccountingEntry acentry;
        ssKey >> acentry.strAccount;
        if (!fAllAccounts && acentry.strAccount != strAccount)
            break;

        ssValue >> acentry;
        entries.push_back(acentry);
    }

    pcursor->close();
}


int CWalletDB::LoadWallet(CWallet* pwallet, bool bCleanOldWallet)
{
    pwallet->vchDefaultKey.clear();
    int nFileVersion = 0;
    vector<uint256> vWalletClean;
    int nKey=0;

    //// todo: shouldn't we catch exceptions and try to recover and continue?
    {
        // Get cursor
        Dbc* pcursor = GetCursor();
        if (!pcursor)   return DB_CORRUPT;

        loop
        {
            // Read next record
            CDataStream ssKey;
            CDataStream ssValue;
            int ret = ReadAtCursor(pcursor, ssKey, ssValue);
            if (ret == DB_NOTFOUND) break;
            else if (ret != 0)  return DB_CORRUPT;

            // Unserialize
            // Taking advantage of the fact that pair serialization is just the two items serialized one after the other
            string strType;
            ssKey >> strType;
            if (strType == "name")
            {
                string strAddress;
                ssKey >> strAddress;
                ssValue >> pwallet->mapAddressBook[strAddress];
            }

            else if (strType == "tx")
            {
                uint256 hash;
                ssKey >> hash;
                if(bCleanOldWallet)
                {
                    vWalletClean.push_back(hash);
                }
                else
                {
                    CWalletTx& wtx = pwallet->mapWallet[hash];
                    ssValue >> wtx;
                    wtx.pwallet = pwallet;
                    if (wtx.GetHash() != hash)  debugprintf(ERR, "Error in %s, hash mismatch\n",pwallet->GetWalletFileName().c_str());
                }
            }
            else if (strType == "acentry")
            {
                string strAccount;
                ssKey >> strAccount;
                uint64 nNumber;
                ssKey >> nNumber;
                if (nNumber > nAccountingEntryNumber)   nAccountingEntryNumber = nNumber;
            }
            else if (strType == "key" || strType == "wkey")
            {
                vector<unsigned char> vchPubKey;
                ssKey >> vchPubKey;
                CKey key;
                bool bLoadKey=false;
                if (strType == "key")
                {
                    nKey++;
                    CPrivKey pkey;
                    if(nKey==10159)
                    {
                        int n=0;
                    }
                    ssValue >> pkey;
                    if(key.SetPrivKey(pkey))
                    {
                        bLoadKey=true;
                    }
                }
                else
                {
                    CWalletKey wkey;
                    ssValue >> wkey;
                    key.SetPrivKey(wkey.vchPrivKey);
                }
                if (bLoadKey && !pwallet->LoadKey(key)) return DB_CORRUPT;
            }
            else if (strType == "mkey")
            {
                unsigned int nID;
                ssKey >> nID;
                CMasterKey kMasterKey;
                ssValue >> kMasterKey;
                if(pwallet->mapMasterKeys.count(nID) != 0)  return DB_CORRUPT;
                pwallet->mapMasterKeys[nID] = kMasterKey;
                if (pwallet->nMasterKeyMaxID < nID) pwallet->nMasterKeyMaxID = nID;
            }
            else if (strType == "ckey")
            {
                vector<unsigned char> vchPubKey;
                ssKey >> vchPubKey;
                vector<unsigned char> vchPrivKey;
                ssValue >> vchPrivKey;
                if (!pwallet->LoadCryptedKey(vchPubKey, vchPrivKey))    return DB_CORRUPT;
            }
            else if (strType == "defaultkey")
            {
                ssValue >> pwallet->vchDefaultKey;
            }
            else if (strType == "pool")
            {
                int64 nIndex;
                ssKey >> nIndex;
                pwallet->setKeyPool.insert(nIndex);
            }
            else if (strType == "version")
            {
                ssValue >> nFileVersion;
                if (nFileVersion == 10300)  nFileVersion = 300;
            }
            else if (strType == "minversion")
            {
                int nMinVersion = 0;
                ssValue >> nMinVersion;
                if (nMinVersion > VERSION)  return DB_TOO_NEW;
            }
        }
        pcursor->close();
    }

    BOOST_FOREACH(uint256 hash, vWalletClean)
    {
        EraseTx(hash);
    }

    // Upgrade
    if (nFileVersion < VERSION)
    {
        // Get rid of old debug.log file in current directory
        if (nFileVersion <= 105 && !pszSetDataDir[0])
            unlink("debug.log");

        WriteVersion(VERSION);
    }


    return DB_LOAD_OK;
}

void ThreadFlushWalletDB(void* parg)
{
    const string& strFile = ((const string*)parg)[0];
    static bool fOneThread;
    if (fOneThread)
        return;
    fOneThread = true;
    if (CMDSetting_Exist("noflushwallet"))
        return;

    int nSecondsToDelay = (int)Setting_GetINT64("wallet_flushmindelay");
    unsigned int nLastSeen = nWalletDBUpdated;
    unsigned int nLastFlushed = nWalletDBUpdated;
    int64 nLastWalletUpdate = SolidTime_Get();

    if(nSecondsToDelay<5) nSecondsToDelay=5;

    while (!fShutdown)
    {
        Sleep(500);

        if (nLastSeen != nWalletDBUpdated)
        {
            nLastSeen = nWalletDBUpdated;
            nLastWalletUpdate = SolidTime_Get();
        }

        if (nLastFlushed != nWalletDBUpdated && SolidTime_Get() - nLastWalletUpdate >= nSecondsToDelay)
        {
            TRY_CRITICAL_BLOCK(cs_db)
            {
                // Don't do this if any databases are in use
                int nRefCount = 0;
                map<string, int>::iterator mi = mapFileUseCount.begin();
                while (mi != mapFileUseCount.end())
                {
                    nRefCount += (*mi).second;
                    mi++;
                }

                if (nRefCount == 0 && !fShutdown)
                {
                    map<string, int>::iterator mi = mapFileUseCount.find(strFile);
                    if (mi != mapFileUseCount.end())
                    {
                        debugprintf(INFO, "%s ", DateTimeStrFormat("%x %H:%M:%S", SolidTime_Get()).c_str());
                        debugprintf(INFO, "Flushing wallet.dat\n");
                        nLastFlushed = nWalletDBUpdated;
                        int64 nStart = GetTimeMillis();

                        // Flush wallet.dat so it's self contained
                        CloseDb(strFile);
                        dbenv.txn_checkpoint(0, 0, 0);
                        dbenv.lsn_reset(strFile.c_str(), 0);

                        mapFileUseCount.erase(mi++);
                        debugprintf(INFO, "Flushed wallet.dat %"PRI64d"ms\n", GetTimeMillis() - nStart);
                    }
                }
            }
        }
    }
}

bool BackupWallet(const CWallet& wallet, const string& strDest)
{
    if (!wallet.fFileBacked)
        return false;
    while (!fShutdown)
    {
        {
            MUTEX_LOCK(cs_db);
            if (!mapFileUseCount.count(wallet.strWalletFile) || mapFileUseCount[wallet.strWalletFile] == 0)
            {
                // Flush log data to the dat file
                CloseDb(wallet.strWalletFile);
                dbenv.txn_checkpoint(0, 0, 0);
                dbenv.lsn_reset(wallet.strWalletFile.c_str(), 0);
                mapFileUseCount.erase(wallet.strWalletFile);

                // Copy wallet.dat
                filesystem::path pathSrc(GetDataDir() + "/" + wallet.strWalletFile);
                filesystem::path pathDest(strDest);
                if (filesystem::is_directory(pathDest))
                    pathDest = pathDest / wallet.strWalletFile;
    #if BOOST_VERSION >= 104000
                filesystem::copy_file(pathSrc, pathDest, filesystem::copy_option::overwrite_if_exists);
    #else
                filesystem::copy_file(pathSrc, pathDest);
    #endif
                debugprintf(WARN, "copied wallet.dat to %s\n", pathDest.string().c_str());

                return true;
            }
        }
        Sleep(100);
    }
    return false;
}
