/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2002  ASAMI, Tomoharu (asami@AsamiOffice.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.xml.relax.expanded;

import java.util.*;
import org.w3c.dom.*;
import jp.gr.java_conf.jaba2.util.UArray;
import jp.gr.java_conf.jaba2.util.ElementCounter;
import jp.gr.java_conf.jaba2.util.LooseList;
import jp.gr.java_conf.jaba2.text.UString;
import jp.gr.java_conf.jaba2.datatype.IXMLDatatype;
import jp.gr.java_conf.jaba2.xml.UElement;
import jp.gr.java_conf.jaba2.xml.datatype.*;
import jp.gr.java_conf.jaba2.xml.relax.cooked.*;
import jp.gr.java_conf.jaba2.xml.relax.expanded.*;

/**
 * OptimizerThread1
 *
 * @since   Dec. 14, 2000
 * @version Jan. 21, 2002
 * @author  ASAMI, Tomoharu (asami@AsamiOffice.com)
 */
public class OptimizerThread1 extends Thread implements IOptimizerThread {
    private ERuleNode rule_;
    private EModule module_;
    private OptimizerController controller_;
    private SyncCounter sync_ = null;
    private String command_ = null;

    public OptimizerThread1(
	ThreadGroup group,
	ERuleNode node,
	EModule module
    ) {
	super(group, "optimizer-" + node.getLabel());
	rule_ = node;
	module_ = module;
    }

    public void setup() {
	try {
	    rule_.lockWrite();
	} catch (InterruptedException e) {
	    throw (new InternalError());
	}
    }

    public synchronized void prepare(SyncCounter sync) {
//System.out.println("prepare");
	command_ = "prepare";
	sync_ = sync;
	notifyAll();
    }

    public synchronized void cyclicResolve(SyncCounter sync) {
//System.out.println("cyclicResolve");
	command_ = "cyclicResolve";
	sync_ = sync;
	notifyAll();
    }

    public synchronized void noneResolve(SyncCounter sync) {
//System.out.println("noneResolve");
	command_ = "noneResolve";
	sync_ = sync;
	notifyAll();
    }

    public synchronized void optimize(SyncCounter sync) {
//System.out.println("optimize");
	command_ = "optimize";
	sync_ = sync;
	notifyAll();
    }

    public synchronized void naming(SyncCounter sync) {
//System.out.println("naming");
	command_ = "naming";
	sync_ = sync;
	notifyAll();
    }

    public void setController(OptimizerController controller) {
	controller_ = controller;
    }

    public synchronized void run() {
//System.out.println("start");
	for (;;) {
	    try {
		while (command_ == null) {
		    wait();
		}
//System.out.println("command = " + command_);
		if ("prepare".equals(command_)) {
		    _prepare();
		} else if ("cyclicResolve".equals(command_)) {
		    _cyclicResolve();
		} else if ("noneResolve".equals(command_)) {
		    _noneResolve();
		} else if ("optimize".equals(command_)) {
		    _optimize();
		} else if ("naming".equals(command_)) {
		    _naming();
		} else {
		    throw (new InternalError(command_));
		}
		command_ = null;
	    } catch (InterruptedException e) {
		throw (new InternalError());
	    }
	}
    }

    private void _prepare() throws InterruptedException {
	try {
	    _prepare(rule_);
	} finally {
	    _post();
	}
    }

    private void _prepare(ENode node) {
	node.setHometown(rule_);
	ENode[] children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    _prepare(children[i]);
	}
    }

    private void _cyclicResolve() throws InterruptedException {
	try {
	    Set refs = new HashSet();
	    _cyclicResolve(rule_, refs);
	    rule_.setCyclicRefs(refs);
	} finally {
	    _post();
	}
    }

    private void _cyclicResolve(ENode node, Set refs) {
	if (node instanceof EElementNode) {
	    if (!refs.contains(node)) {
		refs.add(node);
		_cyclicResolveChildren(node, refs);
	    }
	} else if (node instanceof EContentNode) {
	    if (!refs.contains(node)) {
		refs.add(node);
		_cyclicResolveChildren(node, refs);
	    }
	} else if (node instanceof EElementRefNode) {
	    EElementRefNode ref = (EElementRefNode)node;
	    _cyclicResolve(ref.getElementNode(), refs);
	} else if (node instanceof EContentRefNode) {
	    EContentRefNode ref = (EContentRefNode)node;
	    _cyclicResolve(ref.getContentNode(), refs);
	} else if (node instanceof EExternalRefNode) {
	    EExternalRefNode ref = (EExternalRefNode)node;
	    _cyclicResolve(ref.getElementNode(), refs);
	} else if (node instanceof EExternalContentRefNode) {
	    EExternalContentRefNode ref = (EExternalContentRefNode)node;
	    _cyclicResolve(ref.getContentNode(), refs);
	} else if (node instanceof ESequenceNode) {
	    _cyclicResolveChildren(node, refs);
	} else if (node instanceof EChoiceNode) {
	    _cyclicResolveChildren(node, refs);
	} else if (node instanceof EMixedNode) {
	    _cyclicResolveChildren(node, refs);
	}
    }

    private void _cyclicResolveChildren(ENode node, Set refs) {
	ENode[] children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    _cyclicResolve(children[i], refs);
	}
    }

    private void _noneResolve() throws InterruptedException {
	try {
	    if (rule_.isNone()) {
		return;
	    }
	    ENode[] children = rule_.getChildren();
	    for (int i = 0;i < children.length;i++) {
		if (children[i] instanceof ENoneNode) {
		    controller_.needRetry();
		    rule_.setNone(true);
		    return;
		} else {
		    boolean isNone = _noneResolve(children[i]);
		    if (isNone) {
			controller_.needRetry();
			rule_.setNone(true);
		    }
		}
	    }
	    rule_.setNone(false);
	} finally {
	    _post();
	}
    }

    private boolean _noneResolve(ENode node) {
	return (false);		// XXX
    }

    private void _optimize() throws InterruptedException {
	_adjustTree(rule_);
	_post();
    }

    private void _naming() throws InterruptedException {
	_post();
    }

    private void _post() throws InterruptedException {
	rule_.unlockWrite();
	sync_.done();
	sync_ = null;
    }

    private void _adjustTree(ERuleNode node) throws InterruptedException {
	if (node instanceof EElementNode) {
	    _adjustTree((EElementNode)node);
	} else if (node instanceof EContentNode) {
	    _adjustTree((EContentNode)node);
	} else {
	    throw (new InternalError());
	}
    }

    private void _adjustTree(EElementNode node) throws InterruptedException {
	node.setImmutable(false);
	ENode[] children = node.getChildren();
	node.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTree(node, children[i]);
	}
	if (node.size() == 0) {
	    if (node.getDatatype() == null) {
		node.setDatatype(new DEmptyString());
	    }
	}
	node.setImmutable(true);
	children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    children[i].setImmutable(true);
	}
    }

    private void _adjustTree(EContentNode node) throws InterruptedException {
	node.setImmutable(false);
	ENode[] children = node.getChildren();
	node.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTree(node, children[i]);
	}
	node.setImmutable(true);
	children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    children[i].setImmutable(true);
	}
    }

/*
    private void _adjustTree(ERuleNode parent, ENode child)
	throws InterruptedException {

	if (parent instanceof EElementNode) {
	    _adjustTree((EElementNode)parent, child);
	} else if (parent instanceof EContentNode) {
	    _adjustTree((EContentNode)parent, child);
	} else {
	    throw (new InternalError(child + " on " + parent));
	}
    }
*/

    private void _adjustTree(EElementNode parent, ENode child)
	throws InterruptedException {

	if (child instanceof EElementSlot) {
	    _adjustTreeInElementNode(parent, (EElementSlot)child);
	} else if (child instanceof EAttributeSlot) {
	    _adjustTreeInElementNode(parent, (EAttributeSlot)child);
	} else if (child instanceof EElementRefNode) {
	    _adjustTreeInElementNode(parent, (EElementRefNode)child);
	} else if (child instanceof EContentRefNode) {
	    _adjustTreeInElementNode(parent, (EContentRefNode)child);
	} else if (child instanceof EExternalRefNode) {
	    _adjustTreeInElementNode(parent, (EExternalRefNode)child);
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustTreeInElementNode(parent, (EExternalContentRefNode)child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInElementNode(parent, (ESequenceNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInElementNode(parent, (EChoiceNode)child);
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInElementNode(parent, (EMixedNode)child);
	} else if (child instanceof ENoneNode) {
	    parent.addChild(child);
	    parent.setNone(true);
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTree(EContentNode parent, ENode child)
	throws InterruptedException {

	if (child instanceof EElementSlot) {
	    _adjustTreeInContentNode(parent, (EElementSlot)child);
	} else if (child instanceof EAttributeSlot) {
	    _adjustTreeInContentNode(parent, (EAttributeSlot)child);
	} else if (child instanceof EElementRefNode) {
	    _adjustTreeInContentNode(parent, (EElementRefNode)child);
	} else if (child instanceof EContentRefNode) {
	    _adjustTreeInContentNode(parent, (EContentRefNode)child);
	} else if (child instanceof EExternalRefNode) {
	    _adjustTreeInContentNode(parent, (EExternalRefNode)child);
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustTreeInContentNode(parent, (EExternalContentRefNode)child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInContentNode(parent, (ESequenceNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInContentNode(parent, (EChoiceNode)child);
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInContentNode(parent, (EMixedNode)child);
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    // EElementNode context
    private void _adjustTreeInElementNode(
	EElementNode parent,
	EElementSlot eSlot
    ) throws InterruptedException {
	String typeName = eSlot.getDatatype().getName();
	if ("none".equals(typeName)) {
	    parent.addNoneElementSlot(eSlot);
	} else {
	    parent.addChild(eSlot);
	}
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EAttributeSlot aSlot
    ) throws InterruptedException {
	String typeName = aSlot.getDatatype().getName();
	if ("none".equals(typeName)) {
	    parent.addNoneAttributeSlot(aSlot);
	} else {
	    parent.addChild(aSlot);
	}
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EElementRefNode eRef
    ) throws InterruptedException {
	int occurs = eRef.getOccurs();
	EElementNode eNode = eRef.getElementNode();
	_adjustElementRefNodeInRule(parent, eRef, eNode, occurs);
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EContentRefNode cRef
    ) throws InterruptedException {
	int occurs = cRef.getOccurs();
	EContentNode cNode = cRef.getContentNode();
	_adjustContentRefNodeInRule(parent, cRef, cNode, occurs);
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EExternalRefNode eRef
    ) throws InterruptedException {
	EElementNode eNode = eRef.getElementNode();
	int occurs = eRef.getOccurs();
	_adjustExternalElementRefNodeInRule(parent, eRef, eNode, occurs);
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EExternalContentRefNode cRef
    ) throws InterruptedException {
	int occurs = cRef.getOccurs();
	EContentNode cNode = cRef.getContentNode();
	_adjustExternalContentRefNodeInRule(parent, cRef, cNode, occurs);
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	int occurs = sequence.getOccurs();
	ENode[] children = sequence.getChildren();
	sequence.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInSequenceNode(sequence, children[i]);
	}
	children = sequence.getChildren();
	if (children.length == 0) {
	    return;
	}
	if (sequence.getOccurs() == OCCURS_ONE) {
	    parent.addChildren(sequence);
	} else {
	    EContentRefNode cRef = _makeImplicitContentNode(
		_makeUniqueName("sequence"),
		occurs,
		children
	    );
	    parent.addChild(cRef);
	}
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	_adjustChoiceNode(parent, choice);
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EMixedNode mixed
    ) throws InterruptedException {
	ENode[] children = mixed.getChildren();
	mixed.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInMixedNode(mixed, children[i]);
	}
	parent.addChild(mixed);
    }

    // EContentNode context
    private void _adjustTreeInContentNode(
	EContentNode parent,
	EElementSlot eSlot
    ) throws InterruptedException {
	String typeName = eSlot.getDatatype().getName();
	if ("none".equals(typeName)) {
	    parent.addNoneElementSlot(eSlot);
	} else {
	    parent.addChild(eSlot);
	}
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EAttributeSlot aSlot
    ) throws InterruptedException {
	parent.addChild(aSlot);
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EElementRefNode eRef
    ) throws InterruptedException {
	int occurs = eRef.getOccurs();
	EElementNode eNode = eRef.getElementNode();
	// XXX : handle elementSlot more precisely
	_adjustElementRefNodeInRule(parent, eRef, eNode, occurs); // XXX
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EContentRefNode cRef
    ) throws InterruptedException {
	int occurs = cRef.getOccurs();
	EContentNode cNode = cRef.getContentNode();
	// XXX : handle elementSlot more precisely
	_adjustContentRefNodeInRule(parent, cRef, cNode, occurs); // XXX
/*
	if (cRef.getOccurs() == OCCURS_ONE) {
	    EContentNode cNode = cRef.getContentNode();
	    cNode.lockRead();
	    try {
		parent.addChildren(
		    _deepCopyChildren(cNode.getChildren())
		);
	    } finally {
		cNode.unlockRead();
	    }
	} else {
	    parent.addChild(cRef);
	}
*/
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EExternalRefNode eRef
    ) throws InterruptedException {
	int occurs = eRef.getOccurs();
	EElementNode eNode = eRef.getElementNode();
	_adjustExternalElementRefNodeInRule(parent, eRef, eNode, occurs);
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EExternalContentRefNode cRef
    ) throws InterruptedException {
	int occurs = cRef.getOccurs();
	EContentNode cNode = cRef.getContentNode();
	_adjustExternalContentRefNodeInRule(parent, cRef, cNode, occurs);
/*
	if (cRef.getOccurs() == OCCURS_ONE) {
	    EContentNode cNode = cRef.getContentNode();
	    cNode.lockRead();
	    try {
		parent.addChildren(
		    _deepCopyChildren(cNode.getChildren())
		);
	    } finally {
		cNode.unlockRead();
	    }
	} else {
	    parent.addChild(cRef);
	}
*/
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	if (sequence.getOccurs() == OCCURS_ONE) {
	    parent.addChildren(sequence);
	} else {
	    parent.addChild(sequence);
	}
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	_adjustChoiceNode(parent, choice);
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EMixedNode mixed
    ) throws InterruptedException {
	ENode[] children = mixed.getChildren();
	mixed.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInMixedNode(mixed, children[i]);
	}
	parent.addChild(mixed);
    }

    // ESequence context
    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	ENode child
    ) throws InterruptedException {
	if (child instanceof EElementSlot) {
	    _adjustTreeInSequenceNode(parent, (EElementSlot)child);
	} else if (child instanceof EAttributeSlot) {
	    _adjustTreeInSequenceNode(parent, (EAttributeSlot)child);
	} else if (child instanceof EElementRefNode) {
	    _adjustTreeInSequenceNode(parent, (EElementRefNode)child);
	} else if (child instanceof EContentRefNode) {
	    _adjustTreeInSequenceNode(parent, (EContentRefNode)child);
	} else if (child instanceof EExternalRefNode) {
	    _adjustTreeInSequenceNode(parent, (EExternalRefNode)child);    
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustTreeInSequenceNode(parent, (EExternalContentRefNode)child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInSequenceNode(parent, (ESequenceNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInSequenceNode(parent, (EChoiceNode)child);    
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInSequenceNode(parent, (EMixedNode)child);    
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EElementSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    // RELAX NG
    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EAttributeSlot child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EElementRefNode child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EContentRefNode child
    ) throws InterruptedException {
	parent.addChild(child);	// XXX
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EExternalRefNode child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EExternalContentRefNode child
    ) throws InterruptedException {
	parent.addChild(child);	// XXX
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	ENode[] children = sequence.getChildren();
	sequence.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInSequenceNode(sequence, children[i]);
	}
	children = sequence.getChildren();
	if (children.length == 0) { // XXX : unify sequence child series
	    return;
	} else if (children.length == 1) {
	    int occurs = sequence.getOccurs();
	    ENode child = children[0];
	    if (child instanceof EElementSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAttributeSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EElementRefNode) {
		EElementRefNode oldRef = (EElementRefNode)child;
		EElementRefNode newRef = new EElementRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EContentRefNode) {
		EContentRefNode oldRef = (EContentRefNode)child;
		EContentRefNode newRef = new EContentRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EExternalRefNode) {
		EExternalRefNode oldRef = (EExternalRefNode)child;
		EExternalRefNode newRef = new EExternalRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EExternalContentRefNode) {
		EExternalContentRefNode oldRef
		    = (EExternalContentRefNode)child;
		EExternalContentRefNode newRef
		    = new EExternalContentRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof ESequenceNode) {
		ESequenceNode childSequence = (ESequenceNode)child;
		sequence.setOccurs(
		    UERule.unifyOccurs(
			childSequence.getOccurs(),
			occurs
		    )
		);
		parent.addChild(childSequence);
	    } else if (child instanceof EChoiceNode) {
		EChoiceNode childChoice = (EChoiceNode)child;
		childChoice.setOccurs(
		    UERule.unifyOccurs(
			childChoice.getOccurs(),
			occurs
		    )
		);
		parent.addChild(childChoice);
	    } else if (child instanceof EMixedNode) {
		parent.addChild(child);
	    } else if (child instanceof EAnyOtherAttributeNode) {
		parent.addChild(child);
	    } else if (child instanceof EAnyOtherElementNode) {
		parent.addChild(child);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	} else {
	    EContentRefNode cRef = _makeImplicitContentNode(
		_makeUniqueName("sequence"),
		sequence.getOccurs(),
		children
	    );
	    parent.addChild(cRef);
	}
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	_adjustChoiceNode(parent, choice);
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EMixedNode mixed
    ) throws InterruptedException {
	ENode[] children = mixed.getChildren();
	mixed.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInMixedNode(mixed, children[i]);
	}
	parent.addChild(mixed);
    }

    // EChoiceNode context
    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	ENode child
    ) throws InterruptedException {
	if (child instanceof EElementSlot) {
	    _adjustTreeInChoiceNode(parent, (EElementSlot)child);
	} else if (child instanceof EAttributeSlot) {
	    _adjustTreeInChoiceNode(parent, (EAttributeSlot)child);
	} else if (child instanceof EElementRefNode) {
	    _adjustTreeInChoiceNode(parent, (EElementRefNode)child);
	} else if (child instanceof EContentRefNode) {
	    _adjustTreeInChoiceNode(parent, (EContentRefNode)child);
	} else if (child instanceof EExternalRefNode) {
	    _adjustTreeInChoiceNode(parent, (EExternalRefNode)child);    
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustTreeInChoiceNode(parent, (EExternalContentRefNode)child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInChoiceNode(parent, (ESequenceNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInChoiceNode(parent, (EChoiceNode)child);    
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInChoiceNode(parent, (EMixedNode)child);
	} else if (child instanceof ENoneNode) {
	    // do nothing
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EElementSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EAttributeSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EElementRefNode eRef
    ) throws InterruptedException {
	_adjustElementRefNodeInChoice(
	    parent,
	    eRef,
	    eRef.getElementNode(),
	    eRef.getOccurs()
	);
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EContentRefNode cRef
    ) throws InterruptedException {
	_adjustContentRefNodeInChoice(
	    parent,
	    cRef,
	    cRef.getContentNode(),
	    cRef.getOccurs()
	);
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EExternalRefNode eRef
    ) throws InterruptedException {
	_adjustElementRefNodeInChoice(
	    parent,
	    eRef,
	    eRef.getElementNode(),
	    eRef.getOccurs()
	);
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EExternalContentRefNode cRef
    ) throws InterruptedException {
	_adjustContentRefNodeInChoice(
	    parent,
	    cRef,
	    cRef.getContentNode(),
	    cRef.getOccurs()
	);
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	ENode[] children = sequence.getChildren();
	sequence.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInSequenceNode(sequence, children[i]);
	}
	children = sequence.getChildren();
	if (children.length == 0) { // XXX : unify Sequence Series
	    return;
	} else if (children.length == 1) {
	    int occurs = sequence.getOccurs();
	    ENode child = children[0];
	    if (child instanceof EElementSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAttributeSlot) {
		EContentRefNode cRef = _makeImplicitContentNode(
		    _makeUniqueName("attribute"),
		    occurs,
		    children
		);
		parent.addChild(cRef);
	    } else if (child instanceof EElementRefNode) {
		EElementRefNode oldRef = (EElementRefNode)child;
		EElementRefNode newRef = new EElementRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EContentRefNode) {
		EContentRefNode oldRef = (EContentRefNode)child;
		EContentRefNode newRef = new EContentRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EExternalRefNode) {
		EExternalRefNode oldRef = (EExternalRefNode)child;
		EExternalRefNode newRef = new EExternalRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EExternalContentRefNode) {
		EExternalContentRefNode oldRef
		    = (EExternalContentRefNode)child;
		EExternalContentRefNode newRef
		    = new EExternalContentRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof ESequenceNode) {
		ESequenceNode childSequence = (ESequenceNode)child;
		sequence.setOccurs(
		    UERule.unifyOccurs(
			childSequence.getOccurs(),
			occurs
		    )
		);
		parent.addChild(childSequence);
	    } else if (child instanceof EChoiceNode) {
		EChoiceNode childChoice = (EChoiceNode)child;
		childChoice.setOccurs(
		    UERule.unifyOccurs(
			childChoice.getOccurs(),
			occurs
		    )
		);
		parent.addChild(childChoice);
	    } else if (child instanceof EMixedNode) {
		parent.addChild(child);
	    } else if (child instanceof EAnyOtherAttributeNode) {
		parent.addChild(child);
	    } else if (child instanceof EAnyOtherElementNode) {
		parent.addChild(child);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	} else {
	    EContentRefNode cRef = _makeImplicitContentNode(
		_makeUniqueName("sequence"),
		sequence.getOccurs(),
		children
	    );
	    parent.addChild(cRef);
	}
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	ENode[] children = choice.getChildren();
	choice.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInChoiceNode(choice, children[i]);
	}
	children = choice.getChildren();
	choice.clear();
	if (children.length == 0) {
	    return;
	} else if (children.length == 1) {
	    _makeOptimizedEntryForChoiceChoiceOne(parent, children[0]);
	} else {
	    if (UERule.canUnification(
		parent.getOccurs(), choice.getOccurs())
	    ) {
		parent.addChildren(children);
	    } else {
		EChoiceNode newChoice = new EChoiceNode();
		newChoice.setOccurs(choice.getOccurs());
		newChoice.setHometown(rule_); // XXX
		newChoice.addChildren(children);
		EContentRefNode cRef = _makeImplicitContentNode(
		    _makeUniqueName("choice"),
		    OCCURS_ONE,
		    new ENode[] { newChoice }
		);
		parent.addChild(cRef);
	    }
	}
    }

    private void _makeOptimizedEntryForChoiceChoiceOne(
	EChoiceNode parent,
	ENode child
    ) throws InterruptedException {
	int choiceOccurs = parent.getOccurs();
	if (child instanceof EElementSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EAttributeSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EElementRefNode) {
	    EElementRefNode childRef = (EElementRefNode)child;
	    int childOccurs = childRef.getOccurs();
	    childRef.setOccurs(
		UERule.unifyOccurs(choiceOccurs, childOccurs)
	    );
	    parent.addChild(childRef);
	} else if (child instanceof EContentRefNode) {
	    _adjustContentRefNodeForExpandInChoice(
		parent,
		(EContentRefNode)child,
		choiceOccurs
	    );
	} else if (child instanceof EExternalRefNode) {
	    EExternalRefNode childRef = (EExternalRefNode)child;
	    int childOccurs = childRef.getOccurs();
	    childRef.setOccurs(
		UERule.unifyOccurs(choiceOccurs, childOccurs)
	    );
	    parent.addChild(childRef);
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustExternalContentRefNodeForExpandInChoice(
		parent,
		(EExternalContentRefNode)child,
		choiceOccurs
	    );
	} else if (child instanceof ESequenceNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EChoiceNode) {
	    throw (new InternalError(child.toString()));
/*
	    EChoiceNode childChoice = (EChoiceNode)child;
	    int childOccurs = childChoice.getOccurs();
	    int newOccurs = UERule.unifyOccurs(choiceOccurs, childOccurs);
	    choice.setOccurs(newOccurs);
	    choice.addChildren(child);
	    parent.addChild(choice);
*/
	} else if (child instanceof EMixedNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof ENoneNode) {
	    // do nothing
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EMixedNode mixed
    ) throws InterruptedException {
	EContentRefNode cRef = _makeImplicitContentNode(
	    _makeUniqueName("mixed"),
	    OCCURS_ONE,
	    new ENode[] { mixed }
	);
	parent.addChild(cRef);
    }

    // EMixed context
    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	ENode child
    ) throws InterruptedException {
	if (child instanceof EElementSlot) {
	    _adjustTreeInMixedNode(parent, (EElementSlot)child);
	} else if (child instanceof EAttributeSlot) {
	    _adjustTreeInMixedNode(parent, (EAttributeSlot)child);
	} else if (child instanceof EElementRefNode) {
	    _adjustTreeInMixedNode(parent, (EElementRefNode)child);
	} else if (child instanceof EContentRefNode) {
	    _adjustTreeInMixedNode(parent, (EContentRefNode)child);
	} else if (child instanceof EExternalRefNode) {
	    _adjustTreeInMixedNode(parent, (EExternalRefNode)child);    
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustTreeInMixedNode(parent, (EExternalContentRefNode)child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInMixedNode(parent, (ESequenceNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInMixedNode(parent, (EChoiceNode)child);    
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInMixedNode(parent, (EMixedNode)child);    
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EElementSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EAttributeSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EElementRefNode child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EContentRefNode child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EExternalRefNode child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EExternalContentRefNode child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	ENode[] children = sequence.getChildren();
	sequence.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInSequenceNode(sequence, children[i]);
	}
	children = sequence.getChildren();
	if (children.length == 0) {
	    return;
	} else if (children.length == 1) {
	    _makeOptimizedEntryForMixedSequenceOne(parent, children[0]);
	} else {
	    EContentRefNode cRef = _makeImplicitContentNode(
		_makeUniqueName("sequence"),
		sequence.getOccurs(),
		children
	    );
	    parent.addChild(cRef);
	}
    }

    private void _makeOptimizedEntryForMixedSequenceOne(
	EMixedNode parent,
	ENode child
    ) {
	if (child instanceof EElementSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EAttributeSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EElementRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EContentRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EExternalRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EExternalContentRefNode) {
	    parent.addChild(child);
	} else if (child instanceof ESequenceNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EChoiceNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EMixedNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	ENode[] children = choice.getChildren();
	choice.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInChoiceNode(choice, children[i]);
	}
	children = choice.getChildren();
	if (children.length == 0) {
	    return;
	} else {
	    for (int i = 0;i < children.length;i++) {
		_makeOptimizedEntryForMixed(parent, children[i]);
	    }
	}
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EMixedNode mixed
    ) throws InterruptedException {
	ENode[] children = mixed.getChildren();
	mixed.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInMixedNode(mixed, children[i]);
	}
	children = mixed.getChildren();
	if (children.length == 0) {
	    return;
	} else {
	    for (int i = 0;i < children.length;i++) {
		_makeOptimizedEntryForMixed(parent, children[i]);
	    }
	}
    }

    private void _makeOptimizedEntryForMixed(
	EMixedNode parent,
	ENode child
    ) throws InterruptedException {
	if (child instanceof EElementSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EAttributeSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EElementRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EContentRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EExternalRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EExternalContentRefNode) {
	    parent.addChild(child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInMixedNode(parent, (ESequenceNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInMixedNode(parent, (EChoiceNode)child);
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInMixedNode(parent, (EMixedNode)child);
	} else if (child instanceof ENoneNode) {
	    // do nothing
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    // common procudures
    private void _adjustElementRefNodeInRule(
	ENode parent,
	ENode eRef,
	EElementNode eNode,
	int occurs
    ) throws InterruptedException {
	if (eNode.isNone()) {
	    parent.addChild(new ENoneNode());
	    return;
	}
	if (eNode.isCyclic(rule_)) {
	    parent.addChild(eRef);
	    return;
	}
	eNode.lockRead();
	try {
	    if (eNode.isNone()) {
		parent.addChild(new ENoneNode());
		return;
	    }
	    if (eNode.getSize() > 0) {
		parent.addChild(eRef);
		return;
	    }
	    IXMLDatatype datatype = eNode.getDatatype();
	    if (datatype == null) {
		datatype = new DEmptyString();
	    }
	    if (datatype instanceof DEmptyString) {
		parent.addChild(eRef);
		return;
	    }
	    EElementSlot eSlot = new EElementSlot(
		eNode.getElementName(),
		datatype,
		occurs
	    );
	    eSlot.setAppinfo(eNode.getAppinfo());
	    eSlot.setBase(eNode.getBase());
	    eSlot.setModule(module_);
	    eSlot.setElementNode(eNode);
	    String datatypeName = datatype.getName();
	    if ("none".equals(datatypeName)) {
		parent.addNoneElementSlot(eSlot);
	    } else {
		parent.addChild(eSlot);
	    }
/*
	    if (eNode instanceof EElementSlotNode) {
		EElementSlotNode esNode = (EElementSlotNode)eNode;
		IXMLDatatype datatype = esNode.getDatatype();
		String datatypeName = datatype.getName();
		if (esNode.getSize() > 0) {
		    parent.addChild(eRef);
		    return;
		}
		if ("emptyString".equals(datatypeName)) {
		    parent.addChild(eRef);
		    return;
		}
		EElementSlot eSlot = new EElementSlot(
		    esNode.getElementName(),
		    datatype,
		    occurs
		);
		eSlot.setAppinfo(esNode.getAppinfo());
		eSlot.setBase(esNode.getBase());
		eSlot.setModule(module_);
		if ("none".equals(datatypeName)) {
		    parent.addNoneElementSlot(eSlot);
		} else {
		    parent.addChild(eSlot);
		}
	    } else {
		parent.addChild(eRef);
	    }
*/
	} finally {
	    eNode.unlockRead();
	}
    }

    private void _adjustExternalElementRefNodeInRule(
	ENode parent,
	ENode eRef,
	EElementNode eNode,
	int occurs
    ) throws InterruptedException {
	if (eNode.isNone()) {
	    parent.addChild(new ENoneNode());
	    return;
	}
	if (eNode.isCyclic(rule_)) {
	    parent.addChild(eRef);
	    return;
	}
	eNode.lockRead();
	try {
	    if (eNode.isNone()) {
		parent.addChild(new ENoneNode());
		return;
	    }
	    parent.addChild(eRef);
	} finally {
	    eNode.unlockRead();
	}
    }

    private void _adjustContentRefNodeInRule(
	ENode parent,
	ENode cRef,
	EContentNode cNode,
	int occurs
    ) throws InterruptedException {
	if (cNode.isNone()) {
	    if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		parent.addChild(new ENoneNode());
	    }
	    return;
	}
	cNode.lockRead();
	try {
	    if (cNode.isNone()) {
		if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		    parent.addChild(new ENoneNode());
		}
		return;
	    }
	    ENode[] children = cNode.getChildren();
	    if (children.length == 0) {
		return;
	    } else if (occurs == OCCURS_ONE) {
		// differ against _adjustContentRefNode
		cNode.lockRead();
		try {
		    parent.addChildren(
			_deepCopyChildren(cNode.getChildren())
		    );
		} finally {
		    cNode.unlockRead();
		}
	    } else if (children.length == 1) {
		ENode child = _deepCopy(children[0]);
		if (child instanceof EElementSlot) {
		    parent.addChild(child);
		} else if (child instanceof EAttributeSlot) {
		    throw (new InternalError(child.toString()));
		} else if (child instanceof EElementRefNode) {
		    EElementRefNode oldRef = (EElementRefNode)child;
		    EElementRefNode newRef = new EElementRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EContentRefNode) {
		    EContentRefNode oldRef = (EContentRefNode)child;
		    EContentRefNode newRef = new EContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalRefNode) {
		    EExternalRefNode oldRef = (EExternalRefNode)child;
		    EExternalRefNode newRef = new EExternalRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalContentRefNode) {
		    EExternalContentRefNode oldRef
			= (EExternalContentRefNode)child;
		    EExternalContentRefNode newRef
			= new EExternalContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof ESequenceNode) {
		    parent.addChild(child);
		} else if (child instanceof EChoiceNode) {
		    EChoiceNode choice = (EChoiceNode)child;
		    choice.setOccurs(
			UERule.unifyOccurs(
			    choice.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(child);
		} else if (child instanceof EMixedNode) {
		    parent.addChild(child);
		} else if (child instanceof ENoneNode) {
		    return;
		} else if (child instanceof EAnyOtherAttributeNode) {
		    parent.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    parent.addChild(child);
		} else {
		    throw (new InternalError(child.toString()));
		}
	    } else {
		parent.addChild(cRef);
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    private void _adjustExternalContentRefNodeInRule(
	ENode parent,
	ENode cRef,
	EContentNode cNode,
	int occurs
    ) throws InterruptedException {
	if (cNode.isNone()) {
	    if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		parent.addChild(new ENoneNode());
	    }
	    return;
	}
	cNode.lockRead();
	try {
	    if (cNode.isNone()) {
		if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		    parent.addChild(new ENoneNode());
		}
		return;
	    }
	    ENode[] children = cNode.getChildren();
	    if (children.length == 0) {
		return;
	    } else if (occurs == OCCURS_ONE) {
		cNode.lockRead();
		try {
		    // differ against _adjustContentRefNodeInRule
		    parent.addChildren(
			_deepCopyChildrenInContentRef(cNode.getChildren())
		    );
		} finally {
		    cNode.unlockRead();
		}
	    } else if (children.length == 1) {
		// differ against _adjustContentRefNodeInRule
		ENode child = _deepCopyInContentRef(children[0]);
		if (child instanceof EElementSlot) {
		    parent.addChild(child);
		} else if (child instanceof EAttributeSlot) {
		    throw (new InternalError(child.toString()));
		} else if (child instanceof EElementRefNode) {
		    EElementRefNode oldRef = (EElementRefNode)child;
		    EElementRefNode newRef = new EElementRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EContentRefNode) {
		    EContentRefNode oldRef = (EContentRefNode)child;
		    EContentRefNode newRef = new EContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalRefNode) {
		    EExternalRefNode oldRef = (EExternalRefNode)child;
		    EExternalRefNode newRef = new EExternalRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalContentRefNode) {
		    EExternalContentRefNode oldRef
			= (EExternalContentRefNode)child;
		    EExternalContentRefNode newRef
			= new EExternalContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof ESequenceNode) {
		    parent.addChild(child);
		} else if (child instanceof EChoiceNode) {
		    EChoiceNode choice = (EChoiceNode)child;
		    choice.setOccurs(
			UERule.unifyOccurs(
			    choice.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(child);
		} else if (child instanceof EMixedNode) {
		    parent.addChild(child);
		} else if (child instanceof ENoneNode) {
		    return;
		} else if (child instanceof EAnyOtherAttributeNode) {
		    parent.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    parent.addChild(child);
		} else {
		    throw (new InternalError(child.toString()));
		}
	    } else {
		parent.addChild(cRef);
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    private ENode[] _deepCopyChildrenInContentRef(ENode[] children) {
	ENode[] result = new ENode[children.length];
	for (int i = 0;i < children.length;i++) {
	    result[i] = _deepCopyInContentRef(children[i]);
	}
	return (result);
    }

    private ENode _deepCopyInContentRef(ENode node) {
	if (node instanceof EElementSlot) {
	    EElementSlot eSlot = (EElementSlot)node;
	    EElementNode eNode = eSlot.getElementNode();
	    if (eNode == null) {
		throw (new InternalError());
	    }
	    return (new EElementRefNode(eNode, eSlot.getOccurs()));
	} else {
	    return (_deepCopy(node));
	}
    }

    private void _adjustElementRefNodeInChoice(
	EChoiceNode parent,
	ENode eRef,
	EElementNode eNode,
	int occurs
    ) throws InterruptedException {
	if (eNode.isNone()) {
	    return;
	}
	if (UERule.canUnification(parent.getOccurs(), occurs)) {
	    parent.addChild(eRef);
	} else {
	    EContentRefNode cRef = _makeImplicitContentNode(
		_makeUniqueName("ref"),
		OCCURS_ONE,
		new ENode[] { eRef }
	    );
	    parent.addChild(cRef);
	}
    }

    private void _adjustElementRefNode(	// XXX : not used
	ENode parent,
	ENode eRef,
	EElementNode eNode,
	int occurs
    ) throws InterruptedException {
	if (eNode.isNone()) {
	    if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		parent.addChild(new ENoneNode());
		return;
	    }
	}
	parent.addChild(eRef);
    }

    private void _adjustContentRefNode(	// XXX : not used
	ENode parent,
	ENode cRef,
	EContentNode cNode,
	int occurs
    ) throws InterruptedException {
	if (cNode.isNone()) {
	    if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		parent.addChild(new ENoneNode());
		return;
	    }
	}
	cNode.lockRead();
	try {
	    if (cNode.isNone()) {
		if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		    parent.addChild(new ENoneNode());
		    return;
		}
	    }
	    ENode[] children = cNode.getChildren();
	    if (children.length == 0) {
		return;
	    } else if (children.length == 1) {
		ENode child = _deepCopy(children[0]);
		if (child instanceof EElementSlot) {
		    parent.addChild(child);
		} else if (child instanceof EAttributeSlot) {
		    throw (new InternalError(child.toString()));
		} else if (child instanceof EElementRefNode) {
		    EElementRefNode oldRef = (EElementRefNode)child;
		    EElementRefNode newRef = new EElementRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EContentRefNode) {
		    EContentRefNode oldRef = (EContentRefNode)child;
		    EContentRefNode newRef = new EContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalRefNode) {
		    EExternalRefNode oldRef = (EExternalRefNode)child;
		    EExternalRefNode newRef = new EExternalRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalContentRefNode) {
		    EExternalContentRefNode oldRef
			= (EExternalContentRefNode)child;
		    EExternalContentRefNode newRef
			= new EExternalContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof ESequenceNode) {
		    parent.addChild(child);
		} else if (child instanceof EChoiceNode) {
		    EChoiceNode choice = (EChoiceNode)child;
		    choice.setOccurs(
			UERule.unifyOccurs(
			    choice.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(child);
		} else if (child instanceof EMixedNode) {
		    parent.addChild(child);
		} else if (child instanceof ENoneNode) {
		    return;
		} else if (child instanceof EAnyOtherAttributeNode) {
		    parent.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    parent.addChild(child);
		} else {
		    throw (new InternalError(child.toString()));
		}
	    } else {
		parent.addChild(cRef);
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    private void _adjustContentRefNodeInChoice(
	EChoiceNode parent,
	ENode cRef,
	EContentNode cNode,
	int occurs
    ) throws InterruptedException {
	if (cNode.isNone()) {
	    return;
	}
	cNode.lockRead();
	try {
	    if (cNode.isNone()) {
		    return;
	    }
	    ENode[] children = cNode.getChildren();
	    if (children.length == 0) {
		return;
	    } else if (children.length == 1) {
		ENode child = _deepCopy(children[0]);
		if (child instanceof EElementSlot) {
		    parent.addChild(cRef);
		} else if (child instanceof EAttributeSlot) {
		    throw (new InternalError(child.toString()));
		} else if (child instanceof EElementRefNode) {
		    EElementRefNode oldRef = (EElementRefNode)child;
		    EElementRefNode newRef = new EElementRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EContentRefNode) {
		    EContentRefNode oldRef = (EContentRefNode)child;
		    EContentRefNode newRef = new EContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalRefNode) {
		    EExternalRefNode oldRef = (EExternalRefNode)child;
		    EExternalRefNode newRef = new EExternalRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalContentRefNode) {
		    EExternalContentRefNode oldRef
			= (EExternalContentRefNode)child;
		    EExternalContentRefNode newRef
			= new EExternalContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof ESequenceNode) {
		    ESequenceNode childSequence = (ESequenceNode)child;
		    if (childSequence.getSize() <= 1) {
			throw (new InternalError());
		    }
		    parent.addChild(cRef);
		} else if (child instanceof EChoiceNode) {
		    EChoiceNode childChoice = (EChoiceNode)child;
		    if (UERule.canUnification(
			  parent.getOccurs(),
			  childChoice.getOccurs())) {

			parent.addChildren(childChoice);
		    } else {
			parent.addChild(cRef);
		    }
		} else if (child instanceof EMixedNode) {
		    parent.addChild(cRef);
		} else if (child instanceof ENoneNode) {
		    return;
		} else if (child instanceof EAnyOtherAttributeNode) {
		    parent.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    parent.addChild(child);
		} else {
		    throw (new InternalError(child.toString()));
		}
	    } else {
		if (UERule.canUnification(parent.getOccurs(), occurs)) {
		    parent.addChild(cRef);
		} else {
		    EContentRefNode newRef = _makeImplicitContentNode(
			_makeUniqueName("ref"),
			OCCURS_ONE,
			new ENode[] { cRef }
		    );
		    parent.addChild(newRef);
		}
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    private void _adjustContentRefNodeForExpandInChoice(
	ENode parent,
	EContentRefNode cRef,
	int currentOccurs
    ) throws InterruptedException {
	EContentNode cNode = cRef.getContentNode();
	int refOccurs = cRef.getOccurs();
	cNode.lockRead();
	try {
	    if (cNode.getSize() == 0) {
		return;
	    } else if (cNode.getSize() == 1) {
		if (refOccurs == OCCURS_ONE) {
		    parent.addChildren(
			_deepCopyChildren(cNode.getChildren())
		    );
		} else {
		    parent.addChild(cRef);
		}
	    } else {
		cRef.setOccurs(	// XXX
		    UERule.unifyOccurs(currentOccurs, refOccurs)
		);
		parent.addChild(cRef);
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    private void _adjustExternalContentRefNodeForExpandInChoice(
	ENode parent,
	EExternalContentRefNode cRef,
	int currentOccurs
    ) throws InterruptedException {
	EContentNode cNode = cRef.getContentNode();
	int refOccurs = cRef.getOccurs();
	cNode.lockRead();
	try {
	    if (cNode.getSize() == 0) {
		return;
	    } else if (cNode.getSize() == 1) {
		if (refOccurs == OCCURS_ONE) {
		    parent.addChildren(
			_deepCopyChildren(cNode.getChildren())
		    );
		} else {
		    parent.addChild(cRef);
		}
	    } else {
		cRef.setOccurs(
		    UERule.unifyOccurs(currentOccurs, refOccurs)
		);
		parent.addChild(cRef);
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    // _adjustTreeInElementNode
    // _adjustTreeInContentNode
    // _adjustTreeInSequenceNode
    private void _adjustChoiceNode(
	ENode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	ENode[] children = choice.getChildren();
	choice.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInChoiceNode(choice, children[i]);
	}
	children = _sortChoiceChildren(choice.getChildren());
	choice.clear();
	if (children.length == 0) {
	    return;
	} else if (children.length == 1) {
	    _makeOptimizedEntryForChoiceOne(
		parent,
		children[0],
		choice.getOccurs()
	    );
	} else {
	    choice.addChildren(children);
	    parent.addChild(choice);
	}
    }

    private ENode[] _sortChoiceChildren(ENode[] children) {
	LooseList listByWidth = new LooseList();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAttributeSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EElementRefNode ||
		       child instanceof EContentRefNode ||
		       child instanceof EExternalRefNode ||
		       child instanceof EExternalContentRefNode) {

		int width = _calcTargetWidth(child);
		List list;
		if (listByWidth.size() <= width) {
		    list = new ArrayList();
		    listByWidth.put(width, list);
		} else {
		    list = (List)listByWidth.get(width);
		    if (list == null) {
			list = new ArrayList();
			listByWidth.put(width, list);
		    }
		}
		list.add(child);
	    } else if (child instanceof ESequenceNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EChoiceNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EMixedNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAnyOtherAttributeNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAnyOtherElementNode) {
		throw (new InternalError(child.toString()));
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
	ArrayList resultList = new ArrayList();
	for (int i = listByWidth.size() - 1;i >= 0;i--) {
	    List list = (List)listByWidth.get(i);
	    if (list != null) {
		resultList.addAll(list);
	    }
	}
	ENode[] result = new ENode[resultList.size()];
	return ((ENode[])resultList.toArray(result));
    }

    private int _calcTargetWidth(ENode node) {
	ENode target;
	if (node instanceof EElementRefNode) {
	    target = ((EElementRefNode)node).getElementNode();
	} else if (node instanceof EContentRefNode) {
	    target = ((EContentRefNode)node).getContentNode();
	} else if (node instanceof EExternalRefNode) {
	    target = ((EExternalRefNode)node).getElementNode();
	} else if (node instanceof EExternalContentRefNode) {
	    target = ((EExternalContentRefNode)node).getContentNode();
	} else {
	    throw (new InternalError());
	}
	return (target.getSize());
    }

    private void _makeOptimizedEntryForChoiceOne(
	ENode parent,
	ENode child,
	int occurs
    ) {
	if (child instanceof EElementSlot) {
	    parent.addChild(child);
	} else if (child instanceof EAttributeSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EElementRefNode) {
	    EElementRefNode oldRef = (EElementRefNode)child;
	    EElementRefNode newRef = new EElementRefNode(oldRef);
	    newRef.setOccurs(
		UERule.unifyOccurs(
		    oldRef.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newRef);
	} else if (child instanceof EContentRefNode) {
	    EContentRefNode oldRef = (EContentRefNode)child;
	    EContentRefNode newRef = new EContentRefNode(oldRef);
	    newRef.setOccurs(
		UERule.unifyOccurs(
		    oldRef.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newRef);
	} else if (child instanceof EExternalRefNode) {
	    EExternalRefNode oldRef = (EExternalRefNode)child;
	    EExternalRefNode newRef = new EExternalRefNode(oldRef);
	    newRef.setOccurs(
		UERule.unifyOccurs(
		    oldRef.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newRef);
	} else if (child instanceof EExternalContentRefNode) {
	    EExternalContentRefNode oldRef
		= (EExternalContentRefNode)child;
	    EExternalContentRefNode newRef
		= new EExternalContentRefNode(oldRef);
	    newRef.setOccurs(
		UERule.unifyOccurs(
		    oldRef.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newRef);
	} else if (child instanceof ESequenceNode) {
	    parent.addChild(child); // XXX : more optimize
	} else if (child instanceof EChoiceNode) {
	    parent.addChild(child); // XXX : more optimize
	} else if (child instanceof EMixedNode) {
	    parent.addChild(child); // XXX : more optimize
	} else if (child instanceof ENoneNode) {
	    return;
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private EContentRefNode _makeImplicitContentNode(
	String name,
	int occurs,
	ENode[] children
    ) {
	EContentNode cNode = new EContentNode(name);
	cNode.addChildren(children);
	cNode.setModule(module_);
	module_.addOptimizedContentRule(cNode);
	return (new EContentRefNode(cNode, occurs));
    }

    private ENode _deepCopy(ENode node) {
	if (node instanceof EElementSlot) {
	    return (new EElementSlot((EElementSlot)node));
	} else if (node instanceof EAttributeSlot) {
	    return (new EAttributeSlot((EAttributeSlot)node));
	} else if (node instanceof EElementRefNode) {
	    return (new EElementRefNode((EElementRefNode)node));
	} else if (node instanceof EContentRefNode) {
	    return (new EContentRefNode((EContentRefNode)node));
	} else if (node instanceof EChoiceNode) {
	    return (new EChoiceNode((EChoiceNode)node)); // XXX : modify
	} else if (node instanceof ENoneNode) {
	    return (new ENoneNode());
	} else if (node instanceof EExternalRefNode) {
	    return (new EExternalRefNode((EExternalRefNode)node));
	} else if (node instanceof EExternalContentRefNode) {
	    return (
		new EExternalContentRefNode((EExternalContentRefNode)node)
	    );
	} else if (node instanceof EAnyOtherAttributeNode) {
	    return (new EAnyOtherAttributeNode((EAnyOtherAttributeNode)node));
	} else if (node instanceof EAnyOtherElementNode) {
	    return (new EAnyOtherElementNode((EAnyOtherElementNode)node));
	} else {
	    throw (new InternalError());
	}
    }

    private ENode[] _deepCopyChildren(ENode[] children) {
	ENode[] result = new ENode[children.length];
	for (int i = 0;i < children.length;i++) {
	    result[i] = _deepCopy(children[i]);
	}
	return (result);
    }

    //
    private static ElementCounter nameCounter__ = new ElementCounter();

    private static String _makeUniqueName(String base) {
	nameCounter__.add(base);
	return (UString.uncapitalize(base + nameCounter__.getCount(base)));
    }

/*
    //
    private static int seqNo__ = 0;

    private static int _getSequenceNo() {
	return (++seqNo__);
    }
*/
}

/*
	if (child instanceof EElementSlot) {
	} else if (child instanceof EAttributeSlot) {
	} else if (child instanceof EElementRefNode) {
	} else if (child instanceof EContentRefNode) {
	} else if (child instanceof EExternalRefNode) {
	} else if (child instanceof EExternalContentRefNode) {
	} else if (child instanceof ESequenceNode) {
	} else if (child instanceof EChoiceNode) {
	} else if (child instanceof EMixedNode) {
	} else if (child instanceof EAnyOtherElementNode) {
	} else {
	    throw (new InternalError(child.toString()));
	}
 */
