/*
This source file is part of Rigs of Rods
Copyright 2005,2006,2007,2008,2009 Pierre-Michel Ricordel
Copyright 2007,2008,2009 Thomas Fischer

For more information, see http://www.rigsofrods.com/

Rigs of Rods is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License version 3, as 
published by the Free Software Foundation.

Rigs of Rods is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Rigs of Rods.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "airbrake.h"
#include "ResourceBuffer.h"

Airbrake::Airbrake(SceneManager *manager, char* basename, int num, node_t *ndref, node_t *ndx, node_t *ndy, node_t *nda, Vector3 pos, float width, float length, float maxang, char* texname, float tx1, float ty1, float tx2, float ty2)
    {
		snode=0;
		noderef=ndref;
		nodex=ndx; nodex->iIsSkin=true;
		nodey=ndy; nodey->iIsSkin=true;
		nodea=nda; nodea->iIsSkin=true;
		offset=pos;
		maxangle=maxang;
		area=width*length;
		char meshname[256];
		sprintf(meshname, "airbrakemesh-%s-%i", basename, num);
		/// Create the mesh via the MeshManager
        msh = MeshManager::getSingleton().createManual(meshname, ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME, new ResourceBuffer());

		union
		{
		float *vertices;
		CoVertice_t *covertices;
		};

        /// Create submesh
        SubMesh* sub = msh->createSubMesh();

		//materials
		sub->setMaterialName(texname);

        /// Define the vertices
        size_t nVertices = 4;
        size_t vbufCount = (2*3+2)*nVertices;
		vertices=(float*)malloc(vbufCount*sizeof(float));

		//textures coordinates
		covertices[0].texcoord=Vector2(tx1, ty1);
		covertices[1].texcoord=Vector2(tx2, ty1);
		covertices[2].texcoord=Vector2(tx2, ty2);
		covertices[3].texcoord=Vector2(tx1, ty2);

        /// Define triangles
        /// The values in this table refer to vertices in the above table
        size_t ibufCount = 3*4;
        unsigned short *faces=(unsigned short*)malloc(ibufCount*sizeof(unsigned short));
		faces[0]=0; faces[1]=1; faces[2]=2;
		faces[3]=0; faces[4]=2; faces[5]=3;
		faces[6]=0; faces[7]=2; faces[8]=1;
		faces[9]=0; faces[10]=3; faces[11]=2;

		//set coords
		covertices[0].vertex=Vector3(0,0,0);
		covertices[1].vertex=Vector3(width,0,0);
		covertices[2].vertex=Vector3(width,0,length);
		covertices[3].vertex=Vector3(0,0,length);

		covertices[0].normal=Vector3(0,1,0);
		covertices[1].normal=Vector3(0,1,0);
		covertices[2].normal=Vector3(0,1,0);
		covertices[3].normal=Vector3(0,1,0);

        /// Create vertex data structure for vertices shared between submeshes
        msh->sharedVertexData = new VertexData();
        msh->sharedVertexData->vertexCount = nVertices;

        /// Create declaration (memory format) of vertex data
        VertexDeclaration* decl = msh->sharedVertexData->vertexDeclaration;
        size_t offset = 0;
        decl->addElement(0, offset, VET_FLOAT3, VES_POSITION);
        offset += VertexElement::getTypeSize(VET_FLOAT3);
        decl->addElement(0, offset, VET_FLOAT3, VES_NORMAL);
        offset += VertexElement::getTypeSize(VET_FLOAT3);
//        decl->addElement(0, offset, VET_FLOAT3, VES_DIFFUSE);
//        offset += VertexElement::getTypeSize(VET_FLOAT3);
        decl->addElement(0, offset, VET_FLOAT2, VES_TEXTURE_COORDINATES, 0);
        offset += VertexElement::getTypeSize(VET_FLOAT2);

        /// Allocate vertex buffer of the requested number of vertices (vertexCount)
        /// and bytes per vertex (offset)
        HardwareVertexBufferSharedPtr vbuf =
          HardwareBufferManager::getSingleton().createVertexBuffer(
              offset, msh->sharedVertexData->vertexCount, HardwareBuffer::HBU_DYNAMIC_WRITE_ONLY_DISCARDABLE);

        /// Upload the vertex data to the card
        vbuf->writeData(0, vbuf->getSizeInBytes(), vertices, true);

        /// Set vertex buffer binding so buffer 0 is bound to our vertex buffer
        VertexBufferBinding* bind = msh->sharedVertexData->vertexBufferBinding;
        bind->setBinding(0, vbuf);

		/// Allocate index buffer of the requested number of vertices (ibufCount)
        HardwareIndexBufferSharedPtr faceibuf = HardwareBufferManager::getSingleton().
         createIndexBuffer(
             HardwareIndexBuffer::IT_16BIT,
                ibufCount,
                HardwareBuffer::HBU_STATIC_WRITE_ONLY);

        /// Upload the index data to the card
        faceibuf->writeData(0, faceibuf->getSizeInBytes(), faces, true);

        /// Set parameters of the submesh
        sub->useSharedVertices = true;
        sub->indexData->indexBuffer = faceibuf;
        sub->indexData->indexCount = ibufCount;
        sub->indexData->indexStart = 0;

        /// Set bounding information (for culling)
        msh->_setBounds(AxisAlignedBox(-1,-1,0,1,1,0), true);
        //msh->_setBoundingSphereRadius(Math::Sqrt(1*1+1*1));

        /// Notify Mesh object that it has been loaded
		msh->buildEdgeList();
		msh->load();

		//create the entity and scene node
		char entname[256];
		sprintf(entname, "airbrakenode-%s-%i", basename, num);
		Entity *ec = manager->createEntity(entname, meshname);
		snode=manager->getRootSceneNode()->createChildSceneNode();
		snode->attachObject(ec);

		updatePosition(0.0);
    }

void Airbrake::updatePosition(float amount)
{
	ratio=amount;
	if (!snode) return;
	Vector3 normal=(nodey->smoothpos-noderef->smoothpos).crossProduct(nodex->smoothpos-noderef->smoothpos);
	normal.normalise();
	//position
	Vector3 mposition=noderef->smoothpos+offset.x*(nodex->smoothpos-noderef->smoothpos)+offset.y*(nodey->smoothpos-noderef->smoothpos);
	snode->setPosition(mposition+normal*offset.z);
	//orientation
	Vector3 refx=nodex->smoothpos-noderef->smoothpos;
	refx.normalise();
	Vector3 refy=refx.crossProduct(normal);
	Quaternion orientation=Quaternion(Degree(-ratio*maxangle), (nodex->smoothpos-noderef->smoothpos).normalisedCopy())*Quaternion(refx, normal, refy);
	snode->setOrientation(orientation);
}

void Airbrake::applyForce()
{

			//tropospheric model valid up to 11.000m (33.000ft)
			float altitude=noderef->AbsPosition.y;
			float sea_level_temperature=273.15+15.0; //in Kelvin
			float sea_level_pressure=101325; //in Pa
			float airtemperature=sea_level_temperature-altitude*0.0065; //in Kelvin
			float airpressure=sea_level_pressure*pow(1.0-0.0065*altitude/288.15, 5.24947); //in Pa
			float airdensity=airpressure*0.0000120896;//1.225 at sea level

			Vector3 wind=-noderef->Velocity;
			float wspeed=wind.length();

			Vector3 drag=(1.2*area*sin(fabs(ratio*maxangle/57.3))*0.5*airdensity*wspeed/4.0)*wind;
			noderef->Forces+=drag;
			nodex->Forces+=drag;
			nodey->Forces+=drag;
			nodea->Forces+=drag;

}
