#! /usr/bin/env python
# -*- python coding: utf-8 -*-
# Copyright © 2011,2012 R.F. Smith <rsmith@xs4all.nl>. All rights reserved.
# Time-stamp: <>
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 
# THIS SOFTWARE IS PROVIDED BY AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.

'''Program for converting a view of an STL file into PostScript output.'''

import os
import sys
import time

import stl
import xform

name = "stl2ps [rev. 31a5b12] (2012-01-08)"

def usage():
    print name
    print "Usage: stl2ps infile [outfile] [transform [transform ...]]"
    print "where [transform] is [x number|y number|z number]"

## This is the main program ##
# Process the command-line arguments
validargs = ['x', 'y', 'z', 'X', 'Y', 'Z']
if len(sys.argv) == 1:
    usage()
    sys.exit(0)
infile = sys.argv[1]
if len(sys.argv) < 3 or sys.argv[2] in validargs:
    outfile = None
    del sys.argv[:2]
else:
    outfile = sys.argv[2]
    del sys.argv[:3]
tr = xform.Xform()
while len(sys.argv) > 1:
    if not sys.argv[0] in validargs:
        print "Unknown argument '{}' ignored.".format(sys.argv[0])
        del sys.argv[0]
        continue
    try:
        ang = float(sys.argv[1])
        if sys.argv[0] in ['x','X']:
            tr.rotx(ang)
        elif sys.argv[0] in ['y','Y']:
            tr.roty(ang)
        else:
            tr.rotz(ang)
        del sys.argv[:2]
    except:
        print "Argument '{}' is not a number, ignored.".format(sys.argv[1])
        continue
# Open the file
try:
    stlobj = stl.Surface(infile)
except:
    print "The file '{}' cannot be read or parsed. Exiting.".format(infile)
    sys.exit(1)
# Remove spaces from name
stlobj.name = stlobj.name.strip()
# Apply transformations
if tr.unity == False:
    stlobj.xform(tr)
# Calculate viewport and transformation
xmin, xmax, ymin, ymax, zmin, zmax = stlobj.extents()
pr = xform.Zpar(xmin, xmax, ymin, ymax)
# Prepare output string.
outs = "%!PS-Adobe-1.0\n"
outs += "%%BoundingBox: 0 0 {:.0f} {:.0f}\n".format(pr.w, pr.h)
outs += "% Generated by {} on {}.\n".format(name, time.asctime())
outs += stlobj.stats('% ')+'\n'
s = "% The scale factor used is: {} PostScript points/STL-unit\n"
outs += s.format(pr.s)
s = "% This becomes a picture of {:.0f}×{:.0f} PostScript points;"\
    " {:.0f}×{:.0f} mm.\n"
outs += s.format(pr.w, pr.h, pr.w/72*25.4, pr.h/72*25.4)
# Calculate the visible facets
vizfacets = [f for f in stlobj.facets if pr.visible(f.n)]
outs += "% {} of {} facets are visible.\n".format(len(vizfacets), len(stlobj))
# Next, depth-sort the facets using the smallest distance to the viewer
# of the three vertices.
vizfacets.sort(None, lambda f: max([f.v[0].z, f.v[1].z, f.v[2].z]))
# PostScript settings and macros.
outs += ".5 setlinewidth\n"
outs += "/g {setgray} def\n"
outs += "/f {moveto} def\n"
outs += "/s {lineto} def\n"
outs += "/t {lineto closepath gsave fill grestore stroke} def\n"
# Project and illuminate the facets
pf = [stl.ProjectedFacet(f, pr) for f in vizfacets]
# Draw the triangles
for f in pf:
    s = "{:4.2f} g {:.3f} {:.3f} f {:.3f} {:.3f} s {:.3f} {:.3f} t\n"
    outs += s.format(f.gray, f.x1, f.y1, f.x2, f.y2, f.x3, f.y3)
# Showpage must be the last line in the PostScript output.
outs += "showpage\n"
# Send output.
if outfile == None:
    # derive output filename from input filename.
    outbase = os.path.basename(infile)
    if outbase.endswith((".stl", ".STL")):
        outbase = outbase[:-4]
    outfile = outbase+".ps"
try:
    outf = open(outfile, "w+")
    outf.write(outs)
    outf.close()
except:
    print "Cannot write output file '{}'".format(sys.argv[2])
    sys.exit(2)
