/*
 * $Id: PostGISConnection.java,v 1.1.1.1 2004/01/06 00:13:16 pramsey Exp $
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */
package net.refractions.postgis;

import com.vividsolutions.jts.geom.*;
import com.vividsolutions.jts.io.WKTReader;
import com.vividsolutions.jump.feature.*;
import com.vividsolutions.jump.io.datasource.Connection;
import com.vividsolutions.jump.task.TaskMonitor;

import java.io.Reader;
import java.sql.*;
import java.util.*;

/**
 * This class represents the actual connection of a PostGIS data source. 
 */
public class PostGISConnection implements Connection {
  Map properties;
  String server;
  String port;
  String database;
  String table;
  String username;
  String password;
  String uniqueCol;
  String method;
  
  /**
   * Creates an unconfigured connection.
   */
  public PostGISConnection() {
    this(null);
  }
  
  /**
   * Creates a configured connection. See {@link PostGISDataSource} for 
   * property keys.
   */
  public PostGISConnection(Map properties) {
    this.properties = properties;        
  }
  
  /**
   * Configures the connection. See {@link PostGISDataSource} for property
   * keys.
   */
  public void setProperties(Map properties) {
    this.properties = properties;
  }
  
  /**
   * Executes a query against a connection. The result of the query is returned as 
   * a FeatureCollection.
   * @param query SQL statement.
   */
  public FeatureCollection executeQuery(String query) {
    FeatureDataset dataSet;
    
    readProperties();
    java.sql.Connection conn = connect();
      
    try {
      Statement st = conn.createStatement();
      
      ResultSet rs = st.executeQuery(query + " LIMIT 0");
      ResultSetMetaData meta = rs.getMetaData();
      
      boolean geomCol = false;
      FeatureSchema schema = new FeatureSchema();
      StringBuffer sql = new StringBuffer( "SELECT" );
      int num_cols = meta.getColumnCount();
      for( int col_idx = 1; col_idx <= num_cols; col_idx++ ) {
        String attr_name = meta.getColumnName( col_idx );
        String table_name = meta.getTableName(col_idx);
        AttributeType attr_type = AttributeType.STRING;
      
        // determine the attribute type based on the sql type
        if( meta.getColumnTypeName( col_idx ).equalsIgnoreCase( "geometry") && !geomCol ) {
          // found the first geometry column (any extra geometry columns are treated as strings)
          geomCol = true;
          attr_type = AttributeType.GEOMETRY;
          sql.append( " asText(" + attr_name + ") AS " + attr_name + "," );
        } 
        else {
          int sql_type = meta.getColumnType( col_idx );
          switch( sql_type ) {
            case Types.BIGINT:
            case Types.INTEGER:
            case Types.SMALLINT:
            case Types.TINYINT:
              attr_type = AttributeType.INTEGER;
              sql.append( " " + attr_name + "," );
              break;
            case Types.DOUBLE:
            case Types.FLOAT:
            case Types.REAL:
              attr_type = AttributeType.DOUBLE;
              sql.append( " " + attr_name + "," );
              break;
            default:
              attr_type = AttributeType.STRING;
              sql.append( " " + attr_name + "," );
          }
        }
        schema.addAttribute( attr_name, attr_type );
      }
      rs.close();
      
      if (!geomCol) {
        st.close();
        conn.close();
        throw new IllegalStateException("The table you have selected does not contain any geometric data.");  
      }
      
      sql.deleteCharAt( sql.lastIndexOf( "," ) );
      sql.append( " FROM " + table);
      
      st.execute( "BEGIN" );
      String s = "DECLARE my_cursor CURSOR FOR " + sql.toString();
      
      st.execute( "DECLARE my_cursor CURSOR FOR " + sql.toString() );
      rs = st.executeQuery( "FETCH FORWARD ALL IN my_cursor" );
      
      dataSet = new FeatureDataset( schema ) ;
      GeometryFactory factory = new GeometryFactory( new PrecisionModel(), 0 );
      WKTReader wktReader = new WKTReader( factory );
      while( rs.next() ) {
        Feature f = new BasicFeature( schema );
        for( int attr_idx = 0; attr_idx < schema.getAttributeCount(); attr_idx++ ) {
          AttributeType attr_type = schema.getAttributeType( attr_idx );
          if( attr_type.equals( AttributeType.GEOMETRY ) ) {
            Reader wkt = rs.getCharacterStream( schema.getAttributeName( attr_idx ) );
            Geometry geom;
            if( wkt == null ) {
              geom = new GeometryCollection( null, factory);
            } 
            else {
              geom = wktReader.read( wkt );
            } 
            f.setAttribute( attr_idx, geom );
          } 
          else if( attr_type.equals( AttributeType.INTEGER ) ) {
            f.setAttribute( attr_idx, new Integer( rs.getInt( schema.getAttributeName( attr_idx ) ) ) );
          } 
          else if( attr_type.equals( AttributeType.DOUBLE ) ) {
            f.setAttribute( attr_idx, new Double( rs.getDouble( schema.getAttributeName( attr_idx ) ) ) );
          } 
          else if( attr_type.equals( AttributeType.STRING ) ) {
            f.setAttribute( attr_idx, rs.getString( schema.getAttributeName( attr_idx ) ) );
          } 
        }
        dataSet.add( f );
      }
      
      st.execute( "CLOSE my_cursor" );
      st.execute( "END" );
      st.close();
      conn.close();
    }
    catch(Exception e) {
      if (PostGISPlugIn.DEBUG) e.printStackTrace();
      throw new IllegalStateException(e.getMessage());
    }
    
    return(dataSet);
  }
  
  /**
   * Since any exceptions will cause the query to fail, this function 
   * simply calls executeQuery(String query) and does not ignore any 
   * exceptions.
   */
  public FeatureCollection executeQuery(String query, List exceptions) {
    return(executeQuery(query));
  }
  
  /**
   * Executes an update against the connection.
   * @param query This parameter is ignored, the connection determines everything 
   * it needs to know from setProperties(HashMap properties) and collection.
   * @param collection The updated features.
   */
  public void executeUpdate(String query, FeatureCollection collection) {
    
    if (collection.isEmpty()) 
      throw new IllegalStateException("No data to write, empty Feature Collection");
    
    int SRID = ((Feature)collection.iterator().next()).getGeometry().getSRID();
      
    // get the feature schema
    FeatureSchema schema = collection.getFeatureSchema();
    HashSet schemaCols = new HashSet( schema.getAttributeCount() );
    for( int i = 0; i < schema.getAttributeCount(); i++ ) {
      String name = schema.getAttributeName( i );
      if( schemaCols.contains( name ) ) {
        throw new UnsupportedOperationException( "The FeatureSchema contains duplicate attribute names; you must remove duplicate attribute names before saving to a PostGIS table." );
      }
      schemaCols.add( name );
    }
    
    java.sql.Connection conn = null;
    Statement stmt = null;
    
    HashSet cols = null;
    StringBuffer sqlBuf = null;
    String sql = null;
    
    //connect
    readProperties();  
    conn = connect();
    
    // determine if the table already exists or not
    boolean table_exists = false;
    try {
      stmt = conn.createStatement();
      ResultSet rs = stmt.executeQuery( "SELECT * FROM " + table + " LIMIT 0" );
      ResultSetMetaData meta = rs.getMetaData();
      int num_cols = meta.getColumnCount();
      
      cols = new HashSet( num_cols );
      for( int col_idx = 1; col_idx <= num_cols; col_idx++ ) {
        cols.add( meta.getColumnName( col_idx ) );
      }
      rs.close();
      table_exists = true;
    } 
    catch( SQLException sqle ) {
      table_exists = false;
      if( method == PostGISDataSource.SAVE_METHOD_UPDATE ) {
        throw new IllegalStateException( "Save method is set to UPDATE and table does not exist; cannot update a non-existant table" );
      }       
    }
    
    // begin the transaction
    try {
      conn.setAutoCommit( false ); // don't know why transactions don't work...
      String geomCol = null;
      if( !table_exists ) { 
        // build the create table statement
        sqlBuf = new StringBuffer( "CREATE TABLE " + table + " (" );
        int num_attrs = schema.getAttributeCount();
        cols = new HashSet( num_attrs );
        for( int i = 0; i < num_attrs; i++ ) {
          String name = schema.getAttributeName( i );
          cols.add( name );
          AttributeType type = schema.getAttributeType( i );
          if( type.equals( AttributeType.INTEGER ) ) {
            sqlBuf.append( " " + name + " INT," );
          } 
          else if( type.equals( AttributeType.DOUBLE ) ) {
            sqlBuf.append( " " + " DOUBLE," );
          } 
          else if( type.equals( AttributeType.GEOMETRY ) ) {
            // do not add to the query string
            if( geomCol == null ) {
              // save the name of the geometry column for later
              geomCol = name;
            } 
          } 
          else {
            sqlBuf.append( " " + name + " VARCHAR(255)," );
          }
        }
        sqlBuf.deleteCharAt( sqlBuf.lastIndexOf( "," ) );
        sqlBuf.append( " ) " );
        sql = sqlBuf.toString();
      
        // create the table
        try {
          stmt.executeUpdate( sql );
        } 
        catch( SQLException sqle ) {
          if (PostGISPlugIn.DEBUG) { 
            System.out.println(sql);
            sqle.printStackTrace();
          }
          throw new Exception( "Create table statement failed: " + sqle.toString() + "\n" + sql );
        }
      
        // add the geometry column
        sql = "SELECT AddGeometryColumn( '" + database + "', '" + table + "', '" + geomCol + "', " + SRID + ", 'GEOMETRY', 2 )";
        
        try {
          stmt.execute( sql );
        } 
        catch( SQLException sqle ) {
          if (PostGISPlugIn.DEBUG) { 
            System.out.println(sql);
            sqle.printStackTrace();
          } 
          throw new Exception( "AddGeometryColumn statement failed: " + sqle.toString() + "\n" + sql );
        }
      }
    
      if( method == PostGISDataSource.SAVE_METHOD_UPDATE ) {
        // make sure the table has the unique column in it to do updates on
        if( !cols.contains( uniqueCol ) ) {
          throw new Exception( "The table " + table + " doesn't contain the column " + uniqueCol + ", required in order to do updates." );
        }
        String uniqueVal;
        String val;
        
        Iterator it = collection.iterator();
        while( it.hasNext() ) {
          uniqueVal = null;
          Feature f = (Feature) it.next();
          sqlBuf = new StringBuffer( "UPDATE " + table + " SET " );
          for( int i = 0; i < schema.getAttributeCount(); i++ ) {
            String name = schema.getAttributeName( i );
            if( cols.contains( name ) ) {
              AttributeType type = schema.getAttributeType( i );
              if( type.equals( AttributeType.INTEGER ) ) {
                val = "" + f.getInteger( i );
              } 
              else if( type.equals( AttributeType.DOUBLE ) ) {
                val = "" + f.getDouble( i );
              } 
              else if( type.equals( AttributeType.GEOMETRY ) ) {
                val = "GeometryFromText( '" + f.getGeometry().toText() + "', " + f.getGeometry().getSRID()+ ")";
              } 
              else {
                val = "'" + f.getString( i ) + "'";
              }
              if( name.equals( uniqueCol ) ) {
                uniqueVal = val;
              } 
              else {
                sqlBuf.append( " " + name + " = " + val + "," );
              }
            }
          }
          sqlBuf.deleteCharAt( sqlBuf.lastIndexOf( "," ) );
          sqlBuf.append( " WHERE " + uniqueCol + " = " + uniqueVal );
          sql = sqlBuf.toString();
          try {
            stmt.executeUpdate( sql );
          } 
          catch( SQLException sqle ) {
            throw new Exception( "Update statement failed: " + sqle.toString() + "\n" + sql );
          }
        } // end while(featureIterator)
      } 
      else {
        // build the initial part of the insert statement
        sqlBuf = new StringBuffer( "INSERT INTO " + table + " (" );
        for( int i = 0; i < schema.getAttributeCount(); i++ ) {
          String name = schema.getAttributeName( i );
          if( cols.contains( name ) ) {
            sqlBuf.append( " " + name + "," );
          }
        }
        sqlBuf.deleteCharAt( sqlBuf.lastIndexOf( "," ) );
        sqlBuf.append( " ) VALUES (" );
        String insertQueryHead = sqlBuf.toString();
      
        Iterator it = collection.iterator();
        while( it.hasNext() ) {
          Feature f = (Feature) it.next();
          sqlBuf = new StringBuffer( insertQueryHead );
          for( int i = 0; i < schema.getAttributeCount(); i++ ) {
            String name = schema.getAttributeName( i );
            if( cols.contains( name ) ) {
              AttributeType type = schema.getAttributeType( i );
              if( type.equals( AttributeType.INTEGER ) ) {
                sqlBuf.append( " " + f.getInteger( i ) + "," );
              } 
              else if( type.equals( AttributeType.DOUBLE ) ) {
                sqlBuf.append( " " + f.getDouble( i ) + "," );
              } 
              else if( type.equals( AttributeType.GEOMETRY ) ) {
                sqlBuf.append( " GeometryFromText( '" + f.getGeometry().toText() + "', " + f.getGeometry().getSRID() + ")," );
              } 
              else {
                sqlBuf.append( " '" + f.getString( i ) + "'," );
              }
            }
          }
          sqlBuf.deleteCharAt( sqlBuf.lastIndexOf( "," ) );
          sqlBuf.append( " ) " );
          sql = sqlBuf.toString();
          try {
            stmt.executeUpdate( sql );
          } catch( SQLException sqle ) {
            throw new Exception( "Insert statement failed: " + sqle.toString() + "\n" + sql );
          }
        } // end while(featureIterator)
      } // end if(method)
      
      // end the transaction
      conn.commit(); // don't know why transactions don't work...
      
      stmt.close();
      conn.close();        
    } 
    catch( Exception e ) {
      try {
        conn.rollback();
        stmt.close();
        conn.close();  
      }
      catch(SQLException sqle) {
        if (PostGISPlugIn.DEBUG) sqle.printStackTrace();     
      }
      
      if (PostGISPlugIn.DEBUG) e.printStackTrace();
      throw new IllegalStateException(e.getMessage());
    }
  }

  /*
   * Reads the query + connection properties into global variables.
   */
  private void readProperties() {
    server = (String)properties.get( PostGISDataSource.SERVER_KEY );
    port = (String)properties.get( PostGISDataSource.PORT_KEY );
    database = (String)properties.get( PostGISDataSource.DATABASE_KEY );
    table = (String)properties.get( PostGISDataSource.TABLE_KEY );
    username = (String)properties.get( PostGISDataSource.USERNAME_KEY );
    password = (String)properties.get( PostGISDataSource.PASSWORD_KEY );
    uniqueCol = (String)properties.get( PostGISDataSource.UNIQUE_COLUMN_KEY );
    method = (String)properties.get( PostGISDataSource.SAVE_METHOD_KEY );
  }
  
  /*
   * Opens a connection to a PostgresSQL database.
   */
  private java.sql.Connection connect() {
    try {
      String jdbcUrl = "jdbc:postgresql://" + server + ":" + port + "/" + database;
      Class.forName( "org.postgresql.Driver" );
      return(DriverManager.getConnection( jdbcUrl, username, password ));
    }
    catch(ClassNotFoundException cnfe) {
      if (PostGISPlugIn.DEBUG) cnfe.printStackTrace();
      throw new IllegalStateException("Could not load PostgresSQL driver: " + cnfe.getMessage());  
    }
    
    catch(SQLException sqle) {
      if (PostGISPlugIn.DEBUG) sqle.printStackTrace();
      throw new IllegalStateException("Could not connection to database: " + sqle.getMessage());
    } 
  }
  
  /**
   * @see Connection#close()
   */
  public void close() {}

/* (non-Javadoc)
 * @see com.vividsolutions.jump.io.datasource.Connection#executeQuery(java.lang.String, java.util.Collection, com.vividsolutions.jump.task.TaskMonitor)
 */
public FeatureCollection executeQuery(String query, Collection exceptions, TaskMonitor monitor) {
	// TODO Auto-generated method stub (temporary implementation [brent owens])
	return executeQuery(query);
}

/* (non-Javadoc)
 * @see com.vividsolutions.jump.io.datasource.Connection#executeQuery(java.lang.String, com.vividsolutions.jump.task.TaskMonitor)
 */
public FeatureCollection executeQuery(String query, TaskMonitor monitor) throws Exception {
	// TODO Auto-generated method stub (temporary implementation [brent owens])
	return executeQuery(query);
}

/* (non-Javadoc)
 * @see com.vividsolutions.jump.io.datasource.Connection#executeUpdate(java.lang.String, com.vividsolutions.jump.feature.FeatureCollection, com.vividsolutions.jump.task.TaskMonitor)
 */
public void executeUpdate(String query, FeatureCollection featureCollection, TaskMonitor monitor) throws Exception {
	// TODO Auto-generated method stub (temporary implementation [brent owens])
	executeUpdate(query, featureCollection);
}
  
  
  
}