
// 2004-11-05 [15:35]
//  *. Preamable 

// 2004-11-05 [14:48]
//  **. Abbreviations 

/*

TV :: a set of Truth Values within the system
D :: a set of Designation values (values for which the system yields true)
C :: a set of Connective symbols
f_C :: a set of function symbols for each element in C

*/

/**
Filename: LogicalSystems.java
Created on: 2004-11-06
@author T. Hoshi
Last modified: 2004-11-14 [05:54]
Description
*/

import java.util.*;


public class LogicalSystems {

// 2004-11-07 [19:47]
//  **. Fields 

	/** logical systems being implemented so far. */
	Hashtable logical_systems_table;

	public Hashtable getLogicalSystemsTable () {
		return logical_systems_table;
	}


// 2004-11-07 [20:21]
//  **. Constructor 

	public LogicalSystems() {
		logical_systems_table = new Hashtable();
		logical_systems_table.put( "CPL" , new SystemCPL( ) );
		logical_systems_table.put( "K3" , new SystemK3( ) );
		logical_systems_table.put( "L3" , new SystemL3( ) );
		logical_systems_table.put( "L" , new SystemL( ) );
		logical_systems_table.put( "LAleph" , new SystemLAleph( ) );
		logical_systems_table.put( "LP" , new SystemLP( ) );
		logical_systems_table.put( "RM3" , new SystemRM3( ) );
		logical_systems_table.put( "FDE" , new SystemFDE( ) );
	}


// 2004-11-05 [15:44]
//  *. CPL - Classical two-valued Propositional Logic

/*
  TV := { 0, 1 } 
  D := { 1 }
  C := { NOT, AND, OR, IMPLIES, IFF }
  or { , , , ,  }
  f_C := { f_NOT, f_AND, f_OR, f_IMPLIES, f_IFF }
  or { f_, f_, f_, f_, f_ }

*/


/**
 * Last modified: 2004-11-14 [05:54]
 *
 * @author Takayuki Hoshi <hoshi103@chapman.edu>
 * File: SystemCPL.java
 * 
 *
 */
 
public class SystemCPL implements LogicalSystem {
	
	final String system_name = "CPL" ;

	/** Designated values -- Ones which are interpreted as TRUE within the system.  */
	int[] D = { 1 } ;

	/** Whether truth table is constructable within the system or not. Used to
	 * determine if we could use :table, :satisfiable, or :valid command. If false, those commands are disabled. */
	public boolean truthTableConstructable( ) {
		return true;
	}

	/** returns the number of truth values within the system. If the number is infinity, then it will return -1. */
	public int getNTruthValues( ) {
		return 2;
	}

	public String getSystemName() {
		return system_name;
	}


/*
 * See how the element is interpreted in terms of being either true or false, and
 * nothing else.
 */
	public boolean trueOrFalse( Object o ) {
		int  a  = ((Integer) o).intValue();
		for( int i = 0; i < D.length; i++ ) {
			if( a == D[i] ) 
				return true;
		}
		return false; 
	}

/*
 * Translates a truth value to english
 */
	public String truthValueToEnglish( Object o ) {
		int  a  = ((Integer) o).intValue();

		if (a == 0) {
			return "0 ; False" ;
		}
		else {
			return "1 ; True" ;
		}
	}
		
/*
 * [ ~x ] For any x, return ~x
 */
	public Object interpretNot( Object o1 ) {
		int  x  = ((Integer) o1).intValue();

		if ( x == 0 ) return new Integer( 1 );
		else return new Integer( 0 );
	}

/*
 * [ x & y ] Return 1 iff  x != y != 0 (normally if x = y = 1) .
 */
	
	public Object interpretAnd( Object o1, Object o2 ) {
		int  x  = ((Integer) o1).intValue();
		int  y  = ((Integer) o2).intValue();

		if ( x != 0 && y != 0 ) return new Integer( 1 );
		else return new Integer( 0 );
	}


/*
 * [ x | y ] Return 1 iff x = y = 0; otherwise, return 1
 */
	public Object interpretOr( Object o1, Object o2 ) {
		int  x  = ((Integer) o1).intValue();
		int  y  = ((Integer) o2).intValue();

		if ( x == 0 && y == 0 ) return new Integer( 0 );
		else return new Integer( 1 );
	}

/*
 * [ x => y ] Return 0 iff x != 0 and y = 1; return 1 o.w. I might add that,
 * since this is the system of CPL, the implication is material.
 */
	public Object interpretImplies( Object o1, Object o2 ) {
		int  x  = ((Integer) o1).intValue();
		int  y  = ((Integer) o2).intValue();

		if ( x != 0 && y == 0 ) return new Integer( 0 );
		else return new Integer( 1 );
	}

/*
 * [ x <=> y ] Return 0 iff (x => y) & (y => x) yields true; 1 o.w.
 */
	public Object interpretIff( Object x, Object y ) {

		return interpretAnd ( 
			interpretImplies( x, y ), 
			interpretImplies( y, x ) );
	}

	public Object getCorrespondingNumberFormat( String number_string ) {

		/* in case of user passing an double value, we want to safely convert String to Double and then Double to int.    */
		Double d = new Double( number_string );
		int i = d.intValue();		// numbers after the floating point will be truncated. eg, 1.3333 ---> 1

		return new Integer( i );
	}

	public Object getTrueValue( ) {
		return new Integer( 1 );
	}
		
	public Object getFalseValue( ) {
		return new Integer( 0 );
	}


}		// end of class `SystemCPL'


// 2004-11-05 [16:05]
//  *. K3 - Classical 3-valued Propositional Logic 

/**
 * Last modified: 2004-11-14 [05:54]
 *
 * @author Takayuki Hoshi <hoshi103@chapman.edu>
 * File: SystemK3.java
 * 
 Classical 3-valued propositional logic known as `K_3'

 * TV = { 0, 1, u } = { 0, 1, 2 }		// 2 means `undetermined'
 * D = { 1 }
 * C = { ~, &, |, =>, <=> }
 * 
 */

/* Note that in this class, we are assuming that the truth value passed to each
 * function is necessarily either 0, 1 or 2.  */
	
public class SystemK3 implements LogicalSystem{
	/** Designated values -- Ones which are interpreted as TRUE within the system.  */
	int[] D = { 1 } ;

	final String system_name = "K3" ;

	/** Whether truth table is constructable within the system or not. Used to
	 * determine if we could use :table, :satisfiable, or :valid command. If false, those commands are disabled. */
	public boolean truthTableConstructable( ) {
		return true;
	}

	public String getSystemName() {
		return system_name;
	}


	/** returns the number of truth values within the system. If the number is infinity, then it will return -1. */
	public int getNTruthValues( ) {
		return 3;
	}

/*
 * See how the element is interpreted in terms of being either true or false, and
 * nothing else.
 */
	public boolean trueOrFalse( Object o ) {
		int  a  = ((Integer) o).intValue();
		for( int i = 0; i < D.length; i++ ) {
			if( a == D[i] ) 
				return true;
		}
		return false; 
	}


/*
 * Translates a truth value to english
 */
	public String truthValueToEnglish( Object o ) {
		int  a  = ((Integer) o).intValue();
		if (a == 0) {
			return "0 ; False" ;
		}
		else if ( a == 1 ) {
			return "1 ; True" ;
		}
		else 
			return "2 ; Unknown" ;
	}
		
/*
 * [ ~x ] 
 */
	public Object interpretNot( Object o1 ) {
		int  x  = ((Integer) o1).intValue();
		if ( x == 1 ) return new Integer( 0 );
		else if ( x == 0 ) return new Integer( 1 );
		else return new Integer( 2 );
	}
		
/*
 * [ x & y ] 
 */
	public Object interpretAnd( Object o1, Object o2 ) {
		int  x  = ((Integer) o1).intValue();
		int  y  = ((Integer) o2).intValue();

		if ( x == 1 &&  y == 1 ) return new Integer( 1 );
		else if ( x == 0 || y == 0) return new Integer( 0 );
		else return new Integer( 2 );
	}
		
		
/*
 * [ x | y ] 
 */
	public Object interpretOr( Object o1, Object o2 ) {
		int  x  = ((Integer) o1).intValue();
		int  y  = ((Integer) o2).intValue();

		if ( x == 1 || y == 1) return new Integer( 1 );
		else if ( x == 0 &&  y == 0 ) return new Integer( 0 );
		else return new Integer( 2 );
	}
		
/*
 * [ x => y ] 
 */
	public Object interpretImplies( Object o1, Object o2 ) {
		int  x  = ((Integer) o1).intValue();
		int  y  = ((Integer) o2).intValue();

		if ( x == 0 || y == 1) return new Integer( 1 );
		else if ( x == 1 && y == 0 ) return new Integer( 0 );
		else return new Integer( 2 );
	}
		
/*
 * [ x <=> y ] 
 */
	public Object interpretIff( Object x, Object y ) {

		return interpretAnd ( 
			interpretImplies( x, y ), 
			interpretImplies( y, x ) );
	}
		
	public Object getCorrespondingNumberFormat( String number_string ) {

		/* in case of user passing an double value, we want to safely convert String to Double and then Double to int.    */
		Double d = new Double( number_string );
		int i = d.intValue();		// numbers after the floating point will be truncated. eg, 1.3333 ---> 1

		return new Integer( i );
	}

	public Object getTrueValue( ) {
		return new Integer( 1 );
	}
		
	public Object getFalseValue( ) {
		return new Integer( 0 );
	}

}		// end of class `SystemK3'



//  *. L3 - Lukasiewicz 3-valued Logic 
															// 2004-11-05 [16:08]

/**
 * Last modified: 2004-11-14 [05:54]
 *
 * @author Takayuki Hoshi <hoshi103@chapman.edu>
 * File: SystemL3.java
 * 
 Lukasiewicz 3-valued logic, known as `L_3' 

 * This system of logic is essentially the same as K_3 except that L_3 modifies
 * the behavior of => (Implication) truth functions so that the inference of
 * Modus Ponens is valid in the system.
 */
 
/* Note that in this class, we are assuming that the truth value passed to each
 * function is necessarily either 0, 1 or 2.  */
public class SystemL3 extends SystemK3 implements LogicalSystem{

	final String system_name = "L3" ;


	public String getSystemName() {
		return system_name;
	}


	public Object interpretImplies( Object o1, Object o2 ) {
		int  x  = ((Integer) o1).intValue();
		int  y  = ((Integer) o2).intValue();

		if ( x == 0 || y == 1) return new Integer( 1 );
		else if ( x == 1 && y == 0 ) return new Integer( 0 );
		else if ( x == 2 && y == 2) return new Integer( 1 );		// !! this is the place added in addition. (from K3)
		else return new Integer( 2 );
	}
}		// end of class `SystemL3'

// 2004-11-05 [16:11]
//  *. LP - Logic of Paradox 

/**
 * Last modified: 2004-11-14 [05:54]
 *
 * @author Takayuki Hoshi <hoshi103@chapman.edu>
 * File: SystemLP.java
 * 
 System: Logic of Paradox ( LP)

 * TV = { 0, u, 1 } = { 0, 2, 1}
 * D = { 1, 2 }
 * C = { ~, &, |, =>, <=> }
 * 
 Exactly the same as K3 except that D := { 1, 2 } ( or {1, u] ).

 In this system, it has been suggested to translate `2' as "both true and
 false," 1 as "true and true only," and 0 as "false and false only," all of which we will 
 employ in truthValueToEnglish().

*/
 
public class SystemLP extends SystemK3 implements LogicalSystem{
	/** Designated values -- Ones which are interpreted as TRUE within the system.  */
	int[] D = { 1, 2 } ;

	final String system_name = "LP" ;


	public String getSystemName() {
		return system_name;
	}

/*
 * See how the element is interpreted in terms of being either true or false, and
 * nothing else.
 */
	public boolean trueOrFalse( Object o ) {
		int  a  = ((Integer) o).intValue();

		for( int i = 0; i < D.length; i++ ) {
			if( a == D[i] ) 
				return true;
		}
		return false; 
	}

	public String truthValueToEnglish( Object o ) {
		int  a  = ((Integer) o).intValue();
		if (a == 0) {
			return "0 ; False and false only" ;
		}
		else if ( a == 1 ) {
			return "1 ; True and true only" ;
		}
		else		// !! this is the change (from K3)
			return "2 ; Both true and false" ;
	}
}		// end of class `SystemLP'

// 2004-11-05 [16:12]
//  *. RM3 - Slight modification of LP 

/**
 * Last modified: 2004-11-14 [05:54]
 *
 * @author Takayuki Hoshi <hoshi103@chapman.edu>
 * File: SystemRM3.java
 * 
 System: RM3

 This is exactly the same as LP except for => and <=> so that MP is valid in RM3
 * 
 */
 
public class SystemRM3 extends SystemLP implements LogicalSystem{

	final String system_name = "RM3" ;


	public String getSystemName() {
		return system_name;
	}


	public Object interpretImplies( Object o1, Object o2 ) {
		int  x  = ((Integer) o1).intValue();
		int  y  = ((Integer) o2).intValue();

		if ( x == 0 || y == 1) return new Integer( 1 );
		else if ( x == 2 && y == 2) return new Integer( 2 );
		else return new Integer( 0 );
	}
}		// end of class `SystemRM3'


// 2004-11-05 [16:07]
//  *. L - Fuzzy Logic 

/**
 * Last modified: 2004-11-14 [05:54]
 *
 * @author Takayuki Hoshi <hoshi103@chapman.edu>
 * File: SystemL.java
 * 
 System of fuzzy logic called L.
 TV := { x in R | 0 <= x <= 1 }

 *
 */
 
public class SystemL implements LogicalSystem{

	/** a real number s.t. a sentence having the truth value equal to or above
	 * yeilds TRUE within the system. */
	private double border_line_lambda = 0.5;

	double completely_false = 0.0;
// 	private double half_true = 0.5;
	double completely_true = 1.0;
	
	final String system_name = "L" ;


	public void setBorderLine( double border_line_lambda ) {
		this.border_line_lambda = border_line_lambda;
	}

	/** Whether truth table is constructable within the system or not. Used to
	 * determine if we could use :table, :satisfiable, or :valid command. If false, those commands are disabled. */
	public boolean truthTableConstructable( ) {
		return false;
	}

	public String getSystemName() {
		return system_name;
	}

	public SystemL () {}

	public SystemL ( double border_line_lambda ) {
		this.border_line_lambda = border_line_lambda;
	}

	public SystemL ( Object border_line_object ) {
		double d = ((Double) border_line_object).doubleValue();
		this.border_line_lambda = d;
	}


	/** returns the number of truth values within the system. If the number is infinity, then it will return -1. */
	public int getNTruthValues( ) {
		return -1;
	}

/*
 * See how the element is interpreted in terms of being either true or false, and
 * nothing else.
 */
	public boolean trueOrFalse( Object o ) {
		double  a  = ((Double) o).doubleValue();

		if( a >= border_line_lambda )
			return true;
		else return false;
	}

/*
 * Translates a truth value to english
 */
	public String truthValueToEnglish( Object o ) {
		double  a  = ((Double) o).doubleValue();

		if ( a == completely_true ) { return "1.0 ; Completely True"; }
		else if (a >= border_line_lambda) { return "1.0 > value >= " + border_line_lambda + " ; True"  ; }
		else if ( a == completely_false ) { return "0.0 ; Completely False" ; }
		else { return border_line_lambda + " > value > 0.0 ; False"  ; }
	}
		
/*
 * [ ~x ] 
 */
// 	public Object interpretNot( Object o ) throws FuzzyLogicNumberOutOfRangeException {
	public Object interpretNot( Object o )  {
		double  x  = ((Double) o).doubleValue();

		if ( (x >= 0.0) && (x <= 1.0) ) return new Double( 1.0 - x );
//  		else throw new FuzzyLogicNumberOutOfRangeException("Number must be the real number within [0, 1.0].");
 		else {
			System.out.println( "WARNING: Value taken by a sentence symbol within this system must be a real number between 0 and 1.0, inclusive (i.e., [0.0, 1.0]).  "  );
			System.out.println( "Returning 0.0 instead..."  );
			return new Double( 0.0 );
		}
	}
		
/*
 * [ x & y ] 
 */
	public Object interpretAnd( Object o1, Object o2 ) {
		double  x  = ((Double) o1).doubleValue();
		double  y  = ((Double) o2).doubleValue();

		return new Double( Math.min(x, y) );
	}
		
/*
 * [ x | y ] 
 */
	public Object interpretOr( Object o1, Object o2 ) {
		double  x  = ((Double) o1).doubleValue();
		double  y  = ((Double) o2).doubleValue();

		return new Double( Math.max(x, y) );
	}
		
/*
 * [ x => y ] 
 */
// 	public Object interpretImplies( Object o1, Object o2 ) throws FuzzyLogicNumberOutOfRangeException {
	public Object interpretImplies( Object o1, Object o2 )  {
		double  x  = ((Double) o1).doubleValue();
		double  y  = ((Double) o2).doubleValue();

		if ( (x >= 0.0) && (x <= 1.0) && (y >= 0.0) && (y <= 1.0)) {
			if ( x <= y ) return new Double( 1.0 );
			else return new Double( 1.0 - ( x - y ) );
		}
// 		else throw new FuzzyLogicNumberOutOfRangeException("Number must be the real number within [0, 1.0].");
 		else {
			System.out.println( "WARNING: Value taken by a sentence symbol within this system must be a real number between 0 and 1.0, inclusive (i.e., [0.0, 1.0]).  "  );
			System.out.println( "Returning 0.0 instead..."  );
			return new Double( 0.0 );
		}
	}
		
/*
 * [ x <=> y ] 
 */
// 	public Object interpretIff( Object x, Object y )  throws FuzzyLogicNumberOutOfRangeException{
	public Object interpretIff( Object x, Object y )  {
		return interpretAnd( interpretImplies( x, y ), interpretImplies( y, x ) );
	}
		

	public Object getCorrespondingNumberFormat( String number_string ) {
		Double d = new Double( number_string );
		return d;
	}

	public Object getTrueValue( ) {
		return new Double( 1.0 );
	}
		
	public Object getFalseValue( ) {
		return new Double( 0.0 );
	}


}		// end of class `SystemL'

// 2004-11-05 [16:09]
//  *. LAleph - variation of fuzzy logic 


/**
 * Last modified: 2004-11-14 [05:54]
 *
 * @author Takayuki Hoshi <hoshi103@chapman.edu>
 * File: SystemLAleph.java
 * 
 System of fuzzy logic called L_aleph

 This is the same as system L except that the destination value
 D = {1}. 
*/
 
public class SystemLAleph extends SystemL implements LogicalSystem {

	final String system_name = "LAleph" ;

// 	public void setBorderLine( double border_line_lambda ) {
// 		this.border_line_lambda = border_line_lambda;
// 	}


	public String getSystemName() {
		return system_name;
	}

/*
 * See how the element is interpreted in terms of being either true or false, and
 * nothing else.
 */
	public boolean trueOrFalse( Object o ) {
		double  a  = ((Double) o).doubleValue();

		if( a == 1.0 )
			return true;
		else return false;
	}


	public String truthValueToEnglish( Object o ) {
		double  a  = ((Double) o).doubleValue();

		if ( a == completely_true ) { return "1.0 ; Completely True"; }
		else if ( a == completely_false ) { return "0.0 ; Completely False" ; }
		else { return "1.0 > answer > 0.0 ; False" ; }
	}
}		// end of class `SystemLAleph'

// 2004-11-05 [20:36]
//  *. FDE - First Degree Entailment (as a 4-valued logic)

/*

 TV := { 0, 1, 2, 3 }		
 0 means false and not true; 1 true and not false; 
 2 (or `b') both true and false; and 3 (or `n') neither true nor false.

 D := { 1, 2 }
 C := { NOT, AND, OR }

*/

public class SystemFDE implements LogicalSystem{
	private final int b = 2;		// both
	private final int n = 3;
	private BinaryTwoValuedRelation R;
	/** Designated values -- Ones which are interpreted as TRUE within the system.  */
	int[] D = { 1, 2 } ;

	final String system_name = "FDE" ;

	/** Whether truth table is constructable within the system or not. Used to
	 * determine if we could use :table, :satisfiable, or :valid command. If false, those commands are disabled. */
	public boolean truthTableConstructable( ) {
		return true;
	}

	public String getSystemName() {
		return system_name;
	}

	public SystemFDE(BinaryTwoValuedRelation relation) {
		this.R = relation;
	}
	
	public SystemFDE() {
		this.R = new BinaryTwoValuedRelation();
	}

	public void setRelation(BinaryTwoValuedRelation R) {
		this.R = R;
	}

	/** returns the number of truth values within the system. If the number is infinity, then it will return -1. */
	public int getNTruthValues( ) {
		return 4;
	}


/*
 * See how the element is interpreted in terms of either being true or false, and
 * nothing else.
 */
	public boolean trueOrFalse( Object o ) {
		// true iff 

		int  a  = ((Integer) o).intValue();
		for( int i = 0; i < D.length; i++ ) {
			if( a == D[i] ) 
				return true;
		}
		return false; 
	}

/*
 * Translates a truth value to english
 */
	public String truthValueToEnglish( Object o ) {
		int  a  = ((Integer) o).intValue();
		if (a == 0) { return "0 ; False and not True" ; }
		else if (a == 1) { return "1 ; True and not False" ; }
		else if (a == b) { return "2 ; Both True and False" ; }
		else if (a == n) { return "3 ; Neither True nor False" ; }
		else { return "Invalid, unrecognized value (Value should be either 0, 1, 2, or 3.)"; }
	}
	
/*
 * Given a symbol string `o', return Integer object representing the truth value of `o' 
 */
	public Object interpretSymbol( Object o ) {
		String symbol = o.toString();

		if ( R.contains( symbol, "1" ) && R.contains( symbol, "0") )
			return new Integer( b );
		else if( (! R.contains( symbol, "1")) && (! R.contains( symbol, "0")))
			return new Integer( n );
		else if ( R.contains( symbol, "1" ))
			return new Integer( "1" );
		else return new Integer( "0" );
	}

/*
 * [ ~x ] `~x R 1' is 1 (or true) iff `x R 0' ; `~x R 0' is 1 (or true) iff `x R 1' 
 */
	public Object interpretNot(Object o ) {		// `o' should be an integer object which has been interpreted already.

		int value = ((Integer) o).intValue();
		if( value == 1 ) { return new Integer( 0 ); }
		else if( value == 0 ) { return new Integer( 1 ); }
		else { return new Integer( value ); }
	}

// /*
//  * [ ~x ] `~x R 1' is 1 (or true) iff `x R 0' ; `~x R 0' is 1 (or true) iff `x R 1' 
//  */
// 	public Object interpretNot(Object o ) {
// 		String symbol = o.toString();

// 		int value = ((Integer) interpretSymbol( symbol )).intValue();
// 		if( value == 1 ) { return new Integer( 0 ); }
// 		else if( value == 0 ) { return new Integer( 1 ); }
// 		else { return new Integer( value ); }
// 	}

/*
 * [ x & y ] 
 */
	public Object interpretAnd( Object value_x, Object value_y ) {		//  value_x and value_y should be an integer object which has been interpreted already.
		int x = ((Integer) value_x).intValue();
		int y = ((Integer) value_y).intValue();

		/* why not use `switch'? --- I just don't like it  */
		if( y == 1 ) { return new Integer( x ); }
		else if ( y == 0 ) { return new Integer( 0 ); }
		else if ( (x == 1 || x == b) && y == b ) { return new Integer( b ); }
		else if ( (x == 1 || x == n) && y == n ) { return new Integer( n ); }
		else { return new Integer( 0 ); }
	}

// /*
//  * [ x & y ] 
//  */
// // 	public Object interpretAnd( String symbol_x, String symbol_y ) {
// 	public Object interpretAnd( Object symbol_x, Object symbol_y ) {
// 		int x = ((Integer) interpretSymbol( symbol_x )).intValue();
// 		int y = ((Integer) interpretSymbol( symbol_y )).intValue();
// 		/* why not use `switch'? --- I just don't like it  */
// 		if( y == 1 ) { return new Integer( x ); }
// 		else if ( y == 0 ) { return new Integer( 0 ); }
// 		else if ( (x == 1 || x == b) && y == b ) { return new Integer( b ); }
// 		else if ( (x == 1 || x == n) && y == n ) { return new Integer( n ); }
// 		else { return new Integer( 0 ); }
// 	}

/*
 * [ x | y ] Return 1 iff x = y = 0; otherwise, return 1
 */
	public Object interpretOr( Object value_x, Object value_y ) {		//  value_x and value_y should be an integer object which has been interpreted already.
		int x = ((Integer) value_x).intValue();
		int y = ((Integer) value_y).intValue();
		
		/* there's a neater way to implement the same behavior, which I didn't
		 * employ because of the time constrant. why not use `switch'? --- I
		 * just don't like it */
		if( x == 1 || y == 1 ) { return new Integer( 1 ); }
		else if ( x == 0 && y == 0 ) { return new Integer( 0 ); }
		else if ( (x == 0 || x == n) && (y == 0 || y == n)) { return new Integer( n ); }
		else if ( (x == n && y == b) || (x == b && y == n) ) { return new Integer( 1 ); }
		else { return new Integer( b ); }
	}
	

// /*
//  * [ x | y ] Return 1 iff x = y = 0; otherwise, return 1
//  */
// 	public Object interpretOr( Object symbol_x, Object symbol_y ) {
 
// 		int x = ((Integer) interpretSymbol( symbol_x )).intValue();
// 		int y = ((Integer) interpretSymbol( symbol_y )).intValue();
		
// 		/* there's a neater way to implement the same behavior, which I didn't
// 		 * employ because of the time constrant. why not use `switch'? --- I
// 		 * just don't like it */
// 		if( x == 1 || y == 1 ) { return new Integer( 1 ); }
// 		else if ( x == 0 && y == 0 ) { return new Integer( 0 ); }
// 		else if ( (x == 0 || x == n) && (y == 0 || y == n)) { return new Integer( n ); }
// 		else if ( (x == n && y == b) || (x == b && y == n) ) { return new Integer( 1 ); }
// 		else { return new Integer( b ); }
// 	}
	
	public Object interpretImplies( Object o1, Object o2 )  {
		return interpretOr( interpretNot( o1 ) , o2 );
	}

// 		System.out.println( "WARNING: In system \"FDE\", IMPLIES ( => ) operator is undefined."  );
// 		System.out.println( "Returning False ( 0 ) for the expression..."  );
// 		return new Integer( 0 );

	public Object interpretIff( Object x, Object y )  {
		return interpretAnd ( interpretImplies( x, y ) , interpretImplies( y, x ) );
	}

// 		System.out.println( "WARNING: In system \"FDE\", IFF ( <=> ) operator is undefined."  );
// 		System.out.println( "Returning False ( 0 ) for the expression..."  );
// 		return new Integer( 0 );


/* WELL, I DECIDED NOT TO BE SO OVERTLY STRICT HERE... :-) 


	public Object getCorrespondingNumberFormat( String number_string ) {

		// In FDE, a truth value may be obtained by confirming whether a
		// certain sentence symbol(s) relates to a certain value(s). Thus,
		// given a number string, there is no corresponding number
		// representation that this system has, in which case we return integer
		// 0.
		System.out.println( "WARNING: System`" + this.getSystemName() + "' doesn't have a corresponding number representation for the input number `" + number_string + "'."   );
		System.out.println( "(In this system, a certain sentence symbol(s) relates to a certain truth value, truth values, or none."   );
		System.out.println( "One must, thus, provide a sentence symbol(s) in order to have a truth value returned.) "  );
		System.out.println( "Returning integer value 0 instead..."  );
		return new Integer( 0 );

	}

*/

	public Object getCorrespondingNumberFormat( String number_string ) {
		Double d = new Double( number_string );
		int i = d.intValue();
		return new Integer( i );
	}


	public Object getTrueValue( ) {
		return new Integer( 1 );
	}
		
	public Object getFalseValue( ) {
		return new Integer( 0 );
	}



}


// 2004-11-06 [19:51]
//  *. Propositional Modal Logic


// 2004-11-06 [15:48]
//  *. Test 


	public static void main(String[] args) {
		LogicalSystems logicalSystems = new LogicalSystems( );
		String operator = "" ;
		String operator_name = "" ;

// 2004-11-06 [18:11]
//  **. CPL test 

		SystemCPL cpl = logicalSystems.new SystemCPL();

		System.out.println(  );System.out.println(  );
		System.out.println( "========= CPL TEST ========="  );

		System.out.println( "1 in English : "  + cpl.truthValueToEnglish( new Integer( 1 ) ));
		System.out.println( "0 in English : "  + cpl.truthValueToEnglish( new Integer( 0 ) ));

		System.out.println(  );
		System.out.println( "NOT Table ----------"  );
		System.out.println( "~1 : " + cpl.interpretNot( new Integer( 1 ) ) ); 
		System.out.println( "~0 : " + cpl.interpretNot( new Integer( 0 ) ) ); 

		operator_name = "AND" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 1; i >= 0; i-- ) {
			for( int j = 1; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + cpl.interpretAnd( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "OR" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 1; i >= 0; i-- ) {
			for( int j = 1; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + cpl.interpretOr( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "IMPLIES" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 1; i >= 0; i-- ) {
			for( int j = 1; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + cpl.interpretImplies( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "IFF" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 1; i >= 0; i-- ) {
			for( int j = 1; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + cpl.interpretIff( new Integer( i ), new Integer( j ) ) );
			}
		}


// 2004-11-06 [18:24]
//  **. K3 test 

		SystemK3 k3 = logicalSystems.new SystemK3();
		
		System.out.println(  ); System.out.println(  );
		System.out.println( "========= K3 TEST ========="  );

		System.out.println( "2 in English : "  + k3.truthValueToEnglish( new Integer( 2 ) ) + "  ( regarded as: " + k3.trueOrFalse( new Integer( 2 ) ) + " )");
		System.out.println( "1 in English : "  + k3.truthValueToEnglish( new Integer( 1 ) ) + "  ( regarded as: " + k3.trueOrFalse( new Integer( 1 ) ) + " )");
		System.out.println( "0 in English : "  + k3.truthValueToEnglish( new Integer( 0 ) ) + "  ( regarded as: " + k3.trueOrFalse( new Integer( 0 ) ) + " )");

		System.out.println(  );
		System.out.println( "NOT Table ----------"  );
		System.out.println( "~2 : " + k3.interpretNot( new Integer( 2 ) ) ); 
		System.out.println( "~1 : " + k3.interpretNot( new Integer( 1 ) ) ); 
		System.out.println( "~0 : " + k3.interpretNot( new Integer( 0 ) ) ); 

		operator_name = "AND" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + k3.interpretAnd( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "OR" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + k3.interpretOr( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "IMPLIES" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + k3.interpretImplies( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "IFF" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + k3.interpretIff( new Integer( i ), new Integer( j ) ) );
			}
		}



// 2004-11-06 [18:37]
//  **. L3 test 


		SystemL3 l3 = logicalSystems.new SystemL3();
		
		System.out.println(  ); System.out.println(  );
		System.out.println( "========= L3 TEST ========="  );

		System.out.println( "2 in English : "  + l3.truthValueToEnglish( new Integer( 2 ) ) + "  ( regarded as: " + l3.trueOrFalse( new Integer( 2 ) ) + " )");
		System.out.println( "1 in English : "  + l3.truthValueToEnglish( new Integer( 1 ) ) + "  ( regarded as: " + l3.trueOrFalse( new Integer( 1 ) ) + " )");
		System.out.println( "0 in English : "  + l3.truthValueToEnglish( new Integer( 0 ) ) + "  ( regarded as: " + l3.trueOrFalse( new Integer( 0 ) ) + " )");

		System.out.println(  );
		System.out.println( "NOT Table ----------"  );
		System.out.println( "~2 : " + l3.interpretNot( new Integer( 2 ) ) ); 
		System.out.println( "~1 : " + l3.interpretNot( new Integer( 1 ) ) ); 
		System.out.println( "~0 : " + l3.interpretNot( new Integer( 0 ) ) ); 

		operator_name = "AND" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + l3.interpretAnd( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "OR" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + l3.interpretOr( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "IMPLIES" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + l3.interpretImplies( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "IFF" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + l3.interpretIff( new Integer( i ), new Integer( j ) ) );
			}
		}



// 2004-11-06 [18:40]
//  **. LP test 

		SystemLP lp = logicalSystems.new SystemLP();
		
		System.out.println(  ); System.out.println(  );
		System.out.println( "========= LP TEST ========="  );

		System.out.println( "2 in English : "  + lp.truthValueToEnglish( new Integer( 2 ) ) + "  ( regarded as: " + lp.trueOrFalse( new Integer( 2 ) ) + " )");
		System.out.println( "1 in English : "  + lp.truthValueToEnglish( new Integer( 1 ) ) + "  ( regarded as: " + lp.trueOrFalse( new Integer( 1 ) ) + " )");
		System.out.println( "0 in English : "  + lp.truthValueToEnglish( new Integer( 0 ) ) + "  ( regarded as: " + lp.trueOrFalse( new Integer( 0 ) ) + " )");

		System.out.println(  );
		System.out.println( "NOT Table ----------"  );
		System.out.println( "~2 : " + lp.interpretNot( new Integer( 2 ) ) ); 
		System.out.println( "~1 : " + lp.interpretNot( new Integer( 1 ) ) ); 
		System.out.println( "~0 : " + lp.interpretNot( new Integer( 0 ) ) ); 

		operator_name = "AND" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + lp.interpretAnd( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "OR" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + lp.interpretOr( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "IMPLIES" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + lp.interpretImplies( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "IFF" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + lp.interpretIff( new Integer( i ), new Integer( j ) ) );
			}
		}



// 2004-11-06 [18:40]
//  **. RM3 test 

		SystemRM3 rm3 = logicalSystems.new SystemRM3();
		
		System.out.println(  ); System.out.println(  );
		System.out.println( "========= RM3 TEST ========="  );

		System.out.println( "2 in English : "  + rm3.truthValueToEnglish( new Integer( 2 ) ) + "  ( regarded as: " + rm3.trueOrFalse( new Integer( 2 ) ) + " )");
		System.out.println( "1 in English : "  + rm3.truthValueToEnglish( new Integer( 1 ) ) + "  ( regarded as: " + rm3.trueOrFalse( new Integer( 1 ) ) + " )");
		System.out.println( "0 in English : "  + rm3.truthValueToEnglish( new Integer( 0 ) ) + "  ( regarded as: " + rm3.trueOrFalse( new Integer( 0 ) ) + " )");

		System.out.println(  );
		System.out.println( "NOT Table ----------"  );
		System.out.println( "~2 : " + rm3.interpretNot( new Integer( 2 ) ) ); 
		System.out.println( "~1 : " + rm3.interpretNot( new Integer( 1 ) ) ); 
		System.out.println( "~0 : " + rm3.interpretNot( new Integer( 0 ) ) ); 

		operator_name = "AND" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + rm3.interpretAnd( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "OR" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + rm3.interpretOr( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "IMPLIES" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + rm3.interpretImplies( new Integer( i ), new Integer( j ) ) );
			}
		}

		operator_name = "IFF" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for( int i = 2; i >= 0; i-- ) {
			for( int j = 2; j >= 0; j-- ) {
				System.out.println( i + operator + j + " : " + rm3.interpretIff( new Integer( i ), new Integer( j ) ) );
			}
		}



// 2004-11-06 [18:40]
//  **. L test 

		double border_line = 0.5;
		double[]  ex = { 1.0, 0.75, 0.5, 0.25, 0.0 } ;
		SystemL l = logicalSystems.new SystemL( border_line ) ;
		
		System.out.println(  ); System.out.println(  );
		System.out.println( "========= L TEST ========="  );

		System.out.println( "The boarder line lambda value is : " + border_line );
		for (int i = 0; i < ex.length ; i++ ) {
			System.out.println( ex[i] + " in English : "  + l.truthValueToEnglish( new Double( ex[i] ) ) + "  ( regarded as: " + l.trueOrFalse( new Double( ex[i] ) ) + " )");
		}

		System.out.println( l.interpretNot( new Double( 1.5 ) ) );		// bigger than 1.0


// 		try {

		System.out.println(  );
		System.out.println( "NOT Table ----------"  );
		for (int i = 0; i < ex.length ; i++ ) {
			System.out.println( "~" + ex[i] + "   :   " + l.interpretNot( new Double( ex[i] ) ) ); 
		}

		operator_name = "AND" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for (int i = 0; i < ex.length ; i++ ) {
			for (int j = 0; j < ex.length ; j++ ) {
				System.out.println( ex[i] + "   "  + ex[j] + "   :   " + l.interpretAnd( new Double( ex[i] ), new Double( ex[j] )) );
			}
		}

		operator_name = "OR" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for (int i = 0; i < ex.length ; i++ ) {
			for (int j = 0; j < ex.length ; j++ ) {
				System.out.println( ex[i] + "   "  + ex[j] + "   :   " + l.interpretOr( new Double( ex[i] ), new Double( ex[j] )) );
			}
		}

		operator_name = "IMPLIES" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for (int i = 0; i < ex.length ; i++ ) {
			for (int j = 0; j < ex.length ; j++ ) {
				System.out.println( ex[i] + "   "  + ex[j] + "   :   " + l.interpretImplies( new Double( ex[i] ), new Double( ex[j] )) );
			}
		}

		operator_name = "IFF" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for (int i = 0; i < ex.length ; i++ ) {
			for (int j = 0; j < ex.length ; j++ ) {
				System.out.println( ex[i] + "   "  + ex[j] + "   :   " + l.interpretIff( new Double( ex[i] ), new Double( ex[j] )) );
			}
		}

// 		} 
// 		catch (FuzzyLogicNumberOutOfRangeException e) { }

// 2004-11-06 [18:40]
//  **. L Aleph test 

// 		double border_line = 0.5;
// 		double[]  ex = { 1.0, 0.75, 0.5, 0.25, 0.0 } ;
		SystemLAleph lAleph = logicalSystems.new SystemLAleph(  ) ;
		
		System.out.println(  ); System.out.println(  );
		System.out.println( "========= LALEPH TEST ========="  );

// 		System.out.println( "The boarder line lambda value is : " + border_line );
		for (int i = 0; i < ex.length ; i++ ) {
			System.out.println( ex[i] + " in English : "  + lAleph.truthValueToEnglish( new Double( ex[i] ) ) + "  ( regarded as: " + lAleph.trueOrFalse( new Double( ex[i] ) ) + " )");
		}

// 		try {

		System.out.println(  );
		System.out.println( "NOT Table ----------"  );
		for (int i = 0; i < ex.length ; i++ ) {
			System.out.println( "~" + ex[i] + "   :   " + lAleph.interpretNot( new Double( ex[i] ) ) ); 
		}

		operator_name = "AND" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for (int i = 0; i < ex.length ; i++ ) {
			for (int j = 0; j < ex.length ; j++ ) {
				System.out.println( ex[i] + "   "  + ex[j] + "   :   " + lAleph.interpretAnd( new Double( ex[i] ), new Double( ex[j] )) );
			}
		}

		operator_name = "OR" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for (int i = 0; i < ex.length ; i++ ) {
			for (int j = 0; j < ex.length ; j++ ) {
				System.out.println( ex[i] + "   "  + ex[j] + "   :   " + lAleph.interpretOr( new Double( ex[i] ), new Double( ex[j] )) );
			}
		}

		operator_name = "IMPLIES" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for (int i = 0; i < ex.length ; i++ ) {
			for (int j = 0; j < ex.length ; j++ ) {
				System.out.println( ex[i] + "   "  + ex[j] + "   :   " + lAleph.interpretImplies( new Double( ex[i] ), new Double( ex[j] )) );
			}
		}

		operator_name = "IFF" ;
		System.out.println(  );
		System.out.println( operator_name+" Table ----------"  );
		for (int i = 0; i < ex.length ; i++ ) {
			for (int j = 0; j < ex.length ; j++ ) {
				System.out.println( ex[i] + "   "  + ex[j] + "   :   " + lAleph.interpretIff( new Double( ex[i] ), new Double( ex[j] )) );
			}
		}

// 		} 
// 		catch (FuzzyLogicNumberOutOfRangeException e) { }

// 2004-11-06 [16:29]
//  **. FDE test 

		/* constructing a relation */
		BinaryTwoValuedRelation r = new BinaryTwoValuedRelation( );
		// a: 1, b: b, c: n, d: 0
		r.add( "a" , "1");
		r.add( "b" , "1");
		r.add( "b" , "0");
		r.add( "d" , "0");
		
		SystemFDE fde = logicalSystems.new SystemFDE(r);

		System.out.println(  ); System.out.println(  );
		System.out.println( "========= FDE TEST ========="  );

		System.out.println( "3 in English : "  + fde.truthValueToEnglish( new Integer( 3 ) ) + "  ( regarded as: " + fde.trueOrFalse( new Integer( 3 ) ) + " )");
		System.out.println( "2 in English : "  + fde.truthValueToEnglish( new Integer( 2 ) ) + "  ( regarded as: " + fde.trueOrFalse( new Integer( 2 ) ) + " )");
		System.out.println( "1 in English : "  + fde.truthValueToEnglish( new Integer( 1 ) ) + "  ( regarded as: " + fde.trueOrFalse( new Integer( 1 ) ) + " )");
		System.out.println( "0 in English : "  + fde.truthValueToEnglish( new Integer( 0 ) ) + "  ( regarded as: " + fde.trueOrFalse( new Integer( 0 ) ) + " )");

		System.out.println(  );
		System.out.println( "NOT Table:"  );
		System.out.print( fde.interpretNot( fde.interpretSymbol("a" ) )); System.out.print( fde.interpretNot( fde.interpretSymbol("b" ) )); System.out.print( fde.interpretNot( fde.interpretSymbol("c" ) )); System.out.print( fde.interpretNot( fde.interpretSymbol( "d" ) )); System.out.println(  );

		System.out.println(  );
		System.out.println( "AND Table:"  );
		System.out.print( fde.interpretAnd( fde.interpretSymbol("a"), fde.interpretSymbol("a") ) ); System.out.print( fde.interpretAnd( fde.interpretSymbol("a"), fde.interpretSymbol("b") )); System.out.print( fde.interpretAnd( fde.interpretSymbol("a"), fde.interpretSymbol("c") )); System.out.print( fde.interpretAnd( fde.interpretSymbol("a"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretAnd( fde.interpretSymbol("b"), fde.interpretSymbol("a") )); System.out.print( fde.interpretAnd( fde.interpretSymbol("b"), fde.interpretSymbol("b") )); System.out.print( fde.interpretAnd( fde.interpretSymbol("b"), fde.interpretSymbol("c") )); System.out.print( fde.interpretAnd( fde.interpretSymbol("b"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretAnd( fde.interpretSymbol("c"), fde.interpretSymbol("a") )); System.out.print( fde.interpretAnd( fde.interpretSymbol("c"), fde.interpretSymbol("b") )); System.out.print( fde.interpretAnd( fde.interpretSymbol("c"), fde.interpretSymbol("c") )); System.out.print( fde.interpretAnd( fde.interpretSymbol("c"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretAnd( fde.interpretSymbol("d"), fde.interpretSymbol("a") )); System.out.print( fde.interpretAnd( fde.interpretSymbol("d"), fde.interpretSymbol("b") )); System.out.print( fde.interpretAnd( fde.interpretSymbol("d"), fde.interpretSymbol("c") )); System.out.print( fde.interpretAnd( fde.interpretSymbol("d"), fde.interpretSymbol("d") )); System.out.println(  );

		System.out.println(  );
		System.out.println( "OR Table:"  );
		System.out.print( fde.interpretOr( fde.interpretSymbol("a"), fde.interpretSymbol("a") )); System.out.print( fde.interpretOr( fde.interpretSymbol("a"), fde.interpretSymbol("b") )); System.out.print( fde.interpretOr( fde.interpretSymbol("a"), fde.interpretSymbol("c") )); System.out.print( fde.interpretOr( fde.interpretSymbol("a"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretOr( fde.interpretSymbol("b"), fde.interpretSymbol("a") )); System.out.print( fde.interpretOr( fde.interpretSymbol("b"), fde.interpretSymbol("b") )); System.out.print( fde.interpretOr( fde.interpretSymbol("b"), fde.interpretSymbol("c") )); System.out.print( fde.interpretOr( fde.interpretSymbol("b"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretOr( fde.interpretSymbol("c"), fde.interpretSymbol("a") )); System.out.print( fde.interpretOr( fde.interpretSymbol("c"), fde.interpretSymbol("b") )); System.out.print( fde.interpretOr( fde.interpretSymbol("c"), fde.interpretSymbol("c") )); System.out.print( fde.interpretOr( fde.interpretSymbol("c"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretOr( fde.interpretSymbol("d"), fde.interpretSymbol("a") )); System.out.print( fde.interpretOr( fde.interpretSymbol("d"), fde.interpretSymbol("b") )); System.out.print( fde.interpretOr( fde.interpretSymbol("d"), fde.interpretSymbol("c") )); System.out.print( fde.interpretOr( fde.interpretSymbol("d"), fde.interpretSymbol("d") )); System.out.println(  );

		System.out.println(  );
		System.out.println( "IMPLIES Table:"  );
		System.out.print( fde.interpretImplies( fde.interpretSymbol("a"), fde.interpretSymbol("a") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("a"), fde.interpretSymbol("b") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("a"), fde.interpretSymbol("c") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("a"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretImplies( fde.interpretSymbol("b"), fde.interpretSymbol("a") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("b"), fde.interpretSymbol("b") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("b"), fde.interpretSymbol("c") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("b"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretImplies( fde.interpretSymbol("c"), fde.interpretSymbol("a") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("c"), fde.interpretSymbol("b") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("c"), fde.interpretSymbol("c") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("c"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretImplies( fde.interpretSymbol("d"), fde.interpretSymbol("a") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("d"), fde.interpretSymbol("b") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("d"), fde.interpretSymbol("c") )); System.out.print( fde.interpretImplies( fde.interpretSymbol("d"), fde.interpretSymbol("d") )); System.out.println(  );


		System.out.println(  );
		System.out.println( "IFF Table:"  );
		System.out.print( fde.interpretIff( fde.interpretSymbol("a"), fde.interpretSymbol("a") )); System.out.print( fde.interpretIff( fde.interpretSymbol("a"), fde.interpretSymbol("b") )); System.out.print( fde.interpretIff( fde.interpretSymbol("a"), fde.interpretSymbol("c") )); System.out.print( fde.interpretIff( fde.interpretSymbol("a"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretIff( fde.interpretSymbol("b"), fde.interpretSymbol("a") )); System.out.print( fde.interpretIff( fde.interpretSymbol("b"), fde.interpretSymbol("b") )); System.out.print( fde.interpretIff( fde.interpretSymbol("b"), fde.interpretSymbol("c") )); System.out.print( fde.interpretIff( fde.interpretSymbol("b"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretIff( fde.interpretSymbol("c"), fde.interpretSymbol("a") )); System.out.print( fde.interpretIff( fde.interpretSymbol("c"), fde.interpretSymbol("b") )); System.out.print( fde.interpretIff( fde.interpretSymbol("c"), fde.interpretSymbol("c") )); System.out.print( fde.interpretIff( fde.interpretSymbol("c"), fde.interpretSymbol("d") )); System.out.println(  );
		System.out.print( fde.interpretIff( fde.interpretSymbol("d"), fde.interpretSymbol("a") )); System.out.print( fde.interpretIff( fde.interpretSymbol("d"), fde.interpretSymbol("b") )); System.out.print( fde.interpretIff( fde.interpretSymbol("d"), fde.interpretSymbol("c") )); System.out.print( fde.interpretIff( fde.interpretSymbol("d"), fde.interpretSymbol("d") )); System.out.println(  );



	}		// end of main


}		// END OF `LogicalSystems' 