//shaibutton.h

#ifndef SHAIBUTTON_H
#define SHAIBUTTON_H

// for ANSI memory commands - memcpy,memcmp,memset
#include <string.h>
#include <time.h>
#include "ownet.h"
#include "owfile.h"

// ********************************************************************** //
// DS1963S SHA iButton commands
// ********************************************************************** //
#define SHA_COMPUTE_FIRST_SECRET 0x0F
#define SHA_COMPUTE_NEXT_SECRET  0xF0
#define SHA_VALIDATE_DATA_PAGE   0x3C
#define SHA_SIGN_DATA_PAGE       0xC3
#define SHA_COMPUTE_CHALLENGE    0xCC
#define SHA_AUTHENTICATE_HOST    0xAA
#define CMD_READ_MEMORY          0xF0
#define CMD_MATCH_SCRATCHPAD     0x3C
#define CMD_WRITE_SCRATCHPAD     0x0F
#define CMD_READ_SCRATCHPAD      0xAA
#define CMD_ERASE_SCRATCHPAD     0xC3
#define CMD_COPY_SCRATCHPAD      0x55
#define CMD_READ_AUTH_PAGE       0xA5
#define CMD_COMPUTE_SHA          0x33
#define ROM_CMD_SKIP             0x3C
#define ROM_CMD_RESUME           0xA5

// ********************************************************************** //
// DS1961S SHA iButton commands - rest are shared with above
// ********************************************************************** //
#define SHA33_LOAD_FIRST_SECRET     0x5A
#define SHA33_COMPUTE_NEXT_SECRET   0x33

// ********************************************************************** //
// other constants
#define SHA_FAMILY_CODE        0x18
#define SHA33_FAMILY_CODE      0x33
// maximum number of buttons to track on the port
#define MAX_SHA_IBUTTONS       16

#define SHACoprFilename       "shacopr.cnf"

// ********************************************************************** //
// SHA Device Structures
// ********************************************************************** //
typedef struct
{
   int   portnum;
   uchar devAN[8];
   uchar serviceFilename[5];
   uchar signPageNumber;
   uchar authPageNumber;
   uchar wspcPageNumber;
   uchar versionNumber;
   uchar bindCode[7];
   uchar bindData[32];
   uchar signChlg[3];
   uchar initSignature[20];
   uchar* providerName; //variable length
   uchar* auxilliaryData; //variable length
   uchar encCode;
   uchar ds1961Scompatible;
} SHACopr;

/* 
   An application note in development at the time of this releases specifies
   a new format for the user's account file, or eCertificate.  As long as
   the struct is used, as opposed to directly indexing the specific fields
   in a byte array that represents the entire page of data, the transition 
   should be smooth when that app note is released.
*/
typedef struct
{
   uchar fileLength;
   uchar transID[2];
   uchar convFactor[2];
   uchar balanceBytes[3];
   uchar signature[20];
   uchar dataTypeCode;
   uchar contPtr;
   uchar crc16[2];
} DebitFile;

typedef struct
{
   int   portnum;
   uchar devAN[8];
   uchar accountPageNumber;
   uchar responseMAC[20];
   int   writeCycleCounter;
   union
   {
      uchar raw[32]; // used for direct writes to button only
      DebitFile file; // use this for accessing individual fields
   } accountFile;
} SHAUser;


// ********************************************************************** //
// DS1963S Low-level Functions - defined in sha18.c
// ********************************************************************** //
// General I/O
extern SMALLINT CopySecretSHA(int portnum, SMALLINT secretnum);
extern SMALLINT ReadScratchpadSHA(int portnum, int* address, 
                                  uchar* es, uchar* data, 
                                  SMALLINT resume);
extern SMALLINT WriteScratchpadSHA(int portnum, int address, 
                                   uchar *data, SMALLINT data_len, 
                                   SMALLINT resume);
extern SMALLINT CopyScratchpadSHA(int portnum, int address, 
                                  SMALLINT len, SMALLINT resume);
extern SMALLINT MatchScratchpadSHA(int portnum, uchar* data, 
                                   SMALLINT resume);
extern SMALLINT EraseScratchpadSHA(int portnum, int address, 
                                   SMALLINT resume);
extern int ReadAuthPageSHA(int portnum, SMALLINT pagenum, uchar* data, 
                           uchar* sign, SMALLINT resume);
extern SMALLINT ReadMemoryPageSHA(int portnum, SMALLINT pagenum, uchar* data, 
                                  SMALLINT resume);
extern SMALLINT WriteDataPageSHA(int portnum, SMALLINT pagenum, 
                                 uchar* data, SMALLINT resume);
extern SMALLINT SHAFunction(int portnum, uchar control_byte, 
                            int address, SMALLINT resume);
// Secret Installation
extern SMALLINT InstallSystemSecret(int portnum, SMALLINT pagenum,
                                    SMALLINT secretnum, uchar* secret, 
                                    int secret_length, SMALLINT resume);
extern SMALLINT BindSecretToiButton(int portnum, SMALLINT pagenum,
                                    SMALLINT secretnum, 
                                    uchar* bindData, uchar* bindCode,
                                    SMALLINT resume);
// ********************************************************************** //


// ********************************************************************** //
// DS1961S Low-level Functions - defined in sha33.c
// ********************************************************************** //
// General I/O
extern SMALLINT ReadScratchpadSHA33(int portnum, int* address, 
                                    uchar* es, uchar* data, 
                                    SMALLINT resume);
extern SMALLINT WriteScratchpadSHA33(int portnum, int address, 
                                     uchar *data, SMALLINT resume);
extern SMALLINT CopyScratchpadSHA33(int portnum, int address, 
                                    uchar* MAC, SMALLINT resume);
extern SMALLINT ReadMemoryPageSHA33(int portnum, SMALLINT pagenum, 
                                    uchar* data, SMALLINT resume);
extern int ReadAuthPageSHA33(int portnum, SMALLINT pagenum, 
                             uchar* data, uchar* sign, 
                             SMALLINT resume);
// Secret Installation
extern SMALLINT LoadFirstSecret33(int portnum, uchar* secret, 
                                  SMALLINT resume);
extern SMALLINT InstallSystemSecret33(int portnum, SMALLINT pagenum,
                                      SMALLINT secretnum, 
                                      uchar* input_secret, 
                                      int secret_length, 
                                      SMALLINT resume);
extern SMALLINT BindSecretToiButton33(int portnum, SMALLINT pagenum,
                                      SMALLINT secretnum, 
                                      uchar* bindData, 
                                      uchar* bindCode,
                                      SMALLINT resume);
// General Util
extern void ReformatSecretFor1961S(uchar* auth_secret, int secret_length);
extern void ComputeSHAVM(uchar* MT, long* hash);
extern void HashToMAC(long* hash, uchar* MAC);
// ********************************************************************** //

// ********************************************************************** //
// Protocol-Level Functions - defined in shaibutton.c
// ********************************************************************** //
// Finding and accessing SHA iButtons
extern SMALLINT SelectSHA(int portnum);
extern SMALLINT FindNewSHA(int portnum, uchar* devAN, SMALLINT forceFirst);
extern SMALLINT FindUserSHA(SHAUser* user, FileEntry* fe, 
                            SMALLINT doBlocking);
extern SMALLINT FindCoprSHA(SHACopr* copr, FileEntry* fe);
extern void GetCoprFromRawData(SHACopr* copr, uchar* raw, int len);
// General Protocol functions for 1963S
extern SMALLINT CreateChallenge(SHACopr* copr, SMALLINT pageNum, 
                                uchar* chlg, SMALLINT offset);
extern int AnswerChallenge(SHAUser* user, uchar* chlg);
extern SMALLINT VerifyAuthResponse(SHACopr* copr, SHAUser* user, 
                                   uchar* chlg, SMALLINT doBind);
extern SMALLINT CreateDataSignature(SHACopr* copr, uchar* data, 
                                    uchar* scratchpad, uchar* signature,
                                    SMALLINT readSignature);
// Useful utility functions
extern SMALLINT IntToBytes(uchar* byteArray, int len, unsigned int val);
extern int BytesToInt(uchar* byteArray, int len);
// ********************************************************************** //

// ********************************************************************** //
// Service-Level Functions - defined in shadebit.c (or shadebitvm.c)
// ********************************************************************** //
extern SMALLINT VerifyUser(SHACopr* copr, SHAUser* user, SMALLINT doBind);
extern SMALLINT VerifyData(SHACopr* copr, SHAUser* user);
extern SMALLINT ExecuteTransaction(SHACopr* copr, SHAUser* user, 
                                   int debitAmount, 
                                   SMALLINT verifySuccess );
extern SMALLINT UpdateServiceData(SHACopr* copr, SHAUser* user);
extern SMALLINT InstallServiceData(SHACopr* copr, SHAUser* user,
                                   uchar* secret, int secret_length);
// ********************************************************************** //

// ********************************************************************** //
// Service/Protocol-Level Functions - defined in shadebitvm.c
// ********************************************************************** //
extern SMALLINT InstallSystemSecretVM(SHACopr* copr, 
                                      uchar* secret, 
                                      int secret_length, 
                                      uchar* dst_secret);
extern SMALLINT InstallSignSecretVM(SHACopr* copr, 
                                    uchar* secret, 
                                    int secret_length);
extern SMALLINT InstallAuthSecretVM(SHACopr* copr, 
                                    uchar* secret, 
                                    int secret_length);
extern SMALLINT VerifyAuthResponseVM(SHACopr* copr, SHAUser* user, 
                                     uchar* chlg, SMALLINT doBind);
extern SMALLINT CreateDataSignatureVM(SHACopr* copr, uchar* secret, 
                                      uchar* data, uchar* scratchpad, 
                                      uchar* signature, SMALLINT readSignature);
extern SMALLINT GetCoprVM(SHACopr* copr, FileEntry* fe);
// ********************************************************************** //

// ********************************************************************** //
// Global - defined in shaibutton.c
extern SMALLINT in_overdrive[MAX_PORTNUM];
// ********************************************************************** //

extern void PrintHexLabeled(char* label,uchar* buffer, int cnt);

extern void PrintSerialNum(uchar* buffer);
extern void PrintHex(uchar* buffer, int cnt);
extern void PrintChars(uchar* buffer, int cnt);
extern void ReadChars(uchar* buffer, int len);
extern int getData(uchar *write_buff, int max_len, SMALLINT gethex);
extern int EnterString(char *msg, char *buf, int min, int max);
extern int EnterNum(char *msg, int numchars, long *value, long min, long max);

extern SMALLINT owTouchReset(int);
extern SMALLINT owWriteByte(int,SMALLINT);
extern SMALLINT owSpeed(int,SMALLINT);
extern SMALLINT owVerify(int,SMALLINT);
extern void owSerialNum(int,uchar *,SMALLINT);
extern SMALLINT  owNext(int,SMALLINT,SMALLINT);
extern SMALLINT  owFirst(int,SMALLINT,SMALLINT);
extern void owFamilySearchSetup(int,SMALLINT);
extern void msDelay(int);
extern SMALLINT owReadByte(int);
extern SMALLINT owBlock(int,SMALLINT,uchar *,SMALLINT);
extern void setcrc16(int,ushort);
extern ushort docrc16(int,ushort);

//file stuff
extern SMALLINT owFormat(int portnum, uchar *SNum);
extern SMALLINT owOpenFile(int portnum, uchar *SNum, FileEntry *flname, short *hnd);
extern SMALLINT owReadFile(int portnum, uchar *SNum, short hnd,
                           uchar *buf, int maxlen, int *fl_len);
extern SMALLINT owWriteFile(int portnum, uchar *SNum, short hnd,
                            uchar *buf, int len);
extern SMALLINT owCloseFile(int portnum, uchar *SNum, short hnd);
extern SMALLINT owDeleteFile(int portnum, uchar *SNum, FileEntry *flname);
extern SMALLINT owCreateFile(int portnum, uchar *SNum, int *maxwrite, short *hnd,
                             FileEntry *flname);


#endif //SHAIBUTTON_H
