/*
 * Copyright 2002-2007 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 */

/*
 * @test
 * @bug 4635086
 * @library ../UTIL
 * @build TestUtil
 * @run main TestKATForECB_VT
 * @summary Known Answer Test for AES cipher with ECB mode
 * @author Valerie Peng
 */
import java.security.*;
import javax.crypto.*;
import javax.crypto.spec.*;
import java.math.*;
import com.sun.crypto.provider.*;

import java.util.*;

public class TestKATForECB_VT
{
    private static final String ALGO = "AES";
    private static final String MODE = "ECB";
    private static final String PADDING = "NoPadding";

    private static String[][] CTS = {
        // Cipher Texts for 128-bit key
        {
"3AD78E726C1EC02B7EBFE92B23D9EC34",
"45BC707D29E8204D88DFBA2F0B0CAD9B",
"161556838018F52805CDBD6202002E3F",
"F5569B3AB6A6D11EFDE1BF0A64C6854A",
"64E82B50E501FBD7DD4116921159B83E",
"BAAC12FB613A7DE11450375C74034041",
"BCF176A7EAAD8085EBACEA362462A281",
"47711816E91D6FF059BBBF2BF58E0FD3",
"B970DFBE40698AF1638FE38BD3DF3B2F",
"F95B59A44F391E14CF20B74BDC32FCFF",
"720F74AE04A2A435B9A7256E49378F5B",
"2A0445F61D36BFA7E277070730CF76DA",
"8D0536B997AEFEC1D94011BAB6699A03",
"674F002E19F6ED47EFF319E51FAD4498",
"292C02C5CB9163C80AC0F6CF1DD8E92D",
"FA321CF18EF5FE727DD82A5C1E945141",
"A5A7AFE1034C39CCCEBE3C584BC0BE05",
"4FF5A52E697E77D081205DBDB21CEA39",
"209E88DC94C9003000CE0769AF7B7166",
"5DEE41AF864CB4B650E5F51551824D38",
"A79A63FA7E4503AE6D6E09F5F9053030",
"A48316749FAE7FAC7002031A6AFD8BA7",
"D6EEE8A7357A0E1D64262CA9C337AC42",
"B013CA8A62A858053E9FB667ED39829E",
"DF6EA9E4538A45A52D5C1A43C88F4B55",
"7D03BA451371591D3FD5547D9165C73B",
"0E0426281A6277E186499D365D5F49FF",
"DBC02169DD2059E6CC4C57C1FEDF5AB4",
"826590E05D167DA6F00DCC75E22788EB",
"34A73F21A04421D9786335FAAB49423A",
"ED347D0E0128EE1A7392A1D36AB78AA9",
"EE944B2FE6E9FC888042608DA9615F75",
"9E7C85A909EF7218BA7947CFB4718F46",
"811AE07A0B2B1F816587FA73699AE77D",
"68466FBF43C2FE13D4B18F7EC5EA745F",
"D20B015C7191B219780956E6101F9354",
"5939D5C1BBF54EE1B3E326D757BDDE25",
"B1FDAFE9A0240E8FFEA19CE94B5105D3",
"D62962ECE02CDD68C06BDFEFB2F9495B",
"B3BB2DE6F3C26587BA8BAC4F7AD9499A",
"E0B1072D6D9FF703D6FBEF77852B0A6B",
"D8DD51C907F478DE0228E83E61FD1758",
"A42DFFE6E7C1671C06A25236FDD10017",
"25ACF141550BFAB9EF451B6C6A5B2163",
"4DA7FCA3949B16E821DBC84F19581018",
"7D49B6347CBCC8919C7FA96A37A7A215",
"900024B29A08C6721B95BA3B753DDB4D",
"6D2182FB283B6934D90BA7848CAB5E66",
"F73EF01B448D23A4D90DE8B2F9666E7A",
"4AD9CDA2418643E9A3D926AF5E6B0412",
"7CAEC8E7E5953997D545B033201C8C5B",
"3C43CA1F6B6864503E27B48D88230CF5",
"44F779B93108FE9FEEC880D79BA74488",
"9E50E8D9CFD3A682A78E527C9072A1CF",
"68D000CBC838BBE3C505D6F814C01F28",
"2CB2A9FEC1ACD1D9B0FA05205E304F57",
"01EB2806606E46444520A5CC6180CD4B",
"DAA9B25168CC702326F217F1A0C0B162",
"3E07E648975D9578D03555B1755807ED",
"0B45F52E802C8B8DE09579425B80B711",
"659595DA0B68F6DF0DD6CA77202986E1",
"05FF42873893536E58C8FA98A45C73C4",
"B5B03421DE8BBFFC4EADEC767339A9BD",
"788BCD111ECF73D4E78D2E21BEF55460",
"909CD9EC6790359F982DC6F2393D5315",
"332950F361535FF24EFAC8C76293F12C",
"A68CCD4E330FFDA9D576DA436DB53D75",
"27C8A1CCFDB0B015D1ED5B3E77143791",
"D76A4B95887A77DF610DD3E1D3B20325",
"C068AB0DE71C66DAE83C361EF4B2D989",
"C2120BCD49EDA9A288B3B4BE79AC8158",
"0C546F62BF2773CD0F564FCECA7BA688",
"18F3462BEDE4920213CCB66DAB1640AA",
"FE42F245EDD0E24B216AEBD8B392D690",
"3D3EEBC8D3D1558A194C2D00C337FF2B",
"29AAEDF043E785DB42836F79BE6CBA28",
"215F90C6744E2944358E78619159A611",
"8606B1AA9E1D548E5442B06551E2C6DC",
"987BB4B8740EC0EDE7FEA97DF033B5B1",
"C0A3500DA5B0AE07D2F450930BEEDF1B",
"525FDF8312FE8F32C781481A8DAAAE37",
"BFD2C56AE5FB9C9DE33A6944572A6487",
"7975A57A425CDF5AA1FA929101F650B0",
"BF174BC49609A8709B2CD8366DAA79FE",
"06C50C43222F56C874B1704E9F44BF7D",
"0CEC48CD34043EA29CA3B8ED5278721E",
"9548EA34A1560197B304D0ACB8A1698D",
"22F9E9B1BD73B6B5B7D3062C986272F3",
"FEE8E934BD0873295059002230E298D4",
"1B08E2E3EB820D139CB4ABBDBE81D00D",
"0021177681E4D90CEAF69DCED0145125",
"4A8E314452CA8A8A3619FC54BC423643",
"65047474F7222C94C6965425FF1BFD0A",
"E123F551A9C4A8489622B16F961A9AA4",
"EF05530948B80915028BB2B6FE429380",
"72535B7FE0F0F777CEDCD55CD77E2DDF",
"3423D8EFC31FA2F4C365C77D8F3B5C63",
"DE0E51C264663F3C5DBC59580A98D8E4",
"B2D9391166680947AB09264156719679",
"10DB79F23B06D263835C424AF749ADB7",
"DDF72D27E6B01EC107EA3E005B59563B",
"8266B57485A5954A4236751DE07F6694",
"669A501E1F1ADE6E5523DE01D6DBC987",
"C20C48F2989725D461D1DB589DC0896E",
"DE35158E7810ED1191825D2AA98FA97D",
"4FE294F2C0F34D0671B693A237EBDDC8",
"087AE74B10CCBFDF6739FEB9559C01A4",
"5DC278970B7DEF77A5536C77AB59C207",
"7607F078C77085184EAA9B060C1FBFFF",
"9DB841531BCBE7998DAD19993FB3CC00",
"D6A089B654854A94560BAE13298835B8",
"E1E223C4CF90CC5D195B370D65114622",
"1CBED73C50D053BDAD372CEEE54836A1",
"D309E69376D257ADF2BFDA152B26555F",
"740F7649117F0DEE6EAA7789A9994C36",
"76AE64417C297184D668C5FD908B3CE5",
"6095FEA4AA8035591F1787A819C48787",
"D1FF4E7ACD1C79967FEBAB0F7465D450",
"5F5AD3C42B9489557BB63BF49ECF5F8A",
"FB56CC09B680B1D07C5A52149E29F07C",
"FF49B8DF4A97CBE03833E66197620DAD",
"5E070ADE533D2E090ED0F5BE13BC0983",
"3AB4FB1D2B7BA376590A2C241D1F508D",
"58B2431BC0BEDE02550F40238969EC78",
"0253786E126504F0DAB90C48A30321DE",
"200211214E7394DA2089B6ACD093ABE0",
"0388DACE60B6A392F328C2B971B2FE78",
"58E2FCCEFA7E3061367F1D57A4E7455A"
        },
        // Cipher Texts for 192-bit key
        {
"6CD02513E8D4DC986B4AFE087A60BD0C",
"423D2772A0CA56DAABB48D2129062987",
"1021F2A8DA70EB2219DC16804445FF98",
"C636E35B402577F96974D8804295EBB8",
"1566D2E57E8393C19E29F892EA28A9A7",
"883C878FED70B36CC09D040F9619DD19",
"06734593A974965790E715594FC34AA9",
"F19B389948D9A45534E5BD36C984134A",
"D8410DFC14FA6D175EC968EA8CAC514C",
"7E6C6EBB4029A177CF7B2FDD9AC6BB7A",
"4B51DD4850DC0A6C3A46D924003D2C27",
"2E510A9D917B15BE32A192B12A668F23",
"88F6F79962B0FB77FEA8E7C632D3108E",
"A3A35AB1D88DAF07B52794A0F065383A",
"DC6CC878433E2B3BB193049A4ECBFC53",
"EFCD3763EB7B1A415938248A9A5B4FD5",
"AB7E9FB9A66DBE5BB44854F07D9015EE",
"8B8E9D3365F8F6743ECF7E33E99255A4",
"54D37B4F176FF3D8F6AFC866066D8572",
"E83310889480FBF3C00342E3126D0D02",
"D321AB2511F92F098174AA2DE6E85DA2",
"D8E3F40B1112D5149D58C481DFA9983F",
"2454C4E0806639DDF19854D6C68054AD",
"A5506D410F7CA32F3955DD79D9D09418",
"7908EE40677699568A7DC1AA317C7E4E",
"B4B7B29DD43B2F5CF765E25192273982",
"92AFE9668159BEFFE2A86F8503260164",
"5C36A232FBA6D187A84657AD4028B18F",
"A2E994DFAB3A798DF8F54F6DA87E58E2",
"6CDAB10A72ADF77D71D0765BAAE95631",
"9FE3C801BCAAF7BB800F2E6BF3278E21",
"B459D90D9A6C392E5493BC91CF5A0863",
"0518A9FA5007F6787E0FB4E5AC27D758",
"BED9795415D28599700ED7952384A963",
"F0140421173D60251EF6CAB0229B1B50",
"460EB4652B3F6779EA28CB11B37529ED",
"C4283D351C960A6AC13CD19CCF03AE38",
"6815A10047B2C834A798EBDCC6786C75",
"99BA19F0CDD5990D0386B32CE56C9C4C",
"DE76F62C61E07915162DA13E79679DEC",
"DD0325D6854803D06D1D2277D5FB8D67",
"580B71A41DE37D6FAC83CCB0B3BB1C97",
"E9B1AB470A1B02EF0FF5E6754A092C96",
"8590620F5AF5993B7410282F4126BC1F",
"8D4914D2F1B22B2E268E66E532D29D7C",
"FD826CE48E62C5E30867044B86BA4B56",
"100E7B831C9F35FA1271F5F1316C6FCF",
"0A2DD0C17F68B996AA96C007003D0B31",
"C95F68C57E06B0A2E1F623C83C5D80BF",
"571CAFC92C7C8A5EC54C0741E186905C",
"22514353E95312C112255E1EED0B2DF6",
"791A8BF462BD17580BD9152C6D11C6C5",
"5882A0178D548F84A165DB809C60DC28",
"3CE4A90EED4458CA6039E42DDADB71C3",
"D3CBAB261207A16BE2751E77044FD7C9",
"24E32B698A7B32217093628B01F424AB",
"9F6AFC0AF27CF565110C77E3C24F4F5B",
"E088AA5CDA20EF267BB039B00C72C45B",
"5CF1018B7E0BA1775601C2E279900360",
"3B1A7388B89FB9416AD8753CF5AF35D2",
"137FA4ED00AFCD9F5D8BC0D14BD5837A",
"806F5C9B663559BB56F234881E4A3E60",
"8069A449152292DF2DE8642992C632B6",
"37C6CF2A1ABD1B1F1922B46C7B4A280D",
"7A2835260E5A0AA2B5DC301800EC8438",
"EE81FAF2F9058213FFCACF281CB8509E",
"57F22D93C37129BA331FDBA38E005A1E",
"EC798782E87B7D9F780CC3C3A46519B5",
"43EA28497F5D40E3A4744FA2EDAA42DE",
"91F004E7DEBF41B3414DD8C5C317372C",
"C249EAE54E7B4DF43B938C1B4CC28314",
"32C289D7EEFB99D2F17AD7B7D45FE1EC",
"A675FB2E8DDBF810CEF01CF2B728CD2B",
"A418AAAB6E6921CC731AA8A349386080",
"2E2B0F44863E67D9B0215C4ABD60417F",
"F0AF7CB19E911D481F6426DAEFDD2240",
"CB1304DAAA2DF6878F56AC2E0F887E04",
"B1B70A7E6A0CD1916D9B78BEA19084AE",
"0CDE9F9BE646A5FCE3436B794A9CFC65",
"68C7946D476A0A36674B36AFD7E5DF33",
"48770159A07DD8DFFF06C80105F8D57C",
"665E62801B3260E3C45BD3BE34DFDEBE",
"4159C1F686BFBE5B0E50BDB0DA532B69",
"6333100A5A4AD917DC2D4E78A04869A3",
"866A4519AB1D199F25886B89D0539ACC",
"EC0CFD37E4CBC7E8BE385283F7AEA75A",
"CA2F383AACCA0810AA13F3E710621422",
"1D0EEF6870444F950937831EC0A55D98",
"37839B35ED6801E7670496D479A95017",
"02317C8C7098C4F94AB867AC7A49DD8D",
"FFB4CB4E3F7F8BF3367EBD43236518B4",
"36BEDEF1E4AA3E4A40A305741713FCBF",
"B2DFE3C4870269C1E3FEEC39161540D9",
"147EF2518AD45DA0026056ECBF6A3DFA",
"027A75E4DE635790E47ACE90D7928804",
"C4CF3CCB59BF87D0AFBD629F48CFBB7B",
"35165C93F564C97E1C32EF97E8151A87",
"449DE37F7D5A1BBD628ABBE7E061701D",
"B1D45EAF218F1799B149BAD677FE129F",
"BE08AC6DB6BD0583AA9D2ABC71C73DCD",
"BCC835BD3DF1A79E4C7C145B899A5C25",
"3D311EA611FF5AF371301C58A8E9912D",
"A5A1BEA594ACC7CA80F09EA5ADDB5C71",
"0F09492429FE7222D6CD8190D9F2FFBF",
"816D2220A16B8AAEE71364FD43636C6F",
"D7E8702408419ED73191B107EAF75A0B",
"9B170EFB1E235B433C78E276BEA082F0",
"03BBECC5598AE974430F29395522F096",
"DB53517766C0E8CF42059607CBA89380",
"2E2AF4B7931F0AEFFAC5471148A5BB97",
"C872C0408266403B984F635FF5683DE4",
"15DCF750B0E3A68AD1F4EFD07E8967B4",
"B41092048E9E6A749F6FD8CE515A23A3",
"4DA9267D62507994312BD5C99ADDE730",
"9E2FCA6D1D626E9C6A924EBF7DBF618A",
"E092E8D7EF2C2465AEFB2493C3063590",
"1C0E58DA37D1068378A88DBE2EDE4E10",
"19063F854232B8509A6A3A6D46809959",
"447FB09E54EFA285F7530F25C4EA0022",
"F6ABE86321BE40E1FBFDAFED37CC1D9B",
"4E8506CD006666341D6CF51F98B41F35",
"53995DE0009CA18BECAFB8307C54C14C",
"2006BF99F4C58B6CC2627856593FAEEA",
"2DA697D2737CB30B744A4644FA1CBC6E",
"47A22ACDB60C3A986A8F76ECD0EA3433",
"FDAA17C2CDE20268FE36E164EA532151",
"98E7247C07F0FE411C267E4384B0F600",
"CD33B28AC773F74BA00ED1F312572435"
        },
        // Cipher Texts for 256-bit key
        {
"DDC6BF790C15760D8D9AEB6F9A75FD4E",
"C7098C217C334D0C9BDF37EA13B0822C",
"60F0FB0D4C56A8D4EEFEC5264204042D",
"73376FBBF654D0686E0E84001477106B",
"2F443B52BA5F0C6EA0602C7C4FD259B6",
"75D11B0E3A68C4223D88DBF017977DD7",
"779B38D15BFFB63D8D609D551A5CC98E",
"5275F3D86B4FB8684593133EBFA53CD3",
"1CEF2074B336CEC62F12DEA2F6AB1481",
"1AEF5ABBAD9D7160874578DCD8BAE172",
"46C525DB17E72F26BF03216846B6F609",
"E24411F941BBE08788781E3EC52CBAA4",
"83A3DEDD1DD27018F6A6477E40527581",
"B68F8A2CDBAB0C923C67FC8F0F1087DE",
"649944A70C32BF87A7409E7AE128FDE8",
"2846526D67387539C89314DE9E0C2D02",
"A9A0B8402E53C70DD1688054BA58DDFD",
"4A72E6E1B79C83AC4BE3EBA5699EED48",
"B0E36B867BA4FF2B77D0614B0E364E4C",
"49B57DE141F6418E3090F24DDD4014B6",
"A6C0D5B9797258E1987AC5F6CD20146D",
"426CF4BDCAA369175965D26E7C71EEA2",
"E27F484CE54BC99BC1A52BDA3B518A26",
"D16D186284C7E6EE64B8104E0EF20BA5",
"6431F8538AD54E1E044A9F71F8EF556B",
"ECD57CEB451D27EB96C55B2042257E8E",
"4F0F188DC911B1954AFBC734C9F68872",
"B54DEF0337626B65614E81EDFDE620F3",
"6655D8074CAE0B90B0D3A3FE72D4D9DB",
"C6B74B6B9EB4FC0C9A237DB1B616D09A",
"D7B5D076EA56EC2B20791D7AD51CCF8F",
"FE160C224BF003CE3BDDC90CB52ED22C",
"5E00DA9BA94B5EC0D258D8A8002E0F6A",
"09AC6DCFF4DACFF1651E2BA212A292A3",
"B283617E318D99AF83A05D9810BA89F7",
"0B5F70CCB40B0EF2538AE9B4A9770B35",
"43282BF180248FB517839B37F4DDAAE4",
"DDBD534C8B2E6D30A268F88C55AD765B",
"A41A164E50EC2D9F175E752B755E0B5C",
"37BFF99FF2F7AA97779E4ADF6F13FB10",
"9BA4F7BD298152903A683C4CEC669216",
"5FB750C7CE10DE7B4504248914D0DA06",
"3E748BFA108E086F51D56EC74A9E0FB9",
"31D4E56B99F5B73C1B8437DF332AFB98",
"9DC6717B84FC55D266E7B1D9B5C52A5F",
"8EF8BA007F23C0A50FC120E07041BCCD",
"C58F38E1839FC1918A12B8C9E88C66B6",
"B695D72A3FCF508C4050E12E40061C2D",
"5D2736AD478A50583BC8C11BEFF16D7A",
"DF0EACA8F17847AD41F9578F14C7B56B",
"E5AA14AD48AD0A3C47CC35D5F8020E51",
"11BE6C8F58EBD8CEF1A53F591A68E8CE",
"ECFE7BAFCBF42C1FEE015488770B3053",
"E552649F8D8EC4A1E1CD6DF50B6E6777",
"521C0629DE93B9119CDB1DDC5809DDEA",
"CB38A62A0BAB1784156BA038CBA99BF6",
"76CCEE8AAACD394DE1EEF3DDA10CB54B",
"6AFF910FA1D5673140E2DB59B8416049",
"064A12C0EF73FB386801BF4F35F3120D",
"2240E374929D5B1BB8FF0FFDDDF640EC",
"D4BA15C904C7692185DE85C02052E180",
"1714A315AB0166728A44CD91D4AE9018",
"6C970BDD9F0E222722EA31A1D12DD0AD",
"F5956EDF02BD36A401BBB6CE77C3D3FB",
"0CA11F122CCD7C259DC597EED3DF9BC4",
"50109AB4912AD2560B206F331B62EB6C",
"DBE7C91A4175614889A2D4BEFD64845E",
"0D3322853A571A6B46B79C0228E0DD25",
"96E4EE0BB9A11C6FB8522F285BADDEB6",
"96705C52D2CFCE82E630C93477C79C49",
"C50130AED6A126149D71F3888C83C232",
"4816EFE3DEB380566EBA0C17BF582090",
"0390857B4C8C98E4CF7A2B6F3394C507",
"422E73A02025EBE8B8B5D6E0FA24FCB2",
"3271AA7F4BF1D7C38050A43076D4FF76",
"D2074946F0D37B8975607BFC2E70234C",
"1A509194C1270AB92E5A42D3A9F8D98B",
"512438946360CCC4A5C6D73F6EED7130",
"98CFCDEC46EBEA1A286B3004F2746A0D",
"A1CF369949677A3AF3D58E3EABF2741B",
"D84C2E1A0E4A52166FA8FF6889D1E5E2",
"4AD91CCEEF60119B5078FD162D2735DE",
"2860793D818E97AAFF1D339D7702438D",
"6F9068BE73364AE250D89D78A6C9CE6F",
"024FC3FEF4883FEB1A8DD005305FECCE",
"08A61FE0816D75EA15EB3C9FB9CCDED6",
"449C86DFA13F260175CE39797686FFA4",
"4FFFFC29A59858E1133F2BFB1A8A4817",
"19425D1F6480B25096561295697DC2B7",
"31974727ECDD2C77C3A428FC3A8CB3FC",
"A57CD704B3C95E744D08DF443458F2F5",
"486D8C193DB1ED73ACB17990442FC40B",
"5E4DBF4E83AB3BC055B9FCC7A6B3A763",
"ACF2E0A693FBBCBA4D41B861E0D89E37",
"32A7CB2AE066A51D2B78FC4B4CFCB608",
"677D494DBB73CAF55C1990158DA12F14",
"082A0D2367512ADF0D75A151BFBE0A17",
"5E5BB7337923C482CE8CBA249E6A8C7D",
"D3001BA7C7026EE3E5003179530AFCFC",
"46EC44F8931E629FE8FD8961312EDDE1",
"C5F8ECD79C7B30E81D17E32079969310",
"5B8AD6919E24CAEBCC55401AEE0C9802",
"C2302B7E701B5CC7F8B29E3516DBBFA6",
"A1D04D6A76F9F7A94D49FAA64A87F244",
"7FB6F92D35B5CB6C631600EDB9E860BA",
"B2EF7078BCFACE07AEEC3F9B48830EB3",
"F475A7493D24C7036E53390374C378B3",
"B36802AC987377A37BD8EADC97C57D60",
"ADDCD3D19689C4DDC738CE5F69DC9505",
"0DAF8CA22884915403C0F0BB1F4BD74F",
"4AF36BAE2660503B3248E4685059FD05",
"7D5631814DD8E917D97A0D514C743971",
"BC3352500FC0CBB9DB5B5F6B491C1BE8",
"6A4A30BA87E87AF65C90AEB7AFEDC76B",
"77E6125897668AC8E73E8C79A6FF8336",
"3FA9D39104EBB323C7AAAA248960DD1E",
"FAD75AD76AB10ADC49036B250E229D39",
"2FACAA5FE35B228A16AC74088D702EC4",
"88B6CBCFDFEF8AD91720A1BB69A1F33E",
"C7E9D250998632D444356242EF04058D",
"B14DAD8D3D9153F46C0D3A1AD63C7A05",
"60ABA678A506608D0845966D29B5F790",
"482DC43F2388EF25D24144E144BD834E",
"1490A05A7CEE43BDE98B56E309DC0126",
"ABFA77CD6E85DA245FB0BDC5E52CFC29",
"DD4AB1284D4AE17B41E85924470C36F7",
"CEA7403D4D606B6E074EC5D3BAF39D18",
"530F8AFBC74536B9A963B4F1C4CB738B"
        }
    };

    private static int[] KEY_SIZES = {
        16, 24, 32
    };

    /**
     * Constructs an AES Key according to the specified key size.
     * @param len key size in bytes, i.e. 16, 24, or 32
     */
    private static SecretKey constructAESKey(int len)
        throws IllegalArgumentException {
        if ((len != 16) && (len != 24) && (len != 32)) {
            throw new IllegalArgumentException("Wrong Key Length: " + len);
        }
        byte[] keyval = new byte[len];
        return new SecretKeySpec(keyval, "AES");
    }
    /**
     * Constructs plain text byte array according to the specified round
     * number.
     * @param rounds round number starting from 0, i.e. valid from 0
     * to 127.
     */
    private static byte[] constructPT(int rounds) {
        byte[] tempValue = new byte[16];
        Arrays.fill(tempValue, (byte)0);

        int whichByte = rounds/8;
        int whichDigit = rounds % 8;
        if ((whichByte >= 16) || (whichDigit < 0) ||
            (whichDigit > 8)) {
            throw new IllegalArgumentException("Invalid rounds: " +
                                               rounds);
        }
        switch (whichDigit) {
        case 0:
            tempValue[whichByte] = (byte)0x80;
            break;
        case 1:
            tempValue[whichByte] = (byte)0x40;
            break;
        case 2:
            tempValue[whichByte] = (byte)0x20;
            break;
        case 3:
            tempValue[whichByte] = (byte)0x10;
            break;
        case 4:
            tempValue[whichByte] = (byte)0x08;
            break;
        case 5:
            tempValue[whichByte] = (byte)0x04;
            break;
        case 6:
            tempValue[whichByte] = (byte)0x02;
            break;
        case 7:
            tempValue[whichByte] = (byte)0x01;
            break;
        }
        return tempValue;
    }

    private static byte[] constructByteArray(String s) {
        int len = s.length()/2;
        byte[] tempValue = new byte[len];
        for (int i = 0; i < len; i++) {
            tempValue[i] = Integer.valueOf(s.substring(2*i, 2*i+2),
                                           16).byteValue();
        }
        return tempValue;
    }

    public boolean execute() throws Exception {
        String transformation = ALGO+"/"+MODE+"/"+PADDING;
        Cipher c = Cipher.getInstance(transformation, "SunJCE");

        for (int i=0; i<KEY_SIZES.length; i++) {
            if (KEY_SIZES[i]*8 >
                Cipher.getMaxAllowedKeyLength(transformation)) {
                // skip if this key length is larger than what's
                // configured in the jce jurisdiction policy files
                continue;
            }
            SecretKey aesKey = constructAESKey(KEY_SIZES[i]);
            byte[] plainText = null;
            byte[] cipherText = null;
            try {
            for (int j=0; j < 128; j++) {
                plainText = constructPT(j);
                c.init(Cipher.ENCRYPT_MODE, aesKey);
                cipherText = c.doFinal(plainText);
                byte[] answer = constructByteArray(CTS[i][j]);
                if (!Arrays.equals(cipherText, answer)) {
                    throw new Exception((i+1) +
                        "th known answer test failed for encryption");
                }
                c.init(Cipher.DECRYPT_MODE, aesKey);
                byte[] restored = c.doFinal(cipherText);
                if (!Arrays.equals(plainText, restored)) {
                    throw new Exception((i+1) +
                        "th known answer test failed for decryption");
                }
            }
            System.out.println(
                "Finished KAT for " + KEY_SIZES[i] + "-byte key");
            } catch (SecurityException se) {
                TestUtil.handleSE(se);
            }
        }

        // passed all tests...hooray!
        return true;
    }

    public static void main (String[] args) throws Exception {
        Security.addProvider(new com.sun.crypto.provider.SunJCE());

        TestKATForECB_VT test = new TestKATForECB_VT();
        String testName = test.getClass().getName() + "[" + ALGO +
            "/" + MODE + "/" + PADDING + "]";
        if (test.execute()) {
            System.out.println(testName + ": Passed!");
        }
    }
}
