/*
 * Copyright 2003-2004 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 */

/*
 * @test
 * @bug 4074599
 * @summary Tests for StrictMath.log10
 * @author Joseph D. Darcy
 */


/**
 * The tests in ../Math/Log10Tests.java test properties that should
 * hold for any log10 implementation, including the FDLIBM-based one
 * required for StrictMath.log10.  Therefore, the test cases in
 * ../Math/Log10Tests.java are run against both the Math and
 * StrictMath versions of log10.  The role of this test is to verify
 * that the FDLIBM log10 algorithm is being used by running golden
 * file tests on values that may vary from one conforming log10
 * implementation to another.
 */

public class Log10Tests {
    private Log10Tests(){}

    static int testLog10Case(double input, double expected) {
        return Tests.test("StrictMath.log10(double)", input,
                          StrictMath.log10(input), expected);
    }

    static int testLog10() {
        int failures = 0;
        double [][] testCases = {
            {0x1.3fffffffffec1p-297,    -0x1.653c6a27ae2f8p6},
            {0x1.4p-297,                -0x1.653c6a27ae2f7p6},
            {0x1.3fffffffffbe1p-296,    -0x1.640828f2a4382p6},
            {0x1.4p-296,                -0x1.640828f2a437dp6},
            {0x1.3fffffffffd52p-295,    -0x1.62d3e7bd9a406p6},
            {0x1.4p-295,                -0x1.62d3e7bd9a403p6},
            {0x1.3fffffffffa72p-294,    -0x1.619fa6889049p6},
            {0x1.4p-294,                -0x1.619fa68890489p6},
            {0x1.3fffffffff9bbp-293,    -0x1.606b655386518p6},
            {0x1.4p-293,                -0x1.606b65538650fp6},
            {0x1.3fffffffffbe4p-292,    -0x1.5f37241e7c59ap6},
            {0x1.4p-292,                -0x1.5f37241e7c595p6},
            {0x1.3ffffffffff7ep-291,    -0x1.5e02e2e97261cp6},
            {0x1.4p-291,                -0x1.5e02e2e97261bp6},
            {0x1.3fffffffffc9ep-290,    -0x1.5ccea1b4686a6p6},
            {0x1.4p-290,                -0x1.5ccea1b4686a1p6},
            {0x1.3fffffffffec7p-289,    -0x1.5b9a607f5e728p6},
            {0x1.4p-289,                -0x1.5b9a607f5e727p6},
            {0x1.3fffffffffbe7p-288,    -0x1.5a661f4a547b2p6},
            {0x1.4p-288,                -0x1.5a661f4a547adp6},
            {0x1.3fffffffff907p-287,    -0x1.5931de154a83cp6},
            {0x1.4p-287,                -0x1.5931de154a833p6},
            {0x1.3fffffffffd59p-286,    -0x1.57fd9ce0408bcp6},
            {0x1.4p-286,                -0x1.57fd9ce0408b9p6},
            {0x1.3fffffffffecap-285,    -0x1.56c95bab3694p6},
            {0x1.4p-285,                -0x1.56c95bab3693fp6},
            {0x1.3fffffffffbeap-284,    -0x1.55951a762c9cap6},
            {0x1.4p-284,                -0x1.55951a762c9c5p6},
            {0x1.3ffffffffff84p-283,    -0x1.5460d94122a4cp6},
            {0x1.4p-283,                -0x1.5460d94122a4bp6},
            {0x1.3fffffffffca4p-282,    -0x1.532c980c18ad6p6},
            {0x1.4p-282,                -0x1.532c980c18ad1p6},
            {0x1.3fffffffff9c4p-281,    -0x1.51f856d70eb6p6},
            {0x1.4p-281,                -0x1.51f856d70eb57p6},
            {0x1.3fffffffffe16p-280,    -0x1.50c415a204bep6},
            {0x1.4p-280,                -0x1.50c415a204bddp6},
            {0x1.3fffffffffd5ep-279,    -0x1.4f8fd46cfac66p6},
            {0x1.4p-279,                -0x1.4f8fd46cfac63p6},
            {0x1.3fffffffffd5fp-278,    -0x1.4e5b9337f0cecp6},
            {0x1.4p-278,                -0x1.4e5b9337f0ce9p6},
            {0x1.3fffffffffedp-277,     -0x1.4d275202e6d7p6},
            {0x1.4p-277,                -0x1.4d275202e6d6fp6},
            {0x1.3fffffffffbfp-276,     -0x1.4bf310cddcdfap6},
            {0x1.4p-276,                -0x1.4bf310cddcdf5p6},
            {0x1.3ffffffffff8ap-275,    -0x1.4abecf98d2e7bp6},
            {0x1.4p-275,                -0x1.4abecf98d2e7bp6},
            {0x1.3fffffffffd62p-274,    -0x1.498a8e63c8f04p6},
            {0x1.4p-274,                -0x1.498a8e63c8f01p6},
            {0x1.3fffffffff9cap-273,    -0x1.48564d2ebef9p6},
            {0x1.4p-273,                -0x1.48564d2ebef87p6},
            {0x1.3fffffffff6eap-272,    -0x1.47220bf9b501ap6},
            {0x1.4p-272,                -0x1.47220bf9b500dp6},
            {0x1.3fffffffffb3cp-271,    -0x1.45edcac4ab09ap6},
            {0x1.4p-271,                -0x1.45edcac4ab093p6},
            {0x1.3fffffffffd65p-270,    -0x1.44b9898fa111cp6},
            {0x1.4p-270,                -0x1.44b9898fa1119p6},
            {0x1.3fffffffffa85p-269,    -0x1.4385485a971a6p6},
            {0x1.4p-269,                -0x1.4385485a9719fp6},
            {0x1.3ffffffffe2c7p-268,    -0x1.425107258d24dp6},
            {0x1.3fffffffffed7p-268,    -0x1.425107258d226p6},
            {0x1.4p-268,                -0x1.425107258d225p6},
            {0x1.3fffffffff916p-267,    -0x1.411cc5f0832b4p6},
            {0x1.4p-267,                -0x1.411cc5f0832abp6},
            {0x1.3fffffffffd68p-266,    -0x1.3fe884bb79334p6},
            {0x1.4p-266,                -0x1.3fe884bb79331p6},
            {0x1.3fffffffffe21p-265,    -0x1.3eb443866f3b9p6},
            {0x1.4p-265,                -0x1.3eb443866f3b7p6},
            {0x1.3fffffffffedap-264,    -0x1.3d8002516543ep6},
            {0x1.4p-264,                -0x1.3d8002516543dp6},
            {0x1.3fffffffffbfap-263,    -0x1.3c4bc11c5b4c9p6},
            {0x1.4p-263,                -0x1.3c4bc11c5b4c3p6},
            {0x1.3fffffffff862p-262,    -0x1.3b177fe751554p6},
            {0x1.4p-262,                -0x1.3b177fe751549p6},
            {0x1.3fffffffffa8bp-261,    -0x1.39e33eb2475d6p6},
            {0x1.4p-261,                -0x1.39e33eb2475cfp6},
            {0x1.3fffffffffeddp-260,    -0x1.38aefd7d3d656p6},
            {0x1.4p-260,                -0x1.38aefd7d3d655p6},
            {0x1.3fffffffffbfdp-259,    -0x1.377abc48336ep6},
            {0x1.4p-259,                -0x1.377abc48336dbp6},
            {0x1.3ffffffffff97p-258,    -0x1.36467b1329762p6},
            {0x1.4p-258,                -0x1.36467b1329761p6},
            {0x1.3fffffffffedfp-257,    -0x1.351239de1f7e8p6},
            {0x1.4p-257,                -0x1.351239de1f7e7p6},
            {0x1.3fffffffffeep-256,     -0x1.33ddf8a91586ep6},
            {0x1.4p-256,                -0x1.33ddf8a91586dp6},
            {0x1.3fffffffffcp-255,      -0x1.32a9b7740b8f8p6},
            {0x1.4p-255,                -0x1.32a9b7740b8f3p6},
            {0x1.3fffffffffd71p-254,    -0x1.3175763f0197cp6},
            {0x1.4p-254,                -0x1.3175763f01979p6},
            {0x1.3fffffffff588p-253,    -0x1.30413509f7a0ep6},
            {0x1.4p-253,                -0x1.30413509f79ffp6},
            {0x1.3fffffffff7b1p-252,    -0x1.2f0cf3d4eda9p6},
            {0x1.4p-252,                -0x1.2f0cf3d4eda85p6},
            {0x1.3fffffffffc03p-251,    -0x1.2dd8b29fe3b1p6},
            {0x1.4p-251,                -0x1.2dd8b29fe3b0bp6},
            {0x1.3fffffffff86bp-250,    -0x1.2ca4716ad9b9cp6},
            {0x1.4p-250,                -0x1.2ca4716ad9b91p6},
            {0x1.3fffffffffcbdp-249,    -0x1.2b703035cfc1cp6},
            {0x1.4p-249,                -0x1.2b703035cfc17p6},
            {0x1.3fffffffffee6p-248,    -0x1.2a3bef00c5c9ep6},
            {0x1.4p-248,                -0x1.2a3bef00c5c9dp6},
            {0x1.3fffffffffc06p-247,    -0x1.2907adcbbbd28p6},
            {0x1.4p-247,                -0x1.2907adcbbbd23p6},
            {0x1.3fffffffffd77p-246,    -0x1.27d36c96b1dacp6},
            {0x1.4p-246,                -0x1.27d36c96b1da9p6},
            {0x1.3fffffffffd78p-245,    -0x1.269f2b61a7e32p6},
            {0x1.4p-245,                -0x1.269f2b61a7e2fp6},
            {0x1.3ffffffffea0bp-244,    -0x1.256aea2c9ded3p6},
            {0x1.3fffffffffee9p-244,    -0x1.256aea2c9deb6p6},
            {0x1.4p-244,                -0x1.256aea2c9deb5p6},
            {0x1.3fffffffffe32p-243,    -0x1.2436a8f793f3ep6},
            {0x1.4p-243,                -0x1.2436a8f793f3bp6},
            {0x1.3ffffffffffa3p-242,    -0x1.230267c289fc2p6},
            {0x1.4p-242,                -0x1.230267c289fc1p6},
            {0x1.3fffffffff591p-241,    -0x1.21ce268d80056p6},
            {0x1.4p-241,                -0x1.21ce268d80047p6},
            {0x1.3fffffffff9e3p-240,    -0x1.2099e558760d6p6},
            {0x1.4p-240,                -0x1.2099e558760cdp6},
            {0x1.3fffffffffc0cp-239,    -0x1.1f65a4236c158p6},
            {0x1.4p-239,                -0x1.1f65a4236c153p6},
            {0x1.3fffffffffd7dp-238,    -0x1.1e3162ee621dcp6},
            {0x1.4p-238,                -0x1.1e3162ee621d9p6},
            {0x1.3fffffffffd7ep-237,    -0x1.1cfd21b958262p6},
            {0x1.4p-237,                -0x1.1cfd21b95825fp6},
            {0x1.3fffffffffeefp-236,    -0x1.1bc8e0844e2e6p6},
            {0x1.4p-236,                -0x1.1bc8e0844e2e5p6},
            {0x1.3fffffffffc0fp-235,    -0x1.1a949f4f4437p6},
            {0x1.4p-235,                -0x1.1a949f4f4436bp6},
            {0x1.3ffffffffffa9p-234,    -0x1.19605e1a3a3f1p6},
            {0x1.4p-234,                -0x1.19605e1a3a3f1p6},
            {0x1.3ffffffffffaap-233,    -0x1.182c1ce530478p6},
            {0x1.4p-233,                -0x1.182c1ce530477p6},
            {0x1.3fffffffff2b7p-232,    -0x1.16f7dbb02651p6},
            {0x1.4p-232,                -0x1.16f7dbb0264fdp6},
            {0x1.3fffffffffef3p-231,    -0x1.15c39a7b1c584p6},
            {0x1.4p-231,                -0x1.15c39a7b1c583p6},
            {0x1.3fffffffff932p-230,    -0x1.148f594612612p6},
            {0x1.4p-230,                -0x1.148f594612609p6},
            {0x1.3fffffffffd84p-229,    -0x1.135b181108692p6},
            {0x1.4p-229,                -0x1.135b18110868fp6},
            {0x1.3fffffffffaa4p-228,    -0x1.1226d6dbfe71cp6},
            {0x1.4p-228,                -0x1.1226d6dbfe715p6},
            {0x1.3fffffffffef6p-227,    -0x1.10f295a6f479cp6},
            {0x1.4p-227,                -0x1.10f295a6f479bp6},
            {0x1.3fffffffffd86p-226,    -0x1.0fbe5471ea824p6},
            {0x1.4p-226,                -0x1.0fbe5471ea821p6},
            {0x1.3fffffffffd87p-225,    -0x1.0e8a133ce08aap6},
            {0x1.4p-225,                -0x1.0e8a133ce08a7p6},
            {0x1.3fffffffffef8p-224,    -0x1.0d55d207d692ep6},
            {0x1.4p-224,                -0x1.0d55d207d692dp6},
            {0x1.3fffffffffef9p-223,    -0x1.0c2190d2cc9b4p6},
            {0x1.4p-223,                -0x1.0c2190d2cc9b3p6},
            {0x1.3fffffffff42fp-222,    -0x1.0aed4f9dc2a4ap6},
            {0x1.4p-222,                -0x1.0aed4f9dc2a39p6},
            {0x1.3fffffffff658p-221,    -0x1.09b90e68b8accp6},
            {0x1.4p-221,                -0x1.09b90e68b8abfp6},
            {0x1.3fffffffffaaap-220,    -0x1.0884cd33aeb4cp6},
            {0x1.4p-220,                -0x1.0884cd33aeb45p6},
            {0x1.3fffffffffefcp-219,    -0x1.07508bfea4bccp6},
            {0x1.4p-219,                -0x1.07508bfea4bcbp6},
            {0x1.3fffffffffc1cp-218,    -0x1.061c4ac99ac56p6},
            {0x1.4p-218,                -0x1.061c4ac99ac51p6},
            {0x1.3fffffffffd8dp-217,    -0x1.04e8099490cdap6},
            {0x1.4p-217,                -0x1.04e8099490cd7p6},
            {0x1.3fffffffffaadp-216,    -0x1.03b3c85f86d65p6},
            {0x1.3fffffffffefep-216,    -0x1.03b3c85f86d5ep6},
            {0x1.4p-216,                -0x1.03b3c85f86d5dp6},
            {0x1.3ffffffffdbbdp-215,    -0x1.027f872a7ce16p6},
            {0x1.3fffffffffeffp-215,    -0x1.027f872a7cde4p6},
            {0x1.4p-215,                -0x1.027f872a7cde3p6},
            {0x1.3fffffffffc1fp-214,    -0x1.014b45f572e6ep6},
            {0x1.4p-214,                -0x1.014b45f572e69p6},
            {0x1.3ffffffffffb9p-213,    -0x1.001704c068efp6},
            {0x1.4p-213,                -0x1.001704c068eefp6},
            {0x1.3fffffffffdbfp-212,    -0x1.fdc58716bdefp5},
            {0x1.4p-212,                -0x1.fdc58716bdeeap5},
            {0x1.3ffffffffff3p-211,     -0x1.fb5d04aca9ff8p5},
            {0x1.4p-211,                -0x1.fb5d04aca9ff6p5},
            {0x1.3ffffffffffe9p-210,    -0x1.f8f4824296102p5},
            {0x1.4p-210,                -0x1.f8f4824296102p5},
            {0x1.3fffffffffd09p-209,    -0x1.f68bffd882216p5},
            {0x1.4p-209,                -0x1.f68bffd88220ep5},
            {0x1.3fffffffffdc2p-208,    -0x1.f4237d6e6e32p5},
            {0x1.4p-208,                -0x1.f4237d6e6e31ap5},
            {0x1.3fffffffff23fp-207,    -0x1.f1bafb045a44cp5},
            {0x1.3fffffffffe7bp-207,    -0x1.f1bafb045a42ap5},
            {0x1.4p-207,                -0x1.f1bafb045a426p5},
            {0x1.3ffffffffffecp-206,    -0x1.ef52789a46532p5},
            {0x1.4p-206,                -0x1.ef52789a46532p5},
            {0x1.3fffffffffdc4p-205,    -0x1.ece9f63032644p5},
            {0x1.4p-205,                -0x1.ece9f6303263ep5},
            {0x1.3ffffffffe211p-204,    -0x1.ea8173c61e79ep5},
            {0x1.3ffffffffff35p-204,    -0x1.ea8173c61e74cp5},
            {0x1.4p-204,                -0x1.ea8173c61e74ap5},
            {0x1.3ffffffffec25p-203,    -0x1.e818f15c0a88ep5},
            {0x1.3fffffffffe7ep-203,    -0x1.e818f15c0a85ap5},
            {0x1.4p-203,                -0x1.e818f15c0a856p5},
            {0x1.3ffffffffdcadp-202,    -0x1.e5b06ef1f69c5p5},
            {0x1.3fffffffffedbp-202,    -0x1.e5b06ef1f6966p5},
            {0x1.4p-202,                -0x1.e5b06ef1f6962p5},
            {0x1.3fffffffffd0fp-201,    -0x1.e347ec87e2a76p5},
            {0x1.4p-201,                -0x1.e347ec87e2a6ep5},
            {0x1.3fffffffffe8p-200,     -0x1.e0df6a1dceb7ep5},
            {0x1.4p-200,                -0x1.e0df6a1dceb7ap5},
            {0x1.3ffffffffff39p-199,    -0x1.de76e7b3bac88p5},
            {0x1.4p-199,                -0x1.de76e7b3bac86p5},
            {0x1.3fffffffffff2p-198,    -0x1.dc0e6549a6d92p5},
            {0x1.4p-198,                -0x1.dc0e6549a6d92p5},
            {0x1.3ffffffffff97p-197,    -0x1.d9a5e2df92eap5},
            {0x1.4p-197,                -0x1.d9a5e2df92e9ep5},
            {0x1.3fffffffffdcbp-196,    -0x1.d73d60757efbp5},
            {0x1.4p-196,                -0x1.d73d60757efaap5},
            {0x1.3ffffffffff3cp-195,    -0x1.d4d4de0b6b0b8p5},
            {0x1.4p-195,                -0x1.d4d4de0b6b0b6p5},
            {0x1.3fffffffffee1p-194,    -0x1.d26c5ba1571c6p5},
            {0x1.4p-194,                -0x1.d26c5ba1571c2p5},
            {0x1.3ffffffffff9ap-193,    -0x1.d003d937432dp5},
            {0x1.4p-193,                -0x1.d003d937432cep5},
            {0x1.3fffffffffdcep-192,    -0x1.cd9b56cd2f3ep5},
            {0x1.4p-192,                -0x1.cd9b56cd2f3dap5},
            {0x1.3fffffffffe87p-191,    -0x1.cb32d4631b4eap5},
            {0x1.4p-191,                -0x1.cb32d4631b4e6p5},
            {0x1.3fffffffffff8p-190,    -0x1.c8ca51f9075f2p5},
            {0x1.4p-190,                -0x1.c8ca51f9075f2p5},
            {0x1.3fffffffffff9p-189,    -0x1.c661cf8ef36fep5},
            {0x1.4p-189,                -0x1.c661cf8ef36fep5},
            {0x1.3fffffffffdd1p-188,    -0x1.c3f94d24df81p5},
            {0x1.4p-188,                -0x1.c3f94d24df80ap5},
            {0x1.3fffffffffe8ap-187,    -0x1.c190cabacb91ap5},
            {0x1.4p-187,                -0x1.c190cabacb916p5},
            {0x1.3ffffffffff43p-186,    -0x1.bf284850b7a24p5},
            {0x1.4p-186,                -0x1.bf284850b7a22p5},
            {0x1.3fffffffffffcp-185,    -0x1.bcbfc5e6a3b2ep5},
            {0x1.4p-185,                -0x1.bcbfc5e6a3b2ep5},
            {0x1.3ffffffffffa1p-184,    -0x1.ba57437c8fc3cp5},
            {0x1.4p-184,                -0x1.ba57437c8fc3ap5},
            {0x1.3fffffffffd79p-183,    -0x1.b7eec1127bd4ep5},
            {0x1.4p-183,                -0x1.b7eec1127bd46p5},
            {0x1.3fffffffffe32p-182,    -0x1.b5863ea867e58p5},
            {0x1.4p-182,                -0x1.b5863ea867e52p5},
            {0x1.3ffffffffe4a7p-181,    -0x1.b31dbc3e53faap5},
            {0x1.3ffffffffffffp-181,    -0x1.b31dbc3e53f5ep5},
            {0x1.4p-181,                -0x1.b31dbc3e53f5ep5},
            {0x1.3fffffffffdd7p-180,    -0x1.b0b539d44007p5},
            {0x1.4p-180,                -0x1.b0b539d44006ap5},
            {0x1.3fffffffffa9bp-179,    -0x1.ae4cb76a2c185p5},
            {0x1.3fffffffffe9p-179,     -0x1.ae4cb76a2c17ap5},
            {0x1.4p-179,                -0x1.ae4cb76a2c177p5},
            {0x1.3ffffffffe3f1p-178,    -0x1.abe43500182d1p5},
            {0x1.3ffffffffff49p-178,    -0x1.abe4350018284p5},
            {0x1.4p-178,                -0x1.abe4350018283p5},
            {0x1.3fffffffffc69p-177,    -0x1.a97bb29604398p5},
            {0x1.4p-177,                -0x1.a97bb2960438fp5},
            {0x1.3fffffffffddap-176,    -0x1.a713302bf04ap5},
            {0x1.4p-176,                -0x1.a713302bf049bp5},
            {0x1.3fffffffffe93p-175,    -0x1.a4aaadc1dc5aap5},
            {0x1.4p-175,                -0x1.a4aaadc1dc5a7p5},
            {0x1.3fffffffff481p-174,    -0x1.a2422b57c86d3p5},
            {0x1.3fffffffffe38p-174,    -0x1.a2422b57c86b8p5},
            {0x1.4p-174,                -0x1.a2422b57c86b3p5},
            {0x1.3fffffffffef1p-173,    -0x1.9fd9a8edb47c2p5},
            {0x1.4p-173,                -0x1.9fd9a8edb47bfp5},
            {0x1.3ffffffffffaap-172,    -0x1.9d712683a08ccp5},
            {0x1.4p-172,                -0x1.9d712683a08cbp5},
            {0x1.3fffffffffddep-171,    -0x1.9b08a4198c9dcp5},
            {0x1.4p-171,                -0x1.9b08a4198c9d7p5},
            {0x1.3ffffffffff4fp-170,    -0x1.98a021af78ae4p5},
            {0x1.4p-170,                -0x1.98a021af78ae3p5},
            {0x1.3fffffffffd27p-169,    -0x1.96379f4564bf6p5},
            {0x1.4p-169,                -0x1.96379f4564befp5},
            {0x1.3fffffffffdep-168,     -0x1.93cf1cdb50dp5},
            {0x1.4p-168,                -0x1.93cf1cdb50cfbp5},
            {0x1.3fffffffffe99p-167,    -0x1.91669a713ce0ap5},
            {0x1.4p-167,                -0x1.91669a713ce07p5},
            {0x1.3ffffffffff52p-166,    -0x1.8efe180728f14p5},
            {0x1.4p-166,                -0x1.8efe180728f13p5},
            {0x1.3fffffffffc72p-165,    -0x1.8c95959d15028p5},
            {0x1.4p-165,                -0x1.8c95959d1501fp5},
            {0x1.3ffffffffffbp-164,     -0x1.8a2d13330112cp5},
            {0x1.4p-164,                -0x1.8a2d13330112bp5},
            {0x1.3fffffffffd88p-163,    -0x1.87c490c8ed23ep5},
            {0x1.4p-163,                -0x1.87c490c8ed237p5},
            {0x1.3fffffffffe9dp-162,    -0x1.855c0e5ed9346p5},
            {0x1.4p-162,                -0x1.855c0e5ed9343p5},
            {0x1.3fffffffffefap-161,    -0x1.82f38bf4c5452p5},
            {0x1.4p-161,                -0x1.82f38bf4c544fp5},
            {0x1.3fffffffffd2ep-160,    -0x1.808b098ab1562p5},
            {0x1.4p-160,                -0x1.808b098ab155bp5},
            {0x1.3fffffffffe9fp-159,    -0x1.7e2287209d66ap5},
            {0x1.4p-159,                -0x1.7e2287209d667p5},
            {0x1.3ffffffffff58p-158,    -0x1.7bba04b689774p5},
            {0x1.4p-158,                -0x1.7bba04b689773p5},
            {0x1.3fffffffff099p-157,    -0x1.7951824c758aap5},
            {0x1.3ffffffffff59p-157,    -0x1.7951824c7588p5},
            {0x1.4p-157,                -0x1.7951824c7587fp5},
            {0x1.3fffffffffd31p-156,    -0x1.76e8ffe261992p5},
            {0x1.4p-156,                -0x1.76e8ffe26198bp5},
            {0x1.3fffffffffdeap-155,    -0x1.74807d784da9cp5},
            {0x1.4p-155,                -0x1.74807d784da97p5},
            {0x1.3fffffffffea3p-154,    -0x1.7217fb0e39ba6p5},
            {0x1.4p-154,                -0x1.7217fb0e39ba3p5},
            {0x1.3fffffffffc7bp-153,    -0x1.6faf78a425cb8p5},
            {0x1.3ffffffffff5cp-153,    -0x1.6faf78a425cbp5},
            {0x1.4p-153,                -0x1.6faf78a425cafp5},
            {0x1.3ffffffffffb9p-152,    -0x1.6d46f63a11dbcp5},
            {0x1.4p-152,                -0x1.6d46f63a11dbbp5},
            {0x1.3fffffffffa54p-151,    -0x1.6ade73cffded6p5},
            {0x1.4p-151,                -0x1.6ade73cffdec7p5},
            {0x1.3fffffffffbc5p-150,    -0x1.6875f165e9fdfp5},
            {0x1.3fffffffffea6p-150,    -0x1.6875f165e9fd6p5},
            {0x1.4p-150,                -0x1.6875f165e9fd3p5},
            {0x1.3ffffffffff5fp-149,    -0x1.660d6efbd60ep5},
            {0x1.4p-149,                -0x1.660d6efbd60dfp5},
            {0x1.3fffffffffdefp-148,    -0x1.63a4ec91c21fp5},
            {0x1.4p-148,                -0x1.63a4ec91c21ebp5},
            {0x1.3fffffffffea8p-147,    -0x1.613c6a27ae2fap5},
            {0x1.4p-147,                -0x1.613c6a27ae2f7p5},
            {0x1.3ffffffffff61p-146,    -0x1.5ed3e7bd9a404p5},
            {0x1.4p-146,                -0x1.5ed3e7bd9a403p5},
            {0x1.3ffffffffff62p-145,    -0x1.5c6b65538651p5},
            {0x1.4p-145,                -0x1.5c6b65538650fp5},
            {0x1.3ffffffffffbfp-144,    -0x1.5a02e2e97261cp5},
            {0x1.4p-144,                -0x1.5a02e2e97261bp5},
            {0x1.3fffffffffcdfp-143,    -0x1.579a607f5e73p5},
            {0x1.4p-143,                -0x1.579a607f5e727p5},
            {0x1.3fffffffffd98p-142,    -0x1.5531de154a83ap5},
            {0x1.4p-142,                -0x1.5531de154a833p5},
            {0x1.3fffffffffe51p-141,    -0x1.52c95bab36944p5},
            {0x1.4p-141,                -0x1.52c95bab3693fp5},
            {0x1.3ffffffffffc2p-140,    -0x1.5060d94122a4cp5},
            {0x1.4p-140,                -0x1.5060d94122a4bp5},
            {0x1.3fffffffffdf6p-139,    -0x1.4df856d70eb5cp5},
            {0x1.4p-139,                -0x1.4df856d70eb57p5},
            {0x1.3fffffffffeafp-138,    -0x1.4b8fd46cfac66p5},
            {0x1.4p-138,                -0x1.4b8fd46cfac63p5},
            {0x1.3ffffffffff68p-137,    -0x1.49275202e6d7p5},
            {0x1.4p-137,                -0x1.49275202e6d6fp5},
            {0x1.3ffffffffffc5p-136,    -0x1.46becf98d2e7bp5},
            {0x1.4p-136,                -0x1.46becf98d2e7bp5},
            {0x1.3fffffffffdf9p-135,    -0x1.44564d2ebef8cp5},
            {0x1.4p-135,                -0x1.44564d2ebef87p5},
            {0x1.3fffffffffeb2p-134,    -0x1.41edcac4ab096p5},
            {0x1.4p-134,                -0x1.41edcac4ab093p5},
            {0x1.3ffffffffff6bp-133,    -0x1.3f85485a971ap5},
            {0x1.4p-133,                -0x1.3f85485a9719fp5},
            {0x1.3ffffffffe699p-132,    -0x1.3d1cc5f0832f2p5},
            {0x1.3ffffffffff1p-132,     -0x1.3d1cc5f0832aep5},
            {0x1.4p-132,                -0x1.3d1cc5f0832abp5},
            {0x1.3ffffffffffc9p-131,    -0x1.3ab443866f3b8p5},
            {0x1.4p-131,                -0x1.3ab443866f3b7p5},
            {0x1.3fffffffffda1p-130,    -0x1.384bc11c5b4cap5},
            {0x1.4p-130,                -0x1.384bc11c5b4c3p5},
            {0x1.3ffffffffed15p-129,    -0x1.35e33eb247604p5},
            {0x1.3ffffffffff6ep-129,    -0x1.35e33eb2475dp5},
            {0x1.4p-129,                -0x1.35e33eb2475cfp5},
            {0x1.3ffffffffdd9dp-128,    -0x1.337abc483373bp5},
            {0x1.3fffffffffd46p-128,    -0x1.337abc48336e2p5},
            {0x1.4p-128,                -0x1.337abc48336dbp5},
            {0x1.3fffffffffdffp-127,    -0x1.311239de1f7ecp5},
            {0x1.4p-127,                -0x1.311239de1f7e7p5},
            {0x1.3fffffffff391p-126,    -0x1.2ea9b7740b916p5},
            {0x1.3ffffffffff7p-126,     -0x1.2ea9b7740b8f4p5},
            {0x1.4p-126,                -0x1.2ea9b7740b8f3p5},
            {0x1.3ffffffffff71p-125,    -0x1.2c413509f7ap5},
            {0x1.4p-125,                -0x1.2c413509f79ffp5},
            {0x1.3fffffffffc91p-124,    -0x1.29d8b29fe3b14p5},
            {0x1.4p-124,                -0x1.29d8b29fe3b0bp5},
            {0x1.3fffffffffd4ap-123,    -0x1.27703035cfc1ep5},
            {0x1.4p-123,                -0x1.27703035cfc17p5},
            {0x1.3fffffffffe03p-122,    -0x1.2507adcbbbd28p5},
            {0x1.4p-122,                -0x1.2507adcbbbd23p5},
            {0x1.3fffffffffebcp-121,    -0x1.229f2b61a7e32p5},
            {0x1.4p-121,                -0x1.229f2b61a7e2fp5},
            {0x1.3ffffffffff19p-120,    -0x1.2036a8f793f3ep5},
            {0x1.4p-120,                -0x1.2036a8f793f3bp5},
            {0x1.3fffffffffd4dp-119,    -0x1.1dce268d8004ep5},
            {0x1.4p-119,                -0x1.1dce268d80047p5},
            {0x1.3fffffffffebep-118,    -0x1.1b65a4236c156p5},
            {0x1.4p-118,                -0x1.1b65a4236c153p5},
            {0x1.3ffffffffff77p-117,    -0x1.18fd21b95826p5},
            {0x1.4p-117,                -0x1.18fd21b95825fp5},
            {0x1.3ffffffffffd4p-116,    -0x1.16949f4f4436bp5},
            {0x1.4p-116,                -0x1.16949f4f4436bp5},
            {0x1.3fffffffffe08p-115,    -0x1.142c1ce53047cp5},
            {0x1.4p-115,                -0x1.142c1ce530477p5},
            {0x1.3fffffffffe09p-114,    -0x1.11c39a7b1c588p5},
            {0x1.4p-114,                -0x1.11c39a7b1c583p5},
            {0x1.3fffffffffec2p-113,    -0x1.0f5b181108692p5},
            {0x1.4p-113,                -0x1.0f5b18110868fp5},
            {0x1.3ffffffffff7bp-112,    -0x1.0cf295a6f479cp5},
            {0x1.4p-112,                -0x1.0cf295a6f479bp5},
            {0x1.3ffffffffffd8p-111,    -0x1.0a8a133ce08a8p5},
            {0x1.4p-111,                -0x1.0a8a133ce08a7p5},
            {0x1.3fffffffffa73p-110,    -0x1.082190d2cc9c2p5},
            {0x1.4p-110,                -0x1.082190d2cc9b3p5},
            {0x1.3fffffffffec5p-109,    -0x1.05b90e68b8ac2p5},
            {0x1.4p-109,                -0x1.05b90e68b8abfp5},
            {0x1.3ffffffffedddp-108,    -0x1.03508bfea4bfep5},
            {0x1.3ffffffffff7ep-108,    -0x1.03508bfea4bccp5},
            {0x1.4p-108,                -0x1.03508bfea4bcbp5},
            {0x1.3ffffffffde65p-107,    -0x1.00e8099490d35p5},
            {0x1.3fffffffffe0ep-107,    -0x1.00e8099490cdcp5},
            {0x1.4p-107,                -0x1.00e8099490cd7p5},
            {0x1.3ffffffffed3dp-106,    -0x1.fcff0e54f9c2fp4},
            {0x1.3fffffffffe82p-106,    -0x1.fcff0e54f9bcep4},
            {0x1.4p-106,                -0x1.fcff0e54f9bc6p4},
            {0x1.3fffffffffff3p-105,    -0x1.f82e0980d1ddep4},
            {0x1.4p-105,                -0x1.f82e0980d1ddep4},
            {0x1.3ffffffffff98p-104,    -0x1.f35d04aca9ff8p4},
            {0x1.4p-104,                -0x1.f35d04aca9ff6p4},
            {0x1.3ffffffffe3b7p-103,    -0x1.ee8bffd8822abp4},
            {0x1.3ffffffffffc7p-103,    -0x1.ee8bffd88221p4},
            {0x1.4p-103,                -0x1.ee8bffd88220ep4},
            {0x1.3fffffffffdcdp-102,    -0x1.e9bafb045a432p4},
            {0x1.4p-102,                -0x1.e9bafb045a426p4},
            {0x1.3ffffffffff3ep-101,    -0x1.e4e9f63032642p4},
            {0x1.4p-101,                -0x1.e4e9f6303263ep4},
            {0x1.3ffffffffe301p-100,    -0x1.e018f15c0a8f8p4},
            {0x1.3fffffffffff7p-100,    -0x1.e018f15c0a856p4},
            {0x1.4p-100,                -0x1.e018f15c0a856p4},
            {0x1.3fffffffffd73p-99,     -0x1.db47ec87e2a7cp4},
            {0x1.4p-99,                 -0x1.db47ec87e2a6ep4},
            {0x1.3ffffffffdd9dp-98,     -0x1.d676e7b3bad46p4},
            {0x1.3fffffffffee4p-98,     -0x1.d676e7b3bac8cp4},
            {0x1.4p-98,                 -0x1.d676e7b3bac86p4},
            {0x1.3ffffffffff9dp-97,     -0x1.d1a5e2df92eap4},
            {0x1.4p-97,                 -0x1.d1a5e2df92e9ep4},
            {0x1.3fffffffffffap-96,     -0x1.ccd4de0b6b0b6p4},
            {0x1.4p-96,                 -0x1.ccd4de0b6b0b6p4},
            {0x1.3ffffffffffcdp-95,     -0x1.c803d937432dp4},
            {0x1.4p-95,                 -0x1.c803d937432cep4},
            {0x1.3ffffffffffcep-94,     -0x1.c332d4631b4e8p4},
            {0x1.4p-94,                 -0x1.c332d4631b4e6p4},
            {0x1.3fffffffffe8cp-93,     -0x1.be61cf8ef3706p4},
            {0x1.4p-93,                 -0x1.be61cf8ef36fep4},
            {0x1.3fffffffff983p-92,     -0x1.b990cabacb93ap4},
            {0x1.3ffffffffff45p-92,     -0x1.b990cabacb91ap4},
            {0x1.4p-92,                 -0x1.b990cabacb916p4},
            {0x1.3fffffffffffep-91,     -0x1.b4bfc5e6a3b2ep4},
            {0x1.4p-91,                 -0x1.b4bfc5e6a3b2ep4},
            {0x1.3fffffffffdd6p-90,     -0x1.afeec1127bd52p4},
            {0x1.4p-90,                 -0x1.afeec1127bd46p4},
            {0x1.3fffffffffeebp-89,     -0x1.ab1dbc3e53f64p4},
            {0x1.4p-89,                 -0x1.ab1dbc3e53f5ep4},
            {0x1.3ffffffffffa4p-88,     -0x1.a64cb76a2c178p4},
            {0x1.4p-88,                 -0x1.a64cb76a2c177p4},
            {0x1.3fffffffffa9bp-87,     -0x1.a17bb296043acp4},
            {0x1.3fffffffffd7cp-87,     -0x1.a17bb2960439cp4},
            {0x1.4p-87,                 -0x1.a17bb2960438fp4},
            {0x1.3fffffffffe91p-86,     -0x1.9caaadc1dc5aep4},
            {0x1.3ffffffffffc5p-86,     -0x1.9caaadc1dc5a8p4},
            {0x1.3fffffffffe36p-85,     -0x1.97d9a8edb47c8p4},
            {0x1.3ffffffffffa7p-84,     -0x1.9308a4198c9d8p4},
            {0x1.3ffffffffff7ap-83,     -0x1.8e379f4564bf2p4},
            {0x1.3ffffffffffd7p-82,     -0x1.89669a713ce08p4},
            {0x1.3fffffffffe95p-81,     -0x1.8495959d15026p4},
            {0x1.3ffffffffff4ep-80,     -0x1.7fc490c8ed23ap4},
            {0x1.3ffffffffeafbp-79,     -0x1.7af38bf4c54c3p4},
            {0x1.3ffffffffffd9p-79,     -0x1.7af38bf4c544fp4},
            {0x1.3fffffffffe3bp-78,     -0x1.762287209d67p4},
            {0x1.3ffffffffde65p-77,     -0x1.7151824c7593ap4},
            {0x1.3fffffffffef4p-77,     -0x1.7151824c75884p4},
            {0x1.3ffffffffffadp-76,     -0x1.6c807d784da98p4},
            {0x1.3fffffffffb8bp-75,     -0x1.67af78a425cc7p4},
            {0x1.3ffffffffffaep-75,     -0x1.67af78a425cbp4},
            {0x1.3ffffffffffddp-74,     -0x1.62de73cffdec8p4},
            {0x1.3fffffffffe3fp-73,     -0x1.5e0d6efbd60e8p4},
            {0x1.3ffffffffff54p-72,     -0x1.593c6a27ae2fap4},
            {0x1.3ffffffffffb1p-71,     -0x1.546b65538651p4},
            {0x1.3fffffffff571p-70,     -0x1.4f9a607f5e762p4},
            {0x1.3fffffffffde5p-70,     -0x1.4f9a607f5e732p4},
            {0x1.3fffffffffe9ep-69,     -0x1.4ac95bab36946p4},
            {0x1.3ffffffffffb3p-68,     -0x1.45f856d70eb58p4},
            {0x1.3ffffffffe1d7p-67,     -0x1.41275202e6e16p4},
            {0x1.3ffffffffffb4p-67,     -0x1.41275202e6d7p4},
            {0x1.3fffffffffe44p-66,     -0x1.3c564d2ebef9p4},
            {0x1.3fffffffffd5fp-65,     -0x1.3785485a971aep4},
            {0x1.3fffffffffaadp-64,     -0x1.32b443866f3d5p4},
            {0x1.3fffffffffefep-64,     -0x1.32b443866f3bcp4},
            {0x1.3ffffffffeb35p-63,     -0x1.2de33eb247643p4},
            {0x1.3ffffffffffb7p-63,     -0x1.2de33eb2475dp4},
            {0x1.3ffffffffffe6p-62,     -0x1.291239de1f7e8p4},
            {0x1.3fffffffffdecp-61,     -0x1.24413509f7a0ap4},
            {0x1.3ffffffffff5dp-60,     -0x1.1f703035cfc1ap4},
            {0x1.3ffffffffffbap-59,     -0x1.1a9f2b61a7e3p4},
            {0x1.3fffffffffd92p-58,     -0x1.15ce268d80054p4},
            {0x1.3ffffffffff03p-57,     -0x1.10fd21b958264p4},
            {0x1.3ffffffffe6e9p-56,     -0x1.0c2c1ce530503p4},
            {0x1.3ffffffffffbcp-56,     -0x1.0c2c1ce530478p4},
            {0x1.3ffffffffe2c7p-55,     -0x1.075b181108731p4},
            {0x1.3ffffffffff61p-55,     -0x1.075b181108692p4},
            {0x1.3ffffffffffecp-54,     -0x1.028a133ce08a8p4},
            {0x1.3fffffffff14fp-53,     -0x1.fb721cd171621p3},
            {0x1.3fffffffffff8p-53,     -0x1.fb721cd17157ep3},
            {0x1.3ffffffffdd9dp-52,     -0x1.f1d0132921b2dp3},
            {0x1.3ffffffffffe2p-52,     -0x1.f1d01329219bp3},
            {0x1.3ffffffffebebp-51,     -0x1.e82e0980d1ebdp3},
            {0x1.3fffffffffe2dp-51,     -0x1.e82e0980d1df2p3},
            {0x1.3ffffffffff7p-50,      -0x1.de8bffd882214p3},
            {0x1.3fffffffffaadp-49,     -0x1.d4e9f6303267ap3},
            {0x1.3ffffffffffcdp-49,     -0x1.d4e9f6303264p3},
            {0x1.3fffffffffd77p-48,     -0x1.cb47ec87e2a8ap3},
            {0x1.3fffffffffda5p-48,     -0x1.cb47ec87e2a89p3},
            {0x1.3ffffffffff16p-47,     -0x1.c1a5e2df92ea8p3},
            {0x1.3fffffffff983p-46,     -0x1.b803d93743316p3},
            {0x1.3ffffffffffcfp-46,     -0x1.b803d937432dp3},
            {0x1.3ffffffffffa2p-45,     -0x1.ae61cf8ef3702p3},
            {0x1.3fffffffffbc5p-44,     -0x1.a4bfc5e6a3b5ep3},
            {0x1.3ffffffffffffp-44,     -0x1.a4bfc5e6a3b2ep3},
            {0x1.3fffffffffdd7p-43,     -0x1.9b1dbc3e53f76p3},
            {0x1.3fffffffffebep-42,     -0x1.917bb2960439cp3},
            {0x1.3ffffffffe3f1p-41,     -0x1.87d9a8edb48f7p3},
            {0x1.3ffffffffff77p-41,     -0x1.87d9a8edb47c4p3},
            {0x1.3fffffffff23fp-40,     -0x1.7e379f4564c87p3},
            {0x1.3ffffffffffbdp-40,     -0x1.7e379f4564bf2p3},
            {0x1.3fffffffffe36p-39,     -0x1.7495959d15032p3},
            {0x1.3ffffffffecdbp-38,     -0x1.6af38bf4c5523p3},
            {0x1.3ffffffffffa7p-38,     -0x1.6af38bf4c5452p3},
            {0x1.3ffffffffffd6p-37,     -0x1.6151824c7588p3},
            {0x1.3ffffffffffeep-36,     -0x1.57af78a425cafp3},
            {0x1.3ffffffffdcadp-35,     -0x1.4e0d6efbd6268p3},
            {0x1.3ffffffffffefp-35,     -0x1.4e0d6efbd60ep3},
            {0x1.3fffffffffe68p-34,     -0x1.446b65538652p3},
            {0x1.3ffffffffeafbp-33,     -0x1.3ac95bab36a28p3},
            {0x1.3ffffffffffd9p-33,     -0x1.3ac95bab3694p3},
            {0x1.3ffffffffffdap-32,     -0x1.31275202e6d7p3},
            {0x1.3fffffffffcb5p-31,     -0x1.2785485a971c4p3},
            {0x1.3ffffffffffc4p-31,     -0x1.2785485a971a2p3},
            {0x1.3fffffffffec7p-30,     -0x1.1de33eb2475dcp3},
            {0x1.3ffffffffffaep-29,     -0x1.14413509f7a02p3},
            {0x1.3ffffffffd9a3p-28,     -0x1.0a9f2b61a7fd9p3},
            {0x1.3ffffffffffddp-28,     -0x1.0a9f2b61a7e3p3},
            {0x1.3fffffffff32fp-27,     -0x1.00fd21b9582edp3},
            {0x1.3fffffffffe6dp-27,     -0x1.00fd21b95827p3},
            {0x1.3ffffffffffe4p-26,     -0x1.eeb6302210d2p2},
            {0x1.3ffffffffeb35p-25,     -0x1.db721cd17174dp2},
            {0x1.3fffffffffffcp-25,     -0x1.db721cd17157ep2},
            {0x1.3ffffffffde65p-24,     -0x1.c82e0980d20cap2},
            {0x1.3fffffffffeafp-24,     -0x1.c82e0980d1dfcp2},
            {0x1.3ffffffffed3dp-23,     -0x1.b4e9f630327ep2},
            {0x1.3ffffffffff7fp-23,     -0x1.b4e9f6303264ap2},
            {0x1.3ffffffffffdcp-22,     -0x1.a1a5e2df92ea2p2},
            {0x1.3ffffffffe91bp-21,     -0x1.8e61cf8ef38fbp2},
            {0x1.3ffffffffffffp-21,     -0x1.8e61cf8ef36fep2},
            {0x1.3fffffffffeb2p-20,     -0x1.7b1dbc3e53f7bp2},
            {0x1.3ffffffffe3b7p-19,     -0x1.67d9a8edb4a33p2},
            {0x1.3ffffffffff3dp-19,     -0x1.67d9a8edb47dp2},
            {0x1.3ffffffffe6e9p-18,     -0x1.5495959d1524dp2},
            {0x1.3fffffffffff6p-18,     -0x1.5495959d1502p2},
            {0x1.3ffffffffdb5bp-17,     -0x1.4151824c75badp2},
            {0x1.3ffffffffffebp-17,     -0x1.4151824c7588p2},
            {0x1.3ffffffffe301p-16,     -0x1.2e0d6efbd6364p2},
            {0x1.3ffffffffff11p-16,     -0x1.2e0d6efbd60f3p2},
            {0x1.3fffffffff14fp-15,     -0x1.1ac95bab36a85p2},
            {0x1.3ffffffffff57p-15,     -0x1.1ac95bab3694ep2},
            {0x1.3fffffffff481p-14,     -0x1.0785485a9729fp2},
            {0x1.3fffffffffff9p-14,     -0x1.0785485a971ap2},
            {0x1.3fffffffff571p-13,     -0x1.e8826a13ef5d4p1},
            {0x1.3fffffffffff1p-13,     -0x1.e8826a13ef4p1},
            {0x1.3fffffffffc7bp-12,     -0x1.c1fa4372b055ap1},
            {0x1.3ffffffffffcfp-12,     -0x1.c1fa4372b04c6p1},
            {0x1.3fffffffff189p-11,     -0x1.9b721cd171802p1},
            {0x1.3fffffffffffep-11,     -0x1.9b721cd17157ep1},
            {0x1.3ffffffffe211p-10,     -0x1.74e9f63032b72p1},
            {0x1.3fffffffffff3p-10,     -0x1.74e9f6303264p1},
            {0x1.3fffffffffbc5p-9,      -0x1.4e61cf8ef37bbp1},
            {0x1.3fffffffffff4p-9,      -0x1.4e61cf8ef37p1},
            {0x1.3fffffffff391p-8,      -0x1.27d9a8edb49e8p1},
            {0x1.3ffffffffffe9p-8,      -0x1.27d9a8edb47c2p1},
            {0x1.3fffffffffaadp-7,      -0x1.0151824c7596cp1},
            {0x1.3ffffffffffeap-7,      -0x1.0151824c75882p1},
            {0x1.3fffffffffc7bp-6,      -0x1.b592b7566d3b6p0},
            {0x1.3fffffffffff2p-6,      -0x1.b592b7566d282p0},
            {0x1.3fffffffffda5p-5,      -0x1.68826a13ef4dp0},
            {0x1.3fffffffffff6p-5,      -0x1.68826a13ef402p0},
            {0x1.3fffffffffb8bp-4,      -0x1.1b721cd17170ap0},
            {0x1.3ffffffffffffp-4,      -0x1.1b721cd17157ep0},
            {0x1.3fffffffffcb5p-3,      -0x1.9cc39f1de7047p-1},
            {0x1.3fffffffffff2p-3,      -0x1.9cc39f1de6e06p-1},
            {0x1.3fffffffffc7bp-2,      -0x1.02a30498eb36fp-1},
            {0x1.4p-2,                  -0x1.02a30498eb0fep-1},
            {0x1.3fffffffffda5p-1,      -0x1.a209a84fbd684p-3},
            {0x1.3fffffffffffcp-1,      -0x1.a209a84fbd002p-3},
            {0x1.3fffffffffff3p0,       0x1.8cf18388647c5p-4},
            {0x1.3fffffffffffdp0,       0x1.8cf18388647fbp-4},
            {0x1.3ffffffffff7bp1,       0x1.977d95ec10b4ap-2},
            {0x1.3fffffffffffcp1,       0x1.977d95ec10bfcp-2},
            {0x1.3ffffffffff2bp2,       0x1.65df657b0426dp-1},
            {0x1.3fffffffffff3p2,       0x1.65df657b042f8p-1},
            {0x1.3fffffffffc83p4,       0x1.4d104d427dd4ap0},
            {0x1.3fffffffffff1p4,       0x1.4d104d427de7ap0},
            {0x1.4p4,                   0x1.4d104d427de8p0},
            {0x1.3ffffffffff2bp5,       0x1.9a209a84fbcb6p0},
            {0x1.4p5,                   0x1.9a209a84fbdp0},
            {0x1.3fffffffffd23p6,       0x1.e730e7c779a81p0},
            {0x1.3fffffffffffep6,       0x1.e730e7c779b7ep0},
            {0x1.4p6,                   0x1.e730e7c779b7fp0},
            {0x1.3ffffffffece3p7,       0x1.1a209a84fb9aep1},
            {0x1.4p7,                   0x1.1a209a84fbdp1},
            {0x1.3fffffffffeb3p8,       0x1.40a8c1263ac06p1},
            {0x1.3ffffffffff71p8,       0x1.40a8c1263ac26p1},
            {0x1.4p8,                   0x1.40a8c1263ac3fp1},
            {0x1.3fffffffffe3bp9,       0x1.6730e7c779b31p1},
            {0x1.3fffffffffffcp9,       0x1.6730e7c779b7ep1},
            {0x1.4p9,                   0x1.6730e7c779b7fp1},
            {0x1.3fffffffff657p10,      0x1.8db90e68b8912p1},
            {0x1.3ffffffffff67p10,      0x1.8db90e68b8aa4p1},
            {0x1.4p10,                  0x1.8db90e68b8abfp1},
            {0x1.3fffffffff8ffp11,      0x1.b4413509f78c8p1},
            {0x1.3ffffffffffecp11,      0x1.b4413509f79fcp1},
            {0x1.4p11,                  0x1.b4413509f79ffp1},
            {0x1.3fffffffffd23p12,      0x1.dac95bab368cp1},
            {0x1.3ffffffffff5dp12,      0x1.dac95bab36922p1},
            {0x1.4p12,                  0x1.dac95bab3693fp1},
            {0x1.3fffffffffe13p13,      0x1.00a8c1263ac15p2},
            {0x1.3fffffffffee2p13,      0x1.00a8c1263ac26p2},
            {0x1.4p13,                  0x1.00a8c1263ac3fp2},
            {0x1.3fffffffff193p14,      0x1.13ecd476da29fp2},
            {0x1.3ffffffffffc9p14,      0x1.13ecd476da3dap2},
            {0x1.4p14,                  0x1.13ecd476da3dfp2},
            {0x1.3fffffffff5b7p15,      0x1.2730e7c779a9bp2},
            {0x1.3ffffffffff1dp15,      0x1.2730e7c779b6cp2},
            {0x1.4p15,                  0x1.2730e7c779b7fp2},
            {0x1.3ffffffffec2fp16,      0x1.3a74fb1819167p2},
            {0x1.3ffffffffffedp16,      0x1.3a74fb181931ep2},
            {0x1.4p16,                  0x1.3a74fb181931fp2},
            {0x1.3fffffffffb07p17,      0x1.4db90e68b8a51p2},
            {0x1.3fffffffffecep17,      0x1.4db90e68b8aa4p2},
            {0x1.4p17,                  0x1.4db90e68b8abfp2},
            {0x1.3ffffffffff2bp18,      0x1.60fd21b95824dp2},
            {0x1.3ffffffffffb5p18,      0x1.60fd21b958258p2},
            {0x1.4p18,                  0x1.60fd21b95825fp2},
            {0x1.3ffffffffffe4p19,      0x1.74413509f79fcp2},
            {0x1.4p19,                  0x1.74413509f79ffp2},
            {0x1.3fffffffff6cfp20,      0x1.8785485a970d3p2},
            {0x1.3ffffffffffd9p20,      0x1.8785485a9719cp2},
            {0x1.4p20,                  0x1.8785485a9719fp2},
            {0x1.3ffffffffed47p21,      0x1.9ac95bab3679fp2},
            {0x1.3fffffffffebap21,      0x1.9ac95bab36922p2},
            {0x1.4p21,                  0x1.9ac95bab3693fp2},
            {0x1.3fffffffff16bp22,      0x1.ae0d6efbd5f9bp2},
            {0x1.3ffffffffff8ap22,      0x1.ae0d6efbd60d4p2},
            {0x1.4p22,                  0x1.ae0d6efbd60dfp2},
            {0x1.3ffffffffe667p23,      0x1.c151824c75646p2},
            {0x1.3fffffffffffep23,      0x1.c151824c7587ep2},
            {0x1.4p23,                  0x1.c151824c7587fp2},
            {0x1.3fffffffffff3p24,      0x1.d495959d1501ep2},
            {0x1.4p24,                  0x1.d495959d1501fp2},
            {0x1.3fffffffffadfp25,      0x1.e7d9a8edb474dp2},
            {0x1.3fffffffffebdp25,      0x1.e7d9a8edb47a2p2},
            {0x1.4p25,                  0x1.e7d9a8edb47bfp2},
        };

        for (double[] testCase: testCases)
            failures+=testLog10Case(testCase[0], testCase[1]);

        return failures;
    }

    public static void main(String [] argv) {
        int failures = 0;

        failures += testLog10();

        if (failures > 0) {
            System.err.println("Testing log10 incurred "
                               + failures + " failures.");
            throw new RuntimeException();
        }
    }
}
