package org.hamcrest.core;

import org.hamcrest.BaseMatcher;
import org.hamcrest.Matcher;
import org.hamcrest.Description;
import org.hamcrest.Factory;

import java.util.Arrays;

/**
 * Calculates the logical disjunction of two matchers. Evaluation is
 * shortcut, so that the second matcher is not called if the first
 * matcher returns <code>true</code>.
 */
public class AnyOf<T> extends BaseMatcher<T> {

    private final Iterable<Matcher<T>> matchers;

    public AnyOf(Iterable<Matcher<T>> matchers) {
        this.matchers = matchers;
    }

    public boolean matches(Object o) {
        for (Matcher<T> matcher : matchers) {
            if (matcher.matches(o)) {
                return true;
            }
        }
        return false;
    }

    public void describeTo(Description description) {
        description.appendText("(");
        boolean seenFirst = false;
        for (Matcher<T> matcher : matchers) {
            if (seenFirst) {
                description.appendText(" or ");
            } else {
                seenFirst = true;
            }
            matcher.describeTo(description);
        }
        description.appendText(")");
    }

    @Factory
    public static <T> Matcher<T> anyOf(Matcher<T>... matchers) {
        return anyOf(Arrays.asList(matchers));
    }

    @Factory
    public static <T> Matcher<T> anyOf(Iterable<Matcher<T>> matchers) {
        return new AnyOf<T>(matchers);
    }
}
