/**
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 */

webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.calendar");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.calendar
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the calendar widget.
 * @constructor This function is used to construct a calendar widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.calendar", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    widgetName: "calendar" // Required for theme properties.
});

/**
 * Helper function to add a day link in a cell.
 *
 * @param {Node} rowNodeClone
 * @param {Object} day
 * @param {String} id
 * @param {String} className
 * @param {boolean} setFocus
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.addDayLink = function(rowNodeClone, day, 
        id, className, setFocus) {
    // Clone <td> and <a> elements. 
    var colNodeClone = this.dayColumnContainer.cloneNode(false);
    rowNodeClone.appendChild(colNodeClone);    
    var linkNodeClone = this.dayLinkContainer.cloneNode(false);            
    colNodeClone.appendChild(linkNodeClone);
    
    // Format the date.      
    var formattedDate = this.formatDate(day.getMonth() + 1, day.getDate(), day.getFullYear()); 
  
    // set the link's properties for this day.
    linkNodeClone.title = formattedDate;
    linkNodeClone.id = id;
    linkNodeClone.href = "#";
    linkNodeClone.className = className;

    // NOTE: If you set this value manually, text must be HTML escaped.
    this.widget.addFragment(linkNodeClone, "" + day.getDate());

    var widgetId = this.id;
    linkNodeClone.onclick = function() { 
        webui.suntheme4_2.dijit.byId(widgetId).daySelected(formattedDate); 
        return false;
    };  
    
    // If the setFocus is set to true, then when you tab out of the linkNode,
    // the focus should go to the close button. 
    if (setFocus) {
        var node = webui.suntheme4_2.dijit.byId(linkNodeClone.id);        
        linkNodeClone.onkeydown = function(event) {
            var widget = webui.suntheme4_2.dijit.byId(widgetId);
            
            // Get hold of the close button and set focus on it.
            var evt = (event) ? event : ((window.event) ? window.event : null);
            if (evt.keyCode == 9) {
                var elem = document.getElementById(widget.closeButtonLink.id);
                if (elem != null) {
                    if (elem.focus) {
                        elem.focus();
                    }
                    if (evt.preventDefault) {
                        evt.preventDefault();
                    } else {
                        evt.returnValue = false;
                    }
                    return false;                                  
                }
            }
            return true;
        };
    }
    return true;
};

/**
 * Helper function to add days in the month -- week data rows.
 *
 * @param {Object} currentValue The current value of the text field.
 * @param {boolean} initialize Flag indicating to initialze the year and month menus
 * with the current value. The value is true only when the calendar is opened.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.addDaysInMonth = function(currentValue, initialize) {
    // Date representing a day in a month.
    var day;    
    // Number of columns in a row.
    var column = 0;
    // Row number.    
    var rowNum = 0;    
    // Today's day.
    var today = 0;
    // Selected date.
    var selected = 0;     
    // Day link number
    var linkNum = 0; 
    // Prefix used for a day link id.
    var id = this.id + "_link:";
    // Day link id. ie, id + linkNum.
    var linkId;
    // One day in milliseconds -- 1000 * 60 * 60 * 24    
    var oneDayInMs = 86400000;     

    var todayDate = new Date();
    var todayYear = todayDate.getFullYear();
    var todayMonth = todayDate.getMonth() + 1;
    var todayDay = todayDate.getDate();                  
    
    // selectedYear, selectedMonth, selectedDay:
    // The date to show as highlighted (currentValue) provided
    // that the user is viewing that month and year.
    var selectedYear = null;
    var selectedMonth = null;
    var selectedDay = null;
    if (currentValue != null) {
        selectedYear = currentValue.getFullYear();
        selectedMonth = currentValue.getMonth() + 1;
        selectedDay = currentValue.getDate();
    }
    
    // Get month and year menu widgets.
    var monthMenuWidget = webui.suntheme4_2.dijit.byId(this.monthMenu.id);        
    var yearMenuWidget = webui.suntheme4_2.dijit.byId(this.yearMenu.id);
    if (monthMenuWidget == null || yearMenuWidget == null) {
        return false;
    }
               
    if (initialize) {
         // Set showMonth as selected in the month menu
	 // Set showYear as selected in the year menu
         // Use todayMonth and todayYear if currentValue is null.
	 var showMonth = todayMonth;
	 var showYear = todayYear;
	 if (currentValue != null) {
             // We have a currentValue, so use that for showMonth and showYear
             showMonth = selectedMonth;
	     showYear = selectedYear;
         }         
         this.setLimitedSelectedValue(monthMenuWidget.getSelectElement(), showMonth);
         this.setLimitedSelectedValue(yearMenuWidget.getSelectElement(), showYear);
    }
    
    var month = parseInt(monthMenuWidget.getSelectedValue());
    var year = parseInt(yearMenuWidget.getSelectedValue());
    
    //set selected
    if (currentValue != null && selectedYear == year && selectedMonth == month) {
        selected = selectedDay;
    }
        
    //set today
    if (todayYear == year && todayMonth == month) {
        today = todayDay;
    }
    
    // Add first week data row.
    var rowNodeClone = this.weekRowContainer.cloneNode(false);
    this.tbodyContainer.appendChild(rowNodeClone); 
    rowNodeClone.id = this.id + ":row" + rowNum;
    
    // Convert to javascript month numbering.
    month--;
    
    // Calculate the first of the main month to display in "first" row.
    var first = new Date(year, month, 1);                         
    var firstDay = first.getDay();    
    var className = this.theme.getClassName("DATE_TIME_OTHER_LINK");
    if (firstDay == this.firstDayOfWeek - 1) {
        // First cell on first row is the first of the current month
        day = first;
    } else {
        // First cell on first row is in previous month.
        var backDays = (firstDay - (this.firstDayOfWeek - 1) + 7) % 7;        
        
        // Calculate the date of first cell on first row in previous month.
        day = new Date(first.getTime() - backDays * oneDayInMs);        
        
        // Generate start of first row up to first of month
        while (day.getMonth() !=  first.getMonth()) {
            linkId = id + linkNum;
            this.addDayLink(rowNodeClone, day, linkId, className);
            day = new Date(day.getTime() + oneDayInMs);
            column++;
            linkNum++;            
        }
    }

    // Add any cells in the first row of the main month.
    while (column < 7) {
        // Set appropriate class name.
        if (day.getDate() == selected) {
            className = this.theme.getClassName("DATE_TIME_BOLD_LINK");
        } else if (day.getDate() == today) {
            className = this.theme.getClassName("DATE_TIME_TODAY_LINK");
        } else {
           className = this.theme.getClassName("DATE_TIME_LINK");
        }
            
        linkId = id + linkNum;
        this.addDayLink(rowNodeClone, day, linkId, className);        
        day = new Date(day.getTime() + oneDayInMs);
        column++;
        linkNum++;
    } 
    
    // This variable is used to decide whether the focus should be set
    // on the calendar's close button when tabbing    
    var setFocus = false;            
    
    // Add intermediate rows
    while (day.getDate() != 1) {
        rowNum++;
        // Clone a <tr> node
        rowNodeClone = this.weekRowContainer.cloneNode(false);
        this.tbodyContainer.appendChild(rowNodeClone); 
        rowNodeClone.id = this.id + ":row" + rowNum;

        column = 0;
        while (column < 7 && day.getDate() != 1) {            
            // Set appropriate class name.
            if (day.getDate() == selected) {
                className = this.theme.getClassName("DATE_TIME_BOLD_LINK");
            } else if (day.getDate() == today) {
                className = this.theme.getClassName("DATE_TIME_TODAY_LINK");
            } else {
                className = this.theme.getClassName("DATE_TIME_LINK");
            }
                 
            linkId = id + linkNum;
            var tmpDate = new Date(day.getTime() + oneDayInMs);

            // On some platforms, the date is not incremented correctly (e.g.,
            // October 28th 1990, 2007, and 2012). In this case, try again.
            if (tmpDate.getDate() == day.getDate()) {
                tmpDate = new Date(tmpDate.getTime() + oneDayInMs);
            } 
            
            // Check whether this is the last date in the calendar and if so
            // set the setFocus variable to true. This will mean that when the
            // user tabs away from this day, the focus will be set on the
            // close button.
            if (tmpDate.getDate() == 1 && column == 6) {
                setFocus = true;
            } else {
                setFocus = false;
            }
            this.addDayLink(rowNodeClone, day, linkId, className, setFocus);
            day = tmpDate;
            column++;
            linkNum++;
        }
    }
    
    // Add any cells in the last row of the following month
    while (column < 7) {
        var className = this.theme.getClassName("DATE_TIME_OTHER_LINK");
        linkId = id + linkNum;
        
        // Check whether this is the last date in the calendar and if so
        // set the setFocus variable to true. This will mean that when the
        // user tabs away from this day, the focus will be set on the
        // close button.        
        if (column == 6) {
            setFocus = true;
        } else {
            setFocus = false;
        }
        this.addDayLink(rowNodeClone, day, linkId, className, setFocus);                    
        day = new Date(day.getTime() + oneDayInMs);
        column++;
        linkNum++;
    }
    return true;
};

/**
 * Helper function to add the week day headers row.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.addWeekDays = function() {            
    var colNodeClone;
    var spanNodeClone;    
    var firstDay = this.firstDayOfWeek - 1;
    
    // Clone the <tr> node and append it to <tbody>
    var rowNodeClone = this.weekDayRow.cloneNode(false);
    this.tbodyContainer.appendChild(rowNodeClone);
        
    for (var i = 0; i < 7; i++) {
        // Clone the <th> node and append it to <tr>
        colNodeClone = this.weekDayColumn.cloneNode(false);
        rowNodeClone.appendChild(colNodeClone);
               
        // Clone the <span> node and append it to <th>
        spanNodeClone = this.weekDayContainer.cloneNode(false);
        colNodeClone.appendChild(spanNodeClone);
        
        // NOTE: If you set this value manually, text must be HTML escaped.
        this.widget.addFragment(spanNodeClone, this.weekDays[firstDay]);

        firstDay++;
        if (firstDay == 7) {
            firstDay = 0;
        }     
    }
    return true;
};

/**
 * Close the calendar if the enter key is pressed and set the initial focus
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */            
webui.suntheme4_2.widget.calendar.prototype.closeCalendar = function(event) {
    var evt = (event) ? event : ((window.event) ? window.event : null);
     
    // If key pressed and enter, then close the menu
    if ((evt.type == "keydown") && (evt.keyCode == 13)) {
        this.toggleCalendar();
        document.getElementById(this.toggleLink.id).focus();        
        return false;
    } 
    this.setInitialFocus();
    return true;    
};

/**
 * Process day selected event.
 * <p>
 * When a day link is selected, an event is published which the 
 * calendarField widget will use to update its text field.
 * </p>
 * @param {String} Formatted date string.
 * @return {boolean} false to cancel JavaScript event.
 */
webui.suntheme4_2.widget.calendar.prototype.daySelected = function(formattedDate) {
    this.toggleCalendar();    
    this.publish(webui.suntheme4_2.widget.calendar.event.day.selectedTopic, [{
        id: this.id,
        date:formattedDate
    }]);
    return false;
};

/**
 * This function is used to decrease the month by one.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.decreaseMonth = function() {
    var monthMenu = webui.suntheme4_2.dijit.byId(this.monthMenu.id).getSelectElement();
    // If the monthMenu has no value, set it to January (that's what
    // it will have appeared like in the browser). Can happen on IE.  
    if (monthMenu.value == null) {
        monthMenu.value = monthMenu.options[0].value;
    }
    
    var month = parseInt(monthMenu.value);
    if (month == 1) {
        var yearMenu = webui.suntheme4_2.dijit.byId(this.yearMenu.id).getSelectElement();        
         if (yearMenu.value == null) {
             // If the yearMenu has no value, set it to the first available year            
             // (that's what it will have appeared like in the browser). Can happen on IE.
             yearMenu.value = yearMenu.options[0].value;
         } else if (yearMenu.value == yearMenu.options[0].value) {
             // No need to update the calendar in this case,
             // we don't change anything.
             return false;           
         } else {
             // Decrease the year by one and set the month to December
             var year = parseInt(yearMenu.value);
             year--;
             yearMenu.value = year;
             month = 12;
         }
    } else {
        month--;
    }
    monthMenu.value = month;    
    return this.updateMonth(false);
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.calendar.event =
        webui.suntheme4_2.widget.calendar.prototype.event = {
    /**
     * This object contains day event topics.
     * @ignore
     */
    day: {
        /** Day event topic for custom AJAX implementations to listen for. */
        selectedTopic: "webui_suntheme4_2_widget_calendar_event_selected"
    },

    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_calendar_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_calendar_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_calendar_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_calendar_event_state_end"
    },

    /**
     * This object contains toggle event topics.
     * @ignore
     */
    toggle: {
        /** Open event topic for custom AJAX implementations to listen for. */
        openTopic: "webui_suntheme4_2_widget_calendar_event_toggle_open",

        /** Close event topic for custom AJAX implementations to listen for. */
        closeTopic: "webui_suntheme4_2_widget_calendar_event_toggle_close"
    }
};

/**
 * Helper function to format the date.
 *
 * @param {String} month
 * @param {String} day
 * @param {String} year
 * @return {String} The date format.
 */
webui.suntheme4_2.widget.calendar.prototype.formatDate = function(month, day, year) {
    var date = new String(this.dateFormat);      
    date = date.replace("yyyy", new String(year));
    if (month < 10) {
        date = date.replace("MM", "0" + new String(month));
    } else {
        date = date.replace("MM", new String(month));
    }
    if (day < 10) {
        date = date.replace("dd", "0" + new String(day));
    } else {
        date = date.replace("dd", new String(day));
    }
    return date;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.calendar.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);
    
    // Set properties.
    if (this.todayDateMsg) { props.todayDateMsg = this.todayDateMsg; }
    if (this.spacerImage) { props.spacerImage = this.spacerImage; }
    if (this.topLeftImage) { props.topLeftImage = this.topLeftImage; }
    if (this.topRightImage) { props.topRightImage = this.topRightImage; }
    if (this.closeButtonLink) { props.closeButtonLink = this.closeButtonLink; }
    if (this.increaseLink) { props.increaseLink = this.increaseLink; }
    if (this.decreaseLink) { props.decreaseLink = this.decreaseLink; }
    if (this.monthMenu) { props.monthMenu = this.monthMenu; }
    if (this.yearMenu) { props.yearMenu = this.yearMenu; }   
    if (this.firstDayOfWeek) { props.firstDayOfWeek = this.firstDayOfWeek; }
    if (this.toggleLink) { props.toggleLink = this.toggleLink; }
    if (this.weekDays) { props.weekDays = this.weekDays; }    
    if (this.maxDate) { props.maxDate = this.maxDate; }
    if (this.minDate) { props.minDate = this.minDate; }
    
    return props;
};

/**
 * Workaround IE bug where popup calendar appears under other components.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.ieStackingContextFix = function() {
    var div = this.calendarContainer;
    if (div.style.display == "block") {
        // This popup should be displayed
        // Get the current zIndex for the div
        var divZIndex = div.currentStyle.zIndex;
        
        // Propogate the zIndex up the offsetParent tree
        var tag = div.offsetParent;
        while (tag != null) {
            var position = tag.currentStyle.position;
            if (position == "relative" || position == "absolute") {

                // Save any zIndex so it can be restored
                tag.raveOldZIndex = tag.style.zIndex;

                // Change the zIndex
                tag.style.zIndex = divZIndex;
            }
            tag = tag.offsetParent;
        }

        // Hide controls unaffected by z-index
        this.ieShowShim();
    } else {
        // This popup should be hidden so restore zIndex-s
        var tag = div.offsetParent;
        while (tag != null) {
            var position = tag.currentStyle.position;
            if (position == "relative" || position == "absolute") {
                if (tag.raveOldZIndex != null) {
                    tag.style.zIndex = tag.raveOldZIndex;
                }
            }
            tag = tag.offsetParent;
        }
        this.ieHideShim();
    }
    return true;
};

/**
 * Hides components unaffected by z-index.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.ieShowShim = function() {  
    var popup = this.calendarContainer;
    var shim = this.shimContainer;
    
    shim.style.position = "absolute";
    shim.style.left = popup.style.left;
    shim.style.top = popup.style.top;
    shim.style.width = popup.offsetWidth;
    shim.style.height = popup.offsetHeight;
    shim.style.zIndex = popup.currentStyle.zIndex - 1;
    shim.style.display = "block";

    return true;
};

/**
 * Hide the shim iframe.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.ieHideShim = function() {
    var shim = this.shimContainer;
    shim.style.display = "none";
    return true;
};

/**
 * This function is used to increment the current month.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.increaseMonth = function() {            
    var monthMenu = webui.suntheme4_2.dijit.byId(this.monthMenu.id).getSelectElement();          
    
    // If the monthMenu has no value, set it to January (that's what
    // it will have appeared like in the browser). Can happen on IE. 
    if (monthMenu.value == null) {
        monthMenu.value = monthMenu.options[0].value;
    }
    
    var month = parseInt(monthMenu.value);
    if (month == 12) {
        var yearMenu = webui.suntheme4_2.dijit.byId(this.yearMenu.id).getSelectElement();
        var numOptions = yearMenu.options.length;
        if (yearMenu.value == null) {
            // If the yearMenu has no value, set it to the first available year            
            // (that's what it will have appeared like in the browser). Can happen on IE.
            yearMenu.value = yearMenu.options[0].value;
        } else if (yearMenu.value == yearMenu.options[numOptions-1].value) {
            // No need to update the calendar in this case,
            // we don't change anything.
            return false;            
        } else {
            // Increase the year by one and set the month to January.
            var year = parseInt(yearMenu.value);
            year++;
            yearMenu.value = year;
            month = 1;
        }
    } else {
        month++;
    }
    monthMenu.value = month;   
    return this.updateMonth(false);    
};

/**
 * This function returns a JSON array of months to be displayed in the month 
 * drop down. 
 * return {Object} A JSON array of months.
 */
webui.suntheme4_2.widget.calendar.prototype.getMonthOptions = function() {
    var monthMenu = new Array();
    
    // Get the number of months in a calendar year.
    // Some calendars may have more than 12 months a year.
    var numOfMonths = parseInt(this.theme.getMessage("calendar.numOfMonths"));
    
    for ( var i = 0; i < numOfMonths; i++ ) {
        monthMenu[i] = {};
        monthMenu[i].value = i+1;
        monthMenu[i].disabled = false;
        monthMenu[i].separator = false;
        monthMenu[i].escape = true;
        monthMenu[i].group = false;
        monthMenu[i].label=this.theme.getMessage("calendar."+i);
    }    
    return monthMenu;
};

/**
 * This function returns a JSON array of years to be displayed in the year
 * drop down
 * @param {String} minYear the minimum year of the calendar display
 * @param {String} maxYear the maximum year of the calendar display
 * @return {Object} A JSON array of calendar years.
 */
webui.suntheme4_2.widget.calendar.prototype.getYearOptions = function(minYear, maxYear) {    
    var yearMenu =new Array();       
    var diff = maxYear - minYear;
    for ( var i = 0; i <= diff; i++ ) {
        yearMenu[i] = {};
        yearMenu[i].value = minYear;
        yearMenu[i].disabled = false;
        yearMenu[i].separator = false;
        yearMenu[i].escape = true;
        yearMenu[i].group = false;
        yearMenu[i].label=minYear;
        minYear++;
    }
    return yearMenu;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.postCreate = function () {
    // Set ids. 
    if (this.id) {
        this.calendarMenuContainer.id = this.id + "_calendarMenuContainer";
        this.linkNode.id = this.id + "_linkNodeContainer";
        this.todayDateContainer.id = this.id + "_todayDateContainer";
        this.closeButtonContainer.id = this.id + "_closeButtonContainer";
        this.previousLinkContainer.id = this.id + "_previousLinkContainer";
        this.monthMenuContainer.id = this.id + "_monthMenuContainer";
        this.nextLinkContainer.id = this.id + "_nextLinkContainer";
        this.yearMenuContainer.id = this.id + "_yearMenuContainer";
        this.shimContainer.id = this.id + "_shim";
    }

    // Create client side widgets for the calendar.
    // When the _setProps() function is called, these widgets will be
    // instantiated via the props param. 

    // If toggle link is null, create the image hyperlink.
    if (this.toggleLink == null) {
        this.toggleLink = this.widget.getWidgetProps("imageHyperlink", {
                id: this.id + "_datePickerLink",
                contents: [],
                imagePosition: "left",
                title: this.theme.getMessage("calendar.popupImageAlt"),
                enabledImage: this.widget.getWidgetProps("image", {
                    border: 0,
                    icon: "CALENDAR_BUTTON",
                    id: this.id + "_datePickerLink_image"
                }),
                disabledImage: this.widget.getWidgetProps("image", {
                    border: 0,
                    icon: "CALENDAR_BUTTON_DISABLED",
                    id: this.id + "_datePickerLink_image_disabled"
                }),
                align:"middle"
            });
    }

    // Create the spacer image.
    if (this.spacerImage == null) {
        this.spacerImage = this.widget.getWidgetProps("image", {
            icon: "DOT",
            id: this.id + ":DOT"
        });
    }
    
    // Create the top left image.
    if (this.topLeftImage == null) {
        this.topLeftImage = this.widget.getWidgetProps("image", {
            icon: "SCHEDULER_TOP_LEFT",
            id: this.id + ":topLeft"
        });        
    }        
        
    //Create the top right image.
    if (this.topRightImage == null) {
        this.topRightImage = this.widget.getWidgetProps("image", {
            icon: "SCHEDULER_TOP_RIGHT",
            id: this.id + ":topRight"
        });        
    }

    // Create the increase link imageHyperlink widget.
    if (this.increaseLink == null) {
        this.increaseLink = this.widget.getWidgetProps("imageHyperlink", {
                id: this.id + ":nextMonthLink",
                enabledImage: this.widget.getWidgetProps("image", {
                    border: 0,
                    icon: "SCHEDULER_FORWARD",
                    id: this.id + ":nextMonthLink_image"
                }),
                title: this.theme.getMessage("CalendarMonth.goForward")             
            });
    }   
    
    // Create the decrease link imageHyperlink widget.
    if (this.decreaseLink == null) {
        this.decreaseLink = this.widget.getWidgetProps("imageHyperlink", {
                "id": this.id + ":previousMonthLink",
                enabledImage: this.widget.getWidgetProps("image", {
                    border: 0,
                    icon: "SCHEDULER_BACKWARD",
                    id: this.id + ":previousMonthLink_image"
                }),
                title: this.theme.getMessage("CalendarMonth.goBack")
            });
    }        
    
    // Create the close button link imageHyperlink widget
    if (this.closeButtonLink == null) {
        this.closeButtonLink = this.widget.getWidgetProps("imageHyperlink", {
                id: this.id + ":closeButtonLink",
                enabledImage: this.widget.getWidgetProps("image", {
                    border: 0,
                    icon: "CALENDAR_CLOSE_BUTTON",
                    id: this.id + "closeButtonLink_close"
                }),
                title: this.theme.getMessage("CalendarMonth.close"),
                className: this.theme.getClassName("CALENDAR_CLOSE_BUTTON")            
            });    
    }
    
    // If the dateFormatPattern is null, get one from the themes.
    if (this.dateFormat == null) {
        this.dateFormat = this.theme.getMessage("calendar.dateFormat");
    }
    
    // If the minDate and maxDate are not specified, create a default values.
    // The minDate's year is set to 100 years previous to the current year
    // and maxDate is set to 200 years forward from the minDate's year'
    var minDate = new Date();
    var maxDate = new Date();
    if (this.minDate == null) {
        minDate.setFullYear(minDate.getFullYear() - 100);
        this.minDate = this.formatDate(minDate.getMonth(), 
                            minDate.getDate(), minDate.getFullYear());        
    } else {
        minDate = this.convertStringToDate(this.minDate);
    } 

    if (this.maxDate == null) {
        maxDate.setFullYear(minDate.getFullYear() + 200);
        this.maxDate = this.formatDate(maxDate.getMonth(), 
                            maxDate.getDate(), maxDate.getFullYear());
    } else {
        maxDate = this.convertStringToDate(this.maxDate);
    }             
  
    // Initialize the days of the week.
    if (this.weekDays == null) {
        this.weekDays = new Array();
        this.weekDays[0] = this.theme.getMessage("CalendarMonth.weekdaySun");
        this.weekDays[1] = this.theme.getMessage("CalendarMonth.weekdayMon");
        this.weekDays[2] = this.theme.getMessage("CalendarMonth.weekdayTue");                
        this.weekDays[3] = this.theme.getMessage("CalendarMonth.weekdayWed");
        this.weekDays[4] = this.theme.getMessage("CalendarMonth.weekdayThu");
        this.weekDays[5] = this.theme.getMessage("CalendarMonth.weekdayFri");
        this.weekDays[6] = this.theme.getMessage("CalendarMonth.weekdaySat");
    }           
    
    // Get the first day of week for that particular locale.
    if (this.firstDayOfWeek == null) {
        this.firstDayOfWeek = parseInt(this.theme.getMessage("calendar.firstDayOfWeek"));
    }
    
    // This will append a localized string along with the
    // today's date
    if (this.todayDateMsg == null) {        
        var d = new Date();
        var todayDateMsg = this.theme.getMessage("CalendarMonth.todayIs");
        
        // Remove the "$0" argument used for the server side param
        var index = todayDateMsg.indexOf(":");
        this.todayDateMsg = todayDateMsg.substr(0, index+1);
        
        var month = this.theme.getMessage(
                        "calendar." + (d.getMonth()));
        month=month.substr(0,3);
        if (this.dateFormat.indexOf("MM") == 0) {
            this.todayDateMsg += " " + month + " " + d.getDay();
        } else {
            this.todayDateMsg += " " + d.getDay() + " " + month;        
        }
        this.todayDateMsg += ", "+d.getFullYear();
    }

    // Initialize the month menu if one does not exist.
    if (this.monthMenu == null) {                  
        this.monthMenu = this.widget.getWidgetProps("dropDown", {
            id: this.id + ":monthMenu",
            options:this.getMonthOptions(),
            title: this.theme.getMessage("CalendarMonth.selectMonth")
        });                  
    }
    
    // Initialize the year menu if one does not exist.
    if (this.yearMenu == null) {
        this.yearMenu = this.widget.getWidgetProps("dropDown", {
            id: this.id + ":yearMenu",
            options: this.getYearOptions(minDate.getYear(), maxDate.getYear()),
            title: this.theme.getMessage("CalendarMonth.selectYear")   
        });          
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to obtain the Date object from the given string
 * date value
 *
 * @param {String} inputDate The date to be converted into Dataee object
 * @param {boolean} yearCheck Check whether the year falls within the specified range
 * @return {Object}  The Date object corresponding to the input String
 */
webui.suntheme4_2.widget.calendar.prototype.convertStringToDate = function(inputDate, yearCheck) {   
    if (inputDate == "") {
        property = null;
        return false;
    }
    
    var pattern = this.dateFormat;
    var yearIndex = pattern.indexOf("yyyy");
    var monthIndex = pattern.indexOf("MM");
    var dayIndex = pattern.indexOf("dd");

    // If the format is invalid, set the current value to null
    if (yearIndex < 0 || monthIndex < 0 || dayIndex < 0) {
        return null;
    } 
    
    var counter = 0;
    var number;
    var selectedDate = new Date();
    var found = 0;
    var dateString;

    while (counter < inputDate.length) {
        if (counter == yearIndex) {
            try {
                number = parseInt(inputDate.substr(counter, 4));
                if (isNaN(number)) {
                    property = null;
                    return false;
                }                
                // Check if the input date's year range is inbetween the 
                // allowed dates.   
                if (yearCheck == true) {
                    var index = 0;
                    var foundYear = false;                               
                    yearMenu = webui.suntheme4_2.dijit.byId(this.yearMenu.id).getSelectElement();
                    while (index < yearMenu.length) {
                        if (number == yearMenu.options[index].value) {
                            selectedDate.setFullYear(number);
                            ++found;
                            foundYear = true;
                            break;
                        }
                        index++;
                    }
                    if (!foundYear) {
                        break;
                    }
                } else {            
                    selectedDate.setFullYear(number);
                    ++found;
                }                    
            } catch(e) {}
        } else if (counter == monthIndex) {
            try {    
                dateString = inputDate.substr(counter, 2);
                // This is a workaround for Firefox! 
                // parseInt() returns 0 for values 08 and 09
                // while all other leading zeros work.
                if (dateString.charAt(0) == '0') {
                    dateString = dateString.substr(1, 1);
                }
                number = parseInt(dateString);
                if (isNaN(number)) {
                    property = null;
                    return false;
                }
                selectedDate.setMonth(number-1);
                ++found;
            } catch(e) {}
        } else if (counter == dayIndex) {
            try {
                dateString = inputDate.substr(counter, 2);
                // This is a workaround for Firefox! 
                // parseInt() returns 0 for values 08 and 09
                // while all other leading zeros work.
                if (dateString.charAt(0) == '0') {
                    dateString = dateString.substr(1, 1);
                }
                number = parseInt(dateString);
                if (isNaN(number)) {
                    return null;
                }
                selectedDate.setDate(number);
                ++found;
            } catch(e) {}
        }
        ++counter;
    }

    if (found == 3) {
        return selectedDate;
    } else {
        return null;
    }    
    return true;       
};

/**
 * Helper function to set the initial focus on the menus.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.setInitialFocus = function() {    
    var pattern = new String(this.dateFormat);
    var yearIndex = pattern.indexOf("yyyy");
    var monthIndex = pattern.indexOf("MM");
    
    // Moving the year menu around based on the date format is not supported yet.
    // So, the code for setting focus on the year menu has been commented out.
    // if (yearIndex < monthIndex) {        
    //    var yearMenu = document.getElementById(this.calendarMonth.yearMenu.id).getSelectElement();
    //    yearMenu.focus();                 
    // } else {
        var monthMenu = webui.suntheme4_2.dijit.byId(this.monthMenu.id).getSelectElement();          
        monthMenu.focus();
    // }
    return true;
};

/**
 * Set the value of an HTML select element, but limit value to min and max.
 *
 * @param {Node} select The HTML select element.
 * @param {String} value The selected value.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.setLimitedSelectedValue = function(select, value) {
    var min = select.options[0].value;
    var max = select.options[select.length - 1].value;
    if (value < min) {        
        select.value = min;
    } else if ( value > max) {        
        select.value = max;
    } else {
        this.setSelectedValue(select, value);        
    }
    return true;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {Object} closeButtonLink 
 * @config {String} date 
 * @config {String} dateFormat 
 * @config {Object} decreaseLink 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} increaseLink 
 * @config {Object} monthMenu
 * @config {String} style Specify style rules inline.
 * @config {Object} todayDateMsg
 * @config {Object} toggleLink
 * @config {boolean} visible Hide or show element.
 * @config {Object} yearMenu
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.calendar.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.        
    if (props.todayDateMsg) {
        // NOTE: If you set this value manually, text must be HTML escaped.
        this.widget.addFragment(this.todayDateContainer, props.todayDateMsg);
    }

    if (props.spacerImage) {
        if (!webui.suntheme4_2.dijit.byId(this.spacerImage.id)) {
            this.widget.addFragment(this.spacerImageContainer, props.spacerImage);
        }
    }

    if (props.topLeftImage) {
        if (!webui.suntheme4_2.dijit.byId(this.topLeftImage.id)) {
            this.widget.addFragment(this.topLeftImageContainer, props.topLeftImage);
        }
    }

    if (props.topRightImage) {
        if (!webui.suntheme4_2.dijit.byId(this.topRightImage.id)) {
            this.widget.addFragment(this.topRightImageContainer, props.topRightImage);
        }
    }

    if (props.date) {
        var selDate = this.convertStringToDate(props.date, true);
        if (selDate != null) {
            this.currentValue = selDate;
        }
    }

    // Set close link properties.
    if (props.closeButtonLink) {
        // Set properties.
        props.closeButtonLink.onClick = 
            "webui.suntheme4_2.dijit.byId('" + this.id + "').toggleCalendar();return false;";
        props.closeButtonLink.onKeyDown = 
            "webui.suntheme4_2.dijit.byId('" + this.id + "').closeCalendar(event);return false;";       

        // Update/add fragment.
        this.widget.updateFragment(this.closeButtonContainer, 
            this.closeButtonLink.id, props.closeButtonLink);
    }

    // Set decrease link properties.
    if (props.decreaseLink) {
        // Set properties.
        props.decreaseLink.onClick = 
            "webui.suntheme4_2.dijit.byId('" + this.id + "').decreaseMonth();return false;";

        // Update/add fragment.
        this.widget.updateFragment(this.previousLinkContainer, 
            this.decreaseLink.id, props.decreaseLink);
    }

    // Set increase link properties.
    if (props.increaseLink) {
        // Set properties.
        props.increaseLink.onClick = 
            "webui.suntheme4_2.dijit.byId('" + this.id + "').increaseMonth();return false;";

        // Update/add fragment.
        this.widget.updateFragment(this.nextLinkContainer, this.increaseLink.id, 
            props.increaseLink);
    }
    
    var minDate = null;
    var maxDate = null;    
    if (props.minDate || props.maxDate) {
        if (props.minDate) {
            
            // Convert the given string to a proper given date format pattern
            // and then store it.
            minDate = this.convertStringToDate(props.minDate);
            if (minDate != null) {
                this.minDate = this.formatDate(minDate.getMonth(), 
                    minDate.getDate(), minDate.getFullYear());
            }
        } 
        
        if (props.maxDate) {
            
            // Convert the given string to a proper given date format pattern
            // and then store it.            
            maxDate = this.convertStringToDate(props.maxDate);      
            if (maxDate != null) {
                this.maxDate = this.formatDate(maxDate.getMonth(), 
                    maxDate.getDate(), maxDate.getFullYear());
            }               
        } 
        
        //Recalculate the year options with new minDate and maxDate values.
        props.yearMenu = this.widget.getWidgetProps("dropDown", {
            id: this.id + ":yearMenu",
            options:this.getYearOptions(minDate.getFullYear(), maxDate.getFullYear()),
            title: this.theme.getMessage("CalendarMonth.selectYear")   
        });  
        
        // update the value of yearMenu
        this.yearMenu = props.yearMenu;                                          
    }
        
    // Set month menu properties
    if (props.monthMenu) {                        
        // Set properties.
        props.monthMenu.onChange =
            "webui.suntheme4_2.dijit.byId('" + this.id + "').updateMonth(false);return false;";
                         
        // Update/add fragment.
        this.widget.updateFragment(this.monthMenuContainer, this.monthMenu.id,
            props.monthMenu);
    }

    // Set year menu properties.
    if (props.yearMenu) {        
        // Set properties.
        props.yearMenu.onChange =
            "webui.suntheme4_2.dijit.byId('" + this.id + "').updateMonth(false);return false;";

        // Update/add fragment.
        this.widget.updateFragment(this.yearMenuContainer, this.yearMenu.id,
            props.yearMenu);
    }

    // Set toggle link properties.
    if (props.disabled != null) {
        this.disabled = new Boolean(props.disabled).valueOf(); 
    }

    // If the popup calendar is still being shown, prevent disabling of the calendar.
    // The widget can only be disabled if the popup calendar is not shown.
    if (props.toggleLink || 
        (props.disabled != null && this.calendarContainer.style.display != "block")) {

        // Ensure property exists so we can call setProps just once.
        if (props.toggleLink == null) {
            props.toggleLink = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.toggleLink.disabled = this.disabled;
        props.toggleLink.onClick =
            "webui.suntheme4_2.dijit.byId('" + this.id + "').toggleCalendar();return false;";

        // Update/add fragment.
        this.widget.updateFragment(this.linkNode, this.toggleLink.id, props.toggleLink); 
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * This function is used to set the value of a select element.
 *
 * @param {Node} select The HTML select element.
 * @param {String} value The selected value.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.setSelectedValue = function(select, value) {
    for (var i = 0; i < select.length; i++) {
        if (select.options[i].value == value) {
            select.selectedIndex = i;
            return true;
        }
    }
    select.selectedIndex = -1;
    return true;
};

/**
 * Process toogle event.
 *
 * @return {boolean} false to cancel JavaScript event.
 */
webui.suntheme4_2.widget.calendar.prototype.toggleCalendar = function() {
    var topic = (this.calendarContainer.style.display != "block")
        ? webui.suntheme4_2.widget.calendar.event.toggle.openTopic
        : webui.suntheme4_2.widget.calendar.event.toggle.closeTopic;

    // Publish an event for other widgets to listen for.
    //
    // Note: This must be done before the calendar is opened so user
    // input can be applied to the current date.
    this.publish(webui.suntheme4_2.widget.calendar.event.toggle.openTopic, [{
        id: this.id
    }]);

    // Open the calendar.
    if (this.calendarContainer.style.display != "block") {
        if (webui.suntheme4_2.widget.calendar.activeCalendarId != null) {
            var cal = webui.suntheme4_2.dijit.byId(webui.suntheme4_2.widget.calendar.activeCalendarId);
            cal.toggleCalendar();
        }
        webui.suntheme4_2.widget.calendar.activeCalendarId = this.id;        
        this.calendarContainer.style.display = "block";
        this.setInitialFocus();
        this.updateMonth(true);    
    } else {
        // Hide the calendar popup
        this.calendarContainer.style.display = "none";
        webui.suntheme4_2.widget.calendar.activeCalendarId = null;
    }

    // Test for IE 
    if (webui.suntheme4_2.browser.isIe5up()) {
        this.ieStackingContextFix();
    }          
    return false;
};

/**
 * This function is used to update the calendar month.
 * It is called when the calendar is opened, the next or previous
 * links are clicked, or the month or year menus are changed.
 *
 * @param {boolean} initialize Flag indicating to initialze the year and month menus
 * with the current value. The value is true only when the calendar is opened.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.updateMonth = function(initialize) {
    
    // Remove all the nodes of <tbody> before cloning its children.
    this.widget.removeChildNodes(this.tbodyContainer);    
    // Add week days
    this.addWeekDays();    
    
    // Add days of the month
    this.addDaysInMonth(this.currentValue, initialize);  
    return true;     
};
