/**
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 */


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.addRemove");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");

/** 
 * @class This class contains functions for addRemove components.
 * @static
 */
webui.suntheme4_2.addRemove = {
    /**
     * This function is used to initialize HTML element properties with Object 
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The HTML element id.
     * @config {String} separator The character deliminator for ordered options.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize addRemove.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Note that _available, and _selected are not a facets
        // and therefore do not require the use of "facetid" as discussed below

        // The select element from which selections are made 
        domNode.availableList = document.getElementById(props.id + "_available");

        // The select element in which selections are shown 
        domNode.selectedList = document.getElementById(props.id + "_selected");

        // Bug 6338492 -
        //     ALL: If a component supports facets or children is must be a
        //      NamingContainer
        // Since AddRemove has become a NamingContainer the id's for
        // the facet children are prefixed with the AddRemove id
        // in addition to their own id, which also has the 
        // AddRemove id, as has been the convention for facets. This introduces
        // a redundancy in the facet id so the add button now looks like
        //
        // "formid:addremoveid:addremoveid_addButton"
        //
        // It used to be "formid:addremoveid_addButton"
        // It would be better to encapsulate that knowledge in the
        // AddRemove renderer as does FileChooser which has the
        // same problem but because the select elements are not
        // facets in AddRemove they really do only have id's of the
        // form "formid:addremoveid_list_value". Note that 
        // in these examples the "id" parameter is "formid:addremoveid"
        //
        // Therefore for now, locate the additional prefix here as the
        // "facet" id. Assume that id never ends in ":" and if there is
        // no colon, id is the same as the component id.
        var componentid = props.id;
        var colon_index = componentid.lastIndexOf(':');
        if (colon_index != -1) {
            componentid = props.id.substring(colon_index + 1);
        }
        var facetid = props.id + ":" + componentid;

        domNode.addButton = document.getElementById(facetid + "_addButton");
        domNode.addAllButton = document.getElementById(facetid + "_addAllButton");
        domNode.removeButton = document.getElementById(facetid + "_removeButton");
        domNode.removeAllButton = document.getElementById(facetid + "_removeAllButton");
        domNode.moveUpButton = document.getElementById(facetid + "_moveUpButton");
        domNode.moveDownButton = document.getElementById(facetid + "_moveDownButton");

        // _list_value and _item_list are not facets and do not need facetid
        domNode.selectedValues = document.getElementById(props.id + "_list_value");

        // Calculate the value indices
        var itemString = document.getElementById(props.id + "_item_list");

        // HTML elements may not have been created, yet. The moveUp/Down and 
        // remove/All buttons may not exist at all.
        if (itemString == null
                || domNode.availableList == null 
                || domNode.selectedList == null
                || domNode.selectedValues == null
                || domNode.addButton == null
                || domNode.removeButton == null
                || (new Boolean(props.selectAll).valueOf() == true
                    && (domNode.addAllButton == null || domNode.removeAllButton == null))
                || (new Boolean(props.moveButtons).valueOf() == true
                    && (domNode.moveUpButton == null || domNode.moveDownButton == null))) {
            return setTimeout(function() {
                webui.suntheme4_2.addRemove._init(props);
            }, 10);
        }

        // Set given properties on domNode.
        webui.suntheme4_2.prototypejs.extend(domNode, props, false);

        // Calculate the value indices
        if (itemString != null) {
            var string = new String(itemString.value);
            domNode.allValues = string.split(props.separator);	
        } else {
            domNode.allValues = new Array();
        }

        // The options of the select element from which selections are made 
        domNode.availableOptions = domNode.availableList.options;

        // The options of the select element in which selections are shown 
        domNode.selectedOptions = domNode.selectedList.options;

        // Set functions.
        domNode.add = webui.suntheme4_2.addRemove.add;
        domNode.addAll = webui.suntheme4_2.addRemove.addAll;
        domNode.remove = webui.suntheme4_2.addRemove.remove;
        domNode.removeAll = webui.suntheme4_2.addRemove.removeAll;
        domNode.moveUp = webui.suntheme4_2.addRemove.moveUp;
        domNode.moveDown = webui.suntheme4_2.addRemove.moveDown;
        domNode.updateButtons = webui.suntheme4_2.addRemove.updateButtons;
        domNode.calculateIndex = webui.suntheme4_2.addRemove.calculateIndex;
        domNode.moveOption = webui.suntheme4_2.addRemove.moveOption;
        domNode.updateValue = webui.suntheme4_2.addRemove.updateValue;
        domNode.allowMultipleAdditions = webui.suntheme4_2.addRemove.allowMultipleAdditions;
        domNode.availableOnChange = webui.suntheme4_2.addRemove.availableOnChange;
        domNode.selectedOnChange = webui.suntheme4_2.addRemove.selectedOnChange;

        // Enable multiple buttons.
        if (new Boolean(props.duplicateSelections).valueOf() == true) {
            domNode.allowMultipleAdditions();
        }

        // Initialize buttons.
        domNode.updateButtons();
        return true;
    },

    /**
     * This function adds options to the selected list.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    add: function() {
        if (this.availableOptions.selectedIndex == -1) {
            return false;
        }

        var sort = this.sort && (this.moveUpButton == null);

        // deselect everything in the selected list
        this.selectedList.selectedIndex = -1;
        return this.moveOption(this.availableOptions, this.selectedOptions,
            this.selectedList, sort);
    },

    /**
     * This function removes options to the selected list.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    remove: function() {
        if (this.selectedOptions.selectedIndex == -1) {
            return false;
        }

        // deselect everything in the selected list
        this.availableList.selectedIndex = -1;
        return this.moveOption(this.selectedOptions, this.availableOptions,
            this.availableList, this.sort);
    },

    /**
     * This function moves options in the selected list.
     *
     * @param {Array} moveFromOptions
     * @param {Array} moveToOptions
     * @param {Array} moveToList
     * @param {boolean} sort
     * @return {boolean} true if successful; otherwise, false.
     */
    moveOption: function(moveFromOptions, moveToOptions, moveToList, sort) {
        var index = moveFromOptions.selectedIndex;
        if (index == -1) {
            return false;
        }

        // Keep moving selected items until there aren't any more valid ones
        while (index != -1 && index < moveFromOptions.length - 1) {
            var lastOption = moveToOptions.length - 1;

            // This is the option we're moving
            var curSelection = moveFromOptions[index];

            // This is the index where we insert the option...
            var insertionIndex = 0;
	
            // ...and this is the option at that index
            var insertionOption;
 
            if (sort) {
                // If there are no buttons to move the selected items up or
                // down, then we preserve the sorting order of the available
                // items. We calculate the index of the selected item (based 
                // on the indices assigned when parsing the allValues
                // variable), and then we check each selected item until we
                // reach an item with a higher index.
                var itemIndex = this.calculateIndex(curSelection.value);
                for (var counter = 0;counter < lastOption + 1;++counter) {
                    insertionOption = moveToOptions[counter];
                    if (itemIndex < this.calculateIndex(insertionOption.value)) {
                        insertionIndex = counter;
                        break;
                    }
                }
            } else {
                // If there are buttons to move the options around, then we
                // simply add the new items in the last position
                insertionIndex = lastOption;
                insertionOption = moveToOptions[lastOption];
            }

            // To insert the item, Mozilla works different from Windows
            // and Opera.
            if (moveFromOptions.remove == null) {
                // Case 1: Mozilla
                moveToList.add(curSelection, insertionOption);
            } else {
                // Case 2: Windows and Opera
                moveFromOptions.remove(index);
                moveToOptions.add(curSelection, insertionIndex);
            }
	
            // Make sure the item is selected (this is needed for Opera)
            moveToOptions[insertionIndex].selected = true;

            // Update the options
            lastOption++;

            // Get the next selected index. 
            index = moveFromOptions.selectedIndex;
        }
        this.updateValue();
        this.updateButtons();
        return false;
    },

    /**
     * This function adds all options to the selected list.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    addAll: function() {
        var numOptions = this.availableOptions.length - 1;
        for (var index = 0;index < numOptions;++index) {
            if (this.availableOptions[index].disabled == false) {
                this.availableOptions[index].selected = true;
            }
        }
        return this.add();
    },

    /**
     * This function removes all options from the selected list.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    removeAll: function() {
        var numOptions = this.selectedOptions.length - 1;
        for (var index = 0;index < numOptions;++index) {
            if (this.selectedOptions[index].disabled == false) {
                this.selectedOptions[index].selected = true;
            }
        }
        return this.remove();
    },

    /**
     * This function moves options up in the selected list.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    moveUp: function() {
        // The original allowed items to be moved on both lists. Surely we
        // only sort items on the selected list? 
        // This does not work on Mozilla.

        // We will not move the last item - it's the separator
        var numOptions = this.selectedOptions.length - 1;

        // If there aren't at least two more selected items, then there is
        // nothing to move 
        if (numOptions < 2) {
            return false;
        }

        // Start by examining the first item 
        var index = 0;

        // We're not going to move the first item. Instead, we will start
        // on the first selected item that is below an unselected
        // item. We identify the first unselected item on the list, and 
        // then we will start on next item after that
        while (this.selectedOptions[index].selected) {
            ++index;
            if (index == numOptions) {
                // We've reached the last item - no more items below it so
                // we return
                return false;
            }
        }

        // Start on the item below this one 
        ++index;

        for (index; index < numOptions; ++index) {
            if (this.selectedOptions[index].selected == true) {
                var curOption = this.selectedOptions[index];
                if (this.selectedOptions.remove == null) {
                    // For Mozilla
                    this.selectedOptions[index] = null;
                    this.selectedList.add(curOption,
                        this.selectedOptions[index - 1]);
                } else {
                    // Windows and Opera do
                    this.selectedOptions.remove(index);
                    this.selectedOptions.add(curOption, index - 1);
                }
                // This is needed for Opera only
                this.selectedOptions[index - 1].selected = true;
            }
        }
        this.updateValue();
        this.updateButtons();
        return false;
    },

    /**
     * This function moves options down in the selected list.
     *
     * @return {boolean} false to cancel JavaScript event.
     */
    moveDown: function() {
        // The original allowed items to be moved on both lists. Surely we
        // only sort items on the selected list? 
        // This does not work on Mozilla

        // Last option is numOption -1. That is the separator and we don't
        // move it. We start by examining the second to last item. 
        var index = this.selectedOptions.length - 2;

        // If this number is less than zero, there was nothing on the list
        // and we return
        if (index < 0) {
            return false;
        }

        // We're not going to move the last item. Instead, we will start
        // on the last selected item that is above an unselected
        // item. We identify the last unselected item before the separator
        // and then we start with the item above that one. 
        while (this.selectedOptions[index].selected) {
            --index;
            if (index == 0) {
                // We've reached the first item - no item above it so we
                // return 
                return false;
            }
        }

        // Start on the item above this one 
        --index;

        for (index;index > -1;--index) {
            if (this.selectedOptions[index].selected == true) {
                var curOption = this.selectedOptions[index];
                if (this.selectedOptions.remove == null) {
                    // For Mozilla
                    this.selectedOptions[index] = null;
                    this.selectedList.add(curOption, 
                        this.selectedOptions[index + 1]);
                } else {
                    // Windows and Opera do
                    this.selectedOptions.remove(index);
                    this.selectedOptions.add(curOption, index + 1);
                }
                // This is needed for Opera only
                this.selectedOptions[index + 1].selected = true;
            }
        }
        this.updateValue();
        this.updateButtons();
        return false;
    },

    /**
     * This function updates the state of all buttons.
     *
     * @return {boolean} false to cancel JavaScript event.
     */
    updateButtons: function() {
        var numOptions = this.availableOptions.length-1;
        var setting;
        
        // Disabled items should not be moved and buttons should not be enabled
        // for selected disabled items (IE problem)
        for (var i = 0; i < numOptions; ++i) {
            if (this.availableOptions[i].disabled == true) {
                this.availableOptions[i].selected = false;
            }
        }

        var index = this.availableOptions.selectedIndex;

        // The Add button is enabled if there is at least one option
        // to select from and at least one item is selected
        if (this.addButton != null) {
            setting = numOptions < 1 || index == -1;
            if (this.addButton.setDisabled != null) {
                this.addButton.setDisabled(setting);
            } else {
                this.addButton.disabled = setting;
            }
        }

        // The Add All button is enabled if there is at least one option
        // to select from, and disabled otherwise
        if (this.addAllButton != null) {
            var counter = 0;
            // If available item list is disabled then AddAll button should be disabled 
            // irrespective of options element in list.
            if (this.availableList.disabled == false) {
                for (index = 0; index < numOptions; ++index) {
                    if (this.availableOptions[index].disabled == false) {
                        ++counter;
                    }
                }
            } 
            setting = (counter < 1);            
            if (this.addAllButton.setDisabled != null) {
                this.addAllButton.setDisabled(setting);
            } else {
                this.addAllButton.disabled = setting;
            }
        }

        // The remaining buttons are enabled/disabled based on the 
        // items on the selected list
        index = this.selectedOptions.selectedIndex;
        numOptions = this.selectedOptions.length - 1;
        
        if (this.removeAllButton != null) {
            counter = 0;
            // If selected item list is disabled then RemoveAll button should be disabled 
            // irrespective of options element in list.  
            if (this.selectedList.disabled == false) {
                for (index = 0; index < numOptions; ++index) {
                     if (this.selectedOptions[index].disabled == false) {
                         ++counter;
                     }
                }
            } 
            setting = (counter < 1);
            if (this.removeAllButton.setDisabled != null) {
                this.removeAllButton.setDisabled(setting);
            } else {
                this.removeAllButton.disabled = setting;
            }
        }

        // If there are no selected items or if none of them are selected,
        // we disable Remove, Move Up, Move Down
        index = this.selectedOptions.selectedIndex;
        var noItems = numOptions < 1 || index == -1;
        if (this.removeButton != null) {
            if (this.removeButton.setDisabled != null) {
                this.removeButton.setDisabled(noItems);
            } else {
                this.removeButton.disabled = noItems;
            }
        }

        // The Move Up button is enabled (setting = false) provided that
        // there is at least one selected item that is below an unselected item 
        if (this.moveUpButton != null) {
            setting = true;
            if (noItems != true) {
                // Find the first un-selected option, then see if there is
                // a selected option below that one
                var found = false;
                var unselected = -1;
                for (index = 0; index < numOptions; ++index) {
                    if (unselected == -1) {
                        if (this.selectedOptions[index].selected == false) {
                            unselected = index;
                        }
                    } else {
                        if (this.selectedOptions[index].selected == true) {
                            setting = false;
                            break;
                        }
                    }
                }
            }
            if (this.moveUpButton.setDisabled != null) {
                this.moveUpButton.setDisabled(setting);
            } else {
                this.moveUpButton.disabled = setting;
            }
        }

        // The Move Down button is enabled (setting = false) provided that
        // there is at least one unselected item below a selected item.
        if (this.moveDownButton != null) {
            setting = true;
            if (noItems != true) {	     
                for (index = this.selectedOptions.selectedIndex; index < numOptions; ++index) {
                    if (this.selectedOptions[index].selected == false) {
                        setting = false;
                    }
                }
            }
            if (this.moveDownButton.setDisabled != null) {
                this.moveDownButton.setDisabled(setting);
            } else {
                this.moveDownButton.disabled = setting;
            }
        }
        // set the focus to the list which has some selected item(s).
        // this needs to be done to shift the focus from disabled button (Mozilla)
        if (this.selectedOptions.selectedIndex > -1) {
             this.selectedList.focus();
        } else if (this.availableOptions.selectedIndex > -1) {
             this.availableList.focus();
        }
        return false;
    },

    /**
     * Calculate the current index.
     *
     * @return {int} The current index.
     */
    calculateIndex: function(value, lastIndex) {
        var string = new String(value);
        for (var counter=0; counter < this.allValues.length; counter++) {
            if (string == this.allValues[counter]) {
                return counter;
            }
        }
        // Something went wrong. Return the index before the separator 
        return this.allValues.length - 2;
    },

    /**
     * Update current value.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    updateValue: function() {
        // Remove the options from the select that holds the actual
        // selected values
        while (this.selectedValues.length > 0) {
            this.selectedValues.remove(0);
        }

        // Create a new array consisting of the options marked as selected
        // on the official list
        var newOptions = new Array();
        var cntr = 0;
        var newOption;

        while (cntr < this.selectedOptions.length-1) {
            newOption = document.createElement("option");
            if (this.selectedOptions[cntr].text != null) {
                newOption.text = this.selectedOptions[cntr].text;
            }
            if (this.selectedOptions[cntr].value != null) {
                newOption.value = this.selectedOptions[cntr].value;
            }
            newOption.selected = true;
            newOptions[newOptions.length] = newOption;
            ++ cntr;
        }

        cntr = 0;
        if (this.selectedOptions.remove == null) {
            // For Mozilla
            while (cntr < newOptions.length) {
                this.selectedValues.add(newOptions[cntr], null);
                ++cntr;
            }
        } else {
            // Windows and Opera do
            while (cntr < newOptions.length) {
                this.selectedValues.add(newOptions[cntr], cntr);
                ++cntr;
            }
        }
        return true;
    },

    /**
     * Initialize multiple additions.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    allowMultipleAdditions: function() {
        // Replace the add and remove functions with functions which 
        // leave the available items as they are
        this.add = webui.suntheme4_2.addRemove.multipleAdd;
        this.remove = webui.suntheme4_2.addRemove.multipleRemove;
        return true;
    },

    /**
     * Add multiple options.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    multipleAdd: function() {
        this.selectedList.selectedIndex = -1;
        var index = this.availableOptions.selectedIndex;
        if (index == -1) {
            return false;
        }
    
        // keep moving selected items until there aren't any more valid ones
        while (index != -1 && index < this.availableOptions.length - 1) {
            var lastOption = this.selectedOptions.length - 1;

            // This is the option we're moving
            var curSelection = this.availableOptions[index];
            curSelection.selected = false;
            var addSelection = new Option();
            addSelection.text = curSelection.text;
            addSelection.value = curSelection.value;

            // This is the index where we insert the option...
            var insertionIndex = 0;
            // ...and this is the option at that index
            var insertionOption;

            // If there are no buttons to move the selected items up or
            // down, then we preserve the sorting order of the available
            // items. We calculate the index of the selected item (based 
            // on the indices assigned when parsing the allValues
            // variable), and then we check each selected item until we
            // reach an item with a higher index. 

            // We sort if there are no move buttons
            var sort = (this.moveUpButton == null);

            if (sort) {
                var itemIndex = this.calculateIndex(curSelection.value);
                for (var counter = 0; counter < lastOption + 1; ++counter) {
                    insertionOption = this.selectedOptions[counter];
                    if (itemIndex < this.calculateIndex(insertionOption.value)) {
                        insertionIndex = counter;
                        break;
                    }
                }
            } else {
                // If there are buttons to move the options around, then we
                // simply add the new items in the last position
                insertionIndex = lastOption;
                insertionOption = this.selectedOptions[lastOption];
            }

            // To insert the item, Mozilla works different from Windows
            // and Opera. 
            if (this.selectedOptions.remove == null) {
                // Case 1: Mozilla
                this.selectedList.add(addSelection, insertionOption);
            } else {
                // Case 2: Windows and Opera
                this.selectedOptions.add(addSelection, insertionIndex);
            }
	
            // Make sure the item is selected (this is needed for Opera)
            this.selectedOptions[insertionIndex].selected = true;

            // Update the options
            lastOption++;

            // Get the next selected index. 
            index = this.availableOptions.selectedIndex;
        }
        this.updateValue();
        this.updateButtons();
        return false;
    },

    /**
     * Remove multiple options.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    multipleRemove: function() {
        this.availableList.selectedIndex = -1;
        var index = this.selectedOptions.selectedIndex;
        if (index == -1) {
            return false;
        }

        while (index < this.selectedOptions.length - 1) {
            if (this.selectedOptions[index].selected) {
                if (this.selectedOptions.remove == null) {
                    // Case 1: Mozilla
                    this.selectedOptions[index] = null;
                } else {
                    // Case 2: Windows and Opera
                    this.selectedOptions.remove(index);
                }
            } else {
                index++;
            }
        }
        this.updateValue();
        this.updateButtons();
        return false;
    },

    /**
     * Process available on change event.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    availableOnChange: function() {
        this.selectedList.selectedIndex = -1;
        this.updateButtons();
        return false;
    },

    /**
     * Process selected on change event.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    selectedOnChange: function() {
        this.availableList.selectedIndex = -1;
        this.updateButtons();
        return false;
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.body");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.common");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.theme.common");

/**
 * @class This class contains functions common to HTML elements.
 * @static
 */
webui.suntheme4_2.common = {
    /**
     * Variables needed when submitting form so timeout will work properly.
     * @private
     */
    _formToSubmit: null,
    _submissionComponentId: null,

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // String functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Replace occurences of delimiter with the escapeChar and the
     * delimiter. For example replace "," with "/," if delimiter == "," and
     * escapeChar is "/".
     *
     * @param {String} s The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {String} The escaped string.
     */
    escapeString: function(s, delimiter, escapeChar) {
        if (s == null) {
            return null;
        }
        if (delimiter == null) {
            return s;
        }
        if (escapeChar == null) {
            return null;
        }
        
        // Escape occurrences of delimiter with 
        // escapeChar and the delimiter.
        //
        // First escape the escape char.
        //
        var escape_escapeChar = escapeChar;
        if (escapeChar == "\\") {
            escape_escapeChar = escapeChar + escapeChar;
        }
        
        var rx = new RegExp(escape_escapeChar, "g");
        var s1 = s.replace(rx, escapeChar + escapeChar);
        
        rx = new RegExp(delimiter, "g");
        return s1.replace(rx, escapeChar + delimiter);
    },
    
    /**
     * Replace occurences of a sequence of 2 instances of delimiter 
     * with 1 instance of the delimiter. For example replace ",," with "," if delimiter == ","
     *
     * @param {String} s The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {String} The unescaped string.
     */
    unescapeString: function(s, delimiter, escapeChar) {
        if (s == null) {
            return null;
        }
        if (delimiter == null) {
            return s;
        }
        if (escapeChar == null) {
            return null;
        }
        
        // UnEscape occurrences of delimiter with 
        // single instance of the delimiter
        //
        var escape_escapeChar = escapeChar;
        if (escapeChar == "\\") {
            escape_escapeChar = escapeChar + escapeChar;
        }
        
        // First unescape the escape char.
        //
        var rx = new RegExp(escape_escapeChar + escape_escapeChar, "g");
        var s1 = s.replace(rx, escapeChar);
        
        // Now replace escaped delimters
        //
        rx = new RegExp(escape_escapeChar + delimiter, "g");
        return s1.replace(rx, delimiter);
    },
    
    /**
     * Return an array of unescaped strings from escapedString
     * where the escaped character is delimiter.
     * If delimiter is "," escapedString might have the form
     * <p><pre>
     * XX\,XX,MM\,MM
     *
     * where "\" is the escape char.
     * 
     * and is returned as an array
     * array[0] == "XX,XX"
     * array[1] == "MM,MM"
     * </pre><p>
     *
     * @param {String} escapedString The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {Array} An array of unescaped strings.
     */
    unescapeStrings: function(escapedString, delimiter, escapeChar) {
        if (escapedString == null || escapedString == "") {
            return null;
        }
        if (delimiter == null || delimiter == "") {
            return escapedString;
        }
        if (escapeChar == null || escapeChar == "") {
            return null;
        }
        
        // Need to do this character by character.
        var selections = new Array();
        var index = 0;
        var escseen = 0;
        var j = 0;
        
        for (var i = 0; i < escapedString.length; ++i) {
            if (escapedString.charAt(i) == delimiter) {
                if (escseen % 2 == 0) {
                    selections[index++] = escapedString.slice(j, i);
                    j = i + 1;
                }
            }
            if (escapedString.charAt(i) == escapeChar) {
                ++escseen;
                continue;
            } else {
                escseen = 0;
            }
        }
        
        // Capture the last split.
        selections[index] = escapedString.slice(j);
        
        // Now unescape each selection
        var unescapedArray = new Array();
        
        // Now replace escaped delimiters
        // i.e.  "\," with ","
        for (i = 0; i < selections.length; ++i) {
            unescapedArray[i] = webui.suntheme4_2.common.unescapeString(
            selections[i], delimiter, escapeChar);
        }
        return unescapedArray;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Style functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Use this function add any styleClass to an html tag
     *
     * @param {Node} element the dom html tag element
     * @param {String} styleClass the name of the class to add
     * @return {boolean} true if successful; otherwise, false.
     */
    addStyleClass: function(element, styleClass) {
        // routine protection in javascript
        if (element == null || styleClass == null) {
            return false;
        }
        
        // handle easy case first
        if (element.className == null) {
            element.className = styleClass;
            return true;
        }
        
        // break out style classes into an array  
        var classes = webui.suntheme4_2.common.splitStyleClasses(element);
        if (classes == null) {
            return false;
        }
        
        // For each element of classes, compare it to styleClass.
        // If it is not in the array, append it to the end.
        for (var i = 0; i < classes.length; i++) {
            if (classes[i] != null && classes[i] == styleClass) {
                return true;
            }
        }
        element.className = element.className + " " + styleClass;
        return true;
    },
    
    /**
     * Use this function to check if an array has a style class
     *
     * @param {Array} styleArray of style classes to check
     * @param {String} styleClass the styleClass to check
     * @return {Array} An array of classes.
     */
    checkStyleClasses: function(styleArray, styleClass) {
        if (styleArray == null || styleClass == null) {
            return false;
        }
        for (var i = 0; i < styleArray.length; i++) {
            if (styleArray[i] != null && styleArray[i] == styleClass) {
                return true;
            }
        }   
        return false;
    },
    
    /**
     * Use this function to get array of style classes
     *
     * @param {Node} element the dom html tag element
     * @return {Array} An array of classes.
     */
    splitStyleClasses: function(element) {
        if (element != null && element.className != null) {
            return element.className.split(" ");
        } else {
            return null;
        }
    },
    
    /**
     * Use this function remove any styleClass for an html tag
     *
     * @param {Node} element the dom html tag element
     * @param {String} styleClass the name of the class to remove
     * @return {boolean} true if successful; otherwise, false.
     */
    stripStyleClass: function(element, styleClass) {
        // routine protection in javascript
        if (element == null || styleClass == null || element.className == null) {
            return false;
        }
        
        // break out style classes into an array  
        var classes = webui.suntheme4_2.common.splitStyleClasses(element);
        if (classes == null) {
            return false;
        }
        
        // For each styleClass, check if it's hidden and remove otherwise write
        // it back out to the class
        for (var i = 0; i < classes.length; i++) {
            if (classes[i] != null && classes[i] == styleClass) {
                classes.splice(i,1);  	
            }
        }
        element.className = classes.join(" ");
        return true;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Submit functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Use this function to insert a hidden field element in the page.
     *
     * @param {String} elementId The element ID of the html tag 
     * @param {String} elementValue The value of the html tag.
     * @param {Node} parentForm The parent form of the html tag.
     * @return {boolean} true if successful; otherwise, false.
     */
    insertHiddenField: function(elementId, elementValue, parentForm) {
        // We have to assume that there is only one element
        // with elementId. document.getElementById, returns
        // the first one it finds, which appears to be the 
        // first one created dynamically, if more than one 
        // element is created dynamically with the same id.
        //
        // appendChild just appends even if there is an element
        // with the same id that exists.
        //
        // The assumption is that there should only be 
        // one element in the document with such an id.
        //
        // If the elementId exists just modifiy its value
        // instead of creating and appending.
        //
        
        var element = document.forms[parentForm.id].elements[elementId];
        if (element != null) {
            element.value = elementValue;            
            return true;            
        } 
        
        var newElement = document.createElement('input');
        newElement.type = 'hidden';
        newElement.id = elementId;
        newElement.value = elementValue;
        newElement.name = elementId;
        parentForm.appendChild(newElement);

        return true;
    },
    
    /**
     * Use this function to submit a virtual form.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    submitForm: function() {
        // "formToSubmit" is a literal (not virtual) form.
        // "submissionComponentId" is a component id (not client id).
        // the virtual form implementation uses _submissionComponentId
        // to determine which virtual form (if any) was submitted.
        if (webui.suntheme4_2.common._formToSubmit == null) {
            return false;
        }
        if (webui.suntheme4_2.common._submissionComponentId != null &&
                webui.suntheme4_2.common._submissionComponentId.length > 0) {
            webui.suntheme4_2.common.insertHiddenField('_submissionComponentId', 
                webui.suntheme4_2.common._submissionComponentId,
                webui.suntheme4_2.common._formToSubmit);
        }
        webui.suntheme4_2.common._formToSubmit.submit();
        return false;
    },
    
    /**
     * Helper function to submit a virtual form.
     *
     * @param {Node} form The HTML form element to submit.
     * @param {String} submissionComponentId The Id of the component submitting the form.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    timeoutSubmitForm: function(form, submissionComponentId) {
        webui.suntheme4_2.common._formToSubmit = form;
        webui.suntheme4_2.common._submissionComponentId = submissionComponentId;
        setTimeout('webui.suntheme4_2.common.submitForm()', 0);
        return true;
    },
    
    /**
     * Helper function to submit a virtual form.
     *
     * @param {Node} form The HTML form element to submit.
     * @param {String} submissionComponentId The Id of the component submitting the form.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    leaveSubmitterTrace: function(form, submissionComponentId) {
        // This function only needs to be called in the onclick handler of 
        // an ActionSource component that appears within a -standard- table.
        // Under those circumstances, if this function is not called, then when
        // the component is clicked, the virtual form implementation will have 
        // no way of knowing that a virtual form was submitted.
        if (form != null && submissionComponentId != null && 
                submissionComponentId.length > 0) {
            webui.suntheme4_2.common.insertHiddenField('_submissionComponentId',
            submissionComponentId, form);
        }
        return true;
    },
    
    /**
     * delete a previously created element by createSubmittableArray.
     *
     * @param {String} name The element ID of the html tag 
     * @param {Node} form The HTML form element to submit.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    deleteSubmittableArray: function(name, parentForm) {
        try {
            var submittableArray  = document.getElementById(name);
            if (submittableArray != null) {
                parentForm.removeChild(submittableArray);
            }
        } catch (e) {}
        return true;
    },
    
    /**
     * This method creates a hidden "select" element with id 
     * and name attributes set name, values taken from the values
     * array argument, and display labels from the labels array.
     * It adds the element to the parentForm argument.
     * <p>
     * The pupose of this method is to create an array of values
     * that can be decoded using "name" as the key from a FacesContext
     * external context's "getRequestParameterValuesMap" as an
     * array of Strings. This reduces the need of rendering hidden input
     * field and delimiting several strings so that a multiple selection
     * can be returned.
     * </p><p>
     * The labels array provides an additional piece of data
     * for use on the client, but it is not contained in the submit.
     * All values added to the select are selected so that the
     * values will be submitted.
     * </p>
     *
     * @param {String} name The element ID of the html tag 
     * @param {Node} form The HTML form element to submit.
     * @param {Array} labels
     * @param {Array} values
     * @return {Node} The newly created select element.
     * @private
     */
    createSubmittableArray: function(name, parentForm, labels, values) {
        // An attempt is made to remove a possibly previously created element
        // by this name. It always deletes an element of name from parentForm.
        webui.suntheme4_2.common.deleteSubmittableArray(name, parentForm);
        
        if (values == null || values.length <= 0) {
            return null;
        }
        
        var selections = document.createElement('select');
        selections.className = webui.suntheme4_2.theme.common.getClassName("HIDDEN");
        selections.name = name;
        selections.id = name;
        selections.multiple = true;
        
        // Start from the end of the array because
        // add puts things in at the head.
        //
        for (var i = 0; i < values.length; ++i) {
            var opt = document.createElement('option');
            opt.value = values[i];
            if (labels != null) {
                opt.label = labels[i];
            }
            opt.defaultSelected = true;
            selections.add(opt, null);
        }
        parentForm.appendChild(selections);
        return selections;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Visible functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Use this function to test if the specified element is visible (i.e., it
     * does not contain the hidden className).
     *
     * @param {String} elementId The element ID of the html tag 
     * @return {boolean} true if visible; otherwise, false
     */
    isVisible: function(elementId) {
        if (elementId == null) {
            return false;
        }
        // Get element.
        var element = document.getElementById(elementId);
        return webui.suntheme4_2.common.isVisibleElement(element);
    },
    
    /**
     * Use this function to test if the given element is visible (i.e., it
     * does not contain the hidden className).
     *
     * @param {Node} element The HTML element
     * @return {boolean} true if visible; otherwise, false
     */
    isVisibleElement: function(element) {
        if (element == null) {
            return false;
        }
        // Test for the hidden style class.
        var styleClasses = webui.suntheme4_2.common.splitStyleClasses(element); 
        return !webui.suntheme4_2.common.checkStyleClasses(styleClasses,
            webui.suntheme4_2.theme.common.getClassName("HIDDEN"));
    },
    
    /**
     * Use this function to show or hide any html element in the page
     *
     * @param {String} elementId The element ID of the html tag 
     * @param {boolean} visible true to make the element visible, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     */
    setVisible: function(elementId, visible) {
        if (elementId == null || visible == null ) {
            return false;
        }
        // Get element.
        var element = document.getElementById(elementId);
        return webui.suntheme4_2.common.setVisibleElement(element, visible);
    },
    
    /**
     * Use this function to show or hide any html element in the page
     *
     * @param {Node} element The HTML element
     * @param {boolean} visible true to make the element visible, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     */
    setVisibleElement: function(element, visible) {
        if (element == null || visible == null) {
            return false;
        }
        if (visible) {
            return webui.suntheme4_2.common.stripStyleClass(element,
                webui.suntheme4_2.theme.common.getClassName("HIDDEN"));
        } else {
            return webui.suntheme4_2.common.addStyleClass(element,
                webui.suntheme4_2.theme.common.getClassName("HIDDEN"));
        }
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.cookie");

/**
 * @class This class contains functions to manipulate cookies.
 * @static
 */
webui.suntheme4_2.cookie = {
    /**
     * This function will get the cookie value.
     *
     * @return {String} The cookie value.
     */
    get: function() {
        // Get document cookie.
        var cookie = document.cookie;

        // Parse cookie value.
        var pos = cookie.indexOf(this.$cookieName + "=");
        if (pos == -1) {
            return null;
        }

        var start = pos + this.$cookieName.length + 1;
        var end = cookie.indexOf(";", start);
        if (end == -1) {
            end = cookie.length;
        }

        // return cookie value
        return cookie.substring(start, end);
    },

    /**
     * This function will load the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    load: function() {
        // Get document cookie.
        var cookieVal = this.get();
        if (cookieVal == null) {
            return false;
        }

        // Break cookie into names and values.
        var a = cookieVal.split('&');

        // Break each pair into an array.
        for (var i = 0; i < a.length; i++) {
            a[i] = a[i].split(':');
        }

        // Set name and values for this object.
        for (i = 0; i < a.length; i++) {
            this[a[i][0]] = unescape(a[i][1]);
        }
        return true;
    },

    /**
     * This function will reset the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    reset: function() {
        // Clear cookie value.
        document.cookie = this.$cookieName + "=";
        return true;
    },

    /**
     * This function will store the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    store: function() {
        // Create cookie value by looping through object properties
        var cookieVal = "";

        // Since cookies use the equals and semicolon signs as separators,
        // we'll use colons and ampersands for each variable we store.
        for (var prop in this) {
            // Ignore properties that begin with '$' and methods.
            if (prop.charAt(0) == '$' || typeof this[prop] == 'function') {
                continue;
            }
            if (cookieVal != "") {
                cookieVal += '&';
            }
            cookieVal += prop + ':' + escape(this[prop]);
        }
        var cookieString = this.$cookieName + "=" + cookieVal;
        if (this.$path != null) {
            cookieString += ";path=" + this.$path;
        }
        // Store cookie value.
        document.cookie = cookieString;
        return true;
    }
};

/**
 * @class This class contains functionality for scrolling.
 * @constructor This function is used to construct a javascript object for
 * maintaining scroll position via cookie.
 * @param {String} viewId
 * @param {String} path
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.scrollCookie = function(viewId, path) {    
    // All predefined properties of this object begin with '$' because
    // we don't want to store these values in the cookie.
    this.$cookieName = viewId;
    this.$path = path;

    // Default properties.
    this.left = "0";
    this.top  = "0";

    // This function will load the cookie and restore scroll position.
    this.restore = function() {
        // Load cookie value.
        this.load();
        scrollTo(this.left, this.top);
        return true;
    };

    // This function will set the cookie value.
    this.set = function() {
        var documentElement = window.document.documentElement;
        if (documentElement && documentElement.scrollTop) {
            this.left = documentElement.scrollLeft;
            this.top  = documentElement.scrollTop;
        } else {
            this.left = window.document.body.scrollLeft;
            this.top  = window.document.body.scrollTop;
        }
        // if the left and top scroll values are still null
        // try to extract it assuming the browser is IE
        if (this.left == null && this.top == null) {
            this.left = window.pageXOffset;
            this.top = window.pageYOffset;
        }
        // Store cookie value.
        this.store();
        return true;
    };
    return true;
};

// Inherit cookie properties.
webui.suntheme4_2.scrollCookie.prototype = webui.suntheme4_2.cookie;

/**
 * @class This class contains functions used to maintain focus and scroll position.
 * <p>
 * There can be an initial focus element and a default focus element. The
 * initial focus element is identifed by the "focusElementId" argument.
 * This argument is typically null on the first display of the page. If
 * the Body component is not preserving the focus then it may also be null,
 * at other times, since it represents the element id that last received
 * the focus before the request.
 * </p><p>
 * Whenever the page is displayed and "focusElementId" is null
 * "defaultFocusElementId" will receive the focus if it is not null. This id is
 * defined by the application using the Body "focus" attribute. If the
 * application wants to control the focus in all cases then it should set
 * the Body component "preserveFocus" attribute to "false". The application then
 * explicitly sets the Body "focus" attribute to the element id to receive
 * focus on every request/response.
 * </p><p>
 * In order to preserve focus across requests, the "focusElementFieldId"
 * element is used to preserve the id of the element that receives the focus
 * last. It is a hidden field that is submitted in the
 * request. See the "com.sun.webui.jsf.util.FocusManager" class for
 * information on how the focus is managed. This field exists in all
 * forms, since that it is the only way to guarantee that the last
 * element that received the focus is sent to the server. Which form
 * is submitted can never be known.
 * </p>
 * @constructor This function is used to construct a body object.
 * @param {String} viewId Used to name the scroll cookie
 * @param {String} path A member of the scroll cookie
 * @param {String} defaultFocusElementId The HTML element id that will receive focus.
 * @param {String} focusElementId The id of the element to receive the initial focus.
 * @param {String} focusElementFieldId The id of a hidden field to maintain
 * the id of the last element to have the focus.
 * @param {boolean} preserveScroll if true or not defined the scroll position is 
 * maintained else scroll is not maintained.
 */
webui.suntheme4_2.body = function(viewId, path, defaultFocusElementId, 
	focusElementId, focusElementFieldId, preserveScroll)  {
    /**
     * The id of the HTML element to receive the focus, if the
     * element identified in focusElementFieldId cannot receive the focus.
     */
    this.defaultFocusId = defaultFocusElementId;

    /**
     * The id of a hidden input element whose value is the id
     * of the HTML element to receive the focus. It is set by
     * the focusListener and calls to setFocusBy{Id,Element}.
     */
    this.focusElementFieldId = focusElementFieldId;

    /**
     * The element id to receive the preserved, or initial focus.
     * This member should not be referenced once the onload listener
     * has been invoked. After that point the hidden field will have
     * have the element with the focus. We do this so that two locations
     * do not have to be kept in sync. Developers can just set the focus
     * to the element itself and the focus handler will manage the
     * focus persisitence.
     */
    this.focusElementId = focusElementId;

    // "==" also handles "null"
    //
    this.preserveScroll = (preserveScroll == null)
        ? true : new Boolean(preserveScroll).valueOf();

    /**
     * Create the scroll cookie object.
     */
    if (this.preserveScroll == true) {
	this.scrollCookie = new webui.suntheme4_2.scrollCookie(viewId, path);
    }

    /**
     * According to HTML spec only these elements have
     * "onfocus" which we will equate to "focus".
     * A, AREA, BUTTON, INPUT, LABEL, SELECT, TEXTAREA
     * However just check for a non null "focus" or 
     * catch the exception when trying to reference it.
     * Returns true if the element has a "focus" method, is not
     * disabled, and isVisible, else false.
     *
     * @param {Node} element The DOM node to have focus.
     * @return {boolean} true if DOM Node can have focus.
     */
    this.canAcceptFocus = function(element) {
	var result = false;
	try {
	    result = element != null && element.focus && !element.disabled
		&& element.type != "hidden"
		&& webui.suntheme4_2.common.isVisible(element.id);
	} catch(err) {}
	return result;
    };

    /**
     * Record the id of the element that has just receivied focus.
     * This is called whenever an element receives the focus.
     * This is set on the document so that the cursor entering the
     * window does not trigger this listener.
     *
     * @param {Event} event The object generated by the focus event.
     * @return {boolean} true if successful; otherwise, false.
     */
    this.focusListener = function(event) {
	// If it's not an element node just return
	//
	var node = null;
	var isElementNode = false;
	
	// is IE 
	//
	if (document.attachEvent) {
	    node = event.srcElement;
	
	    // We have to hard code "1" as the Node.ELEMENT_NODE in
	    // ie, because ie does not make the constant accessible.
	    //
	    isElementNode = (node == null ? false : node.nodeType == 1);
	} else {
	    node = event.target;
	    isElementNode = node.nodeType == Node.ELEMENT_NODE;
	}

	if (isElementNode) {
	    // Note that there is no reliable way to set
	    // focus to some other element if the event element
	    // deemed to receive the focus can't accept the focus.
	    //
	    this.updateFocusElementField(node);
	}
	return true;
    };

    /**
     * Set the initial focus and the scroll position.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    this.onLoadListener = function() {
	// register the focus listener first.
	// Then call "setDefaultFocus" using "setTimeout" to
	// allow javascript processing to complete, probably
	// to allow "onload" to complete. The focus listener
	// will update the hidden focus fields as a result
	// of the call to "focus" in setDefaultFocus.
	//

	// Add the focus listener, in the onload to prevent
	// recursive calls from calling setDefaultFocus.
	//
        if (webui.suntheme4_2.browser.isIe()) {
            webui.suntheme4_2.dojo.connect(document, "onfocusin", this, "focusListener");
        } else {
            webui.suntheme4_2.dojo.connect(window, "onfocus", this, "focusListener");
        }

        // Rely on the focus listener to update the focus
        // hidden fields by catching the 'element.focus()' in
        // setDefaultFocus
        //
        this.setInitialFocus();

	// Set up the scroll position after the focus has been
	// restored. Need to make sure that this takes into
	// account the default focus that was just set.
	//
	return this.setDefaultScrollPosition();
    };

    /**
     * Update the page's scroll position.
     *
     * @param {Event} event The object generated by the onUnload event.
     * @return {boolean} true if successful; otherwise, false.
     */
    this.onUnloadListener = function(event) {
	return this.storeScrollPosition();
    };

    /**
     * Set the default focus to the application's chosen default focus element.
     * This method should only be called once to prevent recursive
     * calls since it calls "focus()" on the focus element.
     * Called currently from the onload listener.
     * <p>
     * If "this.defaultFocusId" is not null it will receive the focus; 
     * otherwise, no focus is set.
     * </p>
     * @return {boolean} false if a default focus cannot be established, else true.
     */
    this.setDefaultFocus = function() {
        // HTML elements may not have been added, yet.
        if (this.defaultFocusId != null) {
            var domNode = document.getElementById(this.defaultFocusId);
            if (domNode == null) {
                var _this = this; // Closure magic.
                return setTimeout(function() { _this.setDefaultFocus(); }, 10);
            }

            // Focus not set try the default.
            //
            if (this.setFocusById(this.defaultFocusId)) {
                return true;
            }
        }

	/* For now it doesn't sound like a good idea to ever set
	 * a "heuristic" default focus. It is better for screen readers to start
	 * from the top of the page which we will assume that that
	 * browser starts from there when focus is not set explicitly.
	 * This code can be removed, but left it in case we want to
	 * for some reason.

	// No previously set focus element and no default.
	// Find the first focusable element in the first available form.
	//
	for each (var f in window.document.forms) {
	    for each (var e in f.elements) {
		if (this.setFocusByElement(e)) {
		    return true;
		}
	    }
	}
	// If there is no form, set on the first available link
	//
	for each (var l in window.document.links) {
	    if (this.setFocusByElement(l)) {
		return true;
	    }
	}
	*/
	return false;
    };

    /**
     * This method is invoked in the onload listener, body.onLoadListener.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    this.setDefaultScrollPosition = function() {
	if (!this.preserveScroll) {
	    return false;
	}
	// # char found, anchor being used. forego scrolling.
	// CR 6342635. 
	//
        if (window.location.href.indexOf('#') == -1) {
	    this.scrollCookie.restore(); 
	} else {
	    // destroy the recent scroll data
	    //
	    this.scrollCookie.reset();
	}
        return true;
    };

    /**
     * Set the initial focus by restoring the focus from a previous
     * request or to the application's chosen default focus element.
     * This method should only be called once to prevent recursive
     * calls since it calls "focus()" on the focus element.
     * Called currently from the onload listener.
     * <p>
     * If "this.focusElementId" is not null it will receive the focus.
     * If that element can't receive the focus then the application defined 
     * "this.defaultFocusId" receives the focus. If that element cannot receive 
     * the focus, no focus is set.
     * </p>
     * @return {boolean} false if focus cannot be established, else true.
     */
    this.setInitialFocus = function() {
        // HTML elements may not have been added, yet.
        if (this.focusElementId != null) {
            var domNode = document.getElementById(this.focusElementId);
            if (domNode == null) {
                var _this = this; // Closure magic.
                return setTimeout(function() { _this.setInitialFocus(); }, 10);
            }

            // Try to set focus to "this.focusElementId". If this fails
            // fallback to the app defined default 
            // "this.defaultFocusElementId", if there is one.
            //
            if (this.setFocusById(this.focusElementId)) {
                return true;
            }
        }
        return this.setDefaultFocus();
    };

    /**
     * Set the focus on "focusElement".
     * If focus can be set returns true else false.
     *
     * @param {Node} focusElement The DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     */
    this.setFocusByElement = function(focusElement) {
	if (focusElement == null || !this.canAcceptFocus(focusElement)) {
	    return false;
	}

	// canAcceptFocus tests the existence of the "focus" handler.
	// So it is safe to call it outside of a try/catch statement.
	// This should trigger the focus listener.
        try {
            // Still need try/catch because canAcceptFocus doesn't account for 
            // when parent is invisible. For example, the table's sort panel 
            // closes during page submit making focus element invisible.
            focusElement.focus();
        } catch(err) {}

	// Assume that this update is performed by the 
	// focus listener. This policy was changed in order to 
	// call "setDefaultFocus" using "setTimeout" in order for
	// javascript to have time to be evaluated, probably for
	// on load processing to complete.
	//this.updateFocusElementField(focusElement);
	return true;
    };

    /**
     * Set the focus on element with id "fid".
     * If focus can be set returns true else false.
     *
     * @param {String} fid The id of the DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     */
    this.setFocusById = function(fid) {
	if (fid == null || fid.length == 0) {
	    return false;
	}
	return this.setFocusByElement(document.getElementById(fid));
    };

    /**
     * This method is invoked in the onunload event listener
     * body.onUnloadListener
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    this.storeScrollPosition = function() {
	if (!this.preserveScroll) {
	    return false;
	}
	try {
	    this.scrollCookie.set(); 
	} catch (e) {
	}
        return true; 
    };

    /** 
     * Update the hidden field that maintains the last element to 
     * receive the focus. If the body has multiple forms every form's
     * hidden field is updated with the "focusElement".
     *
     * @param {Node} focusElement The DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     */
    this.updateFocusElementField = function(focusElement) {
	// Don't know if we'll have issues if multiple forms contain
	// an element with the same id. I know getElementById gets
	// confused.
	//

        if (focusElement == null) {
	    return false;
	}
	// Get the form that contains the focus element.
	//
	for (var i = 0;  i < document.forms.length; ++i) {
	    var form = document.forms[i];
            var field = null;

	    // Get the hidden field that maintains the focus element id.
	    // If it exists return it. We know its name is the same
	    // as its id.
	    //
	    try {
		field = form.elements[this.focusElementFieldId];
		if (field != null) {
		    field.value = focusElement.id;
		    continue;
		}
	    } catch (e) {
		// the array access of a non existent element
		// probably threw exception so create the field.
	    }
		
	    // If it doesn't exist create it.
	    // and add it to the form.
	    //
	    field = document.createElement('input');
	    field.type = 'hidden';
	    field.id = this.focusElementFieldId;
	    field.name = this.focusElementFieldId;
	    field.value = focusElement.id;
	    form.appendChild(field);
	}
	return true;
    };

    // The focus listener is set on the document so that the cursor 
    // entering the window does not trigger this listener. 
    this.onLoadListener();

    // If we are not preserving scroll don't add the unload listener.
    if (this.preserveScroll == true) {
        webui.suntheme4_2.dojo.addOnUnload(this, "onUnloadListener");
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.commonTasksSection");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.theme.common");

/** 
 * @class This class contains functions for commonTasksSection components.
 * @static
 */
webui.suntheme4_2.commonTasksSection = {
    /**
     * This function is used to initialize HTML element properties with Object 
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The HTML element id.
     * @config {String} pic1URL Selected image.
     * @config {String} pic2URL Hover image.
     * @config {String} pic3URL Normal image.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize commonTasksSection.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        webui.suntheme4_2.prototypejs.extend(domNode, props, false);

        // Set functions.
	domNode.captureCloseKey = webui.suntheme4_2.commonTasksSection.captureCloseKey;
	domNode.captureBottomInfoKey = webui.suntheme4_2.commonTasksSection.captureBottomInfoKey;
        domNode.hideAll = webui.suntheme4_2.commonTasksSection.hideAll;
        domNode.addCommonTask = webui.suntheme4_2.commonTasksSection.addCommonTask;
        domNode.addInfoPanel = webui.suntheme4_2.commonTasksSection.addInfoPanel;
        domNode.windowResize = webui.suntheme4_2.commonTasksSection.windowResize;
        domNode.onclick = domNode.hideAll;

        // Set task element array.
        domNode.taskElement = new Array();
        domNode.count = 0;

        // Hide panels on resize.
        webui.suntheme4_2.dojo.connect(window, 'onresize', domNode, domNode.windowResize);

        return true;
    },

    /**
     * Hide all task sections.
     *
     * @param {Event} event The JavaScript event.
     * @return {boolean} true if successful; otherwise, false.
     */
    hideAll: function(event) {
        for (var i = 0; i < this.count; i++) {
            task = this.taskElement[i];
            if (task.infoPanel) {
               webui.suntheme4_2.common.setVisibleElement(task.infoPanel.info, false);
               task.infoPanel.image.src = this.pic3URL;
            }
        }
        if (webui.suntheme4_2.browser.isIe5up()) {
            window.event.cancelBubble = true;
        } else {
            event.stopPropagation();
        }
        return true;
    },

    /**
     * This function handles window resize events.
     *
     * @param {Event} event The JavaScript event.
     * @return {boolean} true if successful; otherwise, false.
     */
    windowResize: function(event) {
        for (var i = 0; i < this.count; i++) {
            task = this.taskElement[i];
            if (task.infoPanel) {
               webui.suntheme4_2.common.setVisibleElement(task.infoPanel.info, false);
               task.infoPanel.image.src = this.pic3URL;
            }
        }
        return true;
    },

    /**
     * This function is used to set common task properties using Object literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} commonTaskId
     * @config {String} closeId
     * @config {String} spacerId
     * @config {String} infoIconId
     * @config {String} infoPanelVar
     * @config {String} imageLinkId
     * @return {boolean} true if successful; otherwise, false.
     */
    addCommonTask: function(props) {
        if (props == null) {
            return false;
        }

        // Get HTML elements.
        var info = document.getElementById(props.commonTaskId + props.infoPanelVar);  //id of the info panel box.
        var image = document.getElementById(props.infoIconId); // id of the "i" image .
        var imageLink = document.getElementById(props.imageLinkId);
        var close = document.getElementById(props.closeId); // id of the close button.	
        var parent = document.getElementById(this.id);
        var task = document.getElementById(props.commonTaskId);
        var bottomInfoLink = (props.bottomInfoLink)
            ? document.getElementById(props.bottomInfoLink) 
            : null; // The bottom info panel id.
        var spacer = props.commonTaskId + ":" + props.spacerId; // id of the spacer image.

        // HTML elements may not have been created, yet.
        if (parent == null
                || (props.bottomInfoLink && bottomInfoLink == null)
                || (props.closeId && close == null)
                || (props.commonTaskId && props.infoPanelVar && info == null)
                || (props.commonTaskId && task == null)
                || (props.infoIconId && image == null)
                || (props.imageLinkId && imageLink == null)) {
            return setTimeout(function() {
                parent.addCommonTask(props);
            }, 10);
        }

        // Set info panel.
        var taskElement = document.getElementById(props.commonTaskId);
        taskElement.infoPanel = new this.addInfoPanel(info, image, imageLink,
            close, parent, task, bottomInfoLink, spacer);

        // Add task element to domNode.
        this.taskElement[this.count] = taskElement;
        this.count++;
        return true;
    },
    
    /**
     * Add info panel to common task section.
     *
     * @param {Node} info The info panel box.
     * @param {Node} image The info panel icon
     * @param {Node} imageLink
     * @param {Node} close The close button.
     * @param {Node} parent
     * @param {Node} task
     * @param {Node} bottomInfoLink The bottom info panel link.
     * @param {String} spacer ID of the spacer image.
     * @return {boolean} true if successful; otherwise, false.
     */
    addInfoPanel: function(info, image, imageLink, close, parent, task, 
            bottomInfoLink, spacer) {
        // Set HTML elements.
        this.info = info;
        this.image = image;
        this.imageLink = imageLink;
        this.close = close;
        this.parent = parent;
        this.task = task;
        this.bottomInfoLink = bottomInfoLink;
        this.spacer = spacer;
        this.theme = webui.suntheme4_2.theme.common;
        
        var that = this;

        // Handle the keypress event for the "more" link if one is present.
        // Tabbing out of the info panel should close the info panel whereas
        // pressing escape key should close the info panel tooo
        if (this.bottomInfoLink) {
            
            this.bottomInfoLink.setProps({
                
                // Only for IE.                
                onKeyDown:function(event) {
                    if (webui.suntheme4_2.browser.isIe5up()) {

                        // For IE, while pressing the shift key along with the tab key
                        // the onkeydown seems to be called twice. To prevent this,
                        // check whether the shift key is the one thats being pressed
                        // before actually calling the event handling function.
                        if (!(window.event.keyCode == 16)) {
                            that.captureBottomInfoKey(window.event);
                        }
                    }
                    return false;
                },
                onKeyPress:function(event) {	    
                    var evt = (event) ? event : ((window.event) ? window.event : null);  
                    if (!webui.suntheme4_2.browser.isIe5up()) {
                        that.captureBottomInfoKey(event);
                    }
                    return false;                                 
                }        
            });
        }
        
        // Function that gets invoked when keypress event happens on the bottom
        // portion of the info panel.
        this.captureBottomInfoKey = function(event) {
            if ((event.keyCode == 9 && !event.shiftKey)|| event.keyCode == 27) {
                // need to remove the focus off the link. Otherwise there seems
                // to be problems setting focus on another element in IE.
                that.bottomInfoLink.blur();

                webui.suntheme4_2.common.setVisibleElement(that.info, false);
                that.image.src = that.parent.pic3URL;	
                that.imageLink.focus();
            }

            if (event.shiftKey && event.keyCode == 9) {
                that.close.focus();

                // If you dont do this, the info panel closes on IE
                // and the focus is set on the "i" icon.
                webui.suntheme4_2.common.setVisibleElement(that.info, true);
            }
            return true;
        };

        // Function that is called when the key press event happens on the
        // close image of the info panel.
        this.captureCloseKey = function(event) {
            // We want to process only key press events which have the tab key pressed.
            if (event.keyCode == 9) {
                // If this is not done IE doesnt set focus on the next available
                // element properly if the info panel closes.
                that.close.blur();     

                // If the "more" link is present, shift focus to that
                // else close the info panel on blur.
                if (that.bottomInfoLink && event.shiftKey == false) {
                    that.bottomInfoLink.focus(); 

                    // If this is not done, the info panel closes
                    // after you tab to the element on IE
                    webui.suntheme4_2.common.setVisibleElement(that.info, true);
                } else {
                    that.image.src = that.parent.pic3URL;	            
                    webui.suntheme4_2.common.setVisibleElement(that.info, false);    
                    that.imageLink.focus();
                }                                      
            }
            return true;
        };
                
        // Events which handle the closing of the div.        
        this.close.setProps({
            onClick:function(event) {
                webui.suntheme4_2.common.setVisibleElement(that.info, false);
                that.image.src = that.parent.pic3URL;	
                if (webui.suntheme4_2.browser.isIe5up()) {
                    window. event.cancelBubble = true;
                } else {
                    event.stopPropagation();
                }
                that.task.focus();
                return false;
            },        
            
            // Need to do this only on IE. "Tab" key doesnt get registered
            // for keypress on IE.                        
            onKeyDown:function(event) {
                if (webui.suntheme4_2.browser.isIe5up()) {

                    // this seems to be called once for the shift key and
                    // once for the tab key. Prevent calling the capture
                    // function when the shift key is pressed
                    if (!(window.event.keyCode == 16)) {
                        that.captureCloseKey(window.event);
                    }

                    // If escape key is pressed, the info panel must close.
                    if (window.event.keyCode == 27 || window.event.keyCode == 13) {
                        webui.suntheme4_2.common.setVisibleElement(that.info, false);
                        that.image.src = that.parent.pic3URL;
                        that.imageLink.focus();
                    }                
                    return false;
                }
                return true;
            },
            
            // Handle the keypress event on the close imageHyperlink.
            // If tab key is pressed, the focus must either pass to
            // the "more" link if it is present or the infoPanel should close. 
            onKeyPress:function(event) {              
                var evt = (event) ? event : ((window.event) ? window.event : null);         
                if (!webui.suntheme4_2.browser.isIe5up()) {
                    that.captureCloseKey(evt);
                }
                // If escape key is pressed, the info panel must close.
                if (evt.keyCode == 27 || evt.keyCode == 13) {
                    webui.suntheme4_2.common.setVisibleElement(that.info, false);
                    that.image.src = that.parent.pic3URL;
                    that.imageLink.focus();
                }
                return false;
            }
        });
        
        this.info.onclick = function(event) {
            webui.suntheme4_2.common.setVisibleElement(that.info, true);
            if (webui.suntheme4_2.browser.isIe5up()) {
                window. event.cancelBubble = true;
            } else {
                event.stopPropagation();
            }
            return true;
        };
        
        // Events which handle the image changes for the "i" image.
        this.imageLink.setProps({
            onMouseOver:function() {
                if (!webui.suntheme4_2.common.isVisibleElement(that.info)) {
                    that.image.src = that.parent.pic2URL;
                } else {
                    that.image.src = that.parent.pic1URL;
                }
                return true;
            },        
            onFocus:function() {
                if (!webui.suntheme4_2.common.isVisibleElement(that.info)) {
                    that.image.src = that.parent.pic2URL;
                } else {
                    that.image.src = that.parent.pic1URL;
                }
                return true;
            },        
            onBlur:function() {
                  if (!webui.suntheme4_2.common.isVisibleElement(that.info)) {
                    that.image.src = that.parent.pic3URL;
                } else {
                    that.image.src = that.parent.pic1URL;
                }
                return true;
            },
            onMouseOut:function() {
                if (!webui.suntheme4_2.common.isVisibleElement(that.info)) {
                    that.image.src = that.parent.pic3URL;
                } else {
                    that.image.src = that.parent.pic1URL;
                }
                return true;
            },
            onKeyPress:function(event) {
                var evt = (event) ? event : ((window.event) ? window.event : null);            
                if (evt.keyCode == 13) {
                    that.showInfoPanel();
                    return false;                
                }
                if (webui.suntheme4_2.browser.isIe5up()) {
                    window.event.cancelBubble = true;
                } else {
                    event.stopPropagation();
                }
                return true;
            }            
        });
        
       // Toggle functionality incorporated
        this.image.setProps({onClick:function(event){
            that.showInfoPanel();
            if (webui.suntheme4_2.browser.isIe5up()) {
                window.event.cancelBubble = true;
            } else {
                event.stopPropagation();
            }
            return true;
        }});
        
        this.showInfoPanel = function() {
            var cts = this.parent;
            for (var i = 0; i < cts.count; i++) {
                task = cts.taskElement[i];
                if (task.infoPanel != null
                        && task.infoPanel.image.id != this.image.id) {
                    webui.suntheme4_2.common.setVisibleElement(task.infoPanel.info, false);
                    task.infoPanel.image.src = cts.pic3URL;
                }
            }
 
            if (!webui.suntheme4_2.common.isVisibleElement(this.info)) {
                webui.suntheme4_2.common.setVisibleElement(this.info, true);
                this.getElementPosition2(this.image.id);
                this.getElementPosition(this.task.id);        
                this.info.style.top = (this.ttop + parseInt(this.theme.getMessage("commonTasks.infoPanelOffsetTop"))) +'px';
                this.info.style.left =  (this.tleft - 1) + 'px';
                this.info.style.width = (this.ileft - this.tleft) + 29 + 'px';
                this.close.focus();
                this.image.src = cts.pic1URL;
            } else {
                this.image.src = cts.pic3URL;
                webui.suntheme4_2.common.setVisibleElement(this.info, false);
            }
            return true;
        };

        // Javascript for setting the common task page's look and feel.

        // The prized coordinate locating function - Thank you Danny Goodman...
        this.getElementPosition = function(elemID) {
            var offsetTrail = document.getElementById(elemID);
            var offsetLeft = 0;
            var offsetTop = 0;

            while (offsetTrail) {
                offsetLeft += offsetTrail.offsetLeft;
                offsetTop += offsetTrail.offsetTop;
                offsetTrail = offsetTrail.offsetParent;
            }
            if (navigator.userAgent.indexOf("Mac") != -1 
                    && typeof document.body.leftMargin != "undefined") {
                offsetLeft += document.body.leftMargin;
                offsetTop += document.body.topMargin;
            }
            this.tleft = offsetLeft;
            this.ttop = offsetTop;
            return true;
        };

        this.getElementPosition2 = function(elemID) {
            var offsetTrail = document.getElementById(elemID);
            var offsetLeft = 0;
            var offsetTop = 0;

            while (offsetTrail) {
                offsetLeft += offsetTrail.offsetLeft;
                offsetTop += offsetTrail.offsetTop;
                offsetTrail = offsetTrail.offsetParent;
            }
            if (navigator.userAgent.indexOf("Mac") != -1
                    && typeof document.body.leftMargin != "undefined") {
                offsetLeft += document.body.leftMargin;
                offsetTop += document.body.topMargin;
            }
            this.ileft = offsetLeft;
            return true;
        };
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.editableList");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.formElements");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.common");

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// button functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for button components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.button
 */
webui.suntheme4_2.button = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated See webui.suntheme4_2.widget.button
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize button.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var widget = webui.suntheme4_2.dijit.byId(props.id);
        if (widget == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set functions
        widget.domNode.isSecondary = webui.suntheme4_2.button.isSecondary;
        widget.domNode.setSecondary = webui.suntheme4_2.button.setSecondary;
        widget.domNode.isPrimary = webui.suntheme4_2.button.isPrimary;
        widget.domNode.setPrimary = webui.suntheme4_2.button.setPrimary;
        widget.domNode.isMini = webui.suntheme4_2.button.isMini;
        widget.domNode.setMini = webui.suntheme4_2.button.setMini;
        widget.domNode.getDisabled = webui.suntheme4_2.button.getDisabled;
        widget.domNode.setDisabled = webui.suntheme4_2.button.setDisabled;
        widget.domNode.getVisible = webui.suntheme4_2.button.getVisible;
        widget.domNode.setVisible = webui.suntheme4_2.button.setVisible;
        widget.domNode.getText = webui.suntheme4_2.button.getText;
        widget.domNode.setText = webui.suntheme4_2.button.setText;
        widget.domNode.doClick = webui.suntheme4_2.button.click;

        return true;
    },

    /**
     * Simulate a mouse click in a button. 
     *
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).click();
     */
    click: function() {
        return this.click();
    },

    /**
     * Get the textual label of a button. 
     *
     * @return {String} The element value.
     * @deprecated Use document.getElementById(id).getProps().value;
     */
    getText: function() {
        return this.getProps().value;
    },

    /**
     * Set the textual label of a button. 
     *
     * @param {String} text The element value
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({value: "text"});
     */
    setText: function(text) {
        return this.setProps({value: text});
    },

    /**
     * Use this function to show or hide a button. 
     *
     * @param {boolean} show true to show the element, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({visible: boolean});
     */
    setVisible: function(show) {
        if (show == null) {
            return null;
        }
        return this.setProps({visible: show});
    },

    /**
     * Use this function to find whether or not this is visible according to our
     * spec.
     *
     * @return {boolean} true if visible; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().visible;
     */
    getVisible: function() {
        return this.getProps().visible;
    },

    /**
     * Test if button is set as "primary".
     *
     * @return {boolean} true if primary; otherwise, false for secondary
     * @deprecated Use document.getElementById(id).getProps().primary;
     */
    isPrimary: function() {
        return this.getProps().primary;
    },

    /**
     * Set button as "primary".
     *
     * @param {boolean} primary true for primary, false for secondary
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({primary: boolean});
     */
    setPrimary: function(primary) {
        if (primary == null) {
            return null;
        }
        return this.setProps({primary: primary});
    },

    /**
     * Test if button is set as "secondary".
     *
     * @return {boolean} true if secondary; otherwise, false for primary
     * @deprecated Use !(document.getElementById(id).getProps().primary);
     */
    isSecondary: function() {
        return !(this.getProps().primary);
    },

    /**
     * Set button as "secondary".
     *
     * @param {boolean} secondary true for secondary, false for primary
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({primary: false});
     */
    setSecondary: function(secondary) {
        if (secondary == null) {
            return null;
        }
        return this.setProps({primary: !secondary});
    },

    /**
     * Test if button is set as "mini".
     *
     * @return {boolean} true if mini; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().mini;
     */
    isMini: function() {
        return this.getProps().mini;
    },

    /**
     * Set button as "mini".
     *
     * @param {boolean} mini true for mini, false for standard button
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({mini: boolean});
     */
    setMini: function(mini) {
        if (mini == null) {
            return null;
        }
        return this.setProps({mini: mini});
    },

    /**
     * Test disabled state of button.
     *
     * @return {boolean} true if disabled; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().disabled;
     */
    getDisabled: function() {
        return this.getProps().disabled;
    },

    /**
     * Test disabled state of button.
     *
     * @param {boolean} disabled true if disabled; otherwise, false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(disabled) {
        if (disabled == null) {
            return null;
        }
        return this.setProps({disabled: disabled});
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// checkbox functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for checkbox components.
 * @static
 * 
 * @deprecated See webui.suntheme4_2.widget.checkbox
 */
webui.suntheme4_2.checkbox = {
    /**
     * Set the disabled state for the given checkbox element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) {
        return webui.suntheme4_2.rbcb.setDisabled(elementId, disabled,
            "checkbox", "Cb", "CbDis");
    },

    /** 
     * Set the disabled state for all the checkboxes in the check box
     * group identified by controlName. If disabled
     * is set to true, the check boxes are shown with disabled styles.
     *
     * @param {String} controlName The checkbox group control name
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled) {    
        return webui.suntheme4_2.rbcb.setGroupDisabled(controlName,
            disabled, "checkbox", "Cb", "CbDis");
    },

    /**
     * Set the checked property for a checkbox with the given element Id.
     *
     * @param {String} elementId The element Id
     * @param checked true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */
    setChecked: function(elementId, checked) {
        return webui.suntheme4_2.rbcb.setChecked(elementId, checked,
            "checkbox");
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// dropdown functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for dropDown components.
 * @static
 * 
 * @deprecated See webui.suntheme4_2.widget.dropDown
 */
webui.suntheme4_2.dropDown = {
    /**
     * Use this function to access the HTML select element that makes up
     * the dropDown.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the span tag enclosing the HTML elements that make up
     * the dropDown).
     * @return {Node} a reference to the select element. 
     * @deprecated Use document.getElementById(elementId).setSelectElement()
     */
    getSelectElement: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectElement();
        }
        return null;
    },

    /**
     * This function is invoked by the choice onselect action to set the
     * selected, and disabled styles.
     *
     * Page authors should invoke this function if they set the 
     * selection using JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).changed();
     */
    changed: function(elementId) {         
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.changed();
        }
        return false;
    },

    /**
     * Set the disabled state for given dropdown element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * Page authors should invoke this function if they dynamically
     * enable or disable a dropdown using JavaScript.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({ disabled: disabled});
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the value of the first
     * selected option on the dropDown. If no option is selected, this
     * function returns null. 
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The value of the selected option, or null if none is
     * selected. 
     * @deprecated Use document.getElementById(elementId).getSelectedValue();
     */
    getSelectedValue: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectedValue();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the label of the first
     * selected option on the dropDown. If no option is selected, this
     * function returns null.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The label of the selected option, or null if none is
     * selected. 
     * @deprecated Use document.getElementById(elementId).getSelectedLabel();
     */
    getSelectedLabel: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectedLabel();
        }
        return null;
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// field functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for field components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.field
 */
webui.suntheme4_2.field = {
    /**
     * Use this function to get the HTML input or textarea element
     * associated with a TextField, PasswordField, HiddenField or TextArea
     * component.
     *
     * @param {String} elementId The element ID of the field 
     * @return {Node} the input or text area element associated with the field component
     * @deprecated Use document.getElementById(elementId).getInputElement()
     */
    getInputElement: function(elementId) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getInputElement();
        }
        return null;
    },

    /**
     * Use this function to get the value of the HTML element 
     * corresponding to the Field component.
     *
     * @param {String} elementId The element ID of the Field component
     * @return {String} the value of the HTML element corresponding to the Field component 
     * @deprecated Use document.getElementById(id).getProps().value;
     */
    getValue: function(elementId) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getProps().value;
        }
        return null;
    },

    /**
     * Use this function to set the value of the HTML element 
     * corresponding to the Field component
     *
     * @param {String} elementId The element ID of the Field component
     * @param {String} newValue The new value to enter into the input element Field component 
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({value: "text"});
     */
    setValue: function(elementId, newValue) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({value: newValue});
        }
        return null;
    },

    /** 
     * Use this function to get the style attribute for the field. 
     * The style retrieved will be the style on the span tag that 
     * encloses the (optional) label element and the input element.
     *
     * @param {String} elementId The element ID of the Field component
     * @return {String} The style property of the field.
     * @deprecated Use document.getElementById(id).getProps().style;
     */
    getStyle: function(elementId) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getProps().style;
        }
        return null;
    },

    /**
     * Use this function to set the style attribute for the field. 
     * The style will be set on the <span> tag that surrounds the field.
     *
     * @param {String} elementId The element ID of the Field component
     * @param {String} newStyle The new style to apply
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({style: newStyle});
     */
    setStyle: function(elementId, newStyle) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({style: newStyle});
        }
        return null;
    },

    /**
     * Use this function to disable or enable a field. As a side effect
     * changes the style used to render the field. 
     *
     * @param {String} elementId The element ID of the field 
     * @param {boolean} newDisabled true to disable the field, false to enable the field
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, newDisabled) {  
        if (newDisabled == null) {
            return null;
        }
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({disabled: newDisabled});
        }
        return null;
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// hyperlink functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for hyperlink components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.hyperlink
 */
webui.suntheme4_2.hyperlink = {
    /**
     * This function is used to submit a hyperlink.
     * <p>
     * Note: Params are name value pairs but all one big string array so 
     * params[0] and params[1] form the name and value of the first param.
     * </p>
     *
     * @params {Object} hyperlink The hyperlink element
     * @params {String} formId The form id
     * @params {Object} params Name value pairs
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated See webui.suntheme4_2.widget.hyperlink
     */
    submit: function(hyperlink, formId, params) {
        // Need to test widget for tab and common task components. If a widget 
        // does not exist, fall back to the old code.
	var widget = webui.suntheme4_2.dijit.byId(hyperlink.id);
	if (widget == null) {
            // If a widget does not exist, we shall create one in order to call
            // the submit function directly.
            webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.hyperlink");
            widget = new webui.suntheme4_2.widget.hyperlink({id: hyperlink.id});
	}
        return widget.submitFormData(formId, params);
    },

    /**
     * Use this function to access the HTML img element that makes up
     * the icon hyperlink.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the outter most tag enclosing the HTML img element).
     * @return {Node} The HTML image element.
     * @deprecated Use document.getElementById(elementId).getProps().enabledImage;
     */
    getImgElement: function(elementId) {
        // Need to test widget for alarmStatus, jobstatus, and notification phrase
        // components. If a widget does not exist, fall back to the old code.
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        var props = (widget) ? widget.getProps() : null;
        if (props && props.enabledImage) {
            var imgWidget = webui.suntheme4_2.dijit.byId(props.enabledImage.id);
            if (imgWidget != null) {
                return imgWidget.domNode;    
            }
        }

        // Image hyperlink is now a naming container and the img element id 
        // includes the ImageHyperlink parent id.
        if (elementId != null) {
            var parentid = elementId;
            var colon_index = elementId.lastIndexOf(":");
            if (colon_index != -1) {
                parentid = elementId.substring(colon_index + 1);
            }
            return document.getElementById(elementId + ":" + parentid + "_image");
        }
        return document.getElementById(elementId + "_image");
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// jumpDropDown functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for jumpDropDown components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.dropDown
 */
webui.suntheme4_2.jumpDropDown = {
    /**
     * This function is invoked by the jumpdropdown onchange action to set the
     * form action and then submit the form.
     *
     * Page authors should invoke this function if they set the selection using 
     * JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).changed()
     */
    changed: function(elementId) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.changed();
        }
        return false;
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// listbox functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for listbox components.
 * @static
 * 
 * @deprecated See webui.suntheme4_2.widget.listbox
 */
webui.suntheme4_2.listbox = {
    /**
     * Use this function to access the HTML select element that makes up
     * the list. 
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the span tag enclosing the HTML elements that make up
     * the list).
     * @return {Node} The HTML select element.
     * @deprecated Use document.getElementById(elementId).getSelectElement()
     */
    getSelectElement: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectElement();
        }
        return null;
    },

    /**
     * This function is invoked by the list onselect action to set the selected, 
     * and disabled styles.
     *
     * Page authors should invoke this function if they set the selection
     * using JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).changed();
     */
    changed: function(elementId) {         
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.changed();
        }
        return false;
    },

    /**
     * Invoke this JavaScript function to set the enabled/disabled state
     * of the listbox component. In addition to disabling the list, it
     * also changes the styles used when rendering the component. 
     *
     * Page authors should invoke this function if they dynamically
     * enable or disable a list using JavaScript.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the value of the first
     * selected option on the listbox. If no option is selected, this
     * function returns null. 
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The value of the selected option, or null if none is
     * selected.
     * @deprecated Use document.getElementById(elementId).getSelectedValue();
     */
    getSelectedValue: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectedValue();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the label of the first
     * selected option on the listbox. If no option is selected, this
     * function returns null. 
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The label of the selected option, or null if none is selected.
     * @deprecated Use document.getElementById(elementId).getSelectedLabel();
     */
    getSelectedLabel: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectedLabel();
        }
        return null;
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Generic checkbox and radio button functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for rbcbGroup components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.rbcbGroup
 */
webui.suntheme4_2.rbcb = {
    /**
     * 
     * @param {String} elementId The element Id.
     * @param {boolean} checked true or false
     * @param {String} type
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */ 
    setChecked: function(elementId, checked, type) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({checked: checked});
        }
        return null; 
    },

    /**
     *
     * @param {String} elementId The element Id.
     * @param {boolean} disabled true or false
     * @param {String} type
     * @param {String} enabledStyle
     * @param {String} disabledStyle
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */ 
    setDisabled: function(elementId, disabled, type, enabledStyle,
            disabledStyle) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null; 
    },

    /** 
     * Set the disabled state for all radio buttons with the given controlName.
     * If disabled is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id
     * @param {String} formName The name of the form containing the element
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled, type, enabledStyle,
            disabledStyle) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null;
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// radiobutton functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for radioButton components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.radioButton
 */
webui.suntheme4_2.radiobutton = {
    /**
     * Set the disabled state for the given radiobutton element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id.
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) {    
        return webui.suntheme4_2.rbcb.setDisabled(elementId, disabled, 
            "radio", "Rb", "RbDis");
    },

    /**
     * Set the disabled state for all the radio buttons in the radio button
     * group identified by controlName. If disabled
     * is set to true, the check boxes are displayed with disabled styles.
     *
     * @param {String} controlName The radio button group control name
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled) {    
        return webui.suntheme4_2.rbcb.setGroupDisabled(controlName, disabled, 
            "radio", "Rb", "RbDis");
    },

    /**
     * Set the checked property for a radio button with the given element Id.
     *
     * @param {String} elementId The element Id
     * @param {boolean} checked true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */
    setChecked: function(elementId, checked) {
        return webui.suntheme4_2.rbcb.setChecked(elementId, checked, "radio");
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// upload functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for upload components.
 * @static
 */
webui.suntheme4_2.upload = {
    /**
     * Use this function to get the HTML input element associated with the
     * Upload component.  
     * @param {String} elementId The client id of the Upload component
     * @return {Node} the input element associated with the Upload component
     * else null if elementId is null or "".
     */
    getInputElement: function(elementId) { 
        if (elementId == null || elementId == "") {
	    return null;
	}

	// The upload component MUST always render the input element
	// with the following suffix on the id 
	// "_com.sun.webui.jsf.upload".
	// This "binds" this version of the component to this theme
	// version.
	// This will change when "field" becomes a widget.
	//
        var element = document.getElementById(elementId + 
            "_com.sun.webui.jsf.upload");
        if (element && element.tagName == "INPUT") { 
            return element; 
        } else {
	    return null;
	}
    },

    /**
     * Use this function to disable or enable a upload. As a side effect
     * changes the style used to render the upload. 
     *
     * @param {String} elementId The client id of the upload component.
     * @param {boolean} disabled true to disable the upload, false to enable the upload
     * @return {boolean} true if successful; otherwise, false.
     */
    setDisabled: function(elementId, disabled) {  

        if (elementId == null || elementId == "" || 
		disabled == null || disabled == "") {
            // must supply an elementId && state
            return false;
        }
        var input = webui.suntheme4_2.upload.getInputElement(elementId); 
        if (input == null) {
            // specified elementId not found
            return false;
        }
        input.disabled = disabled;
	return true;
    },

    /**
     * Set the encoding type of the form to "multipart/form-data".
     * 
     *
     * @param {String} elementId The client id of the upload component.
     * @return {boolean} true if encoding type can be set, else false.
     */
    setEncodingType: function(elementId) { 
	if (elementId == null || elementId == "") {
	    return false;
	}

        var upload = webui.suntheme4_2.upload.getInputElement(elementId); 
        var form = upload != null ? upload.form : null;
	if (form != null) {

            // form.enctype does not work for IE, but works Safari
            // form.encoding works on both IE and Firefox
	    //
            if (webui.suntheme4_2.browser.isSafari()) {
                form.enctype = "multipart/form-data";
            } else {
                form.encoding = "multipart/form-data";
            }
	    return true;
        }
	return false;
    },

    /**
     * Create a hidden field with id "preservePathId" and add a listener
     * to the upload's input element, "uploadId". The listener is
     * is added for the onchange event of the upload's input field,
     * see preservePathListener.
     *
     * @param {String} uploadId The client id of the upload component.
     * @param {String} preservePathId
     * @return {boolean} true if the hidden element is created and a listener is
     * added, else false.
     */
    preservePath: function(uploadId, preservePathId) {
	if (uploadId == null || uploadId == "" ||
		preservePathId == null || preservePathId == "") {
	    return false;
	}

	// If there is no upload component, don't do anything.
	// I'm not sure if there is a timing issue here.
	//
	var uploadElement = webui.suntheme4_2.upload.getInputElement(uploadId);
	if (uploadElement == null) {
	    return false;
	}
	var theForm = uploadElement.form;

	// Create the change listener.
	// The event target/srcElement is the upload input element
	// its value is the changed value, save it in the 
	// preservePath hidden field.
	//
	var onChangeListener = function(evt) {

	    // Is IE
	    if (document.attachEvent) {
		node = evt.srcElement;
	    } else {
		node = evt.target;
	    }
	    // node is the upload input element
	    //
	    var preservePath = null;
	    try {
		preservePath = theForm.elements[preservePathId];
	    } catch (e) {
	    }

	    // If the hidden field isn't there create it and assign
	    // the node's value
	    //
	    if (preservePath != null) {
		preservePath.value = node.value;
	    } else {
		webui.suntheme4_2.common.insertHiddenField(preservePathId, 
			node.value, theForm);
	    }
	    return true;
	};

	if (uploadElement.addEventListener) {
	    uploadElement.addEventListener('change', onChangeListener, true);
	} else {
	    uploadElement.attachEvent('onchange', onChangeListener);
	}
	return true;
    }
};
webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");

/** 
 * @class This class contains functions for editableList components.
 * @static
 */
webui.suntheme4_2.editableList = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize editableList.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Not a facet does not have "extra" editable list id.

        // child elements
        // Get the field by calling the field getInputMethod
        // because only it knows about the underlying structure
        // of the rendered field component
        //
        domNode.list = document.getElementById(props.id + "_list");

        // Bug 6338492 -
        //     ALL: If a component supports facets or children is must be a
        //      NamingContainer
        // Since EditableList has become a NamingContainer the id's for
        // the facet children are prefixed with the EditableList id
        // in addition to their own id, which also has the 
        // EditableList id, as has been the convention for facets. This introduces
        // a redundancy in the facet id so the add button now looks like
        //
        // "formid:editablelistid:editablelistid:editablelistid_addButton"
        //
        // It used to be "formid:editablelistid_addButton"
        // It would be better to encapsulate that knowledge in the
        // EditableList renderer as does FileChooser which has the
        // same problem but because the select elements are not
        // facets in EditableList they really do only have id's of the
        // form "formid:addremoveid_list". Note that 
        // in these examples the "id" parameter is "formid:editablelistid"
        //
        // Therefore for now, locate the additional prefix here as the
        // "facet" id. Assume that id never ends in ":" and if there is
        // no colon, id is the same as the component id.
        //
        var componentid = props.id;
        var colon_index = componentid.lastIndexOf(':');
        if (colon_index != -1) {
            componentid = props.id.substring(colon_index + 1);
        }
        var facetid = props.id + ":" + componentid;

        // Get the field by calling the field getInputMethod
        // because only it knows about the underlying structure
        // of the rendered field component
        //
        domNode.field = webui.suntheme4_2.field.getInputElement(facetid + "_field");
        domNode.addButton = document.getElementById(facetid + "_addButton"); 
        domNode.removeButton = document.getElementById(facetid + "_removeButton"); 

        // HTML elements may not have been created, yet.
        if (domNode.list == null 
                || domNode.field == null 
                || domNode.addButton == null 
                || domNode.removeButton == null) {
            return setTimeout(function() {
                webui.suntheme4_2.editableList._init(props);
            }, 10);
        }

        // Set given properties on domNode.
        webui.suntheme4_2.prototypejs.extend(domNode, props, false);

        // attach methods
        domNode.add = webui.suntheme4_2.editableList.add;
        domNode.enableAdd = webui.suntheme4_2.editableList.enableAdd;
        domNode.enableRemove = webui.suntheme4_2.editableList.enableRemove;
        domNode.setAddDisabled = webui.suntheme4_2.editableList.setAddDisabled;
        domNode.setRemoveDisabled = webui.suntheme4_2.editableList.setRemoveDisabled; 
        domNode.updateButtons = webui.suntheme4_2.editableList.updateButtons;
        domNode.setDisabled = webui.suntheme4_2.editableList.setDisabled;

        // Initialize buttons.
        domNode.updateButtons();
        return true;
    },

    /**
     * Add HTML element.
     *
     * @param {String} elementId The HTML element id.
     * @return {boolean} true if successful; otherwise, false.
     */
    add: function(elementId) {
        this.enableAdd(); 
        this.addButton.click();
        return true;
    },

    /**
     * Enable add button.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    enableAdd: function() {
        var disabled = (this.field.value == ""); 
        return this.setAddDisabled(disabled);
    },

    /**
     * Set add button disabled.
     *
     * @param {boolean} disabled If true, disable element.
     * @return {boolean} true if successful; otherwise, false.
     */
    setAddDisabled: function(disabled) {
        if (this.addButton.setDisabled != null) {
            this.addButton.setDisabled(disabled); 
        } else {
            this.addButton.disabled = disabled; 
        }
        return true;
    },

    /**
     * Enable remove button.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    enableRemove: function() {
        var disabled = (this.list.selectedIndex == -1); 
        return this.setRemoveDisabled(disabled); 
    },

    /**
     * Set remove button disabled.
     *
     * @param {boolean} disabled If true, disable element.
     * @return {boolean} true if successful; otherwise, false.
     */
    setRemoveDisabled: function(disabled) {
        if (this.removeButton.setDisabled != null) {
            this.removeButton.setDisabled(disabled); 
        } else {
            this.removeButton.disabled = disabled; 
        }
        return true;
    },

    /**
     * Update add and remove buttons.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    updateButtons: function() {
        this.enableAdd(); 
        this.enableRemove(); 
        return true;
    },

    /**
     * Set buttons disabled.
     *
     * @param {boolean} disabled If true, disable element.
     * @return {boolean} true if successful; otherwise, false.
     */
    setDisabled: function(disabled) {
        if (this.addButton.setDisabled != null) {
            this.addButton.setDisabled(disabled); 
        } else {
            this.addButton.disabled = disabled; 
        }
        if (this.removeButton.setDisabled != null) {
            this.removeButton.setDisabled(disabled); 
        } else {
            this.removeButton.disabled = disabled; 
        }
        this.field.disabled = disabled; 
        this.list.disabled = disabled; 
        return true;
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.fileChooser");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.formElements");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");

/** 
 * @class This class contains functions for fileChooser components.
 * <p>
 * The filechooser has several intermediate actions in addition to the selection
 * of one or more files of folders. For example, there are actions that are 
 * initiated by buttons and actions initiated by mouse clicks and key presses.
 * Some events generated from key presses and mouse clicks behave like 
 * accelerators for the button actions.
 * </p><p>
 * Some actions are client side only actions, such as placing a
 * selected file or folder into the selected file or folder field.
 * </p><p>
 * The server side events and how they are generated.
 * </p><p><pre>
 * 1. moveup - the moveup button is clicked
 * 2. openfolder - select a folder from the list and the openfolder
 *    button is clicked or press the return key with the selection in
 *    focus or double click the selection
 * 3. sort - a sort selection is made from the sort dropdown menu
 * 4. refresh the list - set focus in the look in field and press the return key
 * 5. change the directory listing - type a new value in the look in field
 *    and press the return key
 *    The directory listing is also changed when a folder is typed into
 *    the selected file field, and submitted.
 * 6. filter the list - the focus is placed in the filter field and the
 *    return key is pressed
 * 7. change filter value - a new value is entered into the filter field 
 *    and the return key is pressed
 * 8. commit the changes - the button assigned as the chooseButton is
 *    is clicked, or programmtically clicked when the return key is 
 *    pressed in the select file field or a file selection is double clicked.
 * </pre></p><p>
 * Mouse clicks and return key activations.
 * </p><p><pre>
 * - The mouse clicks and key presses that correspond to actions are
 *   exectuted by "clicking" the corresponding button action programmatically.
 *   For example, action #2 when activated by double clicking or the
 *   return key, programmatically clicks the open folder button.
 *
 * - Action #4 and #6 explcitly submit the form
 *
 * - Action #8 programmatically clicks the assigned chooserButton.
 * </pre></p><p>
 * Selections are made and submitted in the following ways
 * </p><p><pre>
 *   File chooser or folder chooser
 *
 *   - One or more absolute or relative paths are typed or placed
 *   into the select file or folder field the return key is pressed.
 *
 *   - An external submit button submits the form and there are selections
 *   in the selected file or folder field.
 *
 *   File chooser 
 *
 *   - A file selection is double clicked.
 * </pre></p><p>
 * Client side selections
 * </p><p><pre>
 * - A file or folder is single clicked and it is entered into the
 *   selected file or folders field, depending on whether the chooser
 *   is a file or folder chooser.
 *
 * - When a directory is selected and the open folder button is clicked
 *   the entry is set as the look in field value and the form is submitted.
 * 
 * - When the move up button is clicked the parent directory is placed
 *   into the look in field and the form is submitted.
 * </pre></p>
 * @static
 */
webui.suntheme4_2.fileChooser = {
    // FIXME: Note that the dependence on literal client id's is not sufficient
    // if these components are developer defined facets. The actual
    // literal id's cannot be guaranteed.

    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @config {String} chooserType 
     * @config {String} parentFolder 
     * @config {String} separatorChar 
     * @config {String} escapeChar 
     * @config {String} delimiter 
     * @config {String} currentDir
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize fileChooser.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // This is not a user defined facet. It is created dynamically
        // if the enter key is pressed in the selected file field.
        // This is the expected form of the request paramter name
        // in the renderers decode method.
        //
        // Consider passing the name as a parameter, or some other
        // more well defined manner.
        var idPrefix = props.id;
        var index = props.id.lastIndexOf(':');
        if (index != -1) {
             idPrefix += props.id.substring(index);
        }

        // Get HTML elements.
        domNode.lookinfield = webui.suntheme4_2.field.getInputElement(idPrefix + "_lookinField");
        domNode.filterfield = webui.suntheme4_2.field.getInputElement(idPrefix + "_filterField");
        domNode.selectedfield = webui.suntheme4_2.field.getInputElement(idPrefix + "_selectedField");
        domNode.upButton = document.getElementById(idPrefix + "_upButton");
        domNode.openFolderButton = document.getElementById(idPrefix + "_openButton");
        domNode.listentries = webui.suntheme4_2.listbox.getSelectElement(idPrefix + "_listEntries");
        domNode.sortmenu = webui.suntheme4_2.dropDown.getSelectElement(idPrefix + "_sortMenu");

        // HTML elements may not have been created, yet.
        if (domNode.lookinfield == null 
                || domNode.filterfield == null 
                || domNode.selectedfield == null 
                || domNode.upButton == null
                || domNode.openFolderButton == null 
                || domNode.listentries == null
                || domNode.sortmenu == null) {
            return setTimeout(function() {
                webui.suntheme4_2.fileChooser._init(props);
            }, 10);
        }

        // Set given properties on domNode.
        webui.suntheme4_2.prototypejs.extend(domNode, props, false);

        // boolean identifying the chooser mode.
        domNode.folderChooser = (props.chooserType == "folderChooser");
        domNode.fileAndFolderChooser = (props.chooserType == "fileAndFolderChooser");
        domNode.chooseButton = null;
        domNode.selectionsId = idPrefix + "_selections";
        domNode.listOptions = domNode.listentries.options;

        // FIXME: This encoding needs to be generalized if this code is to
        // become more generic chooser-like.
        // In fact encoding entries this way in not ideal.
        // A more explicit typing needs to be developed if it is 
        // necessary, possible a data structure that maps type to entry.
        if (domNode.folderChooser) {
            domNode.chooser = 'folder';
        } else {
            domNode.chooser = 'file';
        }

        // Set functions.
        domNode.enterKeyPressed = webui.suntheme4_2.fileChooser.enterKeyPressed;
        domNode.handleDblClick = webui.suntheme4_2.fileChooser.handleDblClick;
        domNode.handleOnChange = webui.suntheme4_2.fileChooser.handleOnChange;
        domNode.openFolderClicked = webui.suntheme4_2.fileChooser.openFolderClicked;
        domNode.moveUpButtonClicked = webui.suntheme4_2.fileChooser.moveUpButtonClicked;
        domNode.setChooseButton = webui.suntheme4_2.fileChooser.setChooseButton;
        domNode.getCurrentDirectory = webui.suntheme4_2.fileChooser.getCurrentDirectory;
        domNode.getOptionElements = webui.suntheme4_2.fileChooser.getOptionElements;
        domNode.getSelectedOptions = webui.suntheme4_2.fileChooser.getSelectedOptions;
        domNode.isFolderChooser = webui.suntheme4_2.fileChooser.isFolderChooser;
        domNode.isFolderSelected = webui.suntheme4_2.fileChooser.isFolderSelected;
        domNode.getSelectionValue = webui.suntheme4_2.fileChooser.getSelectionValue;
        domNode.getSelectionValueByIndex = webui.suntheme4_2.fileChooser.getSelectionValueByIndex;
        domNode.getSelectionType = webui.suntheme4_2.fileChooser.getSelectionType;
        domNode.getSelectionTypeByIndex = webui.suntheme4_2.fileChooser.getSelectionTypeByIndex;
        domNode.getValueType = webui.suntheme4_2.fileChooser.getValueType;
        domNode.itemSelected = webui.suntheme4_2.fileChooser.itemSelected;
        domNode.getSelectedFolders = webui.suntheme4_2.fileChooser.getSelectedFolders;
        domNode.getSelectedFiles = webui.suntheme4_2.fileChooser.getSelectedFiles;
        domNode.getSelectedValuesByType = webui.suntheme4_2.fileChooser.getSelectedValuesByType;
        domNode.setSelectedFieldValue = webui.suntheme4_2.fileChooser.setSelectedFieldValue;
        domNode.clearSelections = webui.suntheme4_2.fileChooser.clearSelections;
        domNode.deselectFolders = webui.suntheme4_2.fileChooser.deselectFolders;
        domNode.deselectSelectionsByType = webui.suntheme4_2.fileChooser.deselectSelectionsByType;
        domNode.setSelected = webui.suntheme4_2.fileChooser.setSelected;
        domNode.clearSelectedField = webui.suntheme4_2.fileChooser.clearSelectedField;
        domNode.armChooseButton = webui.suntheme4_2.fileChooser.armChooseButton;
        domNode.getFileNameOnly = webui.suntheme4_2.fileChooser.getFileNameOnly;
        domNode.setChooseButtonDisabled = webui.suntheme4_2.fileChooser.setChooseButtonDisabled;

        // For supporting valid entries in look in field and filter field.
        //
        // It is imperative that the look in field and filter field
        // are never submitted if the value does not imply a valid action.
        // Not currently used.
        domNode.onFocus = webui.suntheme4_2.fileChooser.onFocus;
        domNode.onBlur = webui.suntheme4_2.fileChooser.onBlur;

        // Save the initial lookin and filter values.
        domNode.lastLookInValue = domNode.lookinfield.value;
        domNode.lastFilterValue = domNode.filterfield.value;
        domNode.lookinCommitted = false;
        domNode.filterCommitted = false;
        domNode.openFolderButton.setDisabled(true);

	if (props.currentFolder != null) {
	  if (props.parentFolder == props.currentFolder) {
	    domNode.upButton.setDisabled(true);
   	  } else {
	    domNode.upButton.setDisabled(false);
	  }
	}
        return true;
    },

    /**
     * Handler for enter key presses.
     * <p><pre>
     * - Enter key in LookInField
     * - Enter key in FilterField
     * - Enter key in SelectedFileField
     * - Enter key in Listbox with folder selection.
     * Submit the chooser from the various mouse clicks
     * key presses.
     * </pre></p><p>
     * Handles doubleclick on a file selection in the list box.
     * This is equivalent to an enter key press in the selected file field.
     * </p>
     *
     * @param {Node} element
     * @return {boolean} false to cancel JavaScript event.
     */
    enterKeyPressed: function(element) {
	// Return pressed in the list
	if (element.id == this.listentries.id) {
	    // If the selected item is a folder call the click method
	    // on the openFolderButton
	    if (this.isFolderSelected()) {
		// Done in openFolderClicked
		//
		//this.lookinfield.value = this.getSelectionValue();
		this.openFolderButton.click();
	    }
	    return false;
	}

	// The FileChooser's value must only change when selections
	// have been made, not from just intermediate operations.
	//
	// Enter key pressed in the selectedFileField
	// or dbl click in the list.
	if (this.selectedfield && element.id == this.selectedfield.id) {
	    var escapedSelections = this.selectedfield.value;
	    var selections = webui.suntheme4_2.common.unescapeStrings(escapedSelections,
		    this.delimiter, this.escapeChar);

	    // If a choose button has been defined call its click method
	    // otherwise do nothing. This behavior allows the filechooser
	    // to behave like any other component on a page.
	    if (this.chooseButton) {
		// Make sure its enabled.
		this.chooseButton.setDisabled(false);
		this.chooseButton.click();
	    }
	    return false;
	}

	// Enter key pressed in the filter field
	// Call the open folder button's click method.
	// Since there is no JSF action for mouse clicks or key presses
	// overload the open folder action to ensure that the 
	// sort value is updated.
	if (element.id == this.filterfield.id) {
	    this.filterCommitted = true;
	    // Don't let "" get submitted.
	    var fv = this.filterfield.value;
	    if (fv == null || fv == "") {
		this.filterfield.value = this.lastFilterValue;
		return false;
	    }
	    this.lastFilterValue = fv;
	    this.clearSelections();
	    element.form.submit();
	    return false;
	}

	// Enter key pressed in the LookIn field.
	// Call the open folder button's click method.
	// Since there is no JSF action for mouse clicks or key presses
	// overload the open folder action to ensure that the 
	// look in value is updated. This is needed anyway to display
	// the new folder's content.
	if (element.id == this.lookinfield.id) {
	    this.lookinCommitted = true;
	    // Don't let "" get submitted.
	    var lv = this.lookinfield.value;
	    if (lv == null || lv == "") {
		this.lookinfield.value = this.lastLookInValue;
		return false;
	    }
	    this.lastLookInValue = lv;
	    this.clearSelections();
	    element.form.submit();
	    return false;
	}
	return false;
    },

    /**
     * In file chooser mode
     * <p><pre>
     *    - a file selection, call enterKeyPressed with selected file field
     *    - a folder selection, call open folder click handler
     * In folder chooser mode
     *    - a folder selection, call open folder click handler
     * </pre></p>
     * @return {boolean} true if successful; otherwise, false.
     */
    handleDblClick: function() {
	// Nothing selected. Not sure if this can happen since
	// doubleclick implies selection.
	if (!this.itemSelected()) {
	    return false; 
	}

	var fldrSelected = this.isFolderSelected();

	// If the selected item is a folder call the click method
	// on the openFolderButton
	if (fldrSelected) {
	    // Set the look in field, since the selected folder will be
	    // the new look in field value. Done in openFolderClicked.
	    // This only works now because values are full paths.
	    //
	    this.openFolderButton.click();
	    return true;
	}

	// doubleclick is not valid for file selections in a
	// folder chooser.
	// If a file chooser, this is equivalent to a return key
	// in the selected file field.
	if (this.isFolderChooser()) {
	    if (!fldrSelected) {
		return false;
	    }
	} else {
	    // file chooser
	    // double click a file in file chooser mode
	    if (!fldrSelected) {
		if (this.selectedfield) {
		    return this.enterKeyPressed(this.selectedfield);
		}

		// If a choose button has been defined call its click method
		// otherwise do nothing. This behavior allows the filechooser
		// to behave like any other component on a page.
		if (this.chooseButton) {
		    // Make sure its enabled.
		    //
		    this.chooseButton.setDisabled(false);
		    return this.chooseButton.click();
		}
	    }
	}
	return true;
    },

    /**
     * Set choose button disabled.
     *
     * @param {boolean} disabled
     * @return {boolean} true if successful; otherwise, false.
     */
    setChooseButtonDisabled: function(disabled) {
	if (this.chooseButton) {
	    this.chooseButton.setDisabled(disabled);
	}
        return true;
    },

    // Replaces entries in selectedFileField
    // Get the selected entries from the list box and place
    // them in the selected file field, as comma separated entries.
    //
    // Note that the listbox values are full paths and encoded with
    // a "folder" or "file" designation. They probably should
    // be relative paths. The open folder action now depends on the
    // fact that the value is a full path.
    // This will have an effect on the open folder
    // action when the selected value is placed into the look in field.
    // If relative paths are used for the values then
    // the relative path would need to be appended to the look in
    // field value.
    //
    // However it may be the case that the full paths are edited to
    // take off the last element in the full path and keep the 
    // full path list box entries. Full paths are generally more
    // convenient.
    //
    // Note that this handler should call any required handlers
    // needed by the list box, vs. placing the required listbox
    // handlers in a javascript statement as the value of the
    // onChange attribute.
    //
    // Note also that the SWAED guidelines say to place relavtive
    // paths into the selected file field, this probably means
    // just using the display name vs. the value. However note the
    // dependencies on the full paths as described above.

    /**
     * Handler placed on the list box onchange enent.
     * <p>
     * Place all currently selected entries in to the
     * selected file field. If the chooser mode is file, only
     * files are placed into the selected file field.
     * </p><p>
     * If the chooser mode is folder, only folders are placed in the
     * selected folder field.
     * </p><p>
     * If multiple selections are allowed the entries are separated
     * by the specified delimiter. Enteries are escaped appropriately
     * with the specified escape character.
     * </p>
     * @return {boolean} false to cancel JavaScript event.
     */
    handleOnChange: function() {
	webui.suntheme4_2.listbox.changed(this.listentries.id);

	// If nothing is selected disable buttons.
	if (!this.itemSelected()) {
	    this.openFolderButton.setDisabled(true); 
	    if (this.selectedfield &&
		(this.selectedfield.value == null ||
		    this.selectedfield.value == '')) {
		this.setChooseButtonDisabled(true);
	    }
	    return false;
	}

	// This may not be sufficient.
	// The issue is, should a file be selectable in a folder
	// chooser, period. Are they disabled or read only ?
	// And ditto for a multiple selection in a file chooser.
	// Should a folder be selectable as a multiple selection
	// in a file chooser ?
	//
	// This could be made more efficient
	// by return both arrays at once and making only
	// one pass

	var folders = this.getSelectedFolders();
	var files = this.getSelectedFiles();
	var selections = null;

	// If a file chooser, deselect folders when mixed
	// with file selections and disable the openFolder button);
	if (this.fileAndFolderChooser) {
	    selections = new Array(files.length + folders.length);
	    var i = 0;
	    for (; i< files.length; i++) {
	        selections[i] = files[i];
	    } 
	    for (j=0; j< folders.length; j++) {
	        selections[i+j] = folders[j];
	    } 
	    if (files.length > 0) {
		this.openFolderButton.setDisabled(true);
	    } else if (folders.length > 1) {
		this.openFolderButton.setDisabled(true);
	    } else if ((files.length == 0) || (folders.length == 1)) {
		this.openFolderButton.setDisabled(false);
	    }
	} else if (!this.isFolderChooser()) {
	    if (files.length > 0) {
		this.openFolderButton.setDisabled(true);
		this.deselectFolders();
	    } else if (folders.length > 1) {
		this.openFolderButton.setDisabled(false);
		var index = this.listentries.selectedIndex;
		this.deselectFolders();
		this.setSelected(index, true);
		webui.suntheme4_2.listbox.changed(this.listentries.id);
		this.clearSelectedField();
	    } else if (folders.length == 1) {
		// Only allow one folder to be selected
		this.openFolderButton.setDisabled(false);
		this.clearSelectedField();
	    } else {
		this.openFolderButton.setDisabled(true);
		this.clearSelectedField();
	    }
	    selections = files;
	} else {
	    // If a folder chooser allow more than one folder
	    // to be selected
	    selections = folders;
	    if (selections.length == 1) {
		this.openFolderButton.setDisabled(false);
	    } else {
		this.openFolderButton.setDisabled(true);
	    }
	}

	// Make sure the hidden select option array is up
	// to date in case there isn't a selectedFileField.
	if (!this.setSelectedFieldValue(selections)) {
	    webui.suntheme4_2.common.createSubmittableArray(
                this.selectionsId, this.listentries.form, null, selections);
	}

	var flag = ((selections!= null) && (selections.length > 0));
	this.armChooseButton(flag);
	return false;
    },

    /**
     * Clear selected field.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    clearSelectedField: function() {
	if (this.selectedfield) {
	    this.selectedfield.value = '';
	}
        return true;
    },

    /**
     * This function is the event handler for the onclick event
     * of the openFolder button.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    openFolderClicked: function() {
	if (!this.isFolderSelected()) {
	    return false;
	}
	this.clearSelectedField();

	// Only works because the value is a full path.
	this.lookinfield.value = this.getSelectionValue();
	return true;
    },

    /**
     * Test if folder is selected.
     *
     * @return {boolean} true if folder is selected.
     */
    isFolderSelected: function() {
	return this.getSelectionType() == 'folder';
    },

    /**
     * This function is the event handler for the moveUp button.
     * Set the look in field to contain the parent or move up directory.
     * This is imperative. Be careful of the corner case when the 
     * look in field is already the root directory.
     * @return {boolean} true if successful; otherwise, false.
     */
    moveUpButtonClicked: function() {
	this.clearSelections();
	this.lookinfield.value = this.parentFolder;
        return true;
    },

    /**
     * The values of the list options are encoded as
     * <p>
     * <type>=<value>
     * </p><p>
     * Where type is one of "file" or "folder"
     * </p>
     * @return {String} The selection value.
     */
    getSelectionValue: function() {
	var index = this.listentries.selectedIndex;
	return this.getSelectionValueByIndex(index);
    },

    /**
     * Get selection value by index.
     *
     * @return {String}The selection value.
     */
    getSelectionValueByIndex: function(index) {
	var selection = this.listOptions[index].value;
	var i = selection.indexOf('=', 0);
	if (i < 0) {
	    return null;
	}
	if (i != 0) {
	    i = i + 1;
	}
	return selection.substring(i, selection.length); 
    },

    /**
     * Get selection type.
     *
     * @return {String} The selection type.
     */
    getSelectionType: function() {
	var index = this.listentries.selectedIndex;
	return this.getSelectionTypeByIndex(index);
    },

    /**
     * Get selection type by index.
     *
     * @return {String} The selection type.
     */
    getSelectionTypeByIndex: function(index) {
	var selection = this.listOptions[index].value;
	return this.getValueType(selection);
    },

    /**
     * Get value type.
     *
     * @return {String} The value type.
     */
    getValueType: function(value) {
	var i = value.indexOf('=', 0);
	if (i <= 0) {
	    return null;
	}
	var type = value.substring(0, i); 
	return type;
    },

    /**
     * Test if folder chooser.
     *
     * @return {boolean} true if folder chooser.
     */
    isFolderChooser: function() {
	return this.folderChooser;
    },

    /**
     * Get selected item.
     *
     * @return {String} The selected item.
     */
    itemSelected: function() {
	return (this.listentries.selectedIndex != -1);
    },

    /**
     * Get selected folders.
     *
     * @return {Array} An array of selected folders.
     */
    getSelectedFolders: function() {
	return this.getSelectedValuesByType('folder');
    },

    /**
     * Get selected files.
     *
     * @return {Array} An array of selected files.
     */
    getSelectedFiles: function() {
	return this.getSelectedValuesByType('file');
    },

    /**
     * Return all selected options by type, file or folder.
     *
     * @param {String} type
     * @return {Array} An array of selected values.
     */
    getSelectedValuesByType: function(type) {
	var selections = new Array();
	var i = 0;
	for (var j = 0; j < this.listOptions.length; j++) {
	    if (this.listOptions[j].selected){
		if (this.getSelectionTypeByIndex(j) == type) {
		    selections[i++] = this.getSelectionValueByIndex(j);
		} 
	    } 
	} 
	return selections;
    },

    /**
     * Format the selected file field as a comma separated list.
     *
     * @param {Array} selections
     * @return {boolean} true if successful; otherwise, false.
     */
    setSelectedFieldValue: function(selections) {
	var value;
	if (this.selectedfield == null) {
	    return false;
	}

	if (selections == null || selections.length == 0) {
	    return false;
	} else {
	    value = webui.suntheme4_2.common.escapeString(
                this.getFileNameOnly(selections[0]), this.delimiter,
                this.escapeChar);
	}

	for (var j = 1; j < selections.length; j++) {
	    value = value + ',' + 
                webui.suntheme4_2.common.escapeString(
                    this.getFileNameOnly(selections[j]), this.delimiter,
                    this.escapeChar);
	} 

	if (value != null && value != '') { 
	    this.selectedfield.value = value;
	} else { 
	    this.selectedfield.value = '';
	} 
	return true;
    },

    /**
     *
     * @param {Node} element
     * @return {boolean} true if successful; otherwise, false.
     */
    onFocus: function(element) {
	if (element.id == this.lookinfield.id) {
	    this.lookinCommitted = false;
	    this.lastLookInValue = this.lookinfield.value;
	} else if (element.id == this.filterfield.id) {
	    this.filterCommitted = false;
	    this.lastFilterValue = this.filterfield.value;
	}
	return true;
    },

    /**
     *
     * @param {Node} element
     * @return {boolean} true if successful; otherwise, false.
     */
    onBlur: function(element) {
	if (element.id == this.lookinfield.id) {
	    if (this.lookinCommitted == false) {
		this.lookinfield.value = this.lastLookInValue;
	    }
	} else if (element.id == this.filterfield.id) {
	    if (this.filterCommitted == false) {
		this.filterfield.value = this.lastFilterValue;
	    }
	}
	return true;
    },

    /**
     * Clear the selections whenever the selectedFileField is cleared.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    clearSelections: function() {
	var i = 0;
	for (var j = 0; j < this.listOptions.length; j++) {
	    if (this.listOptions[j].selected){
		this.listOptions[j].selected = false;
	    } 
	} 
	// call listbox.changed to update the
	// private state
	webui.suntheme4_2.listbox.changed(this.listentries.id);

	if (this.selectedfield != null) {
	    this.selectedfield.value = "";
	}
        return true;
    },

    /**
     * Set selected.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    setSelected: function(index, torf) {
	this.listOptions[index].selected = torf;
	return webui.suntheme4_2.listbox.changed(this.listentries.id);
    },

    /**
     * Deselect folders.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    deselectFolders: function() {
	return this.deselectSelectionsByType('folder');
    },

    /**
     * Deselect by type.
     *
     * @param {String} type
     * @return {boolean} true if successful; otherwise, false.
     */
    deselectSelectionsByType: function(type) {
	for (var j = 0; j < this.listOptions.length; j++) {
	    if (this.listOptions[j].selected &&
		    this.getValueType(this.listOptions[j].value) == type) {
		this.listOptions[j].selected = false;
	    } 
	} 
	return webui.suntheme4_2.listbox.changed(this.listentries.id);
    },

    /**
     * Enable the choose button.
     *
     * @param {boolean} flag
     * @return {boolean} true if successful; otherwise, false.
     */
    armChooseButton: function(flag) {
	var disabled = true;
	if (this.selectedfield == null) {
	    disabled = flag;
	} else if (this.selectedfield.value != null 
	    && this.selectedfield.value != '') {
	        disabled = false;
	} 
	return this.setChooseButtonDisabled(disabled);
    },

    // Note that this is the only way that the file chooser can control
    // the submit of selected values. If this button is not set then
    // only an external submit button can submit the selections.
    // That means that if there is no chooser button assigned, double clicking
    // a file entry or hitting the return key in the selected file field
    // will NOT submit the values.
    //
    // This "feature" may become configurable.

    /**
     * convenience function to allow developers to disable their
     * chooser button when no entries from the filechooser are
     * selected. This function is not yet complete.
     *
     * @param {String} buttonId
     * @return {boolean} true if successful; otherwise, false.
     */
    setChooseButton: function(buttonId) {
	this.chooseButton = document.getElementById(buttonId);
	// See if there are selections and if so 
	// enable the button. Needs to be after the assignment
	var selections = document.getElementById(this.selectionsId);
	var disabled = true;
	if ((selections != null) && (selections.length > 0)) {
	    disabled = false;
	}
	return this.armChooseButton(disabled);
    },

    /**
     * Convenience function to get the current directory without 
     * going to the server.
     *
     * @return {String} The current directory.
     */
    getCurrentDirectory: function() {
	if (this.lookinfield) {
	    return this.lookinfield.value;
	}
        return null;
    },

    /**
     * Convenience function returning the list of option elements.
     *
     * @return {Array} An array of list options.
     */
    getOptionElements: function() {
	return this.listOptions;
    },

    /*
     * Convenience function to get the list of selected option elements
     * Return an array of selected values or a 0 length array if there
     * are no selections.
     *
     * @return {Array} An array of selected options.
     */
    getSelectedOptions: function() {
	var selections = new Array();
	var i = 0;
	for (var j = 0; j < this.listOptions.length; j++) {
	    if (this.listOptions[j].selected){
		selections[i++] = this.getSelectionValueByIndex(j);
	    } 
	} 
	return selections;
    },

    /*
     * Convenience function to get the file or folder name when 
     * the entire path name is supplied.
     *
     * @param {String} absoluteFileName
     * @return {String} The file name.
     */
    getFileNameOnly: function(absoluteFileName) {
        arrayOfPaths = absoluteFileName.split(this.separatorChar);
	justTheFileName = arrayOfPaths[arrayOfPaths.length -1];
        return justTheFileName;
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.json");

//
// Copyright (c) 2005 JSON.org
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The Software shall be used for Good, not Evil.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//

/**
 * @class This class contains functions for parsing JSON.
 * @static
 */
webui.suntheme4_2.json = {
    /**
     * JSON escape chars.
     * @private
     */
    _m: {
        '\b': '\\b',
        '\t': '\\t',
        '\n': '\\n',
        '\f': '\\f',
        '\r': '\\r',
        '"' : '\\"',
        '\\': '\\\\'
    },

    /**
     * JSON parsor.
     * @private
     */
    _s: {
        'boolean': function (x) {
            return String(x);
        },
        number: function (x) {
            return isFinite(x) ? String(x) : null;
        },
        string: function (x) {
            if (/["\\\x00-\x1f]/.test(x)) {
                x = x.replace(/([\x00-\x1f\\"])/g, function(a, b) {
                    var c = webui.suntheme4_2.json._m[b];
                    if (c) {
                        return c;
                    }
                    c = b.charCodeAt();
                    return '\\u00' +
                        Math.floor(c / 16).toString(16) +
                        (c % 16).toString(16);
                });
            }
            return '"' + x + '"';
        },
        object: function (x) {
            if (x) {
                var a = [], b, f, i, l, v;
                if (x instanceof Array) {
                    a[0] = '[';
                    l = x.length;
                    for (i = 0; i < l; i += 1) {
                        v = x[i];
                        f = webui.suntheme4_2.json._s[typeof v];
                        if (f) {
                            v = f(v);
                            if (typeof v == 'string') {
                                if (b) {
                                    a[a.length] = ',';
                                }
                                a[a.length] = v;
                                b = true;
                            }
                        }
                    }
                    a[a.length] = ']';
                } else if (typeof x.hasOwnProperty === 'function') {
                    a[0] = '{';
                    for (i in x) {
                        if (x.hasOwnProperty(i)) {
                            v = x[i];
                            f = webui.suntheme4_2.json._s[typeof v];
                            if (f) {
                                v = f(v);
                                if (typeof v == 'string') {
                                    if (b) {
                                        a[a.length] = ',';
                                    }
                                    a.push(webui.suntheme4_2.json._s.string(i), ':', v);
                                    b = true;
                                }
                            }
                        }
                    }
                    a[a.length] = '}';
                } else {
                    return null;
                }
                return a.join('');
            }
            return null;
        }
    },

    /**
     * Stringify a JavaScript value, producing JSON text. 
     *
     * @param {Object} v A non-cyclical JSON object.
     * @return {boolean} true if successful; otherwise, false.
     */
    stringify: function (v) {
        var f = webui.suntheme4_2.json._s[typeof v];
        if (f) {
            v = f(v);
            if (typeof v == 'string') {
                return v;
            }
        }
        return null;
    },

    /**
     * Parse a JSON text, producing a JavaScript object.
     *
     * @param {String} text The string containing JSON text.
     * @return {boolean} true if successful; otherwise, false.
     */
    parse: function (text) {
        try {
            return !(/[^,:{}\[\]0-9.\-+Eaeflnr-u \n\r\t]/.test(
                text.replace(/"(\\.|[^"\\])*"/g, ''))) && eval('(' + text + ')');
        } catch (e) {
            return false;
        }
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.orderableList");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");

/** 
 * @class This class contains functions for orderableList components.
 * @static
 */
webui.suntheme4_2.orderableList = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @config {String} moveMessage
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize orderableList.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Not a facet does not have "extra" editable list id.

        // The select element from which selections are made 
        domNode.list = document.getElementById(props.id + "_list");

        // Bug 6338492 -
        //     ALL: If a component supports facets or children is must be a
        //      NamingContainer
        // Since OrderableList has become a NamingContainer the id's for
        // the facet children are prefixed with the OrderableList id
        // in addition to their own id, which also has the 
        // OrderableList id, as has been the convention for facets. This introduces
        // a redundancy in the facet id so the moveUp button now looks like
        //
        // "formid:orderablelistid:orderablelistid:orderablelistid_moveUpButton"
        //
        // It used to be "formid:orderablelistid_moveUpButton"
        // It would be better to encapsulate that knowledge in the
        // OrderableList renderer as does FileChooser which has the
        // same problem but because the select elements are not
        // facets in OrderableList they really do only have id's of the
        // form "formid:orderablelistid_list". Note that 
        // in these examples the "id" parameter is "formid:orderablelistid"
        //
        // Therefore for now, locate the additional prefix here as the
        // "facet" id. Assume that id never ends in ":" and if there is
        // no colon, id is the same as the component id.
        //
        var componentid = props.id;
        var colon_index = componentid.lastIndexOf(':');
        if (colon_index != -1) {
            componentid = props.id.substring(colon_index + 1);
        }
        var facetid = props.id + ":" + componentid;

        domNode.moveUpButton = document.getElementById(facetid + "_moveUpButton");
        domNode.moveDownButton = document.getElementById(facetid + "_moveDownButton");
        domNode.moveTopButton = document.getElementById(facetid + "_moveTopButton");
        domNode.moveBottomButton = document.getElementById(facetid + "_moveBottomButton");

        // Not a facet
        domNode.values = document.getElementById(props.id + "_list_value");

        // HTML elements may not have been created, yet.
        if (domNode.list == null
                || domNode.moveUpButton == null 
                || domNode.moveDownButton == null 
                || domNode.moveTopButton == null
                || domNode.moveBottomButton == null 
                || domNode.values == null) {
            return setTimeout(function() {
                webui.suntheme4_2.orderableList._init(props);
            }, 10);
        }

        // Set given properties on domNode.
        webui.suntheme4_2.prototypejs.extend(domNode, props, false);

        // The options of the select element from which selections are made 
        domNode.options = domNode.list.options;

        // The messages
        if (domNode.moveMessage == null) {
            "Select at least one item to remove";
        }

        // Set functions.
        domNode.moveUp = webui.suntheme4_2.orderableList.moveUp;
        domNode.moveDown = webui.suntheme4_2.orderableList.moveDown;
        domNode.moveTop = webui.suntheme4_2.orderableList.moveTop;
        domNode.moveBottom = webui.suntheme4_2.orderableList.moveBottom;
        domNode.updateButtons = webui.suntheme4_2.orderableList.updateButtons;
        domNode.updateValue = webui.suntheme4_2.orderableList.updateValue;
        domNode.onChange = webui.suntheme4_2.orderableList.updateButtons;

        // Initialize buttons.
        domNode.updateButtons();
        return true;
    },

    /**
     * The original allowed items to be moved on both lists. Surely we
     * only sort items on the selected list? 
     * This does not work on Mozilla
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    moveUp: function() {
        var numOptions = this.options.length;
    
        // If there aren't at least two more selected items, then there is
        // nothing to move 
        if (numOptions < 2) {
            return false;
        }

        // Start by examining the first item 
        var index = 0;

        // We're not going to move the first item. Instead, we will start
        // on the first selected item that is below an unselected
        // item. We identify the first unselected item on the list, and 
        // then we will start on next item after that
        while (this.options[index].selected) {
            ++index;
            if (index == numOptions) {
                // We've reached the last item - no more items below it so
                // we return
                return false;
            }
        }

        // Start on the item below this one 
        ++index;

        for (index; index < numOptions; ++index) {
            if (this.options[index].selected == true) {
                var curOption = this.options[index];
                if (this.options.remove == null) {
                    // For Mozilla
                    this.options[index] = null;
                    this.list.add(curOption, this.options[index - 1]);
                } else {
                    // Windows and Opera do
                    this.options.remove(index);
                    this.options.add(curOption, index - 1);
                }
                // This is needed for Opera only
                this.options[index].selected = false;
                this.options[index - 1].selected = true;
            }
        }
        this.updateValue();
        return this.updateButtons();
    },

    /**
     * The original allowed items to be moved on both lists. Surely we
     * only sort items on the selected list? 
     * This does not work on Mozilla
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    moveTop: function() {
        var numOptions = this.options.length;
        // If there aren't at least two items, there is nothing to move  
        if (numOptions < 2) {
            return false;
        }

        // Find the first open spot 
        var openSpot = 0;
        while (this.options[openSpot].selected) {
            openSpot++;
        }

        // Find the first selected item below it
        var index = openSpot+1;

        for (index; index < numOptions; ++index) {
            if (this.options[index].selected == true) {
                var curOption = this.options[index];
                if (this.options.remove == null) {
                    // For Mozilla
                    this.options[index] = null;
                    this.list.add(curOption, this.options[openSpot]);
                } else {
                    // Windows and Opera do
                    this.options.remove(index);
                    this.options.add(curOption, openSpot);
                }

                // This is needed for Opera only
                this.options[index].selected = false;
                this.options[openSpot].selected = true;
                openSpot++;
            }
        }
        this.updateValue();
        return this.updateButtons();
    },

    /** 
     * The original allowed items to be moved on both lists. Surely we
     * only sort items on the selected list? 
     * This does not work on Mozilla
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    moveDown: function() {
        // Get the last item
        var index = this.options.length - 1;
    
        // If this number is less than zero, there was nothing on the list
        // and we return
        if (index < 0) {
            return false;
        }

        for (var i = index - 1; i >= 0; i--) {
            if (this.options[i].selected) {          
                var next = i + 1;
                if (this.options[next].selected == false) {
                    tmpText = this.options[i].text;
                    tmpValue = this.options[i].value;
                    this.options[i].text = this.options[next].text;
                    this.options[i].value = this.options[next].value;
                    this.options[i].selected = false;
                    this.options[next].text = tmpText;
                    this.options[next].value = tmpValue;
                    this.options[next].selected = true;
                }
            }
        }

        this.updateValue();
        return this.updateButtons();
    },

    /**
     * Move options to bottom.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    moveBottom: function() {
        var numOptions = this.options.length - 1;

        // If there aren't at least two items, there is nothing to move  
        if (numOptions < 1) {
            return false;
        }

        // Find the last open spot 
        var openSpot = numOptions;
        while (this.options[openSpot].selected) {
            openSpot--;
        }

        // Find the first selected item above it
        var index = openSpot-1;

        for (index; index > -1; --index) {
            if (this.options[index].selected == true) {
                var curOption = this.options[index];
	        if (this.options.remove == null) {
                    // For Mozilla
                    this.options[index] = null;
                    this.list.add(curOption, this.options[openSpot+1]);
                } else {
                    // Windows and Opera do
                    this.options.remove(index);
                    this.options.add(curOption, openSpot);
                }

                // This is needed for Opera only
                this.options[index].selected = false;
                this.options[openSpot].selected = true;
                openSpot--;
            }
        }
        this.updateValue();
        return this.updateButtons();
    },

    /**
     * Update buttons.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    updateButtons: function() {
        var numOptions = this.options.length;
        var selectedIndex = this.options.selectedIndex;
        var disabled = true;
        var index;

        // First, check if move down and move to bottom should be
        // enabled. These buttons should be enabled if and only if at
        // least one of the items are selected and there is at least one
        // open spot below a selected item. 
        if (selectedIndex > -1 && selectedIndex < numOptions -1) {
            index = selectedIndex+1;
            while (index < numOptions) {
                if (this.options[index].selected == false) {
                    disabled = false;
                    break;
                }
                index++;
            }
        }

        if (this.moveDownButton != null) {
            if (this.moveDownButton.setDisabled != null) {
                this.moveDownButton.setDisabled(disabled);
            } else {
                this.moveDownButton.disabled = disabled;
            }
        }

        if (this.moveBottomButton != null) {
            if (this.moveBottomButton.setDisabled != null) {
                this.moveBottomButton.setDisabled(disabled);
            } else {
                this.moveBottomButton.disabled = disabled;
            }
        }

        // First, check if move up and move to top should be
        // enabled. These buttons should be enabled if and only if at
        // least one of the items is selected and there is at least one
        // open spot above a selected item. 
        disabled = true;

        if (selectedIndex > -1) {
            index = numOptions - 1;
            while (index > 0) {
                if (this.options[index].selected) {
                    break;
                }
                index--;
            }
            index--;
            while (index > -1) {
                if (this.options[index].selected == false) {
                    disabled = false;
                    break;
                }
                index--;
            }
        }

        if (this.moveUpButton != null) {
            if (this.moveUpButton.setDisabled != null) {
                this.moveUpButton.setDisabled(disabled);
            } else {
                this.moveUpButton.disabled = disabled;
            }
        }

        if (this.moveTopButton != null) {
            if (this.moveTopButton.setDisabled != null) {
                this.moveTopButton.setDisabled(disabled);
            } else {
                this.moveTopButton.disabled = disabled;
            }
        }
        return true;
    },

    /**
     * Update value.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    updateValue: function() {
        // Remove the options from the select that holds the actual
        // selected values
        while (this.values.length > 0) {
            this.values.remove(0);
        }

        // Create a new array consisting of the options marked as selected
        // on the official list
        var newOptions = new Array();
        var cntr = 0;
        var newOption;

        while (cntr < this.options.length) {
            newOption = document.createElement("option");
            if (this.options[cntr].text != null) {
                newOption.text = this.options[cntr].text;
            }
            if (this.options[cntr].value != null) {
                newOption.value = this.options[cntr].value;
            }
            newOption.selected = true;
            newOptions[newOptions.length] = newOption;
            ++ cntr;
        }
        cntr = 0;
        if (this.options.remove == null) {
            // For Mozilla
            while (cntr < newOptions.length) {
                this.values.add(newOptions[cntr], null);
                ++cntr;
            }
        } else {
            // Windows and Opera do
            while (cntr < newOptions.length) {
                this.values.add(newOptions[cntr], cntr);
                ++cntr;
            }
        }
        return true;
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.scheduler");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.formElements");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");

/** 
 * @class This class contains functions for scheduler components.
 * @static
 */
webui.suntheme4_2.scheduler = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @config {String} datePickerId
     * @config {String} dateFieldId
     * @config {String} dateClass
     * @config {String} selectedClass
     * @config {String} edgeClass
     * @config {String} edgeSelectedClass
     * @config {String} todayClass
     * @config {String} dateFormat
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize scheduler.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        webui.suntheme4_2.prototypejs.extend(domNode, props, false);
        domNode.dateLinkId = props.datePickerId + ":dateLink"; 

        // Set functions.
        domNode.setSelected = webui.suntheme4_2.scheduler.setSelected;
        domNode.setDateValue = webui.suntheme4_2.scheduler.setDateValue; 
        domNode.isToday = webui.suntheme4_2.scheduler.isToday;

        return true;
    },

    /**
     * Set selected link.
     *
     * @param {String} value
     * @param {Node} link
     * @return {boolean} true if successful; otherwise, false.
     */
    setDateValue: function(value, link) {
        webui.suntheme4_2.field.setValue(this.dateFieldId, value); 
        return this.setSelected(link);	
    },

    /**
     * Set selected.
     *
     * @param {Node} link
     * @return {boolean} true if successful; otherwise, false.
     */
    setSelected: function(link) {
        if (link == null) {
            return false;
        }

        var dateLink;	
        var linkNum = 0;	

        // Remove any prior highlight 
        while (linkNum < 42) {
            dateLink = document.getElementById(this.dateLinkId + linkNum);  
            if (dateLink == null) {
                break;    
            }

            if (dateLink.className == this.edgeSelectedClass) {
                dateLink.className = this.edgeClass;
            } else if (dateLink.className == this.selectedClass) {
                if (this.isToday(dateLink.title)) {
                    dateLink.className = this.todayClass;
                } else {
                    dateLink.className = this.dateClass;
                }
            }
            linkNum++;
        }

        // apply the selected style to highlight the selected link
        if (link.className == this.dateClass || 
            link.className == this.todayClass) {	
            link.className = this.selectedClass;
        } else if (link.className == this.edgeClass) {
            link.className = this.edgeSelectedClass;
        }
        this.currentSelection = link;
        return true;
    },

    /**
     * Find out if date is today's date.
     *
     * @param {Object} date
     * @return {boolean} true if date is today.
     */
    isToday: function(date) {
        var todaysDate = new Date();
        var pattern = new String(this.dateFormat); 
        var yearIndex = pattern.indexOf("yyyy"); 
        var monthIndex = pattern.indexOf("MM"); 
        var dayIndex = pattern.indexOf("dd"); 
        var currYear = todaysDate.getFullYear(); 
        var currMonth = todaysDate.getMonth() + 1; 
        var currDay = todaysDate.getDate(); 

        if (currYear == parseInt(date.substr(yearIndex, 4))
            && currMonth == parseInt(date.substr(monthIndex, 2))
                && currDay == parseInt(date.substr(dayIndex, 2))) {
            return true;
        }
        return false;
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.table");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.formElements");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");

/** 
 * @class This class contains functions for table components.
 * <p>
 * Once the table is rendered, you will be able to invoke functions directly on 
 * the HTML element. For example:
 * </p><p><pre>
 * var table = document.getElementById("form1:table1");
 * var count = table.getAllSelectedRowsCount();
 * </pre></p><p>
 * Note: It is assumed that formElements.js has been included in the page. In
 * addition, all given HTML element IDs are assumed to be the outter most tag
 * enclosing the component.
 * </p>
 * @static
 */
webui.suntheme4_2.table = {
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Public functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * This function is used to confirm the number of selected components (i.e., 
     * checkboxes or radiobuttons used to de/select rows of the table), affected
     * by a delete action. This functionality requires the selectId property of
     * the tableColumn component and hiddenSelectedRows property of the
     * tableRowGroup component to be set.
     * <p>
     * If selections are hidden from view, the confirmation message indicates the
     * number of selections not displayed in addition to the total number of
     * selections. If selections are not hidden, the confirmation message indicates
     * only the total selections.
     * </p>
     *
     * @return {boolean} A value of true if the user clicks the "OK" button, or 
     * false if the user clicks the "Cancel" button.
     */
    confirmDeleteSelectedRows: function() {
        return this.confirmSelectedRows(this.deleteSelectionsMsg);
    },

    /**
     * This function is used to confirm the number of selected components (i.e., 
     * checkboxes or radiobuttons used to de/select rows of the table), affected by
     * an action such as edit, archive, etc. This functionality requires the 
     * selectId property of the tableColumn component and hiddenSelectedRows
     * property of the tableRowGroup component to be set.
     * <p>
     * If selections are hidden from view, the confirmation message indicates the
     * number of selections not displayed in addition to the total number of
     * selections. If selections are not hidden, the confirmation message indicates
     * only the total selections.
     * </p>
     *
     * @param {String} message The confirmation message (e.g., Archive all selections?).
     * @return {boolean} A value of true if the user clicks the "OK" button, or 
     * false if the user clicks the "Cancel" button.
     */
    confirmSelectedRows: function(message) {
        // Get total selections message.
        var totalSelections = this.getAllSelectedRowsCount();
        var totalSelectionsArray = this.totalSelectionsMsg.split("{0}");
        var totalSelectionsMsg = totalSelectionsArray[0] + totalSelections;

        // Append hidden selections message.
        var hiddenSelections = this.getAllHiddenSelectedRowsCount();
        if (hiddenSelections > 0) {
            // Get hidden selections message.
            var hiddenSelectionsArray = this.hiddenSelectionsMsg.split("{0}");
            var hiddenSelectionsMsg = hiddenSelectionsArray[0] + hiddenSelections;

            totalSelectionsMsg = hiddenSelectionsMsg + totalSelectionsMsg;
        }
        return (message != null)
            ? confirm(totalSelectionsMsg + message)
            : confirm(totalSelectionsMsg);
    },

    /**
     * This function is used to toggle the filter panel from the filter menu. This
     * functionality requires the filterId of the table component to be set. In 
     * addition, the selected value must be set as well to restore the default
     * selected value when the embedded filter panel is closed.
     * <p>
     * If the "Custom Filter" option has been selected, the table filter panel is 
     * toggled. In this scenario, false is returned indicating the onChange event,
     * generated by the table filter menu, should not be allowed to continue.
     * </p><p>
     * If the "Custom Filter Applied" option has been selected, no action is taken.
     * Instead, the is selected filter menu is reverted back to the "Custom Filter" 
     * selection. In this scenario, false is also returned indicating the onChange 
     * event, generated by the table filter menu, should not be allowed to continue.
     * </p><p>
     * For all other selections, true is returned indicating the onChange event, 
     * generated by the table filter menu, should be allowed to continue.
     * </p>
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    filterMenuChanged: function() {
        // Validate panel IDs.
        if (this.panelToggleIds == null || this.panelToggleIds.length == 0) {
            return false;
        }

        // Get filter menu.
        var menu = webui.suntheme4_2.dropDown.getSelectElement(
            this.panelToggleIds[this.FILTER]);
        if (menu == null) {
            return true;
        }

        // Test if table filter panel should be opened.
        if (menu.options[menu.selectedIndex].value == this.customFilterOptionValue) {
            this.toggleFilterPanel();
            return false;
        } else if (menu.options[menu.selectedIndex].
                value == this.customFilterAppliedOptionValue) {
            // Set selected option.
            menu.selectedIndex = 0;
            for (var i = 0; i < menu.options.length; i++) {
                if (menu.options[i].value == this.customFilterOptionValue) {
                    menu.options[i].selected = true;
                    break;
                }
            }
            return false;
        }
        return true;
    },

    /**
     * This function is used to get the number of selected components in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select 
     * rows of the table). This functionality requires the selectId property of the
     * tableColumn component and hiddenSelectedRows property of the table
     * component to be set.
     *
     * @return {int} The number of components selected in the current page.
     */
    getAllSelectedRowsCount: function() {
        return this.getAllHiddenSelectedRowsCount() +
            this.getAllRenderedSelectedRowsCount();
    },

    /**
     * This function is used to get the number of selected components, in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table), currently hidden from view. This functionality requires 
     * the selectId property of the tableColumn component and hiddenSelectedRows
     * property of the table component to be set.
     *
     * @return {int} The number of selected components hidden from view.
     */
    getAllHiddenSelectedRowsCount: function() {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return count;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            count = count + this._getGroupHiddenSelectedRowsCount(this.groupIds[i]);
        }
        return count;
    },

    /**
     * This function is used to get the number of selected components, in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table), currently rendered. This functionality requires 
     * the selectId property of the tableColumn component to be set.
     *
     * @return {int} The number of selected components hidden from view.
     */
    getAllRenderedSelectedRowsCount: function() {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return count;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            count = count + this._getGroupRenderedSelectedRowsCount(this.groupIds[i]);
        }
        return count;
    },

    /**
     * This function is used to initialize all rows displayed in the table when the
     * state of selected components change (i.e., checkboxes or radiobuttons used to
     * de/select rows of the table). This functionality requires the selectId
     * property of the tableColumn component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    initAllRows: function() {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return false;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            this._initGroupRows(this.groupIds[i]);
        }
        return true;
    },

    /**
     * This function is used to toggle the filter panel open or closed. This
     * functionality requires the filterId of the table component to be set. In 
     * addition, the selected value must be set as well to restore the default
     * selected value when the embedded filter panel is closed.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    toggleFilterPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Toggle filter panel.
        this._togglePanel(this.panelIds[this.FILTER],
        this.panelFocusIds[this.FILTER], this.panelToggleIds[this.FILTER]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    /**
     * This function is used to toggle the preferences panel open or closed. This
     * functionality requires the filterId of the table component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    togglePreferencesPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Toggle preferences panel.
        this._togglePanel(this.panelIds[this.PREFERENCES],
        this.panelFocusIds[this.PREFERENCES], this.panelToggleIds[this.PREFERENCES]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Private functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The HTML element ID for the component.
     *
     * // Panel Properties
     * @config {Array} panelIds An array of embedded panel IDs.</li>
     * @config {Array} panelFocusIds An array of IDs used to set focus for open panels.</li>
     * @config {Array} panelToggleIds An array of IDs used to toggle embedded panels.</li>
     * @config {Array} panelToggleIconsOpen An array of toggle icons for open panels.</li>
     * @config {Array} panelToggleIconsClose An array of toggle icons for closed panels.</li>
     *
     * // Filter Properties
     * @config {String} basicFilterStyleClass The style class for basic or no filters.</li>
     * @config {String} customFilterStyleClass The style class for custom filters.</li>
     * @config {String} customFilterOptionValue The custom filter menu option value.</li>
     * @config {String} customFilterAppliedOptionValue The custom filter applied menu option value.</li>
     *
     * // Sort Panel Properties
     * @config {Array} sortColumnMenuIds An array of HTML element IDs for sort column menu components.</li>
     * @config {Array} sortOrderMenuIds An array of HTML element IDs for sort order menu components.</li>
     * @config {Array} sortOrderToolTips An array of tool tips used for sort order menus.</li>
     * @config {Array} sortOrderToolTipsAscending An array of ascending tool tips used for sort order menus.</li>
     * @config {Array} sortOrderToolTipsDescending An array of descending tool tips used for sort order menus.</li>
     * @config {String} duplicateSelectionMsg The message displayed for duplicate menu selections.</li>
     * @config {String} missingSelectionMsg The message displayed for missing menu selections.</li>
     * @config {String} selectSortMenuOptionValue The sort menu option value for the select column.</li>
     * @config {boolean} hiddenSelectedRows Flag indicating that selected rows might be currently hidden from view.</li>
     * @config {boolean} paginated Flag indicating table is in pagination mode.</li>
     *
     * // Group Properties
     * @config {String} selectRowStylClass The style class for selected rows.</li>
     * @config {Array} selectIds An arrary of component IDs used to select rows of the table.</li>
     * @config {Array} groupIds An array of TableRowGroup IDs rendered for the table.</li>
     * @config {Array} rowIds An array of row IDs for rendered for each TableRowGroup.</li>
     * @config {Array} hiddenSelectedRowCounts An array of selected row counts hidden from view.</li>
     * @config {String} hiddenSelectionsMsg The hidden selections message for confirm dialog.</li>
     * @config {String} totalSelectionsMsg The total selections message for confirm dialog.</li>
     * @config {String} deleteSelectionsMsg The delete selections message for confirm dialog.</li>
     *
     * // Group Panel Properties
     * @param {String} columnFooterId ID for column footer.</li>
     * @param {String} columnHeaderId ID for column header.</li>
     * @param {String} tableColumnFooterId ID for table column footer.</li>
     * @param {String} groupFooterId ID for group footer.</li>
     * @param {String} groupPanelToggleButtonId ID for group panel toggle button.</li>
     * @param {String} groupPanelToggleButtonToolTipOpen tool tip for open row group.</li>
     * @param {String} groupPanelToggleButtonToolTipClose tool tip for closed row group.</li>
     * @param {String} groupPanelToggleIconOpen The toggle icon for open row group.</li>
     * @param {String} groupPanelToggleIconClose The toggle icon for closed row group.</li>
     * @param {String} warningIconId ID for warning icon.</li>
     * @param {String} warningIconOpen The warning icon for open row group.</li>
     * @param {String} warningIconClosed The warning icon for closed row group.</li>
     * @param {String} warningIconToolTipOpen The warning icon tool tip for open row group.</li>
     * @param {String} warningIconToolTipClose The warning icon tool tip for closed row group.</li>
     * @param {String} collapsedHiddenFieldId ID for collapsed hidden field.</li>
     * @param {String} selectMultipleToggleButtonId ID for select multiple toggle button.</li>
     * @param {String} selectMultipleToggleButtonToolTip The select multiple toggle button tool tip.</li>
     * @param {String} selectMultipleToggleButtonToolTipSelected The select multiple toggle button tool tip when selected.</li>
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize table.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        webui.suntheme4_2.prototypejs.extend(domNode, props, false);

        // Misc properties.
        domNode.SEPARATOR = ":";   // NamingContainer separator.

        // Panel toggle keys.
        domNode.SORT        = 0;
        domNode.PREFERENCES = 1;
        domNode.FILTER      = 2;

        // Sort keys.
        domNode.PRIMARY   = 0;
        domNode.SECONDARY = 1;
        domNode.TERTIARY  = 2;

        // Replace extra backslashes, JSON escapes new lines (e.g., \\n).
	domNode.hiddenSelectionsMsg = props.hiddenSelectionsMsg.replace(/\\n/g, "\n");
        domNode.totalSelectionsMsg = props.totalSelectionsMsg.replace(/\\n/g, "\n");
	domNode.missingSelectionMsg = props.missingSelectionMsg.replace(/\\n/g, "\n");
	domNode.duplicateSelectionMsg = props.duplicateSelectionMsg.replace(/\\n/g, "\n");
        domNode.deleteSelectionsMsg = props.deleteSelectionsMsg.replace(/\\n/g, "\n");

        // Private functions.
        domNode._getGroupSelectedRowsCount = webui.suntheme4_2.table._getGroupSelectedRowsCount;
        domNode._getGroupHiddenSelectedRowsCount = webui.suntheme4_2.table._getGroupHiddenSelectedRowsCount;
        domNode._getGroupRenderedSelectedRowsCount = webui.suntheme4_2.table._getGroupRenderedSelectedRowsCount;
        domNode._initGroupRows = webui.suntheme4_2.table._initGroupRows;
        domNode._initPrimarySortOrderMenu = webui.suntheme4_2.table._initPrimarySortOrderMenu;
        domNode._initPrimarySortOrderMenuToolTip = webui.suntheme4_2.table._initPrimarySortOrderMenuToolTip;
        domNode._initSecondarySortOrderMenu = webui.suntheme4_2.table._initSecondarySortOrderMenu;
        domNode._initSecondarySortOrderMenuToolTip = webui.suntheme4_2.table._initSecondarySortOrderMenuToolTip;
        domNode._initSortColumnMenus = webui.suntheme4_2.table._initSortColumnMenus;
        domNode._initSortOrderMenu = webui.suntheme4_2.table._initSortOrderMenu;
        domNode._initSortOrderMenus = webui.suntheme4_2.table._initSortOrderMenus;
        domNode._initSortOrderMenuToolTip = webui.suntheme4_2.table._initSortOrderMenuToolTip;
        domNode._initTertiarySortOrderMenu = webui.suntheme4_2.table._initTertiarySortOrderMenu;
        domNode._initTertiarySortOrderMenuToolTip = webui.suntheme4_2.table._initTertiarySortOrderMenuToolTip;
        domNode._resetFilterMenu = webui.suntheme4_2.table._resetFilterMenu;
        domNode._selectAllRows = webui.suntheme4_2.table._selectAllRows;
        domNode._selectGroupRows = webui.suntheme4_2.table._selectGroupRows;
        domNode._toggleGroupPanel = webui.suntheme4_2.table._toggleGroupPanel;
        domNode._togglePanel = webui.suntheme4_2.table._togglePanel;
        domNode._toggleSortPanel = webui.suntheme4_2.table._toggleSortPanel;
        domNode._validateSortPanel = webui.suntheme4_2.table._validateSortPanel;

        // Public functions.
        domNode.confirmDeleteSelectedRows = webui.suntheme4_2.table.confirmDeleteSelectedRows;
        domNode.confirmSelectedRows = webui.suntheme4_2.table.confirmSelectedRows;
        domNode.filterMenuChanged = webui.suntheme4_2.table.filterMenuChanged;
        domNode.getAllSelectedRowsCount = webui.suntheme4_2.table.getAllSelectedRowsCount;
        domNode.getAllHiddenSelectedRowsCount = webui.suntheme4_2.table.getAllHiddenSelectedRowsCount;
        domNode.getAllRenderedSelectedRowsCount = webui.suntheme4_2.table.getAllRenderedSelectedRowsCount;
        domNode.initAllRows = webui.suntheme4_2.table.initAllRows;
        domNode.toggleFilterPanel = webui.suntheme4_2.table.toggleFilterPanel;
        domNode.togglePreferencesPanel = webui.suntheme4_2.table.togglePreferencesPanel;

        return true;
    },

    /**
     * This function is used to get the number of selected components for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table).
     * This functionality requires the selectId property of the tableColumn component
     * and the hiddenSelectedRows property of the table component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of components selected in the current page.
     * @private
     */
    _getGroupSelectedRowsCount: function(groupId) {
        return this._getGroupHiddenSelectedRowsCount(groupId) + 
            this._getGroupRenderedSelectedRowsCount(groupId);
    },

    /**
     * This function is used to get the number of selected components, for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table),
     * currently hidden from view. This functionality requires the selectId property
     * of the tableColumn component and hiddenSelectedRows property of the table
     * component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of selected components hidden from view.
     * @private
     */
    _getGroupHiddenSelectedRowsCount: function(groupId) {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return count;
        }

        // Find the given group Id in the groupIds array.
        for (var i = 0; i < this.groupIds.length; i++) {
            // Get selectId and rowIds array associated with groupId.
            if (groupId == this.groupIds[i]) {
                count = this.hiddenSelectedRowCounts[i];
                break;
            }
        }
        return count;
    },

    /**
     * This function is used to get the number of selected components, for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table),
     * currently rendered. This functionality requires the selectId property of the
     * tableColumn component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of components selected in the current page.
     * @private
     */
    _getGroupRenderedSelectedRowsCount: function(groupId) {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return count;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + 
                this.SEPARATOR + selectId);
            if (select != null && select.getProps().checked) {
                count++;
            }
        }
        return count;
    },

    /**
     * This function is used to initialize rows for the given groupwhen the state
     * of selected components change (i.e., checkboxes or radiobuttons used to
     * de/select rows of the table). This functionality requires the selectId
     * property of the tableColumn component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initGroupRows: function(groupId) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        var checked = true; // Checked state of multiple select button.
        var selected = false; // At least one component is selected.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + 
                this.SEPARATOR + selectId);
            if (select == null) {
                continue;
            }

            // Set row style class.
            var row = document.getElementById(this.groupIds[i] + 
                this.SEPARATOR + rowIds[k]);
            var props = select.getProps();
            if (select.getProps().checked == true) {
                webui.suntheme4_2.common.addStyleClass(row, 
                    this.selectRowStyleClass);
                selected = true;
            } else {
                webui.suntheme4_2.common.stripStyleClass(row, 
                    this.selectRowStyleClass);
                checked = false;
            }
        }

        // Set multiple select button state.
        var title;
        var checkbox = document.getElementById(groupId + this.SEPARATOR + 
            this.selectMultipleToggleButtonId);
        if (checkbox != null) {
            title = (checked) 
                ? this.selectMultipleToggleButtonToolTipSelected
                : this.selectMultipleToggleButtonToolTip;
            checkbox.setProps({
                "checked": checked,
                "title": title
            });
        }

        // Get flag indicating groupis collapsed.
        var prefix = groupId + this.SEPARATOR;
        var collapsed = !webui.suntheme4_2.common.isVisible(prefix + rowIds[0]);

        // Set next warning image.
        var image = document.getElementById(prefix + this.warningIconId);
        if (image != null) {
            var src = this.warningIconOpen;
            title = this.warningIconToolTipOpen;

            // Don't show icon when multiple select is checked.
            if (collapsed && selected && !checked) {
                src = this.warningIconClose;
                title = this.warningIconToolTipClose;
            }
            image.setProps({
               "src": src,
               "title": title
            });
        }
        return true;
    },

    /**
     * This function is used to initialize the primary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initPrimarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.PRIMARY], 
            this.sortOrderMenuIds[this.PRIMARY]);
    },  

    /**
     * This function is used to initialize the primary sort order menu tool tips 
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initPrimarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortOrderMenuIds[this.PRIMARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.PRIMARY], 
            this.sortOrderMenuIds[this.PRIMARY]);
    },

    /**
     * This function is used to initialize the secondary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSecondarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.SECONDARY], 
            this.sortOrderMenuIds[this.SECONDARY]);
    },

    /**
     * This function is used to initialize the secondary sort order menu tool tips
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSecondarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortOrderMenuIds[this.SECONDARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.SECONDARY], 
            this.sortOrderMenuIds[this.SECONDARY]);
    },

    /**
     * This function is used to initialize the primary, secondary, and tertiary 
     * sort column menus used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortColumnMenus: function() {
        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Set initial selected option for all sort column menus.
        for (var i = 0; i < this.sortColumnMenuIds.length; i++) {
            // Get sort column menu.
            var sortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
                this.sortColumnMenuIds[i]);
            if (sortColumnMenu == null) {
                continue;
            }

            // Set default selected value.
            sortColumnMenu.selectedIndex = 0;

            // Set selected option.
            for (var k = 0; k < sortColumnMenu.options.length; k++) {
                if (sortColumnMenu.options[k].defaultSelected == true) {
                    sortColumnMenu.options[k].selected = true;
                    break;
                }
            }
            // Ensure hidden filed values are updated.
            webui.suntheme4_2.dropDown.changed(this.sortColumnMenuIds[i]);
        }
        return true;
    },

    /**
     * This function is used to initialize the primary, secondary, and tertiary 
     * sort order menus used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenus: function() {
        // Validate sort order menu IDs.
        if (this.sortOrderMenuIds == null || this.sortOrderMenuIds.length == 0) {
            return false;
        }

        // Set initial selected option for all sort column menus.
        for (var i = 0; i < this.sortOrderMenuIds.length; i++) {
            // Get sort order menu.
            var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(
                this.sortOrderMenuIds[i]);
            if (sortOrderMenu == null) {
                continue;
            }

            // Set default selected value.
            sortOrderMenu.selectedIndex = 0;

            // Get sort column menu.
            var sortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
                this.sortColumnMenuIds[i]);
            if (sortColumnMenu != null) {
                // If the table is paginated and there are no hidden selected rows, the select
                // column cannot be sorted descending.
                if (sortColumnMenu.options[sortColumnMenu.selectedIndex].
                        value == this.selectSortMenuOptionValue
                        && !this.hiddenSelectedRows && this.paginated) {
                    sortOrderMenu.options[1].disabled = true;
                } else {
                    sortOrderMenu.options[1].disabled = false;
                }
            }

            // Set selected option.
            for (var k = 0; k < sortOrderMenu.options.length; k++) {
                if (sortOrderMenu.options[k].defaultSelected == true) {
                    sortOrderMenu.options[k].selected = true;
                    break;
                }
            }
            // Ensure hidden filed values and styles are updated.
            webui.suntheme4_2.dropDown.changed(this.sortOrderMenuIds[i]);

            // Initialize tool tip.
            this._initSortOrderMenuToolTip(this.sortColumnMenuIds[i], this.sortOrderMenuIds[i]);
        }
        return true;
    },

    /**
     * This function is used to initialize sort order menus used in the
     * sort panel. When a sort column menu changes, the given sort order 
     * menu is initialized based on the the selected value of the given
     * sort column menu.
     *
     * @param {String} sortColumnMenuId The HTML element ID for the sort column menu component.
     * @param {String} sortOrderMenuId The HTML element ID for the sort order menu component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenu: function(sortColumnMenuId, sortOrderMenuId) {
        if (sortColumnMenuId == null || sortOrderMenuId == null) {
            return false;
        }

        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Validate sort order menu IDs.
        if (this.sortOrderMenuIds == null || this.sortOrderMenuIds.length == 0) {
            return false;
        }

        // Get sort column menu.
        var sortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(sortColumnMenuId);
        if (sortColumnMenu == null) {
            return false;
        }

        // Get sort order menu.
        var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(sortOrderMenuId);
        if (sortOrderMenu == null) {
            return false;
        }

        // Reset selected option.
        sortOrderMenu.selectedIndex = 0; // Default ascending.

        // Get sort column menu selected index.            
        var selectedIndex = (sortColumnMenu.selectedIndex > -1)
            ? sortColumnMenu.selectedIndex : 0; // Default to first option.

        // If the table is paginated and there are no hidden selected rows, the select
        // column cannot be sorted descending.
        if (sortColumnMenu.options[selectedIndex].value == this.selectSortMenuOptionValue
                && !this.hiddenSelectedRows && this.paginated) {
            sortOrderMenu.options[1].disabled = true;
        } else {
            sortOrderMenu.options[1].disabled = false;
        }

        // Attempt to match the selected index of the given sort column menu with the
        // default selected value from either sort column menu. If a match is found, the 
        // default selected value of the associated sort order menu is retrieved. This
        // default selected value is set as the current selection of the given sort 
        // order menu.
        for (var i = 0; i < this.sortColumnMenuIds.length; i++) {
            // Get the current sort column menu to test the default selected value.
            var currentSortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
                this.sortColumnMenuIds[i]);
            if (currentSortColumnMenu == null) {
                continue;
            }

            // Find default selected value for the current sort column menu.
            var defaultSelected = null;
            for (var k = 0; k < currentSortColumnMenu.options.length; k++) {
                if (currentSortColumnMenu.options[k].defaultSelected == true) {
                    defaultSelected = currentSortColumnMenu.options[k].value;
                    break;
                }
            }

            // Match default selected value with selected index value.
            if (defaultSelected != null && defaultSelected ==
                    sortColumnMenu.options[selectedIndex].value) {
                // Get current sort order menu to test the default selected value.
                var currentSortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(
                    this.sortOrderMenuIds[i]);
                if (currentSortOrderMenu == null) {
                    continue;
                }

                // Find default selected index for the current sort order menu.
                var defaultSelectedIndex = -1;
                for (var c = 0; c < currentSortOrderMenu.options.length; c++) {
                    if (currentSortOrderMenu.options[c].defaultSelected == true) {
                        defaultSelectedIndex = c;
                        break;
                    }
                }

                // Set selected value for given sort order menu.
                if (defaultSelectedIndex > -1) {
                    sortOrderMenu.options[defaultSelectedIndex].selected = true;
                } else {
                    sortOrderMenu.options[0].selected = true; // Default ascending.
                }
                break;
            }
        }
        // Ensure hidden field values and styles are updated.
        webui.suntheme4_2.dropDown.changed(sortOrderMenuId);

        // Set sort order menu tool tip.
        return this._initSortOrderMenuToolTip(sortColumnMenuId, sortOrderMenuId);
    },

    /**
     * This function is used to initialize sort order menu tool tips used in the 
     * sort panel. When a sort column menu changes, the given sort order 
     * menu tool tip is initialized based on the the selected value of the given
     * sort column menu.
     *
     * @param {String} sortColumnMenuId The HTML element ID for the sort column menu component.
     * @param {String} sortOrderMenuId The HTML element ID for the sort order menu component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenuToolTip: function(sortColumnMenuId, sortOrderMenuId) {
        if (sortColumnMenuId == null || sortOrderMenuId == null) {
            return false;
        }

        // Get sort column menu.
        var sortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(sortColumnMenuId);
        if (sortColumnMenu == null) {
            return false;
        }

        // Get sort order menu.
        var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(sortOrderMenuId);
        if (sortOrderMenu == null) {
            return false;
        }

        // Get tool tip associated with given sort order menu.
        var toolTip = "";
        if (this.sortOrderToolTips != null && this.sortOrderToolTips.length != 0
                && this.sortOrderMenuIds != null) {
            for (var i = 0; i < this.sortOrderMenuIds.length; i++) {
                // The tool tip is at index zero, after splitting the message.
                if (sortOrderMenuId == this.sortOrderMenuIds[i]) {
                    toolTip = this.sortOrderToolTips[i].split("{0}")[0];
                    break;
                }
            }
        }

        // Get sort column menu selected index.            
        var selectedIndex = (sortColumnMenu.selectedIndex > -1)
            ? sortColumnMenu.selectedIndex : 0; // Default to first option.

        // Set tool tip.
        if (sortOrderMenu.options[sortOrderMenu.selectedIndex].value == "true") {
            sortOrderMenu.title = toolTip + this.sortOrderToolTipsDescending[selectedIndex];
        } else {
            // Default ascending.
            sortOrderMenu.title = toolTip + this.sortOrderToolTipsAscending[selectedIndex];
        }
        return true;
    },

    /**
     * This function is used to initialize the tertiary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initTertiarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.TERTIARY], 
            this.sortOrderMenuIds[this.TERTIARY]);
    },

    /**
     * This function is used to initialize the tertiary sort order menu tool tips
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initTertiarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortOrderMenuIds[this.TERTIARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.TERTIARY], 
            this.sortOrderMenuIds[this.TERTIARY]);
    },

    /**
     * This function is used to reset filter drop down menu. This functionality 
     * requires the filterId of the table component to be set. In addition,
     * the selected value must be set as well to restore the default selected
     * value when the embedded filter panel is closed.
     *
     * @param {String} filterId The HTML element ID of the filter menu.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _resetFilterMenu: function(filterId) {
        if (filterId == null) {
            return false;
        }

        // Get filter menu.
        var menu = webui.suntheme4_2.dropDown.getSelectElement(filterId);
        if (menu == null) {
            return true;
        }

        // Get div element associated with the filter panel.
        var div = document.getElementById(this.panelIds[this.FILTER]);
        if (div == null) {
            return false;
        }

        // Set selected style.
        if (webui.suntheme4_2.common.isVisibleElement(div)) {
            webui.suntheme4_2.common.stripStyleClass(menu, this.basicFilterStyleClass);
            webui.suntheme4_2.common.addStyleClass(menu, this.customFilterStyleClass);
        } else {
            // Reset default selected option.
            menu.selectedIndex = 0;
            for (var i = 0; i < menu.options.length; i++) {
                if (menu.options[i].defaultSelected == true) {
                    menu.options[i].selected = true;
                    break;
                }
            }
            webui.suntheme4_2.common.stripStyleClass(menu, this.customFilterStyleClass);
            webui.suntheme4_2.common.addStyleClass(menu, this.basicFilterStyleClass);
        }
        return true;
    },

    /**
     * This function is used to set the selected state components in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table). This functionality requires the selectId property of
     * the tableColumn component to be set.
     *
     * @param {boolean} selected Flag indicating components should be selected.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _selectAllRows: function(selected) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return false;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            this._selectGroupRows(this.groupIds[i], selected);
        }
        return true;
    },

    /**
     * This function is used to set the selected state components for the given row group
     * (i.e., checkboxes or radiobuttons used to de/select rows of the table). This 
     * functionality requires the selectId property of the tableColumn component to be
     * set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @param {boolean} selected Flag indicating components should be selected.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _selectGroupRows: function(groupId, selected) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + this.SEPARATOR + selectId);
            if (select == null) {
                continue;
            }
            select.setProps({checked: new Boolean(selected).valueOf()});
        }
        return this._initGroupRows(groupId); // Set row highlighting.
    },

    /**
     * This function is used to toggle row group panels open or closed.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _toggleGroupPanel: function(groupId) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get rowIds array associated with groupId.
        var rowIds = null;
        for (var c = 0; c < this.groupIds.length; c++) {
            if (groupId == this.groupIds[c]) {
                rowIds = this.rowIds[c];
                break;
            }
        }

        // If row IDs could not be found, do not continue.
        if (rowIds == null) {
            return false;
        }

        // Get flag indicating group is collapsed.
        var prefix = groupId + this.SEPARATOR;
        var collapsed = !webui.suntheme4_2.common.isVisible(prefix + rowIds[0]);

        // Get the number of column headers and table column footers for all 
        // TableRowGroup children.
        var _prefix;
        var columnHeaderId;
        var tableColumnFooterId;
        var columnHeadersCount = 0;
        var tableColumnFootersCount = 0;
        for (var i = 0; i < this.groupIds.length; i++) {
            // Only need to test first nested column header/footer; thus, index 0 is used.        
            _prefix = this.groupIds[i] + this.SEPARATOR;
            columnHeaderId = _prefix + this.columnHeaderId + this.SEPARATOR + "0";
            tableColumnFooterId = _prefix + this.tableColumnFooterId + this.SEPARATOR + "0";
            if (document.getElementById(columnHeaderId) != null) {
                columnHeadersCount++;
            }
            if (document.getElementById(tableColumnFooterId) != null) {
                tableColumnFootersCount++;
            }
        }

        // Toggle nested column footer.
        var rowIndex = 0;
        var columnFooterId;
        while (true) {
            columnFooterId = prefix + this.columnFooterId + 
                this.SEPARATOR + rowIndex++;
            if (document.getElementById(columnFooterId) == null) {
                break;
            }
            webui.suntheme4_2.common.setVisible(columnFooterId, collapsed);
        }

        // Toggle column header only if multiple column headers are shown.
        if (columnHeadersCount > 1) {
            rowIndex = 0;
            while (true) {
                columnHeaderId = prefix + this.columnHeaderId + 
                    this.SEPARATOR + rowIndex++;
                if (document.getElementById(columnHeaderId) == null) {
                    break;
                }            
                webui.suntheme4_2.common.setVisible(columnHeaderId, collapsed);
            }
        }

        // Toggle table column footer only if multiple column footers are shown.
        if (tableColumnFootersCount > 1) {
            rowIndex = 0;
            while (true) {
                tableColumnFooterId = prefix + this.tableColumnFooterId + 
                    this.SEPARATOR + rowIndex++;
                if (document.getElementById(tableColumnFooterId) == null) {
                    break;
                }
                webui.suntheme4_2.common.setVisible(tableColumnFooterId, collapsed);
            }
        }

        // Toggle group rows.
        var rowId;
        for (var k = 0; k < rowIds.length; k++) {
            rowId = prefix + rowIds[k];
            webui.suntheme4_2.common.setVisible(rowId, collapsed);
        }

        // Toggle group footers.
        webui.suntheme4_2.common.setVisible(prefix + this.groupFooterId, collapsed);

        // Set next toggle button image.
        var groupPanelToggleButtonId = prefix + this.groupPanelToggleButtonId;
        var hyperlink = document.getElementById(groupPanelToggleButtonId);
        var image = webui.suntheme4_2.hyperlink.getImgElement(groupPanelToggleButtonId);
        if (hyperlink != null && image != null) {
            image.style.cssText = null; // Need to clear clipped, theme image.
            if (collapsed) {
                // Need to provide themed icon key?
                image.src = this.groupPanelToggleIconOpen;
                hyperlink.setProps({title: this.groupPanelToggleButtonToolTipOpen});
            } else {
                image.src = this.groupPanelToggleIconClose;
                hyperlink.setProps({title: this.groupPanelToggleButtonToolTipClose});
            }
        }

        // Set collapsed hidden field.
        var hiddenField = document.getElementById(prefix + this.collapsedHiddenFieldId);
        if (hiddenField != null) {
            hiddenField.value = !collapsed;
        }
        return this._initGroupRows(groupId); // Set next warning image.
    },

    /**
     * This function is used to toggle embedded panels.
     *
     * @param {String} panelId The panel ID to toggle.
     * @param {String} panelFocusIdOpen The ID used to set focus when panel is opened.
     * @param {String} panelFocusIdClose The ID used to set focus when panel is closed.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _togglePanel: function(panelId, panelFocusIdOpen, panelFocusIdClose) {
        if (panelId == null) {
            return false;
        }

        // Toggle the given panel, hide all others.
        for (var i = 0; i < this.panelIds.length; i++) {
            // Get div element associated with the panel.
            var div = document.getElementById(this.panelIds[i]);
            if (div == null) {
                continue;
            }

            // Set display value. Alternatively, we could set div.style.display
            // equal to "none" or "block" (i.e., hide/show).
            if (this.panelIds[i] == panelId) {
                // Set focus when panel is toggled -- bugtraq 6316565.
                var focusElement = null;

                if (webui.suntheme4_2.common.isVisibleElement(div)) {
                    webui.suntheme4_2.common.setVisibleElement(div, false); // Hide panel.
                    focusElement = document.getElementById(panelFocusIdClose);
                } else {
                    webui.suntheme4_2.common.setVisibleElement(div, true); // Show panel.
                    focusElement = document.getElementById(panelFocusIdOpen);
                }

                // Set focus.
                if (focusElement != null) {
                    focusElement.focus();
                }
            } else {
                // Panels are hidden by default.
                webui.suntheme4_2.common.setVisibleElement(div, false);
            }

            // Get image from icon hyperlink component.
            var image = webui.suntheme4_2.hyperlink.getImgElement(this.panelToggleIds[i]);
            if (image == null) {
                continue; // Filter panel uses a drop down menu.
            }

            // Set image.
            if (webui.suntheme4_2.common.isVisibleElement(div)) {
                image.src = this.panelToggleIconsOpen[i];
            } else {
                image.src = this.panelToggleIconsClose[i];
            }
        }
        return true;
    },

    /**
     * This function is used to toggle the sort panel open or closed. This
     * functionality requires the filterId of the table component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _toggleSortPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Initialize sort column and order menus.
        this._initSortColumnMenus(); 
        this._initSortOrderMenus();

        // Toggle sort panel.
        this._togglePanel(this.panelIds[this.SORT], 
        this.panelFocusIds[this.SORT], this.panelToggleIds[this.SORT]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    /**
     * This function is used to validate sort column menu selections 
     * for the sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _validateSortPanel: function() {
        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Get sort column menus.
        var primarySortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortColumnMenuIds[this.PRIMARY]);
        var secondarySortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortColumnMenuIds[this.SECONDARY]);
        var tertiarySortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortColumnMenuIds[this.TERTIARY]);

        // Test primary and secondary menu selections.
        if (primarySortColumnMenu != null && secondarySortColumnMenu != null) {
            // Test if secondary sort is set, but primary is not.
            if (primarySortColumnMenu.selectedIndex < 1 
                    && secondarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (primarySortColumnMenu.selectedIndex > 0
                  && primarySortColumnMenu.selectedIndex == secondarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }

        // Test primary and tertiary menu selections.
        if (primarySortColumnMenu != null && tertiarySortColumnMenu != null) {
            // Test if tertiary sort is set, but primary is not.
            if (primarySortColumnMenu.selectedIndex < 1 
                    && tertiarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (primarySortColumnMenu.selectedIndex > 0
                    && primarySortColumnMenu.selectedIndex == tertiarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }

        // Test secondary and tertiary menu selections.
        if (secondarySortColumnMenu != null && tertiarySortColumnMenu != null) {
            // Test if tertiary sort is set, but secondary is not.
            if (secondarySortColumnMenu.selectedIndex < 1 
                    && tertiarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (secondarySortColumnMenu.selectedIndex > 0
                    && secondarySortColumnMenu.selectedIndex == tertiarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }
        return true;
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.tree");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.theme.common");

/** 
 * @class This class contains functions for tree components.
 * @static
 */ 
webui.suntheme4_2.tree = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize tree.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        webui.suntheme4_2.prototypejs.extend(domNode, props, false);

	// Set functions.
        domNode.clearAllHighlight = webui.suntheme4_2.tree.clearAllHighlight;
        domNode.clearHighlight = webui.suntheme4_2.tree.clearHighlight;
        domNode.expandCollapse = webui.suntheme4_2.tree.expandCollapse;
        domNode.expandTurner = webui.suntheme4_2.tree.expandTurner;
        domNode.findContainingTreeNode = webui.suntheme4_2.tree.findContainingTreeNode;
        domNode.findNodeByTypeAndProp = webui.suntheme4_2.tree.findNodeByTypeAndProp;
        domNode.getCookieValue = webui.suntheme4_2.tree.getCookieValue;
        domNode.getHighlightTreeBgColor = webui.suntheme4_2.tree.getHighlightTreeBgColor;
        domNode.getHighlightTreeTextColor = webui.suntheme4_2.tree.getHighlightTreeTextColor;
        domNode.getNormalTreeTextColor = webui.suntheme4_2.tree.getNormalTreeTextColor;
        domNode.getParentTreeNode = webui.suntheme4_2.tree.getParentTreeNode;
        domNode.getSelectedTreeNode = webui.suntheme4_2.tree.getSelectedTreeNode;
        domNode.getTree = webui.suntheme4_2.tree.getTree;
        domNode.highlight = webui.suntheme4_2.tree.highlight;
        domNode.highlightParent = webui.suntheme4_2.tree.highlightParent;
        domNode.isAnHref = webui.suntheme4_2.tree.isAnHref;
        domNode.isTreeHandle = webui.suntheme4_2.tree.isTreeHandle;
        domNode.onTreeNodeClick = webui.suntheme4_2.tree.onTreeNodeClick;
        domNode.selectTreeNode = webui.suntheme4_2.tree.selectTreeNode;
        domNode.setCookieValue = webui.suntheme4_2.tree.setCookieValue;
        domNode.treecontent_submit = webui.suntheme4_2.tree.treecontent_submit;
        domNode.treeNodeIsExpanded = webui.suntheme4_2.tree.treeNodeIsExpanded;
        domNode.unhighlightParent = webui.suntheme4_2.tree.unhighlightParent;
        domNode.updateHighlight = webui.suntheme4_2.tree.updateHighlight;

        return true;
    },

    /**
     * Set cookie.
     *
     * @param {String} cookieName
     * @param {String} val
     * @return {boolean} true if successful; otherwise, false.
     */
    setCookieValue: function(cookieName, val) {

	/*
	Fix for bug 6476859 :On initial visit to page, 
        getting selected node returns value from previous 
	session.
	The cookie value should be stored with the global path 
        so that it is applicable for all pages on site.
	*/

	document.cookie = cookieName + "=" + val +";path=/;";
        return true;
    },

    /**
     * Get cookie.
     *
     * @param {String} cookieName
     * @return {String} The cookie value.
     */
    getCookieValue: function(cookieName) {
        var docCookie = document.cookie;
        var pos= docCookie.indexOf(cookieName+"=");

        if (pos == -1) {
            return null;
        }

        var start = pos+cookieName.length+1;
        var end = docCookie.indexOf(";", start );

        if (end == -1) {
            end= docCookie.length;
        }
        return docCookie.substring(start, end);
    },

    /**
     * This function expands or collapses the given tree node.  It expects the
     * source of the given event object (if supplied) to be a tree handle
     * image.  It will change this image to point in the correct direction
     * (right or down).  This implementation depends on the tree handle image
     * names including "tree_handleright" and "tree_handledown" in them.
     * Swapping "right" and "down" in these names must change the handle
     * direction to right and down respectively.
     *
     * @param {Node} treeNode
     * @param {String} imageId
     * @param {Event} event
     * @return {boolean} true if successful; otherwise, false.
     */
    expandCollapse: function(treeNode, imageId, event) {
        var tree = this.getTree(treeNode);
        var childNodes = document.getElementById(treeNode.id+"_children");
        if (childNodes) {
            // Get the event source
            if (!event) {
                event = window.event;
            }
            
            var elt = document.getElementById(imageId);
            
            // get the image widget to compare the actual icon values
            // as opposed to checking the name of the final rendered image.
            
            var imgWidget = webui.suntheme4_2.dijit.byId(imageId);
            var imgProps = imgWidget.getProps();
            var nodeIcon = imgProps.icon;
            
            // First, unhighlight the parent if applicable
            this.unhighlightParent(this.getSelectedTreeNode(tree.id));

            // Change the style to cause the expand / collapse & switch the image
            var display = childNodes.style.display;
            
            if (display == "none") {
                childNodes.style.display = "block";
                if (nodeIcon == "TREE_HANDLE_RIGHT_MIDDLE") {
                    nodeIcon = "TREE_HANDLE_DOWN_MIDDLE";    
                } else if (nodeIcon == "TREE_HANDLE_RIGHT_TOP") {
                    nodeIcon = "TREE_HANDLE_DOWN_TOP";
                } else if (nodeIcon == "TREE_HANDLE_RIGHT_LAST") {
                    nodeIcon = "TREE_HANDLE_DOWN_LAST";
                } else if (nodeIcon == "TREE_HANDLE_RIGHT_TOP_NOSIBLING") {
                    nodeIcon = "TREE_HANDLE_DOWN_TOP_NOSIBLING";
                }
            } else {
                childNodes.style.display = "none";
                if (nodeIcon == "TREE_HANDLE_DOWN_MIDDLE" ) {
                    nodeIcon = "TREE_HANDLE_RIGHT_MIDDLE";    
                } else if (nodeIcon == "TREE_HANDLE_DOWN_TOP" ) {
                    nodeIcon = "TREE_HANDLE_RIGHT_TOP";
                } else if (nodeIcon == "TREE_HANDLE_DOWN_LAST" ) {
                    nodeIcon = "TREE_HANDLE_RIGHT_LAST";
                } else if (nodeIcon == "TREE_HANDLE_DOWN_TOP_NOSIBLING") {
                    nodeIcon = "TREE_HANDLE_RIGHT_TOP_NOSIBLING" ;
                }
            }
            // update the image property to reflect the icon change
            imgWidget.setProps({icon: nodeIcon});

            // Last, update the visible parent of the selected node if now hidden
            this.highlightParent(this.getSelectedTreeNode(tree.id));
        }
        return true;
    },

    /**
     * This function returns the Tree for the given TreeNode.  From
     * a DOM point of view, the tree directly contains all its children
     * (excluding *_children div tags.  This will return the first
     * parentNode that is a div w/ an id != "*_children".
     *
     * @param {Node} treeNode
     * @return {Node} The tree for the given TreeNode.
     */
    getTree: function(treeNode) {
        var tree = treeNode.parentNode;
        var SUFFIX = new String("_children");

        while (tree) {
            // Ignore all div's ending w/ SUFFIX
            if ((tree.nodeName == "DIV")
                    && (tree.id.substr(tree.id.length-SUFFIX.length) != SUFFIX)) {
		break;
            }
            tree = tree.parentNode;
        }
        return tree;
    },

    /**
     * This method handles TreeNode onClick events.  It takes the TreeNode
     * &lt;div&gt; object that was clicked on in order to process the
     * highlighting changes that are necessary.  This object may be obtained by
     * calling <code>getElementById("&lt;TreeNode.getClientId()&gt;")</code>.
     * If this function is invoked from the TreeNode &lt;div&gt; object itself
     * (as is the case when this method is implicitly called), the TreeNode
     * object is simply the <code>this</code> variable.
     *
     * @param {Node} treeNode
     * @param {String} imageId
     * @param {Event} event
     * @return {boolean} true if successful; otherwise, false.
     */
    onTreeNodeClick: function(treeNode, imageId, event) {
        // Check for Tree Handles
        // The handle image and its surrounding area represents the 
        // handler section of the tree node. Clicking on this area
        // alone should cause the node to toggle.
        if (this.isTreeHandle(event)) {
            this.expandCollapse(treeNode, imageId, event);
            return true;
        }

        // Make sure they clicked on an href
        if (!this.isAnHref(event)) {
            // Do nothing
            return true;
        }

        // If we're here, we should select the TreeNode
        return this.selectTreeNode(treeNode.id);
    },

    /**
     * This function may be used to select the given TreeNode. 
     * It will clear the previous TreeNode and select the 
     * given one. The parameter passed to the function is
     * the id of the currently selected treeNode - the
     * value of the tree component from a JSF perspective.
     * This node may not be availbe in the DOM tree as the
     * user may have clicked on the expanded parent node
     * and closed it. In this case the parent node needs
     * to show up in bold.
     *
     * @param {String} treeNodeId
     * @return {boolean} true if successful; otherwise, false.
     */
    selectTreeNode: function(treeNodeId) {
        // Find the top of the tree
	var treeNode = document.getElementById(treeNodeId);

	if (treeNode) {
            var tree = this.getTree(treeNode);

            // Clear the old highlighting
            this.clearAllHighlight(tree.id);

            // Mark the node as selected
            this.setCookieValue(tree.id+"-hi", treeNode.id);

            // first highlight is as a parent
            // when the left frame loads the nodes will 
            // be hightlighted correctly
            this.highlightParent(treeNode);
            this.highlight(treeNode);

            // onClick handler should proceed with hyperlink
            return true;
	} else {
	    // get the parent node ID and highlight the parent.
	    // var arr = treeNodeId.split(":");
	    // var lastBitOfId = arr[arr.length - 1].length;
	    // var lastIndex = treeNodeId.length - lastBitOfId.length;
	    var lastIndex = treeNodeId.lastIndexOf(":");
	    parentNodeId = treeNodeId.substr(0, lastIndex);
	    var parentNode = 
		document.getElementById(parentNodeId);
	    if (parentNode) {	
	        parentNode.style.fontWeight = "bold";
	    }
	    return false;
	}
    },

    /**
     * This function returns the selected TreeNode given the treeId of the
     * Tree.
     *
     * @param {String} treeId
     * @return {Node}The selected TreeNode.
     */
    getSelectedTreeNode: function(treeId) {
        var id = this.getCookieValue(treeId+"-hi");
        if (id) {
            return document.getElementById(id);
        }
        return null;
    },

    /**
     * Clear all highlighted nodes.
     *
     * @param {String} cookieId
     * @return {boolean} true if successful; otherwise, false.
     */
    clearAllHighlight: function(cookieId) {
        // Clear
        var selectedNode = this.getSelectedTreeNode(cookieId);
        this.clearHighlight(selectedNode);
        this.setCookieValue(cookieId+"-hi", "");

        // FIXME: Fix this...
        // this.clearHighlight(document.getElementById(currentHighlightParent));
        return true;
    },

    /**
     * Clear highlighted node
     *
     * @param {Node} node
     * @return {boolean} true if successful; otherwise, false.
     */
    clearHighlight: function(node) {
        if (node) {
	    node.className = 
                webui.suntheme4_2.theme.common.getClassName("TREE_ROW");
        }
        return true;
    },

    /**
     * This function determines if the event source was a tree handle image.
     * This implementation depends on the tree handle image file name
     * containing "tree_handle" and no other images containing this
     * string.
     *
     * @param {Event} event
     * @return {boolean} true if event was generated by tree handle.
     */
    isTreeHandle: function(event) {
        if (!event) {
            event = window.event;
            if (!event) {
                return false;
            }
        }
        var elt = (event.target) ? event.target : event.srcElement;

        // Ignore Tree Handles b/c they should not update highlighting
        
        if (elt.nodeName == "IMG") {
            var imgWidget = webui.suntheme4_2.dijit.byId(elt.id);
            var imgProps = imgWidget.getProps();
            var nodeIcon = imgProps.icon;
            if (nodeIcon.indexOf("TREE_HANDLE_") != -1) {
                return true;
            }
        } else if (elt.nodeName == "A") {
            // User might have been pressing enter around an image.
            // Note: I have never managed to get control to come here.
            
            aID = elt.id;
            var lastIndex = aID.lastIndexOf("_handle");
            if (lastIndex == -1) {
                return false;
            }
            var result = aID.substring(lastIndex, aID.length - 1);
            if (result == "_handle") {
                return true; 
            }
        }
        // Not a tree handle
        return false;
    },

    /**
     * This method checks to see if the event.target is an href, or if any of
     * the parent nodes which contain it is an href.  To be an href, it must be
     * an "A" tag with an "href" attribute containing atleast 4 characters.
     * (Note: Browsers will add on the protocol if you supply a relative URL
     * such as one starting with a '#', '/', or filename).
     *
     * @param {Event} event
     * @return {boolean} true if event was generated by a link.
     */
    isAnHref: function(event) {
        if (!event) {
            event = window.event;
            if (!event) {
                return false;
            }
        }
        var elt = (event.target) ? event.target : event.srcElement;

        // Look for parent href
        while (elt != null) {
            if (elt.nodeName == "A") {
                // Creates a String containing the url
                var url = new String(elt);
                if (url.length > 4) {
                    // All URLs are atleast this long
                    return true;
                }
            }
            elt = elt.parentNode;
        }
        // Not an href
        return false;
    },

    /**
     * This function updates the highlighting for the given Tree client id.
     * This function provides a way to restore the highlighting when a Tree is
     * reloaded in a window (necessary each page load).
     *
     * @param {String} cookieId
     * @return {boolean} true if successful; otherwise, false.
     */
    updateHighlight: function(cookieId) {
        var selNode = this.getSelectedTreeNode(cookieId);
        this.highlight(selNode);

        // FIXME: This doesn't work if the TreeNode element doesn't exist 
        // (which is the case for the server-side tree)
        return this.highlightParent(selNode);
    },

    /**
     * This function highlights the given <code>TreeNode</code>.  The
     * <code>obj</code> passed in is actually the &lt;div&gt; around the html
     * for the <code>TreeNode</code> and may be obtained by calling
     * <code>getElementById("&lt;TreeNode.getClidentId()&gt;")</code>.
     *
     * @param {Node} node
     * @return {boolean} true if successful; otherwise, false.
     */
    highlight: function(node) {
        if (node) {
	    node.className = 
                webui.suntheme4_2.theme.common.getClassName("TREE_SELECTED_ROW");
            return true;
        }
        return false;
    },

    /**
     * This function finds the handler image ICON associated with a given 
     * tree node. The ICON value is used to identify if the node in 
     * question is expanded or not. This is a private function and should
     * not be used by developers on the client side.
     *
     * @param {Node} node
     * @return {boolean} true if the node has an image whose ICON 
     *        indicates the node is expanded.
     */
    findNodeByTypeAndProp: function(node) {
        if (node == null) {        
            return true;
        }
        // First check to see if node is a handler image.
        // Then check if it is of the right type. "RIGHT" icon
        // type indicates the node is not expanded.
        if (node.nodeName == "IMG") {
        
            var imgWidget = webui.suntheme4_2.dijit.byId(node.id);
            var imgProps = imgWidget.getProps();
            var nodeIcon = imgProps.icon;

            if ((nodeIcon == "TREE_HANDLE_RIGHT_MIDDLE") ||
                (nodeIcon == "TREE_HANDLE_RIGHT_TOP") ||
                (nodeIcon == "TREE_HANDLE_RIGHT_LAST") ||
                (nodeIcon == "TREE_HANDLE_RIGHT_TOP_NOSIBLING")) {
                
                return false;
                
            } else if ((nodeIcon == "TREE_HANDLE_DOWN_MIDDLE") ||
                (nodeIcon == "TREE_HANDLE_DOWN_TOP") ||
                (nodeIcon == "TREE_HANDLE_DOWN_LAST") ||
                (nodeIcon == "TREE_HANDLE_DOWN_TOP_NOSIBLING")) {
                
                return true;
            }
        }        
        // Not what we want, walk its children if any
        // Return true for when null conditions arise.
        var nodeList = node.childNodes;
        if (!nodeList || (nodeList.length == 0)) {
            return true;
        }
        var result;
        for (var count = 0; count<nodeList.length; count++) {
            // Recurse
            result = this.findNodeByTypeAndProp(nodeList[count]);
            if (result) {
                // Propagate the result
                return result;
            }
        }
        // Not found
        return true;
    },

    /**
     * This function determines if the given TreeNode is expanded.  It returns
     * <code>true</code> if it is, <code>false</code> otherwise.
     *
     * @param {Node} treeNode
     * @return {boolean} true if TreeNode is expanded.
     */
    treeNodeIsExpanded: function(treeNode) {
        // Find the div containing the handler images for this TreeNode row
        // and pass it to a function that looks for the right image within
        // the div and returns true if the image has been found and is of the
        // type that indicates the node is expanded, false otherwise.
        var node = document.getElementById(treeNode.id + "LineImages");
        return this.findNodeByTypeAndProp(node);
        
    },

    /**
     * This function returns the parent TreeNode of the given TreeNode.
     *
     * @param {Node} treeNode
     * @return {Node} The parent TreeNode.
     */
    getParentTreeNode: function(treeNode) {
        // Get the parent id
        var parentId = treeNode.parentNode.id;
        var childrenIdx = parentId.indexOf("_children");
        if (childrenIdx == -1) {
            return null;
        }
        // This is really a peer div id to what we really want... remove _children
        parentId = parentId.substring(0, childrenIdx);
        // Return the parent TreeNode
        return document.getElementById(parentId);
    },

    /**
     * Unhighlight parent node.
     *
     * @param {Node} childNode
     * @return {boolean} true if successful; otherwise, false.
     */
    unhighlightParent: function(childNode) {
        if (!childNode) {
            return false;
        }

        // First find the parent node and make sure it is collapsed (we don't
        // highlight parent nodes when the selected node is visible)
        var parentNode = this.getParentTreeNode(childNode);
        var highlight = null;
        while (parentNode != null) {
            if (!this.treeNodeIsExpanded(parentNode)) {
                highlight = parentNode;
            }
            parentNode = this.getParentTreeNode(parentNode);
        }
        if (highlight) {
            highlight.style.fontWeight = "normal";
        }
        return true;
    },

    /**
     * Highlight parent node.
     *
     * @param {Node} childNode
     * @return {boolean} true if successful; otherwise, false.
     */
    highlightParent: function(childNode) {
        if (!childNode) {
            return false;
        }

        // First find the parent node and make sure it is collapsed (we don't
        // highlight parent nodes when the selected node is visible)
        var parentNode = this.getParentTreeNode(childNode);
        var highlight = null;
        while (parentNode != null) {
            if (!this.treeNodeIsExpanded(parentNode)) {
                highlight = parentNode;
            }
            parentNode = this.getParentTreeNode(parentNode);
        }
        if (highlight) {
            highlight.style.fontWeight = "bold";
        }
        return true;
    },

    /**
     * Get normal tree text color.
     *
     * @return {String} The text color.
     */
    getNormalTreeTextColor: function() {
        return "#003399";
    },

    /**
     * Get highlight background color.
     *
     * @return {String} The background color.
     */
    getHighlightTreeBgColor: function() {
        return "#CBDCAF";  // ~greenish color
    },

    /**
     * Get highlight tree text color.
     *
     * @return {String} The text color.
     */
    getHighlightTreeTextColor: function() {
        return "#000000";  // black
    },

    /**
     * Returns the TreeNode that contains the given link. This assumes the link
     * is a direct child of the node.
     *
     * @param {Node} link
     * @return {Node} The TreeNode containing the given link.
     */
    findContainingTreeNode: function(link) {     
        var linkId = link.id;
        var nodeId = linkId.substring(0, linkId.lastIndexOf(":"));
        return document.getElementById(nodeId);
    },

    /**
     * If the Tree's expandOnSelect property is true, this method is called to 
     * expand the turner of the tree node with the given labelLink.
     *
     * @param {Node} labelLink
     * @param {String} turnerId
     * @param {String} imageId
     * @param {boolean} isClientSide
     * @param {Event} event
     * @return {boolean} true if successful; otherwise, false.
     */
    expandTurner: function(labelLink, turnerId, imageId, isClientSide, event) {
        var labelLinkId = labelLink.id;
	var formId = labelLinkId.substring(0, labelLinkId.indexOf(":"));
	var node = this.findContainingTreeNode(labelLink);
	var turnerLink = document.getElementById(turnerId); 

	if (turnerLink == null) {
            return false;
	}
	if (!this.treeNodeIsExpanded(node)) {
            // folder is currently closed, expand it
            if (isClientSide) {
		this.expandCollapse(node, imageId, event);      
            } else {
		turnerLink.onclick();
            }    
        }
        return true;
    },

    /**
     * When the tree node link has an associated action, this method should
     * be called to ensure selection highlighting and (if necessary) node 
     * expansion occurs.
     * <p>
     * If the developer specifies the content facet for a given TreeNode, he 
     * should call this function from his facet hyperlink's onClick.
     * </p>
     *
     * @param {String} nodeId
     * @return {boolean} true if successful; otherwise, false.
     */
    treecontent_submit: function(nodeId) {
	if (nodeId == null) {
            return false;
        }
	var node = document.getElementById(nodeId);
	var tree = this.getTree(node);

	// update the current selection
	this.selectTreeNode(node.id);

	// set a cookie that the Tree's decode method will 
	// inspect and expand the corresponding node if necessary
	return this.setCookieValue(tree.id + "-expand", nodeId);
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.accordion");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.accordionTab");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.widgetBase");
 
webui.suntheme4_2.dojo.require("webui.suntheme4_2.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.theme.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.common");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.eventBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.config");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.dijit._Widget"); 
webui.suntheme4_2.dojo.require("webui.suntheme4_2.dijit._Templated");

/**
 * @name webui.suntheme4_2.widget.eventBase
 * @extends webui.suntheme4_2.dijit._Widget, webui.suntheme4_2.dijit._Templated
 * @class This class contains functions for widgets that extend eventBase.
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.eventBase", [
    webui.suntheme4_2.dijit._Widget, webui.suntheme4_2.dijit._Templated]);

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.eventBase.event =
        webui.suntheme4_2.widget.eventBase.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: null,

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: null
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: null,

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: null
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: null,

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: null
    }
};

/**
 * Initialize public events and functions.
 * <p>
 * Note: If this.event.<eventName> is not null, a public function shall be added
 * to the DOM node. To avoid name clashes, do not create private functions with
 * the names; refresh, stateChanged, or submit.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.initEvents = function () {
    if (this.event == null) {
        return false;
    }

    // Since the anchor id and name must be the same on IE, we cannot obtain the
    // widget using the DOM node ID via the public functions below. Therefore, 
    // we need to set the widget id via closure magic.
    var _id = this.id;

    // Flag to add subscribe function.
    var subscribe = false;

    // Add event topics.
    this.domNode.event = this.event;

    // Refresh.
    if (this.event.refresh != null) {
        // Set public function.
        this.domNode.refresh = function(execute) {
            return webui.suntheme4_2.dijit.byId(_id).refresh(execute);
        };
        subscribe = true;
    } else {
        this.domNode.event.refresh = null; // clean.
    }

    // Submit.
    if (this.event.submit != null) {
        // Set public function.
        this.domNode.submit = function(execute) {
            return webui.suntheme4_2.dijit.byId(_id).submit(execute);    
        };
        subscribe = true;
    } else {
        this.domNode.event.submit = null; // clean.
    }

    // State.
    if (this.event.state == null) {
        // Remove prototyped function.
        this.stateChanged = null;
        this.domNode.event.state = null; // clean.
    }
    
    // Subscribe.
    if (subscribe == true) {
        this.domNode.subscribe = function(topic, obj, func) {
            return webui.suntheme4_2.dijit.byId(_id).subscribe(topic, obj, func);
        };
    }
    return true;
};

/**
 * Publish an event topic.
 *
 * @param {String} topic The event topic to publish.
 * @param {Object} props Key-Value pairs of properties. This will be applied
 * to each topic subscriber.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.publish = function(topic, props) {
    // Obtain the Ajax module associated with this widget.
    var config = webui.suntheme4_2.config;
    if (config.ajax.module) {
        webui.suntheme4_2.dojo.require(config.ajax.module + "." + this.widgetName);
    }
    return webui.suntheme4_2.widget.eventBase.prototype._publish(topic, props);
};

/**
 * Publish an event topic.
 * <p>
 * Note: In order to obtain Ajax modules dynamically, this function shall be 
 * overridden by a custom AJAX implementation.
 * </p>
 * @param {String} topic The event topic to publish.
 * @param {Object} props Key-Value pairs of properties. This will be applied
 * to each topic subscriber.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.eventBase.prototype._publish = function(topic, props) {
    // Publish an event for custom AJAX implementations to listen for.
    webui.suntheme4_2.dojo.publish(topic, props);
    return true;
};

/**
 * Subscribe to an event topic.
 *
 * @param {String} topic The event topic to subscribe to.
 * @param {Object} obj The object in which a function will be invoked, or
 * null for default scope.
 * @param {String|Function} func The name of a function in context, or a 
 * function reference to invoke when topic is published. 
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.subscribe = function(topic, obj, func) {
    webui.suntheme4_2.dojo.subscribe(topic, obj, func);
    return true;
};

/**
 * Process refresh event.
 * <p>
 * Note: If this.event.refresh is not null, an event is published for custom
 * Ajax implementations to listen for. If event topics are not implemented for 
 * this widget, the function returns and a message is output to the console.
 * </p>
 * @param {String} execute The string containing a comma separated list 
 * of client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.refresh = function(execute) {
    if (this.event.refresh == null) {
        console.debug("Error: Refresh event topics not implemented for " + 
            this.widgetName); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this.publish(this.event.refresh.beginTopic, [{
        id: this.id,
        execute: execute,
        endTopic: this.event.refresh.endTopic
    }]);
    return true;
};

/**
 * Process state event.
 * <p>
 * Note: If this.event.state is not null, an event is published for custom
 * Ajax implementations to listen for. If event topics are not implemented for 
 * this widget, the function returns and a message is output to the console.
 * </p>
 * @param {Object} props Key-Value pairs of widget properties to update.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.stateChanged = function(props) {
    if (this.event.state == null) {
        console.debug("Error: State event topics not implemented for " + 
            this.widgetName); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this.publish(this.event.state.beginTopic, [{
        id: this.id,
        endTopic: this.event.state.endTopic,
        props: props
    }]);
    return true;
};

/**
 * Process submit event.
 * <p>
 * Note: If this.event.submit is not null, an event is published for custom
 * Ajax implementations to listen for. If event topics are not implemented for 
 * this widget, the function returns and a message is output to the console.
 * </p>
 * @param {String} execute Comma separated string containing a list of 
 * client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.submit = function(execute) {
    if (this.event.submit == null) {
        console.debug("Error: Submit event topics not implemented for " + 
            this.widgetName); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this.publish(this.event.submit.beginTopic, [{
        id: this.id,
        execute: execute,
        endTopic: this.event.submit.endTopic
    }]);
    return true;
};

/**
 * @name webui.suntheme4_2.widget.widgetBase
 * @extends webui.suntheme4_2.widget.eventBase
 * @class This class contains functions used for base functionality in all 
 * widgets. 
 * <p>
 * The widgetBase class inherits from webui.suntheme4_2.dijit._Widget and 
 * webui.suntheme4_2.dijit._Templated. The webui.suntheme4_2.dijit._Widget class is 
 * responsible for calling the buildRendering() and postCreate() functions in 
 * that order. The dijit_Templated function overrides the buildRendering() 
 * functon to fill in template properties.
 * <p></p>
 * The postCreate() function is responible for initializing CSS selectors, 
 * events, and public functions. Commonly used functions (e.g., getProps(), 
 * setProps(), and refresh() are set on the outermost DOM node via the 
 * "superclass" function of widgetBase. This inherited function is also 
 * responsible for invoking the private _setProps() function. 
 * <p></p>
 * The private _setProps() function is used to set widget properties that can be
 * updated by a web app developer. This helps encapsolate functionality and 
 * brand changes while providing a common API for all widgets. In addition, the 
 * widget is selctively updated (i.e., if and only if a key-value pair has been 
 * given). Saving given properties is deferred to the public setProps() function
 * which allows _setProps() to be used during initialization.
 * <p></p>
 * The private _setProps() function is also responsible for invoking 
 * setCommonProps() and setEventProps(). These properties may not always be set 
 * on the outermost DOM node; however, core (i.e., id, class, style, etc.) 
 * properties are. Core properties are set on the DOM via the "superclass" 
 * function of widgetBase which invokes the setCoreProps() function.
 * <p></p>
 * The getClassName() function is responsible for obtaining the selector that
 * will be set on the outermost DOM node. The private _setProps() function 
 * of widgetBase ensures that the getClassName() function is called prior to 
 * invoking setCoreProps(). In most cases, this function will be overridded in
 * order to apply widget specific selectors. However, selectors should be 
 * concatinated in order of precedence (e.g., the user's className property is 
 * always appended last).
 * <p></p>
 * The public setProps() function is responsible for extending the widget class
 * with properties so they can be used during later updates. After extending the
 * widget, the private _setProps() function is called. In some cases, the public
 * setProps() function may be overridden. For example, the label clears the
 * contents property from the widget because that is something we do not want to
 * extend.
 * <p></p>
 * The startup() function is typically called after the widget has been 
 * instantiated. For example, a progressBar might start a timer to periodically
 * refresh. 
 * <p></p>
 * Warning: It's not possible to append HTML elements from script that is 
 * not a direct child of the BODY element. If there is any Javascript
 * running inside the body that is a direct child of body, IE will throw
 * an "Internet Explorer cannot open the Internet site" error. For example,
 * webui.suntheme4_2.dijit._Templated._createNodesFromText generates such an error by calling
 * appendChild(). Therefore, widget creation must be deferred to the
 * window.onLoad event. See http://trac.dojotoolkit.org/ticket/4631
 * </p>
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.widgetBase", webui.suntheme4_2.widget.eventBase, {
    // Note: If your class contains arrays or other objects, they should be
    // declared in the constructor function so that each instance gets it's own
    // copy. Simple types (literal strings and numbers) are fine to declare in 
    // the class directly.
    constructor: function() {
        // Do nothing...
    },

    // Set defaults.
    common: webui.suntheme4_2.common, // Common utils.
    dojo: webui.suntheme4_2.dojo, // Dojo utils.
    prototypejs: webui.suntheme4_2.prototypejs, // Prototype utils.
    theme: webui.suntheme4_2.theme.common, // Theme utils.
    widget: webui.suntheme4_2.widget.common // Widget utils. 
});

/**
 * This function is used to render the widget from a template.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.buildRendering = function () {
    // Get default templates.
    if (this.templatePath == null && this.templateString == null) {
        this.templatePath = this.widget.getTemplatePath(this.widgetName);
        this.templateString = this.widget.getTemplateString(this.widgetName);
    }

    // The templatePath should have precedence. Therefore, in order for the 
    // templatePath to be used, templateString must be null.
    if (this.templatePath != null) {
        this.templateString = null;
    }

    // Template must be set prior to calling "superclass".
    return this.inherited("buildRendering", arguments);
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.widgetBase.prototype.getClassName = function() {
    return this.className;
};

/**
 * This function is used to get common properties from the widget. Please see
 * the setCommonProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.widgetBase.prototype.getCommonProps = function() {
    var props = {};

    // Set properties.
    if (this.accessKey) { props.accessKey = this.accessKey; }
    if (this.dir) { props.dir = this.dir; }
    if (this.lang) { props.lang = this.lang; }
    if (this.tabIndex) { props.tabIndex = this.tabIndex; }
    if (this.title) { props.title = this.title; }

    return props;
};

/**
 * This function is used to get core properties from the widget. Please see
 * the setCoreProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.widgetBase.prototype.getCoreProps = function() {
    var props = {};

    // Set properties.
    if (this.className) { props.className = this.className; }
    if (this.id) { props.id = this.id; }
    if (this.style) { props.style = this.style; }
    if (this.visible != null) { props.visible = this.visible; }

    return props;
};

/**
 * This function is used to get event properties from the widget. Please
 * see the setEventProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.widgetBase.prototype.getEventProps = function() {
    var props = {};

    // Set properties.
    if (this.onBlur) { props.onBlur = this.onBlur; }
    if (this.onChange) { props.onChange = this.onChange; }
    if (this.onClick) { props.onClick = this.onClick; }
    if (this.onDblClick) { props.onDblClick = this.onDblClick; }
    if (this.onFocus) { props.onFocus = this.onFocus; }
    if (this.onKeyDown) { props.onKeyDown = this.onKeyDown; }
    if (this.onKeyPress) { props.onKeyPress = this.onKeyPress; }
    if (this.onKeyUp) { props.onKeyUp = this.onKeyUp; }
    if (this.onMouseDown) { props.onMouseDown = this.onMouseDown; }
    if (this.onMouseOut) { props.onMouseOut = this.onMouseOut; }
    if (this.onMouseOver) { props.onMouseOver = this.onMouseOver; }
    if (this.onMouseUp) { props.onMouseUp = this.onMouseUp; }
    if (this.onMouseMove) { props.onMouseMove = this.onMouseMove; }
    if (this.onSelect) { props.onSelect = this.onSelect; }

    return props;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.widgetBase.prototype.getProps = function() {
    var props = {};

    // Set properties.
    this.prototypejs.extend(props, this.getCommonProps(), false);
    this.prototypejs.extend(props, this.getCoreProps(), false);
    this.prototypejs.extend(props, this.getEventProps(), false);

    return props;
};

/**
 * This function is used to test if widget has been initialized.
 * <p>
 * Note: It is assumed that an HTML element is used as a place holder for the
 * document fragment.
 * </p>
 * @return {boolean} true if widget is initialized.
 */
webui.suntheme4_2.widget.widgetBase.prototype.isInitialized = function() {
    // Testing if the outermost DOM node has been added to the document and
    // ensuring a Dojo attach point exists works fine for JSP. However, the 
    // following code always returns null for facelets.
    //
    // var domNode = document.getElementById(this.id);
    // if (domNode && domNode.getAttribute("dojoattachpoint")) {
    if (this.initialized == true) {
        return true;
    }
    return false;
};

/**
 * This is called after the buildRendering() function.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.postCreate = function () {
    this.inherited("postCreate", arguments);

    // In order to register widgets properly, the DOM node id must be set prior 
    // to creating any widget children. Otherwise, widgets may not be destroyed.
    this.domNode.id = this.id;

    // Since the anchor id and name must be the same on IE, we cannot obtain the
    // widget using the DOM node ID via the public functions below. Therefore, 
    // we need to set the widget id via closure magic.
    var _id = this.id;

    // Set public functions.
    this.domNode.getProps = function() { return webui.suntheme4_2.dijit.byId(_id).getProps(); };
    this.domNode.setProps = function(props, notify) { return webui.suntheme4_2.dijit.byId(_id).setProps(props, notify); };

    // Initialize public events and functions.
    this.initEvents();

    // Set properties.
    this._setProps(this.getProps());

    // All widget properties have been set.
    return this.initialized = true;
};

/**
 * This function is used to set common properties for the given domNode.
 *
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey Shortcut key.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.setCommonProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }
    if (props.accessKey) { 
        domNode.accessKey = props.accessKey;
    }
    if (props.dir) {
        domNode.dir = props.dir;
    }
    if (props.lang) {
        domNode.lang = props.lang;
    }
    if (props.tabIndex > -1 && props.tabIndex < 32767) {
        domNode.tabIndex = props.tabIndex;
    }
    if (props.title) {
        domNode.title = props.title;
    }
    return true;
};

/**
 * This function is used to set core properties for the given domNode. These
 * properties are typically set on the outermost element.
 *
 * Note: The className is typically provided by a web app developer. If 
 * the widget has a default className, it should be added to the DOM node
 * prior to calling this function. For example, the "myCSS" className would
 * be appended to the existing "Tblsun4" className (e.g., "Tbl_sun4 myCSS").
 *
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} style Specify style rules inline.
 * @config {boolean} visible Hide or show element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.setCoreProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }
    if (props.className) {
        domNode.className = props.className;
    }
    if (props.id) { 
        domNode.id = props.id;
    }
    if (props.style) { 
        domNode.style.cssText = props.style;
    }
    if (props.visible != null) {
        this.common.setVisibleElement(domNode, 
            new Boolean(props.visible).valueOf());
    }
    return true;
};

/**
 * This function is used to set event properties for the given domNode.
 *
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Element value changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Element text selected.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.setEventProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }

    // Note: JSON strings are not recognized as JavaScript. In order for
    // events to work properly, an anonymous function must be created.
    if (props.onBlur) { 
        domNode.onblur = (typeof props.onBlur == 'string')
            ? new Function("event", props.onBlur)
            : props.onBlur;
    }
    if (props.onClick) {
        domNode.onclick = (typeof props.onClick == 'string')
            ? new Function("event", props.onClick)
            : props.onClick;
    }
    if (props.onChange) {
        domNode.onchange = (typeof props.onChange == 'string')
            ? new Function("event", props.onChange)
            : props.onChange;
    }
    if (props.onDblClick) {
        domNode.ondblclick = (typeof props.onDblClick == 'string')
            ? new Function("event", props.onDblClick)
            : props.onDblClick;
    }
    if (props.onFocus) {
        domNode.onfocus = (typeof props.onFocus == 'string')
            ? new Function("event", props.onFocus)
            : props.onFocus;
    }
    if (props.onKeyDown) {
        domNode.onkeydown = (typeof props.onKeyDown == 'string')
            ? new Function("event", props.onKeyDown)
            : props.onKeyDown;
    }
    if (props.onKeyPress) {
        domNode.onkeypress = (typeof props.onKeyPress == 'string')
            ? new Function("event", props.onKeyPress)
            : props.onKeyPress;
    }
    if (props.onKeyUp) {
        domNode.onkeyup = (typeof props.onKeyUp == 'string')
            ? new Function("event", props.onKeyUp)
            : props.onKeyUp;
    }
    if (props.onMouseDown) {
        domNode.onmousedown = (typeof props.onMouseDown == 'string')
            ? new Function("event", props.onMouseDown)
            : props.onMouseDown;
    }
    if (props.onMouseOut) {
        domNode.onmouseout = (typeof props.onMouseOut == 'string')
            ? new Function("event", props.onMouseOut)
            : props.onMouseOut;
    }
    if (props.onMouseOver) {
        domNode.onmouseover = (typeof props.onMouseOver == 'string')
            ? new Function("event", props.onMouseOver)
            : props.onMouseOver;
    }
    if (props.onMouseUp) {
        domNode.onmouseup = (typeof props.onMouseUp == 'string')
            ? new Function("event", props.onMouseUp)
            : props.onMouseUp;
    }
    if (props.onMouseMove) {
        domNode.onmousemove = (typeof props.onMouseMove == 'string')
            ? new Function("event", props.onMouseMove)
            : props.onMouseMove;
    }
    if (props.onSelect) {
        domNode.onselect = (typeof props.onSelect == 'string')
            ? new Function("event", props.onSelect)
            : props.onSelect;
    }
    return true;
};

/**
 * This function is used to set widget properties.
 *
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 *
 * Note: If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} style Specify style rules inline.
 * @config {boolean} visible Hide or show element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Extend widget object for later updates.
    this.prototypejs.extend(this, props);

    // Set properties.
    this._setProps(props);

    // Notify listeners state has changed.
    if (new Boolean(notify).valueOf() == true &&
            typeof this.stateChanged == "function") {
        this.stateChanged(props);
    }
    return true;
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.widgetBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set style class -- must be set before calling setCoreProps().
    props.className = this.getClassName();

    // Set more properties.
    return this.setCoreProps(this.domNode, props);
};

/**
 * This function is used to "start" the widget, after the widget has been
 * instantiated.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.startup = function () {
    if (this._started) {
        return false;
    }
    this.inherited("startup", arguments);
    return this._started = true;
};

/**
 * @name webui.suntheme4_2.widget.accordionTab
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the accordionTab widget.
 * @constructor This function is used to construct an accordionTab widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.accordionTab", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.isContainer = true;
        this.selected = false;
        this.focusState = "title";
    },
    widgetName: "accordionTab" // Required for theme properties.    
});

/**
 * The callback function for key press on the accordion tabs.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.createOnKeyDownCallBack = function() {
    var _id = this.id;
    var _widget = this.widget;
    return function(event) {
        var elem = document.getElementById(_id);
        if (elem == null) {
            return false;
        }
        var widget = webui.suntheme4_2.dijit.byId(_id);

        event = _widget.getEvent(event);
        var keyCode = _widget.getKeyCode(event);
        
        // if onkeypress returns false, we do not traverse the menu.
        var keyPressResult = true;
        if (this._onkeypress) {
            keyPressResult = (this._onkeypress) ? this._onkeypress() : true;
        }        
       
        if (keyPressResult != false) {
            widget.traverseMenu(keyCode, event, _id);
        }
        return true;
   };
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.accordionTab.event =
        webui.suntheme4_2.widget.accordionTab.prototype.event = {
    /**
     * This object contains load event topics.
     * @ignore
     */
    load: {
        /** Load event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_accordionTab_event_load_begin",

        /** Load event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_accordionTab_event_load_end"
    },

    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_accordionTab_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_accordionTab_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_accordionTab_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_accordionTab_event_state_end"
    },

    /**
     * This object contains title event topics.
     * @ignore
     */
    title: {
        /** Action event topic for custom AJAX implementations to listen for. */
        selectedTopic: "webui_suntheme4_2_widget_accordionTab_event_tab_selected"
    }
};

/**
 * Process load event.
 *
 * @param {String} execute The string containing a comma separated list 
 * of client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.loadContent = function(execute) {
    // Publish event.
    this.publish(webui.suntheme4_2.widget.accordionTab.event.load.beginTopic, [{
        id: this.id
    }]);
    return true;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.accordionTab.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.selected) { props.selected = this.selected; }
    if (this.title) { props.title = this.title; }
    if (this.tabContent) { props.tabContent = this.tabContent; }
    if (this.visible != null) { props.visible = this.visible; }
    if (this.actions != null) { props.actions = this.actions; }
    if (this.className != null) { props.className = this.className; }
    if (this.style != null) { props.style = this.style; }
    if (this.contentHeight != null) { props.contentHeight = this.contentHeight; }
    if (this.id) { props.id = this.id; }
    if (this.tabContent) { props.tabContent = this.tabContent; }
    if (this.focusId) { props.focusId = this.focusId; }
    if (this.type) { props.type = this.type; }

    return props;
};

/**
 * Get title height.
 *
 * @return {int} The title height.
 */
webui.suntheme4_2.widget.accordionTab.prototype.getTitleHeight = function () {
    // Warning: This function has been made private.
    return this.dojo._getMarginBox(this.titleContainer).height;
};

/**
 * Handle menu onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.onMenuClickCallback = function(event) {
    this.dojo.stopEvent(event);
    return true;
};

/**
 * Handle title onClick event.
 * <p>
 * This function selects the child tab when the user clicks on its label. The 
 * actual behavior of the accordion depends on multipleSelect being enabled.
 * </p>
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.onTitleClickCallback = function (event) {
    this.publish(webui.suntheme4_2.widget.accordionTab.event.title.selectedTopic, [{
        id: this.id
    }]);
    if (this.titleContainer.focus) {
        this.titleContainer.focus();
    }
    return true;
};

/**
 * Handle title onMouseOut event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.onTitleMouseOutCallback = function(event) {
    if (this.selected) {
        this.titleContainer.className = this.theme.getClassName("ACCORDION_TABEXPANDED");
        this.turnerContainer.className = this.theme.getClassName("ACCORDION_DOWNTURNER");
        if (this.titleContainer.focus) {
            this.titleContainer.focus();
        }
        return true;
    }
    this.titleContainer.className = this.theme.getClassName("ACCORDION_TABCOLLAPSED");
    this.turnerContainer.className = this.theme.getClassName("ACCORDION_RIGHTTURNER");
    return true;
};

/**
 * Handle title onMouseOver event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.onTitleMouseOverCallback = function(event) {
    if (this.selected) {
        this.turnerContainer.className = this.theme.getClassName("ACCORDION_DOWNTURNER");
    } else {
        this.turnerContainer.className = this.theme.getClassName("ACCORDION_RIGHTTURNER");
    }
    if (this.titleContainer.focus) {
        this.titleContainer.focus();
    }
    return true;
};

/**
 * Handle the case when the contentNode is about to lose focus. The tabIndex for the tab 
 * and the contentNode should be set to -1 and the focusElement should also be nulled.
 * 
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.onContentEndCallback = function(event) {
    this.focusState = "end";
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.domNode.id = this.id;
        this.titleNode.id = this.id + "_tabTitle";
        this.turnerContainer.id = this.id + "_tabTitleTurner";
        this.menuContainer.id = this.id + "_tabMenu";
        this.hiddenFieldNode.id = this.id + ":selectedState";
        this.hiddenFieldNode.name = this.hiddenFieldNode.id;
        this.domNode.tabIndex = -1;
        this.contentEnd.id = this.id + "_contentEnd";
    }

    // Set style classes.
    this.titleContainer.className = this.theme.getClassName("ACCORDION_TABCOLLAPSED");
    this.turnerContainer.className = this.theme.getClassName("ACCORDION_RIGHTTURNER");
    this.menuContainer.className = this.theme.getClassName("HIDDEN");
    this.titleNode.className = this.theme.getClassName("ACCORDION_TABTITLE");
    this.contentNode.className = this.theme.getClassName("ACCORDION_TABCONTENT");

    // Set public functions.
    // TBD...

    // Set events.
    this.dojo.connect(this.titleContainer, "onclick", this, "onTitleClickCallback");
    this.dojo.connect(this.titleContainer, "onmouseover", this, "onTitleMouseOverCallback");
    this.dojo.connect(this.turnerContainer, "onmouseover", this, "onTitleMouseOverCallback");
    this.dojo.connect(this.turnerContainer, "onclick", this, "onTitleClickCallback");
    this.dojo.connect(this.menuContainer, "onmouseover", this, "onTitleMouseOverCallback");
    this.dojo.connect(this.menuContainer, "onclick", this, "onMenuClickCallback");
    this.dojo.connect(this.titleContainer, "onmouseout", this, "onTitleMouseOutCallback");
    this.dojo.connect(this.turnerContainer, "onmouseout", this, "onTitleMouseOutCallback");
    this.dojo.connect(this.menuContainer, "onmouseout", this, "onTitleMouseOutCallback");
    this.dojo.connect(this.contentEnd, "onblur", this, "onContentEndCallback");
    //Create callback function for onkeydown event.
    this.dojo.connect(this.domNode, "onkeydown", this.createOnKeyDownCallBack()); 
    
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {int} contentHeight CSS selector.
 * @config {String} hiddenField Field set to true if tab is selected.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Array} tabContent The content area of the tab.
 * @config {String} style Specify style rules inline.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.tabContent) {
        this.tabContent = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.accordionTab.prototype._setProps = function(props) {

    if (props == null) {
        return false;
    }

    if (props.tabIndex == null) {
        props.tabIndex = -1;
        this.tabIndex = -1;
    } else { //new code
        this.domNode.tabIndex = props.tabIndex;
        this.titleContainer.tabIndex = props.tabIndex;
        this.contentNode.tabIndex = props.tabIndex;
        this.tabIndex = props.tabIndex;
        this.contentEnd.tabIndex = this.tabIndex;
    }  // end of new code.

    // Set properties.
    if (props.contentHeight) {
        this.contentNode.style.height = props.contentHeight;
    }

    if (props.title) {
        this.setTitle(props.title);
    }

    if (props.tabContent) {
        this.setTabContent(props.tabContent);
        if (this.selected) {
            this.hiddenFieldNode.value = "true";
            this.titleContainer.className = this.theme.getClassName("ACCORDION_TABEXPANDED");
            this.turnerContainer.className = this.theme.getClassName("ACCORDION_DOWNTURNER");
            this.contentNode.style.display = "block";
        } else {
            this.hiddenFieldNode.value = "false";
            this.titleContainer.className = this.theme.getClassName("ACCORDION_TABCOLLAPSED");
            this.turnerContainer.className = this.theme.getClassName("ACCORDION_RIGHTTURNER");
            this.contentNode.style.display = "none";
        }
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    if (props.focusId != null) {
        this.focusId = props.focusId;
        if (this.selected) {
            if (this.focusState == "content") {
                var focusElem = document.getElementById(this.focusId);
                if (focusElem.focus) {
                    focusElem.focus();
                }
            }
        } else {
            this.focusState = "title";
        }
    }
    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * Set tab selected.
 *
 * @param {boolean} isSelected true if selected.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.setSelected = function (isSelected) {
    if (this.selected) {
        this.selected = false;
    } else {
        this.selected = isSelected;
    }

    if (this.selected) {
        this.hiddenFieldNode.value = "true";
        this.titleContainer.className = this.theme.getClassName("ACCORDION_TABEXPANDED");
        this.turnerContainer.className = this.theme.getClassName("ACCORDION_DOWNTURNER");
        this.contentNode.style.display = "block";

        // if the tab does not have content and "loadOnSelect" is set
        // to true go ahead and refresh the widget. 
        if (!this.tabContent) {
            if (this.parent.loadOnSelect) {
                this.loadContent();
            }
        }
    } else {
        this.hiddenFieldNode.value = "false";
        this.titleContainer.className = this.theme.getClassName("ACCORDION_TABCOLLAPSED");
        this.turnerContainer.className = this.theme.getClassName("ACCORDION_RIGHTTURNER");
        this.contentNode.style.display = "none";
    }
    return true;
};

/**
 * Set the contents of the accordion tab.
 *
 * @param {Array} content The Contents of the tab body.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.setTabContent = function(content) {
    if (content) {
        for (var i = 0; i < content.length; i++) {
            this.widget.addFragment(this.contentNode, content[i], "last");
        }
    }
    return true;
};

/**
 * Set the title associated with the accordion tab.
 *
 * @param {String} title Title property associated with the tab.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.setTitle = function (title) {
    
    if (title) {
        // NOTE: If you set this value manually, text must be HTML escaped.
        var titleHref = webui.suntheme4_2.widget.common.getWidgetProps("hyperlink", {
                id: this.id + "_titleLink",
                title: title,
                onClick: "return false;",
                className: "",
                contents: [title]
                
        });
        this.widget.addFragment(this.titleNode, titleHref);
    }
    return true;
};

/**
 * This function takes care of traversing through the accordionTab depending
 * on which key is pressed. If tab is open and current focus is on the title, 
 * the focus should move to the first element in the the open tab. Else it 
 * should simply move out accordion.
 * @param (String) keyCode The value of the key which was pressed
 * @param (Event) event The key press event.
 * @param (String) nodeId The id of the accordionTab. 
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordionTab.prototype.traverseMenu = function(keyCode, event, nodeId) {
    // The focus can either be on the title or the content.
    // If tab is open and current focus is on the title, the focus should move
    // to the first element in the the open tab. Else it should simply move out 
    // accordion.

    // do nothing if only shift is pressed.
    if (event.shiftKey && keyCode != 9) {  
        return true;
    }

    // handle the tab case.
    if (keyCode == 9) {         

        // shift+tab is handled by the parent accordion, simply remove focus
        // from this tab and set tabIndex to -1.
        if (event.shiftKey) {
            
            this.focusState == "title";
            if (this.contentNode.blur) {
                this.contentNode.blur();
            }
            if (this.titleContainer.blur) {
                this.titleContainer.blur();
            }
            // this.focusId = null;
            // this.domNode.tabIndex = -1;  //new
            return true;
        }

        // If tab is open and focus is on the title section
        // shift focus to the content section, else move out of the
        // tab.
        if (this.selected) {
            if (this.focusState == "title") {
                this.contentNode.tabIndex = this.tabIndex;
                this.contentEnd.tabIndex = this.tabIndex;
                this.contentNode.focus();
                if (this.focusId == null) {
                    var fChild = this.contentNode.firstChild;
                    if (fChild) {
                        if (fChild.focus) {
                            fChild.focus();
                        }
                        this.focusId = fChild;
                    }
                    return true;
                } else {
                    var focusElem = document.getElementById(this.focusId);
                    if (focusElem) {
                        if (focusElem.focus) {
                            focusElem.focus();
                        }
                    }
                }
                this.focusState = "content";
            } else if (this.focusState == "content") {
                return true;
            } else {
                this.contentNode.focus();
                if (this.focusId == null) {
                    this.contentNode.tabIndex = this.tabIndex;
                    return true;
                } else {
                    var focusElem = document.getElementById(this.focusId);
                    if (focusElem) {
                        if (focusElem.focus) {
                            focusElem.focus();
                        }
                    }
                }
                this.focusState = "content";
            }
        } else {
            // reset the tabIndex as control is moving out of the tab.
            // this.domNode.tabIndex = -1; //new
            return true;
        }
    } else if (keyCode == 38 && event.ctrlKey) {  // handle ctrl + up

        // Move focus to the header of the accordion tab if its contents is
        // currently in focus. Else, do nothing. Also, stop event propagation.

        if (this.focusState == "content") {
            this.focusState == "title";
            if (this.contentNode.blur) {
                this.contentNode.blur();
            }
            if (this.titleContainer.focus) {
                this.titleContainer.focus();
            }
            if (webui.suntheme4_2.browser.isIe5up()) {
                window. event.cancelBubble = true;
                window.event.returnValue = false;
            } else {
                event.stopPropagation();
                event.preventDefault();
            }
        }
    }
    return true;
};
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.accordion
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the accordion widget.
 * @constructor This function is used to construct an accordion widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.accordion", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.duration = 250;
        this.isContainer = true;
        this.loadOnSelect = false;
        this.multipleSelect = false;
        this.focusElement = null;
    },
    widgetName: "accordion" // Required for theme properties.
});

/**
 * Helper function to add accordion header controls
 *
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.addControls = function(props) {       
    // Add expand and collapse icons only if multiple select is set to
    // true and the icons have been supplied.
    if (props.toggleControls && props.multipleSelect) {
        // Set expand all image properties.
        if (props.expandAllImage) {
            // Update/add fragment.
            this.widget.updateFragment(this.expandAllImgContainer, 
                this.expandAllImage.id, props.expandAllImage);
        }

        // Set collapse all image properties.
        if (props.collapseAllImage) {
            // Update/add fragment.
            this.widget.updateFragment(this.collapseAllImgContainer,
                this.collapseAllImage.id, props.collapseAllImage);
        }
        
        // a divider should only be added if expand/collapse icons exist.
        this.dividerNodeContainer.className = this.theme.getClassName("ACCORDION_HDR_DIVIDER");
    }

    // Set refresh image properties.
    if (props.isRefreshIcon && props.refreshImage) {
        // Update/add fragment.
        this.widget.updateFragment(this.refreshImgContainer,
            this.refreshImage.id, props.refreshImage);
    }
    return true;
};

/**
 * Close all open accordions and leave the others as is.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.collapseAllTabs = function(event) {
    // Iterate over all tabs.
    for (var i = 0; i < this.tabs.length; i++) {
        var widget = webui.suntheme4_2.dijit.byId(this.tabs[i].id);
        if (widget && widget.selected) {
            widget.setSelected(false);
        }
    }
    this.updateFocus(this.collapseAllContainer);
    return true;
};

/**
 * The callback function for key press on the accordion header.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.createOnKeyDownCallBack = function() {
    var _id = this.id;
    var _widget = this.widget;
    return function(event) {
        var elem = document.getElementById(_id);
        if (elem == null) {
            return false;
        }
        var widget = webui.suntheme4_2.dijit.byId(_id);

        event = _widget.getEvent(event);
        var keyCode = _widget.getKeyCode(event);
        
        // if onkeypress returns false, we do not traverse the accordion
        var keyPressResult = true;
        if (this._onkeypress) {
            keyPressResult = (this._onkeypress) ? this._onkeypress() : true;
        }        
       
        if (keyPressResult != false) {
            widget.traverseMenu(keyCode, event, _id);
        }
        return true;
   };
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.accordion.event = 
        webui.suntheme4_2.widget.accordion.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_accordion_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_accordion_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_accordion_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_accordion_event_state_end"
    }
};

/**
 * Open all closed tabs and leave the others as is.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.expandAllTabs = function(event) {
    // Iterate over all tabs.
    for (var i = 0; i < this.tabs.length; i++) {
        var widget = webui.suntheme4_2.dijit.byId(this.tabs[i].id);
        if (widget && !widget.selected) {
            widget.setSelected(true);
        }
    }
    this.updateFocus(this.expandAllContainer);
    return true;
};

/**
 * Set the appropriate focus before invoking tabSelected. This
 * function has been put in place because tabSelected is called
 * from various places but the focus does not have to be set
 * in all the cases.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.focusAndSelectTab = function(props) {
    
    // Iterate over all tabs to ensure id is valid.
    for (var i = 0; i < this.tabs.length; i++) {
        if (props.id == this.tabs[i].id) {
            this.focusElement = this.tabs[i];
            break;   
        }
    }
    this.tabSelected(props);
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.accordion.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.collapseAllImage != null) { props.collapseAllImage = this.collapseAllImage; }
    if (this.expandAllImage != null) { props.expandAllImage = this.expandAllImage; }
    if (this.isRefreshIcon != null) { props.isRefreshIcon = this.isRefreshIcon; }
    if (this.loadOnSelect) { props.loadOnSelect = this.loadOnSelect; }
    if (this.multipleSelect) { props.multipleSelect = this.multipleSelect; }
    if (this.refreshImage != null) { props.refreshImage = this.refreshImage; }
    if (this.style != null) { props.style = this.style; }
    if (this.tabs != null) { props.tabs = this.tabs; }
    if (this.toggleControls) { props.toggleControls = this.toggleControls; }
    if (this.type) { props.type = this.type; }
 
    return props;
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.accordion.prototype.getClassName = function() {
    // Get theme property.
    var className = this.theme.getClassName("ACCORDION_DIV", "");
    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.domNode.id = this.id;
        this.headerContainer.id = this.id + "_accHeader";
        this.expandAllContainer.id = this.id + "_expandAllNode";
        this.expandAllImgContainer.id = this.expandAllContainer.id + "_expandAllImage";
        this.collapseAllContainer.id = this.id + "_collapseAllNode";
        this.collapseAllImgContainer.id = this.collapseAllImgContainer.id + "_collapseAllImage";
        this.dividerNodeContainer.id = this.id + "_dividerNode";
        this.refreshNodeContainer.id = this.id + "_refreshNode";
    }

    // Set class names.
    this.headerContainer.className = this.theme.getClassName("ACCORDION_HDR");
    this.collapseAllContainer.className = this.theme.getClassName("ACCORDION_HDR_CLOSEALL");
    this.expandAllContainer.className = this.theme.getClassName("ACCORDION_HDR_OPENALL");
    
    // the divider should initially be hidden
    this.dividerNodeContainer.className = this.theme.getClassName("HIDDEN");
    this.refreshNodeContainer.className = this.theme.getClassName("ACCORDION_HDR_REFRESH");

    // Set events.
    var _id = this.id;
    this.dojo.connect(this.collapseAllContainer, "onclick", this, "collapseAllTabs");
    this.dojo.connect(this.expandAllContainer, "onclick", this, "expandAllTabs");
    this.dojo.connect(this.refreshNodeContainer, "onclick", function(event) {
        // New literals are created every time this function is called, and it's 
        // saved by closure magic.
        var widget = webui.suntheme4_2.dijit.byId(_id);
        widget.updateFocus(this.refreshNodeContainer);
        widget.refresh(_id);
        return false;
    });
    
    // Create callback function for onkeydown event.
    this.dojo.connect(this.domNode, "onkeydown", this.createOnKeyDownCallBack());
    
    // Subscribe to the "tabSelected" event present in the accordion widget.
    this.subscribe(webui.suntheme4_2.widget.accordionTab.event.title.selectedTopic,
        this, "focusAndSelectTab");

    // Generate the accordion header icons on the client side.
    if (this.toggleControls && this.multipleSelect) {
        if (this.expandAllImage == null) {
            this.expandAllImage = this.widget.getWidgetProps("imageHyperlink", {
                    id: this.id + "_expandImageLink",
                    onClick: "return false;",
                    enabledImage: this.widget.getWidgetProps("image", {
                        icon: "ACCORDION_EXPAND_ALL",
                        id: this.id + "_expandAll"
                    }),
                    title: this.theme.getMessage("Accordion.expandAll")            
                });
        }
        
        if (this.collapseAllImage == null) {
            this.collapseAllImage = this.widget.getWidgetProps("imageHyperlink", {
                    id: this.id + "_collapseImageLink",
                    onClick: "return false;",
                    enabledImage: this.widget.getWidgetProps("image", {
                        icon: "ACCORDION_COLLAPSE_ALL",
                        id: this.id + "_collapseAll"
                    }),
                    title: this.theme.getMessage("Accordion.collapseAll")            
                });
        }
    }

    // Set refresh image hyperlink properties.
    if (this.isRefreshIcon) {
        if (this.refreshImage == null) {
            this.refreshImage = this.widget.getWidgetProps("imageHyperlink", {
                    id: this.id + "_refreshImageLink",
                    onClick: "return false;",
                    enabledImage: this.widget.getWidgetProps("image", {
                        icon: "ACCORDION_REFRESH",
                        id: this.id + "_refresh"
                    }),
                    title: this.theme.getMessage("Accordion.refresh")            
                });
         }
    }
    
    if (this.isRefreshIcon) {
        this.refreshNodeContainer.tabIndex = this.tabIndex;
    }
    if (this.toggleControls && this.multipleSelect) {
            this.expandAllContainer.tabIndex = this.tabIndex;
            this.collapseAllContainer.tabIndex = this.tabIndex;
    }
    if (this.tabs.length > 0) {
        for (var i=0; i< this.tabs.length; i++) {
            this.tabs[i].tabIndex = this.tabIndex;
        }
    }

    with (this.domNode.style) {
        if (position != "absolute") {
            position = "relative";
        }
        overflow = "hidden";
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {Object} collapseAllImage Image associated with collapse all icon
 * @config {Object} expandAllImage Image associated with expand all icon
 * @config {String} id Uniquely identifies an element within a document.
 * @config {boolean} isRefreshIcon True if refresh icon should be set
 * @config {boolean} multipleSelect Set to true if multiple tabs can be selected
 * @config {Object} refreshImage Image associated with refresh icon.
 * @config {String} style Specify style rules inline.
 * @config {Array} tabs Tabs constituting the accordion's children
 * @config {boolean} toggleControls Set to true if expand/collapse icons should be set.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace tabs -- do not extend.
    if (props.tabs) {
        this.tabs = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.accordion.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    if (props.tabIndex == null) {
        props.tabIndex = -1;
        this.tabIndex = -1;
    }

    // add control icons - refresh, expandall, collapseall.
    this.addControls(props); 

    // If we are coming here for
    // the first time there will be no children. The other case is when 
    // the accordion is being rerendered because of a refresh in which 
    // we want to use the latest set of children. addFragment is supposed
    // to do that.
    if (props.tabs) {
        // Remove child nodes.
        this.widget.removeChildNodes(this.tabsContainer);

        // set the tab focus 
        var tabFocus = true;
        
        if (props.tabs.length > 0) {
            for (var i=0; i< props.tabs.length; i++) {
                props.tabs[i].tabIndex = this.tabIndex;
            }
        }
       
        // Add tabs.
        for (var i = 0; i < props.tabs.length; i++) {
            this.widget.addFragment(this.tabsContainer, props.tabs[i], "last");
        }
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * Set a different look for refresh/expand/collapse icons when focus is set
 * on them.
 *
 * @param {String} nodeId The node ID of the icon.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.setFocusStyleClass = function(nodeId) {
    if (nodeId == this.collapseAllContainer.id) {
        //set focus style on collapseNode
        this.collapseAllContainer.className = this.theme.getClassName("ACCORDION_HDR_CLOSEALL_FOCUS");
    } else if (nodeId == this.expandAllContainer.id) {
        //set focus style on expandNode
        this.expandAllContainer.className = this.theme.getClassName("ACCORDION_HDR_OPENALL_FOCUS");
    } else if (nodeId == this.refreshNodeContainer.id) {
        //set focus style on refreshNode
        this.refreshNodeContainer.className = this.theme.getClassName("ACCORDION_HDR_REFRESH_FOCUS");
    }
    return true;
};

/**
 * Reset the styles asscociated with refresh/expand/collapse icons when these 
 * icons are blurred.
 *
 * @param {String} nodeId The node ID of the icon.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.setBlurStyleClass = function(nodeId) {
    if (nodeId == this.collapseAllContainer.id) {
        //set normal className on collapseNode
        this.collapseAllContainer.className = this.theme.getClassName("ACCORDION_HDR_CLOSEALL");
    } else if (nodeId == this.expandAllContainer.id) {
        //set normal className on expandNode
        this.expandAllContainer.className = this.theme.getClassName("ACCORDION_HDR_OPENALL");
    } else if (nodeId == this.refreshNodeContainer.id) {
        //set normal className on refreshNode
        this.refreshNodeContainer.className = this.theme.getClassName("ACCORDION_HDR_REFRESH");
    }
    return true;
};

/**
 * Set appropriate styles when a tab is in focus.
 *
 * @param {String} nodeId The node ID of the icon.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.setTabFocus = function(nodeId) {
    // update the tab with the appropriate tabIndex
    var tabWidget = webui.suntheme4_2.dijit.byId(nodeId);
    var props = {tabIndex: this.tabIndex};
    this.widget.updateFragment(this.tabsContainer, nodeId, props);

    // set the style class to indicate that the tab is in focus.
    if (tabWidget.selected) {
        tabWidget.titleContainer.className = this.theme.getClassName("ACCORDION_TABEXPANDED_FOCUS");
    } else {
        tabWidget.titleContainer.className = this.theme.getClassName("ACCORDION_TABCOLLAPSED_FOCUS");
    }
    tabWidget.domNode.focus();
    return true;
};

/**
 * Set appropriate styles when a tab loses focus.
 *
 * @param {String} nodeId The node ID of the icon.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.setTabBlur = function(nodeId) {
    // update the tab with the appropriate tabIndex
    var tabWidget = webui.suntheme4_2.dijit.byId(nodeId);
    
    if (tabWidget.selected) {
        tabWidget.titleContainer.className = this.theme.getClassName("ACCORDION_TABEXPANDED");
        tabWidget.turnerContainer.className = this.theme.getClassName("ACCORDION_DOWNTURNER");
    } else { 
        tabWidget.titleContainer.className = this.theme.getClassName("ACCORDION_TABCOLLAPSED");
        tabWidget.turnerContainer.className = this.theme.getClassName("ACCORDION_RIGHTTURNER");
    }    

    if (tabWidget) {
        tabWidget.titleContainer.blur();
    }
    return true;
};

/**
 * Process tab selected events.
 *
 * @param props Key-Value pairs of properties.
 * @config {String} id The id of the selected tab.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.tabSelected = function(props) {
    var widget = null;

    // Iterate over all tabs to ensure id is valid.
    for (var i = 0; i < this.tabs.length; i++) {
        if (props.id == this.tabs[i].id) {
            widget = webui.suntheme4_2.dijit.byId(this.tabs[i].id);
            break;   
        }
    }
    
    // Return if id was not valid.
    if (widget == null) {
            return false;
    }
    
    if (this.multipleSelect) {
        widget.setSelected(true);
    } else {
        for (var i = 0; i < this.tabs.length; i++) {
            widget = webui.suntheme4_2.dijit.byId(this.tabs[i].id);
            if (widget) {
                widget.setSelected(props.id == this.tabs[i].id);
                this.tabs[i] = widget.getProps();
            }
        }
    }
    return true;
};

/**
 * This function traverses through the accordion depending
 * on which key is pressed. If the down or right arrow key is pressed the
 * next focusable element of the accordion is focused on. If the last element 
 * was on focus this changes to the first element. If the enter or space key
 * was pressed the onclick function associated with the focused element is 
 * activated. 
 * @param (String) keyCode The valye of the key which was pressed
 * @param (Event) event The key press event.
 * @param (String) nodeId The id of the accordion item. 
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.traverseMenu = function(keyCode, event, nodeId) {
    var savedFocusElement;

    if (this.focusElement != null) {
        savedFocusElement = this.focusElement;
    }

    // Operations to be performed if the arrow keys are pressed.
    // down or right arrow key ==> forward
    // up or left arrow key ==> backward
    // During forward traversal control should go from the refresh icon
    // to the collapseAll icon to the expandAll icon to the tabs from
    // top to bottom. If control is on the last tab, clicking a forward
    // key will cause control to come back to the first tab and not the
    // accordion control icons. A similar but opposite behavior is 
    // exhibited when the up or left keys are pressed. Once control is
    // on the first tab backward traversal will take control to the last 
    // tab in the accordion.

    if (keyCode >= 37 && keyCode <= 40) {
        var forward = true;
        if (keyCode == 37 || keyCode == 38) {
            forward = false;
        }
        var focusSet = false;
        if (forward) {  // handling the down and right arrow keys
            if (this.isRefreshIcon) {
                if (this.focusElement.id == this.refreshNodeContainer.id) {
                    if (this.toggleControls && this.multipleSelect) {
                        this.updateFocus(this.collapseAllContainer);
                        focusSet = true;
                    } else {
                        this.updateFocus(this.tabs[0]);
                        focusSet = true;
                    }
                }
            }
            
            if (!focusSet && this.toggleControls && this.multipleSelect) {
                
                if (this.focusElement.id == this.collapseAllContainer.id) {
                    this.updateFocus(this.expandAllContainer);
                    focusSet = true;

                } else if (this.focusElement.id == this.expandAllContainer.id) {
                    this.updateFocus(this.tabs[0]);
                    focusSet = true;
                
                } else {
                    for (var i = 0; i < this.tabs.length; i++) {
                        if (this.tabs[i].id == this.focusElement.id) {
                            var newIndex = (i + 1) % this.tabs.length;
                            this.updateFocus(this.tabs[newIndex]);
                            focusSet = true;
                            break;
                        }
                    }
                }
            }
            if (!focusSet) {
                for (var i = 0; i < this.tabs.length; i++) {
                    if (this.tabs[i].id == this.focusElement.id) {
                        var newIndex = (i + 1) % this.tabs.length;
                        this.updateFocus(this.tabs[newIndex]);
                        focusSet = true;
                        break;
                    }
                }
                if (this.focusElement == null) {
                    this.updateFocus(this.tabs[0]);
                }
            }

         } else {  // traverse backward

            if (this.isRefreshIcon) {
                if (this.focusElement.id == this.refreshNodeContainer.id) {
                    var index = this.tabs.length;
                    this.updateFocus(this.tabs[index-1]);
                    focusSet = true;
                }
            }
            
            if (!focusSet && this.toggleControls && this.multipleSelect) {
                if (this.focusElement.id == this.collapseAllContainer.id) {
                    if (this.isRefreshIcon) {
                        this.updateFocus(this.refreshNodeContainer);
                        focusSet = true;
                    } else {
                        var index = this.tabs.length;
                        focusSet = true;
                        this.updateFocus(this.tabs[index-1]);
                    }
 
                } else if (this.focusElement.id == this.expandAllContainer.id) {
                    this.updateFocus(this.collapseAllContainer);
                    focusSet = true;
                }
            }
            if (!focusSet) {
                for (var i = 0; i < this.tabs.length; i++) {
                    if (this.tabs[i].id == this.focusElement.id) {
                        if (i == 0) {
                            var index = this.tabs.length;
                            this.updateFocus(this.tabs[index-1]);
                            focusSet = true;
                            break;
                            
                        } else {
                            this.updateFocus(this.tabs[i-1]);
                            focusSet = true;
                        }
                        break;
                    }
                }
                if (this.focusElement == null) { //focus on the last tab
                    var index = this.tabs.length;
                    this.updateFocus(this.tabs[index - 1]);
                    focusSet = true;
                }
            }
        }

        if (webui.suntheme4_2.browser.isIe5up()) {
            window.event.cancelBubble = true;
            window.event.returnValue = false;
        } else {
            event.stopPropagation();
            event.preventDefault();
        }
    } else if(keyCode == 13 || keyCode == 32) {  // handle enter and space bar
        var actionComplete = false;
        if (this.isRefreshIcon) {
            if (this.focusElement.id == this.refreshNodeContainer.id) {
                var accWidget = webui.suntheme4_2.dijit.byId(nodeId);
                accWidget.refresh(nodeId);
                actionComplete = true;
            }    
        }
        if (!actionComplete && this.toggleControls && this.multipleSelect) {
            if (this.focusElement.id == this.collapseAllContainer.id) {
                this.collapseAllTabs();
                actionComplete = true;
                
            } else if (this.focusElement.id == this.expandAllContainer.id) {
                this.expandAllTabs();
                actionComplete = true;
            }
        } 
        if (!actionComplete) { // has to be an accordion tab
            if (this.focusElement) {
                var props;
                if (this.focusElement.selected) {
                    var widget = webui.suntheme4_2.dijit.byId(this.focusElement.id);
                    widget.setSelected(false);
                } else {
                    var widget = webui.suntheme4_2.dijit.byId(this.focusElement.id);
                    widget.setSelected(true);
                }
             
            }
        }

        if (webui.suntheme4_2.browser.isIe5up()) {
            window.event.cancelBubble = true;
            window.event.returnValue = false;
        } else {
            event.stopPropagation();
            event.preventDefault();
        }
    } else if (keyCode == 9) {  // handle tabbing
        
        // Tabbing order:  refresh -> expand/collapse -> first tab header
        // -> first tabable element in tab content (if tab is open)
        // -> tab out of the accordion completely. Reverse tabbing order
        // is also supported.
        
        var forward = true;
        if (event.shiftKey) {
            forward = false;
        }
        if (this.focusElement) {
            var focusSet = false;
            if (this.isRefreshIcon) {
                if (this.focusElement.id == this.refreshNodeContainer.id) {
                    if (forward) {
                        if (this.toggleControls && this.multipleSelect) {
                            this.focusElement = this.collapseAllContainer;
                            this.setFocusStyleClass(this.collapseAllContainer.id);
                            this.setBlurStyleClass(this.refreshNodeContainer.id);
                            focusSet = true;
                            return true;
                        } else {
                            this.focusElement = this.tabs[0];
                            this.setBlurStyleClass(this.refreshNodeContainer.id);
                            this.setTabFocus(this.focusElement.id);
                            focusSet = true;
                            return true;
                        }
                    } else {
                        this.focusElement = null;
                        this.setBlurStyleClass(this.refreshNodeContainer.id);
                        focusSet = true;
                        return true;
                    }
                }
            }

            if (!focusSet && this.toggleControls && this.multipleSelect) {

                if (this.focusElement.id == this.collapseAllContainer.id) {
                    if (forward) {
                        this.focusElement = this.expandAllContainer;
                        this.setBlurStyleClass(this.collapseAllContainer.id);
                        this.setFocusStyleClass(this.expandAllContainer.id);
                        focusSet = true;
                    } else {
                        this.focusElement = this.refreshNodeContainer;
                        this.setFocusStyleClass(this.refreshNodeContainer.id);
                        this.setBlurStyleClass(this.collapseAllContainer.id);
                        focusSet = true;
                    }
                } else if (this.focusElement.id == this.expandAllContainer.id) {
                    if (forward) {
                        this.focusElement = this.tabs[0];
                        this.setTabFocus(this.focusElement.id);
                        this.setBlurStyleClass(this.expandAllContainer.id);
                        focusSet = true;
                    } else {
                        this.focusElement = this.collapseAllContainer;
                        this.setFocusStyleClass(this.collapseAllContainer.id);
                        this.setBlurStyleClass(this.expandAllContainer.id);
                        focusSet = true;
                    }
                } 
            }

            if (!focusSet) { // focus is on a tab
                if (forward) {  
                    var widget = webui.suntheme4_2.dijit.byId(this.focusElement.id);
                    if ((widget.getProps().selected == false) ||
                        (widget.focusState == "end")) {
                        
                        // simply move out of the accordion if tab is closed
                        // or tab is open but user has navigated to the end of
                        // the content section.
                        
                        for (var i = 0; i < this.tabs.length; i++) {
                            if (this.tabs[i].id == this.focusElement.id) {
                                var newIndex = (i + 1);
                                if (newIndex == this.tabs.length) {
                                    this.updateFocus(null);
                                    return true;
                                }
                                this.updateFocus(this.tabs[newIndex]);
                                focusSet = true;
                                return true;
                            }
                        }

                    }
                    focusSet = true;
                    
                } else {    // move to the expand/collapse/refresh icons
                    
                    this.setTabBlur(this.focusElement.id);
                    if (this.toggleControls && this.multipleSelect) {
                        this.focusElement = this.expandAllContainer;
                        this.setFocusStyleClass(this.expandAllContainer.id);
                    } else { 
                        this.focusElement = this.refreshNodeContainer;
                        this.setFocusStyleClass(this.refreshNodeContainer.id);
                    }
                    focusSet = true;
                    return true;
                }
            }

        } else { //get focus to the first element
            var focusSet = false;
            if (this.isRefreshIcon) {
               this.focusElement = this.refreshNodeContainer;
               this.setFocusStyleClass(this.refreshNodeContainer.id);
               focusSet = true;
            }
            if (!focusSet && this.toggleControls && this.multipleSelect) {
                this.focusElement = this.collapseAllContainer;
                this.setFocusStyleClass(this.collapseAllContainer.id);
                focusSet = true;
            } 
            if (!focusSet) {
                this.focusElement = this.tabs[0];
                this.setTabFocus(this.focusElement.id);
            }
            
        } 
    } 
    return true;
};

/**
 * This function updates the focused node within the accordion.
 * The old focus element is blurred and the new focus element is
 * set to focus. Appropriate style selectors are applied. 
 * @param (String) newFocusNode The new focus node.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.accordion.prototype.updateFocus = function(newFocusNode) {

    if (this.focusElement) {
        if (this.focusElement.id == this.refreshNodeContainer.id) {
            this.setBlurStyleClass(this.refreshNodeContainer.id);
        } else if (this.focusElement.id == this.collapseAllContainer.id) {
            this.setBlurStyleClass(this.collapseAllContainer.id);
        } else if (this.focusElement.id == this.expandAllContainer.id) {
            this.setBlurStyleClass(this.expandAllContainer.id);
        } else {
            for (var i = 0; i < this.tabs.length; i++) {
                if (this.tabs[i].id == this.focusElement.id) {
                    this.setTabBlur(this.tabs[i].id);
                    break;
                }
            }
        }
    }
    
    // set the new focusElement and then the associate syles etc.
    if (newFocusNode) {
        this.focusElement = newFocusNode;
        if (this.focusElement.id == this.refreshNodeContainer.id) {
                this.setFocusStyleClass(this.refreshNodeContainer.id);
        } else if (this.focusElement.id == this.collapseAllContainer.id) {
                this.setFocusStyleClass(this.collapseAllContainer.id);
        } else if (this.focusElement.id == this.expandAllContainer.id) {
                this.setFocusStyleClass(this.expandAllContainer.id);
        } else {
            for (var i = 0; i < this.tabs.length; i++) {
                if (this.tabs[i].id == this.focusElement.id) {
                    this.setTabFocus(this.tabs[i].id);
                    break;
                }
            }
        }
    }
    return true;
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.alarm");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.alarm
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the alarm widget.
 * @constructor This function is used to construct an alarm widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.alarm", webui.suntheme4_2.widget.widgetBase, {
    widgetName: "alarm" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.alarm.event =
        webui.suntheme4_2.widget.alarm.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_alarm_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_alarm_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_alarm_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_alarm_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.alarm.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.text != null) { props.text = this.text; }
    if (this.indicators != null) { props.indicators = this.indicators; }
    if (this.textPosition != null) { props.textPosition = this.textPosition; }
    if (this.type != null) { props.type = this.type; }
    
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.alarm.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.rightText.id = this.id + "_rightText";
        this.leftText.id = this.id + "_leftText";
        this.imageContainer.id = this.id + "_imageContainer";        
    }
    // default set of indicators
        
       var  defaultIndicators =[ {
                "type": "down",
                "image": this.widget.getWidgetProps("image", {
                id: this.id + "_down",
                icon: "DOWN_ALARM_INDICATOR"
                })
            },
        {
                "type": "critical",
                "image": this.widget.getWidgetProps("image", {
                id: this.id + "_critical",
                icon: "CRITICAL_ALARM_INDICATOR"
            })
            },
        {
                "type": "major",
                "image": this.widget.getWidgetProps("image", {
                id: this.id + "_major",
                icon: "MAJOR_ALARM_INDICATOR"
            })
            },
        {
                "type": "minor",
                "image": this.widget.getWidgetProps("image", {
                id: this.id + "_minor",
                icon: "MINOR_ALARM_INDICATOR"
            })
            },
        {
                "type": "ok",
                "image": this.widget.getWidgetProps("image", {
                id: this.id + "_ok",
                icon: "OK_ALARM_INDICATOR"
            })
        }];
        
    if (this.indicators == null) {
        this.indicators = defaultIndicators;    
    } else {
      for (var i = 0; i < this.indicators.length; i++) {          
          for (var j = 0; j < defaultIndicators.length; j++) {
              if (this.indicators[i].type == defaultIndicators[j].type) {
                  defaultIndicators[j].image = this.indicators[i].image;
                  this.indicators[i] = null;                  
                  break;
              }
          }          
      }
      
      // merge the indicators (defaultset + custom set)
      for (var i = 0; i < this.indicators.length; i++) {   
          if (this.indicators[i] != null) {                         
                defaultIndicators = defaultIndicators.concat(this.indicators[i]);
          }      
      }
      this.indicators = defaultIndicators;     
    }
    
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Array} indicators 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} style Specify style rules inline.
 * @config {String} text 
 * @config {String} textPosition
 * @config {String} title Provides a title for element.
 * @config {String} type Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.alarm.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.alarm.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.dir) { this.domNode.dir = props.dir; }
    if (props.lang) { this.domNode.lang = props.lang; }    
    
    // Set right text.
    if (props.textPosition == "right" || props.textPosition == null && props.text != null) {
        this.common.setVisibleElement(this.leftText, false);
        this.widget.addFragment(this.rightText, props.text);
    }

    // Set left text.
    if (props.textPosition == "left" && props.text != null) {
        this.common.setVisibleElement(this.rightText, false);
        this.widget.addFragment(this.leftText, props.text);
    }    
    
    // Set indicator properties.
    if (props.indicators || props.type != null && this.indicators) {
        // Iterate over each indicator.
        for (var i = 0; i < this.indicators.length; i++) {
            // Ensure property exists so we can call setProps just once.
            var indicator = this.indicators[i]; // get current indicator.
            if (indicator == null) {
                indicator = {}; // Avoid updating all props using "this" keyword.
            }

            // Set properties.
            indicator.image.visible = (indicator.type == this.type) ? true: false;

            // Update/add fragment.
            this.widget.updateFragment(this.imageContainer, indicator.image.id, 
                indicator.image, "last");
        }
    }

    // Do not call setCommonProps() here. 

    // Set more properties.
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.alert");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.alert
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the alert widget.
 * @constructor This function is used to construct an alert widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.alert", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    widgetName: "alert" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.alert.event =
        webui.suntheme4_2.widget.alert.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_alert_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_alert_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_alert_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_alert_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.alert.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.detail != null) { props.detail = this.detail; }
    if (this.indicators != null) { props.indicators = this.indicators; }
    if (this.summary != null) { props.summary = this.summary; }
    if (this.type != null) { props.type = this.type; }
    if (this.moreInfo != null) { props.moreInfo = this.moreInfo; }
    if (this.spacerImage != null) { props.spacerImage = this.spacerImage; }    
    return props;
};

/**
 * This function is used to process notification events with Object
 * literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} detail Message detail text.
 * @config {String} summary Message summary text.
 * @config {boolean} valid Flag indicating validation state.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.alert.prototype.notify = function(props) {
    if (props == null) {
        return false;
    }
    return this.setProps({
        summary: props.summary,
        detail: props.detail,
        type: "error",
        visible: !props.valid
    });
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.alert.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.bottomLeftContainer.id = this.id + "_bottomLeftContainer";
        this.bottomMiddleContainer.id = this.id + "_bottomMiddleContainer";
        this.bottomRightContainer.id = this.id + "_bottomRightContainer";
        this.detailContainer.id = this.id + "_detailContainer";
        this.imageContainer.id = this.id + "_imageContainer";
        this.leftMiddleContainer.id = this.id + "_leftMiddleContainer";
        this.rightMiddleContainer.id = this.id + "_rightMiddleContainer";
        this.summaryContainer.id = this.id + "_summaryContainer";
        this.topLeftContainer.id = this.id + "_topLeftContainer";
        this.topMiddleContainer.id = this.id + "_topMiddleContainer";
        this.topRightContainer.id = this.id + "_topRightContainer";
        this.detailContainerLink.id = this.id + "_detailContainerLink";
    }
        //create default indicators.   
        var  defaultIndicators =[{
                    "type": "error",
                    "image": this.widget.getWidgetProps("image", {
                    id: this.id + "_error",
                    icon: "ERROR_ALERT_INDICATOR"
                    })
                },
            {
                    "type": "warning",
                    "image": this.widget.getWidgetProps("image", {
                    id: this.id + "_warning",
                    icon: "WARNING_ALERT_INDICATOR"
                })
                },
            {
                    "type": "success",
                    "image": this.widget.getWidgetProps("image", {
                    id: this.id + "_success",
                    icon: "SUCCESS_ALERT_INDICATOR"
                })
                },
            {
                    "type": "information",
                    "image": this.widget.getWidgetProps("image", {
                    id: this.id + "_info",
                    icon: "INFORMATION_ALERT_INDICATOR"
                })
            }];           
    
        
    if (this.indicators == null) {
        this.indicators = defaultIndicators;    
    } else {
      for (var i = 0; i < this.indicators.length; i++) {          
          for (var j = 0; j < defaultIndicators.length; j++) {
              if (this.indicators[i].type == defaultIndicators[j].type) {
                  defaultIndicators[j].image = this.indicators[i].image;
                  this.indicators[i] = null;                  
                  break;
              }
          }          
      }
      
      // merge the indicators (defaultset + custom set)
      for (var i = 0; i < this.indicators.length; i++) {   
          if (this.indicators[i] != null) {                         
                defaultIndicators = defaultIndicators.concat(this.indicators[i]);
          }      
      }
      this.indicators = defaultIndicators;     
    }
    
    //spacer image
    if (this.spacerImage == null) {
        this.spacerImage = this.widget.getWidgetProps("image", {
             icon: "DOT",
             id: this.id + "_dot"
        }); 
    }
    // moreInfo link
    if (this.moreInfo != null && this.moreInfo.id == null && this.moreInfo.widgetType == null) {
        this.moreInfo = this.widget.getWidgetProps("imageHyperlink",{
                    id: this.id + "_" + "alertLink",
                    enabledImage: this.widget.getWidgetProps("image", {
                            id: this.id + "_moreInfoLinkImg",
                            icon: "HREF_LINK"
                    }),     
                    target: this.moreInfo.target,
                    href: this.moreInfo.url,
                    contents: [this.moreInfo.value],
                    imagePosition: "right",                   
                    title: this.moreInfo.tooltip    
                }                            
            );
    }    
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} detail
 * @config {String} dir Specifies the directionality of text.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {Array} indicators 
 * @config {String} moreInfo 
 * @config {String} spacerImage 
 * @config {String} summary 
 * @config {String} type 
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.alert.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.alert.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.dir) { this.domNode.dir = props.dir; }
    if (props.lang) { this.domNode.lang = props.lang; }    
    
    // Set summary.
    if (props.summary) {
        this.widget.addFragment(this.summaryContainer, props.summary);
    }

    // Set detail.
    if (props.detail) {
        this.widget.addFragment(this.detailContainer, props.detail);
    }

    // Set moreInfo.
    if (props.moreInfo) {
        this.widget.addFragment(this.detailContainerLink, props.moreInfo);
    }

    // Set spacer image.
    if (props.spacerImage) {
        var containers = [
            this.bottomLeftContainer,
            this.bottomMiddleContainer,
            this.bottomRightContainer,
            this.leftMiddleContainer,
            this.rightMiddleContainer,
            this.topLeftContainer,
            this.topMiddleContainer,
            this.topRightContainer];

        // Avoid widget ID collisions.
        for (var i = 0; i < containers.length; i++) {
            if (typeof props != 'string') {
                props.spacerImage.id = this.id + "_spacerImage" + i;
            }
            // Replace container with image.
            if (!webui.suntheme4_2.dijit.byId(props.spacerImage.id)) {
                this.widget.addFragment(containers[i], props.spacerImage);
            }
        }
    }

    // Set indicator properties.
    if (props.indicators || props.type != null && this.indicators) {
        // Iterate over each indicator.
        for (var i = 0; i < this.indicators.length; i++) {
            // Ensure property exists so we can call setProps just once.
            var indicator = this.indicators[i]; // get current indicator.
            if (indicator == null) {
                indicator = {}; // Avoid updating all props using "this" keyword.
            }

            // Set properties.
            indicator.image.visible = (indicator.type == this.type) ? true: false;
            indicator.image.tabIndex = this.tabIndex;

            // Update/add fragment.
            this.widget.updateFragment(this.imageContainer, indicator.image.id, 
                indicator.image, "last");
        }
    }

    // Do not call setCommonProps() here. 

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.anchor");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.anchorBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.anchorBase
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for widgets that extend anchorBase.
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.anchorBase", webui.suntheme4_2.widget.widgetBase);

/**
 * Helper function to add children.
 *
 * @param props Key-Value pairs of properties.
 * @config {Array} contents The contents of the anchor body.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.anchorBase.prototype.addContents = function(props) {
    if (props.contents == null) {
        return false;
    }

    // Remove child nodes.
    this.widget.removeChildNodes(this.domNode);

    // Add contents.
    for (i = 0; i < props.contents.length; i++) {
        this.widget.addFragment(this.domNode, props.contents[i], "last");
    }
    return true;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.anchorBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.hrefLang) { props.hrefLang = this.hrefLang; }
    if (this.target) { props.target = this.target; }
    if (this.type) { props.type = this.type; }
    if (this.rev) { props.rev = this.rev; }
    if (this.rel) { props.rel = this.rel; }
    if (this.shape) { props.shape = this.shape; }
    if (this.coords) { props.coords = this.coords; }
    if (this.charset) { props.charset = this.charset; }
    if (this.accessKey) { props.accesskey = this.accessKey; }
    if (this.href) { props.href = this.href; }
    if (this.name) { props.name = this.name; } 
    if (this.contents) { props.contents = this.contents; }
    if (this.disabled != null) { props.disabled = this.disabled; }

    return props;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.anchorBase.prototype.onClickCallback = function(event) {
    if (this.disabled == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the request.
    var result = (this.domNode._onclick)
        ? this.domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    return true;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.anchorBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.anchorBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Add contents.
    this.addContents(props);

    // Set properties.
    if (props.accessKey) { this.domNode.accesskey = props.accessKey; }
    if (props.charset) { this.domNode.charset = props.charset; }
    if (props.coords) { this.domNode.coords = props.coords; }
        if (props.href) {
                
            // If context path is provided, then check whether the image has
            // context path already appended and if not, append it.
            if (this.prefix) {
                props.href = 
                    webui.suntheme4_2.widget.common.appendPrefix(this.prefix, props.href);
            }
            this.domNode.href = props.href; 
        }
    if (props.hrefLang) { this.domNode.hrefLang =  props.hrefLang; }
    if (props.name) { this.domNode.name = props.name; }
    if (props.rev) { this.domNode.rev = props.rev; }
    if (props.rel) { this.domNode.rel = props.rel; }
    if (props.shape) { this.domNode.shape = props.shape; }
    if (props.target) { this.domNode.target = props.target; }
    if (props.type) { this.domNode.type = props.type; }

    // Set id -- anchors must have the same id and name on IE.
    if (props.name) {
        props.id = props.name;
    }

    // A web app devleoper could return false in order to cancel the 
    // submit. Thus, we will handle this event via the onClick call back.
    if (props.onClick) {
        // Set private function scope on DOM node.
        this.domNode._onclick = (typeof props.onClick == 'string')
            ? new Function("event", props.onClick) : props.onClick;

        // Must be cleared before calling setEventProps() below.
        props.onClick = null;
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * @name webui.suntheme4_2.widget.anchor
 * @extends webui.suntheme4_2.widget.anchorBase
 * @class This class contains functions for the anchor widget.
 * @constructor This function is used to construct an anchor widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.anchor", webui.suntheme4_2.widget.anchorBase, {
    // Set defaults.
    widgetName: "anchor" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.anchor.event =
        webui.suntheme4_2.widget.anchor.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_anchor_event_refresh_begin",
        
        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_anchor_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_anchor_event_state_begin",
        
        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_anchor_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.anchor.prototype.getClassName = function() {
    // Set default style.
    var className = (this.href && this.disabled == false)
        ? this.widget.getClassName("ANCHOR","")
        : this.widget.getClassName("ANCHOR_DISABLED","");

    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.anchor.prototype.postCreate = function () {
    // Create callback function for onclick event.
    this.dojo.connect(this.domNode, "onclick", this, "onClickCallback");

    return this.inherited("postCreate", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.bubble");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.bubble
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the bubble widget.
 * @constructor This function is used to construct a bubble widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.bubble", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.defaultTime = 2000;
        this.openDelayTime = 500;
        this.bubbleLeftConst = 5;
        this.topConst = 2;
    },   
    widgetName: "bubble" // Required for theme properties.
});

/**
 * This function is used to close bubble help.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.bubble.prototype.close = function() {
    if (this.openTimerId != null) {
        clearTimeout(this.openTimerId);
    }
    if (this.getProps().visible == false) {
        return false;
    }
     
    var _id = this.id;
    this.timerId = setTimeout(function() {
        // New literals are created every time this function is called, and it's 
        // saved by closure magic.
        webui.suntheme4_2.dijit.byId(_id).setProps({visible: false});
        webui.suntheme4_2.dijit.byId(_id).srcElm.focus();
    }, this.defaultTime);

    return true;
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.bubble.event =
        webui.suntheme4_2.widget.bubble.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_bubble_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_bubble_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_bubble_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_bubble_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.bubble.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.title != null) { props.title = this.title; }
    if (this.contents != null) { props.contents = this.contents; }
    if (this.height != null) { props.height = this.height; }
    if (this.width != null) { props.width = this.width; }
    if (this.autoClose != null) { props.autoClose = this.autoClose; }
    if (this.duration != null) { props.duration = this.duration; }
    if (this.closeButton != null) {props.closeButton = this.closeButton;}
    if (this.openDelay != null) {props.openDelay = this.openDelay;}
    if (this.focusId != null) {props.focusId = this.focusId;}
    if (this.tabIndex != null) {props.tabIndex = this.tabIndex;}
    
    return props;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.bubble.prototype.onClickCallback = function(event) {
    // Close the popup if close button is clicked.
    event = this.widget.getEvent(event);

    var target = (event.target)
        ? event.target 
        : ((event.srcElement) 
            ? event.srcElement : null);

    if (webui.suntheme4_2.browser.isIe5up()) {
        if (window.event != null) {
            window.event.cancelBubble = true;
        }
    } else {
        event.stopPropagation();
    }
    if (this.closeBtn == target) {
        clearTimeout(this.timerId);
        this.setProps({visible: false});
        this.srcElm.focus();
    }
    return true;
};

/**
 * Helper function to create callback for close event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.bubble.prototype.onCloseCallback = function(event) {
    if (event == null) {
        return false;
    }
    
    if ((event.type == "keydown" && event.keyCode == 27)
            || event.type == "click") {
        clearTimeout(this.timerId); 
        
        if (this.srcElm != null && this.getProps().visible) {
            if (this.srcElm.focus) {
                this.srcElm.focus();
            }
        }      
        this.setProps({visible: false});
    }
    return true;
};

/**
 * Helper function to create callback for shift + tab event.
 * Shift+Tab should not allow user to tab out of bubble component.
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.bubble.prototype.onShftTabCallback = function(event) {
    if (event == null) {
        return false;
    }
    event = this.widget.getEvent(event);

    var target = (event.target)
        ? event.target 
        : ((event.srcElement) 
            ? event.srcElement : null);
    if (target == this.bubbleHeader) {                    
        if (webui.suntheme4_2.browser.isFirefox() && (event.shiftKey && (event.keyCode == 9))) {
            if (this.focusId != null) {
                document.getElementById(this.focusId).focus();        
            } else {                
                 this.bubbleHeader.focus();
            }
            event.stopPropagation();
            event.preventDefault(); 
        }
     }
     return true;
};

/**
 * Helper function to create callback for tab event.
 * Cyclic tabbing behavior is implemented for bubble to prevent tab out of bubble component. 
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.bubble.prototype.onTabCallback = function(event) {
    if (event == null) {
        return false;
    }
    event = this.widget.getEvent(event);

    var target = (event.target)
        ? event.target 
        : ((event.srcElement) 
            ? event.srcElement : null);
    if (webui.suntheme4_2.browser.isFirefox()) {        
        if (this.contentEnd == target) {
            this.bubbleHeader.focus();
        } else if (this.bubbleHeader == target && this.focusId != null && (event.keyCode == 9)) {
            document.getElementById(this.focusId).focus();            
        }
        event.stopPropagation();
        event.preventDefault(); 
    }
    return true;
};
    
/**
 * Helper function to create callback for onMouseOver event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.bubble.prototype.onMouseOverCallback = function(event) {
    clearTimeout(this.timerId);
    return true;
};

/**
 * Helper function to create callback for onMouseOut event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.bubble.prototype.onMouseOutCallback = function(event) {
    if (this.autoClose == true) {
        clearTimeout(this.timerId);            
        this.close();            
    }
    return true;
};

/**
 * This function is use to invoke buuble help.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.bubble.prototype.open = function(event) {
    // Get the absolute position of the target.
    var evt = this.widget.getEvent(event);
    // A11Y - open the bubble if its Ctrl key + F1
    if (evt.type == "keydown") {
        if (!(evt.ctrlKey && (evt.keyCode == 112))) {
            return false;
        }
        evt.stopPropagation();
        evt.preventDefault();  
    }
    this.srcElm = (evt.target) 
        ? evt.target : ((evt.srcElement) 
            ? evt.srcElement : null);

    var absPos = this.widget.getPosition(this.srcElm);
    this.srcElm.targetLeft = absPos[0];
    this.srcElm.targetTop = absPos[1];
   
    if (this.timerId != null) {
        clearTimeout(this.timerId);
        this.timerId = null;
    }
    
    if (this.openDelay != null && this.openDelay >= 0) {
        this.openDelayTime = this.openDelay;
    }

    // There should be delay before opening the bubble if open delay is specified.
    // If openDelay is less than zero then there will be dafault 0.5 sec delay.  
    
    var id = this.id; // Closure magic.
    this.openTimerId = setTimeout(function() {
        // Store the active bubble id to form element.
        // Check for the id if its available then close the pending bubble.
        if (webui.suntheme4_2.widget.bubble.activeBubbleId && webui.suntheme4_2.widget.bubble.activeBubbleId != id) {                
            clearTimeout(webui.suntheme4_2.dijit.byId(webui.suntheme4_2.widget.bubble.activeBubbleId).timerId);
            webui.suntheme4_2.dijit.byId(webui.suntheme4_2.widget.bubble.activeBubbleId).setProps({visible: false});
            webui.suntheme4_2.widget.bubble.activeBubbleId = null;                
        }     
        webui.suntheme4_2.widget.bubble.activeBubbleId = id;            
        webui.suntheme4_2.dijit.byId(id).setProps({visible: true});
        webui.suntheme4_2.dijit.byId(id).setPosition();
    }, this.openDelayTime);           
    
    if (this.duration != null && this.duration >= 0) {
        this.defaultTime = this.duration;
    } 
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.bubble.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.bottomLeftArrow.id = this.id + "_bottomLeftArrow";
        this.bottomRightArrow.id = this.id + "_bottomRightArrow";
        this.topLeftArrow.id = this.id + "_topLeftArrow";
        this.topRightArrow.id = this.id + "_topRightArrow";
    }

    // Set public functions.
    this.domNode.close = function() { return webui.suntheme4_2.dijit.byId(this.id).close(); };
    this.domNode.open = function(event) { return webui.suntheme4_2.dijit.byId(this.id).open(event); };

    // Set events.

    // The onClick on window should close bubble.
    this.dojo.connect(document, "onclick", this, "onCloseCallback");

    // The escape key should also close bubble.
    this.dojo.connect(document, "onkeydown", this, "onCloseCallback");

    // The onClick event for component body. Closes the bubble only when
    // close button is clicked.
    this.dojo.connect(this.domNode, "onclick", this, "onClickCallback");

    // Do not close the popup if mouseover on bubble if mouseover on bubble 
    // component then clear the timer and do not close bubble.
    this.dojo.connect(this.domNode, "onmouseover", this, "onMouseOverCallback");

    // Close the popup if mouseout and autoClose is true if onmouseout and 
    // autoClose is true then close the bubble.
    this.dojo.connect(this.domNode, "onmouseout", this, "onMouseOutCallback");
    
    // The onfocus event for contentEnd. This is needed to handle tab event. 
    this.dojo.connect(this.contentEnd, "onfocus", this, "onTabCallback");
    
    // The onkeydown event for bubbleHeader. This is needed to handle tab event. 
    this.dojo.connect(this.bubbleHeader, "onkeydown", this, "onTabCallback");
    
    // The onkeydown event for component body. This is needed to handle shift+tab event.
    this.dojo.connect(this.domNode, "onkeydown", this, "onShftTabCallback");
    
    // Initialize the BubbleTitle width as a percentage of the bubble header.    
    if (this.bubbleTitle != null) {
        this.bubbleTitle.style.width = this.theme.getProperty("styles", 
            "BUBBLE_TITLEWIDTH") + "%";
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to position the bubble.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.bubble.prototype.setPosition = function() {
    
    // THIS CODE BLOCK IS NECESSARY WHEN THE PAGE FONT IS VERY SMALL,
    // AND WHICH OTHERWISE CAUSES THE PERCENTAGE OF THE HEADER WIDTH
    // ALLOCATED TO THE BUBBLE TITLE TO BE TOO LARGE SUCH THAT IT
    // ENCROACHES ON THE SPACE ALLOCATED FOR THE CLOSE BUTTON ICON,
    // RESULTING IN LAYOUT MISALIGNMENT IN THE HEADER.

    // Assume BubbleTitle width max percentage of the bubble header.
    var maxPercent = this.theme.getProperty("styles", "BUBBLE_TITLEWIDTH");

    // Sum of widths of all elements in the header BUT the title.  This includes
    // the width of the close button icon, and the margins around the button and
    // the title.  This should be a themeable parameter that matches the left/right
    // margins specified in the stylesheet for "BubbleTitle" and "BubbleCloseBtn".
    var nonTitleWidth = this.theme.getProperty("styles", "BUBBLE_NONTITLEWIDTH");

    // Get the widths (in pixels) of the bubble header and title
    var headerWidth = this.bubbleHeader.offsetWidth;
    var titleWidth = this.bubbleTitle.offsetWidth;

    // Revise the aforementioned percentage downward until the title no longer
    // encroaches on the space allocated for the close button.  We decrement by
    // 5% each time because doing so in smaller chunks when the font gets very small so 
    // only results in unnecessary extra loop interations.
    //
    if (headerWidth > nonTitleWidth) {
        while ((maxPercent > 5) && (titleWidth > (headerWidth - nonTitleWidth))) {
            maxPercent -= 5;
            this.bubbleTitle.style.width = maxPercent + "%";
            titleWidth = this.bubbleTitle.offsetWidth;
        }
    }

    // Get DOM bubble object associated with this Bubble instance.
    var bubble = this.domNode;

    // If this.style is not null that means developer has specified positioning
    // for component. 
    if (this.domNode != null && this.style != null && this.style.length > 0) {
        if (bubble.style.length != null) {
            for (var i = 0; i < bubble.style.length; i++) {
                if (bubble.style[i] == "top") {
                    this.top = bubble.style.top;
                }
                if (bubble.style[i] == "left") {
                    this.left = bubble.style.left;
                }
            }
        } else {
            // For IE, simply query the style attributes.
            if (bubble.style.top != "") {
                this.top = bubble.style.top;
            }
            if (bubble.style.left != "") {
                this.left = bubble.style.left;
            }
        }
    }

    if ((this.top != null) && (this.left != null)) {
        bubble.style.left = this.left;
        bubble.style.top = this.top;    
    } else {        

        var topLeftArrow = document.getElementById(this.topLeftArrow.id);
        var topRightArrow = document.getElementById(this.topRightArrow.id);
        var bottomLeftArrow = document.getElementById(this.bottomLeftArrow.id);
        var bottomRightArrow = document.getElementById(this.bottomRightArrow.id);
        // hide all callout arrows.
        this.common.setVisible(bottomLeftArrow, false);
        this.common.setVisible(bottomRightArrow, false);
        this.common.setVisible(topLeftArrow, false);
        this.common.setVisible(topRightArrow, false);

        bottomLeftArrow.style.display = "none";
        bottomRightArrow.style.display = "none";
        topLeftArrow.style.display = "none";
        topRightArrow.style.display = "none";

        var slidLeft = false;

        // Assume default bubble position northeast of target, which implies a 
        // bottomLeft callout arrow
        this.arrow = bottomLeftArrow;

        // Try to position bubble to right of srcElm.
        var bubbleLeft = this.srcElm.targetLeft + this.srcElm.offsetWidth + this.bubbleLeftConst;

        // Check if right edge of bubble exceeds page boundary.
        var rightEdge = bubbleLeft + bubble.offsetWidth;
        if (rightEdge > this.widget.getPageWidth()) {

            // Shift bubble to left side of target;  implies a bottomRight arrow.
            bubbleLeft = this.srcElm.targetLeft - bubble.offsetWidth;
            this.arrow = bottomRightArrow;
            slidLeft = true;

            // If left edge of bubble crosses left page boundary, then
            // reposition bubble back to right of target and implies to go
            // back to bottomLeft arrow.  User will need to use scrollbars
            // to position bubble into view.
            if (bubbleLeft <= 0) {
                bubbleLeft = this.srcElm.targetLeft + this.srcElm.offsetWidth + this.bubbleLeftConst;
                this.arrow = bottomLeftArrow;
                slidLeft = false;
            }
        }

        // Try to position bubble above source element
        var bubbleTop = this.srcElm.targetTop - bubble.offsetHeight;

        // Check if top edge of bubble crosses top page boundary
        if (bubbleTop <= 0) {
            // Shift bubble to bottom of target.  User may need to use scrollbars
            // to position bubble into view.
            bubbleTop = this.srcElm.targetTop + this.srcElm.offsetHeight + this.bubbleLeftConst;

            // Use appropriate top arrow depending on left/right position.
            if (slidLeft == true)
                this.arrow = topRightArrow;
            else
                this.arrow = topLeftArrow;
        }

        // Set new bubble position.
        bubble.style.left = bubbleLeft + "px";
        bubble.style.top = bubbleTop + "px";

        // If rendering a callout arrow, set it's position relative to the bubble.
        if (this.arrow != null) {
           this.arrow.style.display = "block";
           this.common.setVisible(this.arrow, true);

           if (this.arrow == topLeftArrow) {
               this.arrow.style.top = -(bubble.offsetHeight - this.topConst) + "px";               
           }
           if (this.arrow == topRightArrow) {
               this.arrow.style.top = -(bubble.offsetHeight - this.topConst) + "px";               
           }
        }
    }
    if (this.focusId != null) {
        document.getElementById(this.focusId).focus();        
    } else {
        if (webui.suntheme4_2.browser.isFirefox()) {
            this.bubbleHeader.focus();
        }
    }
    return true;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} autoClose 
 * @config {Object} closeButton 
 * @config {Array} contents 
 * @config {int} duration 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {int} openDelay 
 * @config {String} title Provides a title for element.
 * @config {int} width 
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.bubble.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }
    
    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.bubble.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }
    //Cyclic focus behavior is supported for firefox browser only.
    //If tabIndex values are provided for elements inside bubble then developer needs to set a valid tabIndex 
    //value for bubble component to achieve cyclic focus behavior. 
    if (webui.suntheme4_2.browser.isFirefox()) {
        if (this.getProps().tabIndex >= 0) {
            this.contentEnd.tabIndex = this.getProps().tabIndex;
        } else {
            this.contentEnd.tabIndex = 0;
        }   
    }
    // Set title.
    if (props.title) {
        this.widget.addFragment(this.titleNode, props.title);
    }

    // hide/display close button
    if (props.closeButton != null) {
        var classNames = this.closeBtn.className.split(" ");
        var closeButtonClass = this.theme.getClassName("BUBBLE_CLOSEBTN");
        var noCloseButtonClass = this.theme.getClassName("BUBBLE_NOCLOSEBTN");

        if (props.closeButton == false) {
            this.common.stripStyleClass(this.closeBtn, closeButtonClass);
            if (!this.common.checkStyleClasses(classNames, noCloseButtonClass))
             this.common.addStyleClass(this.closeBtn, noCloseButtonClass);
        } else {          
          if (!this.common.checkStyleClasses(classNames, closeButtonClass))
             this.common.addStyleClass(this.closeBtn, closeButtonClass);
        }
    }

    // Set width.
    if (props.width > 0) {                    
        this.domNode.style.width = props.width + "px";        
    }

    // Set contents.
    if (props.contents) {
        // Remove child nodes.
        this.widget.removeChildNodes(this.childNode);

        for (var i = 0; i < props.contents.length; i++) {
            this.widget.addFragment(this.childNode, props.contents[i], "last");
        }
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.button");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");
 
/**
 * @name webui.suntheme4_2.widget.button
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the button widget.
 * @constructor This function is used to construct a button widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.button", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
        this.escape = true;
        this.mini = false;
        this.primary = true;
    },
    widgetName: "button" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.button.event =
        webui.suntheme4_2.widget.button.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_button_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_button_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_button_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_button_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.button.prototype.getClassName = function() {
    var key = null;

    if (this.mini == true && this.primary == true) {
        key = (this.disabled == true)
            ? "BUTTON1_MINI_DISABLED" // primaryMiniDisabledClassName
            : "BUTTON1_MINI";         // primaryMiniClassName;
    } else if (this.mini == true) {
        key = (this.disabled == true)
            ? "BUTTON2_MINI_DISABLED" // secondaryMiniDisabledClassName
            : "BUTTON2_MINI";         // secondaryMiniClassName;
    } else if (this.primary == true) {
        key = (this.disabled == true)
            ? "BUTTON1_DISABLED"      // primaryDisabledClassName
            : "BUTTON1";              // primaryClassName
    } else {
        key = (this.disabled == true)
            ? "BUTTON2_DISABLED"      // secondaryDisabledClassName
            : "BUTTON2";	      // secondaryClassName
    }

    var className = this.widget.getClassName(key, "");
    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to obtain the outermost HTML element class name during
 * an onFocus or onMouseOver event.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.button.prototype.getHoverClassName = function() {
    var key = null;

    if (this.mini == true && this.primary == true) {
        key = "BUTTON1_MINI_HOVER"; 	// primaryMiniHovClassName;
    } else if (this.mini == true) {
        key = "BUTTON2_MINI_HOVER"; 	// secondaryMiniHovClassName;
    } else if (this.primary == true) {
        key = "BUTTON1_HOVER"; 		// primaryHovClassName;
    } else {
        key = "BUTTON2_HOVER";		// secondaryHovClassName;
    }

    var className = this.widget.getClassName(key, "");
    return (this.className)
        ? className + " " + this.className
        : className;
};
    
/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.button.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.alt) { props.alt = this.alt; }
    if (this.align) { props.align = this.align; }
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.escape != null) { props.escape = this.escape; }
    if (this.mini != null) { props.mini = this.mini; }
    if (this.primary != null) { props.primary = this.primary; }
    if (this.value) { props.value = this.value; }

    return props;
};

/**
 * Helper function to create callback for onBlur event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.button.prototype.onBlurCallback = function(event) {
    if (this.disabled == true) {
        return true;
    }
    // Prevent errors during page submit, when modules have not been loaded.
    try {
        // Set style class.
        this.domNode.className = this.getClassName();
    } catch (err) {}
    return true;
};

/**
 * Helper function to create callback for onFocus event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.button.prototype.onFocusCallback = function(event) {
    if (this.disabled == true) {
        return true;
    }
    // Prevent errors during page submit, when modules have not been loaded.
    try {
        // Set style class.
        this.domNode.className = this.getHoverClassName();
    } catch (err) {}
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.button.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.domNode.name = this.id;
    }

    // Initialize deprecated public functions. 
    // 
    // Note: Although we now have a setProps function to update properties,
    // these functions were previously added to the DOM node; thus, we must
    // continue to be backward compatible.
    this.domNode.isSecondary = function() { return !(webui.suntheme4_2.dijit.byId(this.id).getProps().primary); };
    this.domNode.setSecondary = function(secondary) { return webui.suntheme4_2.dijit.byId(this.id).setProps({primary: !secondary}); };
    this.domNode.isPrimary = function() { return webui.suntheme4_2.dijit.byId(this.id).getProps().primary; };
    this.domNode.setPrimary = function(primary) { return webui.suntheme4_2.dijit.byId(this.id).setProps({primary: primary}); };
    this.domNode.isMini = function() { return webui.suntheme4_2.dijit.byId(this.id).getProps().mini; };
    this.domNode.setMini = function(mini) { return webui.suntheme4_2.dijit.byId(this.id).setProps({mini: mini}); };
    this.domNode.getDisabled = function() { return webui.suntheme4_2.dijit.byId(this.id).getProps().disabled; };
    this.domNode.setDisabled = function(disabled) { return webui.suntheme4_2.dijit.byId(this.id).setProps({disabled: disabled}); };
    this.domNode.getVisible = function() { return webui.suntheme4_2.dijit.byId(this.id).getProps().visible; };
    this.domNode.setVisible = function(show) { return webui.suntheme4_2.dijit.byId(this.id).setProps({visible: show}); };
    this.domNode.getText = function() { return webui.suntheme4_2.dijit.byId(this.id).getProps().value; };
    this.domNode.setText = function(text) { return webui.suntheme4_2.dijit.byId(this.id).setProps({value: text}); };
    this.domNode.doClick = this.domNode.click;

    // Set events.
    this.dojo.connect(this.domNode, "onblur", this, "onBlurCallback");
    this.dojo.connect(this.domNode, "onfocus", this, "onFocusCallback");
    this.dojo.connect(this.domNode, "onmouseout", this, "onBlurCallback");
    this.dojo.connect(this.domNode, "onmouseover", this, "onFocusCallback");

    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} escape HTML escape value (default).
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {boolean} mini Set button as mini if true.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} primary Set button as primary if true.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.button.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.button.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.alt) { this.domNode.alt = props.alt; }
    if (props.align) { this.domNode.align = props.align; }

    // Set disabled.
    if (props.disabled != null) { 
        this.domNode.disabled = new Boolean(props.disabled).valueOf();
    }

    // Set value (i.e., button text).
    if (props.value) {
        // If escape is true, we want the text to be displayed literally. To 
        // achieve this behavior, do nothing.
        //
        // If escape is false, we want any special sequences in the text 
        // (e.g., "&nbsp;") to be displayed as evaluated (i.e., unescaped).
        this.domNode.value = (new Boolean(this.escape).valueOf() == false)
            ? this.prototypejs.unescapeHTML(props.value)
            : props.value;
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.calendar");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.calendar
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the calendar widget.
 * @constructor This function is used to construct a calendar widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.calendar", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    widgetName: "calendar" // Required for theme properties.
});

/**
 * Helper function to add a day link in a cell.
 *
 * @param {Node} rowNodeClone
 * @param {Object} day
 * @param {String} id
 * @param {String} className
 * @param {boolean} setFocus
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.addDayLink = function(rowNodeClone, day, 
        id, className, setFocus) {
    // Clone <td> and <a> elements. 
    var colNodeClone = this.dayColumnContainer.cloneNode(false);
    rowNodeClone.appendChild(colNodeClone);    
    var linkNodeClone = this.dayLinkContainer.cloneNode(false);            
    colNodeClone.appendChild(linkNodeClone);
    
    // Format the date.      
    var formattedDate = this.formatDate(day.getMonth() + 1, day.getDate(), day.getFullYear()); 
  
    // set the link's properties for this day.
    linkNodeClone.title = formattedDate;
    linkNodeClone.id = id;
    linkNodeClone.href = "#";
    linkNodeClone.className = className;

    // NOTE: If you set this value manually, text must be HTML escaped.
    this.widget.addFragment(linkNodeClone, "" + day.getDate());

    var widgetId = this.id;
    linkNodeClone.onclick = function() { 
        webui.suntheme4_2.dijit.byId(widgetId).daySelected(formattedDate); 
        return false;
    };  
    
    // If the setFocus is set to true, then when you tab out of the linkNode,
    // the focus should go to the close button. 
    if (setFocus) {
        var node = webui.suntheme4_2.dijit.byId(linkNodeClone.id);        
        linkNodeClone.onkeydown = function(event) {
            var widget = webui.suntheme4_2.dijit.byId(widgetId);
            
            // Get hold of the close button and set focus on it.
            var evt = (event) ? event : ((window.event) ? window.event : null);
            if (evt.keyCode == 9) {
                var elem = document.getElementById(widget.closeButtonLink.id);
                if (elem != null) {
                    if (elem.focus) {
                        elem.focus();
                    }
                    if (evt.preventDefault) {
                        evt.preventDefault();
                    } else {
                        evt.returnValue = false;
                    }
                    return false;                                  
                }
            }
            return true;
        };
    }
    return true;
};

/**
 * Helper function to add days in the month -- week data rows.
 *
 * @param {Object} currentValue The current value of the text field.
 * @param {boolean} initialize Flag indicating to initialze the year and month menus
 * with the current value. The value is true only when the calendar is opened.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.addDaysInMonth = function(currentValue, initialize) {
    // Date representing a day in a month.
    var day;    
    // Number of columns in a row.
    var column = 0;
    // Row number.    
    var rowNum = 0;    
    // Today's day.
    var today = 0;
    // Selected date.
    var selected = 0;     
    // Day link number
    var linkNum = 0; 
    // Prefix used for a day link id.
    var id = this.id + "_link:";
    // Day link id. ie, id + linkNum.
    var linkId;
    // One day in milliseconds -- 1000 * 60 * 60 * 24    
    var oneDayInMs = 86400000;     

    var todayDate = new Date();
    var todayYear = todayDate.getFullYear();
    var todayMonth = todayDate.getMonth() + 1;
    var todayDay = todayDate.getDate();                  
    
    // selectedYear, selectedMonth, selectedDay:
    // The date to show as highlighted (currentValue) provided
    // that the user is viewing that month and year.
    var selectedYear = null;
    var selectedMonth = null;
    var selectedDay = null;
    if (currentValue != null) {
        selectedYear = currentValue.getFullYear();
        selectedMonth = currentValue.getMonth() + 1;
        selectedDay = currentValue.getDate();
    }
    
    // Get month and year menu widgets.
    var monthMenuWidget = webui.suntheme4_2.dijit.byId(this.monthMenu.id);        
    var yearMenuWidget = webui.suntheme4_2.dijit.byId(this.yearMenu.id);
    if (monthMenuWidget == null || yearMenuWidget == null) {
        return false;
    }
               
    if (initialize) {
         // Set showMonth as selected in the month menu
	 // Set showYear as selected in the year menu
         // Use todayMonth and todayYear if currentValue is null.
	 var showMonth = todayMonth;
	 var showYear = todayYear;
	 if (currentValue != null) {
             // We have a currentValue, so use that for showMonth and showYear
             showMonth = selectedMonth;
	     showYear = selectedYear;
         }         
         this.setLimitedSelectedValue(monthMenuWidget.getSelectElement(), showMonth);
         this.setLimitedSelectedValue(yearMenuWidget.getSelectElement(), showYear);
    }
    
    var month = parseInt(monthMenuWidget.getSelectedValue());
    var year = parseInt(yearMenuWidget.getSelectedValue());
    
    //set selected
    if (currentValue != null && selectedYear == year && selectedMonth == month) {
        selected = selectedDay;
    }
        
    //set today
    if (todayYear == year && todayMonth == month) {
        today = todayDay;
    }
    
    // Add first week data row.
    var rowNodeClone = this.weekRowContainer.cloneNode(false);
    this.tbodyContainer.appendChild(rowNodeClone); 
    rowNodeClone.id = this.id + ":row" + rowNum;
    
    // Convert to javascript month numbering.
    month--;
    
    // Calculate the first of the main month to display in "first" row.
    var first = new Date(year, month, 1);                         
    var firstDay = first.getDay();    
    var className = this.theme.getClassName("DATE_TIME_OTHER_LINK");
    if (firstDay == this.firstDayOfWeek - 1) {
        // First cell on first row is the first of the current month
        day = first;
    } else {
        // First cell on first row is in previous month.
        var backDays = (firstDay - (this.firstDayOfWeek - 1) + 7) % 7;        
        
        // Calculate the date of first cell on first row in previous month.
        day = new Date(first.getTime() - backDays * oneDayInMs);        
        
        // Generate start of first row up to first of month
        while (day.getMonth() !=  first.getMonth()) {
            linkId = id + linkNum;
            this.addDayLink(rowNodeClone, day, linkId, className);
            day = new Date(day.getTime() + oneDayInMs);
            column++;
            linkNum++;            
        }
    }

    // Add any cells in the first row of the main month.
    while (column < 7) {
        // Set appropriate class name.
        if (day.getDate() == selected) {
            className = this.theme.getClassName("DATE_TIME_BOLD_LINK");
        } else if (day.getDate() == today) {
            className = this.theme.getClassName("DATE_TIME_TODAY_LINK");
        } else {
           className = this.theme.getClassName("DATE_TIME_LINK");
        }
            
        linkId = id + linkNum;
        this.addDayLink(rowNodeClone, day, linkId, className);        
        day = new Date(day.getTime() + oneDayInMs);
        column++;
        linkNum++;
    } 
    
    // This variable is used to decide whether the focus should be set
    // on the calendar's close button when tabbing    
    var setFocus = false;            
    
    // Add intermediate rows
    while (day.getDate() != 1) {
        rowNum++;
        // Clone a <tr> node
        rowNodeClone = this.weekRowContainer.cloneNode(false);
        this.tbodyContainer.appendChild(rowNodeClone); 
        rowNodeClone.id = this.id + ":row" + rowNum;

        column = 0;
        while (column < 7 && day.getDate() != 1) {            
            // Set appropriate class name.
            if (day.getDate() == selected) {
                className = this.theme.getClassName("DATE_TIME_BOLD_LINK");
            } else if (day.getDate() == today) {
                className = this.theme.getClassName("DATE_TIME_TODAY_LINK");
            } else {
                className = this.theme.getClassName("DATE_TIME_LINK");
            }
                 
            linkId = id + linkNum;
            var tmpDate = new Date(day.getTime() + oneDayInMs);

            // On some platforms, the date is not incremented correctly (e.g.,
            // October 28th 1990, 2007, and 2012). In this case, try again.
            if (tmpDate.getDate() == day.getDate()) {
                tmpDate = new Date(tmpDate.getTime() + oneDayInMs);
            } 
            
            // Check whether this is the last date in the calendar and if so
            // set the setFocus variable to true. This will mean that when the
            // user tabs away from this day, the focus will be set on the
            // close button.
            if (tmpDate.getDate() == 1 && column == 6) {
                setFocus = true;
            } else {
                setFocus = false;
            }
            this.addDayLink(rowNodeClone, day, linkId, className, setFocus);
            day = tmpDate;
            column++;
            linkNum++;
        }
    }
    
    // Add any cells in the last row of the following month
    while (column < 7) {
        var className = this.theme.getClassName("DATE_TIME_OTHER_LINK");
        linkId = id + linkNum;
        
        // Check whether this is the last date in the calendar and if so
        // set the setFocus variable to true. This will mean that when the
        // user tabs away from this day, the focus will be set on the
        // close button.        
        if (column == 6) {
            setFocus = true;
        } else {
            setFocus = false;
        }
        this.addDayLink(rowNodeClone, day, linkId, className, setFocus);                    
        day = new Date(day.getTime() + oneDayInMs);
        column++;
        linkNum++;
    }
    return true;
};

/**
 * Helper function to add the week day headers row.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.addWeekDays = function() {            
    var colNodeClone;
    var spanNodeClone;    
    var firstDay = this.firstDayOfWeek - 1;
    
    // Clone the <tr> node and append it to <tbody>
    var rowNodeClone = this.weekDayRow.cloneNode(false);
    this.tbodyContainer.appendChild(rowNodeClone);
        
    for (var i = 0; i < 7; i++) {
        // Clone the <th> node and append it to <tr>
        colNodeClone = this.weekDayColumn.cloneNode(false);
        rowNodeClone.appendChild(colNodeClone);
               
        // Clone the <span> node and append it to <th>
        spanNodeClone = this.weekDayContainer.cloneNode(false);
        colNodeClone.appendChild(spanNodeClone);
        
        // NOTE: If you set this value manually, text must be HTML escaped.
        this.widget.addFragment(spanNodeClone, this.weekDays[firstDay]);

        firstDay++;
        if (firstDay == 7) {
            firstDay = 0;
        }     
    }
    return true;
};

/**
 * Close the calendar if the enter key is pressed and set the initial focus
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */            
webui.suntheme4_2.widget.calendar.prototype.closeCalendar = function(event) {
    var evt = (event) ? event : ((window.event) ? window.event : null);
     
    // If key pressed and enter, then close the menu
    if ((evt.type == "keydown") && (evt.keyCode == 13)) {
        this.toggleCalendar();
        document.getElementById(this.toggleLink.id).focus();        
        return false;
    } 
    this.setInitialFocus();
    return true;    
};

/**
 * Process day selected event.
 * <p>
 * When a day link is selected, an event is published which the 
 * calendarField widget will use to update its text field.
 * </p>
 * @param {String} Formatted date string.
 * @return {boolean} false to cancel JavaScript event.
 */
webui.suntheme4_2.widget.calendar.prototype.daySelected = function(formattedDate) {
    this.toggleCalendar();    
    this.publish(webui.suntheme4_2.widget.calendar.event.day.selectedTopic, [{
        id: this.id,
        date:formattedDate
    }]);
    return false;
};

/**
 * This function is used to decrease the month by one.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.decreaseMonth = function() {
    var monthMenu = webui.suntheme4_2.dijit.byId(this.monthMenu.id).getSelectElement();
    // If the monthMenu has no value, set it to January (that's what
    // it will have appeared like in the browser). Can happen on IE.  
    if (monthMenu.value == null) {
        monthMenu.value = monthMenu.options[0].value;
    }
    
    var month = parseInt(monthMenu.value);
    if (month == 1) {
        var yearMenu = webui.suntheme4_2.dijit.byId(this.yearMenu.id).getSelectElement();        
         if (yearMenu.value == null) {
             // If the yearMenu has no value, set it to the first available year            
             // (that's what it will have appeared like in the browser). Can happen on IE.
             yearMenu.value = yearMenu.options[0].value;
         } else if (yearMenu.value == yearMenu.options[0].value) {
             // No need to update the calendar in this case,
             // we don't change anything.
             return false;           
         } else {
             // Decrease the year by one and set the month to December
             var year = parseInt(yearMenu.value);
             year--;
             yearMenu.value = year;
             month = 12;
         }
    } else {
        month--;
    }
    monthMenu.value = month;    
    return this.updateMonth(false);
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.calendar.event =
        webui.suntheme4_2.widget.calendar.prototype.event = {
    /**
     * This object contains day event topics.
     * @ignore
     */
    day: {
        /** Day event topic for custom AJAX implementations to listen for. */
        selectedTopic: "webui_suntheme4_2_widget_calendar_event_selected"
    },

    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_calendar_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_calendar_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_calendar_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_calendar_event_state_end"
    },

    /**
     * This object contains toggle event topics.
     * @ignore
     */
    toggle: {
        /** Open event topic for custom AJAX implementations to listen for. */
        openTopic: "webui_suntheme4_2_widget_calendar_event_toggle_open",

        /** Close event topic for custom AJAX implementations to listen for. */
        closeTopic: "webui_suntheme4_2_widget_calendar_event_toggle_close"
    }
};

/**
 * Helper function to format the date.
 *
 * @param {String} month
 * @param {String} day
 * @param {String} year
 * @return {String} The date format.
 */
webui.suntheme4_2.widget.calendar.prototype.formatDate = function(month, day, year) {
    var date = new String(this.dateFormat);      
    date = date.replace("yyyy", new String(year));
    if (month < 10) {
        date = date.replace("MM", "0" + new String(month));
    } else {
        date = date.replace("MM", new String(month));
    }
    if (day < 10) {
        date = date.replace("dd", "0" + new String(day));
    } else {
        date = date.replace("dd", new String(day));
    }
    return date;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.calendar.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);
    
    // Set properties.
    if (this.todayDateMsg) { props.todayDateMsg = this.todayDateMsg; }
    if (this.spacerImage) { props.spacerImage = this.spacerImage; }
    if (this.topLeftImage) { props.topLeftImage = this.topLeftImage; }
    if (this.topRightImage) { props.topRightImage = this.topRightImage; }
    if (this.closeButtonLink) { props.closeButtonLink = this.closeButtonLink; }
    if (this.increaseLink) { props.increaseLink = this.increaseLink; }
    if (this.decreaseLink) { props.decreaseLink = this.decreaseLink; }
    if (this.monthMenu) { props.monthMenu = this.monthMenu; }
    if (this.yearMenu) { props.yearMenu = this.yearMenu; }   
    if (this.firstDayOfWeek) { props.firstDayOfWeek = this.firstDayOfWeek; }
    if (this.toggleLink) { props.toggleLink = this.toggleLink; }
    if (this.weekDays) { props.weekDays = this.weekDays; }    
    if (this.maxDate) { props.maxDate = this.maxDate; }
    if (this.minDate) { props.minDate = this.minDate; }
    
    return props;
};

/**
 * Workaround IE bug where popup calendar appears under other components.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.ieStackingContextFix = function() {
    var div = this.calendarContainer;
    if (div.style.display == "block") {
        // This popup should be displayed
        // Get the current zIndex for the div
        var divZIndex = div.currentStyle.zIndex;
        
        // Propogate the zIndex up the offsetParent tree
        var tag = div.offsetParent;
        while (tag != null) {
            var position = tag.currentStyle.position;
            if (position == "relative" || position == "absolute") {

                // Save any zIndex so it can be restored
                tag.raveOldZIndex = tag.style.zIndex;

                // Change the zIndex
                tag.style.zIndex = divZIndex;
            }
            tag = tag.offsetParent;
        }

        // Hide controls unaffected by z-index
        this.ieShowShim();
    } else {
        // This popup should be hidden so restore zIndex-s
        var tag = div.offsetParent;
        while (tag != null) {
            var position = tag.currentStyle.position;
            if (position == "relative" || position == "absolute") {
                if (tag.raveOldZIndex != null) {
                    tag.style.zIndex = tag.raveOldZIndex;
                }
            }
            tag = tag.offsetParent;
        }
        this.ieHideShim();
    }
    return true;
};

/**
 * Hides components unaffected by z-index.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.ieShowShim = function() {  
    var popup = this.calendarContainer;
    var shim = this.shimContainer;
    
    shim.style.position = "absolute";
    shim.style.left = popup.style.left;
    shim.style.top = popup.style.top;
    shim.style.width = popup.offsetWidth;
    shim.style.height = popup.offsetHeight;
    shim.style.zIndex = popup.currentStyle.zIndex - 1;
    shim.style.display = "block";

    return true;
};

/**
 * Hide the shim iframe.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.ieHideShim = function() {
    var shim = this.shimContainer;
    shim.style.display = "none";
    return true;
};

/**
 * This function is used to increment the current month.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.increaseMonth = function() {            
    var monthMenu = webui.suntheme4_2.dijit.byId(this.monthMenu.id).getSelectElement();          
    
    // If the monthMenu has no value, set it to January (that's what
    // it will have appeared like in the browser). Can happen on IE. 
    if (monthMenu.value == null) {
        monthMenu.value = monthMenu.options[0].value;
    }
    
    var month = parseInt(monthMenu.value);
    if (month == 12) {
        var yearMenu = webui.suntheme4_2.dijit.byId(this.yearMenu.id).getSelectElement();
        var numOptions = yearMenu.options.length;
        if (yearMenu.value == null) {
            // If the yearMenu has no value, set it to the first available year            
            // (that's what it will have appeared like in the browser). Can happen on IE.
            yearMenu.value = yearMenu.options[0].value;
        } else if (yearMenu.value == yearMenu.options[numOptions-1].value) {
            // No need to update the calendar in this case,
            // we don't change anything.
            return false;            
        } else {
            // Increase the year by one and set the month to January.
            var year = parseInt(yearMenu.value);
            year++;
            yearMenu.value = year;
            month = 1;
        }
    } else {
        month++;
    }
    monthMenu.value = month;   
    return this.updateMonth(false);    
};

/**
 * This function returns a JSON array of months to be displayed in the month 
 * drop down. 
 * return {Object} A JSON array of months.
 */
webui.suntheme4_2.widget.calendar.prototype.getMonthOptions = function() {
    var monthMenu = new Array();
    
    // Get the number of months in a calendar year.
    // Some calendars may have more than 12 months a year.
    var numOfMonths = parseInt(this.theme.getMessage("calendar.numOfMonths"));
    
    for ( var i = 0; i < numOfMonths; i++ ) {
        monthMenu[i] = {};
        monthMenu[i].value = i+1;
        monthMenu[i].disabled = false;
        monthMenu[i].separator = false;
        monthMenu[i].escape = true;
        monthMenu[i].group = false;
        monthMenu[i].label=this.theme.getMessage("calendar."+i);
    }    
    return monthMenu;
};

/**
 * This function returns a JSON array of years to be displayed in the year
 * drop down
 * @param {String} minYear the minimum year of the calendar display
 * @param {String} maxYear the maximum year of the calendar display
 * @return {Object} A JSON array of calendar years.
 */
webui.suntheme4_2.widget.calendar.prototype.getYearOptions = function(minYear, maxYear) {    
    var yearMenu =new Array();       
    var diff = maxYear - minYear;
    for ( var i = 0; i <= diff; i++ ) {
        yearMenu[i] = {};
        yearMenu[i].value = minYear;
        yearMenu[i].disabled = false;
        yearMenu[i].separator = false;
        yearMenu[i].escape = true;
        yearMenu[i].group = false;
        yearMenu[i].label=minYear;
        minYear++;
    }
    return yearMenu;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.postCreate = function () {
    // Set ids. 
    if (this.id) {
        this.calendarMenuContainer.id = this.id + "_calendarMenuContainer";
        this.linkNode.id = this.id + "_linkNodeContainer";
        this.todayDateContainer.id = this.id + "_todayDateContainer";
        this.closeButtonContainer.id = this.id + "_closeButtonContainer";
        this.previousLinkContainer.id = this.id + "_previousLinkContainer";
        this.monthMenuContainer.id = this.id + "_monthMenuContainer";
        this.nextLinkContainer.id = this.id + "_nextLinkContainer";
        this.yearMenuContainer.id = this.id + "_yearMenuContainer";
        this.shimContainer.id = this.id + "_shim";
    }

    // Create client side widgets for the calendar.
    // When the _setProps() function is called, these widgets will be
    // instantiated via the props param. 

    // If toggle link is null, create the image hyperlink.
    if (this.toggleLink == null) {
        this.toggleLink = this.widget.getWidgetProps("imageHyperlink", {
                id: this.id + "_datePickerLink",
                contents: [],
                imagePosition: "left",
                title: this.theme.getMessage("calendar.popupImageAlt"),
                enabledImage: this.widget.getWidgetProps("image", {
                    border: 0,
                    icon: "CALENDAR_BUTTON",
                    id: this.id + "_datePickerLink_image"
                }),
                disabledImage: this.widget.getWidgetProps("image", {
                    border: 0,
                    icon: "CALENDAR_BUTTON_DISABLED",
                    id: this.id + "_datePickerLink_image_disabled"
                }),
                align:"middle"
            });
    }

    // Create the spacer image.
    if (this.spacerImage == null) {
        this.spacerImage = this.widget.getWidgetProps("image", {
            icon: "DOT",
            id: this.id + ":DOT"
        });
    }
    
    // Create the top left image.
    if (this.topLeftImage == null) {
        this.topLeftImage = this.widget.getWidgetProps("image", {
            icon: "SCHEDULER_TOP_LEFT",
            id: this.id + ":topLeft"
        });        
    }        
        
    //Create the top right image.
    if (this.topRightImage == null) {
        this.topRightImage = this.widget.getWidgetProps("image", {
            icon: "SCHEDULER_TOP_RIGHT",
            id: this.id + ":topRight"
        });        
    }

    // Create the increase link imageHyperlink widget.
    if (this.increaseLink == null) {
        this.increaseLink = this.widget.getWidgetProps("imageHyperlink", {
                id: this.id + ":nextMonthLink",
                enabledImage: this.widget.getWidgetProps("image", {
                    border: 0,
                    icon: "SCHEDULER_FORWARD",
                    id: this.id + ":nextMonthLink_image"
                }),
                title: this.theme.getMessage("CalendarMonth.goForward")             
            });
    }   
    
    // Create the decrease link imageHyperlink widget.
    if (this.decreaseLink == null) {
        this.decreaseLink = this.widget.getWidgetProps("imageHyperlink", {
                "id": this.id + ":previousMonthLink",
                enabledImage: this.widget.getWidgetProps("image", {
                    border: 0,
                    icon: "SCHEDULER_BACKWARD",
                    id: this.id + ":previousMonthLink_image"
                }),
                title: this.theme.getMessage("CalendarMonth.goBack")
            });
    }        
    
    // Create the close button link imageHyperlink widget
    if (this.closeButtonLink == null) {
        this.closeButtonLink = this.widget.getWidgetProps("imageHyperlink", {
                id: this.id + ":closeButtonLink",
                enabledImage: this.widget.getWidgetProps("image", {
                    border: 0,
                    icon: "CALENDAR_CLOSE_BUTTON",
                    id: this.id + "closeButtonLink_close"
                }),
                title: this.theme.getMessage("CalendarMonth.close"),
                className: this.theme.getClassName("CALENDAR_CLOSE_BUTTON")            
            });    
    }
    
    // If the dateFormatPattern is null, get one from the themes.
    if (this.dateFormat == null) {
        this.dateFormat = this.theme.getMessage("calendar.dateFormat");
    }
    
    // If the minDate and maxDate are not specified, create a default values.
    // The minDate's year is set to 100 years previous to the current year
    // and maxDate is set to 200 years forward from the minDate's year'
    var minDate = new Date();
    var maxDate = new Date();
    if (this.minDate == null) {
        minDate.setFullYear(minDate.getFullYear() - 100);
        this.minDate = this.formatDate(minDate.getMonth(), 
                            minDate.getDate(), minDate.getFullYear());        
    } else {
        minDate = this.convertStringToDate(this.minDate);
    } 

    if (this.maxDate == null) {
        maxDate.setFullYear(minDate.getFullYear() + 200);
        this.maxDate = this.formatDate(maxDate.getMonth(), 
                            maxDate.getDate(), maxDate.getFullYear());
    } else {
        maxDate = this.convertStringToDate(this.maxDate);
    }             
  
    // Initialize the days of the week.
    if (this.weekDays == null) {
        this.weekDays = new Array();
        this.weekDays[0] = this.theme.getMessage("CalendarMonth.weekdaySun");
        this.weekDays[1] = this.theme.getMessage("CalendarMonth.weekdayMon");
        this.weekDays[2] = this.theme.getMessage("CalendarMonth.weekdayTue");                
        this.weekDays[3] = this.theme.getMessage("CalendarMonth.weekdayWed");
        this.weekDays[4] = this.theme.getMessage("CalendarMonth.weekdayThu");
        this.weekDays[5] = this.theme.getMessage("CalendarMonth.weekdayFri");
        this.weekDays[6] = this.theme.getMessage("CalendarMonth.weekdaySat");
    }           
    
    // Get the first day of week for that particular locale.
    if (this.firstDayOfWeek == null) {
        this.firstDayOfWeek = parseInt(this.theme.getMessage("calendar.firstDayOfWeek"));
    }
    
    // This will append a localized string along with the
    // today's date
    if (this.todayDateMsg == null) {        
        var d = new Date();
        var todayDateMsg = this.theme.getMessage("CalendarMonth.todayIs");
        
        // Remove the "$0" argument used for the server side param
        var index = todayDateMsg.indexOf(":");
        this.todayDateMsg = todayDateMsg.substr(0, index+1);
        
        var month = this.theme.getMessage(
                        "calendar." + (d.getMonth()));
        month=month.substr(0,3);
        if (this.dateFormat.indexOf("MM") == 0) {
            this.todayDateMsg += " " + month + " " + d.getDay();
        } else {
            this.todayDateMsg += " " + d.getDay() + " " + month;        
        }
        this.todayDateMsg += ", "+d.getFullYear();
    }

    // Initialize the month menu if one does not exist.
    if (this.monthMenu == null) {                  
        this.monthMenu = this.widget.getWidgetProps("dropDown", {
            id: this.id + ":monthMenu",
            options:this.getMonthOptions(),
            title: this.theme.getMessage("CalendarMonth.selectMonth")
        });                  
    }
    
    // Initialize the year menu if one does not exist.
    if (this.yearMenu == null) {
        this.yearMenu = this.widget.getWidgetProps("dropDown", {
            id: this.id + ":yearMenu",
            options: this.getYearOptions(minDate.getYear(), maxDate.getYear()),
            title: this.theme.getMessage("CalendarMonth.selectYear")   
        });          
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to obtain the Date object from the given string
 * date value
 *
 * @param {String} inputDate The date to be converted into Dataee object
 * @param {boolean} yearCheck Check whether the year falls within the specified range
 * @return {Object}  The Date object corresponding to the input String
 */
webui.suntheme4_2.widget.calendar.prototype.convertStringToDate = function(inputDate, yearCheck) {   
    if (inputDate == "") {
        property = null;
        return false;
    }
    
    var pattern = this.dateFormat;
    var yearIndex = pattern.indexOf("yyyy");
    var monthIndex = pattern.indexOf("MM");
    var dayIndex = pattern.indexOf("dd");

    // If the format is invalid, set the current value to null
    if (yearIndex < 0 || monthIndex < 0 || dayIndex < 0) {
        return null;
    } 
    
    var counter = 0;
    var number;
    var selectedDate = new Date();
    var found = 0;
    var dateString;

    while (counter < inputDate.length) {
        if (counter == yearIndex) {
            try {
                number = parseInt(inputDate.substr(counter, 4));
                if (isNaN(number)) {
                    property = null;
                    return false;
                }                
                // Check if the input date's year range is inbetween the 
                // allowed dates.   
                if (yearCheck == true) {
                    var index = 0;
                    var foundYear = false;                               
                    yearMenu = webui.suntheme4_2.dijit.byId(this.yearMenu.id).getSelectElement();
                    while (index < yearMenu.length) {
                        if (number == yearMenu.options[index].value) {
                            selectedDate.setFullYear(number);
                            ++found;
                            foundYear = true;
                            break;
                        }
                        index++;
                    }
                    if (!foundYear) {
                        break;
                    }
                } else {            
                    selectedDate.setFullYear(number);
                    ++found;
                }                    
            } catch(e) {}
        } else if (counter == monthIndex) {
            try {    
                dateString = inputDate.substr(counter, 2);
                // This is a workaround for Firefox! 
                // parseInt() returns 0 for values 08 and 09
                // while all other leading zeros work.
                if (dateString.charAt(0) == '0') {
                    dateString = dateString.substr(1, 1);
                }
                number = parseInt(dateString);
                if (isNaN(number)) {
                    property = null;
                    return false;
                }
                selectedDate.setMonth(number-1);
                ++found;
            } catch(e) {}
        } else if (counter == dayIndex) {
            try {
                dateString = inputDate.substr(counter, 2);
                // This is a workaround for Firefox! 
                // parseInt() returns 0 for values 08 and 09
                // while all other leading zeros work.
                if (dateString.charAt(0) == '0') {
                    dateString = dateString.substr(1, 1);
                }
                number = parseInt(dateString);
                if (isNaN(number)) {
                    return null;
                }
                selectedDate.setDate(number);
                ++found;
            } catch(e) {}
        }
        ++counter;
    }

    if (found == 3) {
        return selectedDate;
    } else {
        return null;
    }    
    return true;       
};

/**
 * Helper function to set the initial focus on the menus.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.setInitialFocus = function() {    
    var pattern = new String(this.dateFormat);
    var yearIndex = pattern.indexOf("yyyy");
    var monthIndex = pattern.indexOf("MM");
    
    // Moving the year menu around based on the date format is not supported yet.
    // So, the code for setting focus on the year menu has been commented out.
    // if (yearIndex < monthIndex) {        
    //    var yearMenu = document.getElementById(this.calendarMonth.yearMenu.id).getSelectElement();
    //    yearMenu.focus();                 
    // } else {
        var monthMenu = webui.suntheme4_2.dijit.byId(this.monthMenu.id).getSelectElement();          
        monthMenu.focus();
    // }
    return true;
};

/**
 * Set the value of an HTML select element, but limit value to min and max.
 *
 * @param {Node} select The HTML select element.
 * @param {String} value The selected value.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.setLimitedSelectedValue = function(select, value) {
    var min = select.options[0].value;
    var max = select.options[select.length - 1].value;
    if (value < min) {        
        select.value = min;
    } else if ( value > max) {        
        select.value = max;
    } else {
        this.setSelectedValue(select, value);        
    }
    return true;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {Object} closeButtonLink 
 * @config {String} date 
 * @config {String} dateFormat 
 * @config {Object} decreaseLink 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} increaseLink 
 * @config {Object} monthMenu
 * @config {String} style Specify style rules inline.
 * @config {Object} todayDateMsg
 * @config {Object} toggleLink
 * @config {boolean} visible Hide or show element.
 * @config {Object} yearMenu
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.calendar.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.        
    if (props.todayDateMsg) {
        // NOTE: If you set this value manually, text must be HTML escaped.
        this.widget.addFragment(this.todayDateContainer, props.todayDateMsg);
    }

    if (props.spacerImage) {
        if (!webui.suntheme4_2.dijit.byId(this.spacerImage.id)) {
            this.widget.addFragment(this.spacerImageContainer, props.spacerImage);
        }
    }

    if (props.topLeftImage) {
        if (!webui.suntheme4_2.dijit.byId(this.topLeftImage.id)) {
            this.widget.addFragment(this.topLeftImageContainer, props.topLeftImage);
        }
    }

    if (props.topRightImage) {
        if (!webui.suntheme4_2.dijit.byId(this.topRightImage.id)) {
            this.widget.addFragment(this.topRightImageContainer, props.topRightImage);
        }
    }

    if (props.date) {
        var selDate = this.convertStringToDate(props.date, true);
        if (selDate != null) {
            this.currentValue = selDate;
        }
    }

    // Set close link properties.
    if (props.closeButtonLink) {
        // Set properties.
        props.closeButtonLink.onClick = 
            "webui.suntheme4_2.dijit.byId('" + this.id + "').toggleCalendar();return false;";
        props.closeButtonLink.onKeyDown = 
            "webui.suntheme4_2.dijit.byId('" + this.id + "').closeCalendar(event);return false;";       

        // Update/add fragment.
        this.widget.updateFragment(this.closeButtonContainer, 
            this.closeButtonLink.id, props.closeButtonLink);
    }

    // Set decrease link properties.
    if (props.decreaseLink) {
        // Set properties.
        props.decreaseLink.onClick = 
            "webui.suntheme4_2.dijit.byId('" + this.id + "').decreaseMonth();return false;";

        // Update/add fragment.
        this.widget.updateFragment(this.previousLinkContainer, 
            this.decreaseLink.id, props.decreaseLink);
    }

    // Set increase link properties.
    if (props.increaseLink) {
        // Set properties.
        props.increaseLink.onClick = 
            "webui.suntheme4_2.dijit.byId('" + this.id + "').increaseMonth();return false;";

        // Update/add fragment.
        this.widget.updateFragment(this.nextLinkContainer, this.increaseLink.id, 
            props.increaseLink);
    }
    
    var minDate = null;
    var maxDate = null;    
    if (props.minDate || props.maxDate) {
        if (props.minDate) {
            
            // Convert the given string to a proper given date format pattern
            // and then store it.
            minDate = this.convertStringToDate(props.minDate);
            if (minDate != null) {
                this.minDate = this.formatDate(minDate.getMonth(), 
                    minDate.getDate(), minDate.getFullYear());
            }
        } 
        
        if (props.maxDate) {
            
            // Convert the given string to a proper given date format pattern
            // and then store it.            
            maxDate = this.convertStringToDate(props.maxDate);      
            if (maxDate != null) {
                this.maxDate = this.formatDate(maxDate.getMonth(), 
                    maxDate.getDate(), maxDate.getFullYear());
            }               
        } 
        
        //Recalculate the year options with new minDate and maxDate values.
        props.yearMenu = this.widget.getWidgetProps("dropDown", {
            id: this.id + ":yearMenu",
            options:this.getYearOptions(minDate.getFullYear(), maxDate.getFullYear()),
            title: this.theme.getMessage("CalendarMonth.selectYear")   
        });  
        
        // update the value of yearMenu
        this.yearMenu = props.yearMenu;                                          
    }
        
    // Set month menu properties
    if (props.monthMenu) {                        
        // Set properties.
        props.monthMenu.onChange =
            "webui.suntheme4_2.dijit.byId('" + this.id + "').updateMonth(false);return false;";
                         
        // Update/add fragment.
        this.widget.updateFragment(this.monthMenuContainer, this.monthMenu.id,
            props.monthMenu);
    }

    // Set year menu properties.
    if (props.yearMenu) {        
        // Set properties.
        props.yearMenu.onChange =
            "webui.suntheme4_2.dijit.byId('" + this.id + "').updateMonth(false);return false;";

        // Update/add fragment.
        this.widget.updateFragment(this.yearMenuContainer, this.yearMenu.id,
            props.yearMenu);
    }

    // Set toggle link properties.
    if (props.disabled != null) {
        this.disabled = new Boolean(props.disabled).valueOf(); 
    }

    // If the popup calendar is still being shown, prevent disabling of the calendar.
    // The widget can only be disabled if the popup calendar is not shown.
    if (props.toggleLink || 
        (props.disabled != null && this.calendarContainer.style.display != "block")) {

        // Ensure property exists so we can call setProps just once.
        if (props.toggleLink == null) {
            props.toggleLink = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.toggleLink.disabled = this.disabled;
        props.toggleLink.onClick =
            "webui.suntheme4_2.dijit.byId('" + this.id + "').toggleCalendar();return false;";

        // Update/add fragment.
        this.widget.updateFragment(this.linkNode, this.toggleLink.id, props.toggleLink); 
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * This function is used to set the value of a select element.
 *
 * @param {Node} select The HTML select element.
 * @param {String} value The selected value.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.setSelectedValue = function(select, value) {
    for (var i = 0; i < select.length; i++) {
        if (select.options[i].value == value) {
            select.selectedIndex = i;
            return true;
        }
    }
    select.selectedIndex = -1;
    return true;
};

/**
 * Process toogle event.
 *
 * @return {boolean} false to cancel JavaScript event.
 */
webui.suntheme4_2.widget.calendar.prototype.toggleCalendar = function() {
    var topic = (this.calendarContainer.style.display != "block")
        ? webui.suntheme4_2.widget.calendar.event.toggle.openTopic
        : webui.suntheme4_2.widget.calendar.event.toggle.closeTopic;

    // Publish an event for other widgets to listen for.
    //
    // Note: This must be done before the calendar is opened so user
    // input can be applied to the current date.
    this.publish(webui.suntheme4_2.widget.calendar.event.toggle.openTopic, [{
        id: this.id
    }]);

    // Open the calendar.
    if (this.calendarContainer.style.display != "block") {
        if (webui.suntheme4_2.widget.calendar.activeCalendarId != null) {
            var cal = webui.suntheme4_2.dijit.byId(webui.suntheme4_2.widget.calendar.activeCalendarId);
            cal.toggleCalendar();
        }
        webui.suntheme4_2.widget.calendar.activeCalendarId = this.id;        
        this.calendarContainer.style.display = "block";
        this.setInitialFocus();
        this.updateMonth(true);    
    } else {
        // Hide the calendar popup
        this.calendarContainer.style.display = "none";
        webui.suntheme4_2.widget.calendar.activeCalendarId = null;
    }

    // Test for IE 
    if (webui.suntheme4_2.browser.isIe5up()) {
        this.ieStackingContextFix();
    }          
    return false;
};

/**
 * This function is used to update the calendar month.
 * It is called when the calendar is opened, the next or previous
 * links are clicked, or the month or year menus are changed.
 *
 * @param {boolean} initialize Flag indicating to initialze the year and month menus
 * with the current value. The value is true only when the calendar is opened.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendar.prototype.updateMonth = function(initialize) {
    
    // Remove all the nodes of <tbody> before cloning its children.
    this.widget.removeChildNodes(this.tbodyContainer);    
    // Add week days
    this.addWeekDays();    
    
    // Add days of the month
    this.addDaysInMonth(this.currentValue, initialize);  
    return true;     
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.calendarField");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.calendar");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.textField");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.fieldBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.fieldBase
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for widgets that extend fieldBase.
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.fieldBase", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
        this.required = false;
        this.size = 20;
    },
    valid: true
});

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.fieldBase.prototype.getInputClassName = function() {   
    return null; // Overridden by subclass.
};

/**
 * Returns the HTML input element that makes up the text field.
 *
 * @return {Node} The HTML input element.
 */
webui.suntheme4_2.widget.fieldBase.prototype.getInputElement = function() {
    return this.fieldNode;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.fieldBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);
    
    // Set properties.
    if (this.alt) { props.alt = this.alt; }
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.label) { props.label= this.label; }
    if (this.maxLength > 0) { props.maxLength = this.maxLength; }    
    if (this.notify) { props.notify = this.notify; }
    if (this.submitForm != null) { props.submitForm = this.submitForm; }
    if (this.text != null) { props.text = this.text; }
    if (this.title != null) { props.title = this.title; }
    if (this.type) { props.type= this.type; }
    if (this.readOnly != null) { props.readOnly = this.readOnly; }
    if (this.required != null) { props.required = this.required; }
    if (this.size > 0) { props.size = this.size; }
    if (this.style != null) { props.style = this.style; }
    if (this.valid != null) { props.valid = this.valid; }
    
    // After widget has been initialized, get user's input.
    if (this.isInitialized() == true && this.fieldNode.value != null) {
        props.value = this.fieldNode.value;
    } else if (this.value != null) {
        props.value = this.value;
    }
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.fieldBase.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.fieldNode.id = this.id + "_field";
        this.fieldNode.name = this.id + "_field";
        this.labelContainer.id = this.id + "_label";
    }
    
    //initialize label
    if (this.label && this.label.value != null &&
	    !this.widget.isFragment(this.label)) {

	this.label = this.widget.getWidgetProps("label", this.label);
	this.label.id = this.labelContainer.id;
    }

    
    // Set public functions.
    this.domNode.getInputElement = function() { return webui.suntheme4_2.dijit.byId(this.id).getInputElement(); };
    
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.fieldBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.submitForm == false || props.submitForm == true) { 
        // connect the keyPress event
        this.dojo.connect(this.fieldNode, "onkeypress", this, "submitFormData");
    }
    if (props.maxLength > 0) { this.fieldNode.maxLength = props.maxLength; }
    if (props.size > 0) { this.fieldNode.size = props.size;  }
    if (props.value != null) { this.fieldNode.value = props.value; }
    if (props.title != null) { this.fieldNode.title = props.title; }   
    if (props.disabled != null) { 
        this.fieldNode.disabled = new Boolean(props.disabled).valueOf();
    }
    if (props.valid != null) { 
        this.valid = new Boolean(props.valid).valueOf();
        if (props.label == null) props.label = {};
        props.label.valid = this.valid;
    }
    if (props.required != null) { 
        this.required = new Boolean(props.required).valueOf();
        if (props.label == null) props.label = {};
        props.label.required = this.required;
    }
    if (props.readOnly != null) { 
        this.fieldNode.readOnly = new Boolean(props.readOnly).valueOf();
    }
    
    // Set label properties.  
    // If _setProps is called during initializat then we will be
    // creating the label and props.label == this.label.
    // If _setProps is called from setProps, then we are updating the
    // label. The label needs to be updated if 
    // required or valid properties are changed.
    // The application may also be creating a label after the
    // widget was created.
    //
    if (props.label) {
	// Now update or create the label.
	// If we don't have an existing label, this.label.id == null
	// then call addFragment in case the application is
	// creating the label after the selectBase widget was created.
	//
	if (this.label != null && this.label.id != null) {
	    this.widget.updateFragment(this.labelContainer, this.label.id,
		props.label);
	} else {
	    this.widget.addFragment(this.labelContainer, props.label);
	}
    }
      
    
    // Set HTML input element class name.
    this.fieldNode.className = this.getInputClassName();
    
    // Set more properties.
    this.setCommonProps(this.fieldNode, props);
    this.setEventProps(this.fieldNode, props);
    
    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * Process keyPress events on the field, which enforces/disables 
 * submitForm behavior.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.fieldBase.prototype.submitFormData = function(event) {
    if (event == null) {
        return false;
    }
    if (event.keyCode == event.KEY_ENTER) {               
        if (this.submitForm == false) {
            // Disable form submission.
            if (window.event) {
                event.cancelBubble = true;
                event.returnValue = false;
            } else{
                event.preventDefault();
                event.stopPropagation();
            }
            return false;
        } else {
            // Submit the form                    
            if (event.currentTarget.form) {
                event.currentTarget.form.submit();
            }
        }
    }
    return true;    
};

/**
 * @name webui.suntheme4_2.widget.textField
 * @extends webui.suntheme4_2.widget.fieldBase
 * @class This class contains functions for the textField widget.
 * @constructor This function is used to construct a textField widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.textField", webui.suntheme4_2.widget.fieldBase, {
    // Set defaults.
    constructor: function() {
        // Array of list values; may be empty; if null - then no autocomplete 
        // functionality is provided
        this.autoCompleteOptions = null; 
        this.autoCompleteSize = 15;
        this.autoCompleteCloseDelayTime = 100;
    },                  
    widgetName: "textField" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.textField.event =
        webui.suntheme4_2.widget.textField.prototype.event = {
  /**
   * This object contains filter event topics.
   * @ignore
   */
    autoComplete: {
        /** Filter event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textField_event_autoComplete_begin",

        /** Filter event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textField_event_autoComplete_end"
    },
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textField_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textField_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textField_event_submit_end"
    },

    /**
     * This object contains validation event topics.
     * @ignore
     */
    validation: {
        /** Validation event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textField_event_validation_begin",

        /** Validation event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textField_event_validation_end"
    }
};

/**
 * Utility function to adjust input and list widths to match
 * the one of surrounding domNode node    
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.adjustListGeometry = function () {
    this.listContainer.style.width = this.fieldNode.offsetWidth;
    this.listContainer.style.left = this.fieldNode.offsetLeft;
    this.listContainer.style.top = this.fieldNode.offsetTop + this.fieldNode.offsetHeight;
    this.listNode.style.width = this.fieldNode.offsetWidth;
    this.listContainer.style.zIndex = "999";
    return true;
};

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.textField.prototype.getInputClassName = function() {          
    // Set readOnly style.
    if (this.fieldNode.readOnly) {
        return this.widget.getClassName("TEXT_FIELD_READONLY", "");
    }

    // Apply invalid style.
    var validStyle =  (this.valid == false) 
        ? " " + this.widget.getClassName("TEXT_FIELD_INVALID", "")
        : " " + this.widget.getClassName("TEXT_FIELD_VALID", "");
    
    // Set default style.    
    return (this.disabled == true)
        ? this.widget.getClassName("TEXT_FIELD_DISABLED", "") 
        : this.widget.getClassName("TEXT_FIELD", "") + validStyle;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.textField.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.autoValidate != null) { props.autoValidate = this.autoValidate; }
    if (this.autoComplete != null) { props.autoComplete = this.autoComplete; } 
    if (this.autoCompleteOptions != null) { props.autoCompleteOptions = this.autoCompleteOptions; } //TODO clone array?
        
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.postCreate = function () {
    // Set events.
    if (this.autoValidate == true) {
        // Generate the following event ONLY when 'autoValidate' == true.
        this.dojo.connect(this.fieldNode, "onblur", this, "validate");
    }
    return this.inherited("postCreate", arguments);;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {boolean} autoValidate
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} maxLength 
 * @config {Array} notify 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {int} size 
 * @config {String} style Specify style rules inline.
 * @config {boolean} submitForm
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * Helper function to create callback for autoComplete list closing event.
 *
 * @return {Function} The callback function.
 */
webui.suntheme4_2.widget.textField.prototype.createCloseListCallback = function() {
    var _id = this.id;
    return function(event) { 
        var widget = webui.suntheme4_2.dijit.byId(_id);
        if (widget == null) {
            return false;
        }
        widget.showAutoComplete = false;
        widget.updateListView();
        return true;
    };
};

/**
 * Publishes event to filter the options list according to specified 
 * filter string. Note that in default implementation such options will be 
 * updated remotely by Ajax call and then are automatically refreshed on the page.
 *
 * @param {String} filter
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.filterOptions = function() {    
    // Publish the event for custom AJAX implementations to listen for.
    // The implementation of this Ajax call will retrieve the value of the filter
    // and will obtain an updated lookup list ( either locally, or by submit to the server)
    // Data returned from Ajax call will be pushed into this.listWidget 
    //
    // @see javascript.widget.jsfx.autoComplete for default Ajax implementation
    this.dojo.publish(webui.suntheme4_2.widget.textField.event.autoComplete.beginTopic, [{
        id: this.id
    }]);
    return true;        
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.textField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }
    
    //we can receive updated autoComplete flag from ajax call back
    if (props.autoComplete != null) { this.autoComplete = new Boolean(props.autoComplete).valueOf();  }
    
    //initialize list, if required    
    //autocomplete list may be initiated only on widget start, and cannot 
    //be introduced as part of dynamic options on setProps
    if (this.autoComplete && this.listWidget == null) {
        //create and populate props for listbox
        this.listWidgetProps = this.widget.getWidgetProps("listbox", {
           id: this.id + "_list",
           onFocus: "webui.suntheme4_2.dijit.byId('" + this.id + "').processFocusEvent(this.event);", 
           onBlur: "webui.suntheme4_2.dijit.byId('" + this.id + "').processBlurEvent(this.event);"
        });         
        //?? use of event registration as in following disables field processing keys 
        //onChange: "webui.suntheme4_2.dijit.byId('" + this.id + "').processListChange(this.event);"

        this.widget.addFragment(this.listContainer, this.listWidgetProps);
        
        //store reference to the list
        this.listWidget = webui.suntheme4_2.dijit.byId(this.listWidgetProps.id);
        this.listNode = this.listWidget.getSelectElement();
        
        //since original list box is created empty, make sure it is not shown
        this.updateListView();
  
         //disable browser autocomplete
         
         //we assume here that once the field is armed for autocomplete,
         //this mode will not get disabled. Thus there is no logic provided
         //for restoration of "autocomplete" attribute
         //
        this.fieldNode.setAttribute("autocomplete", "off");
        
        //use focus event to open the list
        this.dojo.connect(this.fieldNode, "onfocus", this, "processFocusEvent");
 
        //use blur events to close the list
        this.dojo.connect(this.fieldNode, "onblur", this, "processBlurEvent");        
         
        // onChange event of the list will change the content of the input field
        this.dojo.connect(this.listNode, "onchange", this, "processListChange");
 
        //onclick will allow to reopen options list after it has been closed with ESC
        this.dojo.connect(this.fieldNode, "onclick", this, "processFocusEvent");

        // input field changes will trigger updates to the autoComplete list options
        this.dojo.connect(this.fieldNode, "onkeyup", this, "processFieldKeyUpEvent");
        
        //additional logic applied to ENTER, ESCAPE, ARROWs on keydown in order to cancel the event bubble
        this.dojo.connect(this.fieldNode, "onkeydown", this, "processFieldKeyDownEvent");
    }        
    
    if (this.autoComplete && props.autoCompleteOptions != null && this.listWidget != null ) {
        //autoComplete param has changed
        
        //we can receive new options from ajax call
        this.autoCompleteOptions = props.autoCompleteOptions;
        
        // Push properties into listbox.  
        var listProps = {};
        listProps.options = props.autoCompleteOptions;
        
        //change list box size up to autoCompleteSize elem
        if (this.autoCompleteOptions.length > this.autoCompleteSize) {
            listProps.size = this.autoCompleteSize;
        } else if (this.autoCompleteOptions.length == 1) {
            //provide extra space for one-line
            listProps.size = 2;
        } else {
            listProps.size = this.autoCompleteOptions.length;
        }

        this.listWidget.setProps(listProps);
        
        /* // display list on initiation
        this.showAutoComplete = true;        
        */
        this.updateListView();
    }   

    // Set remaining properties.
    var ret = this.inherited("_setProps", arguments);
    
    if (props.autoComplete && props.autoCompleteOptions != null && this.listWidget != null ) {
        this.adjustListGeometry();  //even if autocomplete options are not defined in this set of props
    }
    return ret;
};

/**
 * Process the blur  event - activate delayed timer to close the list.
 * Such timer will be canceled if either list of text field receive 
 * the focus back within a delay period.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.processBlurEvent = function(event) {    
    //clear timeout for list closing, thereby preventing list from being closed    
    if (this.closingTimerId) {
        clearTimeout(this.closingTimerId);
        this.closingTimerId = null;
    }
    this.closingTimerId = setTimeout( 
        this.createCloseListCallback(), this.autoCompleteCloseDelayTime);   

    return true;
};

/**
 * Process keyPress events on the field, which enforces/disables 
 * submitForm behavior.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.processFieldKeyDownEvent = function(event) {
    event = this.widget.getEvent(event);
    if (event == null) {
        return false;
    }

    if (event.keyCode == this.widget.keyCodes.ENTER && this.autoCompleteIsOpen) {               
        // Disable form submission. Note that form submission is disabled
        // only once when listbox is open. If it closed, form will be submitted
        // according to submitForm flag
        if (window.event) {
            event.cancelBubble = true;
            event.returnValue = false;
        } else{
            event.preventDefault();
            event.stopPropagation();
        } 
        this.fieldNode.value = this.listWidget.getSelectedValue();

        //force close the list box
        this.showAutoComplete = false;
        this.updateListView();    
        
        return true;
    }   

    //close the list in case of ESCAPE pressed
    if (event.keyCode == this.widget.keyCodes.ESCAPE  ) {               
        this.fieldNode.value = ""; //Warning: IE empties all fields on the page on 2nd ESC independently of setting value here
        this.showAutoComplete = false;
        this.updateListView();     
        this.filterOptions();
        return true;
    }
      
    //even when the text field is in focus, we want arrow keys ( up and down)
    //navigate through options in the select list
    if (event.keyCode == this.widget.keyCodes.DOWN_ARROW && this.listWidget.getSelectedIndex() < this.listNode.options.length) {               
        try {
            this.showAutoComplete = true;

            if (!this.autoCompleteIsOpen || this.listNode.options.length <=1) {
                this.filterOptions();
                this.listWidget.setSelectedIndex(0) ;
            } else {     
                //list already open
                this.listWidget.setSelectedIndex(this.listWidget.getSelectedIndex() + 1) ;
                this.updateListView();     
                
            }
            this.processListChange(event);
            return true;
       } catch (doNothing) {}
    }
    if (event.keyCode == this.widget.keyCodes.UP_ARROW && this.listWidget.getSelectedIndex() > 0) {               
        try {
            this.showAutoComplete = true;

                this.listWidget.setSelectedIndex(this.listWidget.getSelectedIndex() - 1) ;
            this.processListChange(event);
            return true;
        } catch (doNothing) {}
    }
    return true;
};

/**
 * Process keyPress events on the filter, which chages the filter
 * and submits a request for updated list.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.processFieldKeyUpEvent = function(event) {    
    event = this.widget.getEvent(event);

    if (event != null &&
        ( event.keyCode == this.widget.keyCodes.ESCAPE ||
          event.keyCode == this.widget.keyCodes.ENTER ||
          event.keyCode == this.widget.keyCodes.DOWN_ARROW ||
          event.keyCode == this.widget.keyCodes.UP_ARROW  ||
          event.keyCode == this.widget.keyCodes.SHIFT  ||
          event.keyCode == this.widget.keyCodes.TAB
        )) {
        //these special keys are processed somewhere else - no filtering
        return false; 
    }
    this.showAutoComplete = true;
    this.filterOptions();
    return true;
};

/**
 * Process the focus event - turn on the flag for autoComplete list display
 * <p>
 * Displays autoComplete box on focus 
 * </p>
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.processFocusEvent = function(event) {
    //clear timeout for list closing, thereby preventing list from being closed
    if (this.closingTimerId) {
        clearTimeout(this.closingTimerId);
        this.closingTimerId = null;
    }
    return true;
};

/**
 * Process onChange event on the select list, which results in filter being
 * updated with the new value. Note that no data submission is initiated here.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.processListChange = function(event) {    
    event = this.widget.getEvent(event);

    if (event.type == "change") {               
        try {
            this.fieldNode.value = this.listWidget.getSelectedValue();
            
            //close the list
            this.showAutoComplete = false;
            this.updateListView();  
            this.fieldNode.focus(); 
                           
            return true;
       } catch (doNothing) {}
    }    

    // else - usually from selection movement with the arrow keys 
    this.fieldNode.value = this.listWidget.getSelectedValue();
    this.fieldNode.focus(); //keep the focus on filter field so that user can incrementally type additional data
    return true;
};

/**
 * Helper function to update the view of the component ( what is commonly known
 * in UI world as /refresh/ - but renamed not to be confused with Ajax refresh)
 *  - if size of the list box >=1, shows autocomplete list box.
 *  - if size of the list box <1, hides autocomplete list box.
 * 
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.updateListView = function() {
    if ( this.showAutoComplete == true && this.autoCompleteOptions.length >= 1) {
        //TODO a place for optimization here - not to adjust geometry each time
        this.adjustListGeometry();    

        //optionally we could add check for this.listNode.options.length >0
        this.listNode.className = this.widget.getClassName("TEXT_FIELD_AUTO_COMPLETE_LIST", "");

        // the following is preferred way of setting class, but it does not work
        //this.listWidget.setProps({className: this.widget.getClassName("TEXT_FIELD_AUTO_COMPLETE_LIST", "")}) ;
        
        //this.autoCompleteIsOpen flag indicates whether list box is open or not
        this.autoCompleteIsOpen = true;
    } else {
        this.listNode.className = this.theme.getClassName("HIDDEN");
        //this.listWidget.setProps(visible: 'false');
        this.autoCompleteIsOpen = false;
    }
    return true;
};

/**
 * Process validation event.
 * <p>
 * This function interprets an event (one of onXXX events, such as onBlur,
 * etc) and extracts data needed for subsequent Ajax request generation. 
 * Specifically, the widget id that has generated the event. If widget
 * id is found, publishBeginEvent is called with extracted data.
 * </p>
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.validate = function(event) {
    // Publish an event for custom AJAX implementations to listen for.
    this.publish(webui.suntheme4_2.widget.textField.event.validation.beginTopic, [{
        id: this.id
    }]);
    return true;
};

/**
 * @name webui.suntheme4_2.widget.calendarField
 * @extends webui.suntheme4_2.widget.textField
 * @class This class contains functions for the calendarField widget.
 * @constructor This function is used to construct a calendarField widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.calendarField", webui.suntheme4_2.widget.textField, {
    // Set defaults.
    widgetName: "calendarField" // Required for theme properties.
});

/**
 * This function is called when a day link is selected from the calendar.
 * It updates the field with the value of the clicked date.
 *
 * @param props Key-Value pairs of properties.
 * @config {String} id 
 * @config {String} date
 * @return {boolean} false to cancel JavaScript event.
 */
webui.suntheme4_2.widget.calendarField.prototype.dayClicked = function(props) {
    // Check whether the calendar associated with this particular calendarField
    // broadcasted the event.
    if (props.date != null && props.id == this.calendar.id) {
        // Set the selected date on the field.
        this.domNode.setProps({value: props.date});
    }
    return false;
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.calendarField.event =
        webui.suntheme4_2.widget.calendarField.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_calendarField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_calendarField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_calendarField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_calendarField_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.calendarField.prototype.getClassName = function() {
    // Set default style.
    var className = this.widget.getClassName("CALENDAR_ROOT_TABLE","");

    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.calendarField.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.  
    if (this.align) { props.align = this.align; }
    if (this.calendar) { props.calendar = this.calendar; }  
    if (this.patternHelp) { props.patternHelp = this.patternHelp; }   

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendarField.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.inlineHelpNode.id = this.id + "_pattern";
        this.linkContainer.id = this.id + "_linkContainer";
        this.calendarContainer.id = this.id + "_calendarContainer";
    }     

    // If a patternHelp is not specified by the developer
    // try to get one from the themes. 
    if (this.patternHelp == null) {
        
        var pattern;
        if (this.calendar != null && this.calendar.dateFormat != null) {
            pattern = this.calendar.dateFormat;
        } else {        
            pattern = this.theme.getMessage("calendar.dateFormat");
        }
        var help = this.theme.getMessage("calendar."+pattern);
        if (help != null) {
            this.patternHelp = help;
        } 
    }
    // Set events.

    // Subscribe to the "dayClicked" event present in the calendar widget.
    this.subscribe(webui.suntheme4_2.widget.calendar.event.day.selectedTopic,
        this, "dayClicked");
    // Subscribe to the "toggle" event that occurs whenever the calendar is opened.
    this.subscribe(webui.suntheme4_2.widget.calendar.event.toggle.openTopic,
        this, "toggleCalendar");
        
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {Object} calendar 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {Array} notify 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} patternHelp 
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid 
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendarField.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }
    
    // If the popup calendar is visible, prevent disabling of the calendar.
    // The widget can only be disabled if the popup calendar is not visible.
    if (props.disabled != null) { 
        var widget = webui.suntheme4_2.dijit.byId(this.calendar.id); 
        if (widget != null && !(widget.calendarContainer.style.display != "block")) {
            props.disabled = this.disabled;
        }        
    }
    
    // Set remaining properties.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.calendarField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set disabled.
    if (props.disabled != null) { this.disabled = new Boolean(props.disabled).valueOf(); }

    // Set calendar.
    if (props.calendar || props.disabled != null) {
        // Ensure property exists so we can call setProps just once.
        if (props.calendar == null) {
            props.calendar = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.calendar.disabled = this.disabled;
        
        // Update/add fragment.
        this.widget.updateFragment(this.calendarContainer, this.calendar.id,
            props.calendar); 
    }
    
    // Set date format pattern help.
    if (props.patternHelp) {                            
        // NOTE: If you set this value manually, text must be HTML escaped.
        this.widget.addFragment(this.inlineHelpNode, props.patternHelp);
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

// Cannot do this in the postCreate or setProps as the dom element hasnt yet been
// created on the page. So, offsetWidth would return zero. 
webui.suntheme4_2.widget.calendarField.prototype.startup = function () {
    
    // Adjust the size of the inline help text so that it fits to the
    // size of the text field     
    var width = this.fieldNode.offsetWidth;
    this.inlineHelpNode.style.cssText ="width:"+width+"px;";
    return this.inherited("startup", arguments);
};

/**
 * This function subscribes to the toggleCalendar function of the calendar widget.
 * Whenever the calendar is opened, it updates the value of the calendar with
 * the value present in the field.
 * 
 * @param props Key-Value pairs of properties.
 * @config {String} id
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.calendarField.prototype.toggleCalendar = function(props) {   
    if (props.id != null && props.id == this.calendar.id) {
        var widget = webui.suntheme4_2.dijit.byId(props.id);
        widget.setProps({date: this.getProps().value});
    }
    return true;
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.checkbox");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.checkedBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.checkedBase
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for widgets that extend checkedBase.
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.checkedBase", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    idSuffix: "" // Overridden by subclass
});

/**
 * Helper function to obtain image class names.
 *
 * @return {String} The HTML image element class name.
 */
webui.suntheme4_2.widget.checkedBase.prototype.getImageClassName = function() {
    return null; // Overridden by subclass.
};

/**
 * Helper function to obtain input class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.checkedBase.prototype.getInputClassName = function() {
    return null; // Overridden by subclass.
};

/**
 * Returns the HTML input element that makes up the chekcbox.
 *
 * @return {Node} The HTML input element. 
 */
webui.suntheme4_2.widget.checkedBase.prototype.getInputElement = function() {
    return this.inputNode;
};

/**
 * Helper function to obtain label class names.
 *
 * @return {String} The HTML label element class name.
 */
webui.suntheme4_2.widget.checkedBase.prototype.getLabelClassName = function() {
    return null; // Overridden by subclass.
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.checkedBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.  
    if (this.align) { props.align = this.align; }
    if (this.disabled != null) { props.disabled = this.disabled; }   
    if (this.image) { props.image = this.image; }
    if (this.label) { props.label = this.label; }
    if (this.name) { props.name = this.name; }        
    if (this.readOnly != null) { props.readOnly = this.readOnly; }
    if (this.value) { props.value = this.value; }

    // After widget has been initialized, get user's input.
    if (this.isInitialized() == true && this.inputNode.checked != null) {
        props.checked = this.inputNode.checked;
    } else if (this.checked != null) {
        props.checked = this.checked;
    }
    return props;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkedBase.prototype.onClickCallback = function(event) {
    if (this.readOnly == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the request.
    var result = (this.domNode._onclick)
        ? this.domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkedBase.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.inputNode.id = this.id + this.idSuffix;
        this.imageContainer.id = this.id + "_imageContainer";
        this.labelContainer.id = this.id + "_labelContainer";

        // If null, use HTML input id.
        if (this.name == null) {
            this.name = this.inputNode.id;
        }
    }

    // Set public functions.
    this.domNode.getInputElement = function() { return webui.suntheme4_2.dijit.byId(this.id).getInputElement(); }
    
    // Create callback function for onclick event.
    this.dojo.connect(this.domNode, "onclick", this, "onClickCallback");

    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.checkedBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.value) { 
        this.inputNode.value = props.value;
    }
    if (props.readOnly != null) { 
        this.inputNode.readOnly = new Boolean(props.readOnly).valueOf();       
    }
    if (props.disabled != null) {
        this.inputNode.disabled = new Boolean(props.disabled).valueOf();        
    }
    
    // Set HTML input element class name.
    this.inputNode.className = this.getInputClassName();
    
    if (props.name) { 
        this.inputNode.name = props.name;
    }

    if (props.checked != null) {
        var checked = new Boolean(props.checked).valueOf();

        // Dynamically setting the checked attribute on IE 6 does not work until
        // the HTML input element has been added to the DOM. As a work around, 
        // we shall use a timeout to set the property during initialization.
        if (this.isInitialized() != true &&
                webui.suntheme4_2.browser.isIe()) {
            var _id = this.id;
            setTimeout(function() {
                // New literals are created every time this function
                // is called, and it's saved by closure magic.
                var widget = webui.suntheme4_2.dijit.byId(_id);
                widget.inputNode.checked = checked;
            }, 0); // (n) milliseconds delay.
        } else {
            this.inputNode.checked = checked;
        }
    }

    // Set image properties.
    if (props.image || props.disabled != null && this.image) {     
        // Ensure property exists so we can call setProps just once.
        if (props.image == null) {
            props.image = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.image.className = this.getImageClassName();

        // Update/add fragment.
        this.widget.updateFragment(this.imageContainer, this.image.id, props.image);
    } 

    // Set label properties.
    if (props.label || props.disabled != null && this.label) {     
        // Ensure property exists so we can call setProps just once.
        if (props.label == null) {
            props.label = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.label.className = this.getLabelClassName();

        // Update/add fragment.
        this.widget.updateFragment(this.labelContainer, this.label.id, props.label);
    }

    // Set more properties.
    this.setCommonProps(this.inputNode, props);
    this.setEventProps(this.inputNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * @name webui.suntheme4_2.widget.checkbox
 * @extends webui.suntheme4_2.widget.checkedBase
 * @class This class contains functions for the checkbox widget.
 * @constructor This function is used to construct a checkbox widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.checkbox", webui.suntheme4_2.widget.checkedBase, {
    // Set defaults.
    idSuffix: "_cb",
    widgetName: "checkbox" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.checkbox.event =
        webui.suntheme4_2.widget.checkbox.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_checkbox_event_refresh_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_checkbox_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_checkbox_event_state_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_checkbox_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_checkbox_event_submit_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_checkbox_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.checkbox.prototype.getClassName = function() {
    // Set default style.
    var className = (this.disabled == true)
        ? this.widget.getClassName("CHECKBOX_SPAN_DISABLED", "")
        : this.widget.getClassName("CHECKBOX_SPAN", ""); 

    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * Helper function to obtain image class names.
 *
 * @return {String} The HTML image element class name.
 */
webui.suntheme4_2.widget.checkbox.prototype.getImageClassName = function() {
    return (this.disabled == true)
        ? this.widget.getClassName("CHECKBOX_IMAGE_DISABLED", "")
        : this.widget.getClassName("CHECKBOX_IMAGE", "");  
};

/**
 * Helper function to obtain input class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.checkbox.prototype.getInputClassName = function() {
    // readOnly style.
    if (this.readOnly == true) {
        return this.widget.getClassName("CHECKBOX_READONLY", "");        
    }

    // disabled style.
    return (this.disabled == true)
        ? this.widget.getClassName("CHECKBOX_DISABLED", "")
        : this.widget.getClassName("CHECKBOX", "");  
};

/**
 * Helper function to obtain label class names.
 *
 * @return {String} The HTML label element class name.
 */
webui.suntheme4_2.widget.checkbox.prototype.getLabelClassName = function() {
    return (this.disabled == true)
        ? this.widget.getClassName("CHECKBOX_LABEL_DISABLED", "")
        : this.widget.getClassName("CHECKBOX_LABEL", "");  
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {boolean} checked 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} image 
 * @config {String} label 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect 
 * @config {boolean} readOnly 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkbox.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.checkboxGroup");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.checkedGroupBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.checkedGroupBase
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for widgets that extend checkedGroupBase.
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.checkedGroupBase", webui.suntheme4_2.widget.widgetBase);

/**
 * Helper function to add elements with Object literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled 
 * @config {Array} columns 
 * @config {Array} contents 
 * @config {Object} label
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkedGroupBase.prototype.addContents = function(props) {   
    if (props == null) {
        return false;
    }
    
    if (props.contents) {
        this.widget.removeChildNodes(this.tbodyContainer);
        var rowContainerCloneNode = this.rowContainer.cloneNode(false);
        this.tbodyContainer.appendChild(rowContainerCloneNode);
        if (props.label) {
            var rowNodeClone = this.rowNode.cloneNode(false);
            rowContainerCloneNode.appendChild(rowNodeClone);
            var labelContainerClone = this.labelContainer.cloneNode(false);
            rowNodeClone.appendChild(labelContainerClone);
            this.widget.addFragment(labelContainerClone, props.label, "last");            
              
        }
        var itemN = 0;
        var length = props.contents.length;
        var columns = (props.columns <= 0) ? 1 : props.columns;
        var rows = (length + (columns-1))/columns;
        var propsdisabledvalue = props.disabled == null ? false : props.disabled;
        for (var row = 0; row <= rows; row++) {
            for (var column = 0; column < columns; column++) {
                if (itemN < length) {
                    // Clone < td> node.
                    var contentsRowNodeClone = this.contentsRowNode.cloneNode(false);
                    rowContainerCloneNode.appendChild(contentsRowNodeClone);
                    // Set disabled.                   
                    props.contents[itemN].disabled = propsdisabledvalue;
                   
                    // Add child to the group.
                    this.widget.addFragment(contentsRowNodeClone, props.contents[itemN], "last");
                    itemN++;
                }
            }
            if (row + 1 <= rows) {
                rowContainerCloneNode = this.rowContainer.cloneNode(false);
                this.tbodyContainer.appendChild(rowContainerCloneNode);
                // This check is required here. Else the elements won't be
                // aligned properly when there is no label.
                if (props.label != null) {
                    var contentsRowNodeClone = this.contentsRowNode.cloneNode(false);
                    rowContainerCloneNode.appendChild(contentsRowNodeClone);
                }
              
            }
        }
    } else {
        // Update the disabled property client side
        if (props.disabled != null && this.contents) {
            for (var i = 0; i < this.contents.length; i++) {
                var contentWidget = webui.suntheme4_2.dijit.byId(this.contents[i].id);
                if (contentWidget) {
                    contentWidget.setProps({disabled: props.disabled});
                }
            }
        }
    }
    return true;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.checkedGroupBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.     
    if (this.columns) { props.columns = this.columns; }
    if (this.contents) { props.contents = this.contents; }    
    if (this.disabled != null) { props.disabled = this.disabled; }   
    if (this.id) { props.id = this.id; }
    if (this.label) { props.label = this.label; }    
    if (this.name) { props.name = this.name; }
    if (this.readOnly != null) { props.readOnly = this.readOnly; }  

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkedGroupBase.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {                    
        this.contentsRowNode.id = this.id + "_contentsRowNode";
        this.divContainer.id = this.id + "_divContainer";
        this.labelContainer.id = this.id + "_labelContainer";                            
        this.rowContainer.id = this.id + "_rowContainer";
        this.rowNode.id = this.id + "_rowNode";
        this.tableContainer.id = this.id + "_tableContainer";   
        this.tbodyContainer.id = this.id + "_tbodyContainer";     
    }

    // Show label.
    if (this.label) {
        this.common.setVisibleElement(this.rowNode, true);
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {int} columns 
 * @config {Array} contents 
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {boolean} readOnly Set button as primary if true.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkedGroupBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.checkedGroupBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set label properties.
    if (props.label) {
        // Update/add fragment.
        this.widget.updateFragment(this.labelContainer, this.label.id, props.label);
    }

    // Set contents.    
    if (props.contents || props.disabled != null) {              
        this.addContents(props);   
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * @name webui.suntheme4_2.widget.checkboxGroup
 * @extends webui.suntheme4_2.widget.checkedGroupBase
 * @class This class contains functions for the checkboxGroup widget.
 * @constructor This function is used to construct a checkboxGroup widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.checkboxGroup", webui.suntheme4_2.widget.checkedGroupBase, {
    // Set defaults.
    widgetName: "checkboxGroup" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.checkboxGroup.event =
        webui.suntheme4_2.widget.checkboxGroup.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_checkboxGroup_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_checkboxGroup_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_checkboxGroup_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_checkboxGroup_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.checkboxGroup.prototype.getClassName = function() {
    // Set default style.
    var className = (this.columns > 1)
        ? this.widget.getClassName("CBGRP_HORIZ", "")
        : this.widget.getClassName("CBGRP_VERT", "");

    return (this.className)
        ? className + " " + this.className
        : className;
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.dndContainer");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.dnd");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

 /**
  * @name webui.suntheme4_2.widget.dndContainer
  * @extends webui.suntheme4_2.widget.widgetBase
  * @class This class contains functions for the dndContainer widget
  * @constructor This function is used to construct a dndContainer widget.
  */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.dndContainer", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    widgetName: "dndContainer" // Required for theme properties.
});

/**
 * Helper function to create callback for user's node creator function
 *
 * @param {String} name of the function to be called for node creation, with 
 * signature function(data, hint).
 * @return {Function} function to be called for node creation.
 */
webui.suntheme4_2.widget.dndContainer.prototype.createCreatorCallback = function(funcName) {
    var dragTypes = this.dragTypes ? this.dragTypes : "";
    var dragSource = this.dragSource;
    var func = new Function("data", "hint", "return " + funcName + "(data, hint)");

    // This function will invoke user provided creator function (onNodeCreateFunc) 
    // and will add default dnd type to the item created by it. All items 
    // created within this container will automatically be associated with the 
    // type provided by the container - a container's 'dragTypes' property.
    //
    // The function returns the same data structure as required to be returned 
    // by the user's creator function.
    //
    // {
    //  node: DOM node that will be inserted.
    //  data: Data that will be associated with newly created node.
    //  type: Drag type array that will be associated with newly created node.
    // }; 
    //
    return function(data, hint) { 
        if (func == null)
            return null;
        var ret = func(data, hint);
        
        if (ret != null) {
            // Add type to it.
            ret.type = dragTypes;
        } else {
            // User func did not create node - create one with default creator.
            ret = dragSource.defaultCreator(data, hint);
        }
        return ret;
    };
};

/**
 * Helper function to create callback for user's onDrop function.
 *
 * @return {Function} function to be called upon drop.
 */
webui.suntheme4_2.widget.dndContainer.prototype.createOnDndDropCallback = function() {
    var containerWidget = this;
 
    return function(source, nodes, copy){
        // Call user's onDropFunction
        if (containerWidget.onDropFunc) {
            try {
                var func = eval(containerWidget.onDropFunc);
                return func(source, nodes, copy);
            } catch (error) {           
                //do nothing        
            }
        }
        return true;
    };
};

/**
 * Helper function to obtain HTML container element class names.
 *
 * @return {String} calculated class name of the container node.
 */
webui.suntheme4_2.widget.dndContainer.prototype.getContainerClassName = function() {   
    // Add default style.
    return  this.dndContainer.className;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.dndContainer.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);
    
    // Set properties.
    if (this.dropTypes != null) { props.dropTypes= this.dropTypes; }
    if (this.contents)          { props.contents = this.contents; }
    if (this.contentsDragData)  { props.contentsDragData = this.contentsDragData; }
    if (this.copyOnly != null)  { props.copyOnly = this.copyOnly; }
    if (this.onDropFunc)        { props.onDropFunc = this.onDropFunc; }
    if (this.onNodeCreateFunc)  { props.onNodeCreateFunc = this.onNodeCreateFunc; }
    if (this.dragTypes != null) { props.dragTypes = this.dragTypes; }
    if (this.style != null)     { props.style = this.style; }
    if (this.title != null)     { this.dndContainer.title = this.title;}
    
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.dndContainer.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.dndContainer.id = this.id + "_container";
    }
    
    // Make things draggable.
    var params = {};
    if (this.onNodeCreateFunc) {
        params.creator = this.createCreatorCallback(this.onNodeCreateFunc);                   
    }
    if (this.copyOnly) {
        params.copyOnly = this.copyOnly;
    }
    if (this.dropTypes) {        
        params.accept = (this.dropTypes instanceof Array) 
            ? this.dropTypes : this.dropTypes.split(',');
    }
    if (this.horizontalIndicator != null) {        
        params.horizontal = this.horizontalIndicator;
    }
    if (this.dragTypes == null) {        
        params.isSource = false;
    }
    params.onDropFunction = this.createOnDndDropCallback();
    
    this.dragSource = new webui.suntheme4_2.dnd.Source(this.dndContainer, params);

    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @config {Array} dragTypes list of space-trimmed strings representing 
 * types of the drag element.
 * @config {String} contents children of the container.
 * @config {String} className CSS selector.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} onNodeCreateFunc Javascript code to create new item.
 * @config {Array} dropTypes list of space-trimmed strings accepted by this 
 * container as a drop.
 * @config {String} style Specify style rules inline.
 * @config {boolean} visible Hide or show element.
 * @return {Boolean} true if operation was successfull, false otherwise.
 */
webui.suntheme4_2.widget.dndContainer.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }
    
    // Replace contents.
    if (props.contents) {
        this.contents = null;
    }
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.dndContainer.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }      
    if (props.dragTypes) {        
        this.dragTypes = (props.dragTypes instanceof Array)
            ? props.dragTypes : props.dragTypes.split(',');        
    }
    if (props.dropTypes) {        
        this.dropTypes = (props.dropTypes instanceof Array)
            ? props.dropTypes : props.dropTypes.split(',');
    }
    if (props.onDropFunc) {
        this.onDropFunc =  props.onDropFunc; 
    }
    
    // Set container class name.
    this.dndContainer.className = this.getContainerClassName();
    
    // Set contents.         
    // Assert there is a dragData and id entry for each fragment
    if (props.contents && props.contentsDragData 
            && props.contents.length == props.contentsDragData.length 
            && this.dragSource) {                   
        // Remove child nodes.
        this.widget.removeChildNodes(this.dndContainer);
        
	for (var i = 0; i < props.contents.length; i++) {
            if (this.dragTypes) {
                // For each item in the contents create a span placeholder using
                // normalized creator function (this will allow for consistent 
                // internal placement of container elements within container)
                // which will be an element of the container and at the same 
                // time will contain the output of addFragment. Add the rendered
                // content into the span.             
                var node = this.dragSource.addItem([""], this.dragTypes, 
                    props.contentsDragData[i]); //empty data content
                this.widget.addFragment(node, props.contents[i], "last");
            } else {
                // Simply add data, without making it draggable
                this.widget.addFragment(this.dndContainer, props.contents[i], "last");            
            }
        }
    }
    
    // Set more properties.
    this.setCommonProps(this.dndContainer, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.dropDown");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.selectBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.label");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.label
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the label widget.
 * @constructor This function is used to construct a label widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.label", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.level = webui.suntheme4_2.widget.common.getMessage("label.level", null, 2);
        this.required = false;
        this.valid = true;
    },
    widgetName: "label" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.label.event =
        webui.suntheme4_2.widget.label.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_label_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_label_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_label_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_label_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.label.prototype.getClassName = function() {
    var key = "LABEL_LEVEL_TWO_TEXT";

    if (this.valid == false) {
        key = "CONTENT_ERROR_LABEL_TEXT";
    } else if (this.level == 1) {
        key = "LABEL_LEVEL_ONE_TEXT";
    } else if (this.level == 3) {
        key = "LABEL_LEVEL_THREE_TEXT";
    }

    // Get theme property.
    var className = this.theme.getClassName(key);
    if (className == null || className.length == 0) {
	return this.className;
    }
    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to process notification events with Object
 * literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} detail Message detail text.
 * @config {boolean} valid Flag indicating validation state.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.label.prototype.notify = function(props) {
    if (props == null) {
        return false;
    }
    return this.setProps({
        valid: props.valid,
        errorImage: {
            title: props.detail
        }
    });
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.label.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.contents) { props.contents = this.contents; }
    if (this.errorImage) { props.errorImage = this.errorImage; }
    if (this.htmlFor) { props.htmlFor = this.htmlFor; }
    if (this.level != null) { props.level = this.level; }
    if (this.required != null) { props.required = this.required; }
    if (this.requiredImage) { props.requiredImage = this.requiredImage; }
    if (this.valid != null) { props.valid = this.valid; }
    if (this.value) { props.value = this.value; }

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.label.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.contentsContainer.id = this.id + "_contentsContainer";
        this.errorImageContainer.id = this.id + "_errorImageContainer";
        this.requiredImageContainer.id = this.id + "_requiredImageContainer";
        this.valueContainer.id = this.id + "_valueContainer";
    }

    // If errorImage or requiredImage are null, create images from the theme.
    // When the _setProps() function is called, image widgets will be
    // instantiated via the props param. 
    if (this.errorImage == null) {
	this.errorImage = this.widget.getWidgetProps("image", {
            icon: "LABEL_INVALID_ICON",
            id: this.id + "_error",
	    className: this.widget.getClassName("LABEL_INVALID_IMAGE", null)
        });
    }
    if (this.requiredImage == null) {
	this.requiredImage = this.widget.getWidgetProps("image", {
            icon: "LABEL_REQUIRED_ICON",
            id: this.id + "_required",
	    className: this.widget.getClassName("LABEL_REQUIRED_IMAGE", null)
        });
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey
 * @config {String} className CSS selector.
 * @config {String} contents 
 * @config {String} dir Specifies the directionality of text.
 * @config {Object} errorImage 
 * @config {String} htmlFor 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} level 
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} primary Set button as primary if true.
 * @config {boolean} required
 * @config {Object} requiredImage
 * @config {String} style Specify style rules inline.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.label.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.label.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.htmlFor) { this.domNode.htmlFor = props.htmlFor; }
    if (props.valid != null) { this.valid = new Boolean(props.valid).valueOf(); }
    if (props.required != null) { this.required = new Boolean(props.required).valueOf(); }
    if (props.value) { this.widget.addFragment(this.valueContainer, props.value); }

    // Set error image properties.
    if (props.errorImage || props.valid != null) {
        // Ensure property exists so we can call setProps just once.
        if (props.errorImage == null) {
            props.errorImage = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.errorImage.visible = !this.valid;

        // Update/add fragment.
        this.widget.updateFragment(this.errorImageContainer, this.errorImage.id, 
            props.errorImage);
    }

    // Set required image properties.
    if (props.requiredImage || props.required != null) {
        // Ensure property exists so we can call setProps just once.
        if (props.requiredImage == null) {
            props.requiredImage = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.requiredImage.visible = this.required;

        // Update/add fragment.
        this.widget.updateFragment(this.requiredImageContainer, 
            this.requiredImage.id, props.requiredImage);
    }

    // Set contents.
    if (props.contents) {
        // Remove child nodes.
        this.widget.removeChildNodes(this.contentsContainer);

	for (var i = 0; i < props.contents.length; i++) {
            this.widget.addFragment(this.contentsContainer, props.contents[i], "last");
        }
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * @name webui.suntheme4_2.widget.selectBase
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class defines functions and properties for
 * widgets based on the "select" HTML element.
 *
 * <h3>Dojo attach points</h3>
 * A <code>selectBase</code>  subclass's template must
 * define the following attach point identifiers.
 * <ul>
 * <li><code>listContainer</code> - the <code>select</code> element.</li>
 * <li><code>brNode</code> - the element that controls the label postion.
 * <li><code>optionNode</code> - the element to clone for an
 * <code>option</code> element.
 * </li>
 * <li><code>optionGroupContainer</code> - the element to clone for an
 * <code>optGroup</code> element.</li>
 * <li><code>memberOptionNode</code> - the element to clone for an
 * <code>option</code> in an <code>optGroup</code> element.</li>
 * </ul>
 * <h3>The <code>label</code> object property</h3>
 * The <code>label</code> property is an object that defines the properties
 * for a widget that is rendered by the <code>selectBase</code>
 * class to represent a label. Minimally, only the <code>value</code> 
 * property of the label object property must be non null for 
 * the <code>selectBase</code> widget to render a label. If only the
 * <code>value</code> property is specified the following default
 * values will be used to create an instance of 
 * <code>webui.suntheme4_2.widget.label</code>.
 * <p>
 * <ul>
 * <li><code>widgetType</code> -
 * <code>webui.suntheme4_2.widget.label</code></li>
 * <li><code>id</code> - this.id + "_label"</li>
 * <li><code>htmlFor</code> - this.listContainer.id</li>
 * </ul>
 * <p>See <code>postCreate</code> and <code>getLabelProps</code>
 * </p>
 * <h3>The <code>options</code> array property</h3>
 * The <code>options</code> array property defines the contents of the
 * <code>select</code> HTML element. The contents of the array are 
 * translated into <code>option</code> and <code>optGroup</code> HTML
 * elements. Each element of the <code>options</code> array property
 * is considered an object with the following properties. Note that
 * "NA" in the following table means "not applicable", meaning there
 * is no HTML element property counterpart.
 * <p>
 * <table border="1px">
 * <tr><th>Property</th><th>Type</th><th>Description</th>
 * <th>HTML option or optGroup element property</th></tr>
 * <tr><td>label</td><td>String</td><td>
 * The text that appears as the choice in the rendered <code>option</code>
 * HTML element. (See note[1] below)
 * </td><td>label</td></tr>
 * <tr><td>value</td><td>String</td><td>
 * The value for this option. It is submitted in a request if this option is
 * selected.</td><td>value</td></tr>
 * <tr><td>separator</td><td>boolean</td><td>
 * If true this option represents a separator and cannot be selected.
 * </td><td>NA</td></tr>
 * <tr><td>group</td><td>boolean</td><td>
 * If true this option represents a group and cannot be selected. An
 * <code>optGroup</code> HTML element is rendered to represent this option.
 * </td><td>NA</td></tr>
 * <tr><td>escape</td><td>boolean</td><td>
 * If false the label string will be evaluated as HTML markup.
 * </td><td>NA</td></tr>
 * <tr><td>selected</td><td>boolean</td><td>
 * If true this option will be initially selected. 
 * </td><td>selected</td></tr>
 * <tr><td>disabled</td><td>boolean</td><td>
 * If true this option will be initially disabled and cannot be selected.
 * </td><td>disabled</td></tr>
 * <tr><td>title</td><td>String</td><td>
 * The HTML title attribute text.</td><td>title</td></tr>
 * </table>
 * </p>
 * <p>
 * The option object may also define the javascript event properties
 * suitable for an <code>option</code> or <code>optGroup</code> HTML
 * element and they will be assigned to the element's DOM node.
 * </p>
 * <p>
 * <ul>
 * <li>Note[1] Browser runtime behavior, contrary to the HTML and DOM
 * specifications, indicates that the DOM <code>label</code> property
 * is not rendered and only the <code>text</code> property is rendered.
 * This means that the <code>label</code> property of an
 * <code>options</code> array element will
 * be assigned to an <code>option</code> or <code>optGroup</code>
 * DOM node's <code>text</code> property.</li>
 * </ul>
 * </p>
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.selectBase", 
	webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
        this.required = false;
        this.valid = true;

        // Flag to remember that last label style class.
        this._lastLabelOnTopClassName = null; // never been set

        // Flag to prevent blank entries in the drop down for the original 
        // empty dojo attach point nodes -- see "setOptions".
        this._alreadyRemoved = false;
    }
});

/**
 * This function is called by the <code>_onChangeCallback</code>
 * function to set the option element's className attribute with appropriate
 * selected and disabled selectors.
 * <p>
 * This method calls <code>this.getOptionClassName</code> once for each
 * option. Subclasses should implement <code>getOptionClassName</code>
 * in order to assign the appropriate CSS class name, to reflect a disabled or
 * selected option and any other properties due to an onChange event.
 * <p>
 * See <code>getOptionClassName</code> for the default behavior.
 * </p>
 * <p>
 * Before this method calls <code>this.getOptionClassName</code>, it has
 * addressed an IE issue where disabled options can be selected, and has
 * deslected, selected disabled options.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.selectBase.prototype.changed = function(ev) {
    var options = this.listContainer.options;

    // IE allows disabled options to be selected. Ensure that
    // disabled options never appear as selected options.
    //
    if (webui.suntheme4_2.browser.isIe()) {
	for (var i = 0; i < options.length; ++i) {
	    if (options[i].disabled == true && options[i].selected == true) {
		if (this.listContainer.multiple == true) {
		    options[i].selected = false;
		} else {
		    // Don't we need to set the disabled option's
		    // selected to false too ?
		    //
		    this.listContainer.selectedIndex = -1;
		}
	    }
	    // Only set the option if its value is different
	    // than what is returned. Note that if this method
	    // returns null, the option may not visually reflect
	    // the true state.
	    //
	    var cn = this.getOptionClassName(options[i]);
	    if (cn != null && options[i].className != cn) {
		options[i].className = cn;
	    }
	} 
    } else {
	for (var i = 0; i < options.length; ++i) { 
	    // Only set the option if its value is different
	    // than what is returned. Note that if this method
	    // returns null, the option may not visually reflect
	    // the true state.
	    //
	    var cn = this.getOptionClassName(options[i]);
	    if (cn != null && options[i].className != cn) {
		options[i].className = cn;
	    }
	}
    }
    return true;
};

/**
 * This method copies option properties from <code>fromOption</code> to 
 * <code>toOption</code>. Note that <code>fromOption</code> is assumed to
 * be a DOM <code>option</code> element. There is not a one to one mapping
 * of properties by name from a DOM <code>option</code> element to a
 * widget's <code>options</code> array <code>option</code> array element.
 * The properties that are set in <code>setOptionProps</code> and
 * <code>setGroupOptionProps</code> are copied.
 * @param {Object} toOption An <code>Object</code> to receive
 * <code>selectBase</code> option properties.
 * @param {Object} fromOption An <code>option</code> or <code>optgroup</code>
 * element DOM node to copy <code>selectBase</code> option properties from.
 * @return {boolean} Returns true.
 * @private
 */
webui.suntheme4_2.widget.selectBase.prototype._copyOption = function(toOption, fromOption) {
    var domhandlers = [ "onblur", "onchange", "onclick", "ondblclick",
	"onfocus", "onkeydown", "onkeypress", "onkeyup", "onmousedown",
	"onmouseout", "onmouseover", "onmouseup", "onmousemove", "onresize"];

    var widgethandlers = [ "onBlur", "onChange", "onClick", "onDblClick",
	"onFocus", "onKeyDown", "onKeyPress", "onKeyUp", "onMouseDown",
	"onMouseOut", "onMouseOver", "onMouseUp", "onMouseMove", "onResize"];

    for (var i = 0; i < domhandlers.length; ++i) {
	if (fromOption[domhandlers[i]]) {
	    toOption[widgethandlers[i]] = fromOption[domhandlers[i]];
	}
    }
    toOption.label = fromOption.label;
    toOption.value = fromOption.value;
    toOption.escape = fromOption.escape;
    toOption.disabled = fromOption.disabled;
    toOption.defaultSelected = fromOption.defaultSelected;
    toOption.selected = fromOption.selected;
    toOption.title = fromOption.title;
    toOption.className = fromOption.className;
    toOption.group = fromOption.group;
    if (toOption.group == true) {
	toOption.options = this._copyOptions(fromOption);
    }
    return true;
};

/**
 * Returns an <code>options</code> array of <code>option</code> objects
 * as defined by <code>selectBase</code> (see selectBase.setOptions).
 * <code>domNode</code> is assumed to be a <code>select</code> element
 * or an <code>optgroup</code>.
 * <p>
 * There is not a one to one mapping of properties by name from a DOM
 * <code>option</code> or <code>optgroup</code> element to a
 * widget's <code>options</code> array <code>option</code> object array element.
 * The properties that are set in <code>setOptionProps</code> and 
 * <code>setGroupOptionProps</code> are copied to <code>Object</code>s
 * and returned as the elements of the returned array. The returned
 * array is equivalent to the <code>options</code> array used to create
 * a selectBase subclass, like listbox or dropdown.
 * @param {Object} domNode A <code>select</code> element DOM node or an
 * <code>optgroup</code> DOM node.
 * @return {Object} An array of <code>selectBase</code> options. If 
 * <code>domNode</code> has no child nodes, an empty array is returned.
 * @private
 */
webui.suntheme4_2.widget.selectBase.prototype._copyOptions = function(domNode) {
    var newoptions = [];
    if (!domNode.hasChildNodes()) {
	return newoptions;
    }
    var len = domNode.childNodes.length;
    for (var j = 0; j < len; ++j) {
	newoptions[j] = new Object();
	if (domNode.childNodes != null) {
	    this._copyOption(newoptions[j], domNode.childNodes[j]);
	}
    }
    return newoptions;
};

/**
 * This function returns the CSS class name for an HTML <code>option</code> or
 * <code>optgroup</code> element.
 * <p>
 * This implementation returns <code>option.className</code>
 * This method should be overridden by subclasses.
 * </p>
 *
 * @param {Object} option Key-Value pairs of properties.
 * @return {String} The HTML option element class name.
 */
webui.suntheme4_2.widget.selectBase.prototype.getOptionClassName = function(option) {
    // Make sure that if a subclass does not implement this method
    // that it causes no change to option.
    //
    return option.className;
};

/**
 * This function replaces all <code>option</code> and <code>optgroup</code>
 * elements in the HTML <code>select</code> element,
 * <code>this.listContainer</code>.
 * <p>
 * All options are replaced with the options specified in <code>props</code>.
 * If <code>props</code> is null, <code>false</code>
 * is returned and no change to the <code>select</code> element occurs. If
 * <code>props</code> contains no properties all options in the 
 * <code>select</code> element are removed.
 * </p>
 * @param {Object} props Key-Value pairs of option properties.
 * @config {String} label The text that appears as the choice in the 
 * rendered <code>option</code>.
 * @config {String} value The value for this option, which is submitted
 * in a request if this option is selected.
 * @config {boolean} separator If true this option represents a separator and
 * cannot be selected.
 * @config {boolean} group If true this option represents a group and
 * cannot be selected.
 * @config {boolean} escape If false the label string will be evaluated as
 * HTML markup.
 * @config {boolean} selected If true this option will be initially
 * selected.
 * @config {boolean} disabled If true this option will be initially 
 * disabled and cannot be selected.
 * @config {String} title The HTML title attribute text.
 * @return {boolean} true if successful; otherwise, false.
 * </p>
 * <p>
 * The option object may also define javascript event properties suitable for
 * an <code>option</code> or <code>optGroup</code> HTML element.
 * </p>
 */
webui.suntheme4_2.widget.selectBase.prototype.setOptions = function(props) {
    if (props == null) {
	return false;
    }

    // Remove all existing options
    //
    // Note: this has to be done. Otherwise, you'll get blank entries in the 
    // drop down for the original empty dojo attach point nodes.
    //

    if (!this._alreadyRemoved) {
        this.listContainer.removeChild(this.optionNode); 
        this.optionGroupContainer.removeChild(this.memberOptionNode);
        this.listContainer.removeChild(this.optionGroupContainer);
        this._alreadyRemoved = true;
    }

    // Cleaning up the old options
    //
    while (this.listContainer.firstChild) {
        this.listContainer.removeChild(this.listContainer.firstChild);
    }

    var thisNode;
    for (var i = 0; i < props.options.length; i++) {

	var pOption = props.options[i];

	var isie = webui.suntheme4_2.browser.isIe();
	if (pOption.group == null || pOption.group == false) {

	    // For some reason, ie is prone to painting problems (esp. after a 
	    // style change) when using the DOM to populate options, but 
	    // apparently not when the options are in a group
	    //
	    // There is no working "clone" on "option" nodes in IE
	    // Manually get the attributes and then the "text"
	    // DOM attribute. I don't think an option can have HTML
	    // markup in the body, at least literally in the template
	    // so this strategy should be be sufficient enough to get
	    // the template's option attributes
	    //
	    if (isie) {
		thisNode = new Option();
		var len = this.optionNode.attributes.length;
		for (var j = 0; j < len; ++j) {
		    thisNode.setAttribute(this.optionNode.attributes[j].name,
			    this.optionNode.attributes[j].value);
		}
		// Need to manually do text, although this is likely null
		// in the template.
		//
		thisNode.text = this.optionNode.text;

	    } else {
		thisNode = this.optionNode.cloneNode(true);
	    }

	    // Set the properties on this option element
	    this.setOptionProps(thisNode, pOption);
	    
	    // Would it be better to create all the nodes and then
	    // add them to the "options" array or append then ?
	    // Granted this would mean iterating twice.
	    // But it might be better if something fails
	    // and if so leave the original list alone.
	    //

	    // Append this option node to the select
	    //
	    if (isie) {
		var idx = this.listContainer.options.length;
		var isSelected = thisNode.selected;
		this.listContainer.options[idx] = thisNode;

		// VERIFY THAT WE STILL NEED THIS
		//

		// explicitly set the selected property again!
		// this is necessary to work around a defect in 
		// some versions of IE6
		//
		this.listContainer.options[idx].selected = isSelected;
	    } else {
		this.listContainer.appendChild(thisNode);
	    }
	} else {
	    // group option optgroup
	    //
	    thisNode = this.optionGroupContainer.cloneNode(true);
	    
	    // Set the properties on this optgroup element
	    //
	    this.setGroupOptionProps(thisNode, pOption);
	    
	    // Add the option elements to this group
	    // Supports only one level
	    //
	    var thisSubNode;
	    for (var ix = 0; ix < pOption.options.length; ix++) {
		thisSubNode = this.memberOptionNode.cloneNode(true);
		this.setOptionProps(thisSubNode, pOption.options[ix]);
		thisNode.appendChild(thisSubNode); 
	    }

	    // Append this optgroup node to the select element
	    // only after constructing its option children
	    //
	    this.listContainer.appendChild(thisNode);
	}
    }
    return true;
};

/**
 * This function is used to get widget properties. Please see 
 * subclass's setProps() function for a complete list of supported properties.
 * <br/>
 * This base class handles the following properties.
 * <ul>
 * <li><code>disabled</code> - If true the select element is disabled.</li>
 * <li><code>label</code> - This object defines the widget and properties
 * for a label.</li>
 * <li><code>labelOnTop</code> - If true the label appears above the select
 * element.</li>
 * <li><code>options</code> - This array object contains the select element 
 * options and their attributes. See <code>setOptions</code> and the 
 * <code>selectBase</code> overview for details on the <code>options</code> 
 * array.</li>
 * <li><code>required</code> - If true the a selection is required.</li>
 * <li><code>valid</code> - If true the widget has a valid selection.</li>
 * <li><code>width</code> - This value will be assigned to the 
 * <code>listContainer.style.width</code> attribute to set the width of the
 * select element.</li>
 * </ul>
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.selectBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Get properties.
    // Should we return the default theme value for labelOnTop
    // if "this.labelOnTop" has not been set ?
    //
    if (this.labelOnTop != null) { props.labelOnTop = this.labelOnTop; }
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.label) { props.label = this.label; }
    if (this.required != null) { props.required = this.required; }
    if (this.valid != null) { props.valid = this.valid; }
    if (this.width != null) { props.width = this.width; }

    // After widget has been initialized, get actual select element state
    // Note that the options that exist on the select element will
    // not include the original "options" props that are not
    // HTML option or HTML optgroup attributes.
    //
    if (this.isInitialized() == true && this.listContainer != null) {
	// We need to copy the options from this.listContainer
	// or else they will be destroyed if passed back to addProps.
	//
	var opts = this._copyOptions(this.listContainer);
        props.options = opts;
	// It's not clear if we should do this.
	//
	// this.options = opts;
    } else if (this.options != null) {
        props.options = this.options;
    }
    return props;
};

/**
 * This function is used to obtain the underlying HTML select element.
 *
 * @return {Node} The HTML select element.
 */
webui.suntheme4_2.widget.selectBase.prototype.getSelectElement = function() {
    return this.listContainer;
};

/**
 * This function is used to obtain the index 
 * of the selected option.
 *
 * @return {Integer} The selected index of underlying select element
 */
webui.suntheme4_2.widget.selectBase.prototype.getSelectedIndex = function() { 
    return this.listContainer.selectedIndex; 
};

/**
 * This function is used to directly set selected index on the underlying select box
 * of the selected option.
 *
 * @return {Boolean} true
 */
webui.suntheme4_2.widget.selectBase.prototype.setSelectedIndex = function(index) { 
    if (index >=0 && index < this.listContainer.options.length) {
        this.listContainer.selectedIndex = index;
    }
    return true;
};

/**
 * This function is used to obtain the <code>label</code> attribute of
 * the selected option.
 * <p>
 * If no option is selected, this function returns null.
 * </p>
 * <p>
 * If the underlying select element's <code>multiple</code> attribute is true,
 * this method returns the first selected option's label.
 * </p>
 * 
 * @return The label of the selected option, or null if none is selected. 
 * @return {String} The label attribute of the selected option.
 */
webui.suntheme4_2.widget.selectBase.prototype.getSelectedLabel = function() { 
    var index = this.listContainer.selectedIndex; 

    if (index == -1) { 
        return null; 
    } else { 
	return this.listContainer.options[index].label;
    }
};

/**
 * This function is used to obtain the <code>value</code> attribute
 * of the selected option.
 * <p>
 * If no option is selected, this function returns null. 
 * </p>
 * <p>
 * If the underlying select element's <code>multiple</code>" attribute is true,
 * this method returns the first selected option's value.
 * </p>
 *
 * @return {String} The selected option value or null if none is selected. 
 */
webui.suntheme4_2.widget.selectBase.prototype.getSelectedValue = function() { 
    var index = this.listContainer.selectedIndex; 
    if (index == -1) { 
        return null; 
    } else { 
        return this.listContainer.options[index].value; 
    }
};

/**
 * This function is invoked for the select element's <code>onchange</code>
 * event.
 * <p>
 * If <code>this.disabled</code> is true, return false.<br/>
 * This method calls the handler defined by the <code>props.onChange</code>
 * property. If that handler returns false, false is returned. If the
 * <code>props.onChanged</code> handler returns true, then
 * <code>this.changed</code> is called and its return value is returned.
 * </p>
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.selectBase.prototype._onChangeCallback = function(event) {
    if (this.disabled == true) {
        return false;
    }

    // If function returns false, we must prevent the auto-submit.
    //
    var result = (this.listContainer._onchange)
        ? this.listContainer._onchange(event) : true;
    if (result == false) {
        return false;
    }

    // Set style classes.
    return this.changed(event);
};

/**
 * This function is used to fill in remaining template properties, after the
 * <code>buildRendering</code> function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.selectBase.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.labelContainer.id = this.id + "_label";
        this.listContainer.id = this.id + "_list";
	this.listContainer.name = this.listContainer.id;
    }

    // Subclasses have set this property
    // make the br node visible else hide it.
    //
    if (this.labelOnTop != null) {
	webui.suntheme4_2.common.setVisibleElement(this.brNode, this.labelOnTop);
	this._lastLabelOnTopClassName = this.getLabelClassName(null);
    }

    if (this.label && this.label.value != null &&
	    !this.widget.isFragment(this.label)) {

	this.label = this.widget.getWidgetProps("label", this.getLabelProps());
	this.label.id = this.id + "_label";
    }

    if (this.label != null) {
	if (this.required != null && this.required == true) {
	    this.label.required = true;
	}
	if (this.valid != null && this.valid != true) {
	    this.label.valid = false;
	}
	if (this._lastLabelOnTopClassName != null) {
	    webui.suntheme4_2.common.addStyleClass(this.label, 
		this._lastLabelOnTopClassName);
	}
    }

    // Set public functions.
    this.domNode.getSelectedValue = function() { 
	return webui.suntheme4_2.dijit.byId(this.id).getSelectedValue(); };
    this.domNode.getSelectedLabel = function() { 
	return webui.suntheme4_2.dijit.byId(this.id).getSelectedLabel(); };
    this.domNode.getSelectedIndex = function() { 
	return webui.suntheme4_2.dijit.byId(this.id).getSelectedIndex(); };
    this.domNode.setSelectedIndex = function(index) { 
	return webui.suntheme4_2.dijit.byId(this.id).setSelectedIndex(index); };
    this.domNode.getSelectElement = function() { 
	return webui.suntheme4_2.dijit.byId(this.id).getSelectElement(); };

    // Set events.
    this.dojo.connect(this.listContainer, "onchange", this, 
	"_onChangeCallback");

    return this.inherited("postCreate", arguments);
};

/**
 * Return an Object Literal of label properties desired
 * by the selectBase widget.
 * <p>
 * This implementation returns null. This method should be implemented
 * in subclasses to return label properties desired by the subclass.
 * </p>
 * @return {Object} This implementation returns null;
 */
webui.suntheme4_2.widget.selectBase.prototype.getLabelProps = function() {
    return null;
};

/**
 * This function is used to set the <code>option</code> properties on the
 * <code>optgroup</code> HTML DOM element, <code>element</code>.
 * <p>
 * This method assigns the <code>option.label</code>,
 * <code>option.disabled</code>, <code>option.title</code> properties
 * to <code>element</code>, non HTML properties defined in 
 * <code>option</code>, calls <code>this.setEventProps</code>
 * and then calls <code>this.getOptionClassName</code>. 
 * Subclasses should implement <code>getOptionClassName</code> if the default
 * behavior of <code>selectBase.getOptionClassName</code> is not appropriate.
 * Subclasses usually subclass this method to provide subclass specific 
 * CSS class names and properties. For all the properties defined in the 
 * option array see <code>setOptions</code>.
 * </p>
 * @param {Node} element The optgroup DOM node
 * @param {Object} option Key-Value pairs of properties for the optgroup node.
 * @config {boolean} disabled If true the optgroup is disabled.
 * @config {String} label The optgroup choice text.
 * @config {String} title The optgrp HTML title attribute value.
 * @config {boolean} escape If false the label string will be evaluated
 * as HTML markup.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.selectBase.prototype.setGroupOptionProps =
	function(element, option) {
    element.label = option.label;
  
    if (option.disabled != null) {
        element.disabled = option.disabled;
    }

    if (option.title != null) {
	element.title = option.title;
    }

    // optGroup elements can have events.
    this.setEventProps(element, option);

    // Non HTML properties
    //
    if (option.escape != null) {
	element.escape = option.escape;
    }
    element.group = true;

    // Do this last so that all other properties will have been set.
    // Note that if this does return null, which it shouldn't
    // then the visual appearance may not reflect the 
    // actual state.
    //
    var cn =  this.getOptionClassName(element);
    if (cn != null) {
	element.className = cn;
    }
    return true;
};

/**
 * This function is used to set the <code>option</code> properties on the
 * <code>option</code> HTML DOM element, <code>element</code>.
 * <p>
 * This method assigns the <code>option.label</code>,
 * <code>option.disabled</code>, <code>option.title</code> properties
 * to <code>element</code>, non HTML properties defined in 
 * <code>option</code>, calls <code>this.setEventProps</code>
 * and then calls <code>this.getOptionClassName</code>. 
 * Subclasses should implement <code>getOptionClassName</code> if the default
 * behavior of <code>selectBase.getOptionClassName</code> is not appropriate.
 * Subclasses usually subclass this method to provide subclass specific CSS
 * class names and properties. For all the properties defined in the option
 * array see <code>setOptions</code>
 * </p>
 * @param {Node} element The <code>option</code> HTML element DOM node
 * @param {Object} option Key-Value pairs of properties for the option node.
 * @config {boolean} disabled If true the option is disabled.
 * @config {String} label The option choice text.
 * @config {String} title The value of the option's HTML <code>title</code>
 * attribute.
 * @config {boolean} escape If false the label string will be evaluated
 * as HTML markup.
 * @config {boolean} separator If true this option acts as a separator.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.selectBase.prototype.setOptionProps =
        function(element, option) {
    element.value = option.value;

    // If option.escape is true, we want the text to be displayed
    // literally. To achieve this behavior, do nothing.
    // If option.escape is false, we want any special sequences in the text 
    // (e.g., "&nbsp;") to be displayed as evaluated (i.e., unescaped).
    //
    // This is because it is assumed that the value of the property has 
    // already been escaped. This may be true if the value was set in the 
    // HTML response. This is why we call "unescapeHTML".
    //
    // However it may not be true if the text was set in javascript.
    //

    // Note that the HTML and ECMA JS DOM language binding indicates
    // that the "label" property should be used. However browsers do not
    // respect those specs. In fact label does not even work at all.
    // Setting the string on label results in no string being displayed.
    //
    var textToUse = option.label;
    if (option.escape != null && option.escape == false) {
	// Prototype method.
	element.text = this.prototypejs.unescapeHTML(textToUse);
    } else {
	element.text = textToUse;
    }
    // We must record the state of the original option
    //
    element.label = option.label;

    // Never allow a disabled option to be spedified as selected
    // This is only a problem if an option is initially selected
    // and disbled on most browsers since once disabled it cannot
    // be selected except for IE. For IE it is ensured that a
    // disabled option cannot be "selected" in the
    // "change" method.
    //
    if (option.disabled != null && option.disabled == true) {
	option.selected = false;
    }

    element.defaultSelected = option.selected != null ? option.selected : false;

    // When creating new options, defaultSelected should also be set.
    // Actually only "defaultSelected" should be set and set only once
    // since it is supposed to be the "initial" value.
    // However all options are always recreated in this implementation
    // when this method is called by setOptions.
    //
    element.selected = element.defaultSelected;
	
    element.disabled = option.disabled != null ? option.disabled : false;

    if (option.title != null) {
	element.title = option.title;
    }

    // option elements can have events.
    this.setEventProps(element, option);

    // In order for subclasses to return the appropriate
    // selector all the properties in the option properties array must
    // be placed on the HTML element option and optgroup instances.
    //
    // This is because "getOptionClassName" is called in two different
    // contexts. It is called here where both the HTML option or
    // optgroup element is available and the original option properties 
    // are known, and during the "onchange" event where only the 
    // HTML option element is known. Therefore if the choice of selector 
    // is based on whether or not an option is a "separator" then, 
    // the separator state must be available from the HTML option or optgroup
    // element instance.
    //
    // This issue could be mitigated by using descendant selectors
    // vs. literal selectors. In other words, if every time an option
    // became disabled, the className attribute would be set (or merged) with 
    // Dis. A selector could be defined as ".Lst option.Dis" which would
    // implicitly match. If attribute selectors worked we could use
    // that instead. Either strategy would eliminate the need for
    // methods like "getOptionClassName".
    //
    // Only set non HTML element props if we have to.
    //
    if (option.separator != null && option.separator == true) {
	element.separator = option.separator;
	element.disabled = true;
    }
    if (option.escape != null && option.escape == false) {
	element.escape = option.escape;
    }
    element.group = false;

    // Call getOptionClassName only after setting HTML props
    // and option props on element.  Note that if this method
    // returns null, the option may not visually reflect
    // the true state.
    //
    var cn = this.getOptionClassName(element);
    if (cn != null) {
	element.className = cn;
    }
    return true;
};

/**
 * This function is used to set widget properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled If true the select element is disabled.
 * @config {Object} label The properties for a label widget.
 * @config {boolean} labelOnTop If false the label will appear to the left
 * of the <code>select</code> element, aligned with the first list option. 
 * @config {Object} options An array of Objects that represent the 
 * <code>select</code> element's <code>option</code> and 
 * <code>optgroup</code> elements. @see #setOptions.
 * @config {boolean} required If true the a selection is required.
 * @config {boolean} valid If true the widget has a valid selection.
 * @config {String} width This value will be assigned to the 
 * <code>listContainer.style.width</code> attribute to set the width of the
 * select element.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.selectBase.prototype.setProps = function(props) {
    // Always call inherited setProps
    //
    if (props == null) {
	return null;
    }

    // Set a flag indicating there is a label
    //
    var havelabel = this.label != null && this.label.id != null;

    // Always update the brNode state, even if there isn't a label
    // but only if it is changing.
    //
    var labelProps = null;
    if (props.labelOnTop != null && props.labelOnTop != this.labelOnTop) {

	// The ontop state is changing
	// If the state is "ontop" then make the br node 
	// visible else hide it.
	//
	webui.suntheme4_2.common.setVisibleElement(this.brNode, props.labelOnTop);

	// Remember the new ontop selector.
	//
	this._lastLabelOnTopClassName = this.getLabelClassName(
	    props.labelOnTop);

	// If we have a label remove the last ontop selector from the label.
	// We must strip it from the dom node.
	//
	if (havelabel) {

	    var labelnode = document.getElementById(this.label.id);
	    webui.suntheme4_2.common.stripStyleClass(labelnode,
		this._lastLabelOnTopClassName);

	    // If we are toggling ontop for an existing label or the
	    // application is updating or creating a label, add the
	    // correct ontop selector to props.
	    //
	    var labelProps = props.label != null 
		? props.label 
		: (props.label = {});

	} else if (props.label && this._lastLabelOnTopClassName != null) {
	    labelProps = props.label;
	    webui.suntheme4_2.common.addStyleClass(labelProps, 
		this._lastLabelOnTopClassName);
	}
    }

    var togglerequired = props.required != null && 
	props.required != this.required;
    var togglevalid = props.valid != null && props.valid != this.valid;

    // Update the label required and valid properties if necessary.
    // If labelProps is not null, then the labelOnTop property is changing.
    // If labelProps is null, then if there is an existing label and
    // the required or valid attributes is changing, make sure label.props
    // is updated with the new required and valid states.
    //
    if (labelProps == null && havelabel && (togglerequired || togglevalid)) {
	labelProps = props.label != null ? props.label : (props.label = {});
    }

    if (labelProps != null) {
	if (togglerequired) {
	    labelProps.required = props.required;
	}
	if (togglevalid) {
	    labelProps.valid = props.valid;
	}
    }
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {Object} label The properties for a label widget.
 * @config {Object} options An array of Objects that represent the 
 * <code>select</code> element's <code>option</code> and 
 * <code>optgroup</code> elements. @see #setOptions.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.selectBase.prototype._setProps = function(props) {
    // Always call inherited _setProps
    //
    if (props == null) {
	return null;
    }

    // A web app devleoper could return false in order to cancel the 
    // auto-submit. Thus, we will handle this event via the onChange
    // call back.
    //
    if (props.onChange) {
        // Set private function scope on DOM node.
        this.listContainer._onchange = (typeof props.onChange == 'string')
	    ? new Function("event", props.onChange) 
	    : props.onChange;

        // Must be cleared before calling setEventProps() below.
        props.onChange = null;
    }

    if (props.disabled != null) {
	if (this.listContainer.disabled != props.disabled) {
	    this.listContainer.disabled = props.disabled;
	}
    }

    // Add option and optgroup elements in the select element
    //
    if (props.options) {
        this.setOptions(props);
    }

    if (props.width != null && props.width != "") {
	this.listContainer.style.width = props.width;
    }


    // If _setProps is called during initializat then we will be
    // creating the label and props.label == this.label.
    // If _setProps is called from setProps, then we are updating the
    // label. The label needs to be updated if the labelOnTop or
    // or required or valid properties are changed.
    // The application may also be creating a label after the
    // selectBase widget was created.
    //
    if (props.label) {
	// Now update or create the label.
	// If we don't have an existing label, this.label.id == null
	// then call addFragment in case the application is
	// creating the label after the selectBase widget was created.
	//
	if (this.label != null && this.label.id != null) {
	    this.widget.updateFragment(this.labelContainer, this.label.id,
		props.label);
	} else {
	    this.widget.addFragment(this.labelContainer, props.label);
	}
    }

    // Set more properties.
    this.setCommonProps(this.listContainer, props);
    this.setEventProps(this.listContainer, props);

    // Set remaining properties.
    //
    return this.inherited("_setProps", arguments);
};

/**
 * @name webui.suntheme4_2.widget.dropDown
 * @extends webui.suntheme4_2.widget.selectBase
 * @class This class contains functions for the dropDown widget.
 * @constructor This function is used to construct a dropDown widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.dropDown", webui.suntheme4_2.widget.selectBase, {
    // Set defaults.
    constructor: function() {
        this.labelOnTop = webui.suntheme4_2.widget.common.getMessageBoolean(
	    "dropDown.labelOnTop", false);
        this.submitForm = false;
        this.width = webui.suntheme4_2.theme.common.getMessage("dropDown.width", null);
    },
    widgetName: "dropDown" // Required for theme properties.
});

/**
 * This function is called by the <code>selectBase::onChangeCallback</code>
 * method in respsonse to an onchange event.
 * <p>
 * If this instance is configured as a "jump" menu,
 * <code>submitForm</code> is true, the containing form is submitted.
 * </p>
 * <p>
 * If this instance is configured as a standard "dropdown" menu, then
 * the superclass's <code>changed</code> method is called.
 * See <code>selectBase.changed</code>.<br/>
 * See <code>selectBase.onChangeCallback</code>
 * </p>
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.dropDown.prototype.changed = function() {

    // A null submitForm is the same as a standard menu.
    //
    if (this.submitForm != true) {
        // Drop down changed.
        return this.inherited("changed", arguments);
    } else {

        // Jump drop down changed.
        var jumpDropdown = this.listContainer; 

        // Find the <form> for this drop down
        var form = jumpDropdown.form; 

        if (typeof form != "undefined" && form != null) { 
            this.submitterHiddenNode.value = "true";

            // Set style classes.
            var options = jumpDropdown.options;
            for (var i = 0; i < options.length; i++) { 
                options[i].className = this.getOptionClassName(options[i]);
            }
            form.submit();
        }
    }
    return true; 
};

/**
 * The dropDown defines the following event topics. The submit topics
 * are only valid if the <code>submitForm</code> property is true.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.dropDown.event =
        webui.suntheme4_2.widget.dropDown.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for.*/
        beginTopic: "webui_suntheme4_2_widget_dropDown_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for.*/
        endTopic: "webui_suntheme4_2_widget_dropDown_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_dropDown_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_dropDown_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for.
	 * This topic is only valid if the <code>submitForm</code>
	 * property is <code>true</code>
	 */
        beginTopic: "webui_suntheme4_2_widget_dropDown_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for.
	 * This topic is only valid if the <code>submitForm</code>
	 * property is <code>true</code>
	 */
        endTopic: "webui_suntheme4_2_widget_dropDown_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element CSS class name.
 * <p>
 * Note: Selectors should be concatenated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.dropDown.prototype.getClassName = function() {

    var cn = this.widget.getClassName("DROPDOWN", "");
    return (this.className) ? cn + " " + this.className : cn;
};

/**
 * Return an Object Literal of label properties desired
 * by the dropdown widget.
 * <p>
 * This implementation sets
 * the <code>dropDown.labelLevel</code> and <code>dropDown.labelAlign</code>
 * theme values from the <code>messages</code> and <code>styles</code>
 * theme categories to the
 * label's <code>level</code> and <code>className</code> properties 
 * respectively. It sets the <code>htmlFor</code> attribute to 
 * <code>listContainer.id</code>.
 * </p>
 * <p>
 * These properties are extended with <code>this.label</code> and the
 * resulting properties are returned.
 * </p>
 * @return {Object} label properties.
 */
webui.suntheme4_2.widget.dropDown.prototype.getLabelProps = function() {

    var cn = this.getLabelClassName(null);
    var lvl = this.widget.getMessage("dropDown.labelLevel", null, 2);
    var props = {};
    if (cn != null) {
       props.className = cn;
    }
    props.labelLevel = lvl;
    props.htmlFor = this.listContainer.id;
    this.prototypejs.extend(props, this.label);
    return props;
};

/**
 * Return a CSS selector for the dropDown label.
 * The chosen CSS selector is a function of 
 * the <code>labelOnTop</code> property and the whether or not the
 * dropdown is a "jump" dropDown. If <code>labelOnTop</code>
 * property is <code>true</code> null is returned. If the 
 * <code>labelOnTop</code> property is true, then the value of the
 * the of the theme keys, <code>MENU_JUMP_LABEL_ALIGN</code> or 
 * <code>MENU_STANDARD_LABEL_ALIGN</code> from the 
 * <code>styles</code> category are returned if the drop down is
 * a "jump" dropDown or plain dropDown, respectively.
 * @param {boolean} ontop The label on top state. In some contexts this state
 * must be passed to the method, for example when called from the 
 * <code>selectBase.setProps</code> method.
 * @return {String} A CSS selector for the dropDown label.
 */
webui.suntheme4_2.widget.dropDown.prototype.getLabelClassName = function(ontop) {

    var labelontop = ontop != null ? ontop : this.labelOnTop;
    if (labelontop == true) {
	return null;
    }
    // Jumpmenu ?
    //
    return (this.submitForm != null && this.submitForm == true)
	? this.widget.getClassName("MENU_JUMP_LABEL_ALIGN", null)
        : this.widget.getClassName("MENU_STANDARD_LABEL_ALIGN", null);
};

/**
 * Return a CSS selector for listContainer HTML element.
 * The chosen CSS selector is a function of 
 * the <code>disabled</code> property and whether or not the dropDown
 * is a "jump" dropDown. The returned selector is the value of a 
 * property defined in the <code>styles</code> category, as shown 
 * in the following table.
 * <table border="1px">
 * <tr><th>key</th><th>disabled</th><th>jump dropDown</th></tr>
 * <tr>
 * <td>MENU_STANDARD</td>
 * <td>false</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>MENU_STANDARD_DISABLED</td>
 * <td>true</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>MENU_JUMP</td>
 * <td>false</td>
 * <td>true</td>
 * </tr>
 * <tr>
 * <td>MENU_JUMP_DISABLED</td>
 * <td>true</td>
 * <td>true</td>
 * </tr>
 * </table>
 * @return {String} A CSS selector for the listContainer HTML element.
 * @private
 */
webui.suntheme4_2.widget.dropDown.prototype._getListContainerClassName =
	function(disabled, jumpmenu) {

    var key = "MENU_STANDARD";
    if (disabled == true) {
	if (jumpmenu == true) {
	    key = "MENU_JUMP_DISABLED";
	} else {
	    key = "MENU_STANDARD_DISABLED";
	}
    } else if (jumpmenu == true) {
	key = "MENU_JUMP";
    }
    return this.widget.getClassName(key, null);
};

/**
 * This function is used to get widget properties.
 * <p>See <code>setProps</code> for the list of supported properties,
 * </p>
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.dropDown.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Get properties.
    if (this.submitForm != null) { props.submitForm = this.submitForm; }

    return props;
};

/**
 * This function returns the CSS class name for an HTML option element,
 * based on the state of <code>element</code>. The CSS class name defined by 
 * the following theme keys from the <code>styles</code> theme category
 * is returned for the given states, based on 
 * the properties in <code>element</code>.
 * <p>
 * <table border="1px">
 * <tr><th>state</th><th>key</th></tr>
 * <tr><td>
 * separator == true</td><td>*_OPTION_SEPARATOR</td>
 * </tr><tr><td>
 * group == true</td><td>*_OPTION_GROUP</td>
 * </tr><tr><td>
 * disabled == true</td><td>*_OPTION_DISABLED</td>
 * </tr><tr><td>
 * selected == true and disabled == false</td>
 * <td>*_OPTION_SELECTED</td>
 * </tr><tr><td>
 * separator == false and group false<br/> 
 * 	and disabled == false<br/>
 *	and selected == false</td><td>*_OPTION</td>
 * </tr>
 * </table><br/>
 * Note:  * is <code>MENU_JUMP</code> for a jump menu and
 * <code>MENU_STANDARD</code> if not.
 * </p>
 * <p>
 * <p>
 * If <code>element</code> is null, the CSS class name for the theme
 * key "MENU_STANDARD_OPTION" is returned.
 * </p>
 * @param {Object} element An HTML option DOM node instance.
 * @config {boolean} separator If true, element is a option separator.
 * @config {boolean} group If true the element is a optGroup instance.
 * @config {boolean} disabled If true the element is disabled.
 * @config {boolean} selected If true the element is selected.
 * @return {String} The HTML option element CSS class name.
 */
webui.suntheme4_2.widget.dropDown.prototype.getOptionClassName = function(element) {

    var jumpmenu = this.submitForm != null && this.submitForm == true;

    var key = null;
    if (jumpmenu) {
	key = "MENU_JUMP_OPTION";
	if (element != null) {
	    if (element.separator == true) {
		key = "MENU_JUMP_OPTION_SEPARATOR";
	    } else
	    if (element.group == true) {
		key = "MENU_JUMP_OPTION_GROUP";
	    } else
	    if (element.selected == true) {
		key = "MENU_JUMP_OPTION_SELECTED";
	    }

	    // Disabled wins
	    //
	    if (element.disabled == true) {
		key = "MENU_JUMP_OPTION_DISABLED";
	    }
	}
    } else {
	key = "MENU_STANDARD_OPTION";
	if (element != null) {
	    if (element.separator == true) {
		key = "MENU_STANDARD_OPTION_SEPARATOR";
	    } else
	    if (element.group == true) {
		key = "MENU_STANDARD_OPTION_GROUP";
	    } else
	    if (element.selected == true) {
		key = "MENU_STANDARD_OPTION_SELECTED";
	    }

	    // Disabled wins
	    //
	    if (element.disabled == true) {
		key = "MENU_STANDARD_OPTION_DISABLED";
	    }
	}
    }
    return this.widget.getClassName(key);
};


/*
 * This function is used to fill in remaining template properties, after the
 * <code>buildRendering</code> function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.dropDown.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.submitterHiddenNode.id = this.id + "_submitter";
    }

    // Set the CSS class name for the select element.
    //
    var jumpmenu = this.submitForm != null && this.submitForm == true;
    var disabled = this.disabled != null && this.disabled == true;

    webui.suntheme4_2.common.addStyleClass(this.listContainer, 
	this._getListContainerClassName(disabled, jumpmenu));
    
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties defined in the
 * Object literal <code>props</code>
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} label Defines the widget and its properties to use for
 * a label.
 * @config {boolean} labelOnTop If true the label appears above the dropdown. 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Option selection is changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Option text is highlighted.
 * @config {Array} options See <code>selectBase.setOptions</code> or
 * <code>selectBase</code> overview for details on the elements of this array.
 * @config {int} size The number of option rows to display.
 * @config {String} style Specify style rules inline.
 * @config {boolean} submitForm If true the dropDown performs like a
 * "jump menu" and submits the entire form when an option is selected.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title The a HTML title attribue for the <code>select</code>
 * element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.dropDown.prototype.setProps = function(props, notify) {
    if (props == null) {
	return this.inherited("setProps", arguments);
    }

	
    // We are trying to deal with a state change which requires
    // knowing what the current state of the widget is and the
    // state as defined in props. In order to compare the states
    // it must be done in setProps before the "props" are extended
    // onto the widget. At that point there is no difference between
    // "props" and "this".
    //
    // We need to clear the selector of the listContainer
    // when state changes from disabled to non disabled state
    // or from submitForm to standard dropdown.
    //
    var isjumpmenu = this.submitForm == true;
    var toggleJumpmenu = props.submitForm != null && 
	props.submitForm != isjumpmenu;

    var isdisabled = this.disabled == true;
    var toggleDisabled = props.disabled != null && 
	props.disabled != isdisabled;

    // Get current state of the classsname, and strip it 
    // and then add the classname for the new state.
    //
    if (toggleJumpmenu || toggleDisabled) {
	var cn = this._getListContainerClassName(isdisabled, isjumpmenu);
	webui.suntheme4_2.common.stripStyleClass(this.listContainer, cn);

	cn = this._getListContainerClassName(
	    toggleDisabled ? props.disabled == true : isdisabled,
	    toggleJumpmenu ? props.jumpmenu == true : isjumpmenu);
	webui.suntheme4_2.common.addStyleClass(this.listContainer, cn);
    }
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.dropDown.prototype._setProps = function(props) {
    // If props is null, create one to pass default theme values
    // to the superClass via props.
    //
    if (props == null) {
	return this.inherited("_setProps", arguments);
	//props = {};
    }

    var jumpmenu = props.submitForm != null && props.submitForm == true;

    // Add attributes to the hidden input for jump drop down.
    //
    if ((jumpmenu && this.submitterHiddenNode.name == null) || (jumpmenu &&
	    this.submitterHiddenNode.name != this.submitterHiddenNode.id)) {

	this.submitterHiddenNode.name = this.submitterHiddenNode.id;
	this.submitterHiddenNode.value = "false";
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.editableField");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.textField");

/**
 * @name webui.suntheme4_2.widget.editableField
 * @extends webui.suntheme4_2.widget.textField
 * @class This class contains functions for the editableField widget.
 * @constructor This function is used to construct a editableField widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.editableField", webui.suntheme4_2.widget.textField, {
    // Set defaults.
    constructor: function() {
        this.edit = false;
    },
    widgetName: "editableField" // Required for theme properties.
});
   
/**
 * Helper function to disable edit mode.
 *
 * This function will commit or rollback the changes made in the field, and
 * setup appropriate style on the field.
 *
 * @param {boolean} acceptChanges Optional parameter. 
 * <p>
 * If true, the entered data (fieldNode value) will be commited through the Ajax
 * submit() request. 
 * </p><p>
 * If false, the entered data (fieldNode value) will be rolled back to previos 
 * state (value is saved before field enters editable state).
 * </p><p>
 * If not specified, no changes to the field value will made, and only styles 
 * will be modified.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.editableField.prototype.disableEdit = function(acceptChanges) {
    if (acceptChanges == true) {
        // If savedValue does not exist, we have not edited the field yet
        if (this.autoSave == true && this.savedValue && 
                this.savedValue != this.fieldNode.value) {
            this.submit();
        }
    } else if (acceptChanges == false) {
        if (this.savedValue)   {
            this.fieldNode.value = this.savedValue;
        }
    }
    this.edit = false;
    this.savedValue = null;
    this.fieldNode.className = this.getInputClassName();   
    this.fieldNode.readOnly = true;
    return true;
};

/**
 * Helper function to enable edit mode.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.editableField.prototype.enableEdit = function() {
    // Save the current value.
    this.savedValue = this.fieldNode.value;
        
    this.edit = true;
    this.fieldNode.className = this.getInputClassName();   
    this.fieldNode.readOnly = false;
    this.fieldNode.focus(); // In case function has been called programmatically, not by event.
    this.fieldNode.select();
    return true;
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.editableField.event =
        webui.suntheme4_2.widget.editableField.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_editableField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_editableField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_editableField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_editableField_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_editableField_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_editableField_event_submit_end"
    }
};

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.editableField.prototype.getInputClassName = function() {    
    // Set default style.
    if (this.disabled == true) {
        return  this.widget.getClassName("EDITABLE_FIELD_DISABLED","");
    }

    // Apply invalid style.
    var validStyle =  (this.valid == false) 
        ? " " + this.widget.getClassName("EDITABLE_FIELD_INVALID","")
        : ""; 

    return (this.edit == true)
        ? this.widget.getClassName("EDITABLE_FIELD_EDIT","") + validStyle
        : this.widget.getClassName("EDITABLE_FIELD_DEFAULT","") + validStyle;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.editableField.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.autoSave!= null) { props.autoSave = this.autoSave; }

    return props;
};

/**
 * Process user gesture events on the field, such as dblclick, onblur, 
 * onkeyup, etc.
 *
 * @param {Event} event The JavaScript event
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.editableField.prototype.onEditCallback = function(event) {
    if (event == null) {
        return false;
    }
    if (event.type == "dblclick") {
        this.enableEdit();
        return true;
    }
    if (event.type == "blur") {
        this.disableEdit(true);
        return true;
    }
    if (event.type == "keyup") {
        if (this.edit == false) {
            // Currently in readOnly state.
            // Allow <SPACE> key.
            if (event.keyCode == 32) {
                this.enableEdit();
            }
        } else {
            // Currently in edit state.
            if (event.keyCode == 27) this.disableEdit(false); // ESC
        }
        // Otherwise do not do anything.
        return true;
    }
    return true;    
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.editableField.prototype.postCreate = function () {
    // Set Initial readOnly state.
    this.fieldNode.readOnly = true;

    // Set events.
    this.dojo.connect(this.fieldNode, "ondblclick", this, "onEditCallback");
    this.dojo.connect(this.fieldNode, "onblur", this, "onEditCallback");
    this.dojo.connect(this.fieldNode, "onkeyup", this, "onEditCallback");

    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {boolean} autoSave
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} escape HTML escape button text (default).
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} maxLength 
 * @config {Array} notify
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} required 
 * @config {int} size
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} valid Value of input.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.editableField.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.editableField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Explicitly provided readOnly property must be ignored.
    props.readOnly = null;

    // Set properties.
    if (props.autoSave != null) { this.autoSave = props.autoSave; }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.hiddenField");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.hiddenField
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the hiddenField widget.
 * @constructor This function is used to construct a hiddenField widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.hiddenField", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
    },
    widgetName: "hiddenField" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.hiddenField.event =
        webui.suntheme4_2.widget.hiddenField.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_hiddenField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_hiddenField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_hiddenField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_hiddenField_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_hiddenField_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_hiddenField_event_submit_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.hiddenField.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.name) { props.name = this.name; }
    if (this.value) { props.value = this.value; }

    return props;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} name
 * @config {String} value Value of input.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.hiddenField.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.hiddenField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.name) { this.domNode.name = props.name; }
    if (props.value) { this.domNode.value = props.value; }
    if (props.disabled != null) { 
        this.domNode.disabled = new Boolean(props.disabled).valueOf();
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.hyperlink");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.anchorBase");

/**
 * @name webui.suntheme4_2.widget.hyperlink
 * @extends webui.suntheme4_2.widget.anchorBase
 * @class This class contains functions for the hyperlink widget.
 * @constructor This function is used to construct a hyperlink widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.hyperlink", webui.suntheme4_2.widget.anchorBase, {
    // Set defaults.
    widgetName: "hyperlink" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.hyperlink.event =
        webui.suntheme4_2.widget.hyperlink.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_hyperlink_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_hyperlink_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_hyperlink_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_hyperlink_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.hyperlink.prototype.getClassName = function() {
    // Set default style.
    var className = (this.disabled == true)
        ? this.widget.getClassName("HYPERLINK_DISABLED","")
        : this.widget.getClassName("HYPERLINK","");

    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.hyperlink.prototype.onClickCallback = function(event) {
    if (this.disabled == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the submit.
    var result = (this.domNode._onclick)
        ? this.domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    if (this.href) {
        return false;
    }
    event.preventDefault();
    
    // If a form id isnt provided, use the utility function to
    // obtain the form id.
    if (this.formId == null) {
        var form = this.widget.getForm(this.domNode);
        this.formId = (form) ? form.id : null;
    }
    return this.submitFormData(this.formId, this.params);
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.hyperlink.prototype.postCreate = function () {
    // If the href attribute does not exist, set "#" as the default value of the
    // DOM node.
    this.domNode.href = "#";

    // Create callback function for onClick event.
    this.dojo.connect(this.domNode, "onclick", this, "onClickCallback");

    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} formId The id of the HTML form element.
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {Array} params The parameters to be passed during request.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.hyperlink.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function submits the hyperlink if the hyperlink is enabled.
 *
 * @param {String} formId The id of the HTML form element.
 * @param {Array} params The parameters to be passed during request.
 * @return {boolean} false to cancel the JavaScript event.
 */
webui.suntheme4_2.widget.hyperlink.prototype.submitFormData = function (formId, params) {
    var theForm = document.getElementById(formId);
    var oldTarget = theForm.target;
    var oldAction = theForm.action;

    // Obtain HTML element for tab and common task components.
    var link = document.getElementById(this.id);
    if (link == null) {
        link = this.domNode;
    }

    // Set new action URL.
    var prefix;
    (theForm.action.indexOf("?") == -1) ? prefix = "?" : prefix = "&";
    theForm.action += prefix + link.id + "_submittedLink=" + link.id;               
        
    // Set new target.
    if (link.target && link.target.length > 0) {
        theForm.target = link.target;
    } else {
        theForm.target = "_self";
    }

    // Append query params to new action URL.
    if (params != null) {
        var x;
        for (var i = 0; i < params.length; i++) {
            x = params[i];
            theForm.action += "&" + params[i] + "=" + params[i + 1]; 
            i++;
        }
    }

    // Submit form.
    theForm.submit(); 

    // Restore target and action URL.
    if (link.target != null) {
        theForm.target = oldTarget;
        theForm.action = oldAction;
    }
    return false;        
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.image");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.image
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the image widget.
 * @constructor This function is used to construct a image widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.image", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.border = 0;
    },
    widgetName: "image" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.image.event =
        webui.suntheme4_2.widget.image.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_image_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_image_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_image_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_image_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.image.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.alt) { props.alt = this.alt; }
    if (this.icon) { props.icon = this.icon; }
    if (this.align) { props.align = this.align; }
    if (this.border != null) { props.border = this.border; }
    if (this.height) { props.height = this.height; }
    if (this.hspace) { props.hspace = this.hspace; }
    if (this.longDesc) { props.longDesc = this.longDesc; }
    if (this.src) { props.src = this.src; }
    if (this.vspace) { props.vspace = this.vspace; }
    if (this.width) { props.width = this.width; }

    return props;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {String} border
 * @config {String} className CSS selector.
 * @config {String} prefix The application context path of image
 * @config {String} dir Specifies the directionality of text.
 * @config {String} height 
 * @config {String} hspace 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} longDesc 
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} src 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @config {String} vspace
 * @config {String} width
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.image.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.image.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Style must be set first (e.g., for absolute positioning) -- some style
    // properties may be overridden later by the combined image.
    this.setCommonProps(this.domNode, props);

    // Clear style properties if icon was previously set.
    if (props.src && this.icon) {
        this.domNode.style.border = "";
        this.domNode.style.backgroundImage = "";
        this.domNode.style.backgroundPosition = "";
        this.domNode.style.height = "";
        this.domNode.style.width = "";
    }

    // Set properties.
    if (props.icon) {
        // IE6 has issues with "png" images. IE6 png issue can be fixed but that
        // needs an outermost <span> tag. 
        //
        // <span style="overflow: hidden; width:13px;height:13px; padding: 0px;zoom: 1";>
        // <img src="dot.gif"
        //  style="filter:progid:DXImageTransform.Microsoft.AlphaImageLoader(src='testImage.png',sizingMethod='crop');
        //  margin-left:-0px;
        //  margin-top:-26px; border: none; height:39px;width:13px;"/>
        // </span>
        //
        // For now, skipping the combined image approach for IE6. Also need fix 
        // for Safari.
        var iconProps = webui.suntheme4_2.theme.common.getImage(props.icon);
        var mapKey = iconProps['map_key'];
        if (mapKey != null && !webui.suntheme4_2.browser.isIe6()
                && !webui.suntheme4_2.widget.common.isHighContrastMode()) {
            // Note: Comparing height/width against "actual" properties is not a
            // valid test -- DOT images don't have a default size, for example.
            if (iconProps['top'] != null && iconProps['actual_height'] != null 
                    && iconProps['actual_width'] != null) {               
                var transImage = webui.suntheme4_2.theme.common.getImage("DOT");
                var combinedImage = webui.suntheme4_2.theme.common.getImage(mapKey);

                // Set style properties.
                this.domNode.style.border = "0";
                this.domNode.style.backgroundImage = "url(" + combinedImage['src'] + ")";
                this.domNode.style.backgroundPosition = "0px " + iconProps['top'] + "px";
                this.domNode.style.height = iconProps['actual_height'] + "px";
                this.domNode.style.width = iconProps['actual_width'] + "px";

                // Set transparent image.
                iconProps.src = transImage['src'];
            }
        }
        // Assign icon properties, even if combined image is not used.
        if (iconProps.alt) { this.domNode.alt = iconProps.alt; }
        if (iconProps.height) { this.domNode.height = iconProps.height; }
        if (iconProps.src) { this.domNode.src = iconProps.src; }
        if (iconProps.width) { this.domNode.width = iconProps.width; }
    } else {
        // Icon properties take precedence.
        if (props.alt) { this.domNode.alt = props.alt; }
        if (props.height) { this.domNode.height = props.height; }
        if (props.src) {
                
            // If context path is provided, then check whether the image has
            // context path already appended and if not, append it.
            if (this.prefix) {
                props.src = 
                    webui.suntheme4_2.widget.common.appendPrefix(this.prefix, props.src);                
            }
            this.domNode.src = props.src;  
        } 
        if (props.width) { this.domNode.width = props.width; }
    }
    if (props.align) { this.domNode.align = props.align; }
    if (props.border != null) { this.domNode.border = props.border; }
    if (props.hspace) { this.domNode.hspace = props.hspace; }
    if (props.longDesc) { this.domNode.longDesc = props.longDesc; }    
    if (props.vspace) { this.domNode.vspace = props.vspace; }

    // Set more properties.
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.imageButton");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.button");

/**
 * @name webui.suntheme4_2.widget.imageButton
 * @extends webui.suntheme4_2.widget.button
 * @class This class contains functions for the imageButton widget.
 * @constructor This function is used to construct a imageButton widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.imageButton", webui.suntheme4_2.widget.button, {
    // Set defaults.
    widgetName: "imageButton"  // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.imageButton.event =
        webui.suntheme4_2.widget.imageButton.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_imageButton_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_imageButton_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_imageButton_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_imageButton_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.imageButton.prototype.getClassName = function() {
    // If it is an image button, only the BUTTON3 selectors are used.
    // Note that the "mini" and "primary" values can still be set but
    // have no effect on image buttons by policy, vs by theme.
    var key = (this.disabled == true)
	? "BUTTON3_DISABLED"
	: "BUTTON3";

    var className = this.widget.getClassName(key, "");
    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to obtain the outermost HTML element class name during
 * an onFocus or onMouseOver event.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.imageButton.prototype.getHoverClassName = function() {
    // If it is an image button, only the BUTTON3 selectors are used.
    // Note that the "mini" and "primary" values can still be set but
    // have no effect on image buttons by policy, vs by theme.
    var key = "BUTTON3_HOVER";
    var className = this.widget.getClassName(key, "");
    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.imageButton.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.src) { props.src = this.src; }

    return props;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {String} prefix The application context path of image 
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} escape HTML escape button text (default).
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} src Source for image.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.imageButton.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.imageButton.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.src) {
        // If context path is provided, then check whether the image has
        // context path already appended and if not, append it.
        if (this.prefix) {
            props.src = 
                webui.suntheme4_2.widget.common.appendPrefix(this.prefix, props.src);                
        }
        this.domNode.src = props.src; 
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.imageHyperlink");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.hyperlink");

/**
 * @name webui.suntheme4_2.widget.imageHyperlink
 * @extends webui.suntheme4_2.widget.hyperlink
 * @class This class contains functions for the imageHyperlink widget.
 * @constructor This function is used to construct a imageHyperlink widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.imageHyperlink", webui.suntheme4_2.widget.hyperlink, {
    // Set defaults.
    widgetName: "imageHyperlink" // Required for theme properties.
});

/**
 * Helper function to add children.
 *
 * @param props Key-Value pairs of properties.
 * @config {Array} contents The contents of the anchor body.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.imageHyperlink.prototype.addContents = function(props) {
    if (props.contents == null) {
        return false;
    }

    // Remove child nodes.
    this.widget.removeChildNodes(this.leftContentsContainer);
    this.widget.removeChildNodes(this.rightContentsContainer);

    // Add contents.
    for (i = 0; i <props.contents.length; i++) {
        this.widget.addFragment(this.leftContentsContainer, props.contents[i], "last");
        this.widget.addFragment(this.rightContentsContainer, props.contents[i], "last");
    }
    return true;
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.imageHyperlink.event =
        webui.suntheme4_2.widget.imageHyperlink.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_imageHyperlink_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_imageHyperlink_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_imageHyperlink_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_imageHyperlink_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.imageHyperlink.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.enabledImage) { props.enabledImage = this.enabledImage; }
    if (this.disabledImage) { props.disabledImage = this.disabledImage; }
    if (this.imagePosition) { props.imagePosition = this.imagePosition; }

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.imageHyperlink.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.enabledImageContainer.id = this.id + "_enabled";
        this.disabledImageContainer.id = this.id + "_disabled";
        this.leftContentsContainer.id = this.id + "_leftContents";
        this.rightContentsContainer.id = this.id + "_rightContents";
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {Object} disabledImage
 * @config {Object} enabledImage
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} imagePosition
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.imageHyperlink.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.imageHyperlink.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Show en/disabled images.
    if (props.disabled != null) {
        var disabled = new Boolean(props.disabled).valueOf();

        // We need to hide/show images only when the disabed image is specified.
        if (this.disabledImage) { 
            this.common.setVisibleElement(this.enabledImageContainer, !disabled);
            this.common.setVisibleElement(this.disabledImageContainer, disabled);
        }
    }

    // Add enabled image.
    if (props.enabledImage) {
        this.widget.addFragment(this.enabledImageContainer, props.enabledImage);
    }

    // Add disabled image.
    if (props.disabledImage) {
        this.widget.addFragment(this.disabledImageContainer, props.disabledImage);
    }

    // Set image position.
    if (props.imagePosition) {
        var left = (props.imagePosition == "left");
        this.common.setVisibleElement(this.leftContentsContainer, !left);
        this.common.setVisibleElement(this.rightContentsContainer, left);    
    }

    // Add contents.
    this.addContents(props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.listbox");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.selectBase");

/**
 * @name webui.suntheme4_2.widget.listbox
 * @extends webui.suntheme4_2.widget.selectBase
 * @class This class contains functions for the listbox widget.
 * @constructor This function is used to construct a listbox widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.listbox", 
	webui.suntheme4_2.widget.selectBase, {
    // Set defaults.
    constructor: function() {
        // Or defer to setProps 
        this.labelOnTop = webui.suntheme4_2.widget.common.getMessageBoolean(
            "listbox.labelOnTop", false);
        this.monospace = webui.suntheme4_2.widget.common.getMessageBoolean(
            "listbox.monospace", false);
        this.multiple = webui.suntheme4_2.widget.common.getMessageBoolean(
            "listbox.multiple", false);
        this.size = webui.suntheme4_2.widget.common.getMessage("listbox.size", null, 12);
        this.width = webui.suntheme4_2.theme.common.getMessage("listbox.width", null);
    },
    widgetName: "listbox" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.listbox.event =
        webui.suntheme4_2.widget.listbox.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for.*/
        beginTopic: "webui_suntheme4_2_widget_listbox_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for.*/
        endTopic: "webui_suntheme4_2_widget_listbox_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_listbox_event_state_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_listbox_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_listbox_event_submit_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_listbox_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element CSS class name.
 * <p>
 * Note: Selectors should be concatenated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.listbox.prototype.getClassName = function() {
    var cn = this.widget.getClassName("LISTBOX", "");
    return (this.className) ? cn + " " + this.className : cn;
};

/**
 * Return an Object Literal of label properties desired
 * by the listbox widget.
 * <p>
 * This implementation sets
 * the <code>listbox.labelLevel</code> and <code>listbox.labelAlign</code>
 * theme values from the <code>messages</code> and <code>styles</code>
 * theme categories to the
 * label's <code>level</code> and <code>className</code> properties 
 * respectively. It sets the <code>htmlFor</code> attribute to 
 * <code>listContainer.id</code>
 * </p>
 * <p>
 * These properties are extended with <code>this.label</code> and the
 * resulting properties are returned.
 * </p>
 * @return {Object} label properties.
 */
webui.suntheme4_2.widget.listbox.prototype.getLabelProps = function() {
    var cn = this.getLabelClassName(null);
    var lvl = this.widget.getMessage("listbox.labelLevel", null, 2);
    var props = {};
    if (cn != null) {
       props.className = cn;
    }
    props.level = lvl;
    props.htmlFor = this.listContainer.id;
    this.prototypejs.extend(props, this.label);
    return props;
};

/**
 * Return a CSS selector for the listbox label.
 * The chosen CSS selector is a function of 
 * the <code>labelOnTop</code> property. If <code>labelOnTop</code>
 * property is <code>false</code> the value of the theme key 
 * <code>LISTBOX_LABEL_ALIGN</code> from the 
 * <code>styles</code> category is returned, else <code>null</code>.
 * @param {boolean} ontop The label on top state. In some contexts this state
 * must be passed to the method, for example when called from the 
 * <code>selectBase.setProps</code> method.
 * @return {String} A CSS selector for the listbox label.
 */
webui.suntheme4_2.widget.listbox.prototype.getLabelClassName = function(ontop) {
    var labelontop = ontop != null ? ontop : this.labelOnTop;
    return labelontop == true
	? null
	: this.widget.getClassName("LISTBOX_LABEL_ALIGN", null);
};

/**
 * Return a CSS selector for listContainer HTML element.
 * The chosen CSS selector is a function of 
 * the <code>disabled</code> and <code>monospace</code>properties.
 * The returned selector is the value of a property defined in the
 * <code>styles</code> category, as shown in the following table.
 * <table border="1px">
 * <tr><th>key</th><th>disabled</th><th>monospace</th></tr>
 * <tr>
 * <td>LIST</td>
 * <td>false</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>LIST_DISABLED</td>
 * <td>true</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>LIST_MONOSPACE</td>
 * <td>false</td>
 * <td>true</td>
 * </tr>
 * <tr>
 * <td>LIST_MONOSPACE_DISABLED</td>
 * <td>true</td>
 * <td>true</td>
 * </tr>
 * </table>
 * @return {String} A CSS selector for the listContainer HTML element.
 * @private
 */
webui.suntheme4_2.widget.listbox.prototype._getListContainerClassName =
	function(disabled, monospace) {
    var key = "LIST";
    if (disabled == true) {
	if (monospace == true) {
	    key = "LIST_MONOSPACE_DISABLED";
	} else {
	    key = "LIST_DISABLED";
	}
    } else if (monospace == true) {
	key = "LIST_MONOSPACE";
    }
    return this.widget.getClassName(key, null);
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.listbox.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Get properties.
    //
    if (this.size != null) {
	props.size = this.size;
    }
    if (this.multiple != null) {
	props.multiple = this.multiple;
    }
    if (this.monospace != null) {
	props.monospace = this.monospace;
    }
    return props;
};

/**
 * This function returns the CSS class name for an HTML option element,
 * based on the state of <code>element</code>. The CSS class name for 
 * the following theme keys is returned for the given states, based on 
 * the properties
 * in "option".
 * <p>
 * <table border="1px">
 * <tr><th>state</th><th>key</th></tr>
 * <tr><td>
 * separator == true</td><td>LIST_OPTION_SEPARATOR</td>
 * </tr><tr><td>
 * group == true</td><td>LIST_OPTION_GROUP</td>
 * </tr><tr><td>
 * disabled == true</td><td>LIST_OPTION_DISABLED</td>
 * </tr><tr><td>
 * selected == true and disabled == false</td><td>LIST_OPTION_SELECTED</td>
 * </tr><tr><td>
 * separator == false and group false<br/> 
 * 	and disabled == false<br/>
 *	and selected == false</td><td>LIST_OPTION</td>
 * </tr>
 * </table>
 * </p>
 * <p>
 * If <code>element</code> is null, the CSS class name for the theme
 * key "LIST_OPTION" is returned.
 * </p>
 * @param {Object} element An HTML option DOM node instance.
 * @config {boolean} separator If true, element is a option separator.
 * @config {boolean} group If true the element is a optGroup instance.
 * @config {boolean} disabled If true the element is disabled.
 * @config {boolean} selected If true the element is selected.
 * @return {String} The HTML option element CSS class name.
 */
webui.suntheme4_2.widget.listbox.prototype.getOptionClassName = function(element) {
    var key = "LIST_OPTION";
    if (element == null) {
	return this.widget.getClassName(key, null);
    }

    if (element.separator == true) {
        key = "LIST_OPTION_SEPARATOR";
    } else
    if (element.group == true) {
        key = "LIST_OPTION_GROUP";
    } else
    if (element.selected == true) {
        key = "LIST_OPTION_SELECTED";
    }

    // disabled wins
    //
    if (element.disabled == true) {
	key = "LIST_OPTION_DISABLED";
    }
    return this.widget.getClassName(key, null);
};

/*
 * This function is used to fill in remaining template properties, after the
 * <code>buildRendering</code> function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.listbox.prototype.postCreate = function () {
    // Don't trash the template.
    //
    webui.suntheme4_2.common.addStyleClass(this.listContainer,
	this._getListContainerClassName(
		this.disabled == true, this.monospace == true));
    return this.inherited("postCreate", arguments);
};


/**
 * This function is used to set widget properties defined in the
 * <code>props</code> Object literal.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * <p>
 * See <code>selectBase.setOptions</code> or overview for the properties
 * of option objects in the <code>options</code> array property.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} label Defines the widget and its properties to use for a
 * label.
 * @config {String} labelOnTop If true the label appears above the listbox.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {boolean} multiple If true allow multiple selections
 * @config {boolean} monospace If true use monospace font styling.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Option selection is changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Option text is highlighted.
 * @config {Array} options See selectBase.setOptions or selectBase overview
 * for details on the elements of this array.
 * @config {int} size The number of option rows to display.
 * @config {String} style Specify style rules inline. 
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.listbox.prototype.setProps = function(props, notify) {
    if (props == null) {
	return this.inherited("setProps", arguments);
    }

    // We are trying to deal with a state change which requires
    // knowing what the current state of the widget is and the
    // state as defined in props. In order to compare the states
    // it must be done in setProps before the "props" are extended
    // onto the widget. At that point there is no difference between
    // "props" and "this".
    //
    var ismonospace = this.monospace == true;
    var toggleMonospace = props.monospace != null && 
	props.monospace != ismonospace;

    var isdisabled = this.disabled == true;
    var toggleDisabled = props.disabled != null && 
	props.disabled != isdisabled;

    // Get current state of the classsname, and strip it 
    // and then add the classname for the new state.
    //
    if (toggleMonospace || toggleDisabled) {
	var cn = this._getListContainerClassName(isdisabled, ismonospace);
	webui.suntheme4_2.common.stripStyleClass(this.listContainer, cn);

	cn = this._getListContainerClassName(
	    toggleDisabled ? props.disabled == true : isdisabled,
	    toggleMonospace ? props.monospace == true : ismonospace);
	webui.suntheme4_2.common.addStyleClass(this.listContainer, cn);
    }
    return this.inherited("setProps", arguments);
};

/**
 * This function sets properties specific to the listbox, the superclass
 * will set properties specific to the <code>select</code> HTML element
 * and handle <code>labelOnTop</code> and the label subcomponent.
 * <p>
 * If the <code>size</code>, <code>multiple</code>, or <code>monospace</code>
 * properties are not set in <code>props</code> or <code>this</code>,
 * the values are obtained from the theme using the keys 
 * <code>listbox.size</code>, <code>listbox.multiple</code>,
 * <code>listbox.monospace</code>, and <code>listbox.labelOnTop</code>
 * respectively, from the theme "messages" category. If the theme does 
 * not define these values then the values <code>12</code>,
 * <code>false</code>, <code>false</code>, and <code>false</code>
 * are used, respectively.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} multiple If true allow multiple selections
 * @config {boolean} monospace  If true the <code>LIST_MONOSPACE</code> and
 * <code>LIST_MONOSPACE_DISABLED</code>CSS class names are used, else
 * <code>LIST</code> and <code>LIST_DISABLED</code>.
 * @config {int} size The number of rows to display.
 * @config {boolean} labelOnTop If true render the label on top of the
 * list box, else to the upper left.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.listbox.prototype._setProps = function(props) {
    // If props == null create one so that the superclass can 
    // handle "labelOnTop" or anything else that needs to be
    // set in props and handled in the superclass
    //
    if (props == null) {
	return this.inherited("_setProps", arguments);
	//props = {};
    }

    if (props.size != null) {
	var size = props.size;
	if (size < 1) {
	    size = this.theme.getMessage("listbox.size", null, 12);
	}
	if (this.listContainer.size != size) {
	    this.listContainer.size = size;
	}
    }

    if (props.multiple != null && 
	    this.listContainer.multiple != props.multiple) {
	this.listContainer.multiple = props.multiple;
    }
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.login");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.login
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the login widget.
 * @constructor This function is used to construct a login widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.login", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.loginState = "INIT";
    },
    widgetName: "login" // Required for theme properties.    
});

/**
 * This function publishes the event that kicks off the authentication process.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.login.prototype.authenticate = function() {
    if (this.keys) {
        for (var i=0; i<this.keys.length; i++) {
            var widget = webui.suntheme4_2.dijit.byId(this.keys[i][0]);
            var keyVal;
            if (widget) {
                var name = widget.widgetName;
                if (name == "textField" || name == "passwordField") {
                    keyVal = widget.getProps().value;
                } else if (name == "dropDown" || name == "listbox") {
                    keyVal = widget.getSelectedValue();
                }
            }
            this.keys[i][1] = keyVal;
        }
    }
                    
    // Publish an event for custom AJAX implementations to listen for.
    this.publish(webui.suntheme4_2.widget.login.event.authenticate.beginTopic, [{
        id: this.id,
        loginState: this.loginState,
        keys: this.keys,
        endTopic: webui.suntheme4_2.widget.login.event.authenticate.endTopic
    }]);
    return true;
};

/**
 * This function executes the steps that need to be followed when the 
 * "Login" button is clicked.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.login.prototype._buttonClicked = function() {
    return this.authenticate();
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.login.event =
        webui.suntheme4_2.widget.login.prototype.event = {
    /**
     * This closure is used to process authentication events.
     * @ignore
     */
    authenticate: {
        /**
         * Authentication begin event topics for custom AJAX implementations to listen for.
         */
        beginTopic: "webui_suntheme4_2_widget_login_event_authenticate_begin",

        /**
         * Authentication end event topics for custom AJAX implementations to listen for.
         */
        endTopic: "webui_suntheme4_2_widget_login_event_authenticate_end"

    },

    /**
     * This closure is used to process refresh events.
     * @ignore
     */
    refresh: {
        /**
         * Refresh event topics for custom AJAX implementations to listen for.
         */
        beginTopic: "webui_suntheme4_2_widget_login_event_refresh_begin",

        /**
         * Refresh event topics for custom AJAX implementations to listen for.
         */
        endTopic: "webui_suntheme4_2_widget_login_event_refresh_end"
    },

    /**
     * This closure is used to report success/failure during the authentication 
     * process.
     * @ignore
     */
    result: {
        /**
         * Successful authentication event topic for applications can listen for.
         */
        successTopic: "webui_suntheme4_2_widget_login_event_result_success",

        /**
         * Authentication failure event topic that applications can listen for.
         */
        failureTopic: "webui_suntheme4_2_widget_login_event_result_failure"
    },

    /**
     * This closure is used to process state change events.
     * @ignore
     */
    state: {
        /**
         * Event topics for custom AJAX implementations to listen for.
         */
        beginTopic: "webui_suntheme4_2_widget_login_event_state_begin",

        /**
         * Event topics for custom AJAX implementations to listen for.
         */
        endTopic: "webui_suntheme4_2_widget_login_event_state_end"
    }
};

/**
 * This function performs error handling on the client side
 * when authentication failure occurs. Typically this involves
 * updating the alter widget with the appropriate messages
 * sent across from the server.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} id ID of the login widget.
 * @config {String} loginState State of the authentication process. Set to "INIT" at the beginning.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.login.prototype._handleFailure = function(props) {
    this._setAlert(props.alert);
    this.publish(webui.suntheme4_2.widget.login.event.result.failureTopic, [{
        id: props.id
    }]);

    // initialize the state so that subsequent requests can start afresh
    this.loginState = "INIT";
    props.loginState = "INIT";
    return true;
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.login.prototype.getClassName = function() {
    var key = "LOGIN_DIV";

    // Get theme property.
    var className = this.theme.getClassName(key);
    if (className == null || className.length == 0) {
	return this.className;
    }
    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.login.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // var props = webui.suntheme4_2.widget.login.superclass.getProps.call(this);
    if (this.loginState) {props.loginState = this.loginState; }    
    if (this.autoStart != null) { props.autoStart = this.autoStart;}
    if (this.tabIndex != null) { props.tabIndex = this.tabIndex;}
    if (this.dotImage != null) {props.dotImage = this.dotImage;}
    if (this.loginButton != null) {props.loginButton = this.loginButton;}
    return props;
};

/**
 * This is a private function that creates a client side widget based on the 
 * widgetType and other properties supplied. 
 *
 * @param {Object} props Key-Value pairs of properties of the widget.
 * @return {boolean} the widget props if successful; false, otherwise.
 * @private
 */
webui.suntheme4_2.widget.login.prototype._getWidgetProps = function(props) {
    if (props == null) {
        return false;
    }
        
    var type = props.type;
    if (type == null) {
        return false;
    }
        
    var _props = {};
            
    // Set default module and widget name
    _props = this.widget.getWidgetProps(type, _props); 
        
    // Add extra properties               
    this.prototypejs.extend(_props, props);
    return _props;
};

/**
 * This function handles the case where authentication is complete
 * and successful. The end result could involve storing a cookie
 * value somewhere or redirecting to some other page.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} id ID of the login widget.
 * @config {String} loginState State of the authentication process. Set to 
 * "INIT" at the beginning.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.login.prototype._handleSuccess = function(props) {
    // Publish the success event topic
    // Remove the alert message if props does not
    // contain any alert information.
    // Clear out the loginTable.
    this._setAlert(props.alert);
    this.publish(webui.suntheme4_2.widget.login.event.result.successTopic, [{
        id: props.id
    }]);

    this.widget.removeChildNodes(this.loginTable);
    this.loginState = "INIT";
    props.loginState = "INIT";

    if (props.redirectURL != null) {
        var loc = document.location;
        var newURL = loc.protocol + "//" + loc.host;
        newURL = newURL + props.redirectURL;
        document.location.href = newURL;
    }
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.login.prototype.postCreate = function () {
    if (this.tabIndex == null) {
        this.tabIndex = -1;
    }

    // If login button and dot image are empty generate them on the
    // client side.
    if (this.dotImage == null) {
	this.dotImage = this.widget.getWidgetProps("image", {
            icon: "DOT",
            id: this.id + "_dotImage"
        });
    }
    
    if (this.loginButton == null) {

        var _props = {};
	_props.id = this.id + "_loginButton";
        _props.tabIndex = this.tabIndex;
        _props.alt = this.theme.getMessage("login.buttonAlt");
        _props.value = this.theme.getMessage("login.buttonTitle");
        _props = this.widget.getWidgetProps("button", _props);
        this.loginButton = _props;
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} id ID of the login widget.
 * @config {String} loginState State of the authentication process. Set to "INIT" at the beginning.
 * @config {boolean} autoStart Flag indicating if authentication process should start on page load.
 * @config {String} style Specify style rules inline.
 * @config {String} className CSS selector of outermost DOM node for this widget.
 * @config {String} tabIndex Attribute required to support tabbed navigation.
 * @config {Object} userData JSON object containing user data that needs to be displayed as user prompt.
 * @config {Object} keys JSON object representing the key value mapping for user data fields.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.login.prototype.setProps = function(props, notify) {
    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.login.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Handle the success and failure and continue states here.
    // The init state is handled in the postCreate function
    // as there needs to be a minimal delay the first time this
    // widget is  being rendered - otherwise only the initial <span>
    // element representing the login widget gets transmitted causing
    // problems on the server side. 

    if (props.loginState == "SUCCESS") {
        this._handleSuccess(props);

    } else if (props.loginState == "FAILURE") {
        this._handleFailure(props);

    } else if (props.loginState == "CONTINUE") {
        this._updateLoginTable(props);
    }
    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * This function adds or updates the alert widget displayed as part of the
 * login widget.
 *
 * @param {Object} alertProps Key-Value pairs of properties.
 * @config {String} type The alert type.
 * @config {String} summary The alert summary.
 * @config {String} detail The alert detail.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.login.prototype._setAlert = function(alertProps) {
    if (alertProps == null) {
        this.widget.removeChildNodes(this.alertContainer);
        return false;
    } else {
        var _props = {};
            
        // Set default module and widget name
        _props = this.widget.getWidgetProps("alert", _props); 
        _props.id = this.id + "_alert";

        // Add extra properties               
        if (alertProps != null) {
            this.prototypejs.extend(_props, alertProps);
        }
        if (_props.summary == null) {
            _props.summary = this.theme.getMessage("login.errorSummary");
        }
        if (_props.detail == null) {
            _props.detail = this.theme.getMessage("login.errorDetail");;
        }

        var tr = this.alertRowContainer.cloneNode(false);
        this.loginTbody.appendChild(tr);
        var td = this.alertCellContainer.cloneNode(false);
        tr.appendChild(td);
        var widget = webui.suntheme4_2.dijit.byId(_props.id);
        if (widget) {
            widget.setProps(_props);
        } else {
            this.widget.addFragment(td, _props); 
        }
    }
    return true;
};

/**
 * This function is used to "start" the widget, after the widget has been
 * instantiated.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.login.prototype.startup = function () {
    if (this._started) {
        return false;
    }
    // If widget set to "autoStart" fire up the authentication process.
    var props = this.getProps();
    if (this.loginState == "INIT") {
        var id = this.id;
        if (this.autoStart) {
            setTimeout(function() {
                webui.suntheme4_2.dijit.byId(id).authenticate();
            }, 10);    
        }
    }
    return this.inherited("startup", arguments);
};

/**
 * This function is used to update the login section of the DOM tree with
 * the new set of widgets. This new set of widgets represents the new set
 * of prompts for entering authentication data. For example, the first set
 * of prompts could have been a textField and a password field. The second
 * set could be dropdown list containing a set of roles to choose from 
 * followed by a role password field.
 * 
 * @param {Object} props Key-Value pairs of properties.
 * @config {Object} userData JSON object containing user data that needs to be displayed as user prompt.
 * @config {Object} keys JSON object representing the key value mapping for user data fields.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.login.prototype._updateLoginTable = function(props) {
    // Remove existing data entries before adding the new ones.
    // This involves destroying the widgets and also deleting
    // the table rows.
    this.widget.removeChildNodes(this.loginTbody);
        
    var rowNum = 1;

    // add the alert row
    var alertAdded = this._setAlert(props.alert);
        
    // set up table rows for each of the user prompts
    if (props.userData) {
        var rowCount = props.userData.length;
        var idCount = 0;

        for (var i=0; i < rowCount; i++) {
            var dataRow = props.userData[i];
            var tr = this.inputDataRowContainer.cloneNode(false);
            this.loginTbody.appendChild(tr);
            for (var j=0; j<dataRow.length; j++) {
                var td;
                var divNode;
                if (j==0) {
                    td = this.labelContainerCell.cloneNode(false);
                    tr.appendChild(td);
                    if (i+1 == rowCount) {
                        divNode = this.lastLabelContainer.cloneNode(true);
                    } else {
                        divNode = this.labelContainer.cloneNode(true);
                    }
                    td.appendChild(divNode);
                    var widgetProps = this._getWidgetProps(dataRow[j]);
                    if (widgetProps) {
                        this.widget.addFragment(divNode, widgetProps, "last"); 
                    }
                } else {
                    td = this.dataContainerCell.cloneNode(false);
                    tr.appendChild(td);
                    if (i+1 == rowCount) {
                        divNode = this.lastInputContainer.cloneNode(true);
                    } else if (dataRow[j].type == "staticText") {
                        divNode = this.stxtContainer.cloneNode(true);
                    } else if (dataRow[j].type == "textField") {
                        divNode = this.textContainer.cloneNode(true);
                    } else {
                        divNode = this.inputContainer.cloneNode(true);
                    }
                    td.appendChild(divNode);
                    var widgetProps = this._getWidgetProps(dataRow[j]);
                    if (widgetProps) {
                        this.widget.addFragment(divNode, widgetProps, "last"); 
                    }
                }
            }
            rowNum++;
        }
        // add table row for spacer image followed by the login button.
        var buttonTR = this.buttonRowContainer.cloneNode(false);
        this.loginTbody.appendChild(buttonTR);
        var td1 = this.dotImageContainer.cloneNode(false);
        buttonTR.appendChild(td1);
        if (this.dotImage) {
            this.widget.addFragment(td1, this.dotImage, "last");
        }

        var td2 = this.buttonContainer.cloneNode(false);
        buttonTR.appendChild(td2);
        var spanNode = this.loginButtonContainer.cloneNode(true);
        var _this = this;
        this.loginButton.onClick = function() {
            _this._buttonClicked(props);
            return false;
        };
        td2.appendChild(spanNode);
        this.widget.addFragment(spanNode, this.loginButton, "last");
    }
    return true;
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.menuBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.menuBase
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for widgets that extend menuBase.
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.menuBase", webui.suntheme4_2.widget.widgetBase);

/**
 * Add the specified options to the dom element.
 *
 * @param {Node} menuNode The node to which the menu items are to be added.
 * @param {Object} props Key-Value pairs of properties.
 * @config {Array} options 
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.menuBase.prototype.addOptions = function(menuNode, props) {
    var groupNode, optionNode, separator, sepNode;
    for (var i = 0; i < props.options.length; i++) {
        
        // create an li node which will represent an option element.
        optionNode = this.optionContainer.cloneNode(false);   
        optionNode.id = this.id + "_" + props.options[i].label + "_container";                
        this.setOptionNodeProps(optionNode, props.options[i], i);

        // Append the li element to the menu element.        
        menuNode.appendChild(optionNode);
        if (props.options[i].group == true) {
            optionNode.group = true;
            groupNode = this.groupOptionContainer.cloneNode(false);
            menuNode.appendChild(groupNode);
            this.addOptions(groupNode, props.options[i]);
        }
        
        if (props.options[i].separator == true) {
            separator = this.menuSeparatorContainer.cloneNode(true);
            if (webui.suntheme4_2.browser.isIe5up()) {
                var sep = this.menuSeparator.cloneNode(true);
                separator.appendChild(sep);
            }
            this.common.setVisibleElement(separator, true);             
            menuNode.appendChild(separator);
        }
    }
    return true;    
};

/**
 * The callback function for clicking on a menu item.
 *
 * @param {String} optionId The id of the option element that is clicked.
 * @return {Function} The callback function.
 */
webui.suntheme4_2.widget.menuBase.prototype.createOnClickCallback = function(optionId) {
    var _id = this.id;

    /**
     * New literals are created every time this function is called, and it's 
     * saved by closure magic.
     *
     * @param {Event} event The JavaScript event.
     */
    return function(event) {
        // Get hold of the particular option element.
        var elem = document.getElementById(optionId);
        if (elem == null) {
            return;
        }        

        // Get hold of the menu element.
        var widget = webui.suntheme4_2.dijit.byId(_id);                
        var val = elem.selectValue;
        var dis = elem.disabled;       
        var group = elem.group;
        
        // process the action only if the menu is not disabled and the selected
        // option is not a group header.
        if (!dis && !group) {
            widget.processOnClickEvent(val);
        }         
    };
};

/**
 * The callback function for key press on a menu item.
 *
 * @param {String} nodeId The id of the option element that is clicked
 * @return {Function} The callback function. 
 */
webui.suntheme4_2.widget.menuBase.prototype.createOnKeyDownCallBack = function(nodeId) {
    if (nodeId == null) {
        return null;
    }
    
    var id = this.id;
    var _widget = this.widget;
    return function(event) {
       var elem = document.getElementById(nodeId);
       if (elem == null) {
          return false;
       }
        var widget = webui.suntheme4_2.dijit.byId(id);

        // If the menu is not visible, we do not need to capture
        // key press events.
        if (!this.common.isVisibleElement(widget.domNode)) {
            return false;
        }
        event = _widget.getEvent(event);
        var keyCode = _widget.getKeyCode(event);
        
        // if onkeypress returns false, we do not traverse the menu.
        var keyPressResult = true;
        if (this._onkeypress) {
            keyPressResult = (this._onkeypress) ? this._onkeypress() : true;
        }        
        
        if (keyPressResult != false) {
            widget.traverseMenu(keyCode, event, nodeId);
        }
        return true;
    };
};

/**
 * Handles the on mouse out for each menuitem.
 *
 * @param {Node} menuItem The DOM node associated with the menu item.
 * @return {Function} The callback function.
 */
webui.suntheme4_2.widget.menuBase.prototype.createOnMouseOutCallBack = function(menuItem) {
    if (menuItem == null) {
        return null;
    }
    var _id = this.id;
 
    // New literals are created every time this function is called, and it's 
    // saved by closure magic.
    return function(event) {
        var widget = webui.suntheme4_2.dijit.byId(_id);
        menuItem.className = widget.theme.getClassName("MENU_GROUP_CONTAINER");
    };
};

/**
 * Handles the on mouse over for each menuitem.
 *
 * @param {Node} menuItem The DOM node associated with the menu item.
 * @return {Function} The callback function.
 */
webui.suntheme4_2.widget.menuBase.prototype.createOnMouseOverCallBack = function(menuItem) {
    if (menuItem == null) {
        return null;
    }
    var _id = this.id;

    /**
     * New literals are created every time this function is called, and it's 
     * saved by closure magic.
     *
     * @param {Event} event The JavaScript event.
     */
    return function(event) {
        var widget = webui.suntheme4_2.dijit.byId(_id);
        menuItem.className = menuItem.className + " " + 
                    widget.theme.getClassName("MENU_ITEM_HOVER");            
        if (widget != null) {
            // Mozilla browser (not firefox/seamonkey) do not support focus/blur
            // for divs                
            if (document.getElementById(widget.menuId[widget.focusPosition]).blur) {
                document.getElementById(widget.menuId[widget.focusPosition]).blur();
            }                
            if (!(menuItem.id == widget.menuId[widget.focusPosition])) {
                document.getElementById(widget.menuId[widget.focusPosition]).className =
                    widget.theme.getClassName("MENU_GROUP_CONTAINER");
            }
        }
        if (webui.suntheme4_2.browser.isIe5up() ) {
            menuItem.className = menuItem.className + " " + 
                widget.theme.getClassName("MENU_ITEM_HOVER");            
        }
    };
};

/**
 * Calculate the maximum width of menu to be set.
 * If a menu if at all contains a group, then set the containsGroup
 * attribute to true. This will help in adding an extra pixel width
 * while assigning the width of th menu. This is to account for the
 * indentation of the menu.
 *
 * @param {Array} props 
 * @return {int} The max menu width.
 */
webui.suntheme4_2.widget.menuBase.prototype.getMaxWidth = function(props) {
    var menuWidth = 0;
    var maxWidth = 0;
    for (var i = 0; i < props.length; i++) {
         menuWidth = props[i].label.length;
         if (menuWidth > maxWidth) {
            maxWidth = menuWidth;
         }
         if (props[i].image != null) {
            this.hasImage = true;
         }
         if (props[i].group) {
            this.containsGroup = true;
            maxWidth = this.getMaxWidth(props[i].options);
         }
    }
    return maxWidth;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.menuBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    if (this.options) { props.options = this.options; }
    if (this.formId) { props.formId = this.formId; }
    if (this.submitForm) { props.submitForm = this.submitForm; }  

    return props;
};

/**
 * Returns the currently selected item in the menu.
 *
 * @return {String} The selected item.
 */
webui.suntheme4_2.widget.menuBase.prototype.getSelectedValue = function() {
    if (this.clickedItem) {
        return this.clickedItem;
    } else {
        return null;
    }
};

/**
 * This function is used to obtain the outermost HTML element style.
 * <p>
 * Note: Styles should be concatinated in order of precedence (e.g., the 
 * user's style property is always appended last).
 * </p>
 * @return {String} The outermost HTML element style.
 */
webui.suntheme4_2.widget.menuBase.prototype.getStyle = function() {
    var style = "width:" + this.maxWidth + "em;";

    // Since the style has to be recalculated each and every time the options
    // are changed, you cannot rely on the the existing style on the dom element
    // to see whether style was already defined. Hence use a separate variable
    // which sets itself to true if the widget sets the width/
    if (!this.widthSet) {
    	var st = this.style;
	var reg = ";?[\\t]*width[\\t]*:";
	if (st != null) {
	    var res = st.match(reg);

            // Return user's style if width is set already.
            if (this.style && res != null) {
                return this.style;
            }
	}
    }
    this.widthSet = true;
    return (this.style)
        ? style + this.style
        : style;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.menuBase.prototype.postCreate = function () {
    // Set public functions.
    this.domNode.getSelectedValue = function(props, optionNode) { return webui.suntheme4_2.dijit.byId(this.id).getSelectedValue(); };
    this.focusPosition = 0;        
    return this.inherited("postCreate", arguments);
};

/**
 * Process the enter key press event.Evaluvate the keyPress/keyDown (for non-IE/IE browsers)
 * and traverse through the menu. Also, if onChange is specified, evaluvate that and 
 * submit the form if submitForm is specified to true.
 *
 * @param (String) value The "value" of the selected option. 
 * @return {boolean} true The enter key press event completed successfully
 */
webui.suntheme4_2.widget.menuBase.prototype.processEnterKeyPressEvent = function(value) {
    var changeResult = true;

    // Check whether the selected value is different than the one previously selected
    var bool = (value == this.getSelectedValue());
    this.setSelectedValue(value);

    if (this._onchange && !bool) {    
        // If function returns false, we must prevent the request.       
        changeResult = (this._onchange) ? this._onchange() : true;
    }
    
    // Functions may sometime return without a value in which case the value
    // of the boolean variable may become undefined. 
    if (changeResult != false) {
        if (this.submitForm) {
            this.submitFormData();
        }  
    }
    return true;
};
    
/**
 * This function executes the onchange and onclick event handlers if provided by 
 * the developer. It then either submits the form if submitForm attribute is 
 * specified to true.
 *
 * @param {String} value The selected value.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.menuBase.prototype.processOnClickEvent = function(value) {
    var clickResult = true;
    var changeResult = true;
    
    //Check if the selected value has changed from the previous selection.
    var bool = (value == this.getSelectedValue());
    this.setSelectedValue(value);

    if (this._onclick) {
        clickResult = (this._onclick) ? this._onclick() : true;
    }
    if (this._onchange && !bool) {    
        // If function returns false, we must prevent the request.       
        changeResult = (this._onchange) ? this._onchange() : true;
    }

    // Functions may sometime return without a value in which case the value
    // of the boolean variable may become undefined. 
    if (clickResult != false && changeResult != false) {
        if (this.submitForm) {
            this.submitFormData();
        }  
    }
    return true;
};

/**
 * Set the appropriate class name on the menu item container.
 *
 * @param {Node} menuItemContainer The container for the menu item.
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled 
 * @config {boolean} group
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.menuBase.prototype.setMenuNodeClassName = function(
        menuItemContainer, props) {        
    if (new Boolean(props.group).valueOf() == true) {
        menuItemContainer.className = this.theme.getClassName("MENU_GROUP_HEADER");
    } else if (new Boolean(props.disabled).valueOf() == true) {
        menuItemContainer.className = this.theme.getClassName("MENU_ITEM_DISABLED");
    } else {
        menuItemContainer.className = this.theme.getClassName("MENU_GROUP_CONTAINER");        

        // Whenever mouse over/out happens, focus must be set on the menu accordingly.
        // Apply an hack for IE for mouse hover on the div element since div:hover type
        // of css declarations do not seem to work. onmouseover and onmouseout events
        // are attached with the div element and a style class is applied each time a
        // mouseover happens. This style represents the "hover" class.
        // Note that the "this" in these functions represent the menuItem's "div" element
        // and not the "menu" widget element.
        this.dojo.connect(menuItemContainer, "onmouseover",
            this.createOnMouseOverCallBack(menuItemContainer));
        this.dojo.connect(menuItemContainer, "onmouseout",
            this.createOnMouseOutCallBack(menuItemContainer));
    }
    return true;
};

/**
 * Helper function to set the properties of an option item. This is invoked
 * by the addOptions function call.
 *
 * @param optionNode The node for which the option is to be added.
 * @param {Object} props Key-Value pairs of properties.
 * @param {String} number The position of the option item in the menu.
 * @config {boolean} disabled 
 * @config {boolean} escape
 * @config {Object} image
 * @config {String} label
 * @config {String} title
 * @config {String} value
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.menuBase.prototype.setOptionNodeProps = function(optionNode, props, number) {
    optionNode.id = this.id + "_" + props.value + "_container";
    var menuItemContainer = this.menuItemContainer.cloneNode(false);
    menuItemContainer.id = optionNode.id + "_label";

    // depending on the kind of node, assign the appropriate style
    // for the menu node.
    this.setMenuNodeClassName(menuItemContainer, props);
    optionNode.appendChild(menuItemContainer);
    
    // valueNode contains a div element which will hold the option.
    var valueNode = this.menuItemNode.cloneNode(false);  
    valueNode.id = this.id + "_" + props.value;

    if (!(new Boolean(props.group).valueOf() == true) && 
            !(new Boolean(props.disabled).valueOf() == true)) {
        this.menuId[this.menuItemCount++] = menuItemContainer.id;   
    }

    menuItemContainer.tabIndex = -1;
    valueNode.tabIndex = -1;

    menuItemContainer.selectValue = props.value;
    menuItemContainer.disabled = props.disabled;
    menuItemContainer.group = props.group;
    if (props.title != null) {
        menuItemContainer.title = props.title;
        valueNode.title = props.title;
    }
    
    if (valueNode.setAttributeNS) {
        valueNode.setAttributeNS(
            "http://www.w3.org/2005/07/aaa", "posinset", number);
    }

    if (valueNode.setAttributeNS) {
        valueNode.setAttributeNS(
            "http://www.w3.org/2005/07/aaa", "disabled", props.disabled);
    }
        
    // Create callback function for onkeydown event.
    this.dojo.connect(menuItemContainer, "onkeydown", 
        this.createOnKeyDownCallBack(menuItemContainer.id));         
        
    // Create callback function for onClick event.
    this.dojo.connect(menuItemContainer, "onclick",
        this.createOnClickCallback(menuItemContainer.id));
        
    // Set label value.
    if (props.label) {
        this.widget.addFragment(valueNode, props.label, "last", props.escape);
    }    

    // Set title.
    if (props.title) {
        optionNode.title = props.title;
    }
    
    // By default have the no image container cloned and kept
    // If an image is present, then replace that with the span
    // placeholder for the image.
    if (new Boolean(this.hasImage).valueOf() == true) {
        var imageNode = this.menuItemNoImageContainer.cloneNode(false);
        if (props.image != null) {
            // Add the widget
            imageNode = this.menuItemImageContainer.cloneNode(false);
            props.image.className = this.theme.getClassName("MENU_ITEM_IMAGE");
            this.widget.addFragment(imageNode, props.image);
        } 
        menuItemContainer.appendChild(imageNode);
    }
    
    // Append the placeholder image node.
    menuItemContainer.appendChild(this.menuItemSubMenu.cloneNode(false));

    // Append the div element to the li element.           
    menuItemContainer.appendChild(valueNode);
    return true;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} formId 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {Array} options 
 * @config {boolean} primary Set button as primary if true.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.menuBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }
    
    // Replace options -- do not extend.
    if (props.options) {
        this.options = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.menuBase.prototype._setProps = function(props){
    if (props == null) {
        return false;
    }

    // A web app devleoper could return false in order to cancel the 
    // submit. Thus, we will handle this event via the onClick call back.
    if (props.onChange) {
        // Set private function scope on widget.
        this._onchange = (typeof props.onChange == 'string')
            ? new Function(props.onChange) : props.onChange;

        // Must be cleared before calling setJavaScriptProps() below.
        props.onChange = null;
    }

    // A web app devleoper could return false in order to cancel the 
    // submit. Thus, we will handle this event via the onClick call back.
    if (props.onClick) {
        
        // Set private function scope on widget.
        this._onclick = (typeof props.onClick == 'string')
            ? new Function(props.onClick) : props.onClick;

        // Must be cleared before calling setJavaScriptProps() below.
        props.onClick = null;
    }
    
    // Add options
    if (props.options) {
        
        // Try to set the width of the menu here.
        this.maxWidth = this.getMaxWidth(props.options);        
        
        // Account for image width if one exists. This property can be got from the
	// theme
        if (this.hasImage) {
            var placeHolderWidth = parseFloat(this.theme.getMessage("Menu.placeHolderImageWidth"));
            this.maxWidth += placeHolderWidth; 
        }       
   
        // If an menuGroup exists, then add one character width. Otherwise menu
        // does not scale properly
        if (this.containsGroup) {
            this.maxWidth += 1;
        }
             
        this.widget.removeChildNodes(this.outerMenuContainer);
        this.menuId = [];
        this.menuItemCount = 0;
        
        // Clone the menu node and add it to the outer container.
        var menuNode = this.groupOptionContainer.cloneNode(false);
        menuNode.className = this.theme.getClassName("MENU_CONTAINER");
        this.outerMenuContainer.appendChild(menuNode);         
        this.addOptions(menuNode, props);
    }

    // Need to redo style calculation if style or options
    // have been specified.
    if (props.style || props.options) {
        props.style = this.getStyle();
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);
    
    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * Set the selected item on the widget.
 *
 * @param {String} item The selected value.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.menuBase.prototype.setSelectedValue = function(item) {
    this.clickedItem = item;
    return true;
};

/**
 * Submits the form. Appends the value of the selected item in the request url.
 *
 * @return {boolean} false to cancel the JavaScript event.
 */
webui.suntheme4_2.widget.menuBase.prototype.submitFormData = function () {
    var theForm = document.getElementById(this.formId);
    var oldAction = theForm.action;
    var oldTarget = theForm.target;
       
    // Specify which option in the menu was clicked.
    theForm.action += "?" + this.id + "_submittedValue=" + this.getSelectedValue();
    theForm.target = "_self";
    theForm.submit();     
    theForm.action = oldAction;
    theForm.target = oldTarget;
    return false;
};

/**
 * This function takes care of traversing through the menu items depending
 * on which key is pressed.
 *
 * @param (String) keyCode The valye of the key which was pressed
 * @param (Event) event The key press event.
 * @param (String) nodeId The id of the menu item. 
 * @return {boolean} true Propagate the javascript event
 */
webui.suntheme4_2.widget.menuBase.prototype.traverseMenu = function(keyCode, event, nodeId) {
    var arr = this.menuId;
    var elem = document.getElementById(nodeId);
    var focusElem = document.getElementById(arr[this.focusPosition]);
    
    if (focusElem.blur) {
        focusElem.blur();
    }
    // Operations to be performed if the arrow keys are pressed.
    if (keyCode >= 37 && keyCode <= 40) {
        
        // Check whether up arrow was pressed.
        if (keyCode == 38) {
            focusElem.className = this.theme.getClassName("MENU_GROUP_CONTAINER");
            this.focusPosition--;
            if (this.focusPosition < 0) {
                this.focusPosition = arr.length-1;
            }
            focusElem = document.getElementById(arr[this.focusPosition]); 

        // Check whether down arrow was pressed
        } else if (keyCode == 40) {
            focusElem.className = 
                this.theme.getClassName("MENU_GROUP_CONTAINER");
            this.focusPosition++;
            if (this.focusPosition == arr.length) {
                this.focusPosition = 0;
            }
            focusElem = document.getElementById(arr[this.focusPosition]);
        }   
        
    if (focusElem.focus) {
        focusElem.focus();
    }        
        focusElem.className = focusElem.className + " "+
            this.theme.getClassName("MENU_FOCUS");           
    // Check if enter key was pressed    
    } else if(keyCode == 13){
        focusElem.className =
            this.theme.getClassName("MENU_GROUP_CONTAINER");
        var val = elem.selectValue;
        this.processEnterKeyPressEvent(val);
       
    }    
    if (webui.suntheme4_2.browser.isIe5up()) {
        window.event.cancelBubble = true;
        window.event.returnValue = false;
    } else {
        event.stopPropagation();
        event.preventDefault();
    }
    return true;
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.passwordField");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.fieldBase");

/**
 * @name webui.suntheme4_2.widget.passwordField
 * @extends webui.suntheme4_2.widget.fieldBase
 * @class This class contains functions for the passwordField widget.
 * @constructor This function is used to construct a passwordField widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.passwordField", webui.suntheme4_2.widget.fieldBase, {
    // Set defaults.
    widgetName: "passwordField" // Required for theme properties.
});

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.passwordField.prototype.getInputClassName = function() {
    if (this.fieldNode.readOnly) {
        return this.widget.getClassName("PASSWORD_FIELD_READONLY", "");
    }

    //invalid style
    var validStyle =  (this.valid == false) 
        ? " " + this.widget.getClassName("PASSWORD_FIELD_INVALID", "")
        : " " + this.widget.getClassName("PASSWORD_FIELD_VALID", "");
    
    // Set default style.    
    return (this.disabled == true)
        ? this.widget.getClassName("PASSWORD_FIELD_DISABLED", "") 
        : this.widget.getClassName("PASSWORD_FIELD", "") + validStyle;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} maxLength 
 * @config {Array} notify 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {int} size 
 * @config {String} style Specify style rules inline.
 * @config {boolean} submitForm
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.passwordField.prototype.setProps = function(props) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.popupMenu");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.menuBase");

/**
 * @name webui.suntheme4_2.widget.popupMenu
 * @extends webui.suntheme4_2.widget.menuBase
 * @class This class contains functions for the popupMenu widget.
 * @constructor This function is used to construct a popupMenu widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.popupMenu", webui.suntheme4_2.widget.menuBase, {
    // Set defaults.
    widgetName: "popupMenu" // Required for theme properties.
});

/**
 * Close the menu. Sets the visibility to false.
 *
 * @return {boolean} false to cancel the JavaScript event.
 */
webui.suntheme4_2.widget.popupMenu.prototype.close = function() {
    if (this.common.isVisibleElement(this.domNode)) {
        if (webui.suntheme4_2.widget.popupMenu.activeMenuId) {
            webui.suntheme4_2.widget.popupMenu.activeMenuId = null;
        }
        if (this.target != null) {
            if (this.target.focus) {
                this.target.focus();
            }   
        }        
        return this.setProps({visible: false});
    }
    return false;    
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.popupMenu.event =
        webui.suntheme4_2.widget.popupMenu.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_popupMenu_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_popupMenu_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_popupMenu_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_popupMenu_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_popupMenu_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_popupMenu_event_submit_end"
    }
};

/**
 * Helper function to create callback to close menu.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.popupMenu.prototype.onCloseMenuCallBack = function(event) {
    // Capture the click and see whether it falls within the boundary of the menu
    // if so do not close the menu.
    var evt = (event) 
        ? event : ((window.event) 
            ? window.event : null);

    var target = (evt.target) 
        ? evt.target 
        : ((evt.srcElement) 
            ? evt.srcElement : null);
        
    // If key pressed and it's NOT the escape key, do NOT cancel.
    if ((evt.type == "keydown") && (evt.keyCode != 27)) {
        return false;
    }
        
    // If the event occured on the menu, do NOT cancel.
    // Instead we let the event propagate to the MenuItem handler.
    // Cannot use 
    while (target != null) {
        if (target.className == "Menu_sun4") {
            return false;
        }
        target = target.parentNode;
    }

    // The above will not catch events on IE which occur on menuitem seperators
    // or empty space between menuitems.
    var menuLeft = this.domNode.offsetLeft;        
    var menuTop = this.domNode.offsetTop;        
    var tmp;

    var menuRight = menuLeft + this.domNode.offsetWidth - this.rightShadow;
    var menuBottom = menuTop + this.domNode.offsetHeight - this.bottomShadow;

    // Having problems with document.body.scrollTop/scrollLeft in firefox.
    // It always seems to return 0. But window.pageXOffset works fine.
    if (window.pageXOffset || window.pageYOffset) {
        var eventX = evt.clientX + window.pageXOffset;
        var eventY = evt.clientY + window.pageYOffset;
    } else if (document.documentElement.scrollLeft ||
            document.documentElement.scrollTop){
        var eventX = evt.clientX + document.documentElement.scrollLeft;
        var eventY = evt.clientY + document.documentElement.scrollTop;
    } else {
        var eventX = evt.clientX + document.body.scrollLeft;
        var eventY = evt.clientY + document.body.scrollTop;
    }
    if ((eventX >= menuLeft) && (eventX <= menuRight) && (eventY >= menuTop) && 
            (eventY <= menuBottom)) {
        return false;
    }
    if ((evt.type == "keydown" && evt.keyCode == 27) || evt.type == "click") {
        this.close();
    }
    return true;
};

/**
 * Use this function to make the menu visible. It takes an event parameter
 * as an argument.It calculates the position where the menu is to be displayed
 * at if one is not already provided by the developer.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.popupMenu.prototype.open = function(event) {    
    var evt = this.widget.getEvent(event);
    var keyCode = this.widget.getKeyCode(evt);
    if(evt.type == "keydown" || evt.type == "keypress") {

        if (!(evt.shiftKey && keyCode == 121)) {
            return false;
        }

        if (webui.suntheme4_2.browser.isIe5up()) {
            window.event.cancelBubble = true;
            window.event.returnValue = false;
        } else {
            evt.stopPropagation();
            evt.preventDefault();
        }         
     }
         
    // Only one menu can be open at a time. Hence, close the previous menu.
    var widget = webui.suntheme4_2.dijit.byId(webui.suntheme4_2.widget.popupMenu.activeMenuId);
    if (widget) {
        widget.close();
    }
    webui.suntheme4_2.widget.popupMenu.activeMenuId = this.id;

    var evt = (event) 
        ? event : ((window.event) 
            ? window.event : null);

    // Note: Must test if event is null. Otherwise, pressing enter key while
    // link has focus generates an error on IE.
    if (evt) {
        evt.cancelBubble = true;
    }

    // If menu already rendered, do nothing.
    if (this.common.isVisibleElement(this.domNode)) {
        return false;
    }
        
    // Check if developer defined styles are set on the widget.
    if (this.style != null && this.style.length > 0) {
        // Mozilla browsers will tell us which styles are set.  If they're not
        // in the list, then the styles appear to be undefined.
        if (this.domNode.style.length != null) {
            for (var i = 0; i < this.domNode.style.length; i++) {
                    var x = this.domNode.style[i];
                if (this.domNode.style[i] == "top")
            this.top = this.domNode.style.top; 
                if (this.domNode.style[i] == "left")
                    this.left = this.domNode.style.left;
        }
        } else {
            // For IE, simply query the style attributes.
            if (this.domNode.style.top != "")
                this.top = this.domNode.style.top;
            if (this.domNode.style.left != "")
            this.left = this.domNode.style.left;
        }
    }

    // Fix: Setting the menu visible here causes flashing. The menu is shown in
    // an old location until moved to the new location in the page.

    // Show the menu. Must do this here, else target properties referenced
    // below will not be valid.
    this.setProps({visible: true});
      
    // If specific positioning specified, then simply use it.  This means
    // no provisions are made to guarantee the menu renders in the viewable area.
    if ((this.top != null) && (this.left != null)) {
        this.domNode.style.left = this.left;
        this.domNode.style.top = this.top;
    } else {
        if (evt == null) {
            return false;
        }
        // No positioning specified, so we calculate the optimal position to guarantee
        // menu is fully viewable.
        // Get the absolute position of the target.
        var target = (evt.target) 
            ? evt.target 
            : ((evt.srcElement) 
                ? evt.srcElement : null);
        var absPos = this.widget.getPosition(target);
        var targetLeft = absPos[0];
        var targetTop = absPos[1];

        // Assume default horizontal position is to align the left edge of the menu with the
        // left edge of the target.
        var menuLeft = targetLeft + this.rightShadow;

        // But can be overridden to align right edges.
        // Check if right edge of menu exceeds page boundary.
        var rightEdge = menuLeft + this.domNode.offsetWidth;
        var pageWidth = this.widget.getPageWidth();
        if (rightEdge > pageWidth) {
            // Shift menu left just enough to bring it into view.
            menuLeft -= (rightEdge - pageWidth);
        }
        
        // If left edge of menu crosses left page boundary, then
        // shift menu right just enough to bring it into view.
        if (menuLeft < 0) {
            menuLeft = 0;
        }

        // Assume default vertical position is to position menu below target.
        var menuTop = targetTop + target.offsetHeight + this.bottomShadow;
        
        // Check if bottom edge of menu exceeds page boundary.
        var bottomEdge = menuTop + this.domNode.offsetHeight - this.bottomShadow;
        if (bottomEdge > this.widget.getPageHeight()) {
            // Shift menu to top of target.
            menuTop = targetTop - this.domNode.offsetHeight;

            // If top edge of menu cross top page boundary, then
            // reposition menu back to below target.
            // User will need to use scrollbars to position menu into view.
            if (menuTop <= 0) {
                menuTop = targetTop + target.offsetHeight - this.bottomShadow;
            }
        }

        // Set new menu position.
        this.domNode.style.left = menuLeft + "px";
        this.domNode.style.top = menuTop + "px";
    }

    // Keep track of the element that opened the popup menu.
    // When the menu is closed, the focus is set back on this element.
    if (evt.target) {
        this.target = evt.target;
    } else if (evt.srcElement) {
        this.target = evt.srcElement;
    }
    if (this.target.blur) {
        this.target.blur();
    }
    
    // Always set the focus on the first element of the menu.
    if (this.focusPosition > 0) {
        var menuNode = document.getElementById(this.menuId[this.focusPosition]); 
        if (menuNode) {
            menuNode.className = this.theme.getClassName("MENU_GROUP_CONTAINER");
        }
    }
    this.focusPosition = 0;
    menuNode = document.getElementById(this.menuId[0]);
    menuNode.className = menuNode.className + " " + 
        this.theme.getClassName("MENU_FOCUS");  
        
    if (menuNode.focus) {
        menuNode.focus();
    }
    return true;        
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.popupMenu.prototype.postCreate = function () {
    // Set public functions.
    this.domNode.open = function(event) { return webui.suntheme4_2.dijit.byId(this.id).open(event); };
    this.domNode.close = function() { return webui.suntheme4_2.dijit.byId(this.id).close(); };

    // Set events.s
    this.dojo.connect(document, "onclick", this, "onCloseMenuCallBack"); 
            
    // escape key should also close menu.
    this.dojo.connect(document, "onkeydown", this, "onCloseMenuCallBack");  

    // Default widths of the drop shadow on each side of the menu.  These MUST 
    // be in pixel units and MUST match the absolute values of the left/top 
    // styles of the "Menu" style class in the CSS.
    this.rightShadow = parseFloat(this.theme.getMessage("Menu.rightShadow"));
    this.bottomShadow = parseFloat(this.theme.getMessage("Menu.bottomShadow"));
    this.shadowContainer.className = this.theme.getClassName("MENU_SHADOW_CONTAINER"); 

    return this.inherited("postCreate", arguments);
};

/**
 * Override the "super class" processKeyPressEvent functionality and close the menu.
 *
 * @param (String) value The "value" of the selected option.  
 * @return {boolean} true The enter key press event completed successfully 
 */
webui.suntheme4_2.widget.popupMenu.prototype.processEnterKeyPressEvent = function(value) {
    this.inherited("processEnterKeyPressEvent", arguments);
    this.close();
    return true;
};

/**
 * Override the "super class" processOnClickEvent functionality and close the menu.
 *
 * @param {String} value The selected value.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.popupMenu.prototype.processOnClickEvent = function(value) {
    this.inherited("processOnClickEvent", arguments);
    this.close();
    return true;
};

/**
 * Traverse through the menu items. This overrides the superclass implementation
 * and handles escape/tab/page up/page down/home/end key press events.
 *
 * @param (String) keyCode The valye of the key which was pressed
 * @param (Event) event The key press event.
 * @param (String) nodeId The id of the menu item. 
 * @return {boolean} true Propagate the javascript event 
 */
webui.suntheme4_2.widget.popupMenu.prototype.traverseMenu = function(keyCode, event, nodeId) {
    
    // Handle the escape key and tab key press
    if (keyCode == 27 || keyCode == 9) {
        var focusElem = document.getElementById(this.menuId[this.focusPosition]);
        focusElem.className = this.theme.getClassName("MENU_GROUP_CONTAINER");        
        this.close();
        return true;
    } else if(keyCode >= 33 && keyCode <= 36) {
        focusElem = document.getElementById(this.menuId[this.focusPosition]);        
        focusElem.className = this.theme.getClassName("MENU_GROUP_CONTAINER");
        
        // Handle the home and page Up keys. Focus is set on the first element.
        if (keyCode == 33 || keyCode == 36) {
            this.focusPosition = 0;
            focusElem = document.getElementById(this.menuId[this.focusPosition]);        
        }
        
        // Handle Page Down and End keys. Focus is set on the last element.
        if (keyCode == 34 || keyCode == 35) {
            this.focusPosition = this.menuId.length - 1;
            focusElem = document.getElementById(this.menuId[this.focusPosition]);        
        }
        if (focusElem.focus) {
            focusElem.focus();
        }                        
        focusElem.className = focusElem.className + " " +
            this.theme.getClassName("MENU_FOCUS"); 
        if (webui.suntheme4_2.browser.isIe5up()) {
            window. event.cancelBubble = true;
            window.event.returnValue = false;
        } else {
            event.stopPropagation();
            event.preventDefault();
        }   
        return true;                 
    }    
    this.inherited("traverseMenu", arguments);
    return true;
};

/**
 * Process submit event.
 *
 * @param {String} execute The string containing a comma separated list 
 * of client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.popupMenu.prototype.submit = function(execute) {
    // Publish an event for custom AJAX implementations to listen for.
    this.publish(webui.suntheme4_2.widget.popupMenu.event.submit.beginTopic, [{
        id: this.id,
        execute: execute,
        value: this.getSelectedValue(),
        endTopic: webui.suntheme4_2.widget.popupMenu.event.submit.endTopic
    }]);
    return true;
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.progressBar");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.progressBar
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the progressBar widget.
 * @constructor This function is used to construct a progressBar widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.progressBar", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.progress = 0;
        this.percentChar = "%";
        this.type = this.determinate;
    },
    busy: "BUSY",
    canceled: "canceled",
    completed: "completed",
    determinate: "DETERMINATE",
    failed: "failed",
    indeterminate: "INDETERMINATE",
    notstarted: "not_started",
    paused: "paused",
    resumed: "resumed",
    stopped: "stopped",
    widgetName: "progressBar"
});

/**
 * This function handles cancel progressBar event.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.cancel = function() {
    clearTimeout(this.timeoutId);

    this.hiddenFieldNode.value = this.canceled;
    if (this.type == this.determinate) {
        this.innerBarContainer.style.width = "0%";
    }
    return this.updateProgress();
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.progressBar.event =
        webui.suntheme4_2.widget.progressBar.prototype.event = {
    /**
     * This closure is used to publish progress events.
     * @ignore
     */
    progress: {
        /** Progress event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_progressBar_event_progress_begin",

        /** Progress event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_progressBar_event_progress_end"
    },

    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_progressBar_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_progressBar_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_progressBar_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_progressBar_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.progressBar.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.height) { props.height = this.height; }
    if (this.width) { props.width = this.width; }
    if (this.bottomText) { props.bottomText = this.bottomText; }
    if (this.busyImage != null) { props.busyImage = this.busyImage; }
    if (this.failedStateText != null) { props.failedStateText = this.failedStateText; }
    if (this.id) { props.id = this.id; }
    if (this.log != null) { props.log = this.log; }
    if (this.logId) { props.logId = this.logId; }
    if (this.logMessage) { props.logMessage = this.logMessage; }
    if (this.overlayAnimation != null) { props.overlayAnimation = this.overlayAnimation; }
    if (this.percentChar) { props.percentChar = this.percentChar; }
    if (this.progress != null) { props.progress = this.progress; }
    if (this.progressImageUrl) { props.progressImageUrl = this.progressImageUrl; }
    if (this.progressControlBottom != null) { props.progressControlBottom = this.progressControlBottom; }
    if (this.progressControlRight != null) { props.progressControlRight = this.progressControlRight; }
    if (this.refreshRate) { props.refreshRate = this.refreshRate; }
    if (this.taskState != null) { props.taskState = this.taskState; }
    if (this.toolTip) { props.toolTip = this.toolTip; }
    if (this.topText) { props.topText = this.topText; }
    if (this.type) { props.type = this.type; }
    if (this.prefix) {props.prefix = this.prefix;}

    return props;
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.progressBar.prototype.getClassName = function() {
    var key = "PROGRESSBAR"; 

    // Get theme property.
    var className = this.theme.getClassName(key);
    if (className == null || className.length == 0) {
	return this.className;
    }
    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This method displays the Bottom Controls if it was hidden.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.isBottomControlVisible = function() {
    return this.common.isVisibleElement(this.bottomControlsContainer);
};

/**
 * This method displays the failed state message and icon if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.isFailedStateMessageVisible = function() {
    return this.common.isVisibleElement(this.failedStateContainer);
};

/**
 * This method displays the log message if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.isLogMsgVisible = function() {
    return this.common.isVisibleElement(this.logContainer);
};

/**
 * This method displays the operation text if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.isOperationTextVisible = function() {
    return this.common.isVisibleElement(this.topTextContainer);
};

/**
 * This method displays the ProgressBar Container if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.isProgressBarContainerVisible = function() {
    return this.common.isVisibleElement(this.barContainer);
};

/**
 * This method displays the ProgressBar if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.isProgressBarVisible = function() {
    return this.common.isVisibleElement(this); 
};

/**
 * This method displays the Right Controls if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.isRightControlVisible = function() {
    return this.common.isVisibleElement(this.rightControlsContainer);
};

/**
 * This method displays the status text if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.isStatusTextVisible = function() {
    return this.common.isVisibleElement(this.bottomTextContainer);
};

/**
 * This function handles pause button event.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.pause = function() {
    clearTimeout(this.timeoutId);

    this.hiddenFieldNode.value = this.paused;
    if (this.type == this.indeterminate) {
        this.innerBarContainer.className =
            this.theme.getClassName("PROGRESSBAR_INDETERMINATE_PAUSED");
    }
    return this.updateProgress();
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.barContainer.id = this.id + "_barContainer";
        this.bottomControlsContainer.id = this.id + "_bottomControlsContainer";
        this.bottomTextContainer.id = this.id + "_bottomTextContainer"; 
        this.failedStateContainer.id = this.id + "_failedStateContainer";
        this.failedLabelContainer.id = this.id + "_failedLabelContainer";
        this.hiddenFieldNode.id = this.id + "_" + "controlType";
        this.hiddenFieldNode.name = this.hiddenFieldNode.id;
        this.innerBarContainer.id = this.id + "_innerBarContainer";
        this.innerBarOverlayContainer.id = this.id + "_innerBarOverlayContainer";
        this.logContainer.id = this.id + "_logContainer";
        this.rightControlsContainer.id = this.id + "_rightControlsContainer";
        this.topTextContainer.id = this.id + "_topTextContainer"; 
    }

    // Set public functions
    this.domNode.cancel = function() { return webui.suntheme4_2.dijit.byId(this.id).cancel(); };
    this.domNode.isBottomControlVisible = function() { return webui.suntheme4_2.dijit.byId(this.id).isBottomControlVisible(); };
    this.domNode.isFailedStateMessageVisible = function() { return webui.suntheme4_2.dijit.byId(this.id).isFailedStateMessageVisible(); };
    this.domNode.isLogMsgVisible = function() { return webui.suntheme4_2.dijit.byId(this.id).isLogMsgVisible(); };
    this.domNode.isOperationTextVisible = function() { return webui.suntheme4_2.dijit.byId(this.id).isOperationTextVisible(); };
    this.domNode.isProgressBarContainerVisible = function() { return webui.suntheme4_2.dijit.byId(this.id).isProgressBarContainerVisible(); };
    this.domNode.isProgressBarVisible = function() { return webui.suntheme4_2.dijit.byId(this.id).isProgressBarVisible(); };
    this.domNode.isRightControlVisible = function() { return webui.suntheme4_2.dijit.byId(this.id).isRightControlVisible(); };
    this.domNode.isStatusTextVisible = function() { return webui.suntheme4_2.dijit.byId(this.id).isStatusTextVisible(); };
    this.domNode.pause = function() { return webui.suntheme4_2.dijit.byId(this.id).pause(); };
    this.domNode.resume = function() { return webui.suntheme4_2.dijit.byId(this.id).resume(); };
    this.domNode.stop = function() { return webui.suntheme4_2.dijit.byId(this.id).stop(); };
    this.domNode.setOnCancel = function(func) { return webui.suntheme4_2.dijit.byId(this.id).setOnCancel(func); };
    this.domNode.setOnComplete = function(func) { return webui.suntheme4_2.dijit.byId(this.id).setOnComplete(func); };
    this.domNode.setOnFail = function(func) { return webui.suntheme4_2.dijit.byId(this.id).setOnFail(func); };
    this.domNode.setBottomControlVisible = function(show) { return webui.suntheme4_2.dijit.byId(this.id).setBottomControlVisible(show); };
    this.domNode.setFailedStateMessageVisible = function(show) { return webui.suntheme4_2.dijit.byId(this.id).setFailedStateMessageVisible(show); };
    this.domNode.setLogMsgVisible = function(show) { return webui.suntheme4_2.dijit.byId(this.id).setLogMsgVisible(show); };
    this.domNode.setOperationTextVisible = function(show) { return webui.suntheme4_2.dijit.byId(this.id).setOperationTextVisible(show); };
    this.domNode.setProgressBarContainerVisible = function(show) { return webui.suntheme4_2.dijit.byId(this.id).setProgressBarContainerVisible(show); };
    this.domNode.setProgressBarVisible = function(show) { return webui.suntheme4_2.dijit.byId(this.id).setProgressBarVisible(show); };
    this.domNode.setRightControlVisible = function(show) { return webui.suntheme4_2.dijit.byId(this.id).setRightControlVisible(show); };
    this.domNode.setStatusTextVisible = function(show) { return webui.suntheme4_2.dijit.byId(this.id).setStatusTextVisible(show); };

    if (this.busyImage == null) {
	this.busyImage = this.widget.getWidgetProps("image", {
            icon: "PROGRESS_BUSY",
            id: this.id + "_busy"             
        });
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function handles resume button event.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.resume = function() {
    clearTimeout(this.timeoutId);

    this.hiddenFieldNode.value = this.resumed;
    if (this.type == this.indeterminate) {
        this.innerBarContainer.className = 
            this.theme.getClassName("PROGRESSBAR_INDETERMINATE");
            
    }
    return this.updateProgress();
};

/**
 * This method hides the Bottom Control.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setBottomControlVisible = function(show) {
    if (show == null) {
        return false;
    }
    this.common.setVisibleElement(this.bottomControlsContainer, show);
    return true;
};

/**
 * This method hides the failed state message and icon area.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setFailedStateMessageVisible = function(show) {
    if (show == null) {
        return false;
    }
    this.common.setVisibleElement(this.failedStateContainer, show);
    return true;
};

/**
 * This method hides the log message area.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setLogMsgVisible = function(show) {
    if (show == null) {
        return false;
    }
    this.common.setVisibleElement(this.logContainer, show);
    return true;
};

/**
 * This function invokes developer define function for cancel event.
 * 
 * @param {Function} func The JavaScript function.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setOnCancel = function(func) {
    if (func) {
        this.funcCanceled = func;
    }
    return true;
};

/**
 * This function invokes developer define function for complete event.
 * 
 * @param {Function} func The JavaScript function.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setOnComplete = function(func) {
    if (func) {
        this.funcComplete = func;
    }
    return true;
};

/**
 * This function invokes developer define function for failed event.
 * 
 * @param {Function} func The JavaScript function.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setOnFail = function(func) {
    if (func) {
        this.funcFailed = func;
    }
    return true;
};

/**
 * This method hides the operation text.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setOperationTextVisible = function(show) {
    if (show == null) {
        return false;
    }
    this.common.setVisibleElement(this.topTextContainer, show);
    return true;
};

/**
 * This function is used to set progress with Object literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} failedStateText
 * @config {String} logMessage
 * @config {int} progress
 * @config {String} status
 * @config {String} taskState
 * @config {String} topText
 * @config {String} type
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setProgress = function(props) {
    if (props == null) {
        return false;
    }
      
    // Adjust max value.
    if (props.progress > 99 
            || props.taskState == this.completed) {
        props.progress = 100;
    }

    // Save properties for later updates.
    this.prototypejs.extend(this, props);    

    // Set status.
    if (props.status) {
        this.widget.addFragment(this.bottomTextContainer, props.status);
    }

    // If top text doesnt get change, dont update.
    if (props.topText) {
        if (props.topText != this.topText) {
            this.widget.addFragment(this.topTextContainer, props.topText);
        }
    }

    // Update log messages.
    if (this.type == this.determinate) { 
        if (props.progress != null && props.progress >= 0 ) {
            this.innerBarContainer.style.width = props.progress + '%';
        }

        if (props.logMessage) {
            var field = webui.suntheme4_2.dijit.byId(this.logId).getInputElement();
            if (field != null) {
                field.value = (field.value)
                   ? field.value + props.logMessage + "\n"
                   : props.logMessage + "\n";
            }
        }

        // Add overlay text.
        if (this.overlayAnimation == true) {
            // NOTE: If you set this value manually, text must be HTML escaped.
            this.widget.addFragment(this.innerBarOverlayContainer, props.progress + "%");
        }
    } 

    // Failed state.
    if (props.taskState == this.failed) {
        clearTimeout(this.timeoutId);
        this.widget.sleep(1000);
        this.setProgressBarContainerVisible(false);
        this.setBottomControlVisible(false);
        this.setRightControlVisible(false);
        this.setLogMsgVisible(false);

        if (props.failedStateText != null) {
            // NOTE: If you set this value manually, text must be HTML escaped.
            this.widget.addFragment(this.failedLabelContainer,
                props.failedStateText + " " + props.progress + this.percentChar);

            this.common.setVisibleElement(this.failedLabelContainer, true);
            this.common.setVisibleElement(this.failedStateContainer, true);
        }
        if (this.funcFailed != null) {
            (this.funcFailed)();
        }
        return true;
    }

    // Cancel state.
    if (props.taskState == this.canceled) {
        clearTimeout(this.timeoutId);
        this.widget.sleep(1000);
        this.setOperationTextVisible(false);
        this.setStatusTextVisible(false);
        this.setProgressBarContainerVisible(false);
        this.setBottomControlVisible(false);
        this.setRightControlVisible(false);
        this.setLogMsgVisible(false);

        if (this.type == this.determinate) {
            this.innerBarContainer.style.width = "0%";
        }
        if (this.funcCanceled != null) {
           (this.funcCanceled)(); 
        }
        return true;    
    }

    // paused state
    if (props.taskState == this.paused) {
        clearTimeout(this.timeoutId);
        return true;
    }

    // stopped state
    if (props.taskState == this.stopped) {
        clearTimeout(this.timeoutId);
        return true;
    }

    if (props.progress > 99 
            || props.taskState == this.completed) {
        clearTimeout(this.timeoutId);
        if (this.type == this.indeterminate) {
            this.innerBarContainer.className =
                this.theme.getClassName("PROGRESSBAR_INDETERMINATE_PAUSED");
        }
        if (this.type == this.busy) {
            this.setProgressBarContainerVisible(false);
        }
        if (this.funcComplete != null) {
           (this.funcComplete)(); 
        }
    }

    // Set progress for A11Y.
    if (props.progress) {
        if (this.bottomTextContainer.setAttributeNS) {
            this.bottomTextContainer.setAttributeNS("http://www.w3.org/2005/07/aaa",
                "valuenow", props.progress);
        }
    }
    return true;
};

/**
 * This method hides the ProgressBar Container.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setProgressBarContainerVisible = function(show) {
    if (show == null) {
        return false;
    }

    if (show == false) {
        this.barContainer.style.display = "none";
    } else {
        this.barContainer.style.display = '';
    }
    return true; 
};

/**
 * This method hides the ProgressBar.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setProgressBarVisible = function(show) {
    if (show == null) {
        return false;
    }
    this.common.setVisibleElement(this, show);
    return true; 
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} bottomText 
 * @config {Object} busyImage 
 * @config {String} failedStateText
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} logId 
 * @config {boolean} logMessage 
 * @config {String} overlayAnimation 
 * @config {String} percentChar 
 * @config {int} progress 
 * @config {String} progressImageUrl 
 * @config {String} progressControlBottom
 * @config {String} progressControlRight 
 * @config {int} refreshRate 
 * @config {String} taskState
 * @config {String} toolTip 
 * @config {String} topText 
 * @config {String} type 
 * @config {boolean} visible Hide or show element.
 * @config {int} width 
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.progressBar.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set tool tip.
    if (props.toolTip) {
        this.barContainer.title = props.toolTip;
    }

    // Add top text.
    if (props.topText) {
        this.widget.addFragment(this.topTextContainer, props.topText); 
        this.common.setVisibleElement(this.topTextContainer, true);
    }

    // Add bottom text.
    if (props.bottomText) {
        this.widget.addFragment(this.bottomTextContainer, props.bottomText);
        this.common.setVisibleElement(this.bottomTextContainer, true);
    }

    if (props.type == this.determinate 
            || props.type == this.indeterminate) {
        // Set style class.
        this.barContainer.className =
            this.theme.getClassName("PROGRESSBAR_CONTAINER");

        // Set height.
        if (props.height != null && props.height > 0) {
            this.barContainer.style.height = props.height + "px;"; 
            this.innerBarContainer.style.height = props.height + "px;";
        }

        // Set width.
        if (props.width != null && props.width > 0) {
            this.barContainer.style.width = props.width + "px;";
        }

        // Add right controls.
        if (props.progressControlRight != null) {
            this.widget.addFragment(this.rightControlsContainer, props.progressControlRight);
            this.common.setVisibleElement(this.rightControlsContainer, true);
        }

        // Add bottom controls.
        if (props.progressControlBottom != null) {
            this.widget.addFragment(this.bottomControlsContainer, props.progressControlBottom);
            this.common.setVisibleElement(this.bottomControlsContainer, true);
        }
    }

    if (props.type == this.determinate) {
        // Set style class.
        this.innerBarContainer.className =
            this.theme.getClassName("PROGRESSBAR_DETERMINATE");

        // Set width.
        if (this.progress != null && this.progress >= 0) {
            this.innerBarContainer.style.width = this.progress + '%';
        }    

        // Add overlay.
        if (props.overlayAnimation == true) {
            if (props.width != null && props.width > 0) {
                this.innerBarOverlayContainer.style.width = props.width + "px;";
            }
            // NOTE: If you set this value manually, text must be HTML escaped.            
            this.widget.addFragment(this.innerBarOverlayContainer, this.progress + "%");
            this.common.setVisibleElement(this.innerBarOverlayContainer, true);
        }

        // Add log.
        if (props.log != null && props.overlayAnimation == false) { 
            this.widget.addFragment(this.logContainer, props.log);
            this.common.setVisibleElement(this.logContainer, true);
        }  
    } else if (props.type == this.indeterminate) {
        // Set style class.
        this.barContainer.className = 
            this.theme.getClassName("PROGRESSBAR_CONTAINER");
        this.innerBarContainer.className = 
            this.theme.getClassName("PROGRESSBAR_INDETERMINATE");
    } else if (props.type == this.busy) {
        // Add busy image.
        if (props.busyImage) {
            if (props.width > 0) {
                props.busyImage.width = props.width;
            } 
            if (props.height > 0) {
                props.busyImage.height = props.height;
            }
            if (props.progressImageUrl != null ) {                                
                if (props.prefix) {               
                    props.busyImage.icon = null;     
                    props.busyImage.src = 
                        webui.suntheme4_2.widget.common.appendPrefix(props.prefix, props.progressImageUrl);               
                }    
            }
            this.widget.addFragment(this.busyImageContainer, props.busyImage);
            this.common.setVisibleElement(this.busyImageContainer, true);
        }
    }

    // Set developer specified image.
    if (props.progressImageUrl != null && (props.type != this.busy)) {
        this.innerBarContainer.style.backgroundImage = 'url(' + props.progressImageUrl + ')';
    }

    // Set A11Y properties.
    if (props.progress != null) {
        if (this.bottomTextContainer.setAttributeNS) {
            this.bottomTextContainer.setAttributeNS(
                "http://www.w3.org/2005/07/aaa", "valuenow", this.progress);
        }
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * This method hides the Right Control.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setRightControlVisible = function(show) {
    if (show == null) {
        return false;
    }
    this.common.setVisibleElement(this.rightControlsContainer, show);
    return true;
};

/**
 * This method hides the status text.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.setStatusTextVisible = function(show) {
    if (show == null) {
        return false;
    }
    this.common.setVisibleElement(this.bottomTextContainer, show);
    return true;
};

/**
 * This function is used to "start" the widget, after the widget has been
 * instantiated.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.startup = function () {
    if (this._started) {
        return false;
    }
    // Start a timer used to periodically publish progress events.
    this.updateProgress();  
    return this.inherited("startup", arguments);
};

/**
 * This function handles stop button event.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.stop = function() {
    clearTimeout(this.timeoutId);

    this.hiddenFieldNode.value = this.stopped;
    if (this.type == this.indeterminate) {
        this.innerBarIdContainer.className =
            this.theme.getClassName("PROGRESSBAR_INDETERMINATE_PAUSED");
    }
    return this.updateProgress();
};

/**
 * Process progress event.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.progressBar.prototype.updateProgress = function() {
    // Publish event.
    if (this.refreshRate > 0) {
        // Publish an event for custom AJAX implementations to listen for.
        this.publish(webui.suntheme4_2.widget.progressBar.event.progress.beginTopic, [{
            id: this.id
        }]);
    }

    // Create a call back function to periodically publish progress events.
    var _id = this.id;
    this.timeoutId = setTimeout(function() {
        // New literals are created every time this function is called, and it's 
        // saved by closure magic.
        webui.suntheme4_2.dijit.byId(_id).updateProgress();
    }, this.refreshRate);
    return true;
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.radioButton");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.checkedBase");

/**
 * @name webui.suntheme4_2.widget.radioButton
 * @extends webui.suntheme4_2.widget.checkedBase
 * @class This class contains functions for the radioButton widget.
 * @constructor This function is used to construct a radioButton widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.radioButton", webui.suntheme4_2.widget.checkedBase, {
    // Set defaults.
    idSuffix: "_rb",
    widgetName: "radioButton" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.radioButton.event =
        webui.suntheme4_2.widget.radioButton.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_radioButton_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_radioButton_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_radioButton_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_radioButton_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_radioButton_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_radioButton_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.radioButton.prototype.getClassName = function() {
    // Set default style.
    var className = (this.disabled == true)
        ? this.widget.getClassName("RADIOBUTTON_SPAN_DISABLED", "")
        : this.widget.getClassName("RADIOBUTTON_SPAN", "");

    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * Helper function to obtain image class names.
 *
 * @return {String} The HTML image element class name.
 */
webui.suntheme4_2.widget.radioButton.prototype.getImageClassName = function() {
    return (this.disabled == true)
        ? this.widget.getClassName("RADIOBUTTON_IMAGE_DISABLED", "")
        : this.widget.getClassName("RADIOBUTTON_IMAGE", "");  
};

/**
 * Helper function to obtain input class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.radioButton.prototype.getInputClassName = function() {
    // Set readOnly style.
    if (this.readOnly == true) {
        return this.widget.getClassName("RADIOBUTTON_READONLY", "");
    }

    // Disabled style.
    return (this.disabled == true)
        ? this.widget.getClassName("RADIOBUTTON_DISABLED", "")
        : this.widget.getClassName("RADIOBUTTON", "");  
};

/**
 * Helper function to obtain label class names.
 *
 * @return {String} The HTML label element class name.
 */
webui.suntheme4_2.widget.radioButton.prototype.getLabelClassName = function() {
    return (this.disabled == true)
        ? this.widget.getClassName("RADIOBUTTON_LABEL_DISABLED", "")
        : this.widget.getClassName("RADIOBUTTON_LABEL", "");  
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {boolean} checked 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} image 
 * @config {String} label 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect 
 * @config {boolean} readOnly 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.radioButton.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.radioButton.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    if (props.name) {
        // IE does not support the name attribute being set dynamically as 
        // documented at:
        //
        // http://msdn.microsoft.com/workshop/author/dhtml/reference/properties/name_2.asp
        //
        // In order to create an HTML element with a name attribute, the name
        // and value must be provided when using the inner HTML property or the
        // document.createElement() function. As a work around, we shall set the
        // attribute via the HTML template using name="${this.name}". In order
        // to obtain the correct value, the name property must be provided to 
        // the widget. Although we're resetting the name below, as the default,
        // this has no affect on IE. 
        this.inputNode.name = props.name;
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.radioButtonGroup");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.checkedGroupBase");

/**
 * @name webui.suntheme4_2.widget.radioButtonGroup
 * @extends webui.suntheme4_2.widget.checkedGroupBase
 * @class This class contains functions for the radioButtonGroup widget.
 * @constructor This function is used to construct a radioButtonGroup widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.radioButtonGroup", webui.suntheme4_2.widget.checkedGroupBase, {
    // Set defaults.
    widgetName: "radioButtonGroup" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.radioButtonGroup.event =
        webui.suntheme4_2.widget.radioButtonGroup.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_radioButtonGroup_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */        
        endTopic: "webui_suntheme4_2_widget_radioButtonGroup_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_radioButtonGroup_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_radioButtonGroup_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.radioButtonGroup.prototype.getClassName = function() {
    // Set default style.
    var className = (this.columns > 1)
        ? this.widget.getClassName("RBGRP_HORIZ", "")
        : this.widget.getClassName("RBGRP_VERT", "");

    return (this.className)
        ? className + " " + this.className
        : className;
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.rating");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.rating
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the rating widget.
 * @constructor This function is used to construct a rating widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.rating", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        // Set defaults for public properties that can be modified.
        this.autoSubmit = false;
        this.includeText = true;
        this.includeNotInterested = true;
        this.includeClear = true;
        this.includeModeToggle = false;
        this.inAverageMode = false;
        this.grade = this.CODE_CLEAR;
        this.averageGrade = 0.0;
        this.maxGrade = 0;
        this.gradeReadOnly = false;
        this.modeReadOnly = false;
        this.tabIndex = -1;

        // Set defaults for private data used internally by the widget.
        this.currentText = null;
        this.clicked = false;
        this.mousedover = false;
        this.createGradeControls = true;
        this.gradeNodes = null;
    },

    // Class constants - must have different values and must be <= 0
    CODE_NOTINTERESTED: -1, // Must match NOT_INTERESTED_GRADE in component/Rating.java
    CODE_MODETOGGLE: -2,
    CODE_CLEAR: 0, // Need not be 0, but must match CLEAR_GRADE in component/Rating.java

    widgetName: "rating" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.rating.event =
        webui.suntheme4_2.widget.rating.prototype.event = {
    /** 
     * This closure is used to process refresh events.
     * @ignore
     */
    refresh: {
        /**
         * Refresh event topics for custom AJAX implementations to listen for.
         */
        beginTopic: "webui_suntheme4_2_widget_rating_event_refresh_begin",

        /**
         * Refresh event topics for custom AJAX implementations to listen for.
         */
        endTopic: "webui_suntheme4_2_widget_rating_event_refresh_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_rating_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_rating_event_submit_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_rating_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_rating_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.rating.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    if (this.autoSubmit != null)
        props.autoSubmit = this.autoSubmit;
    if (this.includeText != null)
        props.includeText = this.includeText;
    if (this.includeNotInterested != null)
        props.includeNotInterested = this.includeNotInterested;
    if (this.includeClear != null)
        props.includeClear = this.includeClear;
    if (this.includeModeToggle != null)
        props.includeModeToggle = this.includeModeToggle;
    if (this.inAverageMode != null)
        props.inAverageMode = this.inAverageMode;
    if (this.grade != null)
        props.grade = this.grade;
    if (this.averageGrade != null)
        props.averageGrade = this.averageGrade;
    if (this.maxGrade != null)
        props.maxGrade = this.maxGrade;
    if (this.gradeReadOnly != null)
        props.gradeReadOnly = this.gradeReadOnly;
    if (this.modeReadOnly != null)
        props.modeReadOnly = this.modeReadOnly;
    if (this.notInterestedHoverText != null)
        props.notInterestedHoverText = this.notInterestedHoverText;
    if (this.gradeHoverTexts != null)
        props.gradeHoverTexts = this.gradeHoverTexts;
    if (this.clearHoverText != null)
        props.clearHoverText = this.clearHoverText;
    if (this.modeToggleHoverTexts != null)
        props.modeToggleHoverTexts = this.modeToggleHoverTexts;
    if (this.notInterestedAcknowledgedText != null)
        props.notInterestedAcknowledgedText = this.notInterestedAcknowledgedText;
    if (this.gradeAcknowledgedText != null)
        props.gradeAcknowledgedText = this.gradeAcknowledgedText;
    if (this.clearAcknowledgedText != null)
        props.clearAcknowledgedText = this.clearAcknowledgedText;
    if (this.modeToggleAcknowledgedTexts != null)
        props.modeToggleAcknowledgedTexts = this.modeToggleAcknowledgedTexts;
    if (this.tabIndex != null)
        props.tabIndex = this.tabIndex;

    return props;
}; // getProps

/**
 * Set the text to be displayed in the textContainer.  If the specified text
 * is null or empty, then display a non-breaking space character.
 *
 * @param {String} text  the text to be displayed
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._setText = function(text) {
    if (this.textContainer != null) {
        if (text != null && (text.replace(/^\s+/g, '').replace(/\s+$/g, '') == "" ))
            text = null;
        this.textContainer.innerHTML = (text == null ? "&nbsp;" : text);
        this.currentText = this.textContainer.innerHTML;
    }
    return true;
}; // _setText

/**
 * Get the image info associated with a given grade control for a given grade.
 *
 * @param {boolean} averageMode true if the image info to be returned is within
 *           the context of the widget displaying average mode;  false if within
 *           the context of displaying in normal mode.
 * @param {int} grade the grade of the widget.  This should be the average grade (if
 *           averageMode is true), otherwise the user's grade.
 * @param {int} rank the grade rank assigned to the control whose image info is
 *           being returned.
 * @return {Array} array containing the image info.  The 1st element is the className
 *           the 2nd element is the image width.
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._getGradeImageInfo = function(
            averageMode, grade, rank) {
    var className = null;
    var width = null;

    if (averageMode == true) {
        // Compute the difference between the grade being displayed and the grade rank
        // associated with this image.
        var diff = grade - rank;

        // Show correct image based on diff
        if (diff < (0 -.5)) {
            // Difference is more than half-grade below zero.
            // Show empty grade.
            className = this.theme.getClassName("RATING_GRADE_EMPTY_IMAGE");
            width = parseInt(this.theme.getProperty("images", "RATING_GRADE_EMPTY_WIDTH"));
        } else if (diff < 0) {
            // Difference is less than a half-grade below 0.
            // Show average half-full grade
            className = this.theme.getClassName("RATING_GRADE_AVG_HALF_IMAGE");
            width = parseInt(this.theme.getProperty("images", "RATING_AVG_GRADE_HALF_WIDTH"));
        } else {
            // Difference is 0 or higher.
            // Show average full grade
            className = this.theme.getClassName("RATING_GRADE_AVG_FULL_IMAGE");
            width = parseInt(this.theme.getProperty("images", "RATING_AVG_GRADE_FULL_WIDTH"));
        }
    } else {
        if (rank <= grade) {
            // Show full user's grade
            className = this.theme.getClassName("RATING_GRADE_FULL_IMAGE");
            width = parseInt(this.theme.getProperty("images", "RATING_GRADE_FULL_WIDTH"));
        } else {
            // Show empty grade
            className = this.theme.getClassName("RATING_GRADE_EMPTY_IMAGE");
            width = parseInt(this.theme.getProperty("images", "RATING_GRADE_EMPTY_WIDTH"));
        }
    }
    return [className, width];
}; // _getGradeImageInfo

/**
 * Preview component state based on mouse hover.
 *
 * @param {int} code indicates on which image the event occurs.
 *              Can be one of the widget constants:
 *                  CODE_NOTINTERESTED, CODE_MODETOGGLE, or CODE_CLEAR
 *              or 1->maxGrade
 * @param {boolean} isMouseOver  true if mouseover event, otherwise false.
 *                  false implies an "undo" of the preview state.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._previewState = function(code, isMouseOver) {
    // Determine if we will be displaying average grade.
    //
    var displayingAvg = false;
    if ((code == this.CODE_MODETOGGLE) && isMouseOver)
        // Moused over modeToggle control, so we will preview the inverse mode of
        // current display mode.
        displayingAvg = !this.inAverageMode;
    else {
        // If mouseout from any control then we will preview what the
        // current display mode is.  This is akin to an "undo" of the preview state
        // from a previous mouseover of the modeToggle control.  Otherwise on a
        // mouseover of any control BUT modeToggle, we will not preview the
        // average grade.
        displayingAvg = (!isMouseOver ? this.inAverageMode : false);
    }
    
    // Determine which grade to display.
    //
    var displayingGrade;
    if (!isMouseOver) {
        // Mouseout from any control.  Then we will preview that the
        // current display grade is.  This is akin to an "undo" of the preview state
        // from a previous mouseover.  The grade is either the average grade (if
        // in average mode) or the user's grade (if in normal mode).
        displayingGrade = this.inAverageMode ? this.averageGrade : this.grade;
    }

    else if (code == this.CODE_MODETOGGLE)
        // Mouseover modeToggle.  Display either the average grade or the user's grade.
        displayingGrade = displayingAvg ? this.averageGrade : this.grade;

    else if (code == this.CODE_CLEAR)
        // Mouseover clear, so no grade to display.
        displayingGrade = 0;
    else
        // Display the grade associated with the control on which the event occurred.
        displayingGrade = code;
    

    var hoverClass = this.theme.getClassName("RATING_HOVER");
    var hoverText = null;

    // ModeToggle image
    if ((this.includeModeToggle == true) && (this.modeToggleNode != null)) {
        // Set style class for this image
        if (displayingAvg)
            this.modeToggleNode.className = this.theme.getClassName("RATING_MODE_AVERAGE_IMAGE");
        else
            this.modeToggleNode.className = this.theme.getClassName("RATING_MODE_NORMAL_IMAGE");

        // Since we reset the className above, we may need to add back the hover class.
        if (!this.modeReadOnly)
            this.common.addStyleClass(this.modeToggleNode, hoverClass);

        // If mouseover on modeToggle, set the hover text to display
        if ((code == this.CODE_MODETOGGLE) && isMouseOver && (this.modeToggleHoverTexts != null)) {
            hoverText = (displayingAvg
                ? (this.modeToggleHoverTexts.length == 2 ? this.modeToggleHoverTexts[1] : null)
                : this.modeToggleHoverTexts[0]);
        }
    }

    // Not interested image
    if ((this.includeNotInterested == true) && (this.notInterestedNode != null)) {
        // Set style class for this image
        if (displayingGrade == this.CODE_NOTINTERESTED)
            this.notInterestedNode.className = this.theme.getClassName("RATING_NOT_INTERESTED_ON_IMAGE");
        else
            this.notInterestedNode.className = this.theme.getClassName("RATING_NOT_INTERESTED_OFF_IMAGE");

        // Since we reset the className above, we may need to add back the hover class.
        if (!this.gradeReadOnly)
            this.common.addStyleClass(this.notInterestedNode, hoverClass);

        // If mouseover on notInterested, set the hover text to display
        if (code == this.CODE_NOTINTERESTED && isMouseOver && this.notInterestedHoverText != null)
            hoverText = this.notInterestedHoverText;
    }

    // Clear image
    if ((this.includeClear == true) && (this.clearNode != null)) {
        // Set style class and hover text for this image.  It will always be off unless mouseover
        // of the clear control.
        if (code == this.CODE_CLEAR && isMouseOver) {
            this.clearNode.className = this.theme.getClassName("RATING_CLEAR_ON_IMAGE");
            if (this.clearHoverText != null)
                hoverText = this.clearHoverText;
        }
        else
            this.clearNode.className = this.theme.getClassName("RATING_CLEAR_OFF_IMAGE");

        // Since we reset the className above, we may need to add back the hover class.
        if (!this.gradeReadOnly)
            this.common.addStyleClass(this.clearNode, hoverClass);
    }

    // Grade images
    for (var i = 1; i <= this.maxGrade; i++) {
        if (i > this.gradeNodes.length)
            break;

        // If this grade image is the one moused over, then get it's hover text.
        if (isMouseOver && (code != this.CODE_MODETOGGLE) && (i == displayingGrade)) {
            if ((this.gradeHoverTexts != null) && (i <= this.gradeHoverTexts.length))
                hoverText = this.gradeHoverTexts[i-1];
        }

        // Set appropriate class for this grade image
        var imageInfo = this._getGradeImageInfo(displayingAvg, displayingGrade, i);
        this.gradeNodes[i-1].className = imageInfo[0];

        // Since we reset the className above, we may need to add back the hover class.
        if (!this.gradeReadOnly)
            this.common.addStyleClass(this.gradeNodes[i-1], hoverClass);
    }

    // Set hover text in textContainer
    this._setText(hoverText);

    return true;
}; // _previewState

/**
 * Modify component state based on mouse click, basically updating our
 * state based on the current preview.
 *
 * @param {int} code indicates on which image the event occurs.
 *              Can be one of the object constants:
 *                  CODE_NOTINTERESTED, CODE_MODETOGGLE, or CODE_CLEAR
 *              or 1->maxGrade
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._modifyState = function(code) {
    if (code == this.CODE_MODETOGGLE) {
        // Toggle mode
        this.inAverageMode = !this.inAverageMode;

        // If acknowledgement text configured for when toggling mode, then render it for the new mode.
        var acknowledgedText = null;
        if (this.modeToggleAcknowledgedTexts != null) {
            acknowledgedText = (this.inAverageMode
                ? (this.modeToggleAcknowledgedTexts.length == 2 ? this.modeToggleAcknowledgedTexts[1] : null)
                : this.modeToggleAcknowledgedTexts[0]);
        }
        this._setText(acknowledgedText);

    }
    else {
        // Normal (not average) mode
        this.inAverageMode = false;

        // Update the widget grade, as well as post it to the hidden input
        // field so it's available to be submitted if the page is submitted.
        this.grade = code;
        this.hiddenFieldNode.value = this.grade;

        // Render acknowledged text for image clicked
        var acknowledgedText = null;
        if (code == this.CODE_CLEAR)
            acknowledgedText = this.clearAcknowledgedText;
        else if (code == this.CODE_NOTINTERESTED)
            acknowledgedText = this.notInterestedAcknowledgedText;
        else
            acknowledgedText = this.gradeAcknowledgedText;
        this._setText(acknowledgedText);

        if (this.autoSubmit)
            this.submit();
    }
    return true;
}; // _modifyState

/**
 * Handler for mouseout and mouseover events.
 *
 * @param {int} code indicates on which image the event occurs.
 *              Can be one of the object constants:
 *                  CODE_NOTINTERESTED, CODE_MODETOGGLE, or CODE_CLEAR
 *              or 1->maxGrade
 * @param {boolean} isMouseOver  true if mouseover event, otherwise false
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._onMouseCallback = function(code, isMouseOver) {
    // Return if either:
    //   1. this is a mouse over, or
    //   2. this is a mouse out, and the component is not considered in a "mousedover" state.
    //      (this occurs if we moused in to a grade control, but gradeReadOnly was true,
    //       or if we moused into the modeToggle control, but modeReadOnly was true)
    if ( (this.gradeReadOnly && (code != this.CODE_MODETOGGLE))
        || (this.modeReadOnly && (code == this.CODE_MODETOGGLE)) ) {
        if (isMouseOver || !this.mousedover)
            return true;
    }

    // Show a preview of the component state if the mouse would be clicked.
    this._previewState(code, isMouseOver);
    
    // Remember we just processed a mouseover/mouseout (ie., non-click) event
    this.mousedover = isMouseOver; 
    this.clicked = false; 
    return true;
}; // _onMouseCallback

/**
 * Handler for click events.
 *
 * @param {int} code indicates on which image the event occurs:
 *              Can be one of the object constants:
 *                  CODE_NOTINTERESTED, CODE_MODETOGGLE, or CODE_CLEAR
 *              or 1->maxGrade
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._onClickCallback = function(code) {
    // Return if either:
    //   1. clicked on a grade control when gradeReadOnly is true, or
    //   2. clicked on modeTogglecontrol, but modeReadOnly is true, or
    //   3. We just processed a click and there's been no new mouse event
    if ( (this.gradeReadOnly && (code != this.CODE_MODETOGGLE)) 
        || (this.modeReadOnly && (code == this.CODE_MODETOGGLE))
        || this.clicked)
        return true;

    // Publish event prior to changing widget state.
    this.publish(webui.suntheme4_2.widget.rating.event.state.beginTopic, [{
        id: this.id
    }]);

    // Modify the component state permanently
    this._modifyState(code);
    
    // Remember we just processed a click (ie, non-mouseover/mouseout) event
    this.clicked = true;
    this.mousedover = false; 

    // Publish event after changing widget state.
    this.publish(webui.suntheme4_2.widget.rating.event.state.endTopic, [{
        id: this.id
    }]);
    return true;
}; // _onClickCallback

/**
 * Handler for focus events.
 *
 * @param {int} code indicates on which image the event occurs:
 *              Can be one of the object constants:
 *                  CODE_NOTINTERESTED, CODE_MODETOGGLE, or CODE_CLEAR
 *              or 1->maxGrade
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._onFocusCallback = function(code) {
    // TBD
//    console.log("_onFocusCallback: code=" + code);
    return true;
}; // _onFocusCallback

/**
 * Setup handler for mouseover events.  We setup in this manner so
 * we can use closure magic to define the scope for the handler with 
 * necessary information to properly process the event.
 *
 * @param {int} code indicates on which image the event occurs:
 * @return {Object} the event handler
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._createOnMouseOverCallback = function(code) {
    var _code = code;
    var _this = this;
    return function(event) {
        _this._onMouseCallback(_code, true);
    }
}; // _createOnMouseOverCallback

/**
 * Setup handler for mouseout events.  We setup in this manner so
 * we can use closure magic to define the scope for the handler with 
 * necessary information to properly process the event.
 *
 * @param {int} code indicates on which image the event occurs:
 * @return {Object} the event handler
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._createOnMouseOutCallback = function(code) {
    var _code = code;
    var _this = this;
    return function(event) {
        _this._onMouseCallback(_code, false);
    }
}; // _createOnMouseOutCallback

/**
 * Setup handler for click events.  We setup in this manner so
 * we can use closure magic to define the scope for the handler with 
 * necessary information to properly process the event.
 *
 * @param {int} code indicates on which image the event occurs:
 * @return {Object} the event handler
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._createOnClickCallback = function(code) {
    var _code = code;
    var _this = this;
    return function(event) {
        _this._onClickCallback(_code);
    }
}; // _createOnClickCallback

/**
 * Setup handler for focus events.  We setup in this manner so
 * we can use closure magic to define the scope for the handler with 
 * necessary information to properly process the event.
 *
 * @param {int} code indicates on which image the event occurs:
 * @return {Object} the event handler
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._createOnFocusCallback = function(code) {
    var _code = code;
    var _this = this;
    return function(event) {
        _this._onFocusCallback(_code);
    }
}; // _createOnFocusCallback

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.rating.prototype.postCreate = function () {
    // Set IDs for the controls
    this.notInterestedID = this.id + "_notInterested";
    this.clearID = this.id + "_clear";
    this.modeToggleID = this.id + "_modeToggle";
    this.gradeID = this.id + "_grade";  // actual ID will have ordinal value appended
    this.textID = this.id + "_text";

    // Configure hidden field to hold the submitted value
    this.hiddenFieldNode.id = this.id + "_submitValue";
    this.hiddenFieldNode.name = this.hiddenFieldNode.id;

    // Listen for post-submit events
    this.subscribe(this.event.submit.endTopic, this, "_submitCallback");

    // Initialize maintenance of width dimensions for controlContainer images.
    // Required because we will always need to maintain the width of the control
    // area, which depends on the number of images to be rendered.  The computed
    // width will be a "best fit" - just enough to encompass the required
    // image controls.
    this.imageWidths = new Array();
    this.imageWidths["notInterested"] = 0;
    this.imageWidths["grades"] = 0;
    this.imageWidths["spacer"] = 0;
    this.imageWidths["clear"] = 0;
    this.imageWidths["modeToggle"] = 0;

    // Set classes on elements that don't change.
    this.common.addStyleClass(this.domNode,
        this.theme.getClassName("RATING"));
    this.common.addStyleClass(this.textContainer,
        this.theme.getClassName("RATING_TEXT_CONTAINER"));
    this.common.addStyleClass(this.controlContainer,
        this.theme.getClassName("RATING_CONTROL_CONTAINER"));
    this.common.addStyleClass(this.spacerNode,
        this.theme.getClassName("RATING_SPACER_NODE"));

    // Configure event handlers for the notInterested control
    this.dojo.connect(this.notInterestedNode,
        "onmouseover", this._createOnMouseOverCallback(this.CODE_NOTINTERESTED));
    this.dojo.connect(this.notInterestedNode,
        "onclick", this._createOnClickCallback(this.CODE_NOTINTERESTED));
    this.dojo.connect(this.notInterestedNode,
        "onmouseout", this._createOnMouseOutCallback(this.CODE_NOTINTERESTED));
/* TBD
    this.dojo.connect(this.notInterestedNode,
        "onfocus", this._createOnFocusCallback(this.CODE_NOTINTERESTED));
*/

    // Configure event handlers for the clear control
    this.dojo.connect(this.clearNode,
        "onmouseover", this._createOnMouseOverCallback(this.CODE_CLEAR));
    this.dojo.connect(this.clearNode,
        "onclick", this._createOnClickCallback(this.CODE_CLEAR));
    this.dojo.connect(this.clearNode,
        "onmouseout", this._createOnMouseOutCallback(this.CODE_CLEAR));
/* TBD
    this.dojo.connect(this.clearNode,
        "onfocus", this._createOnFocusCallback(this.CODE_CLEAR));
*/

    // Configure event handlers for the modeToggle control
    this.dojo.connect(this.modeToggleNode,
        "onmouseover", this._createOnMouseOverCallback(this.CODE_MODETOGGLE));
    this.dojo.connect(this.modeToggleNode,
        "onclick", this._createOnClickCallback(this.CODE_MODETOGGLE));
    this.dojo.connect(this.modeToggleNode,
        "onmouseout", this._createOnMouseOutCallback(this.CODE_MODETOGGLE));
/* TBD
    this.dojo.connect(this.modeToggleNode,
        "onfocus", this._createOnFocusCallback(this.CODE_MODETOGGLE));
*/

/* TBD
    this.domNode.tabIndex = -1;
    this.dojo.connect(this.domNode,
        "onfocus", this._createOnFocusCallback(99));
*/
    return this.inherited("postCreate", arguments);
}; // postCreate

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * <p>
 * @config {boolean} autoSubmit
 *                   Indicates whether the grade is automatically submitted to the 
 *                   server via an Ajax request immediately after the grade is 
 *                   selected.  The default is false - it is NOT automatically submitted.
 * </p><p>
 * @config {double}  averageGrade
 *                   The average grade the general user population has assigned to the item.
 *                   Must be between 0 and the maximum grade.  The default is 0.
 * </p><p>
 * @config {String}  clearAcknowledgedText
 *                   The acknowledged text for the clear control.  There is no default.
 * </p><p>
 * @config {String}  clearHoverText
 *                   The hover text for the clear control.  There is no default.
 * </p><p>
 * @config {int}     grade
 *                   The grade (number of "stars") the user has assigned the item.
 *                   Use the keyword "notInterested" for a not interested grade
 *                   and the keyword "clear" to clear the grade (effective, set it to 0).
 *                   The default is "clear".
 * </p><p>
 * @config {String}  gradeAcknowledgedText
 *                   The text that is displayed after clicking on a grade component.  There is no default.
 * </p><p>
 * @config {Array}   gradeHoverTexts
 *                   The hover texts that will be used for the grade controls, ordered from lowest 
 *                   to highest rating.  That is, hoverTexts[0] will be the hover text associated 
 *                   with the lowest rating;  hoverTexts[hoverTexts.length-1] with the highest rating.
 *                   Null can be specified as a member of the array.  There are no defaults.
 * </p><p>
 * @config {boolean} gradeReadOnly
 *                   Indicates whether the grade of this rating component can be changed by the user.
 *                   The default is false - it is NOT read-only, and therefore can be changed by the user.
 * </p><p>
 * @config {boolean} inAverageMode
 *                   Indicates whether the component will be rendered displaying the average grade.
 *                   The default is false, the component will be rendered showing the user's rating (normal mode).
 * </p><p>
 * @config {boolean} includeClear
 *                   Indicates whether a control to clear the user's rating should be displayed.
                     The default is true.
 * </p><p>
 * @config {boolean} includeModeToggle
 *                   Indicates whether a control to toggle the mode (to show the average 
                     rating or the user's rating) should be rendered.  The default is false.
 * </p><p>
 * @config {boolean} includeNotInterested
 *                   Indicates whether a control to allow the user to assign a
 *                   "not interested" rating should be rendered.  The default is true.
 * </p><p>
 * @config {boolean} includeText
 *                   Indicates whether an area for hover or post-click acknowledeged
 *                   text should be rendered.  The default is true.
 * </p><p>
 * @config {int}     maxGrade
 *                   The maximum grade (number of "stars") this rating instance allows.
 *                   There is no default, and so must be set.
 * </p><p>
 * @config {boolean} modeReadOnly
 *                   Indicates whether the mode of this rating component can be changed by the user.
 *                   The default is false - it is NOT read-only, and therefore can be changed by the user.
 * </p><p>
 * @config {Array}   modeToggleAcknowledgedTexts
 *                   The acknowledged texts to be used for the mode toggle control.  The first element
 *                   of the array is the acknowledged text displayed after clicking on the mode toggle 
 *                   control to preview the user's rating (normal mode).  The second element is 
 *                   the text displayed after clicking to preview the average rating (average mode).
 *                   Null can be specified as a member of the array.
 * </p><p>
 * @config {Array}   modeToggleHoverTexts
 *                   The hover texts to be used for the mode toggle control.  The first element 
 *                   of the array is the hover text displayed when hovering over the mode toggle 
 *                   control to preview the user's rating (normal mode).  The second element is 
 *                   the text displayed when hovering to preview the average rating (normal mode).
 *                   Null can be specified as a member of the array.
 * </p><p>
 * @config {String}  notInterestedAcknowledgedText
 *                   The acknowledged text for the "not interested" control.  There is no default.
 * </p><p>
 * @config {String}  notInterestedHoverText
 *                   The hover text for the "not interested" control.  There is no default.
 * </p><p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.rating.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // We are trying to deal with a state change which requires
    // knowing what the current state of the widget is and the
    // state as defined in props. In order to compare the states
    // it must be done in setProps before the "props" are extended
    // onto the widget. At that point there is no difference between
    // "props" and "this".
    //
    if (props.gradeReadOnly != null) {
        if (props.gradeReadOnly != this.gradeReadOnly)
            this.createGradeControls = true;
    }
    if (props.modeReadOnly != null) {
        if (props.modeReadOnly != this.modeReadOnly)
            this.createGradeControls = true;
    }
    if (props.inAverageMode != null) {
        if (props.inAverageMode != this.inAverageMode)
            this.createGradeControls = true;
    }
    if (props.maxGrade != null) {
        if (props.maxGrade != this.maxGrade)
            this.createGradeControls = true;
    }
    if (props.averageGrade != null) {
        var f = parseFloat(props.averageGrade);
        if (!isNaN(f) && (f != this.averageGrade))
            this.createGradeControls = true;
    }
    if (props.grade != null) {
        var newGrade = this.grade;
	if (props.grade == "notInterested")
            newGrade = this.CODE_NOTINTERESTED;
        else if (props.grade == "clear")
            newGrade = this.CODE_CLEAR;
        else {
            var n = parseInt(props.grade);
            if (!isNaN(n))
                newGrade = n;
        }
        if (newGrade != this.grade)
            this.createGradeControls = true;
    }
    if (props.maxGrade != null) {
        if (props.maxGrade != this.maxGrade)
            this.createGradeControls = true;
    }

    // Extend widget object for later updates.
    var retVal = this.inherited("setProps", arguments);

    // Reset indicators that were dependent on state change.
    this.createGradeControls = false;

    return retVal;

}; // setProps

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    var gradeRightMargin = parseInt(this.theme.getMessage("rating.gradeMarginRight"));
    var hiddenClass = this.theme.getClassName("HIDDEN");
    var hoverClass = this.theme.getClassName("RATING_HOVER");

    // Assume width of control container does NOT need to be recalculated, prove otherwise based on 
    // properties that change.
    var changeControlWidth = false;

    // Text area
    if (props.includeText != null) {
        this.textContainer.id = this.textID;
        var classNames = this.textContainer.className.split(" ");
        if (props.includeText == true) {
            // Remove hidden class
            if (this.common.checkStyleClasses(classNames, hiddenClass))
                this.common.stripStyleClass(this.textContainer, hiddenClass);
        } else {
            // Add hidden class
            if (!this.common.checkStyleClasses(classNames, hiddenClass))
                this.common.addStyleClass(this.textContainer, hiddenClass);
        }
    }

    // Not Interested control
    if (props.includeNotInterested != null) {
        this.notInterestedNode.id = this.notInterestedID;
        var classNames = this.notInterestedNode.className.split(" ");
        var imageWidth = 0;

        if (this.includeNotInterested == true) {
            var notInterestedOff = this.theme.getClassName("RATING_NOT_INTERESTED_OFF_IMAGE");
            var notInterestedOn = this.theme.getClassName("RATING_NOT_INTERESTED_ON_IMAGE");

            // Remove hidden class
            if (this.common.checkStyleClasses(classNames, hiddenClass))
                this.common.stripStyleClass(this.notInterestedNode, hiddenClass);

            if (this.grade == this.CODE_NOTINTERESTED) {
                // Remove notInterested OFF class
                if (this.common.checkStyleClasses(classNames, notInterestedOff))
                    this.common.stripStyleClass(this.notInterestedNode, notInterestedOff);

                // Add notInterested ON class
                if (!this.common.checkStyleClasses(classNames, notInterestedOn))
                    this.common.addStyleClass(this.notInterestedNode, notInterestedOn);

                // Get image width
                imageWidth = parseInt(this.theme.getProperty("images", "RATING_NOT_INTERESTED_ON_WIDTH"));

            } else {
                // Remove notInterested ON class
                if (this.common.checkStyleClasses(classNames, notInterestedOn))
                    this.common.stripStyleClass(this.notInterestedNode, notInterestedOn);

                // Add notInterested OFF class
                if (!this.common.checkStyleClasses(classNames, notInterestedOff))
                    this.common.addStyleClass(this.notInterestedNode, notInterestedOff);

                // Get image width
                imageWidth = parseInt(this.theme.getProperty("images", "RATING_NOT_INTERESTED_OFF_WIDTH"));
            }

            // Add right margin
            imageWidth += gradeRightMargin;

        } else {
            // Add hidden class
            if (!this.common.checkStyleClasses(classNames, hiddenClass))
                this.common.addStyleClass(this.notInterestedNode, hiddenClass);
        }

        // Record image width if changing and flag that control container width must be recomputed.
        if (imageWidth != this.imageWidths["notInterested"]) {
            this.imageWidths["notInterested"] = imageWidth;
            changeControlWidth = true;
        }
    }
    if (this.includeNotInterested) {
        if (this.gradeReadOnly)
            this.common.stripStyleClass(this.notInterestedNode, hoverClass);
        else
            this.common.addStyleClass(this.notInterestedNode, hoverClass);
    }

    // If creating grade controls, delete existing ones if they exist
    if (this.createGradeControls == true) {
        for (var i = 1; (this.gradeNodes != null) && (i <= this.gradeNodes.length); i++) {
            var node = this.gradeNodes[i-1];
            if (node != null)
                node.parentNode.removeChild(node);
        }
        this.gradeNodes = null;
        this.imageWidths["grades"] = 0;
    }

    // Grade controls
    if ((this.createGradeControls == true) && (this.maxGrade > 0)) {
        var imageWidths = 0;
        this.gradeNodes = new Array(this.maxGrade);

        for (var i = 1; i <= this.maxGrade; i++) {
            // Clone the gradeNode element and assign ID
            var clone = this.gradeNode.cloneNode(false);
            clone.id = this.gradeID + i;

            // Get image info for this grade control for the display mode
            var imageInfo = null;
            if (this.inAverageMode)
                imageInfo = this._getGradeImageInfo(true, this.averageGrade, i);
            else
                imageInfo = this._getGradeImageInfo(false, this.grade, i);

            // Set class for this grade control
            clone.className = imageInfo[0];
            if (!this.gradeReadOnly)
                this.common.addStyleClass(clone, hoverClass);

            // Maintain running image width for grades
            imageWidths += (imageInfo[1] + gradeRightMargin);

            // Add the clone to the grade container
            this.gradeContainer.appendChild(clone);

            // Save handle to cloned node for quick access later on.
            this.gradeNodes[i-1] = clone;

            // Configure event handlers for this grade control
            this.dojo.connect(clone, "onmouseover", this._createOnMouseOverCallback(i));
            this.dojo.connect(clone, "onclick", this._createOnClickCallback(i));
            this.dojo.connect(clone, "onmouseout", this._createOnMouseOutCallback(i));
/* TBD
            this.dojo.connect(clone, "onfocus", this._createOnFocusCallback(i));
*/
        }

        // Record image widths if changing and flag that control container width must be recomputed.
        if (imageWidths != this.imageWidths["grades"]) {
            this.imageWidths["grades"] = imageWidths;
            changeControlWidth = true;
        }
    }

    // Clear grade control
    if (props.includeClear != null) {
        this.clearNode.id = this.clearID;
        var classNames = this.clearNode.className.split(" ");
        var imageWidth = 0;

        if (props.includeClear == true) {
            var clearOff = this.theme.getClassName("RATING_CLEAR_OFF_IMAGE");
            var clearOn = this.theme.getClassName("RATING_CLEAR_ON_IMAGE");

            // Remove hidden class
            if (this.common.checkStyleClasses(classNames, hiddenClass))
                this.common.stripStyleClass(this.clearNode, hiddenClass);

            // Remove clear ON class.  Shouldn't need to since it's only on upon hover, but Murphy's Law ...
            if (this.common.checkStyleClasses(classNames, clearOn))
                this.common.stripStyleClass(this.clearNode, clearOn);

            // Add clear OFF class
            if (!this.common.checkStyleClasses(classNames, clearOff))
                this.common.addStyleClass(this.clearNode, clearOff);

            // Get image width
            imageWidth = parseInt(this.theme.getProperty("images", "RATING_CLEAR_OFF_WIDTH")) + gradeRightMargin;

        } else {
            // Add hidden class
            if (!this.common.checkStyleClasses(classNames, hiddenClass))
                this.common.addStyleClass(this.clearNode, hiddenClass);
        }

        // Record image width if changing and flag that control container width must be recomputed.
        if (imageWidth != this.imageWidths["clear"]) {
            this.imageWidths["clear"] = imageWidth;
            changeControlWidth = true;
        }
    }
    if (this.includeClear) {
        if (this.gradeReadOnly)
            this.common.stripStyleClass(this.clearNode, hoverClass);
        else
            this.common.addStyleClass(this.clearNode, hoverClass);
    }

    // Mode toggle control
    if (props.includeModeToggle != null) {
        this.modeToggleNode.id = this.modeToggleID;
        var classNames = this.modeToggleNode.className.split(" ");
        var imageWidth = 0;

        if (props.includeModeToggle == true) {
            var normalMode = this.theme.getClassName("RATING_MODE_NORMAL_IMAGE");
            var averageMode = this.theme.getClassName("RATING_MODE_AVERAGE_IMAGE");

            // Remove hidden class
            if (this.common.checkStyleClasses(classNames, hiddenClass))
                this.common.stripStyleClass(this.modeToggleNode, hiddenClass);

            if (this.inAverageMode == true) {
                // Remove normal mode class
                if (this.common.checkStyleClasses(classNames, normalMode))
                    this.common.stripStyleClass(this.modeToggleNode, normalMode);

                // Add average mode class
                if (!this.common.checkStyleClasses(classNames, averageMode))
                    this.common.addStyleClass(this.modeToggleNode, averageMode);

                // Get image width
                imageWidth = parseInt(this.theme.getProperty("images", "RATING_MODE_AVG_WIDTH"));
            }
            else {
                // Remove average mode class
                if (this.common.checkStyleClasses(classNames, averageMode))
                    this.common.stripStyleClass(this.modeToggleNode, averageMode);

                // Add normal mode class
                if (!this.common.checkStyleClasses(classNames, normalMode))
                    this.common.addStyleClass(this.modeToggleNode, normalMode);

                // Get image width
                imageWidth = parseInt(this.theme.getProperty("images", "RATING_MODE_NORMAL_WIDTH"));
            }

        } else {
            // Add hidden class
            if (!this.common.checkStyleClasses(classNames, hiddenClass))
                this.common.addStyleClass(this.modeToggleNode, hiddenClass);
        }

        // Record image width if changing and flag that control container width must be recomputed.
        if (imageWidth != this.imageWidths["modeToggle"]) {
            this.imageWidths["modeToggle"] = imageWidth;
            changeControlWidth = true;
        }
    }
    if (this.includeModeToggle) {
        if (this.modeReadOnly)
            this.common.stripStyleClass(this.modeToggleNode, hoverClass);
        else
            this.common.addStyleClass(this.modeToggleNode, hoverClass);
    }

    // Spacer between grade controls and clear/modeToggle controls
    var classNames = this.spacerNode.className.split(" ");
    if ((this.imageWidths["clear"] > 0) || (this.imageWidths["modeToggle"] > 0)) {
        // Remove hidden class
        if (this.common.checkStyleClasses(classNames, hiddenClass))
            this.common.stripStyleClass(this.spacerNode, hiddenClass);

        // Record spacer width if changing and flag that control container width must be recomputed.
        if (this.imageWidths["spacer"] == 0) {
            this.imageWidths["spacer"] = parseInt(this.theme.getMessage("rating.spacerWidth"));
            changeControlWidth = true;
        }
    } else {
        // Add hidden class
        if (!this.common.checkStyleClasses(classNames, hiddenClass))
            this.common.addStyleClass(this.spacerNode, hiddenClass);

        // Record spacer width if changing and flag that control container width must be recomputed.
        if (this.imageWidths["spacer"] != 0) {
            this.imageWidths["spacer"] = 0;
            changeControlWidth = true;
        }
    }

    // Set width on control container, but only if it's changing.
    if (changeControlWidth == true) {
        var controlContainerWidth = 0;
        for (key in this.imageWidths)
            controlContainerWidth += this.imageWidths[key];
        this.controlContainer.style.width = controlContainerWidth + "px";
    }
    
    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
}; // _setProps

/**
 * Submit the selected grade.  This function is useful if autoSubmit is disabled
 * but it is still desired to asynchronously submit the grade.
 *
 * @param {String} execute The string containing a comma separated list 
 * of client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.rating.prototype.submit = function(execute) {
    // Publish an event for custom AJAX implementations to listen for.
    // Note that the current grade value is placed in a hidden input field which 
    // is automatically submitted, and so we don't need to explicitly include
    // the grade value in the props payload associated with beginTopic.
    //
    this.publish(webui.suntheme4_2.widget.rating.event.submit.beginTopic, [{
        id: this.id,
        execute: execute,
        endTopic: webui.suntheme4_2.widget.rating.event.submit.endTopic
    }]);

    return true;

}; // submit

/**
 * Callback function to the end of the submit request
 *
 * @param props Key-Value pairs of properties.
 * @config {String} id The HTML element Id.
 * @config {double} averageGrade The updated average grade
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.rating.prototype._submitCallback = function(props) {
    // Clear hidden field after each asynch submit, otherwise a subsequent
    // page submit would re-submit the same value again.
    this.hiddenFieldNode.value = null;
    return true;
}; // _submitCallback


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.resetButton");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.button");

/**
 * @name webui.suntheme4_2.widget.resetButton
 * @extends webui.suntheme4_2.widget.button
 * @class This class contains functions for the resetButton widget.
 * @constructor This function is used to construct a resetButton widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.resetButton", webui.suntheme4_2.widget.button, {
    // Set defaults.
    widgetName: "resetButton"  // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.resetButton.event =
        webui.suntheme4_2.widget.resetButton.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_resetButton_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_resetButton_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_resetButton_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_resetButton_event_state_end"
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.staticText");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.staticText
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the staticText widget.
 * @constructor This function is used to construct a staticText widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.staticText", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.escape = true;
    },
    widgetName: "staticText" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.staticText.event =
        webui.suntheme4_2.widget.staticText.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_staticText_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_staticText_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_staticText_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_staticText_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.staticText.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.escape) { props.escape = this.escape; }
    if (this.value) { props.value = this.value; }

    return props;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} escape HTML escape value (default).
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} style Specify style rules inline.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.staticText.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.staticText.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }
      
    // Set text value.
    if (props.value) {
        // NOTE: If you set this value manually, text must be HTML escaped.
        this.widget.addFragment(this.domNode, props.value, null, this.escape);
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.table2");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.table2
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the table2 widget.
 * @constructor This function is used to construct a table2 widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.table2", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    widgetName: "table2" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.table2.event =
        webui.suntheme4_2.widget.table2.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_table2_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_table2_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_table2_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_table2_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.table2.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.actions) { props.actions = this.actions; }
    if (this.align) { props.align = this.align; }
    if (this.bgColor) { props.bgColor = this.bgColor; }
    if (this.border) { props.border = this.border; }
    if (this.caption) { props.caption = this.caption; }
    if (this.cellpadding) { props.cellpadding = this.cellpadding; }
    if (this.cellspacing) { props.cellspacing = this.cellspacing; }
    if (this.filterText) { props.filterText = this.filterText; }
    if (this.frame) { props.frame = this.frame; }
    if (this.rowGroups) { props.rowGroups = this.rowGroups; }
    if (this.rules) { props.rules = this.rules; }
    if (this.summary) { props.summary = this.summary; }
    if (this.width) { props.width = this.width; }

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.actionsContainer.id = this.id + "_actionsContainer";
        this.actionsNode.id = this.id + "_actionsNode";
        this.controlsNode.id = this.id + "_controlsNode";
        this.filterPanelContainer.id = this.id + "_filterPanelContainer";
        this.preferencesPanelContainer.id = this.id + "_preferencesPanelContainer";
        this.sortPanelContainer.id = this.id + "_sortPanelContainer";
        this.rowGroupsContainer.id = this.id + "_rowGroupsContainer";
        this.captionContainer.id = this.id + "_captionContainer";
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {Object} actions 
 * @config {String} align Alignment of image input.
 * @config {String} bgColor
 * @config {String} border
 * @config {String} caption
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} frame 
 * @config {String} filterText 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {Array} rowGroups 
 * @config {String} rules 
 * @config {String} style Specify style rules inline.
 * @config {String} summary
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @config {String} width
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace actions -- do not extend.
    if (props.actions) {
        this.actions = null;
    }

    // Replace rows -- do not extend.
    if (props.rowGroups) {
        this.rowGroups = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.table2.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // To do: Add tabIndex to subwidgets, but not table, tr, or td tags.
    props.tabIndex = null;

    // Set properties.
    if (props.align) { this.domNode.align = props.align; }
    if (props.width) { this.domNode.style.width = props.width; }

    // Add caption.
    if (props.caption || props.filterText && this.caption) {       
        var filterText = null;
        if (props.filterText) {
            filterText = this.theme.getMessage("table.title.filterApplied", [
                props.filterText
            ]);
        }

        // To do: Create a new title message.
        
        this.widget.addFragment(this.captionContainer, (filterText) 
            ? props.caption + filterText : props.caption);
        this.common.setVisibleElement(this.captionContainer, true);
    }

    // Add actions.
    if (props.actions) {
        this.widget.addFragment(this.actionsNode, props.actions);
        this.common.setVisibleElement(this.actionsContainer, true);
    }

    // Add row groups.
    if (props.rowGroups) {
        // Remove child nodes.
        this.widget.removeChildNodes(this.rowGroupsContainer);
 
        // Add row group.
        for (var i = 0; i < props.rowGroups.length; i++) {
            // Set properties that must be applied to each HTML table element.
            props.rowGroups[i]._table = {
                bgColor: props.bgColor,
                border: props.border,
                cellpadding: props.cellpadding,
                cellspacing: props.cellspacing,
                frame: props.frame,
                summary: props.summary
            };
            this.widget.addFragment(this.rowGroupsContainer, props.rowGroups[i], "last");

            // To do: Fix me.
            // Actions my be rendered after column headers are positioned. When 
            // this happens, older offsetTop properties are no longer valid and
            // headers are off by one row, at least. We need a better way to 
            // solve this (e.g., set a fixed action bar height), but we'll call
            // resize() for now. Note that the addRows() function already calls
            // this, but with a much shorter timeout.
            var _id = props.rowGroups[i].id;
            setTimeout(function() {
                // New literals are created every time this function is called, 
                // and it's saved by closure magic.
                webui.suntheme4_2.dijit.byId(_id).resize();
            }, 2000);
        }
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.table2RowGroup");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.table2RowGroup
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the table2RowGroup widget.
 * @constructor This function is used to construct a table2RowGroup widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.table2RowGroup", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.currentRow = 0; // Current row in view.
        this.first = 0; // Index used to obtain rows.
    },
    widgetName: "table2RowGroup" // Required for theme properties.
});

/**
 * This function is used to set column headers and footers.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.addColumns = function() {
    // Clear column headers/footers.
    this.widget.removeChildNodes(this.thead);
    this.widget.removeChildNodes(this.tfoot);

    // Clone dojo attach points.
    var headerRowClone = this.colHeaderRow.cloneNode(false);
    var footerRowClone = this.colFooterRow.cloneNode(false);

    // Append row nodes.
    this.thead.appendChild(headerRowClone);
    this.tfoot.appendChild(footerRowClone);

    // Append cell nodes.
    for (var i = 0; i < this.columns.length; i++) {
        var col = this.columns[i];
        var headerCellClone = this.colHeaderCell.cloneNode(true);
        var footerCellClone = this.colFooterCell.cloneNode(true);

        // Set properties.
        headerCellClone.id = col.id + "_colHeader";
        footerCellClone.id = col.id + "_colFooter";
        if (col.width) {
            headerCellClone.style.width = col.width;
            footerCellClone.style.width = col.width;
        }

        // Add text.
        if (col.headerText) {
            // StaticText widget adds span to match styles.
            //
            // To do: Create utility to help create client-side widgets.
            this.widget.addFragment(headerCellClone,
                this.widget.getWidgetProps("staticText", {
                    id: headerCellClone.id + "Text",
                    value: col.headerText
                }));
            headerVisible = true;
        }
        if (col.footerText) {
            // StaticText widget adds span to match styles.
            //
            // To do: Create utility to help create client-side widgets.
            this.widget.addFragment(footerCellClone,
                this.widget.getWidgetProps("staticText", {
                    id: footerCellClone.id + "Text",
                    value: col.footerText
                }));
            footerVisible = true;
        }

        // Append nodes.
        headerRowClone.appendChild(headerCellClone);
        footerRowClone.appendChild(footerCellClone);

        // Set colspan.
        this.groupHeaderCell.colSpan = this.columns.length;
    }
    return true;
};
 
/**
 * This function is used to add rows using the gieven array. Each row contains
 * an array of columns which holds table data.
 *
 * @param {Array} rows An array of rows.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.addRows = function(rows) {
    if (rows == null) {
        return false;
    }

    // Get properties.
    var props = this.getProps();
    // Get className properties to alternate between rows.
    var classNames = (this.className) ? this.className.split(",") : null;          

    // For each row found, clone the tableDataRow attach point.
    for (var i = 0; i < rows.length; i++) {
        var cols = rows[i]; // Get columns.
        var rowId = this.id + ":" + (this.first + i); // Get row id.

        // Clone table data row without cells.
        var rowClone = this.tableDataRow.cloneNode(false);
        this.tbody.appendChild(rowClone);

        // Set properties.
        props.id = rowId;

        // Set className.
        if (classNames) {
            props.className = classNames[i % classNames.length];
        }

        this.setCommonProps(rowClone, props);
        this.setEventProps(rowClone, props);
        this.setCoreProps(rowClone, props);

        // For each column found, clone the tableDataCell attach point.
        for (var k = 0; k < cols.length; k++) {
            var col = this.columns[k]; // Get current column.
            var colId = col.id.replace(this.id, rowId); // Get col id.

            // Clone node.
            var cellClone = this.tableDataCell.cloneNode(true);
            rowClone.appendChild(cellClone);

            // Set properties.
            this.setColumnProps(cellClone, col);
            cellClone.id = colId; // Override id set by setCoreProps.

            // Add cell data.
            this.widget.addFragment(cellClone, cols[k], "last");
        }

        // Save row for destroy() function.
        if (this.first > 0) {
            this.rows[this.rows.length] = rows[i];
        }
    }

    // Set first row value.
    this.first += rows.length;

    // Adjust layout.
    var _id = this.id;
    setTimeout(function() {
        // New literals are created every time this function is called, and it's 
        // saved by closure magic.
        webui.suntheme4_2.dijit.byId(_id).resize();
    }, 10);
    return true;
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.table2RowGroup.event =
        webui.suntheme4_2.widget.table2RowGroup.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_table2RowGroup_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_table2RowGroup_event_refresh_end"
    },

    /**
     * This closure is used to process scroll events.
     * @ignore
     */
    scroll: {
        /** Scroll event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_table2RowGroup_event_scroll_begin",

        /** Scroll event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_table2RowGroup_event_scroll_end"
    },
    
    /**
     * This closure is used to process pagination button events.
     * @ignore
     */
    pagination: {
        /**
         * This closure is used to process next pagination button event.
         * @ignore
         */
        next: {
            /** Scroll event topic for custom AJAX implementations to listen for. */
            beginTopic: "webui_suntheme4_2_widget_table2RowGroup_event_pagination_next_begin",

            /** Scroll event topic for custom AJAX implementations to listen for. */
            endTopic: "webui_suntheme4_2_widget_table2RowGroup_event_pagination_next_end"
        },

        /**
         * This closure is used to process previous pagination button event.
         * @ignore
         */
        previous: {
            /** Scroll event topic for custom AJAX implementations to listen for. */
            beginTopic: "webui_suntheme4_2_widget_table2RowGroup_event_pagination_previous_begin",

            /** Scroll event topic for custom AJAX implementations to listen for. */
            endTopic: "webui_suntheme4_2_widget_table2RowGroup_event_pagination_previous_end"
        }
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_table2RowGroup_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_table2RowGroup_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.align) { props.align = this.align; }
    if (this.bgColor) { props.bgColor = this.bgColor; }
//    if (this.char) { props.char = this.char; } // To do: Rename -- keyword is reserved.
    if (this.charOff) { props.charOff = this.charOff; }
    if (this.columns) { props.columns = this.columns; }
    if (this.first) { props.first = this.first; }
    if (this.headerText) { props.headerText = this.headerText; }
    if (this.height) { props.height = this.height; }
    if (this.maxRows) { props.maxRows = this.maxRows; }
    if (this.rows) { props.rows = this.rows; }
    if (this.totalRows) { props.totalRows = this.totalRows; }
    if (this.valign) { props.valign = this.valign; }
    if (this.paginationControls != null) {props.paginationControls = this.paginationControls;}
    if (this.paginationNextButton) {props.paginationNextButton = this.paginationNextButton;}
    if (this.paginationPrevButton) {props.paginationPrevButton = this.paginationPrevButton;} 

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.postCreate = function () {
    // Set ids.    
    if (this.id) {
        this.colFooterRow.id = this.id + "_colFooterRow";
        this.colFooterCell.id = this.id + "_colFooterCell";
        this.colHeaderRow.id = this.id + "_colHeaderRow";
        this.colHeaderCell.id = this.id + "_colHeaderCell";
        this.groupHeaderControls.id = this.id + "_groupHeaderControls";
        this.groupHeaderText.id = this.id + "_groupHeaderText";        
        this.rowsText.id = this.id + "_rowsText";
        this.table.id = this.id + "_table";
        this.tableContainer.id = this.id + "_tableContainer";
        this.tableDataRow.id = this.id + "_tableDataRow";
        this.tableDataCell.id = this.id + "_tableDataCell";
        this.tbody.id = this.id + "_tbody";
        this.tfoot.id = this.id + "_tfoot";
        this.thead.id = this.id + "_thead";        
        this.paginationButtonsNode.id = this.id + "_paginationButtonsNode";           
    }

    // Set events.
    this.dojo.connect(this.tableContainer, "onscroll", this, "scroll");

    // Set pagination controls.
    if (this.paginationPrevButton == null) {
        this.paginationPrevButton = this.widget.getWidgetProps("imageHyperlink", {
                id: this.id + "_paginationPrevButton",
                enabledImage: this.widget.getWidgetProps("image", {
                    icon: "TABLE2_PAGINATION_PREV",
                    id: this.id + "_paginationPrevButtonImg"
                }),
                disabledImage: this.widget.getWidgetProps("image", {
                    icon: "TABLE2_PAGINATION_PREV_DISABLED",
                    id: this.id + "_paginationPrevButtonImgDis"
                }),
                title: this.theme.getMessage("table2.pagination.previous")      
            });
    }
      
    if (this.paginationNextButton == null) {
        this.paginationNextButton = this.widget.getWidgetProps("imageHyperlink", {
                id: this.id + "_paginationNextButton",
                enabledImage: this.widget.getWidgetProps("image", {
                    icon: "TABLE2_PAGINATION_NEXT",
                    id: this.id + "_paginationNextButtonImg"
                }),
                disabledImage: this.widget.getWidgetProps("image", {
                    icon: "TABLE2_PAGINATION_NEXT_DISABLED",
                    id: this.id + "_paginationNextButtonImgDis"
                }),
                title: this.theme.getMessage("table2.pagination.next")          
            });
    }
    
    // Resize hack for Moz/Firefox.
    if (webui.suntheme4_2.browser.isNav()) {
        this.dojo.connect(window, "onresize", this, "resize");
    }        
    return this.inherited("postCreate", arguments);
};

/**
 * Process resize event.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.resize = function() {
    // Update rows text.
    this.updateRowsText();

    // Get height offset of each row.
    var offset = 0;
    for (var i = this.currentRow; i < this.currentRow + this.maxRows; i++) {
        var tableDataRow = document.getElementById(this.id + ":" + i);
        if (tableDataRow != null) {
            offset += tableDataRow.offsetHeight;
        } else {
            break;
        }
    }

    // Set the scrollable height.
    if (offset > 0) {
        this.tableContainer.style.height = offset + "px";
    }

    // Set width of each column header & footer.
    var rowId = this.id + ":0"; // ID of first row.
    for (var i = 0; i < this.columns.length; i++) {
        var col = this.columns[i]; // Get default column props.
        var colId = col.id.replace(this.id, rowId);

        // Get row node.
        var tableDataCell = document.getElementById(colId);
        if (tableDataCell == null) {
            continue;
        }

        // Get nodes.
        var colHeaderCell = document.getElementById(col.id + "_colHeader");
        var colFooterCell = document.getElementById(col.id + "_colFooter");

        // Set width.
        if (colHeaderCell) {
            // Column width plus offset for border.
            colHeaderCell.style.width = (tableDataCell.offsetWidth - 1) + "px";
        }
        if (colFooterCell) {
            // Column width plus offset for border.
            colFooterCell.style.width = (tableDataCell.offsetWidth - 1) + "px";
        }
    }

    // Update header & footer position.
    var colHeaderRow = document.getElementById(this.id + "_colHeaderRow");
    var colFooterRow = document.getElementById(this.id + "_colFooterRow");

    var headerHeight = (colHeaderRow) ? colHeaderRow.offsetHeight : 0;
    var footerHeight = (colFooterRow) ? colFooterRow.offsetHeight : 0;

    this.tableContainer.style.marginTop = (headerHeight - 1) + 'px';
    this.tableContainer.style.marginBottom = footerHeight + 'px';

    // Column header height plus offset for border.
    if (colHeaderRow) {
        colHeaderRow.style.top = (this.tableContainer.offsetTop - 
            headerHeight + 1) + 'px';
    }

    // Column footer height plus offset for border.
    if (colFooterRow) {
        colFooterRow.style.top = (this.tableContainer.offsetTop + 
            this.tableContainer.offsetHeight - 1) + 'px';
    }
    return true;
};

/**
 * This function is used to set column properties with Object literals.
 *
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} abbr
 * @config {String} axis
 * @config {String} bgColor
 * @config {String} char
 * @config {String} charOff
 * @config {String} className CSS selector.
 * @config {int} colspan
 * @config {String} dir Specifies the directionality of text.
 * @config {String} height
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {boolean} noWrap 
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {int} rowSpan 
 * @config {String} scope 
 * @config {String} style Specify style rules inline.
 * @config {String} title Provides a title for element.
 * @config {String} valign 
 * @config {boolean} visible Hide or show element.
 * @config {String} width
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.setColumnProps = function(domNode, props) {
    // Set properties.
    if (this.abbr) { domNode.abbr = this.abbr; }
    if (this.axis) { domNode.axis = this.axis; }
    if (this.bgColor) { domNode.bgColor = this.bgColor; }
//    if (this.char) { domNode.char = this.char; } // To do: Rename -- keyword is reserved.
    if (this.charOff) { domNode.charoff = this.charOff; }
    if (this.thisspan) { domNode.colspan = this.colspan; }
    if (this.headers) { domNode.headers = this.headers; }
    if (this.height) { domNode.height = this.height; }
    if (this.noWrap) { domNode.nowrap = "nowrap"; }
    if (this.rowSpan) { domNode.rowspan = this.rowSpan; }
    if (this.scope) { domNode.scope = this.scope; }
    if (this.valign) { domNode.valign = this.valign; }
    if (this.width) { domNode.width = this.width; }

    // Set more properties.
    this.setCommonProps(domNode, props);
    this.setEventProps(domNode, props);
    this.setCoreProps(domNode, props);

    return true;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} align Alignment of image input.
 * @config {String} bgColor
 * @config {String} char
 * @config {String} charOff
 * @config {String} className CSS selector.
 * @config {Array} columns
 * @config {String} dir Specifies the directionality of text.
 * @config {int} first
 * @config {String} headerText
 * @config {int} height
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} maxRows 
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {Array} rows 
 * @config {String} style Specify style rules inline.
 * @config {String} title Provides a title for element.
 * @config {int} totalRows 
 * @config {String} valign 
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace columns -- do not extend.
    if (props.columns) {
        this.columns = null;
    }

    // Replace rows -- do not extend.
    if (props.rows) {
        this.rows = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.table2RowGroup.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.id) { this.domNode.id = props.id; }

    // Set private properties for table widget.
    if (props._table) {
        if (props._table.bgColor) { this.table.bgColor = props._table.bgColor; }
        if (props._table.border) { this.table.border = props._table.border; }
        if (props._table.cellpadding) { this.table.cellpadding = props._table.cellpadding; }
        if (props._table.cellspacing) { this.table.cellspacing = props._table.cellspacing; }
        if (props._table.frame) { this.table.frame = props._table.frame; }
        if (props._table.rules) { this.table.rules = props._table.rules; }
        if (props._table.summary) { this.table.summary = props._table.summary; }
    }

    // Add header.
    if (props.headerText) {
        this.widget.addFragment(this.groupHeaderText, props.headerText);
        this.common.setVisibleElement(this.groupHeaderContainer, true);
    }
    // Add paginationControl.    
    if (props.paginationPrevButton) {
        // set onclick for previous button.
        props.paginationPrevButton.onClick = 
            "webui.suntheme4_2.dijit.byId('" + this.id + "').paginationPrevious();return false;";        
        this.widget.addFragment(this.paginationButtonsNode, props.paginationPrevButton,"last");
    }
    if (props.paginationNextButton) {
        // set onclick for next button.
        props.paginationNextButton.onClick = 
            "webui.suntheme4_2.dijit.byId('" + this.id + "').paginationNext();return false;";
        this.widget.addFragment(this.paginationButtonsNode, props.paginationNextButton,"last");
    }
    if (props.paginationControls != null) {        
        this.paginationControls = props.paginationControls;
    }
    //set enabled/disabled state for pagination controls
    this.updatePaginationControls(); 
        
    // Set columns.
    if (props.columns && this.refreshCols != false) {
        this.addColumns();
        // To Do: Cannot refresh column headers/footers due to poor CSS styles.
        this.refreshCols = false;
    }

    // Add rows.
    if (props.rows) {
        this.first = 0; // Reset index used to obtain rows.
        this.currentRow = 0; // Reset current row in view.

        // Clear rows.
        this.widget.removeChildNodes(this.tbody);
        this.addRows(props.rows);
    }
   
    // Cannot call "superclass" here because properties are set on each row.
    return true;
};

/**
 * Process scroll event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.scroll = function(event) {
    // Publish event to retrieve data.    
    if (this.first < this.totalRows
            && this.currentRow % this.maxRows == 0) {
        // Publish an event for custom AJAX implementations to listen for.
        this.publish(webui.suntheme4_2.widget.table2RowGroup.event.scroll.beginTopic, [{
            id: this.id,
            first: this.first
        }]);
    }    
    var scrollTop = this.tableContainer.scrollTop;
    var rowHeight =  document.getElementById(this.id + ":" + (this.currentRow + 1)).offsetTop -
                    document.getElementById(this.id + ":" + this.currentRow).offsetTop;
    var moveScroll = scrollTop % rowHeight;
    this.currentRow = Math.floor((this.tableContainer.scrollTop) / rowHeight);    
    
    if (moveScroll > (rowHeight / 2)) {
        this.currentRow = this.currentRow + 1;   
    }   
    // Set rows text.    
    return this.updateRowsText();
};

/**
 * Updates pagination control buttons enabled/disabled state.
 * 
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.updatePaginationControls = function() {
    if (this.paginationPrevButton && this.paginationNextButton) {
        var domNodePrev = webui.suntheme4_2.dijit.byId(this.paginationPrevButton.id);
        var domNodeNext = webui.suntheme4_2.dijit.byId(this.paginationNextButton.id);

        if (domNodePrev != null && domNodeNext != null) {
            if (this.currentRow / this.maxRows == 0) {
                domNodePrev.setProps({disabled:true});              
            } else {
                domNodePrev.setProps({disabled:false});  
            } 
            if ((this.currentRow / this.maxRows) == (this.totalRows / this.maxRows) - 1) {
                domNodeNext.setProps({disabled:true});  
            } else {
                domNodeNext.setProps({disabled:false});  
            }
            domNodePrev.setProps({visible:this.paginationControls});
            domNodeNext.setProps({visible:this.paginationControls});  
        }           
    }    
};

/**
 * Process next control button.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.paginationNext = function(event) {
    // Publish event to retrieve data.
    var currentPage = Math.floor(this.currentRow / this.maxRows) + 1;
    var totalPage = Math.floor(this.totalRows / this.maxRows);
    if (this.first < this.totalRows
            && currentPage < totalPage) {
        // Publish an event for custom AJAX implementations to listen for.
        this.publish(webui.suntheme4_2.widget.table2RowGroup.event.pagination.next.beginTopic, [{
            id: this.id,
            first: this.first 
        }]);
    }     
    if (currentPage < totalPage) {          
        // Calculate current row.          
        this.currentRow = currentPage * this.maxRows;        
        // set scroll position to make the current row completely visible
        this.tableContainer.scrollTop =  
            document.getElementById(this.id + ":" + this.currentRow).offsetTop;
    }       
    return this.updateRowsText();
};

/**
 * Process previous control button.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.paginationPrevious = function(event) {
   
    var currentPage = Math.ceil(this.currentRow / this.maxRows) + 1;
    var totalPage = Math.floor(this.totalRows / this.maxRows);
    if (currentPage > 1) {                 
        this.currentRow = (currentPage - 2) * this.maxRows;
        // set scroll position to make the current row completely visible
        this.tableContainer.scrollTop = 
            document.getElementById(this.id + ":" + this.currentRow).offsetTop;        
    }    
    return this.updateRowsText();
};

/**
 * This function is used to set rows text (e.g., "1 - 5 of 20").
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.table2RowGroup.prototype.updateRowsText = function() {
    // Add title augment.
    var firstRow = this.currentRow + 1;
    var lastRow = Math.min(this.totalRows, this.currentRow + this.maxRows);

    // To do: Need to create a new rows message.

    // NOTE: If you set this value manually, text must be HTML escaped.
    var msg = this.theme.getMessage("table.title.paginated", [
        "", 
        firstRow, 
        lastRow, 
        this.totalRows, 
        ""
    ]);

    // "Items: " + firstRow + " - " + lastRow + " of " + this.totalRows);
    if (msg) {
        this.widget.addFragment(this.rowsText, msg);
    }
     //set disabled/enabled state
    this.updatePaginationControls(); 
    return true;
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.textArea");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.textField");

/**
 * @name webui.suntheme4_2.widget.textArea
 * @extends webui.suntheme4_2.widget.textField
 * @class This class contains functions for the textArea widget.
 * @constructor This function is used to construct a textArea widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.textArea", webui.suntheme4_2.widget.textField, {
    // Set defaults.
    constructor: function() {
        this.autoSave = 0;
        this.cols = 20;
        this.rows = 3;
    },
    widgetName: "textArea" // Required for theme properties.
});

/**
 * Helper function to create callback for timer event.
 *
 * @return {Function} The callback function.
 */
webui.suntheme4_2.widget.textArea.prototype.createSubmitCallback = function() {
    var _id = this.id;

    // New literals are created every time this function
    // is called, and it's saved by closure magic.
    return function(event) { 
        var widget = webui.suntheme4_2.dijit.byId(_id);
        if (widget == null) {
            return false;
        }
        //Create a submit request only if field has been modified
        if (widget.lastSaved != widget.fieldNode.value) {
            widget.lastSaved = widget.fieldNode.value;
            widget.submit();
        }
        return true;
    };
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.textArea.event =
        webui.suntheme4_2.widget.textArea.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textArea_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textArea_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textArea_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textArea_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textArea_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textArea_event_submit_end"
    }
};

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.textArea.prototype.getInputClassName = function() {
    // Set readOnly style
    if (this.fieldNode.readOnly) {
        return this.widget.getClassName("TEXT_AREA_READONLY", "");
    }

    // Apply invalid style.
    var validStyle =  (this.valid == false) 
        ? " " + this.widget.getClassName("TEXT_AREA_INVALID", "")
        : " " + this.widget.getClassName("TEXT_AREA_VALID", "");

    // Set default style.    
    return (this.disabled == true)
        ? this.widget.getClassName("TEXT_AREA_DISABLED", "") 
        : this.widget.getClassName("TEXT_AREA", "") + validStyle;    
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.textArea.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);
    
    // Set properties.
    if (this.cols > 0 ) { props.cols = this.cols; }
    if (this.rows > 0) { props.rows = this.rows; }
    if (this.autoSave > 0 ) { props.autoSave = this.autoSave; }
   
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textArea.prototype.postCreate = function () {
    // Set events.                
    if (this.autoSave > 0) {
        this.autoSaveTimerId = setInterval(this.createSubmitCallback(), 
            this.autoSave);  
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {boolean} autoSave 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {int} rows 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textArea.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.textArea.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.   
    if (props.cols > 0) { this.fieldNode.cols = props.cols; }
    if (props.rows > 0) { this.fieldNode.rows = props.rows; }
    
    // Cancel autosave if it has been changed to <=0
    if (props.autoSave <= 0 && this.autoSaveTimerId && this.autoSaveTimerId != null ) {
        clearTimeout(this.autoSaveTimerId);
        this.autoSaveTimerId = null;
    }

    // Set label className -- must be set before calling superclass.
    if (props.label) {
        props.label.className = (props.label.className)
            ? this.widget.getClassName("TEXT_AREA_TOPLABELALIGN", "")  + " " + props.label.className
            : this.widget.getClassName("TEXT_AREA_TOPLABELALIGN", "") ;
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.wizard");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");

/** 
 * @class This class contains functions for wizard components.
 * <p>
 * The wizard JavaScript object is accessed using the getElementById()
 * function. Methods defined on that javascript object instance maybe
 * called using that identifier. For example, the following javascript
 * could be used to close and forward to a page when the wizard closes.
 * </p><p><code>
 *   <ui:wizard ...other attributes... 
 *	onPopupDismiss="document.getElementById('form1:wizard1').closeAndForward('launchform', '/faces/wizardData.jsp', true);" >
 *
 *	...wizard step tags...
 *
 *   </ui:wizard>
 * </code></p>
 * @static
 */
webui.suntheme4_2.wizard = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize wizard.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        webui.suntheme4_2.prototypejs.extend(domNode, props, false);

        // Set functions.
        domNode.nextClicked = webui.suntheme4_2.wizard.nextClicked;
        domNode.previousClicked = webui.suntheme4_2.wizard.previousClicked;
        domNode.cancelClicked = webui.suntheme4_2.wizard.cancelClicked;
        domNode.finishClicked = webui.suntheme4_2.wizard.finishClicked;
        domNode.closeClicked = webui.suntheme4_2.wizard.closeClicked;
        domNode.gotoStepClicked = webui.suntheme4_2.wizard.gotoStepClicked;
        domNode.closePopup = webui.suntheme4_2.wizard.closePopup;
        domNode.closeAndForward = webui.suntheme4_2.wizard.closeAndForward;
        domNode.wizOnLoad = webui.suntheme4_2.wizard.wizOnLoad;
        domNode.resize_hack = webui.suntheme4_2.wizard.resize_hack;

        return true;
    },

    /**
     * @ignore
     */
    nextClicked: function() {
        return true;
    },

    /**
     * @ignore
     */
    previousClicked: function() {
        return true;
    },

    /**
     * @ignore
     */
    cancelClicked: function() {
        return true;
    },

    /**
     * @ignore
     */
    closeClicked: function() {
        return true;
    },

    /**
     * @ignore
     */
    finishClicked: function() {
        return true;
    },

    /**
     * @ignore
     */
    gotoStepClicked: function() {
        return true;
    },

    /**
     * Close popup.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    closePopup: function() {
        window.close();
        return true;
    },

    /**
     * Close the wizard popup and forward to "submitPage" by submitting
     * "submitForm".
     * <p>
     * When the wizard closes it is often necessary to send
     * a request to a page that will make use of the data collected
     * during a wizard session. This method does this by obtaining the
     * form element "submitForm" from the page that launched the
     * the wizard. This means that the page that launched the wizard
     * popup must still be visible in a browser window. If that form
     * is found, the "action" property is set to "submitPage" and the
     * "submit" method of that "submitForm" is executed.
     * The popup window is then closed. 
     * </p><p>
     * However due to JSF's client side state saving mode an extra step
     * must be taken. If the application is operating with client side
     * state saving, JSF will ignore the "submitPage" value of the
     * submitted form's "action" property and will send the request to the
     * view defined in the saved state, saved in an element in "submitForm".
     * </p><p>
     * If the application is configured for client side state saving and
     * the "submitPage" is different from the page that lauched the wizard,
     * set "cleartState" to true. This method will clear the saved state 
     * before submitting the form. The "clearState" default value is false
     * and the saved state will not be cleared.
     * </p><p>
     * The "clearState" functionality only works with Sun's RI.
     * </p>
     *
     * @param {boolean} submitForm
     * @param {boolean} submitPage
     * @param {boolean} clearState
     * @return {boolean} true if successful; otherwise, false.
     */
    closeAndForward: function(submitForm, submitPage, clearState) {
        var f = window.opener.document.getElementById(submitForm);
        if (f == null) {
            console.debug("Can't find form " + submitForm); // See Firebug console.
            window.close();
        }

        if (clearState != null && clearState == true) {
            var elements = f.elements;
            var clientstate = null;
            for (var i = 0; i < elements.length; ++i) {
                // This only works for the Sun RI and is
                // dependent on the RIConstants.FACES_VIEW value
                // of "com.sun.faces.VIEW"
                //
                if (elements[i].name == this.facesViewState) {
                    clientstate = elements[i];
                    break;
                }
            }
            if (clientstate != null) {
                f.removeChild(clientstate);
            }
        }

        f.action = submitPage;
        f.submit();
        window.close();
        return true;
    }, 

    /**
     * This method must be assigned to the onload handler of the onLoad
     * attribute of the ui:body tag if the wizard is to operate properly on IE.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    wizOnLoad: function() {
        var stepsid = this.id + "_stepspane";
        var helpid = this.id + "_helppane";
        var wizbdyid = this.id + "_WizBdy";
        return this.resize_hack(helpid, stepsid, wizbdyid);
    },

    /**
     * used only for popup window and IE, and called by wizOnLoad.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    resize_hack: function(helpid, stepsid, wizbdyid) {
        if (webui.suntheme4_2.browser.isIe5up()) {
            var bdy = document.getElementById(wizbdyid);

            if (bdy != null) {
                bdy.style.height = document.body.clientHeight - 145;

                if (helpid != null && helpid != '') {
                    var help = document.getElementById(helpid);
                    if (help != null) {
                        help.style.height = document.body.clientHeight - 90;
                    }
                }
                if (stepsid != null && stepsid != '') {
                    var steps = document.getElementById(stepsid);
                    if (steps != null) {
                        steps.style.height = document.body.clientHeight - 90;
                    }
                }
            }
        }
        return true;
    }
};
