/***************************************************************************
                             ModelCanvas.cpp
                             -------------------
    begin                : Sat April 5, 2003
    copyright            : (C) 2003 by James Wells
    email                : james@wells.net

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 ***************************************************************************/
#include <wx/wxprec.h>
#ifdef __BORLANDC__
#pragma hdrstop
#endif
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif
#include "ModelCanvas.h"
#include "KlassModelerView.h"
#include "KlassModelerDoc.h"
#include "KlassModelerApp.h"
#include "MainFrame.h"
#include "ModelPrinter.h"
#include "CodeObjectsDlg.h"
#include "Log.h"

extern CLog MasterLog;

#define ZOOM_STEP 1.5

enum {
	MENU_NEW_CLASS = 100,
	MENU_NEW_CLASS_FROM_HEADER,
	MENU_NEW_CLASS_FROM_CLIPBOARD,
	MENU_ZOOM_IN,
	MENU_ZOOM_OUT,
	MENU_DELETE_CLASS,
	MENU_INHERIT_FROM,
	MENU_GENERATE_CODE,
	MENU_HAS_A,
	MENU_USES_A
};
	
BEGIN_EVENT_TABLE( CModelCanvas, wxScrolledWindow )
#ifdef __MSXWT__
	EVT_CONTEXT_MENU( CModelCanvas::OnContextMenu )
#else
	EVT_RIGHT_UP( CModelCanvas::OnRightButtonUp )
#endif
	EVT_LEFT_DOWN( CModelCanvas::OnLeftButtonDown )
	EVT_LEFT_UP( CModelCanvas::OnLeftButtonUp )
	EVT_MOTION( CModelCanvas::OnMouseMove )
	EVT_LEFT_DCLICK( CModelCanvas::OnLeftDoubleClick )
	EVT_MENU( MENU_NEW_CLASS, CModelCanvas::OnNewClass )
	EVT_MENU( MENU_NEW_CLASS_FROM_HEADER, CModelCanvas::OnNewClassFromHeader )
	EVT_MENU( MENU_NEW_CLASS_FROM_CLIPBOARD, CModelCanvas::OnNewClassFromClipboard )
	EVT_MENU( MENU_ZOOM_IN, CModelCanvas::OnZoomIn )
	EVT_MENU( MENU_ZOOM_OUT, CModelCanvas::OnZoomOut )
	EVT_MENU( MENU_DELETE_CLASS, CModelCanvas::OnDeleteClass )
	EVT_MENU( MENU_INHERIT_FROM, CModelCanvas::OnInheritFrom )
	EVT_MENU( MENU_GENERATE_CODE, CModelCanvas::OnGenerateCode )
	EVT_MENU( MENU_HAS_A, CModelCanvas::OnHasA )
	EVT_MENU( MENU_USES_A, CModelCanvas::OnUsesA )
END_EVENT_TABLE()

CModelCanvas::CModelCanvas( CKlassModelerView *pView, wxFrame *pParent )
	: wxScrolledWindow( pParent, -1, wxDefaultPosition, wxDefaultSize, wxHSCROLL|wxVSCROLL|wxSUNKEN_BORDER )
#ifdef __WXMSW__
	, m_ViewFont( *wxNORMAL_FONT )
#else
	, m_ViewFont( 8, wxDEFAULT, wxNORMAL, wxNORMAL )
#endif
{
	m_pView = pView;
	m_Zoom = 1;
	m_LeftDrag = false;
	m_pDragClass = NULL;
	m_pDragInheritence = NULL;
	m_pInheritenceChild = NULL;
	m_pHasAParent = NULL;
	m_pUsesAParent = NULL;
	m_bUseGrid = false;
	m_PrinterPageWidth = 800;
	m_PrinterPageHeight = 1300;
}

CModelCanvas::~CModelCanvas()
{
	m_pView = NULL;
}

void CModelCanvas::ShowContextMenu( const wxPoint &Pos )
{
	wxMenu	Menu;
	int		X, Y;

	CalcUnscrolledPosition( Pos.x, Pos.y, &X, &Y );
	X /= m_Zoom;
	Y /= m_Zoom;
	m_ContentsClickPosition.x = X;
	m_ContentsClickPosition.y = Y;
	if( GetClassAt( m_ContentsClickPosition ) ){
		Menu.Append( MENU_DELETE_CLASS, _T("Delete Class") );
		Menu.Append( MENU_GENERATE_CODE, _T("Generate Code") );
		Menu.AppendSeparator();
		Menu.Append( MENU_INHERIT_FROM, _T("Inherit From") );
		Menu.Append( MENU_HAS_A, _T("Has-a") );
		Menu.Append( MENU_USES_A, _T("Uses-a") );
	} else {
		Menu.Append( MENU_NEW_CLASS, _T("New Class") );
		Menu.Append( MENU_NEW_CLASS_FROM_HEADER, _T("New Class From Header") );
		Menu.Append( MENU_NEW_CLASS_FROM_CLIPBOARD, _T("New Class From Clipboard") );
		Menu.AppendSeparator();
		Menu.Append( MENU_ZOOM_IN, _T("Zoom In") );
		Menu.Append( MENU_ZOOM_OUT, _T("Zoom Out") );
	}	
	PopupMenu( &Menu, Pos.x, Pos.y );
}

void CModelCanvas::OnLeftButtonDown( wxMouseEvent &Event )
{
	// don't turn on a move if we're just creating an inheritence link
	if( m_pInheritenceChild || m_pHasAParent || m_pUsesAParent )
		return;
	wxPoint		Position;

	m_LeftDrag = true;
	m_MouseMovePosition = Event.GetPosition();
	CalcUnscrolledPosition( m_MouseMovePosition.x, m_MouseMovePosition.y, &Position.x, &Position.y );
	Position.x /= m_Zoom;
	Position.y /= m_Zoom;
	m_pDragClass = GetClassAt( Position );
	m_pDragInheritence = GetInheritenceAt( Position );
}

void CModelCanvas::OnLeftButtonUp( wxMouseEvent &Event )
{
	if( m_pInheritenceChild ){
		wxPoint	Pos;
		CalcUnscrolledPosition( Event.GetPosition().x, Event.GetPosition().y, &Pos.x, &Pos.y );
		Pos.x /= m_Zoom;
		Pos.y /= m_Zoom;
		Inherit( Pos );
	} else if( m_pHasAParent ){
		wxPoint	Pos;
		CalcUnscrolledPosition( Event.GetPosition().x, Event.GetPosition().y, &Pos.x, &Pos.y );
		Pos.x /= m_Zoom;
		Pos.y /= m_Zoom;
		HasA( Pos );
	} else if( m_pUsesAParent ){
		wxPoint	Pos;
		CalcUnscrolledPosition( Event.GetPosition().x, Event.GetPosition().y, &Pos.x, &Pos.y );
		Pos.x /= m_Zoom;
		Pos.y /= m_Zoom;
		UsesA( Pos );
	} else {
		if( m_LeftDrag )
			m_pView->GetDoc()->UpdateAllViews( NULL );	// include self in update
		m_LeftDrag = false;
		m_pDragClass = NULL;
		m_pDragInheritence = NULL;
	}

	m_pView->GetDoc()->Modify( true );
}

void CModelCanvas::OnLeftDoubleClick( wxMouseEvent &Event )
{
	wxPoint		Pos;
	CClass		*pClass;
	wxClientDC	DC( this );

	CalcUnscrolledPosition( Event.GetPosition().x, Event.GetPosition().y, &Pos.x, &Pos.y );
	Pos.x /= m_Zoom;
	Pos.y /= m_Zoom;
	pClass = GetClassAt( Pos );
	if( pClass != NULL ){
		CClassDlg					*pClassDlg;
		InheritenceVector::iterator	itVect;
		InheritenceVector			*pVect;
		CClass						*pParent;
		CClass						TheCopy( *pClass );

		pClassDlg = new CClassDlg( this, m_pView->GetDoc(), &TheCopy );
		pClassDlg->Setup();
		if( pClassDlg->ShowModal() == wxID_OK ){
			TheCopy.CopyTo( *pClass );
			pVect = pClass->GetInheritence();
			for( itVect=pVect->begin(); itVect!=pVect->end(); ++itVect ){
				pParent = m_pView->GetDoc()->FindClassNamed( (*itVect)->GetParentName() );
				if( pParent != NULL ){
					(*itVect)->SetParent( pParent );
					(*itVect)->SetVisible( true );
				}
			}
			m_pView->GetDoc()->Modify( true );
			DC.SetFont( m_ViewFont );
			pClass->ComputeRect( &DC );
			wxGetApp().GetMainFrame()->DrawTree( m_pView->GetDoc(), true );
			Refresh();
		}
		pClassDlg->Destroy();
	}
}

void CModelCanvas::OnMouseMove( wxMouseEvent &Event )
{
	if( !m_LeftDrag )
		return;
	if( m_pDragClass ){
		float		dX = (Event.GetPosition().x-m_MouseMovePosition.x)/m_Zoom;
		float		dY = (Event.GetPosition().y-m_MouseMovePosition.y)/m_Zoom;
		m_pDragClass->Translate( dX, dY );
		m_MouseMovePosition = Event.GetPosition();
		Refresh();
	} else if( m_pDragInheritence ){
		m_pDragInheritence->SetHorizontal( m_pDragInheritence->GetHorizontal() - (m_MouseMovePosition.y-Event.GetPosition().y)/m_Zoom );
		m_MouseMovePosition = Event.GetPosition();
		Refresh();
	}
}

void CModelCanvas::Inherit( const wxPoint &Position )
{
	CClass	*pParent;
	
	pParent = GetClassAt( Position );
	if( pParent ){
		CInheritence	*pInheritence = new CInheritence;
		wxRect			ParentRect;

		GetClassRect( NULL, pParent, ParentRect );
        pInheritence->SetParent( pParent );
        pInheritence->SetHorizontal( m_pInheritenceChild->GetPosition().y - (m_pInheritenceChild->GetPosition().y -
									 (ParentRect.y+ParentRect.height))/2 );
		pInheritence->SetVisible( true );
		m_pInheritenceChild->AddInheritence( pInheritence );
		m_pInheritenceChild = NULL;
		m_pView->GetDoc()->UpdateAllViews( NULL );	// include self in update
	}
	m_pInheritenceChild = NULL;
}

void CModelCanvas::HasA( const wxPoint &Position )
{
	CClass	*pChild;
	
	pChild = GetClassAt( Position );
	if( pChild ){
		CRelation	*pRelation = new CRelation;

		pRelation->SetChild( pChild );
		pRelation->SetHasA( true );
		m_pHasAParent->AddRelation( pRelation );
		m_pHasAParent = NULL;
		m_pView->GetDoc()->UpdateAllViews( NULL );	// include self in update
	}
	m_pHasAParent = NULL;
}

void CModelCanvas::UsesA( const wxPoint &Position )
{
	CClass	*pChild;
	
	pChild = GetClassAt( Position );
	if( pChild ){
		CRelation	*pRelation = new CRelation;

		pRelation->SetChild( pChild );
		pRelation->SetHasA( false );
		m_pUsesAParent->AddRelation( pRelation );
		m_pUsesAParent = NULL;
		m_pView->GetDoc()->UpdateAllViews( NULL );	// include self in update
	}
	m_pUsesAParent = NULL;
}

void CModelCanvas::ComputeSizes()
{
	ClassVector					*pVect;
	ClassVector::iterator		itVect;
	wxClientDC					DC( this );

	DC.SetFont( m_ViewFont );
	DC.SetPen( *wxBLACK_PEN );
	pVect = m_pView->GetDoc()->GetClassVector();
	for( itVect=pVect->begin(); itVect!=pVect->end(); ++itVect )
		(*itVect)->ComputeRect( &DC );
}

void CModelCanvas::OnDraw( wxDC &DC )
{
	wxRect	BoundingRect;

	DC.SetUserScale( m_Zoom, m_Zoom );
	if( m_bUseGrid ){
		int 			ContentsWidth, ContentsHeight;
		int				i;
		wxPen			HoldPen = DC.GetPen();
		wxPen			CurrentPen = DC.GetPen();

		CurrentPen.SetColour( wxColour(192,192,192) );
		DC.SetPen( CurrentPen );
		GetVirtualSize( &ContentsWidth, &ContentsHeight );
		i = m_PrinterPageWidth;
		while( i<ContentsWidth ){
			DC.DrawLine( i, 0, i, ContentsWidth );
			i += m_PrinterPageWidth;
  		}
		i = m_PrinterPageHeight;
		while( i<ContentsHeight ){
			DC.DrawLine( 0, i, ContentsHeight, i );
			i += m_PrinterPageHeight;
  		}
		CurrentPen.SetColour( *wxBLACK );
		DC.SetPen( HoldPen );
	}
	// for some reason DC.GetClippingBox returns all zeros
	// at least for Windows XP.
	CalcUnscrolledPosition( 0, 0, &BoundingRect.x, &BoundingRect.y );
	GetClientSize( &BoundingRect.width, &BoundingRect.height );
	BoundingRect.x /= m_Zoom;
	BoundingRect.y /= m_Zoom;
	BoundingRect.width /= m_Zoom;
	BoundingRect.height /= m_Zoom;

	DrawSection( DC, wxPoint(0,0), BoundingRect );
}

void CModelCanvas::DrawSection( wxDC &DC, wxPoint Offset, wxRect Section )
{
	if( !m_pView )
		return;
	ClassVector						*pVect;
	ClassVector::iterator			itVect;
	InheritenceVector				*pInheritence;
	InheritenceVector::iterator		itInheritence;
	RelationVector					*pRelation;
	RelationVector::iterator		itRelation;

	DC.SetFont( m_ViewFont );
	DC.SetPen( *wxBLACK_PEN );
	pVect = m_pView->GetDoc()->GetClassVector();

	for( itVect=pVect->begin(); itVect!=pVect->end(); ++itVect ){
		DrawClass( &DC, *itVect, Offset, Section );
		pInheritence = (*itVect)->GetInheritence();
		for( itInheritence=pInheritence->begin(); itInheritence!=pInheritence->end(); ++itInheritence ){
   			DrawInheritence( &DC, *itVect, *itInheritence, Offset, Section );
        }
		pRelation = (*itVect)->GetRelation();
		for( itRelation=pRelation->begin(); itRelation!=pRelation->end(); ++itRelation ){
   			DrawRelation( &DC, *itVect, *itRelation, Offset, Section );
        }
	}
}

void CModelCanvas::DrawClass( wxDC *pDC, CClass *pClass, wxPoint &Offset, wxRect &Bounding )
{
	wxRect						ClassRect;
	MethodVector				*pMethods;
	MethodVector::iterator		itMethods;
	VariableVector				*pVariables;
	VariableVector::iterator	itVariables;
	int							YPosition;
	int							XPosition;
	int							FontHeight;
	int							TextWidth;

	pDC->GetTextExtent( _T("TextHeight"), &TextWidth, &FontHeight );
	GetClassRect( pDC, pClass, ClassRect );
	if( !Bounding.Intersects( ClassRect ) )
		return;
	ClassRect.Offset( -Offset.x, -Offset.y );
	pMethods = pClass->GetMethods();
	pVariables = pClass->GetVariables();
    pDC->SetBrush(wxBrush(wxColour(0,0,0), wxTRANSPARENT));
	pDC->DrawRectangle( ClassRect );
	YPosition = ClassRect.y;
	XPosition = ClassRect.x + 5;
	// write the name
	pDC->DrawText( wxString((wxChar*)pClass->GetName().c_str()), XPosition, YPosition );
	YPosition += FontHeight;
	pDC->DrawLine( ClassRect.x, YPosition, ClassRect.x+ClassRect.width-1, YPosition );
	YPosition += (int)(FontHeight*.75);
	// write the members
	for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
		if( (*itMethods)->GetAccess() == Public ){
		    pDC->SetBrush(wxBrush(wxColour(0,255,0), wxSOLID));
		} else if( (*itMethods)->GetAccess() == Protected ){
		    pDC->SetBrush(wxBrush(wxColour(255,255,0), wxSOLID));
		} else if( (*itMethods)->GetAccess() == Private ){
		    pDC->SetBrush(wxBrush(wxColour(255,0,0), wxSOLID));
		}
		pDC->DrawRectangle( XPosition, YPosition-FontHeight/2, FontHeight/2, FontHeight/2 );
		pDC->DrawText( wxString((wxChar*)((*itMethods)->GetFullText().c_str())), XPosition+FontHeight/2+5, YPosition-FontHeight+5 );
		YPosition += FontHeight;	
	}
	YPosition -= (int)(FontHeight*.75);
	pDC->DrawLine( ClassRect.x, YPosition, ClassRect.x+ClassRect.width-1, YPosition );
	YPosition += (int)(FontHeight*.75);
	// write the variables
	for( itVariables=pVariables->begin(); itVariables!=pVariables->end(); ++itVariables ){
		if( (*itVariables)->GetAccess() == Public ){
		    pDC->SetBrush(wxBrush(wxColour(0,255,0), wxSOLID));
		} else if( (*itVariables)->GetAccess() == Protected ){
		    pDC->SetBrush(wxBrush(wxColour(255,255,0), wxSOLID));
		} else if( (*itVariables)->GetAccess() == Private ){
		    pDC->SetBrush(wxBrush(wxColour(255,0,0), wxSOLID));
		}
		pDC->DrawRectangle( XPosition, YPosition-FontHeight/2, FontHeight/2, FontHeight/2 );
		pDC->DrawText( wxString((wxChar*)(*itVariables)->GetFullText().c_str()), XPosition+FontHeight/2+5, YPosition-FontHeight+5 );
		YPosition += FontHeight;	
	}
}

void CModelCanvas::DrawInheritence( wxDC *pDC, CClass *pChild, CInheritence *pInheritence, wxPoint &Offset, wxRect &Bounding )
{
	if( !pInheritence->GetVisible() )
		return;
	wxPoint		Start;
	wxPoint		End;
	wxRect		ParentRect;
	wxRect		ChildRect;
	wxRect 		BoundingRect;
	
	GetClassRect( pDC, pInheritence->GetParent(), ParentRect );
	GetClassRect( pDC, pChild, ChildRect );
	Start.x = ChildRect.x+ChildRect.width/2;
	End.x = ParentRect.x+ParentRect.width/2;
	if( ParentRect.y < ChildRect.y ){
		Start.y = ChildRect.y;
		End.y = ParentRect.y + ParentRect.height;
	} else {
		Start.y = ChildRect.y + ChildRect.height;
		End.y = ParentRect.y;
	}
	BoundingRect.x = Start.x<End.x?Start.x:End.x;
	BoundingRect.y = Start.y<End.y?Start.y:End.y;
	BoundingRect.width = Start.x>End.x?Start.x-BoundingRect.x:End.x-BoundingRect.x;
	BoundingRect.height = Start.y>End.y?Start.y-BoundingRect.y:End.y-BoundingRect.y;
	if( !BoundingRect.Intersects( Bounding ) )
		return;
	Start -= Offset;
	End -= Offset;
	// now do the drawing
	pDC->DrawLine( Start.x, Start.y, Start.x, pInheritence->GetHorizontal()-Offset.y );
	pDC->DrawLine( Start.x, pInheritence->GetHorizontal()-Offset.y, End.x, pInheritence->GetHorizontal()-Offset.y);
	pDC->DrawLine( End.x, pInheritence->GetHorizontal()-Offset.y, End.x, End.y );
}

void CModelCanvas::DrawRelation( wxDC *pDC, CClass *pParent, CRelation *pRelation, wxPoint &Offset, wxRect &Bounding )
{
	CClass		*pChild;
	wxPoint		Start;
	wxPoint		End;
	wxRect		ParentRect;
	wxRect		ChildRect;
	wxRect 		BoundingRect;
	wxBrush		OldBrush = pDC->GetBrush();
	wxPen		OldPen = pDC->GetPen();
	wxPen		NewPen;
	
	pChild = pRelation->GetChild();
	if( pChild == NULL )
		return;

	GetClassRect( pDC, pParent, ParentRect );
	GetClassRect( pDC, pChild, ChildRect );
	// lots of different drawing cases.  Nothing to do but
	// slog through them and find the right combo.
	// That's right, I said slog.
	if( ParentRect.GetBottom() < ChildRect.GetTop() ){
		if( pRelation->IsHasA() )
			Start.y = ParentRect.GetBottom();
		else
			Start.y = ParentRect.GetBottom()+3;
		End.y = ChildRect.GetTop();
	} else if( ParentRect.GetTop() > ChildRect.GetBottom() ){
		if( pRelation->IsHasA() )
			Start.y = ParentRect.GetTop()-3;
		else
			Start.y = ParentRect.GetTop();
		End.y = ChildRect.GetBottom();
	} else {
		if( pRelation->IsHasA() )
			Start.y = ParentRect.GetTop() + (float)ParentRect.GetHeight()/2.0+3;
		else
			Start.y = ParentRect.GetTop() + (float)ParentRect.GetHeight()/2.0-3;
		End.y = ChildRect.GetTop() + (float)ChildRect.GetHeight()/2.0;
	}
	// now work on x coords
	if( ParentRect.GetRight() < ChildRect.GetLeft() ){
		Start.x = ParentRect.GetRight();
		End.x = ChildRect.GetLeft();
	} else if( ParentRect.GetLeft() > ChildRect.GetRight() ){
		Start.x = ParentRect.GetLeft();
		End.x = ChildRect.GetRight();
	} else {
		if( pRelation->IsHasA() )
			Start.x = ParentRect.GetLeft() + (float)ParentRect.GetWidth()/2.0+3;
		else
			Start.x = ParentRect.GetLeft() + (float)ParentRect.GetWidth()/2.0-3;
		End.x = ChildRect.GetLeft() + (float)ChildRect.GetWidth()/2.0;
	}
	BoundingRect.x = Start.x<End.x?Start.x:End.x;
	BoundingRect.y = Start.y<End.y?Start.y:End.y;
	BoundingRect.width = Start.x>End.x?Start.x-BoundingRect.x:End.x-BoundingRect.x;
	BoundingRect.height = Start.y>End.y?Start.y-BoundingRect.y:End.y-BoundingRect.y;
	if( BoundingRect.height == 0 )
		BoundingRect.height = 1;
	if( BoundingRect.width == 0 )
		BoundingRect.width = 1;
	if( !BoundingRect.Intersects( Bounding ) )
		return;
	Start -= Offset;
	End -= Offset;
	// now do the drawing
	NewPen = *wxBLACK_PEN;
	if( pRelation->IsHasA() ){
		NewPen.SetStyle( wxDOT );
		pDC->SetPen( NewPen );
		pDC->DrawLine( Start.x, Start.y, End.x, End.y );
		pDC->SetBrush( *wxBLACK_BRUSH );
	} else {
		NewPen.SetStyle( wxSHORT_DASH );
		pDC->SetPen( NewPen );
		pDC->DrawLine( Start.x, Start.y, End.x, End.y );
		pDC->SetBrush( *wxWHITE_BRUSH );
	}
	pDC->SetPen( OldPen );
	pDC->DrawCircle( Start, 5 ); 
	pDC->SetBrush( OldBrush );
}

CClass *CModelCanvas::GetClassAt( const wxPoint &Point )
{
	ClassVector				*pClasses;
	ClassVector::iterator	itVect;
	wxRect					ClassRect;

	pClasses = m_pView->GetDoc()->GetClassVector();
	for( itVect=pClasses->begin(); itVect!=pClasses->end(); ++itVect ){
		GetClassRect( NULL, *itVect, ClassRect );
		if( ClassRect.Inside( Point ) ){
			return *itVect;
		}
	}

	return NULL;
}

CInheritence *CModelCanvas::GetInheritenceAt( const wxPoint &Point )
{
	ClassVector                 *pClasses;
	ClassVector::iterator       itClasses;
	InheritenceVector			*pInheritence;
	InheritenceVector::iterator	itVect;
	wxRect						InheritenceRect;
	wxRect						ChildRect;
	wxRect						ParentRect;

	pClasses = m_pView->GetDoc()->GetClassVector();
	for( itClasses=pClasses->begin(); itClasses!=pClasses->end(); ++itClasses ){
		pInheritence = (*itClasses)->GetInheritence();
		for( itVect=pInheritence->begin(); itVect!=pInheritence->end(); ++itVect ){
			if( (*itVect)->GetVisible() ){
			GetClassRect( NULL, (*itVect)->GetParent(), ParentRect );
			GetClassRect( NULL, *itClasses, ChildRect );
			if( ChildRect.x < ParentRect.x ){
				InheritenceRect.x = ChildRect.x + (ChildRect.width/2);
				InheritenceRect.width = ParentRect.x + (ParentRect.width/2) - InheritenceRect.x;
			} else {
				InheritenceRect.x = ParentRect.x + (ParentRect.width/2);
				InheritenceRect.width = ChildRect.x + (ChildRect.width/2) - InheritenceRect.x;
			}
			InheritenceRect.y = (*itVect)->GetHorizontal()-1;
			InheritenceRect.height = 4;
			if( InheritenceRect.Inside( Point ) )
				return *itVect;
			}
		}
	}
	
	return NULL;
}

void CModelCanvas::GetClassRect( wxDC *pDC, CClass *pClass, wxRect &SizeRect )
{
	pClass->GetRect( SizeRect );	
	SizeRect.x = pClass->GetPosition().x;
	SizeRect.y = pClass->GetPosition().y;
}

void CModelCanvas::OnNewClass( wxCommandEvent &Event )
{
	CClass		*pNewClass = new CClass( *(m_pView->GetDoc()) );
	wxClientDC	DC( this );

	DC.SetFont( m_ViewFont );
	pNewClass->SetPosition( m_ContentsClickPosition );
	pNewClass->ComputeRect( &DC );
	m_pView->GetDoc()->AddClass( pNewClass );
	m_pView->GetDoc()->UpdateAllViews();
	wxGetApp().GetMainFrame()->DrawTree( m_pView->GetDoc(), true );
	m_pView->GetDoc()->Modify( true );
}

void CModelCanvas::OnNewClassFromHeader( wxCommandEvent &Event )
{
	wxFileDialog					FileDialog( this, _T("Select a header file to read"), _T(""), _T(""), _T("Header files (*.h)|*.h|All Files (*)|*")  );
	ClassVector						NewClasses;
	ClassVector::iterator			itVect;
	InheritenceVector				*pInheritence;
	InheritenceVector::iterator		itInheritence;
	CClass							*pParent;
	wxClientDC						DC( this );

	if( FileDialog.ShowModal() != wxID_OK )
		return;

	string FileName = (wxChar*)(FileDialog.GetPath().c_str());
	if( FileName == "" )
		return;
	if( !m_pView->GetDoc()->NewClassesFromHeader(FileName, NewClasses ) ){
		wxMessageBox( _T("Failed to create class"), _T("Error !") );
		return;
	}
	DC.SetFont( m_ViewFont );
	for( itVect=NewClasses.begin(); itVect!=NewClasses.end(); ++itVect ){
		(*itVect)->SetPosition( m_ContentsClickPosition );

		m_pView->GetDoc()->AddClass( *itVect );
		(*itVect)->ComputeRect( &DC );
		m_ContentsClickPosition.x = m_ContentsClickPosition.x+20;
		m_ContentsClickPosition.y = m_ContentsClickPosition.y+20;
	}
	// it's possible that a child class was read before a parent class
	// update the inheritence structure so the correct lines are drawn
	for( itVect=NewClasses.begin(); itVect!=NewClasses.end(); ++itVect ){
		pInheritence = (*itVect)->GetInheritence();
		for( itInheritence=pInheritence->begin(); itInheritence!=pInheritence->end(); ++itInheritence ){
			pParent = m_pView->GetDoc()->FindClassNamed( (*itInheritence)->GetParentName() );
			if( pParent )
				(*itInheritence)->SetParent( pParent );
		}				
	}			
	m_pView->GetDoc()->UpdateAllViews( NULL );	// include self in update
	wxGetApp().GetMainFrame()->DrawTree( m_pView->GetDoc(), true );
	m_pView->GetDoc()->Modify( true );
}

void CModelCanvas::OnNewClassFromClipboard( wxCommandEvent &Event )
{
	ClassVector						NewClasses;
	ClassVector::iterator			itVect;
	InheritenceVector				*pInheritence;
	InheritenceVector::iterator		itInheritence;
	CClass							*pParent;
	wxClientDC						DC( this );

	if( !m_pView->GetDoc()->NewClassesFromClipboard( NewClasses ) ){
		wxMessageBox( _T("Failed to create class"), _T("Error !") );
		return;
	}
	DC.SetFont( m_ViewFont );
	for( itVect=NewClasses.begin(); itVect!=NewClasses.end(); ++itVect ){
		(*itVect)->SetPosition( m_ContentsClickPosition );

		m_pView->GetDoc()->AddClass( *itVect );
		(*itVect)->ComputeRect( &DC );
		m_ContentsClickPosition.x = m_ContentsClickPosition.x+20;
		m_ContentsClickPosition.y = m_ContentsClickPosition.y+20;
	}
	// it's possible that a child class was read before a parent class
	// update the inheritence structure so the correct lines are drawn
	for( itVect=NewClasses.begin(); itVect!=NewClasses.end(); ++itVect ){
		pInheritence = (*itVect)->GetInheritence();
		for( itInheritence=pInheritence->begin(); itInheritence!=pInheritence->end(); ++itInheritence ){
			pParent = m_pView->GetDoc()->FindClassNamed( (*itInheritence)->GetParentName() );
			if( pParent )
				(*itInheritence)->SetParent( pParent );
		}				
	}			
	m_pView->GetDoc()->UpdateAllViews( NULL );	// include self in update
	wxGetApp().GetMainFrame()->DrawTree( m_pView->GetDoc(), true );
	m_pView->GetDoc()->Modify( true );
}

void CModelCanvas::OnZoomIn( wxCommandEvent &Event )
{
	wxClientDC		DC( this );
	int				ContentsWidth, ContentsHeight;
	int				X, Y;

	m_Zoom *= ZOOM_STEP;
	DC.SetFont( m_ViewFont );
	DC.SetUserScale( m_Zoom, m_Zoom );
	GetViewStart( &X, &Y );
	GetVirtualSize( &ContentsWidth, &ContentsHeight );
	SetVirtualSize( ContentsWidth*ZOOM_STEP, ContentsHeight*ZOOM_STEP );
	Scroll( X*ZOOM_STEP, Y*ZOOM_STEP );
	ComputeSizes();
	Refresh();
}

void CModelCanvas::OnZoomOut( wxCommandEvent &Event )
{
	wxClientDC		DC( this );
	int				ContentsWidth, ContentsHeight;
	int				X, Y;

	m_Zoom /= ZOOM_STEP;
	DC.SetFont( m_ViewFont );
	DC.SetUserScale( m_Zoom, m_Zoom );
	GetViewStart( &X, &Y );
	GetVirtualSize( &ContentsWidth, &ContentsHeight );
	SetVirtualSize( ContentsWidth/ZOOM_STEP, ContentsHeight/ZOOM_STEP );
	Scroll( X/ZOOM_STEP, Y/ZOOM_STEP );
	ComputeSizes();
	Refresh();
}

void CModelCanvas::EnableGrid( bool bUseGrid )
{
	m_bUseGrid = bUseGrid;
	Refresh();
}

void CModelCanvas::OnDeleteClass( wxCommandEvent &Event )
{
	CClass							*pClass;
	ClassVector::iterator			itVect;
	InheritenceVector				*pInheritence;
	InheritenceVector::iterator		itInheritence;
	RelationVector					*pRelation;
	RelationVector::iterator		itRelation;
	
	pClass = GetClassAt( m_ContentsClickPosition );
	if( pClass == NULL )
		return;
	for( itVect=m_pView->GetDoc()->GetClassVector()->begin(); itVect!=m_pView->GetDoc()->GetClassVector()->end(); ++itVect ){
		pInheritence = (*itVect)->GetInheritence();
		for( itInheritence=pInheritence->begin(); itInheritence!=pInheritence->end(); ){
			if( (*itInheritence)->GetParent() == pClass )
				itInheritence = pInheritence->erase( itInheritence );
			else
				++itInheritence;
		}
		pRelation = (*itVect)->GetRelation();
		for( itRelation=pRelation->begin(); itRelation!=pRelation->end(); ){
			if( (*itRelation)->GetChild() == pClass )
				itRelation = pRelation->erase( itRelation );
			else
				++itRelation;
		}
	}   
	m_pView->GetDoc()->RemoveClass( pClass );
	m_pView->GetDoc()->UpdateAllViews( NULL );	// include self in update
	wxGetApp().GetMainFrame()->DrawTree( m_pView->GetDoc(), true );
}

void CModelCanvas::OnInheritFrom( wxCommandEvent &Event )
{
	m_pInheritenceChild = GetClassAt( m_ContentsClickPosition );
}

void CModelCanvas::OnHasA( wxCommandEvent &Event )
{
	m_pHasAParent = GetClassAt( m_ContentsClickPosition );
}

void CModelCanvas::OnUsesA( wxCommandEvent &Event )
{
	m_pUsesAParent = GetClassAt( m_ContentsClickPosition );
}

void CModelCanvas::OnGenerateCode( wxCommandEvent &Event )
{
	CClass	*pClass = GetClassAt( m_ContentsClickPosition );

	if( pClass ){
		if( !m_pView->GetDoc()->GenerateCodeFor( pClass ) )
			wxMessageBox( _T("Failed to generate code"), _T("Error !") );
	} else
		wxMessageBox( _T("Failed to generate code"), _T("Error !") );
}

void CModelCanvas::GetPageSize( int &Width, int &Height )
{
	Width = m_PrinterPageWidth;
	Height = m_PrinterPageHeight;
}

void CModelCanvas::GetModelDimensions( wxRect &BoundingRect )
{
	ClassVector				*pClasses;
	ClassVector::iterator	itVect;
	wxRect					ClassRect;
	int						Left, Top;
	int						Right, Bottom;

	Left = Top = 10000;
	Right = Bottom = 0;
	pClasses = m_pView->GetDoc()->GetClassVector();
	for( itVect=pClasses->begin(); itVect!=pClasses->end(); ++itVect ){
		GetClassRect( NULL, *itVect, ClassRect );
		if( ClassRect.x < Left )
			Left = ClassRect.x;
		if( ClassRect.y < Top )
			Top = ClassRect.y;
		if( (ClassRect.x+ClassRect.width) > Right )
			Right = ClassRect.x+ClassRect.width;
		if( (ClassRect.y+ClassRect.height) > Bottom )
			Bottom = ClassRect.y+ClassRect.height;
	}
	BoundingRect.x = Left;
	BoundingRect.y = Top;
	BoundingRect.width = Right-Left;
	BoundingRect.height = Bottom-Top;
}

bool CModelCanvas::GenerateJPGImage( string FileName, int FileType )
{
	wxRect			ClipRect;
	wxPoint			Offset;
	wxMemoryDC		BitmapDC;
	wxBitmap		*pBitmap;

	GetModelDimensions( ClipRect );
	MasterLog.Log( LVL_HIGH, "%d, %d.... %d\n", ClipRect.width, ClipRect.height, wxDisplayDepth() );
	pBitmap = new wxBitmap();
	if( !pBitmap->Create( ClipRect.width, ClipRect.height ) ){
		MasterLog.Log( LVL_HIGH, "CModelCanvas::GenerateJPGImage failed to create Bitmap\n" );
		delete pBitmap;
		return false;
	}
	
	BitmapDC.SelectObject( *pBitmap );
	if( !BitmapDC.FloodFill( 0, 0, *wxWHITE, wxFLOOD_BORDER ) ){
		MasterLog.Log( LVL_HIGH, "CModelCanvas::GenerateJPGImage FloodFill failed\n" );
		BitmapDC.SelectObject( wxNullBitmap );
		delete pBitmap;
		return false;
	}
	Offset.x = ClipRect.x;
	Offset.y = ClipRect.y;
	DrawSection( BitmapDC, Offset, ClipRect );
	if( !pBitmap->SaveFile( (const char*)FileName.c_str(), FileType ) ){
		MasterLog.Log( LVL_HIGH, "CModelCanvas::GenerateJPGImage wxBitmap::SaveFile failed\n" );
		BitmapDC.SelectObject( wxNullBitmap );
		delete pBitmap;
		return false;
	}
	delete pBitmap;

	return true;	
}
