package org.jaxup.tests;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.jaxen.JaxenException;
import org.jaxen.Navigator;
import org.jaxup.Updater;
import org.jaxup.xupdate.NormalizedTextNodeIterator;

public class XMLComparator
{
    private Updater updater;
    private Navigator navigator;

    public XMLComparator(Updater updater)
    {
        this.updater = updater;
        this.navigator = updater.getNavigator();
    }

    public boolean compare(Object nodeA, Object nodeB) throws JaxenException
    {
        if (navigator.isDocument(nodeA))
        {
            if (!navigator.isDocument(nodeB))
            {
                return false;
            }

            return compareDocuments(nodeA, nodeB);
        }

        if (navigator.isElement(nodeA))
        {
            if (!navigator.isElement(nodeB))
            {
                return false;
            }

            return compareElements(nodeA, nodeB);
        }

        if (navigator.isAttribute(nodeA))
        {
            if (!navigator.isAttribute(nodeB))
            {
                return false;
            }

            return compareAttributes(nodeA, nodeB);
        }

        if (navigator.isComment(nodeA))
        {
            if (!navigator.isComment(nodeB))
            {
                return false;
            }

            return compareComments(nodeA, nodeB);
        }

        if (navigator.isText(nodeA))
        {
            if (!navigator.isText(nodeB))
            {
                return false;
            }

            return compareTexts(nodeA, nodeB);
        }

        if (navigator.isProcessingInstruction(nodeA))
        {
            if (!navigator.isProcessingInstruction(nodeB))
            {
                return false;
            }

            return compareProcessingInstructions(nodeA, nodeB);
        }

        if (navigator.isNamespace(nodeA))
        {
            if (!navigator.isNamespace(nodeB))
            {
                return false;
            }

            return compareNamespaces(nodeA, nodeB);
        }

        return false;
    }

    private boolean compareDocuments(Object nodeA, Object nodeB) throws JaxenException
    {
        Iterator childrenA = navigator.getChildAxisIterator(nodeA);
        Iterator childrenB = navigator.getChildAxisIterator(nodeB);
        childrenA = new NormalizedTextNodeIterator(updater, childrenA);
        childrenB = new NormalizedTextNodeIterator(updater, childrenB);
        if (!compareLists(childrenA, childrenB))
        {
            return false;
        }

        // All checks turned out okay.
        return true;
    }

    private boolean compareElements(Object nodeA, Object nodeB) throws JaxenException
    {
        String qnameA = navigator.getElementQName(nodeA);
        String qnameB = navigator.getElementQName(nodeB);
        if (!qnameA.equals(qnameB))
        {
            return false;
        }

        String nsUriA = navigator.getElementNamespaceUri(nodeA);
        String nsUriB = navigator.getElementNamespaceUri(nodeB);
        if (nsUriA == null ? nsUriB != null : !nsUriA.equals(nsUriB))
        {
            return false;
        }

        Iterator attrsA = navigator.getAttributeAxisIterator(nodeA);
        Iterator attrsB = navigator.getAttributeAxisIterator(nodeB);
        if (!compareAttributeLists(attrsA, attrsB))
        {
            return false;
        }

        Iterator childrenA = navigator.getChildAxisIterator(nodeA);
        Iterator childrenB = navigator.getChildAxisIterator(nodeB);
        childrenA = new NormalizedTextNodeIterator(updater, childrenA);
        childrenB = new NormalizedTextNodeIterator(updater, childrenB);
        if (!compareLists(childrenA, childrenB))
        {
            return false;
        }

        // All checks turned out okay.
        return true;
    }

    private boolean compareAttributes(Object nodeA, Object nodeB) throws JaxenException
    {
        String qnameA = navigator.getAttributeQName(nodeA);
        String qnameB = navigator.getAttributeQName(nodeB);
        if (!qnameA.equals(qnameB))
        {
            return false;
        }

        String nsUriA = navigator.getAttributeNamespaceUri(nodeA);
        String nsUriB = navigator.getAttributeNamespaceUri(nodeB);
        if (nsUriA == null ? nsUriB != null : !nsUriA.equals(nsUriB))
        {
            return false;
        }

        String valueA = navigator.getAttributeStringValue(nodeA);
        String valueB = navigator.getAttributeStringValue(nodeB);
        if (!valueA.equals(valueB))
        {
            return false;
        }

        // All checks turned out okay.
        return true;
    }

    private boolean compareTexts(Object nodeA, Object nodeB) throws JaxenException
    {

        String valueA = navigator.getTextStringValue(nodeA);
        String valueB = navigator.getTextStringValue(nodeB);

        // Ignore leading and trailing whitespace
        valueA = valueA.trim();
        valueB = valueB.trim();

        if (!valueA.equals(valueB))
        {
            return false;
        }

        // All checks turned out okay.
        return true;
    }

    private boolean compareComments(Object nodeA, Object nodeB) throws JaxenException
    {

        String valueA = navigator.getCommentStringValue(nodeA);
        String valueB = navigator.getCommentStringValue(nodeB);

        if (!valueA.equals(valueB))
        {
            return false;
        }

        // All checks turned out okay.
        return true;
    }

    private boolean compareProcessingInstructions(Object nodeA, Object nodeB) throws JaxenException
    {
        String targetA = navigator.getProcessingInstructionTarget(nodeA);
        String targetB = navigator.getProcessingInstructionTarget(nodeB);

        if (!targetA.equals(targetB))
        {
            return false;
        }

        String dataA = navigator.getProcessingInstructionData(nodeA);
        String dataB = navigator.getProcessingInstructionData(nodeB);

        if (!dataA.equals(dataB))
        {
            return false;
        }

        // All checks turned out okay.
        return true;
    }

    private boolean compareNamespaces(Object nodeA, Object nodeB) throws JaxenException
    {

        String prefixA = navigator.getNamespacePrefix(nodeA);
        String prefixB = navigator.getNamespacePrefix(nodeB);

        if (!prefixA.equals(prefixB))
        {
            return false;
        }

        String valueA = navigator.getNamespaceStringValue(nodeA);
        String valueB = navigator.getNamespaceStringValue(nodeB);
        if (!valueA.equals(valueB))
        {
            return false;
        }

        // All checks turned out okay.
        return true;
    }

    private boolean compareLists(Iterator childrenA, Iterator childrenB) throws JaxenException
    {

        while (childrenA.hasNext() && childrenB.hasNext())
        {
            Object childA = childrenA.next();
            Object childB = childrenB.next();

            if (!compare(childA, childB))
                return false;
        }

        if (!childrenA.hasNext() && !childrenB.hasNext())
        {
            return true;
        }
        else
        {
            // Different lengths of children
            return false;
        }
    }

    private List collectIterator(Iterator i)
    {
        List l = new ArrayList();
        while (i.hasNext())
        {
            l.add(i.next());
        }
        return l;
    }

    private boolean compareAttributeLists(Iterator iterAttrsA, Iterator iterAttrsB) throws JaxenException
    {

        List attrsA = collectIterator(iterAttrsA);
        List attrsB = collectIterator(iterAttrsB);

        if (checkAllAttrsAInB(attrsA, attrsB) && checkAllAttrsAInB(attrsB, attrsA))
        {
            return true;
        }
        return false;
    }

    private boolean checkAllAttrsAInB(List attrsA, List attrsB) throws JaxenException
    {

        for (Iterator a = attrsA.iterator(); a.hasNext();)
        {
            Object attrA = a.next();
            String qnameA = navigator.getAttributeQName(attrA);
            if (qnameA.startsWith("xmlns:"))
            {
                // Ignore namespace attributes
                continue;
            }

            boolean existsInB = false;
            for (Iterator b = attrsB.iterator(); b.hasNext();)
            {
                Object attrB = b.next();
                String qnameB = navigator.getAttributeQName(attrB);
                if (qnameB.startsWith("xmlns:"))
                {
                    // Ignore namespace attributes
                    continue;
                }

                if (compareAttributes(attrA, attrB))
                {
                    existsInB = true;
                    break;
                }
            }
            if (!existsInB)
            {
                return false;
            }
        }
        return true;
    }
}
