/*
 * Copyright (C) 2005 - 2012 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.jaspersoft.jasperserver.remote.services.impl;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.FolderImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.QueryImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.remote.exception.AccessDeniedException;
import com.jaspersoft.jasperserver.remote.exception.FolderNotFoundException;
import com.jaspersoft.jasperserver.remote.exception.IllegalParameterValueException;
import com.jaspersoft.jasperserver.remote.exception.MandatoryParameterNotFoundException;
import com.jaspersoft.jasperserver.remote.exception.ResourceAlreadyExistsException;
import com.jaspersoft.jasperserver.remote.exception.ResourceNotFoundException;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.security.BadCredentialsException;
import org.testng.annotations.AfterGroups;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeGroups;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import static org.mockito.Matchers.*;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.testng.Assert.assertEquals;

/**
 * <p></p>
 *
 * @author Zakhar.Tomchenco
 * @version $Id$
 */
public class SingleRepositoryServiceImplTest {
    @InjectMocks
    private SingleRepositoryServiceImpl service;// = new SingleRepositoryServiceImpl();
    @Mock
    private RepositoryService repositoryService;


    final private String uri = "/test";
    final private String uri2 = "/test/r";
    final private Folder folder = new FolderImpl();
    final private Resource resource = new QueryImpl();
    final private Resource resource2 = new QueryImpl();

    @BeforeClass
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
    }

    @BeforeMethod
    public void cleanUp(){
        Mockito.reset(repositoryService);
        folder.setURIString("/");
        folder.setLabel("l");
        folder.setName("folder");
        resource.setURIString("/test/folder");
        resource.setLabel("Folder");
        resource2.setURIString(uri2 + "/" + resource.getName());
    }

    @BeforeGroups(groups = "POST")
    public void makePartial() throws Exception{
        service = Mockito.spy(service);
        doReturn(resource).when(service).getResource(anyString());
    }

    @AfterGroups(groups = "POST")
    public void restore() throws Exception{
        MockitoAnnotations.initMocks(this);
    }

    @Test(groups = "DELETE", expectedExceptions = {IllegalParameterValueException.class})
    public void testDeleteResource_Root() throws Exception{
        service.deleteResource("/");

        Mockito.verify(repositoryService).deleteResource(Mockito.any(ExecutionContext.class), Mockito.eq(uri));
    }

    @Test(groups = "DELETE", expectedExceptions = {IllegalParameterValueException.class})
    public void testDeleteResource_Null() throws Exception{
        service.deleteResource(null);
    }

    @Test(groups = "DELETE", expectedExceptions = {IllegalParameterValueException.class})
    public void testDeleteResource_Empty() throws Exception{
        service.deleteResource("");
    }

    @Test(groups = "DELETE")
    public void testDeleteResource_Resource() throws Exception{
        Mockito.when(repositoryService.getResource(Mockito.any(ExecutionContext.class), Mockito.eq(uri))).thenReturn(resource);

        service.deleteResource(uri);

        Mockito.verify(repositoryService).deleteResource(Mockito.any(ExecutionContext.class), Mockito.eq(uri));
    }

    @Test(groups = "DELETE")
    public void testDeleteResource_Folder() throws Exception{
        Mockito.when(repositoryService.getResource(Mockito.any(ExecutionContext.class), Mockito.eq(uri))).thenReturn(null);
        Mockito.when(repositoryService.getFolder(Mockito.any(ExecutionContext.class), Mockito.eq(uri))).thenReturn(folder);

        service.deleteResource(uri);

        Mockito.verify(repositoryService).deleteFolder(Mockito.any(ExecutionContext.class),Mockito.eq(uri));
    }

    @Test(groups = "DELETE", expectedExceptions = {AccessDeniedException.class})
    public void testDeleteResource_Forbidden() throws Exception{
        Mockito.when(repositoryService.getResource(Mockito.any(ExecutionContext.class), Mockito.eq(uri))).thenReturn(null);
        Mockito.when(repositoryService.getFolder(Mockito.any(ExecutionContext.class), Mockito.eq(uri))).thenReturn(folder);
        Mockito.doThrow(new org.springframework.security.AccessDeniedException("NO")).when(repositoryService).deleteFolder(Mockito.any(ExecutionContext.class), Mockito.eq(uri));

        service.deleteResource(uri);
    }

    @Test(groups = "POST")
    public void testCreateResource_created() throws Exception{
        Mockito.doReturn(true).when(repositoryService).folderExists(any(ExecutionContext.class), anyString());
        Mockito.when(repositoryService.getFolder(Mockito.any(ExecutionContext.class), Mockito.eq(resource.getURIString()))).thenReturn(folder);

        Resource response = service.createResource(resource, uri, null, false);

        Mockito.verify(repositoryService).saveResource(any(ExecutionContext.class), any(Resource.class));
    }

    @Test(groups = "POST", expectedExceptions = {ResourceAlreadyExistsException.class})
    public void testCreateResource_oneOfGeneratedIsResource() throws Exception {
        Mockito.when(repositoryService.getResource(Mockito.any(ExecutionContext.class), anyString())).thenReturn(resource);
        Mockito.when(repositoryService.getFolder(any(ExecutionContext.class), eq(""))).thenReturn(folder);

        Resource response = service.createResource(resource, uri, null, true);

        assertEquals(response.getName(), resource.getLabel().concat("_1"));
    }

    @Test(groups = "POST")
    public void testCreateResource_generateNameIfResourceExists() throws Exception{
        Mockito.when(repositoryService.getResource(Mockito.any(ExecutionContext.class), eq(uri + Folder.SEPARATOR + resource.getLabel()))).thenReturn(resource);
        Mockito.when(repositoryService.getFolder(any(ExecutionContext.class), eq(""))).thenReturn(folder);

        Resource response = service.createResource(resource, uri, null, true);

        assertEquals(response.getName(), resource.getLabel().concat("_1"));
    }

    @Test(groups = "POST")
    public void testCreateResource_generateNameIfResourceExistsTwice() throws Exception{
        resource.setLabel(resource.getLabel().concat("_1"));

        Mockito.when(repositoryService.getResource(Mockito.any(ExecutionContext.class), eq(uri + Folder.SEPARATOR + resource.getLabel()))).thenReturn(resource);
        Mockito.when(repositoryService.getFolder(any(ExecutionContext.class), eq(""))).thenReturn(folder);

        Resource response = service.createResource(resource, uri, null, true);

        assertEquals(response.getName(), resource.getLabel().replace("1", "2"));
    }

    @Test(groups = "POST", expectedExceptions = {FolderNotFoundException.class})
    public void testCreateResource_failIfFolderNotExists() throws Exception{
        Mockito.doReturn(true).when(repositoryService).resourceExists(any(ExecutionContext.class), anyString());
        Mockito.doReturn(false).when(repositoryService).folderExists(any(ExecutionContext.class), anyString());

        Resource response = service.createResource(resource, uri, null, false);
    }

    @Test(groups = "POST")
    public void testCreateResource_createNotExistingFolder() throws Exception{
        Mockito.doReturn(false).when(repositoryService).resourceExists(any(ExecutionContext.class), anyString());
        Mockito.when(repositoryService.getFolder(any(ExecutionContext.class), eq(""))).thenReturn(folder);

        Resource response = service.createResource(resource, uri, null, true);

        Mockito.verify(repositoryService).saveFolder(any(ExecutionContext.class), any(Folder.class));
    }

    @Test(groups = "POST")
    public void testCreateResource_createNotExistingFolders() throws Exception{
        Mockito.doReturn(false).when(repositoryService).resourceExists(any(ExecutionContext.class), anyString());
        Mockito.when(repositoryService.getFolder(any(ExecutionContext.class), eq(""))).thenReturn((Folder) folder);

        resource.setURIString(resource.getURIString() + "/gfyu");
        Resource response = service.createResource(resource, resource.getURIString(), null, true);

        Mockito.verify(repositoryService, times(3)).saveFolder(any(ExecutionContext.class), any(Folder.class));
    }

    @Test(groups = "POST", expectedExceptions = {AccessDeniedException.class})
    public void testCreateResource_createNotExistingFoldersWithoutPermission() throws Exception{
        Mockito.doReturn(false).when(repositoryService).resourceExists(any(ExecutionContext.class), anyString());
        Mockito.when(repositoryService.getFolder(any(ExecutionContext.class), eq(""))).thenThrow(BadCredentialsException.class);

        resource.setURIString(resource.getURIString() + "/gfyu");
        Resource response = service.createResource(resource, resource.getURIString(), null, true);

        Mockito.verify(repositoryService, times(2)).saveFolder(any(ExecutionContext.class), any(Folder.class));
    }

    @Test(groups = "POST")
    public void testCreateResource_createsFolder() throws Exception {
        Mockito.when(repositoryService.folderExists(any(ExecutionContext.class), anyString())).thenReturn(true);

        Resource response = service.createResource(folder, uri, null, false);

        Mockito.verify(repositoryService).saveFolder(any(ExecutionContext.class), any(Folder.class));
    }

    @Test(groups = "COPY")
    public void testCopyResource_copy() throws Exception{
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri))).thenReturn(resource);
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri2))).thenReturn(null);
        Mockito.when(repositoryService.folderExists(any(ExecutionContext.class), anyString())).thenReturn(true);

        service.copyResource(uri, uri2, false, false);

        verify(repositoryService).copyResource(any(ExecutionContext.class), eq(uri), eq(uri2 + Folder.SEPARATOR + resource2.getName()));
    }

    @Test(groups = "COPY", expectedExceptions = {IllegalParameterValueException.class})
    public void testCopyResource_sourceNotSet() throws Exception{
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri))).thenReturn(resource);
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri2))).thenReturn(null);
        Mockito.when(repositoryService.folderExists(any(ExecutionContext.class), anyString())).thenReturn(true);

        service.copyResource(null, uri2, false, false);
    }

    @Test(groups = "COPY", expectedExceptions = {ResourceNotFoundException.class})
    public void testCopyResource_copyNotExists() throws Exception{
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri))).thenReturn(null);
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri2))).thenReturn(null);
        Mockito.when(repositoryService.folderExists(any(ExecutionContext.class), anyString())).thenReturn(true);

        service.copyResource(uri, uri2, false, false);
    }

    @Test(groups = "COPY", expectedExceptions = {ResourceAlreadyExistsException.class})
    public void testCopyResource_copyDestinationAlreadyExists() throws Exception{
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri))).thenReturn(resource);
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(resource2.getURIString()))).thenReturn(resource2);
        Mockito.when(repositoryService.folderExists(any(ExecutionContext.class), anyString())).thenReturn(true);

        service.copyResource(uri, uri2, false, false);
    }

    @Test(groups = "COPY")
    public void testCopyResource_copyDestinationAlreadyExistsOverwrite() throws Exception{
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri))).thenReturn(resource);
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(resource2.getURIString()))).thenReturn(resource2);
        Mockito.when(repositoryService.folderExists(any(ExecutionContext.class), anyString())).thenReturn(true);

        service.copyResource(uri, uri2, false, true);

        verify(repositoryService).deleteResource(any(ExecutionContext.class), eq(uri2 + Folder.SEPARATOR + resource2.getName()));
    }

    @Test(groups = "COPY")
    public void testCopyResource_copyFolder() throws Exception{
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri))).thenReturn(folder);
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri2))).thenReturn(null);
        Mockito.when(repositoryService.folderExists(any(ExecutionContext.class), anyString())).thenReturn(true);

        service.copyResource(uri, uri2, false, false);

        verify(repositoryService).copyFolder(any(ExecutionContext.class), eq(uri), eq(uri2 + Folder.SEPARATOR + folder.getName()));
    }

    @Test(groups = "MOVE", dependsOnGroups = {"COPY"})
    public void testMoveResource_move() throws Exception{
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri))).thenReturn(resource);
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri2))).thenReturn(null);
        Mockito.when(repositoryService.folderExists(any(ExecutionContext.class), anyString())).thenReturn(true);

        service.moveResource(uri, uri2, false, false);

        verify(repositoryService).moveResource(any(ExecutionContext.class), eq(uri), eq(uri2 + Folder.SEPARATOR + folder.getName()));
    }

    @Test(groups = "MOVE", dependsOnGroups = {"COPY"})
    public void testMoveResource_moveFolder() throws Exception{
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri))).thenReturn(folder);
        Mockito.when(repositoryService.getResource(any(ExecutionContext.class), eq(uri2))).thenReturn(null);
        Mockito.when(repositoryService.folderExists(any(ExecutionContext.class), anyString())).thenReturn(true);

        service.moveResource(uri, uri2, false, false);

        verify(repositoryService).moveFolder(any(ExecutionContext.class), eq(uri), eq(uri2 + Folder.SEPARATOR + folder.getName()));
    }
}
