/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.remote.resources.converters;

import com.jaspersoft.jasperserver.api.common.util.TimeZoneContextHolder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.service.ResourceFactory;
import com.jaspersoft.jasperserver.dto.resources.ClientResourceLookup;
import com.jaspersoft.jasperserver.remote.exception.IllegalParameterValueException;
import com.jaspersoft.jasperserver.remote.exception.MandatoryParameterNotFoundException;
import com.jaspersoft.jasperserver.remote.resources.GenericParametersReflectionHelper;
import com.jaspersoft.jasperserver.remote.services.PermissionsService;
import com.jaspersoft.jasperserver.war.util.CalendarFormatProvider;
import org.springframework.security.Authentication;
import org.springframework.security.context.SecurityContextHolder;

import java.text.DateFormat;
import java.text.ParseException;

/**
 * <p></p>
 *
 * @author Yaroslav.Kovalchyk
 * @version $Id: ResourceConverterImpl.java 29489 2013-03-11 15:59:46Z ztomchenco $
 */
public abstract class ResourceConverterImpl<ResourceType extends Resource, ClientType extends ClientResourceLookup> implements ResourceConverter<ResourceType, ClientType> {
    @javax.annotation.Resource(name = "mappingResourceFactory")
    protected ResourceFactory objectFactory;

    @javax.annotation.Resource(name = "messagesCalendarFormatProvider")
    protected CalendarFormatProvider calendarFormatProvider;

    @javax.annotation.Resource(name = "concretePermissionsService")
    private PermissionsService permissionsService;

    private Class<ClientType> clientTypeClass;

    private String serverResourceType;

    private String clientResourceType;

    // object factory returns correct type of resource. So, cast below is safe
    @SuppressWarnings("unchecked")
    protected ResourceType getNewResourceInstance() {
        return (ResourceType) objectFactory.newResource(null, getServerResourceType());
    }

    protected DateFormat getDateTimeFormat() {
        return calendarFormatProvider.getDatetimeFormat();
    }

    public String getServerResourceType() {
        if (serverResourceType == null) {
            serverResourceType = GenericParametersReflectionHelper.getGenericTypeArgument(this.getClass(), ResourceConverter.class, 0).getName();
        }
        return serverResourceType;
    }

    @Override
    public ResourceType toServer(ClientType clientObject) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
        final ResourceType newResourceInstance = getNewResourceInstance();
        return toServer(clientObject, newResourceInstance);
    }

    @Override
    public ResourceType toServer(ClientType clientObject, ResourceType resultToUpdate) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
        resultToUpdate = genericFieldsToServer(clientObject, resultToUpdate);
        return resourceSpecificFieldsToServer(clientObject, resultToUpdate);
    }

    protected ResourceType genericFieldsToServer(ClientType clientObject, ResourceType resultToUpdate) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
        if (clientObject.getLabel() == null || "".equals(clientObject.getLabel())){
            throw new MandatoryParameterNotFoundException(getClientResourceType()+".label");
        }
        resultToUpdate.setURIString(clientObject.getUri());
        final DateFormat dateTimeFormatter = getDateTimeFormat();
        dateTimeFormatter.setTimeZone(TimeZoneContextHolder.getTimeZone());
        if (clientObject.getCreationDate() != null) {
            try {
                resultToUpdate.setCreationDate(dateTimeFormatter.parse(clientObject.getCreationDate()));
            } catch (ParseException ex) {
                throw new IllegalParameterValueException("creationDate", clientObject.getCreationDate());
            }
        }
        if (clientObject.getUpdateDate() != null) {
            try {
                resultToUpdate.setUpdateDate(dateTimeFormatter.parse(clientObject.getUpdateDate()));
            } catch (ParseException ex) {
                throw new IllegalParameterValueException("updateDate", clientObject.getUpdateDate());
            }
        }
        resultToUpdate.setDescription(clientObject.getDescription());
        resultToUpdate.setLabel(clientObject.getLabel());
        resultToUpdate.setVersion(clientObject.getVersion() != null ? clientObject.getVersion() : Resource.VERSION_NEW);
        return resultToUpdate;
    }

    protected abstract ResourceType resourceSpecificFieldsToServer(ClientType clientObject, ResourceType resultToUpdate) throws IllegalParameterValueException, MandatoryParameterNotFoundException;

    @Override
    public ClientType toClient(ResourceType serverObject) {
        final ClientType client = genericFieldsToClient(getNewClientObjectInstance(), serverObject);
        return resourceSpecificFieldsToClient(client, serverObject);
    }

    protected ClientType genericFieldsToClient(ClientType client, ResourceType serverObject) {
        final DateFormat dateTimeFormatter = getDateTimeFormat();
        final Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        dateTimeFormatter.setTimeZone(TimeZoneContextHolder.getTimeZone());
        if (serverObject.getCreationDate() != null) {
            client.setCreationDate(dateTimeFormatter.format(serverObject.getCreationDate()));
        }
        client.setDescription(serverObject.getDescription());
        client.setLabel(serverObject.getLabel());
        client.setResourceType(getClientResourceType());
        if (serverObject.getUpdateDate() != null) {
            client.setUpdateDate(dateTimeFormatter.format(serverObject.getUpdateDate()));
        }
        client.setUri(serverObject.getURIString());
        client.setVersion(serverObject.getVersion());
        client.setPermissionMask(permissionsService.getEffectivePermission(serverObject, authentication).getPermissionMask());
        return client;
    }

    protected abstract ClientType resourceSpecificFieldsToClient(ClientType client, ResourceType serverObject);

    protected ClientType getNewClientObjectInstance() {
        try {
            return getClientTypeClass().newInstance();
        } catch (Exception e) {
            throw new IllegalStateException("Couldn't instantiate client object", e);
        }
    }

    // Client object class is extracted from real implementation class by reflection. So, cast is safe.
    @SuppressWarnings("unchecked")
    protected Class<ClientType> getClientTypeClass() {
        if (clientTypeClass == null) {
            clientTypeClass = (Class) GenericParametersReflectionHelper.getGenericTypeArgument(this.getClass(), ResourceConverter.class, 1);
        }
        return clientTypeClass;
    }

    public String getClientResourceType() {
        if (clientResourceType == null) {
            clientResourceType = GenericParametersReflectionHelper.extractClientType(getClientTypeClass());
        }
        return clientResourceType;
    }
}
