/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.remote.resources.converters;

import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceReference;
import com.jaspersoft.jasperserver.api.metadata.common.service.ResourceFactory;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.dto.resources.ClientFile;
import com.jaspersoft.jasperserver.dto.resources.ClientReference;
import com.jaspersoft.jasperserver.dto.resources.ClientReferenceableFile;
import com.jaspersoft.jasperserver.dto.resources.ClientReferenceableInputControl;
import com.jaspersoft.jasperserver.dto.resources.ClientReferenceableQuery;
import com.jaspersoft.jasperserver.dto.resources.ClientReportUnit;
import com.jaspersoft.jasperserver.remote.exception.IllegalParameterValueException;
import com.jaspersoft.jasperserver.remote.exception.MandatoryParameterNotFoundException;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * <p></p>
 *
 * @author Yaroslav.Kovalchyk
 * @version $Id: ReportUnitResourceConverter.java 29531 2013-03-12 09:50:30Z ykovalchyk $
 */
@Service
public class ReportUnitResourceConverter extends DataSourceHolderResourceConverter<ReportUnit, ClientReportUnit> {
    private static final String EXTRACT_RESOURCE_NAME_FROM_URI_PATTERN = ".+/([^/]+)$";
    @Resource
    private FileResourceConverter fileResourceConverter;
    @Resource(name = "mappingResourceFactory")
    private ResourceFactory objectFactory;

    @Override
    protected ReportUnit resourceSpecificFieldsToServer(ClientReportUnit clientObject, ReportUnit resultToUpdate) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
        resultToUpdate.setAlwaysPromptControls(clientObject.isAlwaysPromptControls());
        final byte controlsLayout;
        switch (clientObject.getControlsLayout()) {
            case popupScreen:
                controlsLayout = ReportUnit.LAYOUT_POPUP_SCREEN;
                break;
            case separatePage:
                controlsLayout = ReportUnit.LAYOUT_SEPARATE_PAGE;
                break;
            case inPage:
                controlsLayout = ReportUnit.LAYOUT_IN_PAGE;
                break;
            case topOfPage:
                controlsLayout = ReportUnit.LAYOUT_TOP_OF_PAGE;
                break;
            default:
                controlsLayout = ReportUnit.LAYOUT_POPUP_SCREEN;
        }
        resultToUpdate.setControlsLayout(controlsLayout);
        resultToUpdate.setDataSnapshotId(clientObject.getDataSnapshotId());
        resultToUpdate.setInputControlRenderingView(clientObject.getInputControlRenderingView());
        resultToUpdate.setReportRenderingView(clientObject.getReportRenderingView());
        resultToUpdate.setQuery(resourceReferenceConverterProvider.getConverterForType(ClientReferenceableQuery.class)
                .toServer(clientObject.getQuery(), resultToUpdate.getQuery()));
        resultToUpdate.setMainReport(resourceReferenceConverterProvider.getConverterForType(ClientReferenceableFile.class)
                .addReferenceRestriction(new ResourceReferenceConverter.FileTypeRestriction(ClientFile.FileType.jrxml))
                .toServer(clientObject.getJrxml(), resultToUpdate.getMainReport()));
        List<ResourceReference> inputControls = null;
        final List<ClientReferenceableInputControl> clientInputControls = clientObject.getInputControls();
        if (clientInputControls != null && !clientInputControls.isEmpty()) {
            inputControls = new ArrayList<ResourceReference>(clientInputControls.size());
            final ResourceReferenceConverter<ClientReferenceableInputControl> inputControlReferenceConverter =
                    resourceReferenceConverterProvider.getConverterForType(ClientReferenceableInputControl.class);
            for (ClientReferenceableInputControl inputControlReference : clientInputControls) {
                inputControls.add(inputControlReferenceConverter.toServer(inputControlReference));
            }
        }
        resultToUpdate.setInputControls(inputControls);
        // Raw List is used in core classes, but it is of type List<ResourceReference>. So, cast is safe.
        @SuppressWarnings("unchecked")
        final List<ResourceReference> resources = resultToUpdate.getResources();
        resultToUpdate.setResources(convertResourcesToServer(clientObject.getFiles(), resources));
        return resultToUpdate;
    }

    protected List<ResourceReference> convertResourcesToServer(Map<String, ClientReferenceableFile> clientResources,
            List<ResourceReference> serverReferencesToUpdate) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
        List<ResourceReference> result = new ArrayList<ResourceReference>();
        if (clientResources != null && !clientResources.isEmpty()) {
            Map<String, ResourceReference> serverResourcesMap = getServerResourcesAsMap(serverReferencesToUpdate);
            for (String currentResourceName : clientResources.keySet()) {
                final ClientReferenceableFile clientObject = clientResources.get(currentResourceName);
                ResourceReference serverObjectToUpdate = serverResourcesMap.get(currentResourceName);
                if (serverObjectToUpdate != null && serverObjectToUpdate.isLocal()
                        && !(serverObjectToUpdate.getLocalResource() instanceof FileResource)) {
                    // this case is invalid. Local resources should be of type FileResource only.
                    // but if wrong type comes, then let's proceed like no server object to update.
                    serverObjectToUpdate = null;
                }
                final ResourceReferenceConverter<ClientReferenceableFile> fileResourceReferenceConverter =
                        resourceReferenceConverterProvider.getConverterForType(ClientReferenceableFile.class);
                if (clientObject instanceof ClientReference &&
                        (serverObjectToUpdate == null || serverObjectToUpdate.isLocal())) {
                    // Server side FileResource can be a reference.
                    // It is done for ReportUnit's resources referencing in JRXML.
                    // Currently this is the only known case, where FileResource can be a reference.
                    // So, handle this tricky case here.
                    if (serverObjectToUpdate == null) {
                        // new local FileResource-reference should be created
                        // ensure, that reference points to a valid file resource
                        fileResourceReferenceConverter.toServer(clientObject);
                        FileResource fileResourceReference = (FileResource) objectFactory.newResource(null, FileResource.class);
                        fileResourceReference.setReferenceURI(clientObject.getUri());
                        serverObjectToUpdate = new ResourceReference(fileResourceReference);
                    } else if (!clientObject.getUri().equals(((FileResource) serverObjectToUpdate.getLocalResource()).getReferenceURI())) {
                        // reference URI is changed. Let's validate reference target
                        // ensure, that reference points to a valid file resource
                        fileResourceReferenceConverter.toServer(clientObject);
                        // update reference if conversion above succeed
                        ((FileResource) serverObjectToUpdate.getLocalResource()).setReferenceURI(clientObject.getUri());
                    }
                } else {
                    // normal file reference case.
                    serverObjectToUpdate = fileResourceReferenceConverter.toServer(clientObject, serverObjectToUpdate);
                }
                result.add(serverObjectToUpdate);
            }
            if (serverReferencesToUpdate == null) {
                // no existing resources, let's create new ArrayList with resources
                serverReferencesToUpdate = new ArrayList<ResourceReference>(result);
            } else {
                // there was a list with resources.
                // remove all previously existing resources
                serverReferencesToUpdate.clear();
                // add all updated/created resources
                serverReferencesToUpdate.addAll(result);
            }
        } else if (serverReferencesToUpdate != null) {
            // here is no resources in incoming ClientReportUnit.
            // So, let's remove all existing resources.
            serverReferencesToUpdate.clear();
        }
        return serverReferencesToUpdate;
    }

    protected Map<String, ResourceReference> getServerResourcesAsMap(List<ResourceReference> list) {
        final HashMap<String, ResourceReference> result = new HashMap<String, ResourceReference>();
        if (list != null) {
            for (ResourceReference reference : list) {
                final String name = reference.isLocal() ? reference.getLocalResource().getName() :
                        getResourceNameForReference(reference);
                result.put(name, reference);
            }
        }
        return result;
    }

    protected String getResourceNameForReference(ResourceReference reference) {
        String name;
        final String referenceURI = reference.getReferenceURI();
        Matcher matcher = Pattern.compile(EXTRACT_RESOURCE_NAME_FROM_URI_PATTERN).matcher(referenceURI);
        if (matcher.find()) {
            name = matcher.group(1);
        } else {
            throw new IllegalStateException("Invalid reference URI '" + referenceURI + "'. Resource name can't be extracted");
        }
        return name;
    }

    protected Map<String, ClientReferenceableFile> convertResourcesToClient(List<ResourceReference> serverResources) {
        Map<String, ClientReferenceableFile> result = null;
        if (serverResources != null && !serverResources.isEmpty()) {
            ResourceReferenceConverter<ClientReferenceableFile> fileResourceReferenceConverter =
                    resourceReferenceConverterProvider.getConverterForType(ClientReferenceableFile.class);
            result = new HashMap<String, ClientReferenceableFile>(serverResources.size());
            for (ResourceReference reference : serverResources) {
                if (reference.isLocal()) {
                    // partially process file references here because local file can be reference too.
                    // I hope this is the only case, when FileResource can be a reference ;)
                    if (reference.getLocalResource() instanceof FileResource) {
                        FileResource file = (FileResource) reference.getLocalResource();
                        if (file.isReference()) {
                            result.put(file.getName(),
                                    fileResourceReferenceConverter.toClient(new ResourceReference(file.getReferenceURI())));
                        } else {
                            result.put(file.getName(), fileResourceConverter.toClient(file));
                        }
                    } else {
                        throw new IllegalStateException("ReportUnit can't contain references of type " +
                                reference.getLocalResource());
                    }
                } else {
                    result.put(getResourceNameForReference(reference), fileResourceReferenceConverter.toClient(reference));
                }
            }
        }
        return result;
    }

    @Override
    protected void setDataSourceToResource(ResourceReference dataSourceReference, ReportUnit resource) {
        resource.setDataSource(dataSourceReference);
    }

    @Override
    protected ResourceReference getDataSourceFromResource(ReportUnit resource) {
        return resource.getDataSource();
    }

    @Override
    protected ClientReportUnit resourceSpecificFieldsToClient(ClientReportUnit client, ReportUnit serverObject) {
        client.setAlwaysPromptControls(serverObject.isAlwaysPromptControls());
        final ClientReportUnit.ControlsLayoutType controlsLayout;
        switch (serverObject.getControlsLayout()) {
            case ReportUnit.LAYOUT_POPUP_SCREEN:
                controlsLayout = ClientReportUnit.ControlsLayoutType.popupScreen;
                break;
            case ReportUnit.LAYOUT_SEPARATE_PAGE:
                controlsLayout = ClientReportUnit.ControlsLayoutType.separatePage;
                break;
            case ReportUnit.LAYOUT_IN_PAGE:
                controlsLayout = ClientReportUnit.ControlsLayoutType.inPage;
                break;
            case ReportUnit.LAYOUT_TOP_OF_PAGE:
                controlsLayout = ClientReportUnit.ControlsLayoutType.topOfPage;
                break;
            default:
                controlsLayout = ClientReportUnit.ControlsLayoutType.popupScreen;
        }
        client.setControlsLayout(controlsLayout);
        client.setDataSnapshotId(serverObject.getDataSnapshotId());
        client.setInputControlRenderingView(serverObject.getInputControlRenderingView());
        client.setReportRenderingView(serverObject.getReportRenderingView());
        client.setQuery(resourceReferenceConverterProvider.getConverterForType(ClientReferenceableQuery.class)
                .toClient(serverObject.getQuery()));
        // don't need to add restriction for JRXML type of file.
        // In case of toClient conversion restrictions are not called.
        client.setJrxml(resourceReferenceConverterProvider.getConverterForType(ClientReferenceableFile.class)
                .toClient(serverObject.getMainReport()));
        List<ClientReferenceableInputControl> inputControls = null;
        final List<ResourceReference> serverInputControls = serverObject.getInputControls();
        if (serverInputControls != null && !serverInputControls.isEmpty()) {
            inputControls = new ArrayList<ClientReferenceableInputControl>(serverInputControls.size());
            final ResourceReferenceConverter<ClientReferenceableInputControl> inputControlResourceReferenceConverter =
                    resourceReferenceConverterProvider.getConverterForType(ClientReferenceableInputControl.class);
            for (ResourceReference inputControlReference : serverInputControls) {
                inputControls.add(inputControlResourceReferenceConverter.toClient(inputControlReference));
            }
        }
        client.setInputControls(inputControls);
        // core module uses raw List, but there is List<ResourceReference>. So, cast is safe.
        @SuppressWarnings("unchecked")
        final List<ResourceReference> resources = serverObject.getResources();
        client.setFiles(convertResourcesToClient(resources));
        return client;
    }
}
