/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.war.cascade.handlers.validators;

import com.jaspersoft.jasperserver.api.JSException;
import com.jaspersoft.jasperserver.api.metadata.common.domain.DataType;
import com.jaspersoft.jasperserver.war.cascade.InputControlValidationException;
import com.jaspersoft.jasperserver.war.cascade.handlers.converters.DataConverterService;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Locale;

/**
 * @author Yaroslav.Kovalchyk
 * @version $Id: ComparableValidator.java 24150 2012-06-13 13:03:12Z ykovalchyk $
 */
@Service
public class ComparableValidator<T extends Comparable<?>> implements InputControlValueValidator<T> {

    @Resource
    private DataConverterService dataConverterService;
    @Resource
    protected MessageSource messageSource;

    @Override
    public void validateSingleValue(T value, DataType dataType) throws InputControlValidationException {
        if (value != null && dataType != null) {
            final Locale locale = LocaleContextHolder.getLocale();
            if (dataType.getMinValue() != null) {
                @SuppressWarnings("unchecked") // concrete type of Number implements Comparable<ConcreteNumberType>. Therefore cast is safe for all standard Number types
                        Comparable<T> min = convertRestrictionValueToParameterType(value.getClass(), dataType.getMinValue(), dataType);
                if (dataType.isStrictMin()) {
                    if (min.compareTo(value) >= 0) {
                        throw new InputControlValidationException("fillParameters.error.smallerThan", null, messageSource.getMessage("fillParameters.error.smallerThan", null, locale), null);
                    }
                } else if (min.compareTo(value) > 0) {
                    throw new InputControlValidationException("fillParameters.error.smallerOrEqual", null, messageSource.getMessage("fillParameters.error.smallerOrEqual", null, locale), null);
                }
            }

            if (dataType.getMaxValue() != null) {
                @SuppressWarnings("unchecked") // concrete type of Number implements Comparable<ConcreteNumberType>. Therefore cast is safe for all standard Number types
                        Comparable<T> max = convertRestrictionValueToParameterType(value.getClass(), dataType.getMaxValue(), dataType);
                if (dataType.isStrictMax()) {
                    if (max.compareTo(value) <= 0) {
                        throw new InputControlValidationException("fillParameters.error.greaterThan", null, messageSource.getMessage("fillParameters.error.greaterThan", null, locale), null);
                    }
                } else if (max.compareTo(value) < 0) {
                    throw new InputControlValidationException("fillParameters.error.greaterOrEqual", null, messageSource.getMessage("fillParameters.error.greaterOrEqual", null, locale), null);
                }
            }
        }
    }

    @SuppressWarnings("unchecked")// cast safety guarantied by dataConverterService
    protected Comparable<T> convertRestrictionValueToParameterType(Class<?> type, Object restrictionValue, DataType dataType) {
        Comparable<T> result;
        final Class<?> restrictionValueClass = restrictionValue.getClass();
        if (restrictionValueClass.isAssignableFrom(type)) {
            result = (Comparable<T>) restrictionValue;
        } else if (restrictionValue instanceof String) {
            try {
                result = (Comparable<T>) dataConverterService.convertSingleValue((String) restrictionValue, dataType, type);
            } catch (Exception e) {
                throw new IllegalStateException("Value '" + restrictionValue + "' couldn't be used as restriction for input control of type " + type.getName());
            }
        } else
            throw new JSException("Restriction type " + restrictionValueClass.getName() + " incompatible with value type " + type.getName());
        return result;
    }
}
