/*
 * Copyright (C) 2005 - 2012 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public  License
 *  along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.api.security.externalAuth;

import com.jaspersoft.jasperserver.api.JSException;
import com.jaspersoft.jasperserver.api.metadata.user.domain.impl.client.MetadataUserDetails;
import com.jaspersoft.jasperserver.api.security.externalAuth.processors.ExternalUserProcessor;
import com.jaspersoft.jasperserver.api.security.externalAuth.processors.ProcessorData;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.security.Authentication;
import org.springframework.security.context.SecurityContextHolder;
import org.springframework.security.providers.anonymous.AnonymousAuthenticationToken;
import org.springframework.security.userdetails.UserDetails;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import static com.jaspersoft.jasperserver.api.security.externalAuth.processors.ProcessorData.Key.*;

/**
 *
 *     
 * @author dlitvak
 *
 */
public class ExternalDataSynchronizerImpl implements ExternalDataSynchronizer, InitializingBean {

	private static final Logger logger = LogManager.getLogger(ExternalDataSynchronizerImpl.class);


	private ExternalUserDetailsService externalUserDetailsService = new EmptyExternalUserDetailsService();
    private List<ExternalUserProcessor> externalUserProcessors;

    public void afterPropertiesSet() throws Exception {
        Assert.notNull(externalUserProcessors, "externalUserProcessors must not be null");
        Assert.notEmpty(externalUserProcessors, "externalUserProcessors must not be empty: at least external user setup processor must be present to enter th user into JRS DB.");
    }

	@Override
	@Transactional(propagation = Propagation.REQUIRED)
    public void synchronize() {
		Authentication auth = SecurityContextHolder.getContext().getAuthentication();
		logger.debug("Authentication token: " + (auth == null ? "none" : auth));

		if (auth != null && auth.getPrincipal() != null &&
				auth.isAuthenticated() &&
				!(auth instanceof AnonymousAuthenticationToken)) {

			//make sure this thread has fresh ProcessorData each time it invokes synchronize()
			ProcessorData.getInstance().clearData();

			loadExternalUserDetailsToProcessorData(auth);
			for  (ExternalUserProcessor processor : externalUserProcessors)
				processor.process();

			Authentication newAuth = SecurityContextHolder.getContext().getAuthentication();
			// The authentication can be null if there are no roles. This sets the anonymous user as the
			// logged-in user, if the anonymousUserFilter is in the chain
			if (newAuth != null && newAuth.getPrincipal() instanceof MetadataUserDetails) {
				MetadataUserDetails newPrincipal = (MetadataUserDetails) newAuth.getPrincipal();

				// Keep a hold of the original principal: it may be useful later
				newPrincipal.setOriginalAuthentication(auth);
			}
		}
    }

	/**
	 * This method merges external user data from Authentication and ExternalUserDetailsService and transfers them
	 * to ProcessorData.  ExternalUserProcessor's process ProcessorData into internal database entries.
	 *
	 * @param auth
	 */
	protected void loadExternalUserDetailsToProcessorData(final Authentication auth) {
		final Object principal = auth.getPrincipal();
		if (principal == null)
			throw new JSException("Principal is null.");

		final String username = principal instanceof UserDetails ? ((UserDetails)principal).getUsername() : principal.toString();
		final List<Map<String, Object>> userDetails = externalUserDetailsService.loadDetails(username);

		final ProcessorData processorData = ProcessorData.getInstance();

		if (userDetails != null && !userDetails.isEmpty())
			processorData.addData(EXTERNAL_LOADED_DETAILS, userDetails.get(0));

		processorData.addData(EXTERNAL_AUTH_DETAILS, auth.getPrincipal());
		processorData.addData(EXTERNAL_AUTHORITIES, auth.getAuthorities());
	}


	protected ExternalUserDetailsService getExternalUserDetailsService() {
		return externalUserDetailsService;
	}

	public void setExternalUserDetailsService(ExternalUserDetailsService externalUserDetailsService) {
		Assert.notNull(externalUserDetailsService, "Cannot set user details service to null.");
		this.externalUserDetailsService = externalUserDetailsService;
	}

	public void setExternalUserProcessors(List<ExternalUserProcessor> externalUserProcessors) {
		Assert.notNull(externalUserProcessors, "Cannot set empty processor list.");
		this.externalUserProcessors = externalUserProcessors;
	}

	protected List<ExternalUserProcessor> getExternalUserProcessors() {
		return externalUserProcessors;
	}
}
