/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.jaxrs.resources;

import com.jaspersoft.jasperserver.api.metadata.common.domain.ContentResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.dto.resources.ClientFile;
import com.jaspersoft.jasperserver.dto.resources.ClientResourceListWrapper;
import com.jaspersoft.jasperserver.dto.resources.ClientResourceLookup;
import com.jaspersoft.jasperserver.jaxrs.common.RestConstants;
import com.jaspersoft.jasperserver.remote.exception.IllegalParameterValueException;
import com.jaspersoft.jasperserver.remote.exception.RemoteException;
import com.jaspersoft.jasperserver.remote.exception.ResourceNotFoundException;
import com.jaspersoft.jasperserver.remote.resources.GenericParametersReflectionHelper;
import com.jaspersoft.jasperserver.remote.resources.converters.LookupResourceConverter;
import com.jaspersoft.jasperserver.remote.resources.converters.ResourceConverterProvider;
import com.jaspersoft.jasperserver.remote.services.BatchRepositoryService;
import com.jaspersoft.jasperserver.remote.services.SingleRepositoryService;
import com.jaspersoft.jasperserver.search.common.ResourceDetails;
import com.jaspersoft.jasperserver.search.mode.SearchMode;
import com.jaspersoft.jasperserver.search.service.RepositorySearchCriteria;
import com.jaspersoft.jasperserver.search.service.RepositorySearchService;
import com.jaspersoft.jasperserver.search.service.impl.RepositorySearchCriteriaImpl;
import com.jaspersoft.jasperserver.war.common.JasperServerHttpConstants;
import com.sun.jersey.core.spi.factory.ResponseBuilderImpl;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.HeaderParam;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.ArrayList;
import java.util.List;

/**
 * <p></p>
 *
 * @author Yaroslav.Kovalchyk
 * @version $Id: SearchResourcesJaxrsService.java 29699 2013-03-14 14:23:46Z ztomchenco $
 */
@Service
@Path("/resources")
public class SearchResourcesJaxrsService {
    public static final String FOLDER_URI = "folderUri";
    @Resource(name = "concreteRepository")
    public RepositoryService repositoryService;
    @Resource
    private RepositorySearchService repositorySearchService;
    @Resource
    private BatchRepositoryService batchRepositoryService;
    @Resource
    private SingleRepositoryService singleRepositoryService;
    @Resource
    private LookupResourceConverter lookupResourceConverter;
    @Resource
    private ResourceConverterProvider resourceConverterProvider;

    @GET
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    public Response getResources(
            @QueryParam(RestConstants.QUERY_PARAM_SEARCH_QUERY) String q,
            @QueryParam(FOLDER_URI) String folderUri,
            @QueryParam("type") String type,
            @QueryParam(RestConstants.QUERY_PARAM_OFFSET) Integer start,
            @QueryParam(RestConstants.QUERY_PARAM_LIMIT) Integer limit,
            @QueryParam("recursive") @DefaultValue("true") Boolean recursive,
            @QueryParam("showHiddenItems") @DefaultValue("false") Boolean showHiddenItems,
            @QueryParam("forceTotalCount") @DefaultValue("false") Boolean forceTotalCount,
            @QueryParam(RestConstants.QUERY_PARAM_SORT_BY) String sortBy) throws IllegalParameterValueException, ResourceNotFoundException {
        if(folderUri != null){
            validateFolderUri(folderUri);
        }
        final RepositorySearchCriteriaImpl.Builder builder = new RepositorySearchCriteriaImpl.Builder()
                .setSearchText(q)
                .setFolderUri(folderUri != null ? folderUri : Folder.SEPARATOR)
                .setStartIndex(start != null ? start : 0)
                .setMaxCount(limit != null ? limit : 100)
                .setSearchMode(recursive == null || recursive ? SearchMode.SEARCH : SearchMode.BROWSE)
                .setShowHidden(showHiddenItems)
                .setSortBy(sortBy);
        boolean isSearchSuitable = true;
        if (type != null) {
            if (GenericParametersReflectionHelper.extractClientType(ClientResourceLookup.class).equals(type)) {
                // ResourceLookup is a valid repository resource, but without persistence. So, search for resource lookups isn't suitable.
                isSearchSuitable = false;
            } else if (GenericParametersReflectionHelper.extractClientType(ClientFile.class).equals(type)) {
                // here is an exception from a common rule: one client object corresponds to te only server object and back
                // Single ClientFile client object corresponds to either FileResource or ContentResource on server side.
                // Do this trick here
                builder.setResourceTypes(FileResource.class.getName(), ContentResource.class.getName());
            } else {
                try {
                    builder.setResourceTypes(resourceConverterProvider.getToServerConverter(type).getServerResourceType());
                } catch (IllegalParameterValueException e) {
                    // don't perform search for unsupported resource type
                    isSearchSuitable = false;
                }
            }

        }


        final RepositorySearchCriteria criteria = builder.getCriteria();
        // call search if it is suitable
        final List<ResourceDetails> results = isSearchSuitable ? repositorySearchService.getResults(null, criteria) : null;

        List<ClientResourceLookup> clientResult = null;
        if (results != null && !results.isEmpty()) {
            clientResult = new ArrayList<ClientResourceLookup>(results.size());
            for (ResourceDetails currentItem : results) {
                ClientResourceLookup clientItem = lookupResourceConverter.toClient(currentItem);
                clientResult.add(clientItem);
            }
        }

        Response.ResponseBuilder response = new ResponseBuilderImpl();
        if (clientResult != null && clientResult.size() != 0) {
            response.status(Response.Status.OK).entity(new ClientResourceListWrapper(clientResult));
            response.header(RestConstants.HEADER_START_INDEX, criteria.getStartIndex())
                    .header(RestConstants.HEADER_RESULT_COUNT, results.size());
            if (criteria.getStartIndex() == 0 && results.size() < criteria.getMaxCount()) {
                response.header(RestConstants.HEADER_TOTAL_COUNT, results.size());
            } else if ((criteria.getStartIndex() == 0 && results.size() >= criteria.getMaxCount()) ||
                    (forceTotalCount != null && forceTotalCount)) {
                response.header(RestConstants.HEADER_TOTAL_COUNT, repositorySearchService.getResultsCount(null, criteria));
            }
        } else {
            response.status(Response.Status.NO_CONTENT);
        }
        return response.build();
    }

    /**
     * This method validates if folder URI is correct and exists
     *
     * @param folderUri - folder URI to check
     * @throws IllegalParameterValueException - in case if URI has invalid format
     * @throws ResourceNotFoundException - in case if folder with given URI doesn't exist
     */
    protected void validateFolderUri(String folderUri) throws IllegalParameterValueException, ResourceNotFoundException {
        if(!folderUri.startsWith(Folder.SEPARATOR)){
            throw new IllegalParameterValueException(FOLDER_URI, folderUri);
        } else if(repositoryService.getFolder(null, folderUri) == null){
            throw new ResourceNotFoundException(folderUri);
        }
    }

    @DELETE
    public Response deleteResources(@QueryParam("resourceUri") List<String> uris) throws RemoteException {
        if (uris == null) {
            throw new IllegalParameterValueException("resourceUri", "null");
        }
        batchRepositoryService.deleteResources(uris);
        return Response.noContent().build();
    }

    @POST
    public Response copyResource(
            @HeaderParam(HttpHeaders.CONTENT_LOCATION) String sourceUri,
            @QueryParam("createFolders") @DefaultValue("true") Boolean createFolders,
            @QueryParam("overwrite") @DefaultValue("false") Boolean overwrite) throws RemoteException {
        singleRepositoryService.copyResource(sourceUri, Folder.SEPARATOR, createFolders, overwrite);
        return Response.noContent().build();
    }

    @PUT
    public Response moveResource(
            @HeaderParam(HttpHeaders.CONTENT_LOCATION) String sourceUri,
            @QueryParam("createFolders") @DefaultValue("true") Boolean createFolders,
            @QueryParam("overwrite") @DefaultValue("false") Boolean overwrite) throws RemoteException {
        singleRepositoryService.moveResource(sourceUri, Folder.SEPARATOR, createFolders, overwrite);
        return Response.noContent().build();
    }

}
