package com.jaspersoft.jasperserver.api.metadata.common.util;

import com.jaspersoft.jasperserver.api.JSException;
import org.apache.commons.lang.StringEscapeUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * <p>Resolves JNDI names to JDBC properties.</p>
 *
 * @author Bob
 * @author Yuriy Plakosh
 */
public class JndiFallbackResolver {
    public static final String JDBC_DRIVER_CLASS_NAME = "driverClassName";
    public static final String JDBC_URL = "url";
    public static final String JDBC_USERNAME = "username";
    public static final String JDBC_PASSWORD = "password";

    /**
     * Properties which contain mapping between JNDI name and JDBC properties.
     */
    private Properties jndiFallbackProperties;

    /**
     * Returns the map of JDBC properties which were resolved from fallback properties file using specified JNDI name.
     *
     * @param jndiName the JNDI name.
     *
     * @return the map of JDBC properties.
     */
    public Map<String, String> getJdbcPropertiesMap(String jndiName) {
        if (jndiFallbackProperties == null) {
            throw new JSException("No JNDI fallback properties available.");
        }

        // look for prop matching jndi name
        String propPrefix = null;
        for (Map.Entry<Object, Object> entry : jndiFallbackProperties.entrySet()) {
            String key = (String) entry.getKey();
            if (key.endsWith(".jndi") && entry.getValue().equals(jndiName)) {
                propPrefix = key.replace(".jndi", "");
                break;
            }
        }
        if (propPrefix == null) {
            throw new JSException("No matching JNDI property found for '" + jndiName + "' in fallback properties.");
        }

        Map<String, String> jdbcPropertiesMap = new HashMap<String, String>(4);

        // Pull everything out of props and put it to the map.
        jdbcPropertiesMap.put(JDBC_DRIVER_CLASS_NAME, getFallbackProperty(propPrefix, JDBC_DRIVER_CLASS_NAME));
        jdbcPropertiesMap.put(JDBC_URL, StringEscapeUtils.unescapeXml(getFallbackProperty(propPrefix, JDBC_URL)));
        jdbcPropertiesMap.put(JDBC_USERNAME, getFallbackProperty(propPrefix, JDBC_USERNAME));
        jdbcPropertiesMap.put(JDBC_PASSWORD, getFallbackProperty(propPrefix, JDBC_PASSWORD));

        return jdbcPropertiesMap;
    }

    private String getFallbackProperty(String prefix, String suffix) {
        String value = jndiFallbackProperties.getProperty(prefix + ".jdbc." + suffix);
        if (value == null) {
            throw new JSException("No value for '" + suffix + "' in fallback properties.");
        }
        return value;
    }

    public void setJndiFallbackProperties(Properties jndiFallbackProperties) {
        this.jndiFallbackProperties = jndiFallbackProperties;
    }
}
