////////////////////////////
// Message Dialog
////////////////////////////

/**
 * Shows a message dialog.
 * @param {String} message text message
 * @param {String} header optional text header
 * @param {Function} closeHandler optional function that gets called when dialog closes
 *
 */
function showMessageDialog(message, header, closeHandler, dialogWidth) {
    var dialogDiv = document.getElementById("messageDialog");

    if (dialogWidth) {
        dialogDiv.style.width = dialogWidth;
    }

    var msgCell = document.getElementById("messageDialogBodyCell");
    msgCell.innerHTML = message;

    var hdrCell = document.getElementById("messageDialogHeaderCell");
    if (header != null) {
        hdrCell.innerHTML = header;
    } else {
        hdrCell.innerHTML = "<br>";
    }

    var holder = {};

	//fallback to maintable (should always be there) if we don't find outer table
	//TODO: fix this 
	var backgroundObj =  document.getElementById('outerTable');
	backgroundObj = backgroundObj ? backgroundObj : document.getElementById('mainTable');

    holder.hazeLayer = renderHazeLayer(0,getAbsoluteOffsets(backgroundObj)[1]);
    holder.docOnKey = document.onkeypress;

    var okOnClick = function(e) { hideMessageDialog(holder); if (closeHandler) { closeHandler(e); } };
    var okBtn = document.getElementById("messageDialogOkButton");
    okBtn.onclick = okOnClick;

    var keyPress = function(e) {
        return messageDialogOnKeyPress(
            e,
            function(e) { hideMessageDialog(holder); if (closeHandler) { closeHandler(e); } }
        )
    };
    document.onkeypress = keyPress;

    dialogDiv.style.display="block";
    centerLayer(dialogDiv);
}

function hideMessageDialog(holder) {
  var dialogDiv = document.getElementById("messageDialog")
  dialogDiv.style.display="none";
  if (holder.hazeLayer != null) {
      holder.hazeLayer.parentNode.removeChild(holder.hazeLayer);
  }
  document.onkeypress = holder.docOnKey;
}

function messageDialogOnKeyPress(evt, handler) {
    var charCode = whichKeyHit(evt);
    //escape
    if (charCode == Keys.DOM_VK_ESCAPE) {
        if (handler) {
            handler(evt);
        }
    }
    return false;
}

///////////////////////////
// Confirm Dialog
///////////////////////////

/**
 * Shows a confirmation dialog.
 * @param {String} message text message
 * @param {String} header optional text header
 * @param {Function} okHandler function that gets called when dialog Ok button is pressed
 * @param {Function} cancelHandler function that gets called when dialog Cancel button or Esc is pressed
 *
 */
function showConfirmDialog(message, header, okHandler, cancelHandler) {
    var dialogDiv = document.getElementById("confirmDialog")

    var msgCell = document.getElementById("confirmDialogBodyCell");
    msgCell.innerHTML = message;

    var hdrCell = document.getElementById("confirmDialogHeaderCell");
    if (header != null) {
        hdrCell.innerHTML = header;
    } else {
        hdrCell.innerHTML = "<br>";
    }

    var holder = {};

    holder.hazeLayer = renderHazeLayer(0,getAbsoluteOffsets(document.getElementById('outerTable'))[1]);
    holder.docOnKey = document.onkeypress;

    var okOnClick = function(e) { hideConfirmDialog(holder); if (okHandler) { okHandler(e); } };
    var cancelOnClick = function(e) { hideConfirmDialog(holder); if (cancelHandler) { cancelHandler(e); } };

    var okBtn = document.getElementById("confirmDialogOkButton");
    okBtn.onclick = okOnClick;
    var cancelBtn = document.getElementById("confirmDialogCancelButton");
    cancelBtn.onclick = cancelOnClick;

    var keyPress = function(e) {
        return messageDialogOnKeyPress(
            e,
            function(e) { hideConfirmDialog(holder); if (cancelHandler) { cancelHandler(e); } }
        )
    };
    document.onkeypress = keyPress;

    dialogDiv.style.display="block";
    centerLayer(dialogDiv);

}

// Hide
function hideConfirmDialog(holder) {
    var dialogDiv = document.getElementById("confirmDialog")
    dialogDiv.style.display="none";
    if (holder.hazeLayer != null) {
        holder.hazeLayer.parentNode.removeChild(holder.hazeLayer);
    }
  document.onkeypress = holder.docOnKey;
}

var ModalDialog = Class.create({
    initialize : function (id, defaultButtonId, escButtonId, buttons) {

        this.id = id;
        this.dialogDiv = $(this.id);
        this.defaultButtonId = defaultButtonId;
        this.escButtonId = escButtonId;

        this.buttons = {};

        this.setButtons(buttons);

        this.holder = null;
    },

    show : function () {
//        if (this.holder != null) {
//            return;
//        }

        this.holder = {};

        var backgroundObj =  document.getElementById('outerTable');
        backgroundObj = backgroundObj ? backgroundObj : document.getElementById('mainTable');

        this.holder.hazeLayer = renderHazeLayer(0,0/*getAbsoluteOffsets(backgroundObj)[1]*/);
        this.holder.docOnKey = document.onkeypress;

        var dialogInstance = this;
        var escHandler = this.buttons[this.escButtonId].getClickHandler();

        var keyPress = function(e) {

            return messageDialogOnKeyPress(e, function(e) {

                dialogInstance.hide();

                if (escHandler) {

                    escHandler(e);
                }
            })
        };

        document.onkeypress = keyPress;

        this.dialogDiv.style.display="block";
        centerLayer(this.dialogDiv);

        try {
            $(this.defaultButtonId).focus();
        } catch(ex) {}
    },

    hide : function () {

        this.dialogDiv.hide();

        if (this.holder) {
            if (this.holder.hazeLayer != null) {

                this.holder.hazeLayer.parentNode.removeChild(this.holder.hazeLayer);
            }

            document.onkeypress = this.holder.docOnKey;
        }

        this.holder = null;
    },

    setButtons : function (buttons) {

        this.buttons = buttons;

        var dialogInstance = this;

        for (var id in buttons) {
            buttons[id].setDialog(dialogInstance);
        }
    }
});

ModalDialog.Button = Class.create({
    initialize : function (id, clickHandler) {
        this.id = id;
        this.clickHandler = clickHandler;
        this.element = $(this.id);
        this.dialog = null;

        var buttonInstance = this;

        Element.observe(this.id, 'click', function(e) {

            var result = true;

            if (clickHandler) {
                result = clickHandler(e);
            }

            if (buttonInstance.dialog && result) {
                buttonInstance.dialog.hide();
            }
        });
    },

    getClickHandler : function () {
        return this.clickHandler;
    },

    setDialog : function (dialog) {
        return this.dialog = dialog;
    }
});

var MsgBox = Class.create(ModalDialog, {

    initialize : function ($super, header, msg, okBtnTitle) {

        var id = 'msgBox';

        this.header = $(id + 'Header');
        this.message = $(id + 'Message');

        this.okBtnId = id + 'OkBtn';

        $(this.okBtnId).value = (okBtnTitle) ? okBtnTitle : 'OK';

        this.header.innerHTML = (header) ? header : "";
        this.message.innerHTML = (msg) ? msg : "";

        $super(id, this.okBtnId, this.okBtnId, this._getButtons());
    },

    _getButtons : function () {

        var viewerInstance = this;
        var buttons = {};

        buttons[this.okBtnId] = new ModalDialog.Button(this.okBtnId, function (e) {

            return viewerInstance.onok();
        });

        return buttons;
    },

    onok : function () {
        return true;
    },

    oncancel : function () {
        return true;
    }
});

var ConfirmMsgBox = Class.create(ModalDialog, {

    initialize : function ($super, id, header, msg, okBtnTitle, cancelBtnTitle) {
        this.headerId = id + 'Header';
        this.messageId = id + 'Message';

        this.okBtnId = id + 'OkBtn';
        this.cancelBtnId = id + 'CancelBtn';

        var cmbContainer = document.createElement("div");
        cmbContainer.id = id;
        cmbContainer.className = "dialogFrame confirmMsgBox";
        cmbContainer.style.position = "absolute";
        cmbContainer.style.display = "none";
        cmbContainer.innerHTML =
                '    <div class="dialogHeader" onmousedown="dialogOnMouseDown(event);">' +
                '        <div id="' + this.headerId + '" class="dialogHeaderTitle"></div>' +
                '    </div>' +
                '    <div class="dialogContent confirmMsgBoxContent" onkeypress="cancelEventBubbling(event);">' +
                '        <div id="' + this.messageId + '" class="confirmMsgBoxMessage"></div>' +
                '        <div style="text-align:right;">' +
                '            <input id="' + this.cancelBtnId + '" type="button" class="dialogButton confirmMsgBoxCancelBtn"/>' +
                '            <input id="' + this.okBtnId + '" type="button" class="dialogButton"/>' +
                '        </div>' +
                '    </div>';
        document.body.appendChild(cmbContainer);

        this.header = $(this.headerId);
        this.message = $(this.messageId);

        $(this.okBtnId).value = (okBtnTitle) ? okBtnTitle : 'OK';
        $(this.cancelBtnId).value = (cancelBtnTitle) ? cancelBtnTitle : 'Cancel';

        this.header.innerHTML = (header) ? header : "";
        this.message.innerHTML = (msg) ? msg : "";

        $super(id, this.okBtnId, this.cancelBtnId, this._getButtons());
    },

    _getButtons : function () {

        var viewerInstance = this;
        var buttons = {};

        buttons[this.cancelBtnId] = new ModalDialog.Button(this.cancelBtnId, function (e) {

            return viewerInstance.oncancel();
        });

        buttons[this.okBtnId] = new ModalDialog.Button(this.okBtnId, function (e) {

            return viewerInstance.onok();
        });

        return buttons;
    },

    onok : function () {
        return true;
    },

    oncancel : function () {
        return true;
    }
});

