/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata.common.service;

import java.util.List;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.ValidationErrorFilter;
import com.jaspersoft.jasperserver.api.common.domain.ValidationErrors;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResourceData;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceLookup;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterCriteria;


/**
 * Top level class for accessing the repository metadata.
 * This class has various methods for retrieving resources from the repository
 * and for uploading resources to the repository.
 * 
 * @author Teodor Danciu (teodord@users.sourceforge.net)
 * @version $Id: RepositoryService.java 15217 2009-03-05 11:38:24Z lucian $
 */
public interface RepositoryService
{

	/**
	 * 
	 * @param context
	 * @param uri
	 * @return the resource if found or null otherwise
	 */
	public Resource getResource(ExecutionContext context, String uri);
	
	public Resource getResource(ExecutionContext context, String uri, Class resourceType);


	/**
	 * 
	 * @param context
	 * @param uri
	 * @return
	 * @throws JSResourceNotFoundException
	 */
	public FileResourceData getResourceData(ExecutionContext context, String uri) throws JSResourceNotFoundException;


	public FileResourceData getContentResourceData(ExecutionContext context, String uri) throws JSResourceNotFoundException;


	public Folder getFolder(ExecutionContext context, String uri);


	public void saveFolder(ExecutionContext context, Folder folder);


	public List getAllFolders(ExecutionContext context);

	/**
	 * Returns a list of all folders under a specific parent folders.
	 * 
	 * <p>
	 * The list returned by this method contains all visible folders
	 * that have the specified folder as ancestor, including the specified
	 * folder itself.
	 * 
	 * @param context the execution context
	 * @param parentURI the URI of the parent folder
	 * @return a list of {@link Folder} instances
	 */
	public List getAllSubfolders(ExecutionContext context, String parentURI);


	public List getSubFolders(ExecutionContext context, String folderURI);
	
	public boolean folderExists(ExecutionContext context, String uri);

	/**
	 * 
	 */
	public ValidationErrors validateResource(ExecutionContext context, Resource resource, ValidationErrorFilter filter);
	
	public ValidationErrors validateFolder(ExecutionContext context, Folder folder, ValidationErrorFilter filter);
	

	/**
	 * Saves a new resource into the repository or updates an existing one.
	 * 
	 * <p>
	 * Calling this is the same as calling
	 * {@link #saveResource(ExecutionContext, Resource, boolean) saveResource(context, resource, false)}.
	 * 
	 * @param context the execution context
	 * @param resource the resource to be saved or updated
	 */
	public void saveResource(ExecutionContext context, Resource resource);
	
	/**
	 * Saves a new resource into the repository or updates an existing one.
	 * 
	 * <p>
	 * This method does what {@link #saveResource(ExecutionContext, Resource)} does,
	 * except that it allows the caller to specify whether the resource creation
	 * date is to be updated in the case when an existing resource is being
	 * updated.
	 * 
	 * @param context the execution context
	 * @param resource the resource to be saved or updated
	 * @param updateCreationDate used to specify that when updating an existing resource,
	 * its creation date needs to be updated as well
	 * 
	 * @see Resource#isNew()
	 * @see Resource#getCreationDate()
	 */
	public void saveResource(ExecutionContext context, Resource resource,
			boolean updateCreationDate);

	public void deleteResource(ExecutionContext context, String uri);

	
	public void deleteFolder(ExecutionContext context, String uri);

	
	public void delete(ExecutionContext context, String resourceURIs[], String folderURIs[]);
	
	
	/**
	 * Given filtering criteria, find relevant Resources.
	 * 
	 * @param context
	 * @param criteria
	 * @return Array of found Resources
	 */
	public ResourceLookup[] findResource(ExecutionContext context, FilterCriteria criteria);
	
	public ResourceLookup[] findResources(ExecutionContext context, FilterCriteria[] criteria);

	public List loadResourcesList(final FilterCriteria filterCriteria);

	public List loadResourcesList(ExecutionContext context, FilterCriteria filterCriteria);

	public List loadResourcesList(ExecutionContext context, FilterCriteria[] filterCriteria);

	public List loadClientResources(FilterCriteria filterCriteria);

	/**
	 * Return a Resource that does not yet contain content.
	 * 
	 * @param context
	 * @param class - class of resource to create
	 * @return Resource
	 */
	public Resource newResource(ExecutionContext context, Class _class);

	public String getChildrenFolderName(String resourceName);

	public boolean resourceExists(ExecutionContext executionContext, String uri);

	public boolean resourceExists(ExecutionContext executionContext, String uri, Class resourceType);

	public boolean resourceExists(ExecutionContext executionContext, FilterCriteria filterCriteria);

	/**
	 * Determines whether a resource or folder having a specified URI exists in the repository.
	 * <p/>
	 * Repository entities are uniquely identified by their URI.  This method can be used to check
	 * whether a URI is already present in the repository.
	 * 
	 * @param executionContext the execution context
	 * @param uri the URI
	 * @return <code>true</code> iff the URI is present in the repository
	 */
	public boolean repositoryPathExists(ExecutionContext executionContext, String uri);
	
	/*
	 * set hidden flag for a given folder, so it won't appear in the results for any repo operation
	 */
	void hideFolder(String uri);
	/*
	 * unset hidden flag for a given folder
	 */
	void unhideFolder(String uri);
	
	void moveFolder(ExecutionContext context, String sourceURI, String destinationFolderURI);
	
	void moveResource(ExecutionContext context, String sourceURI, String destinationFolderURI);
	
	Resource copyResource(ExecutionContext context, String sourceURI, String destinationURI);
	
	Folder copyFolder(ExecutionContext context, String sourceURI, String destinationURI);
	
	/**
	 * Copies several resources into a destination folder.
	 * 
	 * <p>
	 * Names will be automatically assigned for the copied resources.
	 * If possible, the names of the original resources will be preserved.
	 * If any of the names already exists in the destination folder, a new
	 * name will be automatically set for the resource copy.
	 * 
	 * @param context the execution context
	 * @param resources the URIs of the resources to copy
	 * @param destinationFolder the URI of the destination folder
	 */
	void copyResources(ExecutionContext context, 
			String[] resources, String destinationFolder);
}
