/*
 * Copyright (C) 2006 JasperSoft http://www.jaspersoft.com
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed WITHOUT ANY WARRANTY; and without the 
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt 
 * or write to:
 * 
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */

package com.jaspersoft.jasperserver.api.metadata.tenant.service.impl;


import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Collections;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;

import com.jaspersoft.jasperserver.api.JSException;
import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.metadata.common.service.ResourceFactory;
import com.jaspersoft.jasperserver.api.metadata.common.service.impl.HibernateDaoImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.impl.hibernate.util.RepositoryUtils;
import com.jaspersoft.jasperserver.api.metadata.tenant.service.TenantPersistenceResolver;
import com.jaspersoft.jasperserver.api.metadata.user.domain.Tenant;
import com.jaspersoft.jasperserver.api.metadata.user.domain.impl.hibernate.RepoTenant;
import com.jaspersoft.jasperserver.api.metadata.user.service.TenantService;

/**
 * @author achan
 *
 */
public class TenantServiceImpl extends HibernateDaoImpl 
		implements TenantService, TenantPersistenceResolver {

	protected static final Log log = LogFactory.getLog(TenantServiceImpl.class);
	private ResourceFactory objectMappingFactory;
	private ResourceFactory persistentClassFactory;
	private String userOrgIdDelimiter;
	
	public ResourceFactory getPersistentClassFactory() {
		return persistentClassFactory;
	}
	
	public void setPersistentClassFactory(ResourceFactory persistentClassFactory) {
		this.persistentClassFactory = persistentClassFactory;
	}
	
	public ResourceFactory getObjectMappingFactory() {
		return objectMappingFactory;
	}

	public void setObjectMappingFactory(ResourceFactory objectFactory) {
		this.objectMappingFactory = objectFactory;
	}
	
	protected RepoTenant getRepoTenant(String tenantId, boolean required) {
		DetachedCriteria criteria = DetachedCriteria.forClass(persistentTenantClass());
		criteria.add(Restrictions.naturalId().set("tenantId", tenantId));
		List tenantList = getHibernateTemplate().findByCriteria(criteria);
		RepoTenant tenant;		
		if (tenantList.isEmpty()) {
			if (required) {
				throw new JSException("Tenant not found with Tenant ID \"" + tenantId + "\"");//TODO i18n
			}
			
			log.debug("Tenant not found with Tenant ID \"" + tenantId + "\"");
			tenant = null;
		} else {
			tenant = (RepoTenant) tenantList.get(0);
		}
		return tenant;
	}

	protected Class persistentTenantClass() {
		return getPersistentClassFactory().getImplementationClass(Tenant.class);
	}
	
	
	protected RepoTenant getRepoTenantBasedOnTenantUri(ExecutionContext context, String tenantUri) {
		DetachedCriteria criteria = DetachedCriteria.forClass(persistentTenantClass());
		criteria.add(Restrictions.eq("tenantUri", tenantUri));
		List tenantList = getHibernateTemplate().findByCriteria(criteria);
		RepoTenant tenant = null;		
		if (tenantList.isEmpty()) {
			log.debug("Tenant not found with Tenant Name \"" + tenantUri + "\"");
		} else {
			tenant = (RepoTenant) tenantList.get(0);
		}
		return tenant;
	}	

	/* (non-Javadoc)
	 * @see com.jaspersoft.jasperserver.api.metadata.user.service.UserAuthorityService#putUser(com.jaspersoft.jasperserver.api.common.domain.ExecutionContext, com.jaspersoft.jasperserver.api.metadata.user.domain.User)
	 */
	public void putTenant(ExecutionContext context, Tenant aTenant) {
		RepoTenant existingTenant = getRepoTenant(aTenant.getId(), false);
		if (existingTenant == null) {
			existingTenant = (RepoTenant) getPersistentClassFactory().newObject(Tenant.class);
		}
		existingTenant.copyFromClient(aTenant, this);
		getHibernateTemplate().saveOrUpdate(existingTenant);
	}
	
	protected List getRepoSubTenants(ExecutionContext context, String parentTenantId) {
		RepoTenant parent = getRepoTenant(parentTenantId, false);
		if (parent == null) {
			return new ArrayList(0);
		}
		return new ArrayList(parent.getSubTenants());
	}
	
	public List getSubTenantList(ExecutionContext context, Tenant parentTenant) {
		List persistentTenants = getRepoSubTenants(context, parentTenant.getId());
		return toClientTenantList(persistentTenants);
	}
	
	public List getSubTenantList(ExecutionContext context, String parnentTenantId) {
		List persistentTenants = getRepoSubTenants(context, parnentTenantId);
		return toClientTenantList(persistentTenants);
	}
	
	protected List toClientTenantList(List persistentTenants) {
		if (persistentTenants == null) {
			return null;
		}
		
		List tenants = new ArrayList(persistentTenants.size());
		for (Iterator it = persistentTenants.iterator(); it.hasNext();) {
			RepoTenant persistentTenant = (RepoTenant) it.next();
			Tenant tenant = toClientTenant(persistentTenant);
			tenants.add(tenant);
		}
		return tenants;
	}
	
	public Tenant getTenant(ExecutionContext context, String tenantId) {
		RepoTenant rTenant = getRepoTenant(tenantId, false);
		if (rTenant == null) {
			return null;
		}
		Tenant tenant = toClientTenant(rTenant);
		return tenant;
		
	}
	
	public void deleteTenant(ExecutionContext context, String tenantId) {
		RepoTenant tenant = getRepoTenant(tenantId, false);
		if (tenant != null) {
		   getHibernateTemplate().delete(tenant);
		} else {
			if (log.isInfoEnabled()) {
				log.info("Tenant " + tenantId + " not found for deletion");
			}
		}
	}
	
	public void updateTenant(ExecutionContext context, Tenant aTenant) {
		putTenant(context, aTenant);
	}
	
	public List getAllSubTenantList(ExecutionContext context, String parentTenantId) {
		ArrayList allSubTenants = new ArrayList();
		getAllSubTenantListPrivate(parentTenantId, allSubTenants);
		return allSubTenants;
	}
	
	private void getAllSubTenantListPrivate(String parentTenantId, ArrayList allSubTenants) {
		List subTenant = getSubTenantList(null, parentTenantId);
		for (int i=0; i<subTenant.size(); i++) {
			allSubTenants.add(subTenant.get(i));
			getAllSubTenantListPrivate(((Tenant)subTenant.get(i)).getId(), allSubTenants);
		}
	}
	
	public int getNumberOfTenants(ExecutionContext context) {
		List allTenants = getAllSubTenantList(null, ORGANIZATIONS);
		return allTenants.size();
	}
	
	public Tenant getDefaultTenant(ExecutionContext context) {
		List allTenants = getAllSubTenantList(null, ORGANIZATIONS);	
		if (allTenants.size() == 1) {
			return (Tenant)allTenants.get(0);
		} else {
			return null;
		}
	}
	
	
	public Tenant getTenantBasedOnTenantUri(ExecutionContext context, String tenantUri) {
		RepoTenant rTenant = getRepoTenantBasedOnTenantUri(context, tenantUri);
		if (rTenant == null) {
			return null;
		}
		Tenant tenant = toClientTenant(rTenant);
		return tenant;		
	}

	protected Tenant toClientTenant(RepoTenant rTenant) {
		Tenant tenant = (Tenant) objectMappingFactory.newObject(Tenant.class);
		rTenant.copyToClient(tenant);
		return tenant;
	}

	public String getUserOrgIdDelimiter() {
		return userOrgIdDelimiter;
	}

	public void setUserOrgIdDelimiter(String userOrgIdDelimiter) {
		this.userOrgIdDelimiter = userOrgIdDelimiter;
	}

	public Tenant getTenantBasedOnRepositoryUri(ExecutionContext context,
			String uri) {
		RepoTenant repoTenant = null;
		while (uri != null && repoTenant == null) {
			repoTenant = getRepoTenantBasedOnTenantFolderUri(uri);
			uri = RepositoryUtils.getParentPath(uri);
		}
		if (repoTenant == null || TenantService.ORGANIZATIONS.equals(repoTenant.getTenantId())) {
			return null;
		}
		Tenant tenant = toClientTenant(repoTenant);
		return tenant;		
	}
	
	protected RepoTenant getRepoTenantBasedOnTenantFolderUri(String folderUri) {
		DetachedCriteria criteria = DetachedCriteria.forClass(
				persistentTenantClass());
		criteria.add(Restrictions.eq("tenantFolderUri", folderUri));
		List tenantList = getHibernateTemplate().findByCriteria(criteria);
		RepoTenant tenant = null;		
		if (!tenantList.isEmpty()) {
			tenant = (RepoTenant) tenantList.get(0);
		}
		return tenant;
	}

	public boolean isMultiTenantEnvironment(ExecutionContext context) {
        DetachedCriteria criteria = DetachedCriteria.forClass(
                persistentTenantClass());
        criteria.add(Restrictions.ne("tenantUri", "/"));
        List tenantList = getHibernateTemplate().findByCriteria(criteria);
        return tenantList.size() > 1;
	}

	public RepoTenant getPersistentTenant(String tenantId, boolean required) {
		if (tenantId == null || tenantId.length() == 0) {
			return null;
		}
		
		return getRepoTenant(tenantId, required);
	}

    public String getUniqueTenantId(String proposedTenantId) {
        List similarTenantIdList;
        DetachedCriteria criteria = DetachedCriteria.forClass(
                persistentTenantClass());
        criteria.add(Restrictions.like("tenantId", proposedTenantId + "%"));
        List tenantList = getHibernateTemplate().findByCriteria(criteria);
        RepoTenant tenant = null;
        if (tenantList.isEmpty()) {
            similarTenantIdList = Collections.emptyList();
        } else {
            similarTenantIdList = new ArrayList(tenantList.size());
            for (int i = 0; i < tenantList.size(); i++) {
                similarTenantIdList.add(((RepoTenant) tenantList.get(i)).getTenantId());
            }
        }

        if (similarTenantIdList.size() == 0) {
            return proposedTenantId;
        }

        for (int i = 0; i < similarTenantIdList.size(); i++) {
            String orgId = (String) similarTenantIdList.get(i);
            if (orgId.equalsIgnoreCase(proposedTenantId)) {
                break;
            }
            if (i == (similarTenantIdList.size() - 1)) {
                return proposedTenantId;
            }
        }

        int orgIndex = 1;
        boolean ready = false;
        while (!ready) {
            for (int i = 0; i < similarTenantIdList.size(); i++) {
                String orgId = (String) similarTenantIdList.get(i);
                if ((proposedTenantId + orgIndex).equalsIgnoreCase(orgId)) {
                    orgIndex++;
                    break;
                }
                if (i == (similarTenantIdList.size() - 1)) {
                    ready = true;
                    proposedTenantId = proposedTenantId + orgIndex;
                }
            }
        }

        return proposedTenantId;
    }
}
