/*
 * Copyright (C) 2005 - 2008 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata.olap.service.impl;

import com.jaspersoft.jasperserver.api.JSException;
import com.jaspersoft.jasperserver.api.common.util.CharacterEncodingProvider;
import com.jaspersoft.jasperserver.api.common.util.spring.StaticApplicationContext;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResourceData;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.vfs.FileName;
import org.apache.commons.vfs.FileObject;
import org.apache.commons.vfs.FileType;
import org.apache.commons.vfs.provider.AbstractFileObject;
import org.springframework.context.ApplicationContext;

/**
 * Implementation of Apache VFS FileObject to access the JasperServer repository
 *
 * @author swood
 */
public class RepositoryFileObject extends AbstractFileObject implements FileObject {

    private static final Log log = LogFactory.getLog(RepositoryFileObject.class);

    FileResource fileResource = null;
    FileResourceData resourceData = null;
    String encoding = null;

    protected RepositoryFileObject(final RepositoryFileSystem fs, final FileName fileName) {
        super(fileName, fs);
    }

    @Override
    protected FileType doGetType() throws Exception {

        if (fileResource == null) {
            return FileType.IMAGINARY;
        } else {
            return FileType.FILE;
        }
    }

    @Override
    protected String[] doListChildren() throws Exception {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    protected long doGetContentSize() throws Exception {

        if (fileResource == null) {
            throw new JSException("no resource. URI: " + getName());
        }
        return fileResource.getData().length;
    }

    /**
     * Attaches this file object to its file resource.  This method is called
     * before any of the doBlah() or onBlah() methods.  Sub-classes can use
     * this method to perform lazy initialisation.
     */
    protected void doAttach() throws Exception {

        if (fileResource == null) {
            ApplicationContext ctx = StaticApplicationContext.getApplicationContext();

            if (ctx == null) {
                throw new JSException("StaticApplicationContext not configured in Spring");
            }
            RepositoryService repositoryService = (RepositoryService) ctx.getBean("repositoryService");

            CharacterEncodingProvider encodingProvider = (CharacterEncodingProvider) ctx.getBean("encodingProvider");;
            encoding = encodingProvider.getCharacterEncoding();

            String resourceName = getName().getPath();
            log.debug("Loading resource: " + resourceName);

            fileResource = (FileResource) repositoryService.getResource(null, resourceName);

            log.debug("has data: " + ((fileResource == null) ? "null" : "" + fileResource.hasData()));
            log.debug("isReference: " + ((fileResource == null) ? "null" : "" + fileResource.isReference()));

            if (fileResource != null && !fileResource.hasData()) {
                resourceData = repositoryService.getResourceData(null, fileResource.getURIString());
            }
        }
    }

    @Override
    protected InputStream doGetInputStream() throws Exception {

        if (fileResource == null) {
            throw new JSException("no resource. URI: " + getName());
        }

        InputStream data = resourceData != null ? resourceData.getDataStream(): fileResource.getDataStream();

        // TODO: Should this be cached?

        String line;
        StringBuffer fileString = new StringBuffer();
        try {
            BufferedReader in = new BufferedReader(new InputStreamReader(data,
                    encoding));
            while ((line = in.readLine()) != null) {
                fileString.append(line);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return new BufferedInputStream(new ByteArrayInputStream(fileString.toString().getBytes()));
    }

}
