/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.war.action;

import com.jaspersoft.jasperserver.api.JSException;
import com.jaspersoft.jasperserver.api.common.domain.impl.ExecutionContextImpl;
import com.jaspersoft.jasperserver.api.engine.common.service.ReportInputControlInformation;
import com.jaspersoft.jasperserver.api.metadata.common.domain.DataType;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceReference;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.war.dto.RuntimeInputControlWrapper;
import org.apache.commons.collections.set.ListOrderedSet;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;

import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Set;

public class ReportParametersUtils {

    /**
     * This property actually should be taken from spring context like in ReportParametersAction
     * but this logic is called also from non-spring context e.g. Controller and DWR.
     * This is for bug #23204 - [Case #20132, ...] the issue is urgent
     * and I don't have time to make big refactoring for this small change.
     *
     * TODO: I've analyzed parsing of Input Controls values of different data types and noticed that logic is very decentralized
     * TODO: I think we need to create single API for Input Controls values parsing that will handle all data types in one interface
     */
    public static final String staticDatePattern = "yyyyMMddHHmmss";

    public static void setInputControlParameterValue(RuntimeInputControlWrapper wrapper, Object paramValue, RepositoryService repository) {
        wrapper.setErrorMessage(null); // clear any existing error
        Object inputValue = toInputControlValue(wrapper, paramValue, repository);
        wrapper.setValue(inputValue);
    }

    public static Object toInputControlValue(RuntimeInputControlWrapper wrapper, Object paramValue, RepositoryService repository) {
        if (paramValue == null) {
            return null;
        }

        Object value;
        if (wrapper.isMulti()) {
            Set values = toMultiInputControlValue(paramValue);
            value = values;
        } else {
            DataType datatype = getDatatype(wrapper.getInputControl(), repository);
            if (datatype != null) {
                switch (datatype.getType()) {
                case DataType.TYPE_NUMBER:
                    value = toInputControlNumber(paramValue);
                    break;
                case DataType.TYPE_TEXT:
                    value = paramValue.toString();
                    break;
                case DataType.TYPE_DATE:
                    value = toInputControlDate(paramValue, wrapper, new SimpleDateFormat(staticDatePattern));
                    break;
                case DataType.TYPE_DATE_TIME:
                    value = toInputControlDate(paramValue, wrapper, new SimpleDateFormat(staticDatePattern));
                    break;
                default:
                    value = paramValue;
                    break;
                }
            } else {
                value = paramValue;
            }
        }
        return value;
    }

    public static Set toMultiInputControlValue(Object paramValue) {
        Set values = new ListOrderedSet();
        if (paramValue != null) {
            if (paramValue instanceof Collection) {
                values.addAll((Collection) paramValue);
            } else if (paramValue.getClass().isArray()) {
                int length = Array.getLength(paramValue);
                for (int idx = 0; idx < length; ++idx) {
                    Object val = Array.get(paramValue, idx);
                    values.add(val);
                }
            }
        }
        return values;
    }

    public static DataType getDatatype(InputControl control, RepositoryService repository) {
        ResourceReference dataTypeRef = control.getDataType();
        DataType dataType;
        if (dataTypeRef == null) {
            dataType = null;
        } else if (dataTypeRef.isLocal()) {
            dataType = (DataType) dataTypeRef.getLocalResource();
        } else {
            dataType = (DataType) repository.getResource(new ExecutionContextImpl(), dataTypeRef.getReferenceURI());
        }
        return dataType;
    }

    public static Date toInputControlDate(Object paramValue, RuntimeInputControlWrapper wrapper, DateFormat dateFormat) {
        Date inputValue;
        if (paramValue instanceof Date) {
            Date dateValue = (Date) paramValue;
            inputValue = new Date(dateValue.getTime());
        /* What if values come as string */
        } else if (paramValue instanceof String) {
            /* Parse anf set value to wrapper */
            parseDateAndDateTime((String) paramValue, wrapper, dateFormat, null);
            /* Throw exception if value wasn't parsed and left as string */
            if (wrapper.getValue() instanceof String) {
                throw new JSException("exception.report.unrecognized.date.type", new Object[]{paramValue.getClass().getName()});
            } else {
                inputValue = (Date) wrapper.getValue();
            }
        } else {
            throw new JSException("exception.report.unrecognized.date.type", new Object[]{paramValue.getClass().getName()});
        }
        return inputValue;
    }

    public static BigDecimal toInputControlNumber(Object paramValue) {
        BigDecimal inputValue;
        if (paramValue instanceof Byte
                || paramValue instanceof Short
                || paramValue instanceof Integer
                || paramValue instanceof Long) {
            inputValue = BigDecimal.valueOf(((Number) paramValue).longValue());
        } else if (paramValue instanceof Float
                || paramValue instanceof Double) {
            inputValue = BigDecimal.valueOf(((Number) paramValue).doubleValue());
        } else if (paramValue instanceof BigDecimal) {
            inputValue = (BigDecimal) paramValue;
        } else if (paramValue instanceof BigInteger) {
            inputValue = new BigDecimal((BigInteger) paramValue);
        } else if (paramValue instanceof String) {
            if (((String) paramValue).trim().length() == 0) {
                return null;
            }
            inputValue = new BigDecimal((String) paramValue);
        } else {
            throw new JSException("exception.report.unrecognized.number.type", new Object[]{paramValue.getClass().getName()});
        }
        return inputValue;
    }

    /**
     * Parse string value of Date or DateTime and set value to control wrapper
     * @param strValue
     *          String value coming from user or from saved state
     * @param wrapper
     *          RuntimeInputControlWrapper
     * @param format
     *          DateFormat can have short pattern for Date or long pattern for DateTime
     * @param messageSource
     *          Optional message source - sometimes we don't need to set error message
     * @see com.jaspersoft.jasperserver.war.util.DefaultCalendarFormatProvider
     */
    protected static void parseDateAndDateTime(String strValue, RuntimeInputControlWrapper wrapper, DateFormat format, MessageSource messageSource) {
        try {
            ReportInputControlInformation rici = wrapper.getControlInfo();
            Class paramClass = (rici == null) ? null : rici.getValueType();
            /* Convert empty string value to null for date type */
            if (strValue == null || strValue.trim().length() == 0) {
                wrapper.setValue(null);
            } else {
                /* Format date using DateFormat with Date or DateTime pattern */
                Date dateValue = format.parse(strValue);
                /* Transform to exact parameter type */
                dateValue = convertToProperSubclass(dateValue, paramClass);
                wrapper.setValue(dateValue);
            }
        } catch (ParseException e) {
            if (messageSource != null) {
                wrapper.setErrorMessage(messageSource.getMessage("fillParameters.error.invalidDateTime", null, LocaleContextHolder.getLocale()));
            }
            wrapper.setValue(strValue);
        }
    }

    /**
     * Transform to exact parameter type: Date, Time or DateTime
     * @param value
     *          Date value
     * @param parameterClass
     *          Exact class of parameter
     * @return
     *          Trimmed regarding parameterClass Date value
     */
    protected static Date convertToProperSubclass(Date value, Class parameterClass) {
        if (java.sql.Timestamp.class.equals(parameterClass)) {
            return new java.sql.Timestamp(value.getTime());
        }
        if (java.sql.Date.class.equals(parameterClass)) {
            return new java.sql.Date(value.getTime());
        }
        if (java.sql.Time.class.equals(parameterClass)) {
            /* strip off date part */
            java.util.GregorianCalendar cal = new java.util.GregorianCalendar();
            cal.setTime(value);
            cal.clear(Calendar.YEAR);
            cal.clear(Calendar.MONTH);
            cal.clear(Calendar.DATE);
            return new java.sql.Date(cal.getTimeInMillis());
        }
        return value;
    }
}
