/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.jaspersoft.jasperserver.remote;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import javax.activation.DataSource;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;

/**
 * The scope on the service is always assumed to be the request, so this class should be instanced
 * each time the service has been requested.
 *
 * @author gtoffoli
 */
public class AbstractService {

    public static final String REMOTE_MANAGEMENT_SERVICES = "remoteManagementServices";

    private static final Log log = LogFactory.getLog(AbstractService.class);

    protected static ThreadLocal<AbstractService> context = new ThreadLocal<AbstractService>();

    protected ApplicationContext applicationContext = null;
    private ManagementServices managementServices = null;
    private ServicesUtils      servicesUtils = null;
    private Locale             locale = null;
    private Map<String,DataSource> inputAttachments  = new HashMap<String, DataSource>();
    private Map<String,DataSource> outputAttachments = new HashMap<String, DataSource>();
    private Map<String,Object>     attributes = new HashMap<String, Object>();


    public AbstractService(ApplicationContext ac)
    {
        this(ac, Locale.getDefault());
    }

    public AbstractService(ApplicationContext ac, Locale locale)
    {
        this.applicationContext = ac;
        this.locale = locale;
        servicesUtils = new ServicesUtils(this.applicationContext);
        context.set(this);
    }

    /**
     * @return the managementServices
     */
    public ManagementServices getManagementServices() {
        if (managementServices == null)
        {
            managementServices = (ManagementServices) applicationContext.getBean(REMOTE_MANAGEMENT_SERVICES);
        }
        return managementServices;
    }

    /**
     * @param managementServices the managementServices to set
     */
    public void setManagementServices(ManagementServices managementServices) {
        this.managementServices = managementServices;
    }

    /**
     * @return the servicesUtils
     */
    public ServicesUtils getServicesUtils() {
        return servicesUtils;
    }

    /**
     * @param servicesUtils the servicesUtils to set
     */
    public void setServicesUtils(ServicesUtils servicesUtils) {
        this.servicesUtils = servicesUtils;
    }



    /**
     * This method takes the Locale requested by the client.
     * If the requested locale is null, the default locale
     * is returned.
     *
     * This is just a convenient method to specify a locale using a string
     * instead to pass a locale object.
     * In case the locale is not set, the default locale is set instead.
     *
     * A locale code can be in the form:
     *
     * langagecode[_countrycode]
     *
     * Ex: en_US, it_IT, it, en_UK
     *
     */
    public void setLocale(String requestedLocale)
    {
        try {
            if (requestedLocale != null)
            {
                String language = requestedLocale;
                String country = "";
                if (requestedLocale.indexOf("_") > 0)
                {
                    language = requestedLocale.substring(0, requestedLocale.indexOf("_"));
                    country = requestedLocale.substring(requestedLocale.indexOf("_")+1);
                    setLocale(new Locale(language, country));
                }
                else
                {
                    setLocale(new Locale(language));
                }
            }
        } catch (Exception ex)
        {
            log.error("Unable to get requested locale (" + requestedLocale + ")");
            setLocale(Locale.getDefault());
        }
   }

    /**
     * Return the request locale, or just the default locale if the request did not
     * specified a locale.
     *
     * TODO: Locale should be taked by the HTTP request if available and never set directly
     *
     * @return
     */
    public Locale getLocale() {
        return locale == null ? Locale.getDefault() : locale;
    }

    /**
     * Set the locale of this service execution
     * @param locale
     */
    public void setLocale(Locale locale) {
        this.locale = locale;
    }


    /**
     * The service running in the current thread.
     * The AbstractService class stores in memory the first AbstractService created in the current thread.
     * This allows to retrieve it everywhere just calling this method.
     *
     * @return
     */
    public static AbstractService getContext()
    {
        return context.get();
    }


    /**
     * Return a localized message using the request locale and using the
     * MessageSource to resolve the key and build the message.
     *
     * This is a shortcut for getMessage(key, new Object[]{})
     *
     * @param key
     * @return
     */
    public String getMessage(String key)
    {
        return getMessage(key, new Object[]{});
    }

    /**
     * Return a localized message using the request locale and using the
     * MessageSource to resolve the key and build the message.
     * 
     *
     * @param key
     * @param args
     * @return
     */
    public String getMessage(String key, Object[] args)
    {
        return getManagementServices().getMessage(key, args, getLocale());
    }



    /**
     * Return a localized message using the request locale and using the
     * MessageSource to resolve the key and build the message.
     * If no key is found, the default message is used instead.
     *
     *
     * @param key
     * @param args
     * @return
     */
    public String getMessage(String key, Object[] args, String defMessage)
    {
        return getManagementServices().getMessageSource().getMessage(key, args, defMessage, getLocale());
    }

    /**
     * @return the inputAttachments
     */
    public Map<String, DataSource> getInputAttachments() {
        return inputAttachments;
    }

    /**
     * @return the outputAttachments
     */
    public Map<String, DataSource> getOutputAttachments() {
        return outputAttachments;
    }

    /**
     * @return the attributes
     */
    public Map<String, Object> getAttributes() {
        return attributes;
    }

    /**
     * @param attributes the attributes to set
     */
    public void setAttributes(Map<String, Object> attributes) {
        this.attributes = attributes;
    }
    
}
