/*
// $Id: //open/mondrian/src/main/mondrian/gui/JDBCTreeModel.java#9 $
// This software is subject to the terms of the Common Public License
// Agreement, available at the following URL:
// http://www.opensource.org/licenses/cpl.html.
// Copyright (C) 2002-2008 Julian Hyde and others
// All Rights Reserved.
// You must accept the terms of that agreement to use this software.
*/
package mondrian.gui;

import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreePath;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import org.apache.log4j.Logger;

/**
 *
 * @author  sean
 * @version $Id: //open/mondrian/src/main/mondrian/gui/JDBCTreeModel.java#9 $
 */
public class JDBCTreeModel implements javax.swing.tree.TreeModel {

    private static final Logger LOGGER = Logger.getLogger(JDBCTreeModel.class);

    private Vector treeModelListeners = new Vector();
    JDBCMetaData jdbcMetaData;
    Workbench workbench;

    List catalogs;
    Node root;

    /** Creates a new instance of JDBCTreeModel */
    public JDBCTreeModel(JDBCMetaData metaData, Workbench wb) {
        this.jdbcMetaData = metaData;
        this.workbench = wb;
        try {
            root = new Node("All Schemas", Node.ROOT, null);
            catalogs = new ArrayList();

            for (String schemaName : jdbcMetaData.getAllSchemas()) {
                Node cat = new Node(schemaName, Node.CATALOG, root);

                Vector<String> tables = jdbcMetaData.getAllTables(schemaName);
                for (String tableName : tables) {

                    Node table = new Node(tableName, Node.TABLE, cat);
                    cat.getChildren().add(table);
                }
                catalogs.add(cat);
            }
            root.children = catalogs;
        } catch (Exception ex) {
            LOGGER.error("JDBCTreeModel", ex);
        }
    }


    /** Adds a listener for the <code>TreeModelEvent</code>
     * posted after the tree changes.
     *
     * @param   l       the listener to add
     * @see     #removeTreeModelListener
     *
     */
    public void addTreeModelListener(TreeModelListener l) {
        treeModelListeners.add(l);
    }

    /** Returns the child of <code>parent</code> at index <code>index</code>
     * in the parent's
     * child array.  <code>parent</code> must be a node previously obtained
     * from this data source. This should not return <code>null</code>
     * if <code>index</code>
     * is a valid index for <code>parent</code> (that is <code>index >= 0 &&
     * index < getChildCount(parent</code>)).
     *
     * @param   parent  a node in the tree, obtained from this data source
     * @return  the child of <code>parent</code> at index <code>index</code>
     *
     */
    public Object getChild(Object parent, int index) {
        if (parent instanceof Node) {
            return ((Node)parent).getChildren().get(index);
        }

        return null;
    }

    /** Returns the number of children of <code>parent</code>.
     * Returns 0 if the node
     * is a leaf or if it has no children.  <code>parent</code> must be a node
     * previously obtained from this data source.
     *
     * @param   parent  a node in the tree, obtained from this data source
     * @return  the number of children of the node <code>parent</code>
     *
     */
    public int getChildCount(Object parent) {
        if (parent instanceof Node) {
            return ((Node)parent).getChildren().size();
        }
        return 0;
    }

    /** Returns the index of child in parent.  If <code>parent</code>
     * is <code>null</code> or <code>child</code> is <code>null</code>,
     * returns -1.
     *
     * @param parent a note in the tree, obtained from this data source
     * @param child the node we are interested in
     * @return the index of the child in the parent, or -1 if either
     *    <code>child</code> or <code>parent</code> are <code>null</code>
     *
     */
    public int getIndexOfChild(Object parent, Object child) {
       if (parent instanceof Node) {
            return ((Node)parent).getChildren().indexOf(child);
       }

       return -1;
    }

    /** Returns the root of the tree.  Returns <code>null</code>
     * only if the tree has no nodes.
     *
     * @return  the root of the tree
     *
     */
    public Object getRoot() {
        return root;
    }

    /** Returns <code>true</code> if <code>node</code> is a leaf.
     * It is possible for this method to return <code>false</code>
     * even if <code>node</code> has no children.
     * A directory in a filesystem, for example,
     * may contain no files; the node representing
     * the directory is not a leaf, but it also has no children.
     *
     * @param   node  a node in the tree, obtained from this data source
     * @return  true if <code>node</code> is a leaf
     *
     */
    public boolean isLeaf(Object node) {
        return getChildCount(node) == 0;
    }

    /** Removes a listener previously added with
     * <code>addTreeModelListener</code>.
     *
     * @see     #addTreeModelListener
     * @param   l       the listener to remove
     *
     */
    public void removeTreeModelListener(TreeModelListener l) {
        treeModelListeners.remove(l);
    }

    /** Messaged when the user has altered the value for the item identified
     * by <code>path</code> to <code>newValue</code>.
     * If <code>newValue</code> signifies a truly new value
     * the model should post a <code>treeNodesChanged</code> event.
     *
     * @param path path to the node that the user has altered
     * @param newValue the new value from the TreeCellEditor
     *
     */
    public void valueForPathChanged(TreePath path, Object newValue) {
    }

    class Node {
        static final int CATALOG = 0;
        static final int TABLE = 1;
        static final int COLUMN = 2;
        static final int ROOT = 3;
        String name;
        int type;
        List children;
        Node parent;

        public Node(String n, int t, Node p) {
            name = n;
            type = t;
            parent = p;
            // children = new ArrayList();
        }

        public String toString() {
            if (type == ROOT) {
                return workbench.getResourceConverter().getFormattedString(
                        "jdbcExplorer.root.name", "Schemas",  null);
            }

            if (name == null || name.trim().length() == 0) {
                switch (type) {
                    case CATALOG:
                        return workbench
                                .getResourceConverter()
                                .getFormattedString(
                                    "jdbcExplorer.default.name.catalog",
                                    "Default Schema",
                                    null);
                    case TABLE:
                        return workbench
                                .getResourceConverter()
                                .getFormattedString(
                                    "jdbcExplorer.default.name.table",
                                    "Table",
                                    null);
                    case COLUMN:
                        return workbench
                                .getResourceConverter()
                                .getFormattedString(
                                    "jdbcExplorer.default.name.column",
                                    "Column",
                                    null);
                }
            }
            return name;
        }
        
        public List getChildren() {
            if (children == null) {
                children = new ArrayList();
                
                if (type == TABLE) {
                    
                    // This is a table, parent is a schema
                    
                    Vector<String> columnNames = jdbcMetaData.getAllColumns(parent.name, name);
                    for (String columnName : columnNames) {
                        Node column = new Node(columnName, Node.COLUMN, this);
                        children.add(column);
                    }
                }
            }
            return children;
        }
    }
}

// End JDBCTreeModel.java
