/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.remote.handlers;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;


import com.jaspersoft.jasperserver.remote.RepositoryServiceContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jaspersoft.jasperserver.api.metadata.common.domain.ContentResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResourceData;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.Argument;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.ResourceDescriptor;
import com.jaspersoft.jasperserver.remote.AbstractService;
import com.jaspersoft.jasperserver.remote.ResourceDataSource;
import com.jaspersoft.jasperserver.remote.ServiceException;
import java.io.IOException;
import java.io.InputStream;
import javax.activation.DataSource;

/**
 * @author Lucian Chirita (lucianc@users.sourceforge.net)
 * @version $Id: ContentResourceHandler.java 19933 2010-12-11 15:27:37Z tmatyashovsky $
 */
public class ContentResourceHandler extends RepositoryResourceHandler {

    private static final Log log = LogFactory.getLog(ContentResourceHandler.class);
    private static final String MAIN_ATTACHMENT_ID = "attachment";

    public Class getResourceType() {
        return ContentResource.class;
    }

    protected void doGet(Resource resource, ResourceDescriptor descriptor, Map options) throws ServiceException {
        ContentResource contentRes = (ContentResource) resource;
        descriptor.setWsType(ResourceDescriptor.TYPE_CONTENT_RESOURCE);
        descriptor.setResourceProperty(ResourceDescriptor.PROP_CONTENT_RESOURCE_TYPE, contentRes.getFileType());


        if (options==null || options.containsKey(Argument.NO_RESOURCE_DATA_ATTACHMENT)) {
            descriptor.setHasData(false);
        } else {
            // Retrieve resource data...
            descriptor.setHasData(true);

            addMainAttachment(descriptor);

            if (!options.containsKey(Argument.NO_SUBRESOURCE_DATA_ATTACHMENTS)) {
                addSubAttachments(contentRes, descriptor);
            }
        }

    }

    @Override
    protected void updateResource(Resource resource, ResourceDescriptor descriptor, Map options) throws ServiceException {

        super.updateResource(resource, descriptor, options);

        ContentResource contentRes = (ContentResource) resource;

        String fileType = descriptor.getResourcePropertyValue(ResourceDescriptor.PROP_CONTENT_RESOURCE_TYPE);
        if (fileType != null) {
            contentRes.setFileType(fileType);
        }

        // TODO: Support attachments...

        if (descriptor.getHasData()) {
                
                DataSource ds = locateDataAttachment(descriptor);
                readData(contentRes, ds);
        }

        List children = descriptor.getChildren();
        List subresources = new ArrayList(children == null ? 0 : children.size());
        if (children != null) {
            for (Iterator it = children.iterator(); it.hasNext();) {
                ResourceDescriptor subdescriptor = (ResourceDescriptor) it.next();
                ContentResource subresource = (ContentResource) createChildResource(subdescriptor);
                subresources.add(subresource);
            }
        }
        contentRes.setResources(subresources);
    }

    protected DataSource locateDataAttachment(ResourceDescriptor resourceDescriptor) throws ServiceException {

        String contentName = resourceDescriptor.getResourcePropertyValue(ResourceDescriptor.PROP_DATA_ATTACHMENT_ID);
        if (contentName == null) {
            contentName = resourceDescriptor.getName();
        }

        DataSource ds = AbstractService.getContext().getInputAttachments().get(contentName);

        if (ds == null) {
            throw new ServiceException(ServiceException.GENERAL_ERROR2,
                    AbstractService.getContext().getMessage("webservices.error.content.resource.data.not.present",
                    new Object[]{contentName}));
        }

        return ds;
    }

    protected String getAttachmentContentID(ContentResource contentRes) {
        return contentRes.getName();
    }

    protected void readData(ContentResource resource, DataSource dataSource) throws ServiceException {

        boolean close = true;
        InputStream dataStream = null;
        try {
            dataStream = dataSource.getInputStream();
            resource.readData(dataStream);

            close = false;
            dataStream.close();
        } catch (IOException e) {
            throw new ServiceException(e);
        } finally {
            if (close && dataStream != null) {
                try {
                    dataStream.close();
                } catch (IOException e) {
                    log.error("Error closing attachment stream", e);
                }
            }
        }
    }


    protected void addMainAttachment(ResourceDescriptor descriptor)
    {
        FileResourceData fileData = AbstractService.getContext().getManagementServices().getRepository().getContentResourceData(null, descriptor.getUriString());
        ResourceDataSource attachmentSource = new ResourceDataSource(descriptor.getName(), fileData);

        AbstractService.getContext().getOutputAttachments().put(MAIN_ATTACHMENT_ID, attachmentSource);
        descriptor.setResourceProperty(ResourceDescriptor.PROP_DATA_ATTACHMENT_ID, MAIN_ATTACHMENT_ID);
    }

    protected void addSubAttachments(ContentResource contentRes, ResourceDescriptor descriptor)
    {

        if (contentRes.getFileType().equals(ContentResource.TYPE_HTML))
        {
            List resources = contentRes.getResources();
            if (resources != null && !resources.isEmpty()) {
                for (Iterator it = resources.iterator(); it.hasNext();) {
                    ContentResource subResource = (ContentResource) it.next();
                    FileResourceData subData = AbstractService.getContext().getManagementServices().getRepository().getContentResourceData(null,subResource.getURIString());
                    ResourceDataSource subAttachment = new ResourceDataSource(subResource.getName(), subData);
                    AbstractService.getContext().getOutputAttachments().put(subResource.getName(), subAttachment);
                    descriptor.setResourceProperty(ResourceDescriptor.PROP_DATA_ATTACHMENT_ID, MAIN_ATTACHMENT_ID);
                }
            }
        }
    }

   
}
