/*
 * JaLingo, http://jalingo.sourceforge.net/
 *
 * Copyright (c) 2002-2006 Oleksandr Shyshko
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package ja.lingo.engine.mergedindex;

import ja.JaTestCase;
import ja.lingo.engine.beans.IArticle;

import java.io.IOException;

public class MergedIndexMergerTest extends JaTestCase {
    private static final String ABC = "abc";
    private static final String ABCD = "abcd";
    private static final String ABCDE = "abcde";
    private static final String BBB = "bbb";
    private static final String ZXDDT = "zxddt";
    private static final String ZXY = "zxy";

    private MockDictionaryIndex reader1;
    private MockDictionaryIndex reader2;

    private MergedIndexMerger merger;
    private MockBuchIndexBuilder builder;

    protected void setUp() throws Exception {
        super.setUp();

        reader1 = new MockDictionaryIndex();
        reader1.addArticle( ABC );
        reader1.addArticle( ABCDE );
        reader1.addArticle( ZXDDT );

        reader2 = new MockDictionaryIndex();
        reader2.addArticle( ABC );
        reader2.addArticle( ABCD );
        reader2.addArticle( BBB );
        reader2.addArticle( ZXDDT );
        reader2.addArticle( ZXY );

        merger = new MergedIndexMerger();
        builder = new MockBuchIndexBuilder();
    }

    protected void tearDown() throws Exception {
        builder.verify();

        super.tearDown();
    }

    public void test() throws IOException {
        final String[] TITLES = {
                ABC,
                ABCD,
                ABCDE,
                BBB,
                ZXDDT,
                ZXY
        };

        final IArticle[] articles = new IArticle[] {
                new Article( reader1, 0 ).append( reader2, 0 ),
                new Article( reader2, 1 ),
                new Article( reader1, 1 ),
                new Article( reader2, 2 ),
                new Article( reader1, 2 ).append( reader2, 3 ),
                new Article( reader2, 4 )
        };

        merger.addReaderContents( reader1 );
        merger.addReaderContents( reader2 );

        merger.build( builder );

        IMergedIndex index = builder.getMergedIndex();

        assertEquals( TITLES.length, index.size() );

        for ( int articleIndex = 0; articleIndex < TITLES.length; articleIndex++ ) {
            String expectedTitle = TITLES[articleIndex];

            String title = index.getArticleTitle( articleIndex );
            assertEquals( expectedTitle, title );

            IArticle article = index.getArticle( articleIndex );
            assertEquals( articles[articleIndex].size(), article.size() );

            for ( int groupIndex = 0; groupIndex < articles[articleIndex].size(); groupIndex++ ) {
                assertSame(
                        articles[articleIndex].getReader( groupIndex ),
                        article.getReader( groupIndex ) );

                assertEquals(
                        articles[articleIndex].getInReaderIndex( groupIndex ),
                        article.getInReaderIndex( groupIndex ) );
            }
        }
    }

    public void testDuplicateSequence() throws IOException {
        MockDictionaryIndex reader = new MockDictionaryIndex();
        reader.addArticle( ABCDE );
        reader.addArticle( ABCDE );

        merger.addReaderContents( reader );

        merger.build( builder );
        IMergedIndex index = builder.getMergedIndex();

        assertEquals( 1, index.size() );
    }

    public void testDuplicateSequenceDifferentCase() throws IOException {
        MockDictionaryIndex reader1 = new MockDictionaryIndex();
        reader1.addArticle( ABCDE.toUpperCase() );
        reader1.addArticle( ABCDE );

        MockDictionaryIndex reader2 = new MockDictionaryIndex();
        reader2.addArticle( ABC );
        reader2.addArticle( ABCDE.toUpperCase() );

        merger.addReaderContents( reader1 );
        merger.addReaderContents( reader2 );

        merger.build( builder );
        IMergedIndex index = builder.getMergedIndex();

        assertEquals( 4, index.size() );
    }
}
