/*
 * JaLingo, http://jalingo.sourceforge.net/
 *
 * Copyright (c) 2002-2006 Oleksandr Shyshko
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package ja.centre.util.sort.external;

import ja.JaTestCase;
import ja.centre.util.measurer.TimeMeasurer;
import ja.centre.util.sort.ComparableComparator;
import ja.centre.util.sort.ISorter;
import ja.centre.util.sort.MergeSorter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class ExternalSorterTest extends JaTestCase {
/*
    static {
        LogInitializer.initialize();
    }
*/

    private static final Log LOG = LogFactory.getLog( ExternalSorterTest.class );

    private ISorter<String> sorter;

    protected void setUp() throws Exception {
        super.setUp();
        sorter = new MergeSorter<String>();
    }

    public void testSort() throws IOException {
        List<String> values = Arrays.asList( new String[] { "s", "ddd", "zz", "Fwee", "12", "rerw", "ASD" } );
        List<String> expected = Arrays.asList( new String[] { "12", "ASD", "Fwee", "ddd", "rerw", "s", "zz" } );

        List<String> actual = new ArrayList<String>();

        CollectionReader<String> reader = new CollectionReader<String>( values );
        CollectionWriter<String> writer = new CollectionWriter<String>( actual );

        // sort
        ExternalSorter<String> externalSorter = new ExternalSorter<String>( new StringPersister(), sorter, new ComparableComparator<String>(), 3 );
        externalSorter.sort( reader, writer );

        // check
        assertEquals( expected, actual );
    }

    public void testSortInMemory() throws IOException {
        runRandomStringTest( 5000, 1000, 16 );
    }

    public void testSortEquals() throws IOException {
        runRandomStringTest( 5000, 5000, 16 );
    }

    public void testSortGreater() throws IOException {
        runRandomStringTest( 5000, 10000, 16 );
    }

    public void testOrderedSubject() throws IOException {
        runOrderedStringTest( 5000, 5000 );
    }

    private void runRandomStringTest( int valuesInMemory, final int stringCount, final int stringLength ) throws IOException {
        IReader<String> reader = new RandomStringReader( stringCount, stringLength );
        LOG.info( "Sorting " + stringCount + " strings of length " + stringLength + " using buffer of size " + valuesInMemory );
        runStringTest( valuesInMemory, stringCount, reader );
    }

    private void runOrderedStringTest( int valuesInMemory, final int stringCount ) throws IOException {
        IReader<String> reader = new OrderedStringReader( stringCount );
        LOG.info( "Sorting " + stringCount + " of ordered strings using buffer of size " + valuesInMemory );
        runStringTest( valuesInMemory, stringCount, reader );
    }

    private void runStringTest( int valuesInMemory, int stringCount, IReader<String> reader ) throws IOException {
        LOG.info( "Expected number of parts is " + ((int) Math.ceil( (float) stringCount )) / valuesInMemory );
        Writer writer = new Writer();

        TimeMeasurer measurer = new TimeMeasurer();

        ExternalSorter<String> externalSorter = new ExternalSorter<String>( new StringPersister(), sorter, new ComparableComparator<String>(), valuesInMemory );
        externalSorter.sort( reader, writer );

        assertEquals( stringCount, writer.getWrittenCount() );

        LOG.info( "Sort complete with " + measurer );
    }

    private static String createRandomString( int length ) {
        byte[] string = new byte[length];

        for ( int i = 0; i < string.length; i++ ) {
            string[i] = (byte) ((int) (Math.random() * 32) + ((int) 'A'));
        }

        return new String( string );

    }

    private static class Writer implements IWriter<String> {
        private String lastValue;
        private int writtenCount;

        public void write( String value ) {
            if ( lastValue != null && lastValue.compareTo( value ) > 0 ) {
                throw new RuntimeException( "Last value \"" + lastValue + "\" is greater then next \"" + value + "\"" );
            }

            lastValue = value;
            writtenCount++;
        }

        public void close() {
        }

        public int getWrittenCount() {
            return writtenCount;
        }
    }

    private static class RandomStringReader implements IReader<String> {
        private int count;
        private final int stringLength;

        public RandomStringReader( int count, int stringLength ) {
            this.stringLength = stringLength;
            this.count = count;
        }

        public boolean hasNext() {
            return count != 0;
        }

        public String next() {
            count--;
            return createRandomString( stringLength );
        }

        public void close() {
        }
    }

    private static class OrderedStringReader implements IReader<String> {
        private int count;
        private int index = 100000000;

        public OrderedStringReader( int count ) {
            this.count = count;

        }

        public boolean hasNext() {
            return count != 0;
        }

        public String next() {
            count--;
            return Integer.toString( index++ );
        }

        public void close() {
        }
    }
}
