/*
 * JaLingo, http://jalingo.sourceforge.net/
 *
 * Copyright (c) 2002-2006 Oleksandr Shyshko
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.izforge.izpack.util.os;

import ja.centre.util.assertions.Arguments;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * Taken from IzPack, then cut not neccesary code and rewritten a bit.
 */
public class ShellLink {
    public static final int HIDE = 0;
    public static final int NORMAL = 1;
    public static final int MINIMIZED = 2;
    public static final int MAXIMIZED = 3;

    public static final int MIN_SHOW = 0;
    public static final int MAX_SHOW = 3;

    public static final int DESKTOP = 1;
    public static final int PROGRAM_MENU = 2;
    public static final int START_MENU = 3;
    public static final int STARTUP = 4;

    public static final int CURRENT_USER = 0;
    public static final int ALL_USERS = 1;

    public int linkType = DESKTOP;
    public int userType = CURRENT_USER;

    private static final int MIN_TYPE = 1;
    private static final int MAX_TYPE = 4;

    private static final int SL_OK = 1;
    private static final int SL_ERROR = -1;
    private static final int SL_INITIALIZED = -2;
    private static final int SL_NOT_INITIALIZED = -3;
    private static final int SL_OUT_OF_HANDLES = -4;
    private static final int SL_NO_IPERSIST = -5;
    private static final int SL_NO_SAVE = -6;
    private static final int SL_WRONG_DATA_TYPE = -7;

    private static final int UNINITIALIZED = -1;
    private static final String LINK_EXTENSION = ".lnk";

    private int nativeHandle = UNINITIALIZED;

    private String currentUserLinkPath = "";
    private String allUsersLinkPath = "";
    private String groupName = "";
    private String linkName = "";
    private String linkFileName = "";
    private String linkDirectory = "";

    private String arguments = "";
    private String description = "";
    private String iconPath = "";
    private String targetPath = "";
    private String workingDirectory = "";

    private String dummyString = "";

    private int hotkey = 0;
    private int iconIndex = 0;
    private int showCommand = NORMAL;

    private boolean initializeSucceeded = false;

    private native int initializeCOM();
    private native int releaseCOM();
    private native int getInterface();
    private native int releaseInterface();
    private native int GetArguments();
    private native int GetDescription();
    private native int GetHotkey();
    private native int GetIconLocation();
    private native int GetPath();
    private native int GetShowCommand();
    private native int GetWorkingDirectory();
    private native int Resolve();
    private native int SetArguments();
    private native int SetDescription();
    private native int SetHotkey();
    private native int SetIconLocation();
    private native int SetPath();
    private native int SetShowCommand();
    private native int SetWorkingDirectory();
    private native int saveLink( String name );
    private native int loadLink( String name );
    private native int GetLinkPath( int target );
    private native void FreeLibrary( String name );

    private static boolean nativeLibraryLoaded;

    public ShellLink() throws Exception {
        if ( !nativeLibraryLoaded ) {
            Runtime.getRuntime().load( getPathToShellLinkDll() );
            nativeLibraryLoaded = true;
        }

        if ( initializeCOM() != SL_OK ) {
            throw new Exception( "could not initialize COM" );
        } else {
            initializeSucceeded = true;
        }

        int successCode = getInterface();
        if ( successCode != SL_OK ) {
            releaseCOM();
            initializeSucceeded = false;

            if ( successCode == SL_OUT_OF_HANDLES ) {
                throw new Exception( "could not get an instance of IShellLink, no more handles available" );
            } else {
                throw new Exception( "could not get an instance of IShellLink, failed to co-create instance" );
            }
        }
    }

    private String getPathToShellLinkDll() throws IOException {
        InputStream is = getClass().getResourceAsStream( "ShellLink.dll" );

        if ( is == null ) {
            throw new IOException( "Could not locate resource of native library" );
        }

        File tempFile = File.createTempFile( "_ShellLink.dll_", ".dll" );

        FileOutputStream os = new FileOutputStream( tempFile );
        //
        byte[] buffer = new byte[32768];
        int read;
        while ( (read = is.read( buffer )) != -1 ) {
            os.write( buffer, 0, read );
        }
        is.close();
        os.close();

        // NOTE leak: due to unknown reason, this file is not delete after JVM exit!
        tempFile.deleteOnExit();

        return tempFile.getAbsolutePath();
    }

    protected void finalize() {
        releaseInterface();

        if ( initializeSucceeded ) {
            releaseCOM();
            initializeSucceeded = false;
        }
    }

    private String fullLinkPath( int userType ) {
        StringBuilder path = new StringBuilder();

        // ----------------------------------------------------
        // make sure we hava a valid storage path
        // ----------------------------------------------------
        // TODO: handle errors
        int result = GetLinkPath( linkType );

        // ----------------------------------------------------
        // build the complete name
        // ----------------------------------------------------
        if ( userType == CURRENT_USER ) {
            path.append( currentUserLinkPath );
        } else {
            path.append( allUsersLinkPath );
        }

        if ( groupName != null && groupName.length() > 0 ) {
            path.append( File.separator );
            path.append( groupName );
        }

        return path.toString();
    }
    private String fullLinkName( int userType ) {
        StringBuilder name = new StringBuilder();

        name.append( fullLinkPath( userType ) );

        name.append( File.separator );
        name.append( linkName );
        name.append( LINK_EXTENSION );

        return name.toString();
    }

    private void set() throws Exception {
        if ( SetArguments() != SL_OK ) {
            throw new Exception( "could not set arguments" );
        }
        if ( SetDescription() != SL_OK ) {
            throw new Exception( "could not set description" );
        }
        if ( SetHotkey() != SL_OK ) {
            throw new Exception( "could not set hotkey" );
        }
        if ( SetIconLocation() != SL_OK ) {
            throw new Exception( "could not set icon location" );
        }
        if ( SetPath() != SL_OK ) {
            throw new Exception( "could not set target path" );
        }
        if ( SetShowCommand() != SL_OK ) {
            throw new Exception( "could not set show command" );
        }
        if ( SetWorkingDirectory() != SL_OK ) {
            throw new Exception( "could not set working directory" );
        }
    }

    private void setProgramGroup( String groupName ) {
        this.groupName = groupName;
    }
    public void setArguments( String arguments ) {
        this.arguments = arguments;
    }
    public void setDescription( String description ) {
        this.description = description;
    }
    private void setHotkey( int hotkey ) {
        this.hotkey = hotkey;
    }
    public void setIconLocation( String path, int index ) {
        this.iconPath = path;
        this.iconIndex = index;
    }
    public void setTargetPath( String path ) {
        this.targetPath = path;
    }
    public void setShowCommand( int show ) {
        if ( show < MIN_SHOW || show > MAX_SHOW ) {
            Arguments.doThrow( "Illegal value for show command" );
        }

        this.showCommand = show;
    }
    public void setWorkingDirectory( String dir ) {
        this.workingDirectory = dir;
    }
    public void setLinkName( String name ) {
        linkName = name;
    }
    public void setLinkType( int type ) throws IllegalArgumentException {
        if ( type < MIN_TYPE || type > MAX_TYPE ) {
            Arguments.doThrow( "Illegal value for type" );
        }

        linkType = type;
    }
    public void setUserType( int type ) throws IllegalArgumentException {
        if ( type == CURRENT_USER || type == ALL_USERS ) {
            userType = type;
        } else {
            Arguments.doThrow( type + " is not a recognized user type" );
        }
    }

    public void save() throws Exception {
        // set all values on the native side
        set();

        // make sure the target actually resolves
        int result = Resolve();
        if ( result != SL_OK ) {
            throw new Exception( "cannot resolve target" );
        }

        // make sure the directory exists
        File directory = new File( fullLinkPath( userType ) );
        if ( !directory.exists() ) {
            directory.mkdirs();
            linkDirectory = directory.getPath();
        } else {
            linkDirectory = "";
        }

        // perform the save operation
        String saveTo = fullLinkName( userType );
        result = saveLink( saveTo );

        if ( result == SL_NO_IPERSIST ) {
            throw new Exception( "could not get handle for IPesist" );
        } else if ( result == SL_NO_SAVE ) {
            throw new Exception( "the save operation failed" );
        }

        linkFileName = saveTo;
    }
}
