/*****************************************************************************
*   "Irit" - the 3d (not only polygonal) solid modeller.		     *
*									     *
* Written by:  Gershon Elber				Ver 0.2, Mar. 1990   *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
*   Handles the fetching of characters for the irt scripting language.       *
*****************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <string.h>
#include "program.h"
#include "allocate.h"
#include "ctrl-brk.h"
#include "inptprsg.h"
#include "inptprsl.h"
#include "objects.h"
#include "overload.h"

#define FILE_STACK_SIZE	10
#define MAX_QUEUE_LINES 1000

typedef struct FileStackStruct {
    char Name[LINE_LEN_SHORT];
    char Path[LINE_LEN_LONG];
    FILE *f;
} FileStackStruct;

GLOBAL_DATA IritInputSourceType IritInputSource;

STATIC_DATA char
    UnGetChar = 0,
    *UnGetLines[MAX_QUEUE_LINES];  /* Pushed line pointers on input buffer!? */
STATIC_DATA int
    FileStackPtr = 0,
    InptPrsrEchoSource = 0,
    UnGetLinesTail = 0,
    UnGetLinesHead = 0;
STATIC_DATA FileStackStruct FileStack[FILE_STACK_SIZE];/* Include file stack.*/

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Queue a line to process in input stream.                                 M
*                                                                            *
* PARAMETERS:                                                                M
*   Line:     To queue for processing in input stream.                       M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptPrsrQueueInputLine                                                   M
*****************************************************************************/
void InptPrsrQueueInputLine(char *Line)
{
    UnGetLines[UnGetLinesHead++] = Line;

    if (UnGetLinesHead == MAX_QUEUE_LINES)
	UnGetLinesHead = 0;

    if (UnGetLinesHead == UnGetLinesTail)
        IRIT_NON_FATAL_ERROR("Reevaluation input line queue is full");
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Sets echo level of source irt files.                                       M
*                                                                            *
* PARAMETERS:                                                                M
*   EchoSource:  TRUE for echo, FALSE for no echo.                           M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:  Old value.                                                         M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptPrsrSetEchoSource                                                    M
*****************************************************************************/
int InptPrsrSetEchoSource(int EchoSource)
{
    int OldVal = InptPrsrEchoSource;

    InptPrsrEchoSource = EchoSource;

    return OldVal;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to control all getchar in this module and echo it if requested     M
*   Note it handles the FileStack and decrease it if end of file was found.  M
*                                                                            *
* PARAMETERS:                                                                M
*   InString:   TRUE if now we read between two double quotes (a string).    M
*                                                                            *
* RETURN VALUE:                                                              M
*   char:       Next character in input stream.                              M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptPrsrGetC	                                                     M
*****************************************************************************/
char InptPrsrGetC(int InString)
{
    STATIC_DATA char *p,
	Line[INPUT_LINE_LEN] = "",
	TLine[INPUT_LINE_LEN] = "";
    STATIC_DATA int
	LineLength = 0,
	LineCount = 0;
    char c;
    int i;

    if (UnGetChar == 0) {		       /* One level of unget char... */
	if (LineCount < LineLength) {	 /* Is there anything in local Line? */
	}
	else {
	    do {
	        if (UnGetLinesHead != UnGetLinesTail) {
		    strcpy(Line, UnGetLines[UnGetLinesTail++]);
		    if (UnGetLinesTail == MAX_QUEUE_LINES)
		        UnGetLinesTail = 0;

		    IritInputSource = IRIT_INPUT_SOURCE_LINE_QUEUE;
		    LineCount = 0;
		}
	        else if (FileStackPtr == 0) {
		    IritInputWindowGetStr(Line, INPUT_LINE_LEN);
		    IritInputSource = IRIT_INPUT_SOURCE_KBD;
		    LineCount = 0;
	        }
	        else {
		    sprintf(Line, "%s > ",
			    FileStack[FileStackPtr - 1].Name);
		    IritInputSource = IRIT_INPUT_SOURCE_FILE;
		    IritIdleFunction(0);
		    LineCount = (int) strlen(Line);
		    if (fgets(TLine, INPUT_LINE_LEN,
			      FileStack[FileStackPtr - 1].f) == NULL) {
		        /* Its end of file - close it and update stack. */
			TLine[0] = 0;
		        fclose(FileStack[--FileStackPtr].f);
		    }

		    /* Strip off CR/LF/TAB.  */
		    for (i = LineCount, p = TLine; *p != 0; p++) {
			if (*p == 0x09)
			    do {
				Line[i++] = ' ';
			    }
		            while ((i - LineCount) % 8 != 0);
			else if (*p < ' ' || *p > '~')
			    break;
			else
			    Line[i++] = *p;
		    }
		    Line[i] = 0;
	        }

		if (InptPrsrEchoSource)
#ifdef DJGCC
		    IRIT_WNDW_PUT_STR(Line);
#else
		    if (IritInputSource == IRIT_INPUT_SOURCE_FILE) {
		        /* Input was from file? */
			IRIT_WNDW_PUT_STR(Line);
		    }
#endif /* DJGCC */

	        LineLength = (int) strlen(Line);
	    } while (LineCount >= LineLength);
	}

	c = Line[LineCount++];
	if (c == '#' && !InString) {	  /* Its a comment - skip that line. */
            c = ' ';				   /* Must return something. */
            LineCount = LineLength;    /* Force next time to fetch new line. */
	}
#	ifdef DEBUG1
	    fprintf(stderr, "%c", c);
#	endif /* DEBUG1 */
    }
    else {
	c = UnGetChar;
	UnGetChar = 0;
    }

    return c;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Routine to unget one char						     *
*                                                                            *
* PARAMETERS:                                                                *
*   c:        Character to unget.                                            *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
void InptPrsrUnGetC(char c)
{
    UnGetChar = c;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to quit reading until next ';'. If reading from files, they are    M
* all closed as well.							     M
*                                                                            *
* PARAMETERS:                                                                M
*   FlushStdin:   If not reading from a file, should we skip to next ';'?    M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptPrsrFlushToEndOfExpr                                                 M
*****************************************************************************/
void InptPrsrFlushToEndOfExpr(int FlushStdin)
{
    if (UnGetLinesHead != UnGetLinesTail)
	UnGetLinesHead = UnGetLinesTail;
    if (FileStackPtr > 0)   /* Close all open files - back to stdin. */
	while (FileStackPtr)
	    fclose(FileStack[--FileStackPtr].f);
    else if (FlushStdin && InptPrsrLastToken != TOKENEND)
	while (InptPrsrGetC(FALSE) != ';');
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to push new file to read on the FileStack from INCLUDE command:    M
*                                                                            *
* PARAMETERS:                                                                M
*   PrmFileName:  Name of file to start to read from.                        M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   InptPrsrFileInclude                                                      M
*****************************************************************************/
void InptPrsrFileInclude(char *PrmFileName)
{
    int i;
    FILE *f;
    char FileName[LINE_LEN_LONG], c, *p;

    if (FileStackPtr < FILE_STACK_SIZE) {
	strcpy(FileName, PrmFileName);

  	if (strstr(FileName, ".irt") == NULL &&
	    strstr(FileName, ".IRT") == NULL)
	    strcat(FileName, ".irt");
	
	if ((f = fopen(FileName, "r")) == NULL &&
	    FileStackPtr > 0 &&
	    strlen(FileStack[FileStackPtr - 1].Path) > 0) {
	    char FullFileName[LINE_LEN_LONG];

	    /* Try the directory specified by the parent file, if any. */
	    sprintf(FullFileName, "%s%s",
		    FileStack[FileStackPtr - 1].Path, FileName);
	    if ((f = fopen(FullFileName, "r")) != NULL)
	        strncpy(FileName, FullFileName, LINE_LEN_LONG - 1);
	}

	/* If still a failure, see if we have IRIT_INCLUDE specifications. */
	if (f == NULL && (p = getenv("IRIT_INCLUDE")) != NULL) {
	    char Path[LINE_LEN_LONG];

	    strncpy(Path, p, LINE_LEN_LONG - 1);

	    if ((p = strtok(Path, ";")) != NULL) {
	        do {
		    char FullFileName[LINE_LEN_LONG];

		    /* Try the directory specified by the IRIT_INCLUDE. */
#		    if defined(__WINNT__) || defined(OS2GCC)
		        sprintf(FullFileName, "%s\\%s", p, FileName);
#		    else
		        sprintf(FullFileName, "%s/%s", p, FileName);
#		    endif /* __WINNT__ || OS2GCC */

		    if ((f = fopen(FullFileName, "r")) != NULL) {
		        strncpy(FileName, FullFileName, LINE_LEN_LONG - 1);
			break;
		    }
		}
		while ((p = strtok(NULL, ";")) != NULL);
	    }
	}

	if (f != NULL) {
	    FileStack[FileStackPtr].f = f;
	    for (i = (int) strlen(FileName) - 1;   /* Isolate the file name. */
		 i > 0 && (c = FileName[i]) != '\\' && c != '/' && c != ':';
		 i--);
	    if (i > 0)
		i++;
	    strncpy(FileStack[FileStackPtr].Name, &FileName[i],
							   LINE_LEN_SHORT - 1);
	    if (i > 0) {
	        FileName[i] = 0;
		strncpy(FileStack[FileStackPtr].Path, FileName,
							    LINE_LEN_LONG - 1);
	    }
	    else 
		FileStack[FileStackPtr].Path[0] = 0;

	    if ((p = strstr(FileStack[FileStackPtr].Name,
			    ".irt")) != NULL ||
		(p = strstr(FileStack[FileStackPtr].Name,
			    ".IRT")) != NULL)
		*p = 0;

	    FileStackPtr++;		 /* Now next char is from that file! */
	}
	else {
	    IRIT_WNDW_FPRINTF2("Cannt open file %s - ignored", FileName);
	}
    }
    else
	IRIT_WNDW_PUT_STR("File nesting too deep - ignored");
}

