/*****************************************************************************
*   "Irit" - the 3d (not only polygonal) solid modeller.		     *
*									     *
* Written by:  Gershon Elber				Ver 0.2, Mar. 1990   *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
*   Module to provide the required interfact for the cagd library for the    *
* free form surfaces and curves.					     *
*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include "program.h"
#include "user_lib.h"
#include "geom_lib.h"
#include "objects.h"
#include "mrchcube.h"
#include "mdl_lib.h"
#include "freeform.h"

#define RFLCT_LN_SIL_EPS	0.01

static void ComputeSmoothPolyNormalsAux(IPObjectStruct *PObj,
					RealType MaxAngle);
static void FixPolyNormalsAux(IPObjectStruct *PObj, int TrustPixPt);
static void FixPolyGeometryAux(IPObjectStruct *PObj, int Op, RealType Eps);
static IPPolygonStruct *FilterPolylineData(CagdSrfStruct *Srf,
					   int RflctLines,
					   VectorType ViewDir,
					   IPPolygonStruct *Pls,
					   int Euclidean,
					   RealType *LinePt,
					   RealType *LineDir);
static PointType *ConvertListOfPtsToArray(IPObjectStruct *PtSet, int *n);
static PointType *GetPointList(IPObjectStruct *PtList, int *n);
static IPObjectStruct *GetControlMVMesh(int n,
					int *Lengths,
					int *Orders,
					IPObjectStruct *PtLstObj,
					MvarGeomType GType,
					char **ErrStr);

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Compute smooth normals to vertices of a polygonal object without normals M
* to vertices.                                                               M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:      Polygonal object to estimate normals to its vertices.         M
*   MaxAngle:  Maximum angle between approximated normal and face normal of  M
*	       a vertex to consider valid and smooth it.		     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Same geometry as PObj but with smooth normals        M
*                                                                            *
* SEE ALSO:                                                                  M
*   GMBlendNormalsToVertices                                                 M
*                                                                            *
* KEYWORDS:                                                                  M
*   ComputeSmoothPolyNormals                                                 M
*****************************************************************************/
IPObjectStruct *ComputeSmoothPolyNormals(IPObjectStruct *PObj,
					 RealType *MaxAngle)
{
    if (!IP_IS_OLST_OBJ(PObj) &&
	!IP_IS_POLY_OBJ(PObj) &&
	!IP_IS_POLYGON_OBJ(PObj)) {
	IRIT_WNDW_PUT_STR("Expecting a polygonal (list) object");
	return NULL;
    }

    PObj = IPCopyObject(NULL, PObj, TRUE);

    ComputeSmoothPolyNormalsAux(PObj, *MaxAngle);

    return PObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Auxiliary function of ComputeSmoothPolyNormals                           *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:      Polygonal object to estimate normals to its vertices.         *
*   MaxAngle:  Maximum angle between approximated normal and face normal of  *
*	       a vertex to consider valid and smooth it.		     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void								     *
*****************************************************************************/
static void ComputeSmoothPolyNormalsAux(IPObjectStruct *PObj,
					RealType MaxAngle)
{
    if (IP_IS_OLST_OBJ(PObj)) {
	int i = 0;
        IPObjectStruct *Obj;

        while ((Obj = IPListObjectGet(PObj, i++)) != NULL)
	    ComputeSmoothPolyNormalsAux(Obj, MaxAngle);
    }
    else if (IP_IS_POLY_OBJ(PObj) && IP_IS_POLYGON_OBJ(PObj)) {
        GMBlendNormalsToVertices(PObj -> U.Pl, MaxAngle);
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Correct the normals of a polygonal object.                               M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:         Polygonal object to correct normals.			     M
*   TrustPixPt:   0 to trust the vertices' normal,			     M
*		  1 to trust the orientation of polygons' normals,	     M
*		  2 to reorient the polygons so all plane normals point      M
*		    outside or all inside (based on first poly).	     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Same geometry as PObj but with fixed normals.        M
*                                                                            *
* SEE ALSO:                                                                  M
*   GMBlendNormalsToVertices                                                 M
*                                                                            *
* KEYWORDS:                                                                  M
*   FixPolyNormals		                                             M
*****************************************************************************/
IPObjectStruct *FixPolyNormals(IPObjectStruct *PObj, RealType *TrustPixPt)
{
    if (!IP_IS_OLST_OBJ(PObj) &&
	!IP_IS_POLY_OBJ(PObj) &&
	!IP_IS_POLYGON_OBJ(PObj)) {
	IRIT_WNDW_PUT_STR("Expecting a polygonal (list) object");
	return NULL;
    }

    PObj = IPCopyObject(NULL, PObj, TRUE);

    FixPolyNormalsAux(PObj, REAL_PTR_TO_INT(TrustPixPt));

    return PObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Auxiliary function of FixPolyNormals.	                             *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:         Polygonal object to correct normals.			     *
*   TrustPixPt:   0 to trust the vertices' normal,			     *
*		  1 to trust the orientation of polygons' normals,	     *
*		  2 to reorient the polygons so all plane normals point      *
*		    outside or all inside (based on first poly.		     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void								     *
*****************************************************************************/
static void FixPolyNormalsAux(IPObjectStruct *PObj, int TrustPixPt)
{
    if (IP_IS_OLST_OBJ(PObj)) {
	int i = 0;
        IPObjectStruct *Obj;

        while ((Obj = IPListObjectGet(PObj, i++)) != NULL)
	    FixPolyNormalsAux(Obj, TrustPixPt);
    }
    else if (IP_IS_POLY_OBJ(PObj) && IP_IS_POLYGON_OBJ(PObj)) {
        GMFixNormalsOfPolyModel(PObj -> U.Pl, TrustPixPt);
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Correct the normals of a polygonal object.                               M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:         Polygonal object to correct normals.			     M
*   Op:           0 to remove duplicated polygons,			     *
*		  1 to remove zero length edges.			     *
*   Eps:	  Tolerance of vertices equality, etc.                       M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Same geometry as PObj but with fixed normals.        M
*                                                                            *
* SEE ALSO:                                                                  M
*   GMBlendNormalsToVertices                                                 M
*                                                                            *
* KEYWORDS:                                                                  M
*   FixPolyGeometry		                                             M
*****************************************************************************/
IPObjectStruct *FixPolyGeometry(IPObjectStruct *PObj,
				RealType *Op,
				RealType *Eps)
{
    if (!IP_IS_OLST_OBJ(PObj) &&
	!IP_IS_POLY_OBJ(PObj) &&
	!IP_IS_POLYGON_OBJ(PObj)) {
	IRIT_WNDW_PUT_STR("Expecting a polygonal (list) object");
	return NULL;
    }

    PObj = IPCopyObject(NULL, PObj, TRUE);

    FixPolyGeometryAux(PObj, REAL_PTR_TO_INT(Op), *Eps);

    return PObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Auxiliary function of FixPolyGeometry.	                             *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:         Polygonal object to correct normals.			     *
*   Op:           0 to remove duplicated polygons,			     *
*		  1 to remove zero length edges.			     *
*   Eps:	  Tolerance of vertices equality, etc.                       *
*                                                                            *
* RETURN VALUE:                                                              *
*   void								     *
*****************************************************************************/
static void FixPolyGeometryAux(IPObjectStruct *PObj, int Op, RealType Eps)
{
    if (IP_IS_OLST_OBJ(PObj)) {
	int i = 0;
        IPObjectStruct *Obj;

        while ((Obj = IPListObjectGet(PObj, i++)) != NULL)
	    FixPolyGeometryAux(Obj, Op, Eps);
    }
    else if (IP_IS_POLY_OBJ(PObj) && IP_IS_POLYGON_OBJ(PObj)) {
	switch (Op) {
	    case 0:
	        GMCleanUpDupPolys(&PObj -> U.Pl, Eps);
		break;
	    case 1:
		GMCleanUpPolygonList(&PObj -> U.Pl, Eps);
		break;
	}
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Blossom evaluation of freeforms.                                         M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:        Geometry to blossom.                                        M
*   BlsmVals:    Blossom values.                                             M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  Evaluated Blossom                                     M
*                                                                            *
* SEE ALSO:                                                                  M
*   CagdCrvBlossomEval                                                       M
*                                                                            *
* KEYWORDS:                                                                  M
*   BlossomEvaluation                                                        M
*****************************************************************************/
IPObjectStruct *BlossomEvaluation(IPObjectStruct *PObj,
				  IPObjectStruct *BlsmVals)
{
    int BlsmsLen = 0;
    CagdRType *Blsms;
    IPObjectStruct
	*RetVal = NULL;

    if (IP_IS_OLST_OBJ(BlsmVals)) {
        int Length = IPListObjectLength(BlsmVals);
	IPObjectStruct *B;

	Blsms = (CagdRType *) IritMalloc(sizeof(CagdRType) * Length);

	while ((B = IPListObjectGet(BlsmVals, BlsmsLen)) != NULL &&
	       BlsmsLen < Length) {
	    if (!IP_IS_NUM_OBJ(B)) {
		IritFree(Blsms);
	        IRIT_NON_FATAL_ERROR("Expecting a LIST of blossom values.");
		return NULL;
	    }

	    Blsms[BlsmsLen++] = B -> U.R;
	}
    }
    else {
        IRIT_NON_FATAL_ERROR("Expecting a LIST of blossom values.");
    }

    if (IP_IS_CRV_OBJ(PObj)) {
        CagdCtlPtStruct *CtlPt;

	if (BlsmsLen >= PObj -> U.Crvs -> Order) {
	    IRIT_NON_FATAL_ERROR("Blossom LIST length larger than degree.");
        }
	else {
	    CtlPt = CagdCrvBlossomEval(PObj -> U.Crvs, Blsms, BlsmsLen);

	    RetVal = IPGenCTLPTObject(CtlPt -> PtType, CtlPt -> Coords, NULL);

	    CagdCtlPtFree(CtlPt);
	}
    }
    else
        IRIT_NON_FATAL_ERROR("Only curve can be blossomed.");

    IritFree(Blsms);

    return RetVal;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Remove unnecessary knots from a given Bspline curve.                     M
*                                                                            *
* PARAMETERS:                                                                M
*   CrvObj:       To remove unneeded knots from.                             M
*   SamplesNum:   Number of samples to take for norm computations.  Used     M
*		  for error estimates and should be at least in the dozens.  M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Reduced curve, that is similar to original curve!    M
*                                                                            *
* SEE ALSO:                                                                  M
*   SymbRmKntBspCrvCleanKnots, RemoveKnots                                   M
*                                                                            *
* KEYWORDS:                                                                  M
*   CleanRefinedKnots                                                        M
*****************************************************************************/
IPObjectStruct *CleanRefinedKnots(IPObjectStruct *CrvObj, RealType *SamplesNum)
{
    CagdCrvStruct
	*Crv = SymbRmKntBspCrvCleanKnots(CrvObj -> U.Crvs,
					 REAL_PTR_TO_INT(SamplesNum));

    if (Crv != NULL)
	return IPGenCRVObject(Crv);
    else
	return NULL;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Remove unnecessary knots from a given Bspline curve.                     M
*                                                                            *
* PARAMETERS:                                                                M
*   CrvObj:       To remove unneeded knots from.                             M
*   GlobalLimit:  Global limit - maximal error if positive, maximum number   M
*		  of knots to remove if negative (taking the absolute value).M
*   IterationLimit:  Same as GlobalLimit but always positive with sematics   M
*		  as in GlobalLimit, but for one iteration only.	     M
*   SamplesNum:   Number of samples to take for norm computations.  Used     M
*		  for error estimates and should be at least in the dozens.  M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Reduced curve, that is similar to original curve!    M
*                                                                            *
* SEE ALSO:                                                                  M
*   SymbRmKntBspCrvRemoveKnotsError, SymbRmKntBspCrvRemoveKnots, RemoveKnots M
*                                                                            *
* KEYWORDS:                                                                  M
*   RemoveKnots                                                              M
*****************************************************************************/
IPObjectStruct *RemoveKnots(IPObjectStruct *CrvObj,
			    RealType *GlobalLimit,
			    RealType *IterationLimit,
			    RealType *SamplesNum)
{
    CagdCrvStruct *Crv;

    if (*GlobalLimit > 0.0)
	Crv = SymbRmKntBspCrvRemoveKnotsError(CrvObj -> U.Crvs, 
					      *GlobalLimit, 
					      FABS(*IterationLimit), 
					      REAL_PTR_TO_INT(SamplesNum));
    else
	Crv = SymbRmKntBspCrvRemoveKnots(CrvObj -> U.Crvs, 
					 -REAL_PTR_TO_INT(GlobalLimit), 
					 ABS(REAL_PTR_TO_INT(IterationLimit)),
					 REAL_PTR_TO_INT(SamplesNum));

    if (Crv != NULL)
	return IPGenCRVObject(Crv);
    else
	return NULL;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Computes the reflection lines of a freeform surface.                     M
*                                                                            *
* PARAMETERS:                                                                M
*   SrfObj:       Surface to compute reflection lines for.                   M
*   ViewDir:	  Direction of view.					     M
*   LnsSprs:      A list of either:					     M
*                 1. A vector and list of points prescribing the infinite    M
*		     (parallel) lines to be reflected off the surface.       M
*                 2. A center sphere point and list of cone opening angles   M
*		     prescribing the circular curves to be reflected off the M
*		     surface.						     M
*		  If this object is not a list object, it is assumed the     M
*		  auxiliary data precomputed in previous reflection          M
*		  computation is to be freed.			             M
*   Euclidean:    TRUE for reflection lines in R3 on the surface, FALSE for  M
*		  reflection lines in parametric space of surface.	     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A list of reflection lines.			     M
*                                                                            *
* SEE ALSO:                                                                  M
*   SymbRflctLnGen, SymbRflctCircGen					     M
*                                                                            *
* KEYWORDS:                                                                  M
*   ReflectionLines                                                          M
*****************************************************************************/
IPObjectStruct *ReflectionLines(IPObjectStruct *SrfObj,
				VectorType ViewDir,
				IPObjectStruct *LnsSprs,
				RealType *Euclidean)
{
    STATIC_DATA PlaneType
        Plane = { 1.0, 0.0, 0.0, 1.280791e-6 };
    int i, j;
    IPObjectStruct *PCntrObj, *LinePt, *LinesPts, *LinesDir,
	*SprCntr, *ConeAngles, *CAngObj,
	*PCntrsObj = IPGenLISTObject(NULL);
    CagdBType
	OldInterpFlag = BspMultInterpFlag(FALSE);
    CagdSrfStruct
	*Srf = SrfObj -> U.Srfs;

    if (PT_EQ_ZERO(ViewDir) &&
	IP_IS_OLST_OBJ(LnsSprs) &&
	(LinesDir = IPListObjectGet(LnsSprs, 0)) != NULL &&
	IP_IS_VEC_OBJ(LinesDir) &&
	(LinesPts = IPListObjectGet(LnsSprs, 1)) != NULL &&
	IP_IS_OLST_OBJ(LinesPts)) {
        for (i = j = 0; (LinePt = IPListObjectGet(LinesPts, i++)) != NULL; ) {
	    if (IP_IS_POINT_OBJ(LinePt)) {
	        IPObjectStruct
		    *PZeroObj = IPGenSRFObject(SymbHighlightLnGen(Srf,
							LinePt -> U.Pt,
							LinesDir -> U.Vec,
							NULL));

		if ((PCntrObj = ContourFreeform(PZeroObj, Plane,
						NULL, NULL)) != NULL) {
		    /* Filter out all unrelevant/redundant components. */
		    PCntrObj -> U.Pl = FilterPolylineData(Srf, TRUE, ViewDir,
							  PCntrObj -> U.Pl,
							  APX_EQ(*Euclidean,
								 0.0),
							  LinePt -> U.Pt,
							  LinesDir -> U.Vec);

		    if (PCntrObj -> U.Pl != NULL)
		        IPListObjectInsert(PCntrsObj, j++, PCntrObj);
		    else
		        IPFreeObject(PCntrObj);
		}

		IPFreeObject(PZeroObj);
	    }
	    else {
	        IRIT_NON_FATAL_ERROR("Expecting a point on reflection line.");
	    }
	}
	IPListObjectInsert(PCntrsObj, j, NULL);

	/* Free any left auxiliary data structure related to highlight comp. */
	SymbHighlightLnFree(SrfObj -> U.Srfs, NULL);
    }
    else if (IP_IS_OLST_OBJ(LnsSprs) &&
	     (LinesDir = IPListObjectGet(LnsSprs, 0)) != NULL &&
	     IP_IS_VEC_OBJ(LinesDir) &&
	     (LinesPts = IPListObjectGet(LnsSprs, 1)) != NULL &&
	     IP_IS_OLST_OBJ(LinesPts)) {
        for (i = j = 0; (LinePt = IPListObjectGet(LinesPts, i++)) != NULL; ) {
	    if (IP_IS_POINT_OBJ(LinePt)) {
	        IPObjectStruct
		    *PZeroObj = IPGenSRFObject(SymbRflctLnGen(Srf,
							    ViewDir,
							    LinePt -> U.Pt,
							    LinesDir -> U.Vec,
							    NULL));

		if ((PCntrObj = ContourFreeform(PZeroObj, Plane,
						NULL, NULL)) != NULL) {
		    /* Filter out all unrelevant/redundant components. */
		    PCntrObj -> U.Pl = FilterPolylineData(Srf, TRUE, ViewDir,
							  PCntrObj -> U.Pl,
							  APX_EQ(*Euclidean,
								 0.0),
							  LinePt -> U.Pt,
							  LinesDir -> U.Vec);

		    if (PCntrObj -> U.Pl != NULL)
		        IPListObjectInsert(PCntrsObj, j++, PCntrObj);
		    else
		        IPFreeObject(PCntrObj);
		}

		IPFreeObject(PZeroObj);
	    }
	    else {
	        IRIT_NON_FATAL_ERROR("Expecting a point on reflection line.");
	    }
	}
	IPListObjectInsert(PCntrsObj, j, NULL);

	/* Free any left auxiliary data structure related to reflect. comp. */
	SymbRflctLnFree(SrfObj -> U.Srfs, NULL);
    }
    else if (IP_IS_OLST_OBJ(LnsSprs) &&
	     (SprCntr = IPListObjectGet(LnsSprs, 0)) != NULL &&
	     IP_IS_POINT_OBJ(SprCntr) &&
	     (ConeAngles = IPListObjectGet(LnsSprs, 1)) != NULL &&
	     IP_IS_OLST_OBJ(ConeAngles)) {
        for (i = j = 0;
	     (CAngObj = IPListObjectGet(ConeAngles, i++)) != NULL;
	     ) {
	    if (IP_IS_NUM_OBJ(CAngObj)) {
	        IPObjectStruct
		    *PZeroObj = IPGenSRFObject(SymbRflctCircGen(Srf,
							      ViewDir,
							      SprCntr -> U.Pt,
							      CAngObj -> U.R,
							      NULL));

		if ((PCntrObj = ContourFreeform(PZeroObj, Plane,
						NULL, NULL)) != NULL) {
		    /* Filter out all unrelevant/redundant components. */
		    PCntrObj -> U.Pl = FilterPolylineData(Srf, FALSE, ViewDir,
							  PCntrObj -> U.Pl,
							  APX_EQ(*Euclidean,
								 0.0),
							  NULL, NULL);

		    if (PCntrObj -> U.Pl != NULL)
		        IPListObjectInsert(PCntrsObj, j++, PCntrObj);
		    else
		        IPFreeObject(PCntrObj);
		}

		IPFreeObject(PZeroObj);
	    }
	    else {
	        IRIT_NON_FATAL_ERROR("Expecting a cone's opening angle for reflection circle.");
	    }
	}
	IPListObjectInsert(PCntrsObj, j, NULL);

	/* Free any left auxiliary data structure related to reflect. comp. */
	SymbRflctCircFree(SrfObj -> U.Srfs, NULL);
    }
    else {
	/* Return an empty list object. */
	IPListObjectInsert(PCntrsObj, 0, NULL);
    }

    BspMultInterpFlag(OldInterpFlag);

    return PCntrsObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Filter out all unrelevant/redundant components.                          *
*                                                                            *
* PARAMETERS:                                                                *
*   Srf:        Surface geometry where this polyline data came from.         *
*   RflctLines: Are we reflecting lines (TRUE), or circles (FALSE).	     *
*   ViewDir:	Direction of view, or (0, 0, 0) to ignore.		     *
*   Pls:        Polyline data to filter.                                     *
*   Euclidean:  TRUE for reflection lines in R3 on the surface, FALSE for    *
*		reflection lines in parametric space of surface.	     *
*   LinePt:     If reflecting lines, a point on the reflected line.          *
*   LineDir:    If reflecting lines, the direction of the line.	             *
*                                                                            *
* RETURN VALUE:                                                              *
*   IPPolygonStruct *:  Filtered data                                        *
*****************************************************************************/
static IPPolygonStruct *FilterPolylineData(CagdSrfStruct *Srf,
					   int RflctLines,
					   VectorType ViewDir,
					   IPPolygonStruct *Pls,
					   int Euclidean,
					   RealType *LinePt,
					   RealType *LineDir)
{
    IPPolygonStruct *Pl, *PlPrev;
    CagdRType UMin, UMax, VMin, VMax;

    CagdSrfDomain(Srf, &UMin, &UMax, &VMin, &VMax);

    for (Pl = PlPrev = Pls; Pl != NULL; ) {
        IPVertexStruct *VPrev, *V;

	for (V = VPrev = Pl -> PVertex; V != NULL; ) {
	    int Purge = FALSE;
	    CagdRType t, t1, t2, *P;
	    CagdPType PtE3, Pt1, Pt2;
	    CagdVType RflctDir;
	    CagdVecStruct *Nrml;

	    V -> Coord[0] = BOUND(V -> Coord[0], UMin, UMax);
	    V -> Coord[1] = BOUND(V -> Coord[1], VMin, VMax);

	    P = CagdSrfEval(Srf, V -> Coord[0], V -> Coord[1]);
	    CagdCoerceToE3(PtE3, &P, -1, Srf -> PType);

	    if (!PT_EQ_ZERO(ViewDir)) {
		Nrml = CagdSrfNormal(Srf, V -> Coord[0], V -> Coord[1], TRUE);
		t = DOT_PROD(Nrml -> Vec, ViewDir);
		if (FABS(t) < RFLCT_LN_SIL_EPS)
		    Purge = TRUE;  /* Almost a silhouette point. */
		else {
		    /* Compute reflection direction... */
		    VEC_COPY(RflctDir, Nrml -> Vec);
		    VEC_SCALE(RflctDir, 2.0 * t);
		    VEC_SUB(RflctDir, RflctDir, ViewDir);

		    if (RflctLines) {
			/* Solve for inter. point of reflection line */
			/* and reflection direction off surface.     */
			GM2PointsFromLineLine(PtE3, RflctDir, LinePt, LineDir,
					      Pt1, &t1, Pt2, &t2);

			Purge = ((t < 0.0) ^ (t1 > 0.0));
		    }
		}
	    }

	    if (Purge) {
	        if (V == Pl -> PVertex) {
		    Pl -> PVertex = V -> Pnext;
		    IPFreeVertex(V);
		    V = VPrev = Pl -> PVertex;
		}
		else {
		    /* Break the polyline there. */
		    VPrev -> Pnext = NULL;
		    Pl -> Pnext = IPAllocPolygon(0, V -> Pnext, Pl -> Pnext);
		    IPFreeVertex(V);
		    V = NULL;
		}
	    }
	    else {
	        if (Euclidean) {
		    V -> Coord[0] = V -> Coord[1];
		    V -> Coord[1] = V -> Coord[2];
		    V -> Coord[2] = 0.0;
		}
		else {
		    PT_COPY(V -> Coord, PtE3);
		}

		VPrev = V;
		V = V -> Pnext;
	    }
	}

	if (Pl -> PVertex == NULL) {
	    /* We have a completely deleted polyline. */
	    if (Pl == Pls) {
	        Pls = Pl -> Pnext;
		IPFreePolygon(Pl);
		Pl = PlPrev = Pls;
	    }
	    else {
	        PlPrev -> Pnext = Pl -> Pnext;
		IPFreePolygon(Pl);
		Pl = PlPrev -> Pnext;
	    }
	}
	else {
	    PlPrev = Pl;
	    Pl = Pl -> Pnext;
	}
    }

    return Pls;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Computes the area integral of a given curve.	                     M
*                                                                            *
* PARAMETERS:                                                                M
*   CrvObj:       To compute area integral for.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Integral of area as a scalar.                        M
*                                                                            *
* SEE ALSO:                                                                  M
*   SymbCrvEnclosedArea			                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   CrvAreaIntergal                                                          M
*****************************************************************************/
IPObjectStruct *CrvAreaIntergal(IPObjectStruct *CrvObj)
{
    CagdCrvStruct
	*Crv = SymbCrvEnclosedArea(CrvObj -> U.Crvs);

    if (Crv != NULL)
	return IPGenCRVObject(Crv);
    else
	return NULL;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Computes the volume integral of a given surface.	                     M
*                                                                            *
* PARAMETERS:                                                                M
*   SrfObj:      To compute volume integral for.                             M
*   Method:      Either 1 (for volume to XY plane) or			     M
*			2 (for volume to origin).			     M
*   Eval:	 TRUE to evaluate the volume integral to its numeric result. M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Integral of volume (A surface volume scalar field or M
*			a numeric volume result).			     M
*                                                                            *
* SEE ALSO:                                                                  M
*   SymbSrfVolume1Srf, SymbSrfVolume1, SymbSrfVolume2Srf, SymbSrfVolume2     M
*                                                                            *
* KEYWORDS:                                                                  M
*   SrfVolumeIntergal                                                        M
*****************************************************************************/
IPObjectStruct *SrfVolumeIntergal(IPObjectStruct *SrfObj,
				  RealType *Method,
				  RealType *Eval)
{
    RealType Vol;
    CagdSrfStruct *VolSrf;

    switch (REAL_PTR_TO_INT(Method)) {
	case 1:
	    if (REAL_PTR_TO_INT(Eval)) {
	        Vol = SymbSrfVolume1(SrfObj -> U.Srfs);
		return IPGenNUMValObject(Vol);

	    }
	    else {
	        VolSrf = SymbSrfVolume1Srf(SrfObj -> U.Srfs, TRUE);
		return IPGenSRFObject(VolSrf);
	    }
        case 2:
	default:
	    if (REAL_PTR_TO_INT(Eval)) {
	        Vol = SymbSrfVolume2(SrfObj -> U.Srfs);
		return IPGenNUMValObject(Vol);
	    }
	    else {
	        VolSrf = SymbSrfVolume2Srf(SrfObj -> U.Srfs, TRUE);
		return IPGenSRFObject(VolSrf);
	    }
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Computes the volume integral of a given surface.	                     M
*                                                                            *
* PARAMETERS:                                                                M
*   SrfObj:      To compute volume integral for.                             M
*   Moment:	 Either 1 or 2 for first or second moment.		     M
*   Axis1, Axis2:  1 for X, 2 for Y, 3 for Z.	Axis2 is ignored for first   M
*		 order moments.						     M
*   Eval:	 TRUE to evaluate the volume integral to its numeric result. M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Integral of volume (A surface volume scalar field or M
*			a numeric volume result).			     M
*                                                                            *
* SEE ALSO:                                                                  M
*   SymbSrfFirstMoment, SymbSrfFirstMomentSrf,                               M
*   SymbSrfSecondMoment, SymbSrfSecondMomentSrf                              M
*                                                                            *
* KEYWORDS:                                                                  M
*   SrfMomentIntergal                                                        M
*****************************************************************************/
IPObjectStruct *SrfMomentIntergal(IPObjectStruct *SrfObj,
				  RealType *Moment,
				  RealType *Axis1,
				  RealType *Axis2,
				  RealType *Eval)
{
    CagdRType Mom;
    CagdSrfStruct *MomSrf;

    switch (REAL_PTR_TO_INT(Moment)) {
	case 1:
	    if (REAL_PTR_TO_INT(Eval)) {
	        Mom = SymbSrfFirstMoment(SrfObj -> U.Srfs,
					 REAL_PTR_TO_INT(Axis1));
		return IPGenNUMValObject(Mom);
	    }
	    else {
	        MomSrf = SymbSrfFirstMomentSrf(SrfObj -> U.Srfs,
					       REAL_PTR_TO_INT(Axis1),
					       TRUE);
		return IPGenSRFObject(MomSrf);
	    }
	case 2:
	    if (REAL_PTR_TO_INT(Eval)) {
	        Mom = SymbSrfSecondMoment(SrfObj -> U.Srfs,
					  REAL_PTR_TO_INT(Axis1),
					  REAL_PTR_TO_INT(Axis2));
		return IPGenNUMValObject(Mom);
	    }
	    else {
	        MomSrf = SymbSrfSecondMomentSrf(SrfObj -> U.Srfs,
						REAL_PTR_TO_INT(Axis1),
						REAL_PTR_TO_INT(Axis2),
						TRUE);
		return IPGenSRFObject(MomSrf);
	    }
	default:
	    IRIT_NON_FATAL_ERROR("Invalid moment - only first or second.");
	    return NULL;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Converts the given set of points into an array.                          *
*                                                                            *
* PARAMETERS:                                                                *
*   PtSet:   A list object of a set of points.                               *
*   n:       To place here the number of points converted.                   *
*                                                                            *
* RETURN VALUE:                                                              *
*   PointType *:    Array of n points, allocated dynamically.                *
*****************************************************************************/
static PointType *ConvertListOfPtsToArray(IPObjectStruct *PtSet, int *n)
{
    int i,
        NumVertices = -1;
    PointType *PtArray;
    IPObjectStruct *PtObj;

    while ((PtObj = IPListObjectGet(PtSet, ++NumVertices)) != NULL) {
	if (!IP_IS_CTLPT_OBJ(PtObj) &&
	    !IP_IS_POINT_OBJ(PtObj) &&
	    !IP_IS_VEC_OBJ(PtObj)) {
	    IRIT_NON_FATAL_ERROR("Non point object found in list");
	    return NULL;
	}
    }
    *n = NumVertices;
    PtArray = (PointType *) IritMalloc(sizeof(PointType) * NumVertices);

    for (i = 0; i < NumVertices; i++) {
        PtObj = IPCoerceObjectTo(IPListObjectGet(PtSet, i), IP_OBJ_POINT);

	PT_COPY(PtArray[i], PtObj -> U.Pt);
	IPFreeObject(PtObj);
    }

    return PtArray;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Register (find the rigid motion transformation) to match points set      M
* PointSet1 with either a second point set PointSet2 or with surface Srf2.   M
*   Returns a rigid motion transformation that brings the first point set,   M
* PointsSet1 to match the second set, PointSetSrf2.			     M
*                                                                            *
* PARAMETERS:                                                                M
*   PointsSet1:      The first points set as set of points.                  M
*   PointsSetSrf2:   The second - either a points set or a surface.          M
*   AlphaConverge:   Numerical controller between (more than) zero and one.  M
*		     For (almost) zero, the convergance is slow but stable   M
*		     and for one it will be fast but less stable.	     M
*   Tolerance:       Of registeration, in L infinity sense.		     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:     A matrix object represented the computed           M
*			  matrix of the registration.			     M
*                                                                            *
* SEE ALSO:                                                                  M
*   UserRegisterTwoPointSets                                                 M
*                                                                            *
* KEYWORDS:                                                                  M
*   RegisterPointSet                                                         M
*****************************************************************************/
IPObjectStruct *RegisterPointSet(IPObjectStruct *PointsSet1,
				 IPObjectStruct *PointsSetSrf2,
				 RealType *AlphaConverge,
				 RealType *Tolerance)
{
    if (IP_IS_OLST_OBJ(PointsSetSrf2)) {
        int n1, n2;
        PointType
	    *PtSet1 = ConvertListOfPtsToArray(PointsSet1, &n1),
	    *PtSet2 = ConvertListOfPtsToArray(PointsSetSrf2, &n2);
	MatrixType RegMat;
	RealType CompTol;

	if (PtSet1 == NULL || PtSet2 == NULL) {
	    if (PtSet1 != NULL)
	        IritFree(PtSet1);
	    if (PtSet2 != NULL)
	        IritFree(PtSet2);
	    return NULL;
	}
	
	CompTol = UserRegisterTwoPointSets(n1, PtSet1, n2, PtSet2,
					   *AlphaConverge, *Tolerance,
					   NULL, RegMat);

	IritFree(PtSet1);
	IritFree(PtSet2);

	if (CompTol > *Tolerance)
	    IRIT_NON_FATAL_ERROR("Failed to converge to desired tolerance.");

	return IPGenMATObject(RegMat);
    }
    else if (IP_IS_SRF_OBJ(PointsSetSrf2)) {
        int n1;
        PointType
	    *PtSet1 = ConvertListOfPtsToArray(PointsSet1, &n1);
	MatrixType RegMat;
	RealType CompTol;

	if (PtSet1 == NULL)
	    return NULL;
	
	CompTol = UserRegisterPointSetSrf(n1, PtSet1, PointsSetSrf2 -> U.Srfs,
					  *AlphaConverge, *Tolerance,
					  NULL, RegMat);

	IritFree(PtSet1);

	if (CompTol > *Tolerance)
	    IRIT_NON_FATAL_ERROR("Failed to converge to desired tolerance.");

	return IPGenMATObject(RegMat);
    }
    else {
        IRIT_NON_FATAL_ERROR("Expecting either a points set or a surface.");
	return NULL;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Constructs a real 3D bump mapping over surface Srf with texture Texture  M
* tiled over the surface (DuDup x DvDup) times.                              M
*                                                                            *
* PARAMETERS:                                                                M
*   Srf:       Surface to construct 3D bump texture geometry for.            M
*   Texture:   The geometry of a single tile.  Spans [0, 1] in x and y.      M
*   DuDup:     Number of times to place the tile along the U axis.           M
*   DvDup:     Number of times to place the tile along the V axis.           M
*   LclUVs:    TRUE for using local Texture tile UV's, FALSE for Srf UV's.   M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   The geometry of the bumy surface.                    M
*                                                                            *
* SEE ALSO:                                                                  M
*   UserDDMPolysOverSrf                                                      M
*                                                                            *
* KEYWORDS:                                                                  M
*   EvalDDMForSrf                                                            M
*****************************************************************************/
IPObjectStruct *EvalDDMForSrf(IPObjectStruct *Srf,
			      IPObjectStruct *Texture,
			      RealType *DuDup,
			      RealType *DvDup,
			      RealType *LclUVs)
{
    if (IP_IS_SRF_OBJ(Srf)) {
        return UserDDMPolysOverSrf(Srf -> U.Srfs, Texture, *DuDup, *DvDup,
				   REAL_PTR_TO_INT(LclUVs));
    }
    else if (IP_IS_POLY_OBJ(Srf) && IP_IS_POLYGON_OBJ(Srf)) {
        return UserDDMPolysOverPolys(Srf, Texture, *DuDup, *DvDup,
				     REAL_PTR_TO_INT(LclUVs));
    }
    else {
        IRIT_NON_FATAL_ERROR("Expecting either a polygonal mesh or a surface.");
        return NULL;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Fetch a list of point objects into a vector of PointType.                *
*                                                                            *
* PARAMETERS:                                                                *
*   PtList:   A list object of points.                                       *
*   n:        Updated with the length of the created vector of points.       *
*                                                                            *
* RETURN VALUE:                                                              *
*   PointType *:  A vector of points holding the fetched list.               *
*****************************************************************************/
static PointType *GetPointList(IPObjectStruct *PtList, int *n)
{
    int i, j,
        NumVertices = -1;
    PointType *Pts;
    IPObjectStruct *PtObj;
    CagdPointType PtType;

    if (!IP_IS_OLST_OBJ(PtList)) {
	IRIT_NON_FATAL_ERROR("Input not object list object!");
	return NULL;
    }

    while ((PtObj = IPListObjectGet(PtList, ++NumVertices)) != NULL) {
	if (!IP_IS_CTLPT_OBJ(PtObj) &&
	    !IP_IS_POINT_OBJ(PtObj) &&
	    !IP_IS_VEC_OBJ(PtObj)) {
	    IRIT_NON_FATAL_ERROR("Non point object found in list");
	    return NULL;
	}
    }

    /* Coerce all points to a common space, in place. */
    if ((PtType = IPCoercePtsListTo(PtList, CAGD_PT_E1_TYPE)) == CAGD_PT_NONE)
	return NULL;

    Pts = (PointType *) IritMalloc(sizeof(PointType) * NumVertices);

    for (i = 0; i < NumVertices; i++) {
	IPObjectStruct
	    *VObj = IPListObjectGet(PtList, i);
	RealType
	    *v = VObj -> U.CtlPt.Coords;

	if (CAGD_IS_RATIONAL_PT(PtType)) {
	    CagdRType 
	        w = *v++;

	    for (j = 0; j < 3; j++)
		Pts[i][j] = *v++ / w;
	}
	else
	    for (j = 0; j < 3; j++)
		Pts[i][j] = *++v;
    }

    *n = NumVertices;

    return Pts;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Analytic fit to a given set of points.                                   M
*                                                                            *
* PARAMETERS:                                                                M
*   UVPts:          List of parametric domain points.                        M
*   EucPts:         List of Euclidean points (R^3). Same length as UVPts.    M
*   RFirstAtOrigin: TRUE to coerce the first point to be at UV origin.       M
*   RFitDegree:     1 for a bilinear, 2 for quadrics.                        M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  4/6 coefficient points of the fitted surfaces as:     M
*    RFitDegree = 1:  A + B * u + C * v + D * u * v                          M
*    RFitDegree = 2:  A + B * u + C * v + D * u * u + E * u * v + F * v * v  M
*                                                                            *
* SEE ALSO:                                                                  M
*   GMSrfBilinearFit, GMSrfQuadricFit                                        M
*                                                                            *
* KEYWORDS:                                                                  M
*   AnalyticSrfFit                                                           M
*****************************************************************************/
IPObjectStruct *AnalyticSrfFit(IPObjectStruct *UVPts,
			       IPObjectStruct *EucPts,
			       RealType *RFirstAtOrigin,
			       RealType *RFitDegree)
{
    int i, UVLen, EucLen,
	FitLen = 0,
	FirstAtOrigin = REAL_PTR_TO_INT(RFirstAtOrigin),
	FitDegree = REAL_PTR_TO_INT(RFitDegree);
    PointType *FitPts,
	*Uvs = GetPointList(UVPts, &UVLen),
	*Eucs = GetPointList(EucPts, &EucLen);
    IPObjectStruct *RetFit;

    if (Uvs == NULL || Eucs == NULL || UVLen != EucLen) {
	if (Uvs != NULL)
	    IritFree(Uvs);
	if (Eucs != NULL)
	    IritFree(Eucs);

	return NULL;
    }
    
    switch(FitDegree) {
	case 1:
	    FitPts = GMSrfBilinearFit(Uvs, Eucs, FirstAtOrigin, UVLen);
	    FitLen = 4;
	    break;
	case 2:
	    FitPts = GMSrfQuadricFit(Uvs, Eucs, FirstAtOrigin, UVLen);
	    FitLen = 6;
	    break;
	default:
	    IRIT_NON_FATAL_ERROR("Invalid fit order!");
    }

    IritFree(Uvs);
    IritFree(Eucs);

    if (FitLen == 0)
        return NULL;

    RetFit = IPGenLISTObject(NULL);

    for (i = 0; i < FitLen; i++) {
	IPListObjectInsert(RetFit, i, IPGenPTObject(&FitPts[i][0],
						    &FitPts[i][1],
						    &FitPts[i][2]));
    }
    IPListObjectInsert(RetFit, i, NULL);

    return RetFit;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Clip a polygonal model against a prescribed plane.                       M
*                                                                            *
* PARAMETERS:                                                                M
*   Poly:   A polygonal object to clip.                                      M
*   Plane:  To clip polygonal object Poly against.	                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A list of three polygonal objects (that can be        M
*		empty!): the polygon in the positive side of the Plane, the  M
*		polygons that intersects the Plane and the polygons on the   M
*		negative side of Plane, in this order.	                     M
*		If one of the polygonal lists is empty, a numeric value of   M
*		zero is substituted instead.				     M
*                                                                            *
* SEE ALSO:                                                                  M
*   GMClipPolysAgainstPlane                                                  M
*                                                                            *
* KEYWORDS:                                                                  M
*   PolyPlaneClipping                                                        M
*****************************************************************************/
IPObjectStruct *PolyPlaneClipping(IPObjectStruct *Poly, PlaneType Plane)
{
    IPObjectStruct *PObj;
    IPPolygonStruct *PNeg, *PInter,
	*PPos = GMClipPolysAgainstPlane(Poly -> U.Pl, &PNeg, &PInter, Plane);
    IPObjectStruct *(* IPGenPolyFuncPtr)(IPPolygonStruct *);

    if (IP_IS_POLYGON_OBJ(Poly))
        IPGenPolyFuncPtr = IPGenPOLYObject;
    else
        IPGenPolyFuncPtr = IPGenPOLYLINEObject;

    PObj = IPGenLISTObject(PPos == NULL ? IPGenNUMValObject(0)
			                : IPGenPolyFuncPtr(PPos));
    IPListObjectInsert(PObj, 1, PInter == NULL ? IPGenNUMValObject(0)
			                       : IPGenPolyFuncPtr(PInter));
    IPListObjectInsert(PObj, 2, PNeg == NULL ? IPGenNUMValObject(0)
			                     : IPGenPolyFuncPtr(PNeg));
    IPListObjectInsert(PObj, 3, NULL);

    return PObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Evaluate principle curvatures and directions for the given surface.      M
*                                                                            *
* PARAMETERS:                                                                M
*   PSrf:  Surface to evaluate the prinicple curvatures/directions.          M
*   U, V:   Parametric location where to evaluate the curvature properties.  M
*   REuclidean:  TRUE for asymptotes in Euclidean tangent space,	     M
*		 FALSE for parameteric space.				     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:    A list of four entries as (k1, D1, k2, D2) where    M
*			 ki are principle curvatures and Di are principle    M
*			 directions.					     M
*                                                                            *
* SEE ALSO:                                                                  M
*   SymbEvalSrfCurvature                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   EvalSurfaceCurvature                                                     M
*****************************************************************************/
IPObjectStruct *EvalSurfaceCurvature(IPObjectStruct *PSrf,
				     RealType *U,
				     RealType *V,
				     RealType *REuclidean)
{
    CagdRType k1, k2;
    CagdVType D1, D2;
    IPObjectStruct *PRetVal;

    if (AttrGetPtrAttrib(PSrf -> U.Srfs -> Attr, "_EvalCurv") == NULL)
	SymbEvalSrfCurvPrep(PSrf -> U.Srfs, TRUE);

    SymbEvalSrfCurvature(PSrf -> U.Srfs, *U, *V,
			 !REAL_PTR_TO_INT(REuclidean), &k1, &k2, D1, D2);

    SymbEvalSrfCurvPrep(PSrf -> U.Srfs, FALSE);
    AttrFreeOneAttribute(&PSrf -> U.Srfs -> Attr, "_EvalCurv");

    PRetVal = IPGenLISTObject(IPGenNUMValObject(k1));
    IPListObjectInsert(PRetVal, 1, IPGenVECObject(&D1[0], &D1[1], &D1[2]));
    IPListObjectInsert(PRetVal, 2, IPGenNUMValObject(k2));
    IPListObjectInsert(PRetVal, 3, IPGenVECObject(&D2[0], &D2[1], &D2[2]));
    IPListObjectInsert(PRetVal, 4, NULL);

    return PRetVal;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Evaluate asymptotic directions for the given surface/location.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   PSrf:  Surface to evaluate asymptotic directions for.	             M
*   U, V:  Parametric location where to evaluate the asymptotes.	     M
*   REuclidean:  TRUE for asymptotes in Euclidean tangent space,	     M
*		 FALSE for parameteric space.				     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:    A list of upto two vector, the asymptotes.	     M
*                                                                            *
* SEE ALSO:                                                                  M
*   SymbEvalSrfCurvature                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   EvalSurfaceAsympDir                                                      M
*****************************************************************************/
IPObjectStruct *EvalSurfaceAsympDir(IPObjectStruct *PSrf,
				    RealType *U,
				    RealType *V,
				    RealType *REuclidean)
{
    int n,
	i = 0;
    CagdVType Dir1, Dir2;
    IPObjectStruct *PRetVal;

    /* Note we never free this "_EvalCurv" attribute so it is essentially */
    /* a memory leak...							  */
    if (AttrGetPtrAttrib(PSrf -> U.Srfs -> Attr, "_EvalCurv") == NULL)
	SymbEvalSrfCurvPrep(PSrf -> U.Srfs, TRUE);

    n = SymbEvalSrfAsympDir(PSrf -> U.Srfs, *U, *V,
			    !REAL_PTR_TO_INT(REuclidean), Dir1, Dir2);

    SymbEvalSrfCurvPrep(PSrf -> U.Srfs, FALSE);
    AttrFreeOneAttribute(&PSrf -> U.Srfs -> Attr, "_EvalCurv");

    PRetVal = IPGenLISTObject(NULL);
    if (n >= 1)
        IPListObjectInsert(PRetVal, i++,
			   IPGenVECObject(&Dir1[0], &Dir1[1], &Dir1[2]));
    if (n >= 2)
	IPListObjectInsert(PRetVal, i++,
			   IPGenVECObject(&Dir2[0], &Dir2[1], &Dir2[2]));
    IPListObjectInsert(PRetVal, i, NULL);

    return PRetVal;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Constructs an arc length approximation to a given.                       M
*                                                                            *
* PARAMETERS:                                                                M
*   PCrv:       Curve to construct an arc length approximation for.          M
*   Fineness:   Tolerance to use is sampling the original curve.	     M
*   ROrder:     Order of expected approximation.                             M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Constructed arc length approximating curve.          M
*                                                                            *
* SEE ALSO:                                                                  M
*   SymbCrvArcLenCrv                                                         M
*                                                                            *
* KEYWORDS:                                                                  M
*   CrvArcLenApprox                                                          M
*****************************************************************************/
IPObjectStruct *CrvArcLenApprox(IPObjectStruct *PCrv,
				RealType *Fineness,
				RealType *ROrder)
{
    CagdCrvStruct
	*Crv = SymbCrvArcLenCrv(PCrv -> U.Crvs,	*Fineness,
				REAL_PTR_TO_INT(ROrder));

    return Crv == NULL ? NULL : IPGenCRVObject(Crv);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Warps the given text, using the current loaded font, in Bezier form in   M
* surface Srf using composition.  The characters of Txt are laid one after   M
* the other until the entire surface is filled horizontally, or characters   M
* in Txt are exhausted.							     M
*                                                                            *
* PARAMETERS:                                                                M
*   PSrf:         Surface to warp the text Txt along.  The text is laid      M
*		  along the u axis with the v axis being the height.         M
*   Txt:          Text to warp insid Srf.				     M
*   HSpace:       Horizonal space between characters.			     M
*   VBase, VTop:  Minimal and maximaal fraaction of height for regular       M
*		  charaacters.  Measured on the letter 'A'.                  M
*   RLigatures:   If TRUE, better handle the ligatures between chars.        M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  Warped text.                                          M
*                                                                            *
* SEE ALSO:                                                                  M
*   UserWarpTextOnSurface, GMLoadTextFont, GMMakeTextGeometry,		     M
*   SymbComposeSrfCrv              			                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   TextWarpThroughSrf                                                       M
*****************************************************************************/
IPObjectStruct *TextWarpThroughSrf(IPObjectStruct *PSrf,
				   char *Txt,
				   RealType *HSpace,
				   RealType *VBase,
				   RealType *VTop,
				   RealType *RLigatures)
{
    return UserWarpTextOnSurface(PSrf -> U.Srfs, Txt, *HSpace,
				 *VBase, *VTop, *RLigatures);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Compute the ray-surface intersection using Bezier clipping.              M
*                                                                            *
* PARAMETERS:                                                                M
*   RayPt:    Ray origin location.					     M
*   RayDir:   Ray direction.						     M
*   PSrf:     Surface to process and intersect.				     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   A list object of the form			     M
*                       list( #Inter, UV0, EucPt0, ..., UVn, EucPtn ).	     M
*                                                                            *
* SEE ALSO:                                                                  M
*   CagdRayTraceBzrSrf                                                       M
*                                                                            *
* KEYWORDS:                                                                  M
*   BezierRayClipping                                                        M
*****************************************************************************/
IPObjectStruct *BezierRayClipping(PointType RayPt,
				  VectorType RayDir,
				  IPObjectStruct *PSrf)
{
    int i = 0;
    CagdRType
	Zero = 0.0;
    IPObjectStruct
	*PRetVal = IPGenLISTObject(NULL);
    CagdUVStruct *IntrPrm, *TmpIntrPrm;
    CagdPtStruct *IntrPt, *TmpIntrPt;

    if (CagdRayTraceBzrSrf(RayPt, RayDir, PSrf -> U.Srfs, &IntrPrm, &IntrPt)) {
        IPListObjectInsert(PRetVal, i++,
			   IPGenNUMValObject(CagdListLength(IntrPrm)));
	for (TmpIntrPrm = IntrPrm, TmpIntrPt = IntrPt;
	     TmpIntrPrm != NULL && TmpIntrPt != NULL;
	     TmpIntrPrm = TmpIntrPrm -> Pnext, TmpIntrPt = TmpIntrPt -> Pnext) {
	    IPListObjectInsert(PRetVal, i++,
			       IPGenPTObject(&TmpIntrPrm -> UV[0],
					     &TmpIntrPrm -> UV[1],
					     &Zero));
	    IPListObjectInsert(PRetVal, i++,
			       IPGenPTObject(&TmpIntrPt -> Pt[0],
					     &TmpIntrPt -> Pt[1],
					     &TmpIntrPt -> Pt[2]));
        }
        CagdUVFreeList(IntrPrm);
        CagdPtFreeList(IntrPt);
    }
    else {
        IPListObjectInsert(PRetVal, i++, IPGenNUMValObject(0));
    }
    IPListObjectInsert(PRetVal, i, NULL);

    return PRetVal;
}


/*****************************************************************************
* DESCRIPTION:                                                               *
* Routine to copy the control mesh lists to a multivariate control mesh.     *
*   The multivariate is allocated here as well.				     *
*   Returns the multivariate if o.k., otherwise NULL.			     *
*                                                                            *
* PARAMETERS:                                                                *
*   n:          Number of dimensions to this multivariates.                  *
*   Lengths:    n lengths in each dimension of the multivariate.             *
*   Orders:     n orders in each dimension of the multivariate.              *
*   PtLstObj:   A list of control points of size                             *
*				(Lengths[0] * Lengths[1] * .. Lengths[n-1]). *
*   GType:      Geometry type - Bezier, Bspline etc.                         *
*   ErrStr:     If an error, detected, this is initialized with description. *
*                                                                            *
* RETURN VALUE:                                                              *
*   IPObjectStruct *:   A trivar object if successful, NULL otherwise.       *
*****************************************************************************/
static IPObjectStruct *GetControlMVMesh(int n,
					int *Lengths,
					int *Orders,
					IPObjectStruct *PtLstObj,
					MvarGeomType GType,
					char **ErrStr)
{
    int i, j, PtSize,
	NumCtlPt = 1;
    CagdRType **r;
    RealType *v;
    IPObjectStruct *MVObj, *PtObj;
    CagdPointType
	PtType = CAGD_PT_E1_TYPE;

    for (i = 0; i < n; i++) {
        if (Lengths[i] < 2) {
	    *ErrStr = IRIT_EXP_STR("Less than two points in some direction");
	    return NULL;
	}

        NumCtlPt *= Lengths[i];
    }

    if (IPListObjectLength(PtLstObj) != NumCtlPt) {
        *ErrStr = IRIT_EXP_STR("Wrong number of control points found");
	return NULL;
    }

    if ((PtType = IPCoerceCommonSpace(PtLstObj, PtType)) == CAGD_PT_NONE) {
        *ErrStr = "";
	return NULL;
    }

    /* Coerce all points to a common space, in place. */
    if (IPCoercePtsListTo(PtLstObj, PtType) == CAGD_PT_NONE) {
        *ErrStr = "";
	return NULL;
    }

    MVObj = IPGenMULTIVARObject(NULL);
    switch (GType) {
	case MVAR_POWER_TYPE:
	    MVObj -> U.MultiVars =  MvarMVNew(n, MVAR_POWER_TYPE,
					      PtType, Lengths);
	    CAGD_GEN_COPY(MVObj -> U.MultiVars -> Orders,
			  MVObj -> U.MultiVars -> Lengths, n * sizeof(int));
	    break;
	case MVAR_BEZIER_TYPE:
	    MVObj -> U.MultiVars = MvarBzrMVNew(n, Lengths, PtType);
	    break;
	case MVAR_BSPLINE_TYPE:
	    MVObj -> U.MultiVars = MvarBspMVNew(n, Lengths, Orders, PtType);
	    break;
	default:
	    break;
    }
    PtSize = CAGD_IS_RATIONAL_PT(PtType) + CAGD_NUM_OF_PT_COORD(PtType);

    for (i = 0, r = MVObj -> U.MultiVars -> Points; i < NumCtlPt; i++) {
        PtObj = IPListObjectGet(PtLstObj, i);

	v = PtObj -> U.CtlPt.Coords;

	if (CAGD_IS_RATIONAL_PT(PtType))
	    for (j = 0; j < PtSize; j++)
	        r[j][i] = *v++;
	else
	    for (j = 1; j <= PtSize; j++)
	        r[j][i] = *++v;
      }

    return MVObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Routine to create a Bezier trivar geometric object defined by a list of  M
* lists of lists of control points.					     M
*                                                                            *
* PARAMETERS:                                                                M
*   LensLstObj:    List of lengths in each of the dimensions of the MV.      M
*   PtLstObj:      A list of control points of size                          M
*			(LenLstObj[0] * LenLstObj[1] * .. LenLstObj[n-1]).   M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *: A Bezier trivar object if successful, NULL otherwise.  M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenBezierMVObject                                                        M
*****************************************************************************/
IPObjectStruct *GenBezierMVObject(IPObjectStruct *LensLstObj,
				  IPObjectStruct *PtLstObj)
{
    int i,
	n = IPListObjectLength(LensLstObj),
	*Lens = (int *) IritMalloc(sizeof(int *) * n);
    char *ErrStr;
    IPObjectStruct *MVObj;

    for (i = 0; i < n; i++) {
        IPObjectStruct
	    *PTmp = IPListObjectGet(LensLstObj, i);

	if (IP_IS_NUM_OBJ(PTmp))
	    Lens[i] = REAL_TO_INT(PTmp -> U.R);
	else {
	    IritFree(Lens);

	    IRIT_NON_FATAL_ERROR2("MBEZIER: Dimensions' list contains non-numeric");

	    return NULL;
	}
    }

    MVObj = GetControlMVMesh(n, Lens, Lens, PtLstObj,
			     MVAR_BEZIER_TYPE, &ErrStr);

    IritFree(Lens);

    if (MVObj == NULL) {
	IRIT_NON_FATAL_ERROR2("MBEZIER: %s, empty object result", ErrStr);
    }

    return MVObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Routine to create a Bezier trivar geometric object defined by a list of  M
* lists of lists of control points.					     M
*                                                                            *
* PARAMETERS:                                                                M
*   LensLstObj:    List of lengths in each of the dimensions of the MV.      M
*   PtLstObj:      A list of control points of size                          M
*			(LenLstObj[0] * LenLstObj[1] * .. LenLstObj[n-1]).   M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *: A Bezier trivar object if successful, NULL otherwise.  M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenPowerMVObject                                                         M
*****************************************************************************/
IPObjectStruct *GenPowerMVObject(IPObjectStruct *LensLstObj,
				 IPObjectStruct *PtLstObj)
{
    int i,
	n = IPListObjectLength(LensLstObj),
	*Lens = (int *) IritMalloc(sizeof(int *) * n);
    char *ErrStr;
    IPObjectStruct *MVObj;

    for (i = 0; i < n; i++) {
        IPObjectStruct
	    *PTmp = IPListObjectGet(LensLstObj, i);

	if (IP_IS_NUM_OBJ(PTmp))
	    Lens[i] = REAL_TO_INT(PTmp -> U.R);
	else {
	    IritFree(Lens);

	    IRIT_NON_FATAL_ERROR2("MPOWER: Dimensions' list contains non-numeric");

	    return NULL;
	}
    }

    MVObj = GetControlMVMesh(n, Lens, Lens, PtLstObj,
			     MVAR_POWER_TYPE, &ErrStr);

    IritFree(Lens);

    if (MVObj == NULL) {
	IRIT_NON_FATAL_ERROR2("MPOWER: %s, empty object result", ErrStr);
    }

    return MVObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Routine to create a Bspline trivar geometric object defined by a list    M
* of lists of lists of control points.					     M
*                                                                            *
* PARAMETERS:                                                                M
*   LensLstObj:    List of lengths in each of the dimensions of the MV.      M
*   OrdersLstObj:  List of orders in each of the dimensions of the MV.       M
*   PtLstObj:      A list of control points of size                          M
*			(LenLstObj[0] * LenLstObj[1] * .. LenLstObj[n-1]).   M
*   KVLstObj:      A list of knot vector lists in each of the dimensions of  M
*		   the MV.						     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *: A Bspline trivar object if successful, NULL otherwise. M
*                                                                            *
* KEYWORDS:                                                                  M
*   GenBsplineMVObject                                                       M
*****************************************************************************/
IPObjectStruct *GenBsplineMVObject(IPObjectStruct *LensLstObj,
				   IPObjectStruct *OrdersLstObj,
				   IPObjectStruct *PtLstObj,
				   IPObjectStruct *KVLstObj)
{
    int i,
	n = IPListObjectLength(LensLstObj),
	*Lens = (int *) IritMalloc(sizeof(int *) * n),
	*Orders = (int *) IritMalloc(sizeof(int *) * n);
    char *ErrStr;
    IPObjectStruct *MVObj;

    if (IPListObjectLength(OrdersLstObj) != n ||
	IPListObjectLength(KVLstObj) != n) {
	IritFree(Lens);
	IritFree(Orders);

	IRIT_NON_FATAL_ERROR2("MBSPLINE: Length of orders and lengths list not equal");

	return NULL;
    }

    for (i = 0; i < n; i++) {
        IPObjectStruct
	    *PTmp1 = IPListObjectGet(LensLstObj, i),
	    *PTmp2 = IPListObjectGet(OrdersLstObj, i);

	if (IP_IS_NUM_OBJ(PTmp1) && IP_IS_NUM_OBJ(PTmp2)) {
	    Lens[i] = REAL_TO_INT(PTmp1 -> U.R);
	    Orders[i] = REAL_TO_INT(PTmp2 -> U.R);
	}
	else {
	    IritFree(Lens);
	    IritFree(Orders);

	    IRIT_NON_FATAL_ERROR2("MBSPLINE: lengths/orders list contains non-numeric");

	    return NULL;
	}

	if (Lens[i] < 2 || Orders[i] < Lens[i]) {
	    IritFree(Lens);
	    IritFree(Orders);

	    IRIT_NON_FATAL_ERROR2("MBSPLINE: Invalid length or order");

	    return NULL;
	}
    }

    MVObj = GetControlMVMesh(n, Lens, Orders, PtLstObj,
			     MVAR_BSPLINE_TYPE, &ErrStr);

    if (MVObj == NULL) {
        IritFree(Lens);
	IritFree(Orders);

	IRIT_NON_FATAL_ERROR2("MBSPLINE: %s, empty object result.\n", ErrStr);
	return NULL;
    }
    else {
	for (i = 0; i < n; i++) {
	    IritFree(MVObj -> U.MultiVars -> KnotVectors[i]);
	    if ((MVObj -> U.MultiVars -> KnotVectors[i] =
		 GetKnotVector(IPListObjectGet(KVLstObj, i), Orders[i],
			       &Lens[i], &ErrStr, TRUE)) == NULL) {
	        IPFreeObject(MVObj);

		IritFree(Lens);
		IritFree(Orders);

		IRIT_NON_FATAL_ERROR2("MBSPLINE: Knot vector invalid, %s, empty object result.\n",
				      ErrStr);
		return NULL;
	    }
	}
    }

    IritFree(Lens);
    IritFree(Orders);

    return MVObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes the umbilical points of a given surface.                          M
*   The umbilicals are computed as the zeros of the function C = H^2 - K.    M
* C is never nagative and hence we actually solve for dC/du = dC/dv = 0 and  M
* test the values of C there.						     M
*   Hence (We only consider numerator of C which is sufficient for zeros),   M
*      C = H^2 - K = (2FM - EN - GL)^2 - 4(LN - M^2)(EG - F^2).		     M
*                                                                            *
* PARAMETERS:                                                                M
*   PSrfObj:   Surface to compute its umbilical points, if any.              M
*   SubTol:    Subdivision tolerance of computation.			     M
*   NumTol:    Numerical tolerance of computation.			     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  List of umbilical points of PSrfObj, nil if none.     M
*                                                                            *
* SEE ALSO:                                                                  M
*   UserSrfUmbilicalPts                                                      M
*                                                                            *
* KEYWORDS:                                                                  M
*   SrfUmbilicPts                                                            M
*****************************************************************************/
IPObjectStruct *SrfUmbilicPts(IPObjectStruct *PSrfObj,
			      RealType *SubTol,
			      RealType *NumTol)
{
    int i = 0;
    RealType
	R = 0.0;
    MvarPtStruct *MVPt,
        *MVPts = UserSrfUmbilicalPts(PSrfObj -> U.Srfs, *SubTol, *NumTol);
    IPObjectStruct
	*PRetVal = IPGenLISTObject(NULL);

    for (MVPt = MVPts; MVPt != NULL; MVPt = MVPt -> Pnext) {
        IPListObjectInsert(PRetVal, i++,
			   IPGenPTObject(&MVPt -> Pt[0],
					 &MVPt -> Pt[1],
					 &R));
    }
    MvarPtFreeList(MVPts);

    IPListObjectInsert(PRetVal, i, NULL);

    return PRetVal;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Approximates a given freeform curve using piecewise (bi)arcs.            M
*                                                                            *
* PARAMETERS:                                                                M
*   PCrv:      The curve to piecewise biarc approximate.                     M
*   Tolerance: Of approximation.                                             M
*   MaxAngle:  Maximum angular span of arcs allowed.                         M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:    List of arcs with "center" point obj attributes     M
*                                                                            *
* SEE ALSO:                                                                  M
*   SymbCrvBiArcApprox                                                       M
*                                                                            *
* KEYWORDS:                                                                  M
*   CrvBiArcApprox                                                           M
*****************************************************************************/
IPObjectStruct *CrvBiArcApprox(IPObjectStruct *PCrv,
			       RealType *Tolerance,
			       RealType *MaxAngle)
{
    int i;
    SymbArcStruct *Arc,
	*Arcs = SymbCrvBiArcApprox(PCrv -> U.Crvs, *Tolerance, *MaxAngle);
    IPObjectStruct *PObj,
	*ObjArcList = IPGenLISTObject(NULL);

    
    for (Arc = Arcs, i = 0; Arc != NULL; Arc = Arc -> Pnext) {
        CagdCrvStruct *Crv;
	
	if (Arc -> Arc) {
	    CagdPtStruct Start, Center, End;

	    /* It is an arc. */

	    PT_COPY(Start.Pt, Arc -> Pt1);
	    PT_COPY(Center.Pt, Arc -> Cntr);
	    PT_COPY(End.Pt, Arc -> Pt2);

	    Crv = BzrCrvCreateArc(&Start, &Center, &End);

	    PObj = IPGenCRVObject(Crv);

	    AttrSetObjectObjAttrib(PObj, "center",
				   IPGenPTObject(&Arc -> Cntr[0],
						 &Arc -> Cntr[1],
						 &Arc -> Cntr[2]), FALSE);
	    IPListObjectInsert(ObjArcList, i++, PObj);
	}
	else {
	    CagdPtStruct Pt1, Pt2;

	    /* It is a line segment. */

	    PT_COPY(Pt1.Pt, Arc -> Pt1);
	    PT_COPY(Pt2.Pt, Arc -> Pt2);
	    Crv = CagdMergePtPt(&Pt1, &Pt2);

	    PObj = IPGenCRVObject(Crv);

	    IPListObjectInsert(ObjArcList, i++, PObj);
	}
    }

    SymbArcFreeList(Arcs);

    IPListObjectInsert(ObjArcList, i, NULL);

    return ObjArcList;
}


/*****************************************************************************
* DESCRIPTION:                                                               M
* Computes the distance function square between the given two entities.	     M
*   Entities could be curves and/or surfaces.  If a curve and a surface are  M
* given, the curve must be first.					     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj1, PObj2:  Two entities to compute the distance function square      M
*                  between.		                                     M
*   RDistType:     0 for distance vector function,			     M
*		   1 for distance square function,			     M
*		   2 for distance vector projected on the normal of Entity1, M
*		   3 for distance vector projected on the normal of Entity2. M
*		   In cases 2 and 3 the normal is not normalized.            M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   List of intersection locations. If SelfInter is TRUE M
*                       assume PObj1 and PObj2 are the same and search for   M
*			self intersections.				     M
*                                                                            *
* KEYWORDS:                                                                  M
*   DistanceTwoFreeforms                                                     M
*****************************************************************************/
IPObjectStruct *DistanceTwoFreeforms(IPObjectStruct *PObj1,
				     IPObjectStruct *PObj2,
				     RealType *RDistType)
{
    IPObjectStruct
	*DistObj = NULL;

    if (IP_IS_CRV_OBJ(PObj1)) {
        if (IP_IS_CRV_OBJ(PObj2)) {
	    CagdSrfStruct
		*DSqrSrf = SymbSrfDistCrvCrv(PObj1 -> U.Crvs, PObj2 -> U.Crvs,
					     REAL_PTR_TO_INT(RDistType));

	    DistObj = IPGenSRFObject(DSqrSrf);
	}
	else if (IP_IS_SRF_OBJ(PObj2)) {
	    MvarMVStruct
		*DSqrMV = MvarMVDistCrvSrf(PObj1 -> U.Crvs, PObj2 -> U.Srfs,
					   REAL_PTR_TO_INT(RDistType));

	    DistObj = IPGenMULTIVARObject(DSqrMV);
	}
    }
    else if (IP_IS_SRF_OBJ(PObj1)) {
        if (IP_IS_SRF_OBJ(PObj2)) {
	    MvarMVStruct
		*DSqrMV = MvarMVDistSrfSrf(PObj1 -> U.Srfs, PObj2 -> U.Srfs,
					   REAL_PTR_TO_INT(RDistType));

	    DistObj = IPGenMULTIVARObject(DSqrMV);
	}
    }

    if (DistObj == NULL) {
        IRIT_NON_FATAL_ERROR2("DIST2FF: None curves/surfaces as input");
    }

    return DistObj;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Compares two objects up to rigid motion and scale in the XY plane.         *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj1, PObj2:  The two objects to compare.                               *
*   Eps:           Tolerance of equality.		                     *
*                                                                            *
* RETURN VALUE:                                                              *
*   IPObjectStruct *:   A numeric zero object if failed or a list object     *
*                       with the Trans point, and scale and rotate factors.  *
*                                                                            *
* KEYWORDS:                                                                  *
*   FreeformCompareUptoRigidScale2D                                          *
*****************************************************************************/
IPObjectStruct *FreeformCompareUptoRigidScale2D(IPObjectStruct *PObj1,
						IPObjectStruct *PObj2,
						CagdRType *Eps)
{
    PointType Trans;
    CagdRType Rot;
    CagdRType Scl;
    IPObjectStruct *PRet;
			     
    if (PObj1 -> ObjType != PObj2 -> ObjType)
	return IPGenNUMValObject(0);

    switch (PObj1 -> ObjType) {
	case IP_OBJ_CURVE:
	    if (!CagdCrvsSameUptoRigidScl2D(PObj1 -> U.Crvs,
					    PObj2 -> U.Crvs,
					    Trans, &Rot, &Scl, *Eps))
	        return IPGenNUMValObject(0);
	    break;
	case IP_OBJ_SURFACE:
	    if (!CagdSrfsSameUptoRigidScl2D(PObj1 -> U.Srfs,
					    PObj2 -> U.Srfs,
					    Trans, &Rot, &Scl, *Eps))
	        return IPGenNUMValObject(0);
	    break;
	default:
	    return IPGenNUMValObject(0);
    }

    PRet = IPGenLISTObject(IPGenPTObject(&Trans[0], &Trans[1], &Trans[2]));
    IPListObjectInsert(PRet, 1, IPGenNUMValObject(Rot));
    IPListObjectInsert(PRet, 2, IPGenNUMValObject(Scl));
    IPListObjectInsert(PRet, 3, NULL);

    return PRet;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Computes a piecewise-cubics polynomial approximations to a given curve.  M
*                                                                            *
* PARAMETERS:                                                                M
*   Crv:        Curve to approximate as piecewise cubics.                    M
*   Tol:        Tolerance of approximation.			             M
*   MaxLen:     An optional limit on the length of the cubics segements.     M
*		Non positive value to ignore.				     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A list of cubic segments.                             M
*                                                                            *
* SEE ALSO:                                                                  M
*   BzrApproxBzrCrvAsCubics, ApproxCrvAsQuadratics                           M
*                                                                            *
* KEYWORDS:                                                                  M
*   ApproxCrvAsCubics                                                        M
*****************************************************************************/
IPObjectStruct *ApproxCrvAsCubics(IPObjectStruct *Crv,
				  RealType *Tol,
				  RealType *MaxLen)
{
    CagdCrvStruct
	*CubicCrvs = BzrApproxBzrCrvAsCubics(Crv -> U.Crvs, *Tol,
				  *MaxLen <= 0 ? IRIT_INFNTY : *MaxLen);

    return IPLnkListToListObject(CubicCrvs, IP_OBJ_CURVE);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Computes a piecewise-quadratics polynomial approximations to a given     M
* curve.							             M
*                                                                            *
* PARAMETERS:                                                                M
*   Crv:        Curve to approximate as piecewise quadratics.                M
*   Tol:        Tolerance of approximation.			             M
*   MaxLen:     An optional limit on the length of the quadratics segements. M
*		Non positive value to ignore.				     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  A list of quadratic segments.                         M
*                                                                            *
* SEE ALSO:                                                                  M
*   BzrApproxBzrCrvAsQuadratics, ApproxCrvAsCubics                           M
*                                                                            *
* KEYWORDS:                                                                  M
*   ApproxCrvAsQuadratics                                                    M
*****************************************************************************/
IPObjectStruct *ApproxCrvAsQuadratics(IPObjectStruct *Crv,
				      RealType *Tol,
				      RealType *MaxLen)
{
    CagdCrvStruct
	*QuadraticCrvs = BzrApproxBzrCrvAsQuadratics(Crv -> U.Crvs, *Tol,
				  *MaxLen <= 0 ? IRIT_INFNTY : *MaxLen);

    return IPLnkListToListObject(QuadraticCrvs, IP_OBJ_CURVE);
}
