/*****************************************************************************
* Filter to convert Wavefront's OBJ data files to IRIT .irt files.	     *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
* Written by:  Gershon Elber				Ver 1.0, Apr 1992    *
*****************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <string.h>
#include "irit_sm.h"
#include "iritprsr.h"
#include "allocate.h"
#include "attribut.h"
#include "geom_lib.h"
#include "misc_lib.h"

#define OBJ_LINE_LEN	10000
#define REALLOC_LIST(Lst, LstSize, Struct) { \
	     Lst = (Struct *) IritRealloc(Lst, LstSize * sizeof(Struct), \
					       LstSize * sizeof(Struct) * 2); \
             LstSize *= 2; }

#ifdef NO_CONCAT_STR
STATIC_DATA char *VersionStr =
	"obj2irit		Version 9.5,		Gershon Elber,\n\
	 (C) Copyright 1989-2005 Gershon Elber, Non commercial use only.";
#else
STATIC_DATA char *VersionStr = "obj2irit	" IRIT_VERSION ",	Gershon Elber,	"
	__DATE__ ",   " __TIME__ "\n" IRIT_COPYRIGHT ", Non commercial use only.";
#endif /* NO_CONCAT_STR */

typedef struct ObjVStruct {
    double x, y, z, w;
} ObjVStruct;
struct ObjVStruct *ObjVList, *ObjV;

typedef struct ObjVPStruct {
    double u, v, w;
} ObjVPStruct;
struct ObjVPStruct *ObjVPList, *ObjVP;

typedef struct ObjVNStruct {
    double Nx, Ny, Nz;
} ObjVNStruct;
struct ObjVNStruct *ObjVNList, *ObjVN;

typedef struct ObjVTStruct {
    double u, v, w;
} ObjVTStruct;
struct ObjVTStruct *ObjVTList, *ObjVT;

typedef struct ColorTableStruct {
    char *Name;
    unsigned char Red, Green, Blue;
    RealType Transp;
} ColorTableStruct;

STATIC_DATA int
    ObjVListSize = 1000,
    ObjVPListSize = 1000,
    ObjVNListSize = 1000,
    ObjVTListSize = 1000,
    ObjVListIndex = 0,
    ObjVPListIndex = 0,
    ObjVNListIndex = 0,
    ObjVTListIndex = 0,
    GlblLineCount = 0,
    GlblReversePolys = FALSE,
    GlblMoreFlag = FALSE;

STATIC_DATA unsigned char
    *GlblColor;

STATIC_DATA char
    *CtrlStr = "obj2irit m%- r%- o%-OutName!s z%- OBJFile!*s";

STATIC_DATA ColorTableStruct GlblPredefColorTable[] = {
#   include "irt_clr.h"
};
STATIC_DATA ColorTableStruct *GlblColorTable;
STATIC_DATA char UnGetLine[OBJ_LINE_LEN];

STATIC_DATA int
    GlblColorTableIndex = 0,
    GlblColorTableSize = 2 * sizeof(GlblPredefColorTable) /
                                                     sizeof(ColorTableStruct);
STATIC_DATA RealType
    GlblTransp = 0.0;

static char *GetLine(FILE **OBJFile, FILE **MTLFile);
static void DumpAndFreeObj(IPObjectStruct *PObj, int DATHandle);
static void GetNewMaterial(FILE **OBJFile, FILE **MTLFile, char *Line);
static unsigned char *GetMaterial(char *Line, RealType *Transp);
static IPPolygonStruct *InsertLine(char *Line);
static IPPolygonStruct *InsertFace(char *Line);
static void InsertVertex(char *Line);
static void InsertCurveSurface(char *Line);
static void Obj2IritExit(int ExitCode);

/*****************************************************************************
* DESCRIPTION:                                                               M
* Main module of dat2irit - Read command line and do what is needed...	     M
*                                                                            *
* PARAMETERS:                                                                M
*   argc, argv:  Command line.                                               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   main                                                                     M
*****************************************************************************/
void main(int argc, char **argv)
{
    int Error,
	ProcessingFaces = FALSE,
	DATHandle = -1,
	VerFlag = FALSE,
	OutFileFlag = FALSE,
	NumFiles = 0;
    char *p, ObjName[LINE_LEN],
	*Line,
	*OutFileName = NULL,
	**FileNames = NULL;
    FILE *OBJFile,
	*MTLFile = NULL;
    IPPolygonStruct *Pl;
    IPObjectStruct
	*CurrPObj = NULL;

#ifdef DEBUG_IRIT_MALLOC
    IritInitTestDynMemory();
#endif /* DEBUG_IRIT_MALLOC */

    if ((Error = GAGetArgs(argc, argv, CtrlStr, &GlblMoreFlag,
			   &GlblReversePolys, &OutFileFlag,
			   &OutFileName, &VerFlag,
			   &NumFiles, &FileNames)) != 0) {
	GAPrintErrMsg(Error);
	GAPrintHowTo(CtrlStr);
	Obj2IritExit(1);
    }

    if (VerFlag) {
	fprintf(stderr, "\n%s\n\n", VersionStr);
	GAPrintHowTo(CtrlStr);
	Obj2IritExit(0);
    }

    if (!NumFiles) {
	fprintf(stderr, "No data file names were given, exit.\n");
	GAPrintHowTo(CtrlStr);
	Obj2IritExit(1);
    }
    else if (NumFiles > 1) {
	fprintf(stderr, "Cannot handle more than one OBJ file at a time, exit.\n");
	GAPrintHowTo(CtrlStr);
	Obj2IritExit(1);
    }

    if (strcmp(FileNames[0], "-") == 0)
	OBJFile = stdin;
    else if ((OBJFile = fopen(FileNames[0], "r")) == NULL) {
	fprintf(stderr, "Cannot open OBJ file \"%s\", exit.\n", FileNames[0]);
	Obj2IritExit(1);
    }

    if (OutFileName != NULL) {
	if ((DATHandle = IPOpenDataFile(OutFileName, FALSE, TRUE)) < 0) {
	    fprintf(stderr, "Failed to open \"%s\".\n", OutFileName);
	    Obj2IritExit(2);
	}
    }
    else
        DATHandle = IPOpenStreamFromFile(stdout, FALSE, FALSE, FALSE, FALSE);

    ObjVList = IritMalloc(sizeof(ObjVStruct) * ObjVListSize);
    ObjVPList = IritMalloc(sizeof(ObjVPStruct) * ObjVPListSize);
    ObjVNList = IritMalloc(sizeof(ObjVNStruct) * ObjVNListSize);
    ObjVTList = IritMalloc(sizeof(ObjVTStruct) * ObjVTListSize);

    ObjName[0] = 0;
    UnGetLine[0] = 0;

    GlblColorTable = (ColorTableStruct *) IritMalloc(GlblColorTableSize *
						     sizeof(ColorTableStruct));
    GEN_COPY(GlblColorTable, GlblPredefColorTable,
	     sizeof(GlblPredefColorTable));
    GlblColorTableIndex = sizeof(GlblPredefColorTable) /
						    sizeof(ColorTableStruct);

    GlblColor = GetMaterial("white", &GlblTransp);

    while ((Line = GetLine(&OBJFile, &MTLFile)) != NULL) {
	switch (Line[0]) {
	    case 0:
	    case '#':
	    case ' ':
		break;
	    case 'c':
		if (strnicmp(Line, "cstype", 6) == 0) {
		    InsertCurveSurface(Line);
		}
		break;
	    case 'f':
		/* Face record. */
		ProcessingFaces = TRUE;
		if (CurrPObj != NULL &&
		    (!IP_IS_POLY_OBJ(CurrPObj) ||
		     !IP_IS_POLYGON_OBJ(CurrPObj))) {
		    DumpAndFreeObj(CurrPObj, DATHandle);
		    CurrPObj = NULL;
		}

		if ((Pl = InsertFace(Line)) != NULL) {
		    if (CurrPObj == NULL) {
			CurrPObj = IPGenPOLYObject(Pl);
			if (strlen(ObjName) > 0)
			    IP_SET_OBJ_NAME2(CurrPObj, ObjName);
			IP_SET_POLYGON_OBJ(CurrPObj);
		    }
		    else {
			Pl -> Pnext = CurrPObj -> U.Pl;
			CurrPObj -> U.Pl = Pl;
		    }
		}
		break;
	    case 'g':
	        /* Grouping. */
		if (strlen(Line) > 2)
		    strncpy(ObjName, &Line[2], LINE_LEN - 2);
		ObjName[LINE_LEN - 2] = 0;
		if ((p = strchr(ObjName, ' ')) != NULL ||
		    (p = strchr(ObjName, '\t')) != NULL)
		    *p = 0;

		if (CurrPObj != NULL) { /* Force its dumping out. */
		    DumpAndFreeObj(CurrPObj, DATHandle);
		    CurrPObj = NULL;
		}
		break;
	    case 'l':
		if (strnicmp(Line, "lod", 3) == 0) {
		    fprintf(stderr,
			    "Level of Details %s requested\n",
			    &Line[3]);
		    break;
		}

		/* Line record. */
		if (CurrPObj != NULL &&
		    (!IP_IS_POLY_OBJ(CurrPObj) ||
		     !IP_IS_POLYLINE_OBJ(CurrPObj))) {
		    DumpAndFreeObj(CurrPObj, DATHandle);
		    CurrPObj = NULL;
		}

		if ((Pl = InsertLine(Line)) != NULL) {
		    if (CurrPObj == NULL) {
			CurrPObj = IPGenPOLYObject(Pl);
			if (strlen(ObjName) > 0)
			    IP_SET_OBJ_NAME2(CurrPObj, ObjName);
			IP_SET_POLYLINE_OBJ(CurrPObj);
		    }
		    else {
			Pl -> Pnext = CurrPObj -> U.Pl;
			CurrPObj -> U.Pl = Pl;
		    }
		}
		break;
	    case 'm':
		if (strnicmp(Line, "mtllib", 6) == 0) {
		    fprintf(stderr,
			    "Material library \"%s\" is processed\n",
			    &Line[7]);
		    if ((MTLFile = fopen(&Line[7], "r")) == NULL) {
			fprintf(stderr, "Cannot open MTL file \"%s\", ignored.\n",
				&Line[7]);
		    }
		}
		else if (strnicmp(Line, "maplib", 6) == 0) {
		    fprintf(stderr,
			    "Texture library \"%s\" is processed\n",
			    &Line[7]);
		    if ((MTLFile = fopen(&Line[7], "r")) == NULL) {
			fprintf(stderr, "Cannot open TXTR file \"%s\", ignored.\n",
				&Line[7]);
		    }
		}

	        /* Merging/Mapping group. */
		break;
	    case 'n':
		if (strnicmp(Line, "newmtl", 6) == 0)
		    GetNewMaterial(&OBJFile, &MTLFile, Line);
		break;
	    case 'o':
	        /* Object Name. */
		if (strlen(Line) > 2)
		    strncpy(ObjName, &Line[2], LINE_LEN - 2);
		ObjName[LINE_LEN - 2] = 0;
		if ((p = strchr(ObjName, ' ')) != NULL ||
		    (p = strchr(ObjName, '\t')) != NULL)
		    *p = 0;
		break;
	    case 's':
	        /* Smoothing. */
		break;
	    case 'u':
		if (strnicmp(Line, "usemtl", 6) == 0) {
		    GlblColor = GetMaterial(&Line[7], &GlblTransp);
		    if (GlblMoreFlag) {
			fprintf(stderr,
				"Selected color \"%s\" (%d %d %d) of obj \"%s\"\n",
				&Line[7],
				GlblColor[0], GlblColor[1], GlblColor[2],
				ObjName);
		    }
		}
		else if (strnicmp(Line, "usemap", 6) == 0) {
		    fprintf(stderr,
			    "Texture mapping \"%s\" requested and ignored\n",
			    &Line[7]);
		}
		break;
	    case 'v':
		/* Vertex record. */
		if (ProcessingFaces && CurrPObj != NULL) {
		    /* Dump the previous object and clear the lists. */
		    DumpAndFreeObj(CurrPObj, DATHandle);
		    CurrPObj = NULL;
		    ProcessingFaces = FALSE;
		}

	        InsertVertex(Line);
		break;
	    default:
		fprintf(stderr, "Unsupported record: \"%s\"\n", Line);
		break;
	}
    }

    if (CurrPObj != NULL) {
	DumpAndFreeObj(CurrPObj, DATHandle);
	CurrPObj = NULL;
    }

    IPCloseStream(DATHandle, TRUE);
    fclose(OBJFile);

    IritFree(ObjVList);
    IritFree(ObjVPList);
    IritFree(ObjVNList);
    IritFree(ObjVTList);
    IritFree(GlblColorTable);

    Obj2IritExit(0);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Gets one line from the OBJ file and make sure it is lower case only.     *
*                                                                            *
* PARAMETERS:                                                                *
*   OBJFile:   File to read from - main file.                                *
*   MTLFile:   File to read from - material/texture file.                    *
*                                                                            *
* RETURN VALUE:                                                              *
*   char *:    Read line, or NUL if EOF.                                     *
*****************************************************************************/
static char *GetLine(FILE **OBJFile, FILE **MTLFile)
{
    unsigned int i;
    STATIC_DATA char Line[OBJ_LINE_LEN], *p;

    if (UnGetLine[0] != 0) {
	strcpy(Line, UnGetLine);
	UnGetLine[0] = 0;
	return Line;
    }

    if (*MTLFile != NULL) {
	if (!fgets(Line, OBJ_LINE_LEN - 1, *MTLFile)) {
	    fclose(*MTLFile);
	    *MTLFile = NULL;
	}
    }

    if (*MTLFile == NULL) {
	if (!fgets(Line, OBJ_LINE_LEN - 1, *OBJFile))
	    return NULL;
    }

    GlblLineCount++;

    for (i = 0; i < strlen(Line); i++) {
	if (isupper(Line[i]))
	    Line[i] = tolower(Line[i]);
	else if (Line[i] < ' ' && Line[i] != 0x09) {
	    Line[i] = 0;
	    break;
	}
    }

    for (i = 0; isspace(Line[i]) && Line[i] != 0; i++);

    return &Line[i];
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Dumps and free the given object.                                         *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:        Object to dump to file and free.                            *
*   DATHandle:   File to write to.                                           *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void DumpAndFreeObj(IPObjectStruct *PObj, int DATHandle)
{
    switch (PObj -> ObjType) {
	case IP_OBJ_POLY:
	    if (PObj -> U.Pl == NULL) {
		IPFreeObject(PObj);
		return;
	    }
	    break;
	case IP_OBJ_SURFACE:
	    if (PObj -> U.Srfs == NULL) {
		IPFreeObject(PObj);
		return;
	    }
	    break;
	default:
	    fprintf(stderr, "Dumped object of type %d\n", PObj -> ObjType);
	    break;
    }

    AttrSetObjectRGBColor(PObj, GlblColor[0], GlblColor[1], GlblColor[2]);
    if (GlblTransp > 0.0)
        AttrSetObjectRealAttrib(PObj, "transp", GlblTransp);
    IPPutObjectToHandler(DATHandle, PObj);
    IPFreeObject(PObj);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Gets a newmtl record.                                                    *
*                                                                            *
* PARAMETERS:                                                                *
*   OBJFile:   File to read from - main file.                                *
*   MTLFile:   File to read from - material/texture file.                    *
*   Line:      The 'newmtl NameMaterialName' line.                           *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void GetNewMaterial(FILE **OBJFile, FILE **MTLFile, char *Line)
{
    int Red = 0,
	Green = 0,
	Blue = 0,
	Count = 0;

    ColorTableStruct *ColorEntry;

    if (GlblColorTableIndex >= GlblColorTableSize - 2) {
	REALLOC_LIST(GlblColorTable, GlblColorTableSize, ColorTableStruct);
    }

    ColorEntry = &GlblColorTable[GlblColorTableIndex++];

    ColorEntry -> Name = IritStrdup(&Line[7]);
    ColorEntry -> Red = ColorEntry -> Green = ColorEntry -> Blue = 255;
    ColorEntry -> Transp = 0.0;

    if (GlblMoreFlag)
	fprintf(stderr, "Def. new srf material \"%s\" - ", &Line[7]);

    ColorEntry -> Red = ColorEntry -> Green = ColorEntry -> Blue = 0;

    while ((Line = GetLine(OBJFile, MTLFile)) != NULL && strlen(Line) > 0) {
	char *p;
    fprintf(stderr, "%s\n", Line);


	if ((p = strstr(Line, "ka")) != NULL ||
	    (p = strstr(Line, "kd")) != NULL) {
	    float r, g, b;

	    if (sscanf(&p[2], "%f %f %f", &r, &g, &b) == 3) {
		Red   += (int) (255 * r);
		Green += (int) (255 * g);
		Blue  += (int) (255 * b);
		Count++;
	    }
	}
	else if ((p = strstr(Line, "d")) != NULL) {
	    float t;

	    if (sscanf(&p[2], "%f", &t) == 1)
		ColorEntry -> Transp = 1 - t;
	}
	else if ((p = strstr(Line, "tf")) != NULL) {
	    float t1, t2, t3;

	    if (sscanf(&p[2], "%f %f %f", &t1, &t2, &t3) == 3) {
		/* We support scalar transparency only. */
		ColorEntry -> Transp = (t1 + t2 + t3) / 3.0;
	    }
	}
	else if ((p = strstr(Line, "ks")) != NULL ||
		 (p = strstr(Line, "ns")) != NULL ||
		 (p = strstr(Line, "illum")) != NULL ||
		 (p = strstr(Line, "map_Kd")) != NULL) {
	}
	else {
	    strcpy(UnGetLine, Line);
	    break;
	}
    }

    if (Count > 0) {
	ColorEntry -> Red = Red / Count;
	ColorEntry -> Green = Green / Count;
	ColorEntry -> Blue = Blue / Count;
    }

    if (GlblMoreFlag)
	fprintf(stderr, "color (%d, %d, %d) [Transp = %.3f]\n",
		ColorEntry -> Red, ColorEntry -> Green, ColorEntry -> Blue,
		ColorEntry -> Transp);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Converts "usemtl material" to string with RGB color.                     *
*                                                                            *
* PARAMETERS:                                                                *
*   Line:    The line of the usemtl record.                                  *
*   Transp:  The transparency factor of this material.                       *
*                                                                            *
* RETURN VALUE:                                                              *
*   unsigned char *: A static vector of RGB colors (each between 0 and 255). *
*****************************************************************************/
static unsigned char *GetMaterial(char *Line, RealType *Transp)
{
    STATIC_DATA unsigned char Color[3];
    int i;
    RealType Match,
	MaxMatch = 0.0;
    ColorTableStruct
	*BestMatchColor = NULL;

    if (GlblMoreFlag)
	fprintf(stderr, "Material \"%s\" requested", Line);

    for (i = GlblColorTableIndex - 1; i >= 0; i--) {
	if ((Match = IritApproxStrStrMatch(Line, GlblColorTable[i].Name, TRUE))
								  > MaxMatch) {
	    BestMatchColor = &GlblColorTable[i];
	    MaxMatch = Match;
	}
    }

    if (BestMatchColor != NULL) {
        Color[0] = BestMatchColor -> Red;
	Color[1] = BestMatchColor -> Green;
	Color[2] = BestMatchColor -> Blue;
	*Transp = BestMatchColor -> Transp;
	if (stricmp(Line, BestMatchColor -> Name) != 0) {
	    fprintf(stderr,
		   "\nWarning: color \"%s\" was substituted by color \"%s\"",
		   Line, BestMatchColor -> Name);
	}

	if (GlblMoreFlag)
	    fprintf(stderr,
		    ",\n\tcolor (%d, %d, %d) [Transp = %f] selected",
		    (int) Color[0], (int) Color[1], (int) Color[2], *Transp);
    }
    else {
        Color[0] = (int) IritRandom(64, 255);
	Color[1] = (int) IritRandom(64, 255);
	Color[2] = (int) IritRandom(64, 255);
	*Transp = 0.0;

        fprintf(stderr,
		"\nWarning: color \"%s\" was not found,\n\t using random color (%d, %d, %d) instead",
		Line,
		(int) Color[0], (int) Color[1], (int) Color[2]);
    }

    if (GlblMoreFlag)
	fprintf(stderr, "\n");

    return (unsigned char *) Color;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Creates one line from the cited vertices in the given line.              *
*                                                                            *
* PARAMETERS:                                                                *
*   Line:   An "l ..." record of the obj data file - defining one line.      *
*                                                                            *
* RETURN VALUE:                                                              *
*   IPPolygonStruct *:  An IRIT polyline representing the line.              *
*****************************************************************************/
static IPPolygonStruct *InsertLine(char *Line)
{
    int VIndex, VTIndex;
    IPPolygonStruct
	*Pl = NULL;
    char *p;

    Pl = IPAllocPolygon(0, NULL, NULL);
    strtok(Line, " \t\n\r");  /* And skip the "l" */
    while ((p = strtok(NULL, " \t\n\r")) != NULL) {
	if (sscanf(p, "%d/%d", &VIndex, &VTIndex) == 2 ||
	    sscanf(p, "%d", &VIndex) == 1) {
	    IPVertexStruct
	        *V = IPAllocVertex2(NULL);

	    if (--VIndex >= ObjVListIndex)
	      fprintf(stderr, "Vertex index out of range in line %d\n",
		      GlblLineCount);

	    V -> Coord[0] = ObjVList[VIndex].x;
	    V -> Coord[1] = ObjVList[VIndex].y;
	    V -> Coord[2] = ObjVList[VIndex].z;
	    IP_RST_NORMAL_VRTX(V);
	    LIST_PUSH(V, Pl -> PVertex);
	}
    }

    return Pl;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Creates one polygon from the cited vertices in the given line.           *
*                                                                            *
* PARAMETERS:                                                                *
*   Line:   An "f* ..." record of the obj data file - defining one face.     *
*                                                                            *
* RETURN VALUE:                                                              *
*   IPPolygonStruct *:  An IRIT polygon representing the face.               *
*****************************************************************************/
static IPPolygonStruct *InsertFace(char *Line)
{
    int VIndex, VNIndex, VTIndex;
    IPPolygonStruct
	*Pl = NULL;
    char *p;

    switch (Line[1]) {
	case ' ':
	case 8:
	    /* An "f" record */
	    Pl = IPAllocPolygon(0, NULL, NULL);
	    strtok(Line, " \t\n\r");  /* And skip the "fo" */
	    while ((p = strtok(NULL, " \t\n\r")) != NULL) {
		if (sscanf(p, "%d/%d/%d", &VIndex, &VTIndex, &VNIndex) == 3 ||
		    sscanf(p, "%d//%d", &VIndex, &VNIndex) == 2) {
		    IPVertexStruct
			*V = IPAllocVertex2(NULL);

		    if (--VIndex >= ObjVListIndex ||
			--VNIndex >= ObjVNListIndex)
			fprintf(stderr,
				"Vertex index out of range in line %d\n",
				GlblLineCount);

		    V -> Coord[0] = ObjVList[VIndex].x;
		    V -> Coord[1] = ObjVList[VIndex].y;
		    V -> Coord[2] = ObjVList[VIndex].z;
		    V -> Normal[0] = ObjVNList[VNIndex].Nx;
		    V -> Normal[1] = ObjVNList[VNIndex].Ny;
		    V -> Normal[2] = ObjVNList[VNIndex].Nz;
		    IP_SET_NORMAL_VRTX(V);
		    LIST_PUSH(V, Pl -> PVertex);
		}
		else if (sscanf(p, "%d", &VIndex) == 1) {
		    IPVertexStruct
			*V = IPAllocVertex2(NULL);

		    if (--VIndex >= ObjVListIndex)
			fprintf(stderr,
				"Vertex index out of range in line %d\n",
				GlblLineCount);

		    V -> Coord[0] = ObjVList[VIndex].x;
		    V -> Coord[1] = ObjVList[VIndex].y;
		    V -> Coord[2] = ObjVList[VIndex].z;
		    IP_RST_NORMAL_VRTX(V);
		    LIST_PUSH(V, Pl -> PVertex);
		}
		else {
		    fprintf(stderr, "Illegal face format in line %d\n",
			    GlblLineCount);
		}
	    }
	    break;
	case 'o':
	    /* An "fo" record */
	    Pl = IPAllocPolygon(0, NULL, NULL);
	    strtok(Line, " \t\n\r");  /* And skip the "fo" */
	    while ((p = strtok(NULL, " \t\n\r")) != NULL) {
		if (sscanf(p, "%d/%d", &VIndex, &VTIndex) == 2 ||
		    sscanf(p, "%d", &VIndex) == 1) {
		    IPVertexStruct
			*V = IPAllocVertex2(NULL);

		    if (--VIndex >= ObjVListIndex)
			fprintf(stderr,
				"Vertex index out of range in line %d\n",
				GlblLineCount);

		    V -> Coord[0] = ObjVList[VIndex].x;
		    V -> Coord[1] = ObjVList[VIndex].y;
		    V -> Coord[2] = ObjVList[VIndex].z;
		    IP_RST_NORMAL_VRTX(V);
		    LIST_PUSH(V, Pl -> PVertex);
		}
		else {
		    fprintf(stderr, "Illegal face format in line %d\n",
			    GlblLineCount);
		}
	    }
	    break;
	default:
	    fprintf(stderr, "Unsupported \"f%c\" record detected in line %d\n",
		    Line[1], GlblLineCount);
	    break;
    }

    if (Pl -> PVertex == NULL) {
	fprintf(stderr, "Face definition with no vertices in line %d\n",
		GlblLineCount);
	IPFreePolygon(Pl);
	return NULL;
    }

    if (Pl -> PVertex -> Pnext == NULL ||
	Pl -> PVertex -> Pnext -> Pnext == NULL) {
	fprintf(stderr,
		"Face definition with less than three vertices in line %d\n",
		GlblLineCount);
	IPFreePolygon(Pl);
	return NULL;
    }

    if (GlblReversePolys) {
	IPVertexStruct *V;

        IPReverseVrtxList(Pl);
	for (V = Pl -> PVertex; V != NULL; V = V -> Pnext) {
	    VEC_SCALE(V -> Normal, -1.0);
	}
    }

    return Pl;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Process one vertex line (OBJ line starting with a 'v'.   Updates the     *
* global arrays of vertices.						     *
*                                                                            *
* PARAMETERS:                                                                *
*   Line:      To process.                                                   *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void InsertVertex(char *Line)
{
    int i;

    switch (Line[1]) {
	case 8:
	case ' ':
	    if (ObjVListIndex >= ObjVListSize - 2) {
	        REALLOC_LIST(ObjVList, ObjVListSize, ObjVStruct);
	    }
	    ObjV = &ObjVList[ObjVListIndex++];
	    i = sscanf(&Line[2], "%lf %lf %lf %lf",
		       &ObjV -> x, &ObjV -> y, &ObjV -> z, &ObjV -> w);
	    if (i == 3)
	        ObjV -> w = 1.0;
	    else if (i != 4) {
		fprintf(stderr,
			"Failed to read vertex in OBJ file, line %d\n",
			GlblLineCount);
		Obj2IritExit(3);
	    }
	    break;
	case 'p':
	    if (ObjVPListIndex >= ObjVPListSize - 2) {
	        REALLOC_LIST(ObjVPList, ObjVPListSize, ObjVPStruct);
	    }
	    ObjVP = &ObjVPList[ObjVPListIndex++];
	    i = sscanf(&Line[2], "%lf %lf %lf",
		       &ObjVP -> u, &ObjVP -> v, &ObjVP -> w);
	    if (i == 1) {
		ObjVP -> v = 0.0;
		ObjVP -> w = 1.0;
	    }
	    else if (i == 2)
	        ObjVP -> w = 1.0;
	    else if (i != 3) {
		fprintf(stderr,
			"Failed to read parametric space vertex in OBJ file, line %d\n",
			GlblLineCount);
		Obj2IritExit(3);
	    }
	    break;
	case 'n':
	    if (ObjVNListIndex >= ObjVNListSize - 2) {
	        REALLOC_LIST(ObjVNList, ObjVNListSize, ObjVNStruct);
	    }
	    ObjVN = &ObjVNList[ObjVNListIndex++];
	    if (sscanf(&Line[2], "%lf %lf %lf",
		       &ObjVN -> Nx, &ObjVN -> Ny, &ObjVN -> Nz) != 3) {
		fprintf(stderr,
			"Failed to read normal vertex in OBJ file, line %d\n",
			GlblLineCount);
		Obj2IritExit(3);
	    }
	    break;
	case 't':
	    if (ObjVTListIndex >= ObjVTListSize - 2) {
	        REALLOC_LIST(ObjVTList, ObjVTListSize, ObjVTStruct);
	    }
	    ObjVT = &ObjVTList[ObjVTListIndex++];
	    i = sscanf(&Line[2], "%lf %lf %lf",
		       &ObjVT -> u, &ObjVT -> v, &ObjVT -> w);
	    if (i == 1)
	        ObjVT -> v = ObjVT -> w = 1.0;
	    else if (i == 2)
	        ObjVT -> w = 1.0;
	    else if (i != 3) {
		fprintf(stderr,
			"Failed to read texture vertex in OBJ file, line %d\n",
			GlblLineCount);
		Obj2IritExit(3);
	    }
	    break;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Process one curve/surface (OBJ line starting with a 'cstype'.   Dumps    *
* the curve/surface out, immediately.					     *
*                                                                            *
* PARAMETERS:                                                                *
*   Line:      To process.                                                   *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void InsertCurveSurface(char *Line)
{
    int IsRational = FALSE;

    if (strstr(Line, "rat"))
	IsRational = TRUE;

    if (strstr(Line, "bezier")) {
	fprintf(stderr,
		"Unsupported %s curve/surface type \"%s\" in line %d\n",
		IsRational ? "rational" : "polynomial", Line, GlblLineCount);
    }
    else if (strstr(Line, "bspline")) {
	fprintf(stderr,
		"Unsupported %s curve/surface type \"%s\" in line %d\n",
		IsRational ? "rational" : "polynomial", Line, GlblLineCount);
    }
    else {
	fprintf(stderr, "Unsupported curve/surface type \"%s\" in line %d\n",
		Line, GlblLineCount);

    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Obj2Irit exit routine.						     *
*                                                                            *
* PARAMETERS:                                                                *
*   ExitCode:                                                                *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void Obj2IritExit(int ExitCode)
{
    exit(ExitCode);
}

#ifdef DEBUG

/*****************************************************************************
* DESCRIPTION:                                                               *
*    Dummy function to link at debugging time.                               *
*                                                                            *
* PARAMETERS:                                                                *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*                                                                            *
* KEYWORDS:                                                                  *
*****************************************************************************/
void DummyLinkCagdDebug(void)
{
    IPDbg();
}

#endif /* DEBUG */
