/******************************************************************************
* CagdPrim.c - Primitive surfaces' constructors.                 	      *
*******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                 *
*******************************************************************************
* Written by Gershon Elber, Dec. 96.					      *
******************************************************************************/

#include <string.h>
#include "cagd_loc.h"

STATIC_DATA CagdPtStruct
    Origin = { NULL, NULL, { 0.0, 0.0, 0.0 } };

/*****************************************************************************
* DESCRIPTION:                                                               M
*   A surface constructor of a plane, parallel to XY plane at level Zlevel.  M
*                                                                            *
* PARAMETERS:                                                                M
*   MinX, MinY:  Minimum XY coordinates of plane.                            M
*   MaxX, MaxY:  Maximum XY coordinates of plane.                            M
*   ZLevel:      Z level of plane, parallel to the XY plane.                 M
*                                                                            *
* RETURN VALUE:                                                              M
*   CagdSrfStruct *:   Constructed surface.                                  M
*                                                                            *
* SEE ALSO:                                                                  M
*   CagdPrimSphereSrf, CagdPrimTorusSrf, CagdPrimCone2Srf, CagdPrimConeSrf,  M
*   CagdPrimCylinderSrf                                                      M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdPrimPlaneSrf                                                         M
*****************************************************************************/
CagdSrfStruct *CagdPrimPlaneSrf(CagdRType MinX,
				CagdRType MinY,
				CagdRType MaxX,
				CagdRType MaxY,
				CagdRType ZLevel)
{
    CagdPtStruct P11, P12, P21, P22;
    CagdSrfStruct *Srf;

    P11.Pt[2] = P12.Pt[2] = P21.Pt[2] = P22.Pt[2] = ZLevel;

    P11.Pt[0] = MinX;
    P11.Pt[1] = MinY;

    P12.Pt[0] = MinX;
    P12.Pt[1] = MaxY;

    P21.Pt[0] = MaxX;
    P21.Pt[1] = MinY;

    P22.Pt[0] = MaxX;
    P22.Pt[1] = MaxY;

    Srf = CagdBilinearSrf(&P11, &P21, &P12, &P22);

    CAGD_SET_GEOM_TYPE(Srf, CAGD_GEOM_PLANAR);

    return Srf;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   A surface constructor of a plane, parallel to XY plane at level Zlevel.  M
*                                                                            *
* PARAMETERS:                                                                M
*   MinX, MinY:  Minimum XY coordinates of plane.                            M
*   MaxX, MaxY:  Maximum XY coordinates of plane.                            M
*   ZLevel:      Z level of plane, parallel to the XY plane.                 M
*   Order:	 Order of plane that is requested.			     M
*   Len:         Number of control points (via refinement).		     M
*                                                                            *
* RETURN VALUE:                                                              M
*   CagdSrfStruct *:   Constructed surface.                                  M
*                                                                            *
* SEE ALSO:                                                                  M
*   CagdPrimSphereSrf, CagdPrimTorusSrf, CagdPrimCone2Srf, CagdPrimConeSrf,  M
*   CagdPrimCylinderSrf, CagdPrimPlaneSrf                                    M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdPrimPlaneSrfOrderLen                                                 M
*****************************************************************************/
CagdSrfStruct *CagdPrimPlaneSrfOrderLen(CagdRType MinX,
					CagdRType MinY,
					CagdRType MaxX,
					CagdRType MaxY,
					CagdRType ZLevel,
					int Order,
					int Len)
{
    CagdSrfStruct
	*TSrf = CagdPrimPlaneSrf(MinX, MinY, MaxX, MaxY, ZLevel),
	*Srf = CagdSrfDegreeRaiseN(TSrf, Order, Order);

    CagdSrfFree(TSrf);

    if (Len > Order) {
        CagdRType
	    *KV = BspKnotUniformOpen(Len, Order, NULL);
	CagdSrfStruct
	    *RSrf1 = CagdSrfRefineAtParams(Srf, CAGD_CONST_U_DIR, FALSE,
					   &KV[Order], Len - Order),
	    *RSrf2 = CagdSrfRefineAtParams(RSrf1, CAGD_CONST_V_DIR, FALSE,
					   &KV[Order], Len - Order);

	IritFree(KV);
	CagdSrfFree(Srf);
	CagdSrfFree(RSrf1);
	Srf = RSrf2;
    }

    return Srf;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   A surface constructor of a sphere, centered at Center and radius Radius. M
*                                                                            *
* PARAMETERS:                                                                M
*   Center:   of constructed sphere.                                         M
*   Radius:   of constructed sphere.                                         M
*   Rational: If TRUE exact ration sphere is created.  If FALSE an	     M
*		 approximated integral surface is created.		     M
*                                                                            *
* RETURN VALUE:                                                              M
*   CagdSrfStruct *:  A Bspline surface representing a sphere.		     M
*                                                                            *
* SEE ALSO:                                                                  M
*   CagdPrimPlaneSrf, CagdPrimTorusSrf, CagdPrimCone2Srf, CagdPrimConeSrf,   M
*   CagdPrimCylinderSrf							     M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdPrimSphereSrf                                                        M
*****************************************************************************/
CagdSrfStruct *CagdPrimSphereSrf(CagdVType Center,
				 CagdRType Radius,
				 CagdBType Rational)
{
    CagdMType Mat;
    CagdRType TMin, TMax;
    CagdCrvStruct *Arc180, *TCrv,
	*Circle = Rational ? BspCrvCreateUnitCircle() :
			     BspCrvCreateUnitPCircle();
    CagdSrfStruct *Spr;

    CagdCrvDomain(Circle, &TMin, &TMax);
    Arc180 = CagdCrvRegionFromCrv(Circle, TMin, TMin + (TMax - TMin) * 0.5);

    CagdCrvFree(Circle);

    if (Rational)
        MatGenMatRotY1(M_PI * 0.5, Mat);	         /* Map to YZ plane. */
    else
        MatGenMatRotX1(M_PI * 0.5, Mat);	         /* Map to XZ plane. */
    TCrv = CagdCrvMatTransform(Arc180, Mat);
    CagdCrvFree(Arc180);
    Arc180 = TCrv;

    if (Rational)
	Spr = CagdSurfaceRev(Arc180);
    else
        Spr = CagdSurfaceRevPolynomialApprox(Arc180);

    CagdCrvFree(Arc180);

    CagdSrfTransform(Spr, Origin.Pt, Radius);
    CagdSrfTransform(Spr, Center, 1.0);

    CAGD_SET_GEOM_TYPE(Spr, CAGD_GEOM_SPHERICAL);

    return Spr;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   A surface constructor of a torus, centered at Center and radii of        M
* MajorRadius and MinorRadius.						     M
*                                                                            *
* PARAMETERS:                                                                M
*   Center:        of constructed torus.                                     M
*   MajorRadius:   of constructed torus.                                     M
*   MinorRadius:   of constructed torus.                                     M
*   Rational:      If TRUE exact ration sphere is created.  If FALSE an	     M
*		   approximated integral surface is created.		     M
*                                                                            *
* RETURN VALUE:                                                              M
*   CagdSrfStruct *:  A Bspline surface representing a torus.		     M
*                                                                            *
* SEE ALSO:                                                                  M
*   CagdPrimPlaneSrf, CagdPrimSphereSrf, CagdPrimCone2Srf, CagdPrimConeSrf,  M
*   CagdPrimCylinderSrf                                                      M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdPrimTorusSrf                                                         M
*****************************************************************************/
CagdSrfStruct *CagdPrimTorusSrf(CagdVType Center,
				CagdRType MajorRadius,
				CagdRType MinorRadius,
				CagdBType Rational)
{
    CagdVType Trans;
    CagdMType Mat;
    CagdCrvStruct *TCrv,
	*Circle = Rational ? BspCrvCreateUnitCircle() :
			     BspCrvCreateUnitPCircle();
    CagdSrfStruct *Trs;

    PT_RESET(Trans);
    CagdCrvTransform(Circle, Trans, MinorRadius);
    Trans[1] = MajorRadius;
    CagdCrvTransform(Circle, Trans, 1.0);

    MatGenMatRotY1(M_PI * 0.5, Mat);		         /* Map to YZ plane. */
    TCrv = CagdCrvMatTransform(Circle, Mat);
    CagdCrvFree(Circle);
    Circle = TCrv;

    if (Rational)
	Trs = CagdSurfaceRev(Circle);
    else
	Trs = CagdSurfaceRevPolynomialApprox(Circle);

    CagdCrvFree(Circle);

    CagdSrfTransform(Trs, Center, 1.0);

    CAGD_SET_GEOM_TYPE(Trs, CAGD_GEOM_TORUS);

    return Trs;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   A surface constructor of a truncated cone, centered at Center and radii  M
* of MajorRadius and MinorRadius. A MinorRadius of zero would construct a    M
* regular cone.  Otherwise, a truncated cone.  Axis of cone is Z axis.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   Center:        of constructed cone (center of its base).                 M
*   MajorRadius:   of constructed cone.                                      M
*   MinorRadius:   of constructed cone.                                      M
*   Height:	   of constructed cone.                                      M
*   Rational:      If TRUE exact ration sphere is created.  If FALSE an	     M
*		   approximated integral surface is created.		     M
*   Caps:	   Do we want caps (top and/or bottom) for the cone?	     M
*                                                                            *
* RETURN VALUE:                                                              M
*   CagdSrfStruct *:  A Bspline surface representing a cone.		     M
*                                                                            *
* SEE ALSO:                                                                  M
*   CagdPrimPlaneSrf, CagdPrimSphereSrf, CagdPrimTorusSrf, CagdPrimConeSrf,  M
*   CagdPrimCylinderSrf                                                      M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdPrimCone2Srf                                                         M
*****************************************************************************/
CagdSrfStruct *CagdPrimCone2Srf(CagdVType Center,
				CagdRType MajorRadius,
				CagdRType MinorRadius,
				CagdRType Height,
				CagdBType Rational,
				CagdPrimCapsType Caps)
{
    STATIC_DATA CagdPtStruct
	XAxisPt =  { NULL, NULL, { 1.0, 0.0, 0.0 } },
	ZXAxisPt = { NULL, NULL, { 1.0, 0.0, 1.0 } },
	ZAxisPt =  { NULL, NULL, { 0.0, 0.0, 1.0 } };
    CagdSrfStruct *Cone;
    CagdCrvStruct *CrvAux, *CrvAux2, *Cross;

    /* Build the cross section. */
    XAxisPt.Pt[0] = MajorRadius;
    ZXAxisPt.Pt[0] = MinorRadius;
    ZXAxisPt.Pt[2] = Height;
    ZAxisPt.Pt[2] = Height;
    if (APX_EQ(MinorRadius, 0.0)) {
	CrvAux2 = CagdMergePtPt(&XAxisPt, &ZAxisPt);
    }
    else if (Caps != CAGD_PRIM_CAPS_TOP && Caps != CAGD_PRIM_CAPS_BOTH) {
	CrvAux2 = CagdMergePtPt(&XAxisPt, &ZXAxisPt);
    }
    else {
	CrvAux = CagdMergePtPt(&ZXAxisPt, &ZAxisPt);
	CrvAux2 = CagdMergePtCrv(&XAxisPt, CrvAux);
	CagdCrvFree(CrvAux);
    }

    if (Caps == CAGD_PRIM_CAPS_BOTTOM || Caps == CAGD_PRIM_CAPS_BOTH) {
        Cross = CagdMergePtCrv(&Origin, CrvAux2);
	CagdCrvFree(CrvAux2);
    }
    else
        Cross = CrvAux2;

    if (Rational)
	Cone = CagdSurfaceRev(Cross);
    else
	Cone = CagdSurfaceRevPolynomialApprox(Cross);

    CagdCrvFree(Cross);

    CagdSrfTransform(Cone, Center, 1.0);

    CAGD_SET_GEOM_TYPE(Cone, CAGD_GEOM_CONICAL);

    return Cone;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   A surface constructor of a cone, centered at Center, radii of Radius,    M
* and height of Height.	 Axis of cone is Z axis.			     M
*                                                                            *
* PARAMETERS:                                                                M
*   Center:        of constructed cone (center of its base).                 M
*   Radius:        of constructed cone's base.                               M
*   Height:	   of constructed cone.                                      M
*   Rational:      If TRUE exact ration sphere is created.  If FALSE an	     M
*		   approximated integral surface is created.		     M
*   Caps:	   Do we want caps (top and/or bottom) for the cone?	     M
*                                                                            *
* RETURN VALUE:                                                              M
*   CagdSrfStruct *:  A Bspline surface representing a cone.		     M
*                                                                            *
* SEE ALSO:                                                                  M
*   CagdPrimPlaneSrf, CagdPrimSphereSrf, CagdPrimTorusSrf, CagdPrimCone2Srf, M
*   CagdPrimCylinderSrf                                                      M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdPrimConeSrf                                                          M
*****************************************************************************/
CagdSrfStruct *CagdPrimConeSrf(CagdVType Center,
			       CagdRType Radius,
			       CagdRType Height,
			       CagdBType Rational,
			       CagdPrimCapsType Caps)
{
    return CagdPrimCone2Srf(Center, Radius, 0.0, Height, Rational, Caps);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   A surface constructor of a Cylinder, centered at Center, radii of        M
* Radius, and height of Height.	 Axis of cylinder is Z axis.		     M
*                                                                            *
* PARAMETERS:                                                                M
*   Center:        of constructed Cylinder (center of its base).             M
*   Radius:        of constructed Cylinder.                                  M
*   Height:	   of constructed Cylinder.                                  M
*   Rational:      If TRUE exact ration sphere is created.  If FALSE an	     M
*		   approximated integral surface is created.		     M
*   Caps:	   Do we want caps (top and/or bottom) for the cone?	     M
*                                                                            *
* RETURN VALUE:                                                              M
*   CagdSrfStruct *:  A Bspline surface representing a cylinder.	     M
*                                                                            *
* SEE ALSO:                                                                  M
*   CagdPrimPlaneSrf, CagdPrimSphereSrf, CagdPrimTorusSrf, CagdPrimCone2Srf, M
*   CagdPrimConeSrf							     M
*                                                                            *
* KEYWORDS:                                                                  M
*   CagdPrimCylinderSrf                                                      M
*****************************************************************************/
CagdSrfStruct *CagdPrimCylinderSrf(CagdVType Center,
				   CagdRType Radius,
				   CagdRType Height,
				   CagdBType Rational,
				   CagdPrimCapsType Caps)
{
    CagdSrfStruct
	*Cyl = CagdPrimCone2Srf(Center, Radius, Radius,
				Height, Rational, Caps);

    CAGD_SET_GEOM_TYPE(Cyl, CAGD_GEOM_CYLINDRICAL);

    return Cyl;
}
