/********************************************************************

   Module Name:     SimpleConsumerBean.java
   Creation Date:   6/20/98
   Description:     A basic InfoBusDataConsumer that watches for a single
                    MonetaryDataItem on the InfoBus

*********************************************************************/

/*
 * Copyright (c) 1997-1998 Lotus Development Corporation. All Rights Reserved.
 *
 *
 * LOTUS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. LOTUS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 */

import java.awt.*;
import java.applet.*;

import java.beans.*;
import javax.infobus.*;
import java.util.Locale;

/*
*   Remember that being an InfoBus participant exposes a portion of your
*   applet to the world -- a SECURITY issue -- and any code reachable by
*   your IB interfaces must be THREADSAFE
*/

public class SimpleConsumerBean extends Applet 
    implements  InfoBusBean, InfoBusDataConsumer,
                PropertyChangeListener, DataItemChangeListener,
                java.awt.event.ItemListener
{

    //  SECURITY: use a proxy subobject to implement our InfoBusDataConsumer
    //   This interface is handed to other IB participants with requests,
    //   and the proxy subobject minimizes unwanted introspection
    //
    private transient InfoBusDataConsumer     m_consumerProxy;

    // use an InfoBusMemberSupport to hold our InfoBus
    private InfoBusBeanSupport    m_IBHolder;

    // the name of the data we desire
    private String                  m_dataName = "DATA1";

    // variable to hold the data that we are seeking
    private transient Object                  m_data;

    // object for locking reassignments to our data
    private transient Object                  m_dataLock;


	public void init() 
    {
		super.init();

        // following generated by layout tool
		//{{INIT_CONTROLS
		setLayout(null);
		setSize(375,200);
		setBackground(new Color(255));
		label1 = new java.awt.Label("SimpleConsumerBean");
		label1.setBounds(24,12,230,37);
		label1.setFont(new Font("Dialog", Font.BOLD, 18));
		label1.setForeground(new Color(16777215));
		add(label1);
		labelData1 = new java.awt.Label("");
		labelData1.setBounds(24,108,336,48);
		labelData1.setFont(new Font("Dialog", Font.BOLD, 20));
		labelData1.setBackground(new Color(12632256));
		add(labelData1);
		label2 = new java.awt.Label("Get Price In:",Label.RIGHT);
		label2.setBounds(24,60,168,38);
		label2.setFont(new Font("Dialog", Font.PLAIN, 18));
		label2.setForeground(new Color(16777215));
		add(label2);
		unitChoice = new java.awt.Choice();
		unitChoice.addItem("Dollars");
		unitChoice.addItem("Pounds");
		unitChoice.addItem("Yen");
        unitChoice.addItem("Numeric");
		add(unitChoice);
		unitChoice.setBounds(204,60,149,40);
		unitChoice.setFont(new Font("Dialog", Font.PLAIN, 18));
		unitChoice.setBackground(new Color(16777215));
		//}}
        // end generated code
        
        unitChoice.addItemListener(this);

        // the "this" in the following constructor tells the 
        // support to use this object in the source field of all
        // events it issues
        m_IBHolder = new InfoBusBeanSupport ( this );

        constructTransients();
	}
	
    private void readObject(java.io.ObjectInputStream stream)
        throws java.io.IOException, ClassNotFoundException
    {
        stream.defaultReadObject();
        constructTransients();
    }
            

    private void constructTransients()
    {
        // reconstruct transients
        m_consumerProxy = new InfoBusDataConsumerProxy ( this );
        m_dataLock      = new Object();

        // register as a PropertyChangeListener with our IBMSupport
        // so that we are alerted if an external agent successfully
        // changes our InfoBus via setInfoBus
        m_IBHolder.addInfoBusPropertyListener ( this );
    }


    /*
    * In our applet's start method we request a named InfoBus for this applet
    * and join it, all via the IBMemberSupport.joinInfoBus method.  If the join
    * is successful, a side effect will be a PropertyChangeEvent showing our 
    * InfoBus property changing from null to X -- and this in turn causes us
    * to add our ProducerProxy to X in our propertyChange method
    *
    * On many applets (including this one) if a null infoBusName is detected
    * by the start() method the applet joins a default bus (this is a feature
    * of many applets but is not mandated by the InfoBus specification)
    */
    public void start ()
    {
        
        try
        {
            if ( m_IBHolder.getInfoBusName() == null || m_IBHolder.getInfoBusName().equals(""))
            {
                //gets default bus, will set both infoBus and infoBusName props
                m_IBHolder.joinInfoBus( this ); 
            }
            else
            {
                //tells the IBBSupport to reattach to the InfoBus it has a name for
                //  (no effect if already attached, but at this point in code we aren't)
                m_IBHolder.rejoinInfoBus( ); 
            }
        }
        catch (Exception e)
        {
            System.out.println ( "SimpleConsumerBean.init Error "+
                                    "intializing its InfoBus:  " + e);
            System.exit(1);
        }

        // NOTE that my own propertyChange method was called if the joinInfoBus
        //   succeeded, which puts my InfoBusDataConsumer object onto the bus

        // NOTE 2:  initial rendezvous code goes here -- in case the data was
        //  announced on the bus before this applet finished intializing
        /*   ... for example:
            InfoBus tmpbus = m_IBHolder.getInfoBus();
            Object tmpdata = tmpbus.findDataItem( m_dataName ); 
            if (tmpdata != null)
                processData( tmpdata );
        */

    }


    /*
    * In our applet's stop method we tell our IBMemberSupport to leave the 
    * InfoBus it was holding.  If successful, a side effect will be a 
    * PropertyChangeEvent showing our InfoBus property changing from X to 
    * null, which in turn causes us to remove our ConsumerProxy.  
    *
    * Note that we do NOT reset our infoBusName property!  This is so we 
    * rejoin the same bus if we are restarted -- including after 
    * deserialization.
    */
    public void stop ()
    {
        // tell the IBMSupport to leave its bus
        try
        {
            m_IBHolder.leaveInfoBus();
        }
        catch ( PropertyVetoException pve )
        {
            // do nothing -- our InfoBus property is managed by external bean
        }
        catch ( InfoBusMembershipException ibme )
        {
            // do nothing -- simply means we already had null for IB
        }
    }

    //////////////////////////////////////////
    //  InfoBusBean methods not in InfoBusMember
    
    /**
    * Updates the value of InfoBus by setting a new infoBusName.  If the InfoBus
    * cannot be changed, the set will throw an Exception
    * Setting the infoBusName to null causes this bean to leave its current
    * bus, resulting in both the InfoBus and infoBusName properties being null.
    *
    * On many applets (including this one) if a null infoBusName is detected
    * by the start() method the applet joins a default bus (this is a feature
    * of many applets but is not mandated by the InfoBus specification)
    *
    * @exception InfoBusMembershipException if InfoBus cannot be changed
    */
    public void setInfoBusName ( String newName )
        throws InfoBusMembershipException
    {
        m_IBHolder.setInfoBusName( newName );
    }
    
    public String getInfoBusName()
    {
        return m_IBHolder.getInfoBusName();
    }


    ////////////////////////////////////////////
    //  InfoBusMember methods
    ////////////////////////////////////////////

    // Delegate all calls to our InfoBusMemberSupport object, m_IBHolder

    public InfoBus getInfoBus()
    {
        return m_IBHolder.getInfoBus();
    }

    public void setInfoBus (InfoBus b) throws PropertyVetoException
    {
        m_IBHolder.setInfoBus( b );
    }

    public void addInfoBusVetoableListener(VetoableChangeListener l)
    {
        m_IBHolder.addInfoBusVetoableListener ( l );
    }

    public void removeInfoBusVetoableListener(VetoableChangeListener l)
    {
        m_IBHolder.removeInfoBusVetoableListener ( l );
    }
    
     public void addInfoBusPropertyListener(PropertyChangeListener l)
    {
        m_IBHolder.addInfoBusPropertyListener ( l );
    }

    public void removeInfoBusPropertyListener(PropertyChangeListener l)
    {
        m_IBHolder.removeInfoBusPropertyListener ( l );
    }


    /////////////////////////////////////////////////////////
    //  Reproduced versions of InfoBusDataConsumer methods
    //    to be called by ConsumerProxy
    /////////////////////////////////////////////////////////

    /*
    * A PropertyChangeEvent is issued by any InfoBusMember if it
    * is changing the InfoBus to which it is connected.  Since we
    * use an InfoBusMemberSupport object to hold and administer our
    * InfoBus connection, we take advantage of this fact to track
    * a change to our own InfoBus and move our Consumer Proxy
    */
    public void propertyChange ( PropertyChangeEvent pce )
    {
        // Note: this implementation is written for a PropertyChangeListener
        //  that may be listening to multiple properties and/or sources

        String s = pce.getPropertyName();

        // check if the PCE refers to an InfoBus
        if ( s.equalsIgnoreCase("InfoBus") )
        {
            // we are only concerned with moving our ConsumerProxy
            // if an InfoBus PCE came from our own IBMemberSupport!
            if ( pce.getSource() == this )
            {
                Object oldVal = pce.getOldValue();
                Object newVal = pce.getNewValue();

                //note that either old or new value may validly be NULL

                if ( oldVal != null && oldVal instanceof InfoBus )
                {
                    ((InfoBus) oldVal).removeDataConsumer ( m_consumerProxy );
                }

                if ( newVal != null && newVal instanceof InfoBus )
                {
                    ((InfoBus) newVal).addDataConsumer ( m_consumerProxy );
               }

            }
            // put code here if we're watching other objects' IB properties
        }
        // put code here if we're watching other properties in the system
    }


    /*
    * This method delivers availability announcements for all data on this bus
    */
    public void dataItemAvailable ( InfoBusItemAvailableEvent ibe )
    {
        if ( ibe == null )
        {
            return;
        }
        
        // the following line causes us to ignore additional announcements
        // if we already have a data item by the target name
        if ( m_data != null )
        {
            return;
        }

        String s = ibe.getDataItemName();

        // The following code accepts any ImmediateAccess which matches
        //  our target data name without regard for source

        if ( ( s != null ) && s.equals( m_dataName ) )
        {
            // Get a copy of the advertised data
            Object tmpData = ibe.requestDataItem( m_consumerProxy, null );

            //THREADSAFETY: make our update activity threadsafe
            //              after we have retrieved the data
            synchronized ( m_dataLock )
            {
                // test the retrieved data for interfaces we know how to parse
                if ( tmpData instanceof ImmediateAccess )
                {
                    if (  m_data instanceof DataItemChangeManager )
                    { // unregister from an old data item before replacing
                        ((DataItemChangeManager)m_data).
                            removeDataItemChangeListener ( this );
                    }

                    m_data = tmpData;

                    if ( m_data instanceof DataItemChangeManager )
                    {
                        ((DataItemChangeManager)m_data).
                            addDataItemChangeListener ( this );
                    }

                    labelData1.setText ( getLocalizedString() );
                }
                // test for other retrieval interfaces here if we want
            }
        }              
    }
    
    /*
    * This method delivers revoke announcements for all data on this bus
    */
    public void dataItemRevoked ( InfoBusItemRevokedEvent ibe )
    {
        if ( ibe == null )
        {
            return;
        }

        String s = ibe.getDataItemName();

        // first we see if the name of the item revoked matches our data
        if ( ( s != null ) && s.equals( m_dataName ) )
        {
            //next check to see if we were holding a copy
            if ( m_data != null )
            {
                // if we were, see if we can do a source match as w
                //   well as name match
                if ( ! (m_data instanceof DataItem) )
                {  //can't get a data item source, so drop it
                    dropData();
                }
                else
                {  // can get a source, so only drop on source match
                    if ( ibe.getSource() == ((DataItem)m_data).getSource() )
                    {
                        // the name AND the source match, so drop the data
                        dropData();
                    }
                }
            }
        }
    }


    //////////////////////////////////////
    // End of InfoBusDataConsumer methods
    ///////////////////////////////////////
    

    /////////////////////////////////////////////
    //  DataItemChangeListener methods
    /////////////////////////////////////////////
	/**
	* Indicates a changed value in the data item.  A reference to the 
	* data item that changed can be obtained from the event.
	* @param event contains change information
	*/
	public void dataItemValueChanged(DataItemValueChangedEvent event) 
    {
		if ( event.getChangedItem() != m_data )
		{
			return;
		}

        String newDisplayString;

        // THREADSAFETY: activity on match must be threadsafe
        synchronized ( m_dataLock )
        {
            newDisplayString = getLocalizedString();
            labelData1.setText( newDisplayString );
        }

    }

	/**
	* Since we only know how to handle an ImmediateAccess (not a collection)
	* we don't do dataItemAdded
	*/
	public void dataItemAdded(DataItemAddedEvent event) 
	{}

	/**
	* Indicates that an item was deleted from an aggregate data item 
	* (ArrayAccess, a JDK Collection, etc).  A reference to the data item 
	* that was deleted, and a reference to the one that lost it, can be 
	* obtained from the event.
	* @param event contains details of the deletion
	*/
	public void dataItemDeleted(DataItemDeletedEvent event) 
    {
		if ( event.getChangedItem() != m_data )
		{
			return;
		}

        dropData();
    }


	/** 
	* Indicates that an item (and its sub-items, if any) has been revoked, 
	* and is temporarily unavailable.  A reference to the data item that 
	* was revoked can be obtained from the event.
	* @param event contains details of the revoked data
	*/
	public void dataItemRevoked(DataItemRevokedEvent event) 
    {
		if ( event.getChangedItem() != m_data )
		{
			return;
		}

        dropData();
    }


	/**
	* We don't handle RowsetAccess items, so no implementation here
	* @param event contains details of the cursor move
	*/
	public void rowsetCursorMoved(RowsetCursorMovedEvent event) {}
	
	////////////////////////////////////////////////////////
	// End of DataItemChangeListener methods
	////////////////////////////////////////////////////////
	
    /*
    * Called by dataItemAvailable and dataItemValueChanged
    */
    private String getLocalizedString ()
    {
        Locale lcode;
        String s = null;
        
        int i = unitChoice.getSelectedIndex();

        // i = 3 indicates we want raw numeric string
        if ( i == 3 )
        {
            return ((ImmediateAccess)m_data).getValueAsString();
        }

        // otherwise create a Locale object indicated by the choice
        switch ( i )
        {
            case 0:
                lcode = new Locale ("en", "US");
                break;
                
            case 1:
                lcode = new Locale ("en", "GB" );
                break;
                
            case 2:
                lcode = new Locale ("ja", "JP");
                break;

            default:
                lcode = null;
                break;
        }
        
        try
        {
            s = ((ImmediateAccess)m_data).getPresentationString( lcode );
        }
        catch ( javax.infobus.UnsupportedOperationException e )
        {
            // if our locale is unsupported, use null to request producer's default
            s = ((ImmediateAccess)m_data).getPresentationString( null );
        }
        
        return s;
    }
    

    /*
    * This private method unregisters us as a listener and then drops the data
    */
    private void dropData()
    {
        synchronized ( m_dataLock )
        {
            if ( m_data != null )
            {
                if ( m_data instanceof DataItem )
                {
                    ((DataItem)m_data).release();    
                }
                if ( m_data instanceof DataItemChangeListener )
                {
                    ((DataItemChangeManager) m_data).
                        removeDataItemChangeListener ( this );
                }
                m_data = null; // REALLY drop our reference
            }
        }
    }


    public void itemStateChanged ( java.awt.event.ItemEvent e )
	{
        // THREADSAFETY: other threads may attempt to send new data, etc.
        if ( m_data != null )
        {
            synchronized ( m_dataLock )
            {
                labelData1.setText( getLocalizedString() );
            }
        }
	}
    
    
    public Dimension getMinimumSize()
    {
        return new Dimension( 350, 200 );
    }
    
    
    //{{DECLARE_CONTROLS
	java.awt.Label label1;
	java.awt.Label labelData1;
	java.awt.Label label2;
	java.awt.Choice unitChoice;
	//}}
}
