/********************************************************************

   Module Name:     MonetaryDataItem.java
   Creation Date:   6/20/98
   Description:     ImmediateAccess wrapper for a Float representing
                        US Dollars
*********************************************************************/

/*
 * Copyright (c) 1997-1998 Lotus Development Corporation. All Rights Reserved.
 *
 *
 * LOTUS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. LOTUS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 */

import javax.infobus.*;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Locale;

public class MonetaryDataItem implements ImmediateAccess, DataItem, 
    DataItemChangeManager
{
    // MonetaryDataItem stores a US Dollar value, but can also report it
    // in Yen or British Pounds (5/13/98 exchange rates)
    // constructor note: Locale( ISO 639 Language Code, ISO 3166 Country Code )
    public final Locale     US_LOCALE = new Locale( "en", "US" );
    public final Locale     UK_LOCALE = new Locale( "en", "GB" );
    public final Locale     JP_LOCALE = new Locale( "ja", "JP" );

    protected Float                         m_value;
    protected String                        m_name;
    protected DataItemChangeManagerSupport  m_changeSupport;
    protected InfoBusEventListener          m_source;

    public MonetaryDataItem( Float v, String name, InfoBusEventListener source )
    {
        m_value = v;
        m_name = name;
        m_source = source;
        m_changeSupport = new DataItemChangeManagerSupport( this );
    }

    /** 
    * Sets the Value property and sends a DataItemChangeEvent to 
    * all registered DataItemChangeListeners
    */
    public void setDollarValue( Float v )
    {
        m_value = v;
        m_changeSupport.fireItemValueChanged( this, null );
    }


    /**
    * The MonetaryDataItem is not modifiable by consumers -- so the
    * ImmediateAccess.setValue that we expose throws a notSupported
    * @exception javax.infobus.UnsupportedOperationException
    */
    public void setValue ( Object newValue )
    {
        throw new javax.infobus.UnsupportedOperationException("setValue not supported");
    }


    //Methods from ImmediateAccess interface:
    // getValueAsObject, getValueAsString, getPresentationString
    /**
    * Returns the data as an Object, a clone of the master data
    * held by the MonetaryDataItem
    * @return the data encapsulated in this MonetaryDataItem
    */
    public Object getValueAsObject()
    {
        return m_value;
    }

    /**
    * Returns the data in String form by calling toString on the wrapped Object
    * @return the data encapsulated in this MonetaryDataItem
    */
    public String getValueAsString()
    {
        return m_value != null ? m_value.toString() : new String("");
    }

    /**
    * Returns the data in String form by calling toString on the wrapped Object
    * @return the data encapsulated in this MonetaryDataItem
    */
    public String getPresentationString( java.util.Locale locale)
    {
        // null defaults to US Dollars and dollar-sign
        if ( ( locale == null ) || US_LOCALE.equals( locale ) )
        {
            return new String("$"+ m_value.toString());
        }
        else if ( UK_LOCALE.equals( locale ) )
        {
            float pounds = 0.61f * m_value.floatValue();
            return new String( "\u00A3"     //unicode for pound-sign
                                + pounds ); 
        }
        else if ( JP_LOCALE.equals( locale ) )
        {
            float yen = 133f * m_value.floatValue();
            return new String( "\u00A5"     //unicode for yen-sign
                                + yen );
        }
        else
        {
            throw new javax.infobus.UnsupportedOperationException("Unsupported Locale");
        }
        
    }

    //methods from DataItemChangeManager:  add[remove]DataItemChangeListener, 
    /**
    * Registers a DataItemChagnedListener with the MonetaryDataItem so that
    * future changes will send a DataItemChangeEvent to the Listener
    * @param l the Listener registering for future change notifications
    */
   	public void	addDataItemChangeListener( DataItemChangeListener l )
    {
        // delegate to our DataItemChangeManagerSupport
        m_changeSupport.addDataItemChangeListener( l );
	}

    /**
    * Unregisters l from the list of Listeners to which change notifications
    * are sent
    * @param l the Listener to be removed from the notifications list
    */
	public void	removeDataItemChangeListener(DataItemChangeListener l)
    {
        // delegate to our DataItemChangeManagerSupport
        m_changeSupport.removeDataItemChangeListener( l );
	}

    /**
    * Returns the InfoBusEventListener which offers this DataItem on the
    * InfoBus
    * @return the InfoBusEventListener registered as the source on 
    * this DataItem
    */
    public InfoBusEventListener getSource()
    {
        return m_source;
    }


    /**
    * MonetaryDataItem supports a "Name" property, which returns the data
    * name supplied to the constructor.
    * @param propertyName the name of the property desisred
    * @return the name of the MonetaryDataItem if propertyName = "Name" (case
    * ignored), else returns null
    */
    public Object getProperty( String propertyName )
    {
        if ( "name".equalsIgnoreCase(propertyName) )
        {
            return m_name;
        }
        else return null;
    }

	/**
	* Not supported, no effect.
	*/
	public void release()
	{
	}


}

