/*
==========================================================================

Project:   gpctool - a graphical user interface and development 
           environment for the gpc generic polygon clipper library.

File:      graphics.c
Author:    Alan Murta (email: gpc@cs.man.ac.uk)
Version:   1.01
Date:      11th April 1999

Copyright: (C) 1997-1999, Advanced Interfaces Group,
           University of Manchester.

           This software is free for non-commercial use. It may be copied,
           modified, and redistributed provided that this copyright notice
           is preserved on all copies. The intellectual property rights of
           the algorithms used reside with the University of Manchester
           Advanced Interfaces Group.

           You may not use this software, in whole or in part, in support
           of any commercial product without the express consent of the
           author.

           There is no warranty or other guarantee of fitness of this
           software for any purpose. It is provided solely "as is".

==========================================================================
*/


#include "gpc.h"
#include "appl.h"
#include <float.h>
#include <stdlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#define SHOW_STRIPS    FALSE
#define SHOW_TRIANGLES FALSE
#define SHOW_VERTICES  FALSE

#define PS_A4_X_SIZE   594.0
#define PS_A4_X_CENTRE 297.0
#define PS_A4_Y_CENTRE 421.0
#define EXCESS          1.25
#define GRID_STEPS        50
#define GRID               3
#define SKETCH             4
#define BACKGROUND         5
#define VERTICES           6
#define TRISTRIP           7

GLfloat line_colour[8][3]=
{
  {0.00, 1.00, 0.00}, /* Clip polygon    */
  {1.00, 0.65, 0.00}, /* Subject polygon */
  {1.00, 1.00, 1.00}, /* Result polygon  */
  {0.50, 0.50, 0.50}, /* Grid lines      */
  {1.00, 0.00, 0.00}, /* Sketch lines    */
  {0.00, 0.00, 0.00}, /* Background      */
  {1.00, 1.00, 0.00}, /* Vertices        */
  {0.70, 0.00, 0.00}  /* Tristrip        */
};


void get_layout(gpc_polygon *sp, gpc_polygon *cp)
{
  double x_extent, y_extent, xc, yc, halfspan;
  int    c, v;

  /* Calculate bounding box */
  layout.xl= DBL_MAX;
  layout.xr= -DBL_MAX;
  layout.yb= DBL_MAX;
  layout.yt= -DBL_MAX;

  /* Inspect subject polygon */
  for (c= 0; c < sp->num_contours; c++)
    for (v= 0; v < sp->contour[c].num_vertices; v++)
    {
      if (sp->contour[c].vertex[v].x < layout.xl)
        layout.xl= sp->contour[c].vertex[v].x;
      if (sp->contour[c].vertex[v].x > layout.xr)
        layout.xr= sp->contour[c].vertex[v].x;
      if (sp->contour[c].vertex[v].y < layout.yb)
        layout.yb= sp->contour[c].vertex[v].y;
      if (sp->contour[c].vertex[v].y > layout.yt)
        layout.yt= sp->contour[c].vertex[v].y;
    }

  /* Inspect clip polygon */
  for (c= 0; c < cp->num_contours; c++)
    for (v= 0; v < cp->contour[c].num_vertices; v++)
    {
      if (cp->contour[c].vertex[v].x < layout.xl)
        layout.xl= cp->contour[c].vertex[v].x;
      if (cp->contour[c].vertex[v].x > layout.xr)
        layout.xr= cp->contour[c].vertex[v].x;
      if (cp->contour[c].vertex[v].y < layout.yb)
        layout.yb= cp->contour[c].vertex[v].y;
      if (cp->contour[c].vertex[v].y > layout.yt)
        layout.yt= cp->contour[c].vertex[v].y;
    }

  /* Deal with NULL case */
  if (layout.xl >= layout.xr)
  {
    layout.xl= DEFAULT_XL;
    layout.xr= DEFAULT_XR;
    layout.yb= DEFAULT_YB;
    layout.yt= DEFAULT_YT;
  }

  /* Calculate bounding box extent and centre */

  x_extent= layout.xr - layout.xl;
  y_extent= layout.yt - layout.yb;
  if (x_extent > y_extent)
    layout.extent= EXCESS * x_extent;
  else
    layout.extent= EXCESS * y_extent;
  xc= 0.5 * (layout.xl + layout.xr); 
  yc= 0.5 * (layout.yb + layout.yt); 

  /* Calculate OpenGL orthographic projection parameters */
  
  halfspan= 0.5 * layout.extent;
  layout.gl_xl= xc - halfspan;
  layout.gl_yb= yc - halfspan;
  layout.gl_xr= xc + halfspan;
  layout.gl_yt= yc + halfspan;

  /* Calculate PostScript parameters */

  layout.ps_scale= PS_A4_X_SIZE / layout.extent;
  layout.ps_x_offset= PS_A4_X_CENTRE - layout.ps_scale * xc;
  layout.ps_y_offset= PS_A4_Y_CENTRE - layout.ps_scale * yc;


  /* Update OpenGL canvas axis labels */

  fl_activate_form(opengl_form);
  sprintf(min_label, "(%lf, %lf)", layout.gl_xl, layout.gl_yb);
  fl_set_object_label(min_label_obj, min_label); 
  fl_redraw_object(min_label_obj);

  sprintf(max_label, "(%lf, %lf)", layout.gl_xr, layout.gl_yt);
  fl_set_object_label(max_label_obj, max_label);
  fl_redraw_object(max_label_obj);
  fl_deactivate_form(opengl_form);
}


void draw_vertices(int p)
{
  int c, v;
  double offset;

  offset= layout.extent / 200.0; 
  glColor3fv(line_colour[VERTICES]);
  glLineWidth(2.0);
  for (c= 0; c < poly[p].num_contours; c++)
  {
    for (v= 0; v < poly[p].contour[c].num_vertices; v++)
    {
      glBegin(GL_LINE_LOOP);
      glVertex2d(poly[p].contour[c].vertex[v].x + offset,
                 poly[p].contour[c].vertex[v].y);
      glVertex2d(poly[p].contour[c].vertex[v].x,
                 poly[p].contour[c].vertex[v].y + offset);
      glVertex2d(poly[p].contour[c].vertex[v].x - offset,
                 poly[p].contour[c].vertex[v].y);
      glVertex2d(poly[p].contour[c].vertex[v].x,
                 poly[p].contour[c].vertex[v].y - offset);
      glEnd();
    }
  }
  glLineWidth(1.0);
}


void draw_tristrip(void)
{
  int s, v;

  glColor3fv(line_colour[TRISTRIP]);

  for (s= 0; s < tri.num_strips; s++)
  {
    if (SHOW_STRIPS)
      glColor3d(0.4 + 0.4 * ((float)rand()/(float)RAND_MAX),
                0.4 + 0.4 * ((float)rand()/(float)RAND_MAX),
                0.4 + 0.4 * ((float)rand()/(float)RAND_MAX));

    glBegin(GL_TRIANGLE_STRIP);
    for (v= 0; v < tri.strip[s].num_vertices; v++)
      glVertex2d(tri.strip[s].vertex[v].x, tri.strip[s].vertex[v].y);
    glEnd();

    if (SHOW_TRIANGLES)
    {
      glLineWidth(1.0);
      glColor3d(1.0, 1.0, 1.0);
      glBegin(GL_LINE_STRIP);
      for (v= 0; v < tri.strip[s].num_vertices; v++)
        glVertex2d(tri.strip[s].vertex[v].x, tri.strip[s].vertex[v].y);
      glEnd();
    }


  }
}


void draw_polygon(int p)
{
  int c, v;

  glColor3fv(line_colour[p]);
  if (p == RESULT)
    glLineWidth(2.0);
  else
    glLineWidth(1.0);

  for (c= 0; c < poly[p].num_contours; c++)
  {
    /* Draw holes using dashed lines */
    if (poly[p].hole[c])
    {
      glLineStipple(1, 0x00FF);
      glEnable(GL_LINE_STIPPLE);
    }
    glBegin(GL_LINE_LOOP);
    for (v= 0; v < poly[p].contour[c].num_vertices; v++)
    {
      glVertex2d(poly[p].contour[c].vertex[v].x,
                 poly[p].contour[c].vertex[v].y);
    }
    glEnd();
    glDisable(GL_LINE_STIPPLE);
  }
  glLineWidth(1.0);
}


void draw_grid(void)
{
  double d, step;

  step= layout.extent / (double) GRID_STEPS;
  glColor3fv(line_colour[GRID]);
  glBegin(GL_LINES);
  for (d= layout.gl_xl + step; d < layout.gl_xr; d+= step)
  {
    glVertex2d(d, layout.gl_yb);
    glVertex2d(d, layout.gl_yt);
  }
  for (d= layout.gl_yb + step; d < layout.gl_yt; d+= step)
  {
    glVertex2d(layout.gl_xl, d);
    glVertex2d(layout.gl_xr, d);
  }
  glEnd();
}


void draw_image(void)
{
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  get_layout(&poly[SUBJECT], &poly[CLIP]);
  gluOrtho2D (layout.gl_xl, layout.gl_xr, layout.gl_yb, layout.gl_yt);

  fl_activate_glcanvas(opengl_canvas);
  glClear(GL_COLOR_BUFFER_BIT);
  if (clip_type & 2)
    draw_tristrip();
  draw_grid();
  draw_polygon(SUBJECT); 
  draw_polygon(CLIP); 
  if (clip_type & 1)
  {
    draw_polygon(RESULT);
    if (SHOW_VERTICES) 
      draw_vertices(RESULT);
  }
  glFinish();
  glXSwapBuffers(fl_display, fl_get_canvas_id(opengl_canvas));
}


int canvas_expose(FL_OBJECT *ob, Window win, int w, int h, XEvent *xev,
                  void *ud)
{
  glViewport(0,0, 832, 832);
  glClearColor(0.0,0.0,0.0,0.0);

  draw_image();
  return 0;
}


int canvas_button(FL_OBJECT *ob, Window win, int w, int h, XEvent *xev, 
                  void *ud)
{
  static int v= 0, firstcall= 1;
  static double mx= 0.0, my = 0.0;
  static gpc_vertex_list new_contour;
  static gpc_vertex      new_vertex[1024];
  int    button, rx, ry, i;
  double gx, gy;

  glColor3fv(line_colour[SKETCH]);

  button= xev->xbutton.button;
  fl_activate_glcanvas(opengl_canvas);

  if (button == 1)
  {
    gx= (double) GRID_STEPS / (double) w;
    gy= (double) GRID_STEPS / (double) h;
    rx= (int)(((double)(    xev->xbutton.x) + (0.5 / gx)) * gx);
    ry= (int)(((double)(h - xev->xbutton.y) + (0.5 / gy)) * gy);
    mx= layout.gl_xl + layout.extent * ((double) rx / (gx * (double) w));
    my= layout.gl_yb + layout.extent * ((double) ry / (gy * (double) h));
    new_vertex[v].x= mx;
    new_vertex[v].y= my;
    v++;
  
    if (firstcall)
    {
      glBegin(GL_POINTS);
      glVertex2d(mx, my);
      glEnd();
      glFinish();
      glXSwapBuffers(fl_display, fl_get_canvas_id(opengl_canvas));
      glBegin(GL_LINES);
      glVertex2d(mx, my);
      firstcall= 0;
    }
    else
    {
      glVertex2d(mx, my);
      glEnd();
      glFinish();
      glXSwapBuffers(fl_display, fl_get_canvas_id(opengl_canvas));
      glBegin(GL_LINES);
      glVertex2d(mx, my);
    } 
  }
  else
  {
    /* Finish off the last (degenerate) GL_LINE */
    glVertex2d(mx, my);
    glEnd();

    if (v >= 3)
    {
      /* Store the new contour */
      new_contour.num_vertices= v;
      new_contour.vertex= new_vertex;
      gpc_add_contour(&poly[sketch_poly], &new_contour, FALSE);

      /* Draw the complete shape in the correct colour */
      glColor3fv(line_colour[sketch_poly]);
      glBegin(GL_LINE_LOOP);
      for (i= 0; i < v; i++)
        glVertex2d(new_vertex[i].x, new_vertex[i].y);
      glEnd();
    }
    else
    {
      /* Erase the degenerate contour */ 
      glColor3fv(line_colour[BACKGROUND]);
      glBegin(GL_LINE_STRIP);
      for (i= 0; i < v; i++)
        glVertex2d(new_vertex[i].x, new_vertex[i].y);
      glEnd();
    }
 
    /* Conclude the sketch process */
    glXSwapBuffers(fl_display, fl_get_canvas_id(opengl_canvas));
    fl_deactivate_form(opengl_form);
    fl_activate_form(fd_gpctool->gpctool);
    firstcall= 1;
    v= 0;
    glFinish();
  }

  return 0;
}
