/**
 \brief
  This module implements the gnocl::drawingArea widget.
 \authors
  Peter G. Baum, William J Giddings
 \date 2001-03:
*/

/* user documentation */

/** \page page14 gnocl::drawingArea
  \section sec gnocl::drawingArea
  Implementation of gnocl::drawingArea
  \subsection subsection1 Implemented gnocl::drawingArea Options
  \subsection subsection2 Implemented gnocl::drawingArea Commands

\subsection subsection3 Sample Tcl Script
 \code
   set box [gnocl::box -buttonType 1]
   $box add [gnocl::button -text "%_Do_n't save" -icon "%#No"]
   $box add [gnocl::button -text "%#Cancel"]
   $box add [gnocl::button -text "%#Save"]
   gnocl::window -title "Buttons" -child $box
 \endcode
\subsection subsection4 Produces
 \image html "../pics/drawingArea.png"
*/

/****f* function/drawingArea
 * NAME
 * DESCRIPTION
 * AUTHOR
 *	WJG
 * CREATION DATE
 *	07/08
 * PURPOSE
 * USAGE
 * PARAMETERS
 * COMMANDS
 * OPTIONS
 * EXAMPLE
 * FUNCTION
 * NOTES
 * BUGS
 * SEE ALSO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * 	2008-10: added command, class ?
 * EXAMPLE
 * |hmtl <B>PRODUCES</B><P>
 * |html <image src="../pics/drawingArea.png">
 * SOURCE
 *****
 */

#include "gnocl.h"
#include <gdk/gdkkeysyms.h>
#include <string.h>
#include <assert.h>

/****f* function/expose_event
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	WJG
 * DATE
 * 	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
/* Redraw all of the points when an expose-event occurs. If you do not do this,
 * the drawing area will be cleared. */
static gboolean expose_event ( GtkWidget *area, GdkEventExpose *event, GPtrArray *parray )
{
	guint i, x, y;
	GdkPoint points[5];

	/* Loop through the coordinates, redrawing them onto the drawing area. */

	for ( i = 0; i < parray->len; i = i + 2 )
	{
		x = GPOINTER_TO_INT ( parray->pdata[i] );
		y = GPOINTER_TO_INT ( parray->pdata[i+1] );

		points[0].x = x;   points[0].y = y;
		points[1].x = x + 1; points[1].y = y;
		points[2].x = x - 1; points[2].y = y;
		points[3].x = x;   points[3].y = y + 1;
		points[4].x = x;   points[4].y = y - 1;

		gdk_draw_points ( area->window, area->style->fg_gc[GTK_WIDGET_STATE ( area ) ], points, 5 );
	}

	return TRUE;
}

/*****/

/****f* function/button_pressed
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	WJG
 * DATE
 * 	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
/* Draw a point where the user pressed the mouse and points on each of the
 * four sides of that point. */
static gboolean button_pressed ( GtkWidget *area, GdkEventButton *event, GPtrArray *parray )
{
	gint x = event->x, y = event->y;
	GdkPoint points[5] = { {x, y}, {x + 1, y}, {x - 1, y}, {x, y + 1}, {x, y - 1} };

	gdk_draw_points ( area->window,
					  area->style->fg_gc[GTK_WIDGET_STATE ( area ) ],
					  points, 5 );

	g_print ( "x = %d : y = %d\n", x, y );

	g_ptr_array_add ( parray, GINT_TO_POINTER ( x ) );
	g_ptr_array_add ( parray, GINT_TO_POINTER ( y ) );

	return FALSE;
}

/*****/

/****f* function/motion_notify
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	WJG
 * DATE
 * 	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
/* Draw a point where the moved the mouse pointer while a button was
 * pressed along with points on each of the four sides of that point. */
static gboolean motion_notify ( GtkWidget *area, GdkEventMotion *event, GPtrArray *parray )
{
	gint x = event->x, y = event->y;
	GdkPoint points[5] = { {x, y}, {x + 1, y}, {x - 1, y}, {x, y + 1}, {x, y - 1} };

	gdk_draw_points ( area->window,
					  area->style->fg_gc[GTK_WIDGET_STATE ( area ) ],
					  points, 5 );

	g_ptr_array_add ( parray, GINT_TO_POINTER ( x ) );
	g_ptr_array_add ( parray, GINT_TO_POINTER ( y ) );

	return FALSE;
}

/*****/

/****f* function/key_pressed
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	WJG
 * DATE
 * 	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
/* Clear the drawing area when the user presses the Delete key. */
static gboolean key_pressed ( GtkWidget *area, GdkEventKey *event, GPtrArray *parray )
{
	if ( event->keyval == GDK_Delete )
	{
		gdk_window_clear ( area->window );
		g_ptr_array_remove_range ( parray, 0, parray->len );
	}

	return FALSE;
}

/*****/

/****f* function/drawingAreaFunc
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	PGB
 * DATE
 * 	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
static int drawingAreaFunc ( ClientData data, Tcl_Interp *interp, int objc, Tcl_Obj * const objv[] )
{
	g_print ( "function/drawingAreaFunc\n" );

	static const char *cmds[] = { "draw", "cget", "configure", "delete", "class", "erase", NULL };
	enum cmdIdx {DrawIdx, CgetIdx, ConfigureIdx, DeleteIdx, ClassIdx, EraseIdx };


	static const char *drawOpts[] = {	"point", "points", "line", "lines", "pixbuf", "segments",
									  "rectangle", "arc", "polygon", "trapezoids", "glyph",
									  "glyphTransformed", "layoutLine", "layoutWithColors",
									  "string", "text", "image", NULL
									};

	enum drawOptsIdx { 	PointIdx, PointsIdx, LineIdx, LinesIdx, PixbufIdx, SegmentsIdx,
						RectangleIdx, ArcIdx, PolygonIdx, TrapezoidsIdx, GlypIdx,
						GlyphTransformedIdx, LayoutLineIdx, LayoutWithColorsIdx,
						StringIdx, TextIdx, ImageIdx
					 };

	GtkWidget *area = GTK_WIDGET ( data );

	int idx;
	int idx2;

	if ( objc < 2 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, "command" );
		return TCL_ERROR;
	}

	if ( Tcl_GetIndexFromObj ( interp, objv[1], cmds, "command",
							   TCL_EXACT, &idx ) != TCL_OK )
	{
		return TCL_ERROR;
	}

	switch ( idx )
	{
		case EraseIdx:
			{
				g_print ( "draw $s\n", Tcl_GetString ( objv[2] ) );
			}

			break;
		case DrawIdx:
			{
				g_print ( "draw %s\n", Tcl_GetString ( objv[2] ) );

				if ( Tcl_GetIndexFromObj ( interp, objv[2], drawOpts, "command",
										   TCL_EXACT, &idx2 ) != TCL_OK )
				{
					return TCL_ERROR;
				}

				switch ( idx2 )
				{
					case PointIdx:
						{

							gint x = 50, y = 50;
							GdkPoint points[5] = { {x, y}, {x + 1, y}, {x - 1, y}, {x, y + 1}, {x, y - 1} };


							//gdk_gc_set_rgb_fg_color (GdkGC *gc, const GdkColor *color)

							/* more options parsing here! */

							g_print ( "points %s\n", Tcl_GetString ( objv[3] ) );
							g_print ( "colour %s\n", Tcl_GetString ( objv[4] ) );


							/* then do the business */
							gdk_draw_point ( area->window,
											 area->style->fg_gc[GTK_WIDGET_STATE ( area ) ],
											 x, y );

							gdk_draw_points ( area->window,
											  area->style->fg_gc[GTK_WIDGET_STATE ( area ) ],
											  points, 5 );


						}

						break;

				}

			}

			break;
		case CgetIdx:
			{
				g_print ( "cget\n" );
			}

			break;
		case ConfigureIdx:
			{
				g_print ( "configure\n" );
			}

			break;
		case DeleteIdx:
			{
				g_print ( "delete\n" );
			}

			break;
		case ClassIdx:
			{
				g_print ( "Class = drawingArea\n" );
				Tcl_SetObjResult ( interp, Tcl_NewStringObj ( "drawingArea", -1 ) );
			}

			break;
			/*
			case PointIdx
			case PointsIdx
			case LineIdx
			case LinesIdx
			case PixbufIdx
			case SegmentsIdx
			case RectangleIdx
			case ArcIdx
			case PolygonIdx
			case TrapezoidsIdx
			case GlypIdx
			case GlyphTransformedIdx
			case LayoutLineIdx
			case LayoutWithColorsIdx,
			case StringIdx
			case TextIdx
			case ImageIdx
			*/
	}

	return TCL_OK;
}

/*****/

/****f* function/gnoclDrawingAreaCmd
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	PGB
 * DATE
 * 	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
/* function called to create widget */
int gnoclDrawingAreaCmd (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj * const objv[] )
{
	int       ret;
	GtkWidget *widget;
	GPtrArray *parray;

	/* create and show the wudget */
	parray = g_ptr_array_sized_new ( 5000 );
	widget = gtk_drawing_area_new ();
	gtk_widget_show ( GTK_WIDGET ( widget ) );

	GTK_WIDGET_SET_FLAGS ( widget, GTK_CAN_FOCUS );
	gtk_widget_add_events ( widget, GDK_BUTTON_PRESS_MASK | GDK_BUTTON_MOTION_MASK | GDK_KEY_PRESS_MASK );

	g_signal_connect ( G_OBJECT ( widget ), "button_press_event", G_CALLBACK ( button_pressed ), parray );
	g_signal_connect ( G_OBJECT ( widget ), "motion_notify_event", G_CALLBACK ( motion_notify ), parray );
	g_signal_connect ( G_OBJECT ( widget ), "key_press_event", G_CALLBACK ( key_pressed ), parray );
	g_signal_connect ( G_OBJECT ( widget ), "expose_event", G_CALLBACK ( expose_event ), parray );

	gtk_widget_show ( GTK_WIDGET ( widget ) );

	/* You must do this after the widget is visible because it must first
	 * * be realized for the GdkWindow to be valid! */
	gdk_window_set_cursor ( widget->window, gdk_cursor_new ( GDK_PENCIL ) );


	/* register the new widget for use with the Tcl interpretor */
	return gnoclRegisterWidget ( interp, GTK_WIDGET ( widget ), drawingAreaFunc );

}

/*****/
