/*
 * $Id: DefaultMutableTreeTableNode.java,v 1.2 2008/11/23 17:52:49 dbarashev Exp $
 *
 * Copyright 2004 Sun Microsystems, Inc., 4150 Network Circle,
 * Santa Clara, California 95054, U.S.A. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.jdesktop.swingx.treetable;

import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.Stack;

import javax.swing.tree.TreeNode;

/**
 *
 *@author Karl Schaefer
 */
public class DefaultMutableTreeTableNode extends AbstractMutableTreeTableNode {
    
    /**
     * 
     */
    public DefaultMutableTreeTableNode() {
        super();
    }

    /**
     * @param userObject
     */
    public DefaultMutableTreeTableNode(Object userObject) {
        super(userObject);
    }

    /**
     * @param userObject
     * @param allowsChildren
     */
    public DefaultMutableTreeTableNode(Object userObject, boolean allowsChildren) {
        super(userObject, allowsChildren);
    }

    /**
     * {@inheritDoc}
     */
    public Object getValueAt(int column) {
        return getUserObject();
    }

    /**
     * {@inheritDoc}
     */
    public int getColumnCount() {
        return 1;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isEditable(int column) {
        return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setValueAt(Object aValue, int column) {
        setUserObject(aValue);
    }

    public MutableTreeTableNode getLastLeaf()
    {
      TreeNode current = this;
      int size = current.getChildCount();

      while (size > 0)
        {
          current = current.getChildAt(size - 1);
          size = current.getChildCount();
        }

      return (MutableTreeTableNode) current;
    }
    
    public void removeAllChildren()
    {
      for (int i = getChildCount() - 1; i >= 0; i--)
        remove(i);
    }

    public boolean isRoot() {
        return (parent == null);
    } // isRoot()

    public int getLevel()
    {
      int count = -1;
      TreeNode current = this;

      do
        {
          current = current.getParent();
          count++;
        }
      while (current != null);

      return count;
    }

    
    public TreeNode[] getPath()
    {
      return getPathToRoot(this, 0);
    }
    
    protected TreeNode[] getPathToRoot(TreeNode node, int depth)
    {
      if (node == null)
        {
          if (depth == 0)
            return null;

          return new TreeNode[depth];
        }

      TreeNode[] path = getPathToRoot(node.getParent(), depth + 1);
      path[path.length - depth - 1] = node;
      return path;
    }

    public MutableTreeTableNode getPreviousSibling()
    {
      if (parent == null)
        return null;

      int index = parent.getIndex(this) - 1;

      if (index < 0)
        return null;

      return (MutableTreeTableNode) parent.getChildAt(index);
    }
    
    public boolean isNodeDescendant(TreeNode node)
    {
      if (node == null)
        return false;

      TreeNode current = node;

      while (current != null
             && current != this)
        current = current.getParent();

      return current == this;
    }
    
    
    public Enumeration<TreeNode> preorderEnumeration() {
        return new PreorderEnumeration(this);
    }
    
    static class PreorderEnumeration implements Enumeration<TreeNode>
    {
        TreeNode next;

        Stack childrenEnums = new Stack();

        PreorderEnumeration(TreeNode node)
        {
            next = node;
            childrenEnums.push(node.children());
        }

        public boolean hasMoreElements()
        {
            return next != null;
        }

        public TreeNode nextElement()
        {
            if( next == null )
                throw new NoSuchElementException("No more elements left.");

            TreeNode current = next;

            Enumeration children = (Enumeration) childrenEnums.peek();

            // Retrieves the next element.
            next = traverse(children);

            return current;
        }

        private TreeNode traverse(Enumeration children)
        {
            // If more children are available step down.
            if( children.hasMoreElements() )
            {
                TreeNode child = (TreeNode) children.nextElement();
                childrenEnums.push(child.children());

                return child;
            }

            // If no children are left, we return to a higher level.
            childrenEnums.pop();

            // If there are no more levels left, there is no next
            // element to return.
            if ( childrenEnums.isEmpty() )
                return null;
            else
            {
                return traverse((Enumeration) childrenEnums.peek());
            }
        }
     }
    
}
