/*
 * InetrvalTagHandler.java
 *
 * Created on 26. Februar 2007, 11:52
 *
 */

package net.sourceforge.ganttproject.parser;

import java.awt.Color;
import java.util.Date;
import net.sourceforge.ganttproject.calendar.Interval;
import net.sourceforge.ganttproject.calendar.IntervalManager;
import net.sourceforge.ganttproject.calendar.IntervalType;
import net.sourceforge.ganttproject.util.ColorConvertion;
import org.xml.sax.Attributes;

/**
 * TagHandler for intervals and interval types
 *
 * @author shofmann <sebastian.hofmann@sourcepark.de>
 */
public class IntervalTagHandler implements TagHandler{        
    
    /** Reference to the IntervalManager of the current project */
    private IntervalManager intervalManager;
    
    /**
     * Creates a new instance of IntervalTagHandler
     * @param intervalManager Reference to the IntervalManager of current project
     */
    public IntervalTagHandler(IntervalManager intervalManager) {
        this.intervalManager = intervalManager;
    }
    
    /**
     * Callback-method called by XML-Parser at start of an element. 
     * Parses Elements of type intervalType and Interval.
     * @param namespaceURI namespaceURI
     * @param sName sName
     * @param qName qName
     * @param attrs attrs
     * @throws FileFormatException case of miss-spelled XML
     */
    public void startElement(String namespaceURI, String sName, String qName,
            Attributes attrs) throws FileFormatException {
        if (qName.equals("interval")) {
            handleInterval(attrs);
        }else if(qName.equals("intervalType")){
            handleIntervalType(attrs);
        }
    }
    
    /**
     * Callback-method called by XML-Parser at end of an element. 
     * Parses Elements of type intervalType and Interval.
     * @param namespaceURI namespaceURI
     * @param sName sName
     * @param qName qName     
     */
    public void endElement(String namespaceURI, String sName, String qName) {
        // nothing to do here...
    }
    
    /**
     * Handles parsing the attributs of an IntervalType
     * @param attrs the attributes to parse
     * @throws FileFormatException in case of miss-spelled XML
     */
    private void handleIntervalType(Attributes attrs) throws FileFormatException{
        
        IntervalType type = new IntervalType();
        
        String description = attrs.getValue("description");
        String colorString = attrs.getValue("color");
        String lockedString = attrs.getValue("locked");                

        boolean locked = Boolean.parseBoolean(lockedString);
        
        Color color = null;
        try{
            color = ColorConvertion.hexStringToColor(colorString);
        }catch(Exception e){
            throw new FileFormatException("[Intervalparser] color needs to " +
                    "be specified as hexadecimal value, e.g. #ff00ff");
        }
        
        if(description == null){
            throw new FileFormatException(
                    "[Intervalparser] Missing description for intervalType.");
        }
        
        type.setColor(color);
        type.setDescription(description);
        type.setLocked(locked);
        
        this.intervalManager.addIntervalType(type);
    }
    
    /**
     * Handles parsing the attributs of an Interval
     * @param attrs the attributes to parse
     * @throws FileFormatException in case of miss-spelled XML
     */
    private void handleInterval(Attributes attrs) throws FileFormatException{
        
        Interval interval = new Interval();
        
        String startDateString = attrs.getValue("startdate");
        String endDateString = attrs.getValue("enddate");
        String description = attrs.getValue("description");
        
        if(startDateString == null || endDateString == null || description == null){
            throw new FileFormatException("[Intervalparser] Illegal interval specified");
        }
        
        Date startDate, endDate = null;
        try{
            startDate = IntervalManager.SIMPLE_DATE_FORMAT.parse(startDateString);
            endDate = IntervalManager.SIMPLE_DATE_FORMAT.parse(endDateString);
        }catch(Exception e){
            throw new FileFormatException("[Intervalparser] Illegal syntax for " +
                    "date in interval specified. Must be: "
                    +IntervalManager.SIMPLE_DATE_FORMAT.toPattern());
        }
        
        if(endDate.compareTo(startDate) < 0){
            throw new FileFormatException(
                    "[Intervalparser] Enddate must be after startdate.");
        }

        interval.setEndDate(endDate);
        interval.setStartDate(startDate);
        
        IntervalType type = this.intervalManager.getIntervalTypeByDescription(description);
        
        if(type == null){
            throw new FileFormatException("[Intervalparser] Unknown description for interval: "+description);
        }
        
        interval.setType(type);
        
        this.intervalManager.addInterval(interval);
        
    }

}
