#!/usr/bin/env python
from __future__ import with_statement

import os
import sys
import zipfile
from getopt import getopt
from shutil import copytree, copyfile

def usage(app_name):
    return 'Usage: %s <project name>' % (app_name)

def recursively_rmdir(dirname):
    files = os.listdir(dirname)
    for f in files:
        path = os.path.join(dirname, f)
        if os.path.isdir(path):
            recursively_rmdir(path)
        else:
            os.unlink(path)
    os.rmdir(dirname)

def create_file(file_name, content):
    if not os.path.exists(os.path.dirname(file_name)):
        os.makedirs(os.path.dirname(file_name), 0755)
    with open(file_name, 'w') as f:
        f.write('\n'.join(content))

def create_app_yaml(app_yaml_file, project_name):
    create_file(app_yaml_file, [
        'application: %s' % (project_name),
        'version: 1',
        'api_version: 1',
        'runtime: python',
        '',
        'default_expiration: "1d"',
        '',
        'handlers:',
        '- url: /css',
        '  static_dir: assets/css',
        '- url: /js',
        '  static_dir: assets/js',
        '- url: /img',
        '  static_dir: assets/img',
        '- url: /favicon.ico',
        '  static_files: favicon.ico',
        '  upload: favicon.ico',
        '- url: .*',
        '  script: main.py',
        '',
    ])

def create_controller_base(path):
    create_file(path, [
        '',
        'class Application:',
        '    """',
        '    The base controller init mixin',
        '    """',
        '    def application_init(self):',
        '        """',
        '        default initialization method invoked by dispatcher',
        '        """',
        '        pass'
    ])
    
def create_controller_py(controller_py):
    create_file(controller_py, [
        'from gaeo.controller import BaseController',
        '',
        'class WelcomeController(BaseController):',
        '    """The default Controller',
        '',
        '    You could change the default route in main.py',
        '    """',
        '    def index(self):',
        '        """The default method',
        '',
        '        related to templates/welcome/index.html',
        '        """',
        '        pass',
        '',
    ])

def create_base_template(index_html_file):
    create_file(index_html_file, [
        '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01//EN"',
        '    "http://www.w3.org/TR/html4/strict.dtd">',
        '<html>',
        '    <head>',
        '        <meta http-equiv="Content-Type" content="text/html; charset=UTF-8">',
        '        <title>{% block title %}{% endblock %}</title>',
        '    </head>',
        '    <body>',
        '        {% block content %}{% endblock %}',
        '    </body>',
        '</html>',
        '',
    ])

def create_default_template(index_html_file):
    create_file(index_html_file, [
        '{% extends "../base.html" %}',
        '',
        '{% block title %}GAEO Default Template{% endblock %}',
        '',
        '{% block content %}',
        '        <h1>It works!!</h1>',
        '        You could open <i>application/templates/welcome/index.html</i> to edit this page.',
        '        <!-- this page is related to controller/welcome.py -->',
        '{% endblock %}',
        '',
    ])

def create_eclipse_project(project_home, project_name):
    proj = os.path.join(project_home, '.project')
    pydevproj = os.path.join(project_home, '.pydevproject')
    
    create_file(proj, [
        '<?xml version="1.0" encoding="UTF-8"?>',
        '<projectDescription>',
        '    <name>%s</name>' % project_name,
        '    <comment></comment>',
        '    <projects>',
        '    </projects>',
        '    <buildSpec>',
        '        <buildCommand>',
        '            <name>org.python.pydev.PyDevBuilder</name>',
        '            <arguments>',
        '            </arguments>',
        '        </buildCommand>',
        '    </buildSpec>',
        '    <natures>',
        '        <nature>org.python.pydev.pythonNature</nature>',
        '    </natures>',
        '</projectDescription>'
    ])
    
    create_file(pydevproj, [
        '<?xml version="1.0" encoding="UTF-8" standalone="no"?>',
        '<?eclipse-pydev version="1.0"?>',
        '',
        '<pydev_project>',
        '    <pydev_property name="org.python.pydev.PYTHON_PROJECT_VERSION">python 2.5</pydev_property>',
        '    <pydev_pathproperty name="org.python.pydev.PROJECT_SOURCE_PATH">',
        '        <path>/%s</path>' % project_name,
        '    </pydev_pathproperty>',
        '</pydev_project>'
    ])

def main(argv):
    ignore_exist_proj = False 
    create_eclipse_proj = False
    zipped_core = False

    cur_dir = os.getcwd()

    optlist, args = getopt(argv, '', ['eclipse', 'zipped'])

    for opt, value in optlist:
        if opt == '--eclipse':
            create_eclipse_proj = True
        elif opt == '--zipped':
            zipped_core = True

    project_name = args[0]

    # create project directory
    project_home = os.path.join(cur_dir, project_name)
    if os.path.exists(project_home):
        print '%s exists' % (project_home)
        return
    else:
        os.mkdir(project_home, 0755)

    project_name = os.path.basename(project_name).lower()
    template_base = 'oildrum'

    # create <project_name>/application/__init__.py
    application_dir = os.path.join(project_home, 'application')
    create_file(os.path.join(application_dir, '__init__.py'), [])

    controller_dir = os.path.join(application_dir, 'controller')
    create_file(os.path.join(controller_dir, '__init__.py'), [])
    # create <project_name>/application/controller/application.py
    create_controller_base(os.path.join(controller_dir, 'application.py'))
    # create <project_name>/application/controller/welcome.py
    create_controller_py(os.path.join(controller_dir, 'welcome.py'))

    # create base template
    create_base_template(os.path.join(application_dir, 'templates', 'base.html'))

    # create default template
    create_default_template(os.path.join(application_dir, 'templates', 'welcome', 'index.html'))

    # create blank model module
    model_dir = os.path.join(application_dir, 'model')
    create_file(os.path.join(model_dir, '__init__.py'), [])

    # create app.yaml
    create_app_yaml(os.path.join(project_home, 'app.yaml'), project_name)

    # create assets directories
    assets_dir = os.path.join(project_home, 'assets')
    os.mkdir(assets_dir, 0755)
    for d in ['css', 'img', 'js']:
        target_dir = os.path.join(assets_dir, d)
        os.mkdir(target_dir, 0755)

    # create an empty favicon.ico
    create_file(os.path.join(project_home, 'favicon.ico'), [])

    # copy GAEO directory
    gaeo_folder = os.path.join(project_home, 'gaeo')
    copytree(os.path.join(os.path.dirname(os.path.dirname(__file__)), template_base, 'gaeo'), gaeo_folder)
    
    # if zipped option is enabled
    if zipped_core:
        os.chdir(project_home)
        z = zipfile.ZipFile('gaeo.zip', 'w', zipfile.ZIP_DEFLATED)
        dir_list = []
        for root,dirs,files in os.walk('gaeo'):
            dir_list.extend([d for d in dirs if os.listdir(os.path.join(root, d)) == []])
            for filename in files:
                z.write(os.path.join(root, filename))
        for d in dir_list:
            z_info = zipfile.ZipInfo(os.path.join(root, d))
            z.writestr(z_info, '')
        z.close()
        recursively_rmdir('gaeo')
        os.chdir(os.path.join(project_home, '..'))
    
    # copy plugins directory
    copytree(os.path.join(os.path.dirname(os.path.dirname(__file__)), template_base, 'plugins'), os.path.join(project_home, 'plugins'))

    # copy sample main.py
    copyfile(os.path.join(os.path.dirname(os.path.dirname(__file__)), template_base, 'main-sample.py'), os.path.join(project_home, 'main.py'))

    # create the eclipse project file
    if create_eclipse_proj:
        create_eclipse_project(project_home, project_name)

    print 'The "%s" project has been created.' % project_name

def commandline():
    if len(sys.argv) > 1:
        main(sys.argv[1:])
    else:
        print usage(sys.argv[0]);

#paster version of command, not used yet
try:
    from paste.script import command
    import optparse
    class GaeoCommand(command.Command):
        """Create GAEO project
    
        Example usage::
    
        $ paster gaeo [project name] [--zip] [--eclipse]
        """
        max_args = 3
        min_args = 1
        summary = __doc__.splitlines()[0]
        usage = '\n' + __doc__
        group_name = "GAEO"
    
        parser = command.Command.standard_parser(verbose=True)
        parser = optparse.OptionParser(
                     usage="paster gaeo [project name] [--zip] [--eclipse]")
        """parser.add_option("-p", "--project",
                help="project name for the code",
                dest="project")
        parser.add_option("-z", "--zipped",
                help="zip generated gaeo library",
                dest="zipped")
        parser.add_option("-e", "--eclipse",
                help="generate eclipse project info",
                dest="eclipse")"""

        def command(self):
            self.__dict__.update(self.options.__dict__)
            main(self.args)
except:
    pass    

if __name__ == '__main__':
     commandline()

