/*============================================================================
 * Tracing utility functions for profiling and debugging
 *============================================================================*/

/*
  This file is part of the "Finite Volume Mesh" library, intended to provide
  finite volume mesh and associated fields I/O and manipulation services.

  Copyright (C) 2010  EDF

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/*----------------------------------------------------------------------------
 * Standard C library headers
 *----------------------------------------------------------------------------*/

#include <stdlib.h>

/*----------------------------------------------------------------------------
 * BFT library headers
 *----------------------------------------------------------------------------*/

#include <bft_mem.h>
#include <bft_mem_usage.h>
#include <bft_printf.h>

/*----------------------------------------------------------------------------
 *  Local headers
 *----------------------------------------------------------------------------*/

#include "fvm_defs.h"
#include "fvm_config_defs.h"
#include "fvm_parall.h"

/*----------------------------------------------------------------------------
 *  Header for the current file
 *----------------------------------------------------------------------------*/

#include "fvm_trace.h"

/*----------------------------------------------------------------------------*/

#ifdef __cplusplus
extern "C" {
#if 0
} /* Fake brace to force back Emacs auto-indentation back to column 0 */
#endif
#endif /* __cplusplus */

/*============================================================================
 * Type definitions
 *============================================================================*/

#if defined(FVM_HAVE_MPI)

typedef struct
{
  double val;
  int    rank;
} _fvm_trace_mpi_double_int_t;

#endif

/*============================================================================
 * Static global variables
 *============================================================================*/

/*============================================================================
 * Private function definitions
 *============================================================================*/

/*=============================================================================
 * Public function definitions
 *============================================================================*/

/*----------------------------------------------------------------------------
 * Print memory usage status.
 *
 * If no associated description is given, a call number will be issued.
 *
 * parameters:
 *   descr <-- optional description, or NULL
 *----------------------------------------------------------------------------*/

void
fvm_trace_mem_status(const char  *descr)
{
  int    i, itot;
  double valreal[4];

#if defined(FVM_HAVE_MPI)
  MPI_Comm comm = fvm_parall_get_mpi_comm();
  int rank_id = fvm_parall_get_rank();
  int n_ranks = fvm_parall_get_size();
  int  imax = 0, imin = 0;
  int  id_min[4];
  _fvm_trace_mpi_double_int_t  val_in[4], val_min[4], val_max[4];
#else
  int n_ranks = 1;
#endif

  int   val_flag[4] = {1, 1, 1, 1};
  char  unit[]    = {'k', 'm', 'g', 't', 'p'};

  static int call_id = 0;

  const char  *type_str[] = {"max. measured       ",
                             "max. instrumented   ",
                             "current measured    ",
                             "current instrumented"};

  /* Memory summary */

  if (descr != NULL)
    bft_printf(_("\nMemory use summary: %s\n\n"), descr);
  else
    bft_printf(_("\nMemory use summary (call %d):\n\n"), call_id);

  valreal[0] = (double)bft_mem_usage_max_pr_size();
  valreal[1] = (double)bft_mem_size_max();
  valreal[2] = (double)bft_mem_usage_pr_size();
  valreal[3] = (double)bft_mem_size_current();

  /* Ignore inconsistent measurements */

  for (i = 0; i < 4; i++) {
    if (valreal[i] < 1.0)
      val_flag[i] = 0;
  }

#if defined(FVM_HAVE_MPI)

  if (n_ranks > 1) {

    MPI_Reduce(val_flag, id_min, 4, MPI_INT, MPI_MIN, 0, comm);
    for (i = 0; i < 4; i++) {
      val_in[i].val = valreal[i];
      val_in[i].rank = rank_id;
    }
    MPI_Reduce(&val_in, &val_min, 4, MPI_DOUBLE_INT, MPI_MINLOC, 0, comm);
    MPI_Reduce(&val_in, &val_max, 4, MPI_DOUBLE_INT, MPI_MAXLOC, 0, comm);
    if (rank_id == 0) {
      for (i = 0; i < 4; i++) {
        val_flag[i]  = id_min[i];
        valreal[i] = val_max[i].val;
      }
    }
  }
#endif

  /* Similar handling of several instrumentation methods */

  for (i = 0; i < 4; i++) {

    /* If an instrumentation method returns an apparently consistent
       result, print it. */

    if (val_flag[i] == 1) {

      for (itot = 0;
           valreal[i] > 1024. && unit[itot] != 'p';
           itot++)
        valreal[i] /= 1024.;
#if defined(FVM_HAVE_MPI)
      if (n_ranks > 1 && rank_id == 0) {
        for (imin = 0;
             val_min[i].val > 1024. && unit[imin] != 'p';
             imin++)
          val_min[i].val /= 1024.;
        for (imax = 0;
             val_max[i].val > 1024. && unit[imax] != 'p';
             imax++)
          val_max[i].val /= 1024.;
      }
#endif

      /* Print to log file */

#if defined(FVM_HAVE_MPI)
      if (n_ranks > 1 && rank_id == 0) {
        bft_printf(_("  %s : %10.3f %cb min (rank %d), "
                     " %10.3f %cb max (rank %d)\n"),
                   type_str[i], val_min[i].val, unit[imin], val_min[i].rank,
                   val_max[i].val, unit[imax], val_max[i].rank);
      }
#endif
      if (n_ranks == 1)
        bft_printf(_("  %s : %12.3f %cb\n"),
                   type_str[i], valreal[i], unit[itot]);
    }
  }
}

/*----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif /* __cplusplus */
