////////////////////////////////////////////////////////////////////////////////
//
//  ADOBE SYSTEMS INCORPORATED
//  Copyright 2005-2007 Adobe Systems Incorporated
//  All Rights Reserved.
//
//  NOTICE: Adobe permits you to use, modify, and distribute this file
//  in accordance with the terms of the license agreement accompanying it.
//
////////////////////////////////////////////////////////////////////////////////

package mx.core
{

import flash.display.BitmapData;
import mx.core.FlexBitmap;

/**
 *  BitmapAsset is a subclass of the flash.display.Bitmap class
 *  which represents bitmap images that you embed in a Flex application.
 *  It implements the IFlexDisplayObject interface, which makes it
 *  possible for an embedded bitmap image to be displayed in an Image control,
 *  or to be used as a container background or a component skin.
 *
 *  <p>The bitmap image that you're embedding can be in a JPEG, GIF,
 *  or PNG file.
 *  You can also embed a bitmap symbol that is in a SWF file produced
 *  by Flash.
 *  In each of these cases, the MXML compiler autogenerates a class
 *  that extends BitmapAsset to represent the embedded bitmap image.</p>
 *
 *  <p>You don't generally have to use the BitmapAsset class directly
 *  when you write a Flex application.
 *  For example, you can embed a GIF file and display the image
 *  in an Image control by writing the gollowing:</p>
 *
 *  <pre>
 *  &lt;mx:Image id="logo" source="&#64;Embed(source='Logo.gif')"/&gt;</pre>
 *
 *  <p>or use it as the application's background image in CSS syntax
 *  by writing</p>
 *
 *  <pre>
 *  &lt;mx:Style&gt;
 *      Application
 *      {
 *          backgroundImage: Embed(source="Logo.gif")
 *      }
 *  &lt;mx:Style/&gt;</pre>
 *
 *  <p>without having to understand that the MXML compiler has created
 *  a subclass of BitmapAsset for you.</p>
 *
 *  <p>However, it may be useful to understand what is happening
 *  at the ActionScript level.
 *  To embed a bitmap image in ActionScript, you declare a variable
 *  of type Class, and put <code>[Embed]</code> metadata on it.
 *  For example, you embed a GIF file like this:</p>
 *
 *  <pre>
 *  [Bindable]
 *  [Embed(source="Logo.gif")]
 *  private var logoClass:Class;</pre>
 *
 *  <p>The MXML compiler sees the .gif extension, transcodes the GIF data
 *  into the bitmap format that the player uses, autogenerates
 *  a subclass of the BitmapAsset class, and sets your variable
 *  to be a reference to this autogenerated class.
 *  You can then use this class reference to create instances of the
 *  BitmapAsset using the <code>new</code> operator, and you can use
 *  APIs of the BitmapAsset class on them:</p>
 *
 *  <pre>
 *  var logo:BitmapAsset = BitmapAsset(new logoClass());
 *  logo.bitmapData.noise(4);</pre>
 *
 *  <p>However, you rarely need to create BitmapAsset instances yourself
 *  because image-related properties and styles can simply be set to an
 *  image-producing class, and components will create image instances
 *  as necessary.
 *  For example, to display this image in an Image control, you can 
 *  set the Image's <code>source</code> property to <code>logoClass</code>.
 *  In MXML you could do this as follows:</p>
 *
 *  <pre>
 *  &lt;mx:Image id="logo" source="{logoClass}"/&gt;</pre>
 */
public class BitmapAsset extends FlexBitmap
                         implements IFlexAsset, IFlexDisplayObject
{
    include "../core/Version.as";

    //--------------------------------------------------------------------------
    //
    //  Constructor
    //
    //--------------------------------------------------------------------------

    /**
     *  Constructor.
     *
     *  @param bitmapData The data for the bitmap image. 
     *
     *  @param pixelSnapping Whether or not the bitmap is snapped
     *  to the nearest pixel.
     *
     *  @param smoothing Whether or not the bitmap is smoothed when scaled. 
     */
    public function BitmapAsset(bitmapData:BitmapData = null,
                                pixelSnapping:String = "auto",
                                smoothing:Boolean = false)
    {
        super(bitmapData, pixelSnapping, smoothing);
    }

    //--------------------------------------------------------------------------
    //
    //  Properties
    //
    //--------------------------------------------------------------------------

    //----------------------------------
    //  measuredHeight
    //----------------------------------

    /**
     *  @inheritDoc
     */
    public function get measuredHeight():Number
    {
        if (bitmapData)
            return bitmapData.height
        
        return 0;
    }

    //----------------------------------
    //  measuredWidth
    //----------------------------------

    /**
     *  @inheritDoc
     */
    public function get measuredWidth():Number
    {
        if (bitmapData)
            return bitmapData.width;
        
        return 0;
    }

    //--------------------------------------------------------------------------
    //
    //  Methods
    //
    //--------------------------------------------------------------------------

    /**
     *  @inheritDoc
     */
    public function move(x:Number, y:Number):void
    {
        this.x = x;
        this.y = y;
    }

    /**
     *  @inheritDoc
     */
    public function setActualSize(newWidth:Number, newHeight:Number):void
    {
        width = newWidth;
        height = newHeight;
    }
}

}
