Ext.data.JsonP.data({
  "guide": "<h1>Data</h1>\n\n<hr />\n\n<p>The data package is what loads and saves all of the data in your application, and it saw a massive upgrade for version 4. With the new Model class, it’s easy to handle almost any kind of data on the client side. Validations are performed at the Model level and the new Associations API makes it possible to set up relationships between your models.</p>\n\n<p>The expanded Proxy class can now be used directly on a Model, meaning you can load and save data without the need for a Store, and the new localStorage Proxy enables you to save data on the client with a single line of code. Multiple sorting, filtering and grouping is now all possible on the client side, and the new Readers even understand nested data sets. The data package underpins most of the components in the framework, and we’ve written extensively about it in recent posts:</p>\n\n<p>Original articles: <a href=\"http://www.sencha.com/blog/ext-js-4-anatomy-of-a-model/\">Anatomy of a Model</a> and <a href=\"http://edspencer.net/2011/02/proxies-extjs-4.html\">Proxies in ExtJS 4</a></p>\n\n<h2>What's New</h2>\n\n<p>The data package in <a href=\"#/api/Ext\" rel=\"Ext\" class=\"docClass\">Ext</a> JS 4 consists of 43 classes, but there are three that are more important than all others - Model, Store and Proxy. These are used by almost every application, and are supported by a number of satellite classes:</p>\n\n<p><img src=\"guides/data/data-package.png\" alt=\"Data package overview\" /></p>\n\n<h3>Models and Stores</h3>\n\n<p>The centerpiece of the data package is <a href=\"#/api/Ext.data.Model\" rel=\"Ext.data.Model\" class=\"docClass\">Ext.data.Model</a>. A Model represents some type of data in an application - for example an e-commerce app might have models for Users, Products and Orders. At its simplest a Model is just a set of fields and their data. Anyone familiar with <a href=\"#/api/Ext\" rel=\"Ext\" class=\"docClass\">Ext</a> JS 3 will have used Ext.data.Record, which was the precursor to <a href=\"#/api/Ext.data.Model\" rel=\"Ext.data.Model\" class=\"docClass\">Ext.data.Model</a>. Let's look at how we create a model now:</p>\n\n<pre><code>Ext.define('User', {\n    extend: 'Ext.data.Model',\n    fields: [\n        {name: 'id', type: 'int'},\n        {name: 'name', type: 'string'}\n    ]\n});\n</code></pre>\n\n<p>Models are typically used with a Store, which is basically a collection of Model instances. Setting up a Store and loading its data is simple:</p>\n\n<pre><code>Ext.create('Ext.data.Store', {\n    model: 'User',\n    proxy: {\n        type: 'ajax',\n        url : 'users.json',\n        reader: 'json'\n    },\n    autoLoad: true\n});\n</code></pre>\n\n<p>That's all we need to do to load a set of User model instances from the url 'users.json'. We configured our Store to use an AjaxProxy, telling it the url to load data from and the <a href=\"#/api/Ext.data.reader.Reader\" rel=\"Ext.data.reader.Reader\" class=\"docClass\">Ext.data.reader.Reader</a> class used to decode the data. In this case our server is returning json, so we've set up a JsonReader to read the response. Stores are able to perform sorting, filtering and grouping locally, as well as supporting remote sorting, filtering and grouping. With Ext JS 4, there is no longer a separate GroupingStore - we can now do multi-sorting, filtering and grouping inside a standard Store:</p>\n\n<pre><code>Ext.create('Ext.data.Store', {\n    model: 'User',\n\n    sorters: ['name', 'id'],\n    filters: {\n        property: 'name',\n        value   : 'Ed'\n    },\n    groupers: {\n        property : 'age',\n        direction: 'ASC'\n    }\n});\n</code></pre>\n\n<p>In the store we just created, the data will be sorted first by name then id; it will be filtered to only include Users with the name 'Ed' and the data will be grouped by age. It's easy to change the sorting, filtering and grouping at any time through the Store API.</p>\n\n<h3>Proxies</h3>\n\n<p>We saw above how Store uses a Proxy to load its data, and how that Proxy could in turn be configured with a Reader to decode the server response. One structural change in <a href=\"#/api/Ext\" rel=\"Ext\" class=\"docClass\">Ext</a> JS 4 is that Store no longer has a direct link to Reader and Writer - these are now handled by the Proxy. This gives us an enormous benefit - Proxies can now be defined directly on a Model:</p>\n\n<pre><code>Ext.define('User', {\n    extend: 'Ext.data.Model',    \n    fields: ['id', 'name', 'age'],\n    proxy: {\n        type: 'rest',\n        url : '/users',\n        reader: {\n            type: 'json',\n            root: 'users'\n        }\n    }\n});\n\n// Uses the User Model's Proxy\nExt.create('Ext.data.Store', {\n    model: 'User'\n});\n</code></pre>\n\n<p>This helps us in two ways. First, it's likely that every Store that uses the User model will need to load its data the same way, so we avoid having to duplicate the Proxy definition for each Store. Second, we can now load and save Model data without a Store:</p>\n\n<pre><code>// Gives us a reference to the User class\nvar User = Ext.getModel('User');\n\nvar ed = Ext.create('User', {\n    name: 'Ed Spencer',\n    age : 25\n});\n\n// We can save Ed directly without having to add him to a Store first because we\n// configured a RestProxy this will automatically send a POST request to the url /users\ned.save({\n    success: function(ed) {\n        console.log(\"Saved Ed! His ID is \"+ ed.getId());\n    }\n});\n\n// Load User 123 and do something with it (performs a GET request to /users/123)\nUser.load(123, {\n    success: function(user) {\n        console.log(\"Loaded user 123: \" + user.get('name'));\n    }\n});\n</code></pre>\n\n<p>We've also introduced some brand new Proxies that take advantage of the new capabilities of HTML5 - LocalStorageProxy and SessionStorageProxy. Although older browsers don't support these new HTML5 APIs, they're so useful that we know a lot of applications will benefit enormously from their presence. And if we don't have a Proxy that matches your needs it's easy to create your own.</p>\n\n<h3>Associations</h3>\n\n<p>Proxies aren't the only new capability added to Models; we can now link Models together with the new Associations API. Most applications deal with many different Models, and the Models are almost always related. A blog authoring application might have models for User, Post and Comment. Each User creates Posts and each Post receives Comments. We can express those relationships like so:</p>\n\n<pre><code>Ext.define('User', {\n    extend: 'Ext.data.Model',\n    fields: ['id', 'name'],\n\n    hasMany: 'Posts'\n});\n\nExt.define('Post', {\n    extend: 'Ext.data.Model',\n    fields: ['id', 'user_id', 'title', 'body'],\n\n    belongsTo: 'User',\n    hasMany: 'Comments'\n});\n\nExt.define('Comment', {\n    extend: 'Ext.data.Model',\n    fields: ['id', 'post_id', 'name', 'message'],\n\n    belongsTo: 'Post'\n});\n</code></pre>\n\n<p>It's easy to express rich relationships between different Models in your application. Each Model can have any number of associations with other Models and your Models can be defined in any order. Once we have a Model instance we can easily traverse the associated data - for example, if we wanted to log all Comments made on each Post for a given User, we can do something like this:</p>\n\n<pre><code>// Loads User with ID 123 using User's Proxy\nUser.load(123, {\n    success: function(user) {\n        console.log(\"User: \" + user.get('name'));\n\n        user.posts().each(function(post) {\n            console.log(\"Comments for post: \" + post.get('title'));\n\n            post.comments().each(function(comment) {\n                console.log(comment.get('message'));\n            });\n        });\n    }\n});\n</code></pre>\n\n<p>Each of the hasMany associations we created above results in a new function being added to the Model. We declared that each User model hasMany Posts, which added the user.posts() function we used in the snippet above. Calling user.posts() returns a Store configured with the Post model. In turn, the Post model gets a comments() function because of the hasMany Comments association we set up. You may be wondering why we passed a 'success' function to the User.load call but didn't have to do so when accessing the User's posts and comments. All data is assumed to be loaded asynchronously because it usually has to be loaded from a server somewhere. This usually means passing in callbacks that are called when the data has been loaded, as with the 'success' function above.</p>\n\n<h3>Loading Nested Data</h3>\n\n<p>By setting up associations as we did above, the framework can automatically parse out nested data in a single request. Instead of making a request for the User data, another for the Posts data and then yet more requests to load the Comments for each Post, we can return all of the data in a single server response like this:</p>\n\n<pre><code>  {\n      id: 1\n      name: 'Ed',\n      posts: [\n          {\n              id   : 12,\n              title: 'All about data in <a href=\"#/api/Ext\" rel=\"Ext\" class=\"docClass\">Ext</a> JS 4',\n              body : 'One areas that has seen the most improvement...',\n              comments: [\n                  {\n                      id: 123,\n                      name: 'S Jobs',\n                      message: 'One more thing'\n                  }\n              ]\n          }\n      ]\n  }\n</code></pre>\n\n<p>The data is all parsed out automatically by the framework. It's easy to configure your Models' Proxies to load data from almost anywhere, and their Readers to handle almost any response format. As with <a href=\"#/api/Ext\" rel=\"Ext\" class=\"docClass\">Ext</a> JS 3, Models and Stores are used throughout the framework by many of the components such a Grids, Trees and Forms.</p>\n\n<h2>Validations and Associations</h2>\n\n<p>(orginal article at http://www.sencha.com/blog/using-validations-and-associations-in-sencha-touch/)</p>\n\n<p>Sencha Touch already has a rich Model layer which makes it easy to deal with different types of data. As of Sencha Touch 0.97β, Models became a lot richer with support for validating their data and associating them with other Models. These new capabilities make it easier to write client-side applications by reducing the amount of code you need to write. First up, let's look at using validations in a model. The example we're going to use an e-commerce management application, with models for Users and Products. Let's define the Product model first:</p>\n\n<pre><code>Ext.ns('MyApp');\n\nMyApp.Product = Ext.define('Product', {\n    extend: 'Ext.data.Model',\n    fields: [\n        {name: 'id',      type: 'int'},\n        {name: 'user_id', type: 'int'},\n        {name: 'name',    type: 'string'},\n        {name: 'price',   type: 'float'},\n        {name: 'sku',     type: 'string'}\n    ],\n\n    validations: [\n        {type: 'presence', name: 'name'},\n        {type: 'format',   name: 'sku', matcher: /[0-9]{4}[A-Z]+/},\n        {type: 'length',   name: 'name', min: 3}\n    ],\n\n    proxy: {\n        type: 'localstorage',\n        id  : 'products'\n    }\n});\n</code></pre>\n\n<p>The model definition above is largely self-explanatory: we're defining a model called Product with four fields - id, name, price and sku, plus several validation rules about those fields. The field definitions have always been present in Sencha Touch and validations follow the same format. In each case, we specify a field and a type of validation. Some validations take additional optional configuration - for example the length validation can take min and max properties, format can take a matcher, etc. There are five validations built into Sencha Touch and adding custom rules is easy. First, let's meet the ones built right in:</p>\n\n<ul>\n<li><code>presence</code> simply ensures that the field has a value. Zero counts as a valid value but empty strings do not.</li>\n<li><code>length</code> ensures that a string is between a min and max length. Both constraints are optional.</li>\n<li><code>format</code> ensures that a string matches a regular expression format. In the example above we ensure that the sku is 4 numbers followed by at least one letter.</li>\n<li><code>inclusion</code> ensures that a value is within a specific set of values (e.g. ensuring gender is either male or female).</li>\n<li><code>exclusion</code> ensures that a value is not one of the specific set of values (e.g. blacklisting usernames like 'admin').</li>\n</ul>\n\n\n<p>Now that we have a grasp of what the different validations do, let's try using them against a Product instance. We'll create a product instance and run the validations against it, noting any failures:</p>\n\n<pre><code>var product = new MyApp.Product({\n    name : 'Sencha Touch',\n    sku  : 'not a valid sku',\n    price: 99\n});\n\nvar errors = product.validate();\n\nerrors.isValid()) //returns 'false' as there were validation errors\nerrors.items; //returns the array of all errors found on this model instance\n\nerrors.forField('sku'); //returns the errors for the sku field\n</code></pre>\n\n<p>The key function here is validate(), which runs all of the configured validations and returns an <a href=\"#/api/Ext.data.Errors\" rel=\"Ext.data.Errors\" class=\"docClass\">Ext.data.Errors</a> object. This simple object is just a collection of any errors that were found, plus some convenience methods such as isValid() - which returns true if there were no errors on any field - and forField(), which returns all errors for a given field. In our e-commerce system each Product is created by a User, so let's set up the User model now:</p>\n\n<pre><code>MyApp.User = Ext.define('User', {\n    extend: 'Ext.data.Model',\n    fields: [\n        {name: 'id',       type: 'int'},\n        {name: 'name',     type: 'string'},\n        {name: 'gender',   type: 'string'},\n        {name: 'username', type: 'string'}\n    ],\n\n    validations: [\n        {type: 'inclusion', name: 'gender',   list: ['male', 'female']},\n        {type: 'exclusion', name: 'username', list: ['admin']}\n    ],\n\n    associations: [\n        {type: 'hasMany', model: 'Product', name: 'products'}\n    ],\n\n    proxy: {\n        type: 'localstorage',\n        id  : 'users'\n    }\n});\n</code></pre>\n\n<p>Defining the User follows the same pattern as we used for defining the Product - we set up the fields and a couple of validations. In this case, our validations are expecting the gender field to be either male or female, and the username to be anything but 'admin'. This time, however, we've also added an association to the model. There are two main types of association in Sencha Touch - hasMany and belongsTo. In our application each User creates many Products, so we create a hasMany association linking User to Product. Associations give us a powerful way to manipulate related data. For example, loading all of the Products for a particular User is very easy:</p>\n\n<pre><code>var user = new MyApp.User({id: 10});\n\n//loads all products where user_id = 10\nuser.products().load({\n    callback: function(records, operation) {\n        alert(records.length);\n    }\n});\n</code></pre>\n\n<p>Let's break down what the code above is actually doing. The association that we defined have created a new method on the User object called products(). This method returns an <a href=\"#/api/Ext.data.Store\" rel=\"Ext.data.Store\" class=\"docClass\">Ext.data.Store</a> that is automatically filtered to only load Products where the user_id is equal to the User instance's id (which is 10 in this case). Because we're in the browser and a long way from the database, all loading and saving operations are asynchronous, so we have to pass a callback function to the generated Products store's load method. This callback is given the records that are loaded as well as the <a href=\"#/api/Ext.data.Operation\" rel=\"Ext.data.Operation\" class=\"docClass\">Ext.data.Operation</a> object that is used to load them. Associations aren't just helpful for loading data - they're useful for creating new records too:</p>\n\n<pre><code>user.products().add({\n    name: 'Ext JS 4.0',\n    sku : '1234A'\n});\n\nuser.products().sync();\n</code></pre>\n\n<p>Here we instantiate a new Product, which is automatically given the User's id in the user_id field. Calling sync() saves the new Product via its configured Proxy - this, again, is an asynchronous operation to which you can pass a callback if you want to be notified when the operation completed. It's usually useful to have both sides of a relationship know about the association, so let's update our Product model definition:</p>\n\n<pre><code>MyApp.Product = Ext.define('Product', {\n    extend: 'Ext.data.Model',\n    // Same fields and validations as before\n    ...\n\n    associations: [\n        {type: 'belongsTo', model: 'User'}\n    ]\n});\n</code></pre>\n\n<p>The belongsTo association also generates new methods on the model, here's how we can use those:</p>\n\n<pre><code>var product = new MyApp.Product({id: 100});\n\nproduct.getUser(function(user) {\n    //do something with the loaded user model\n});\n\nproduct.setUser(100, {\n    callback: function(product, operation) {\n        if (operation.wasSuccessful()) {\n            alert('Product user updated');\n        } else {\n            alert('Product user could not be updated');\n        }\n    }\n});\n</code></pre>\n\n<p>Once more, the loading function (getUser) is asynchronous and requires a callback function to get at the user instance. The setUser method simply updates the foreign_key (user_id in this case) to 100 and saves the Product model. As usual, callbacks can be passed in that will be triggered when the save operation has completed - whether successful or not. The best way to find out more about validations and associations is to check out the updated Model documentation, as well as the docs on the hasMany and belongsTo associations. Finally, be aware that since we're still in beta, the API for this new functionality may change slightly before version 1.0.</p>\n\n<h2>Anatomy of a Model</h2>\n\n<p>A Model represents almost any type of persistable data in an application. For example, an e-commerce application might have models for User, Product, Order and so on. Each model contains a set of fields as well as functions that enable the application to operate on its data—for example an Order model might have a ‘shipToCustomer’ function that kicks off the shipping process.</p>\n\n<p>Ext JS 3.x and before had a class called Record, which was very similar to the new Model class. The difference is that whereas Record was only fields and functions, Model is much more powerful. Today, we’re going to look at four of the principal parts of Model—Fields, Proxies, Associations and Validations.</p>\n\n<p><img src=\"guides/data/model.png\" alt=\"Model architecture\" /></p>\n\n<h3>Fields</h3>\n\n<p>Every model consists of one or more fields. At its simplest, a field is just a name: ‘id’, ‘email’ and ‘phone’ could all be fields on a User model. Fields can also do a lot more. They can have a type (int, float, string or auto) and even conversion functions that modify their values when set. For example, if we record our user’s height, we can have a function that converts it from inches to centimeters.</p>\n\n<p>Here’s how we would set up a simple User model with three fields. For the first field, we will specify type ‘int’—e.g. it should be an integer. For the second field, we won’t specify a type, which means it will use the ‘auto’ type by default. The auto type will accept anything passed to it. For the third field, we’ll specify type ‘int’ and also provide a conversion function that takes a number of inches and converts it to centimeters:</p>\n\n<pre><code>Ext.define('User', {\n    extend: 'Ext.data.Model',\n    fields: [\n        {name: 'id', type: 'int'},\n        'name',\n        {\n            name: 'height',\n            type: 'int',\n            convert: function(inches) {\n                return Math.round(inches * 2.54);\n            }\n        }\n    ]\n});\n</code></pre>\n\n<p>It’s easy to create a new User instance now and see how our field definitions affect the data we pass into them:</p>\n\n<pre><code>var abe = new User({\n    id: 123,\n    name: 'Abe Elias',\n    height: 76\n});\n\nconsole.log(abe.get('id')); //logs 123 (a JavaScript Number object)\nconsole.log(abe.get('name')); //logs 'Abe Elias' (A JavaScript String object)\nconsole.log(abe.get('height')); //logs 193 (inches converted to centimeters)\n</code></pre>\n\n<p>Using simple instances of Models is very easy, but usually an application needs to load and save data. For this, we turn to Proxy.</p>\n\n<h3>Proxy</h3>\n\n<p>In <a href=\"#/api/Ext\" rel=\"Ext\" class=\"docClass\">Ext</a> JS 4, a Proxy is responsible for loading and saving data from some source. This can be over AJAX, using HTML5 localStorage or even simply keeping data in memory. <a href=\"#/api/Ext\" rel=\"Ext\" class=\"docClass\">Ext</a> JS 4 comes with a set of Proxies built in by default, but it’s easy to create your own. Let’s see how we might set up a Proxy for our User Model:</p>\n\n<pre><code>Ext.define('User', {\n    extend: 'Ext.data.Model',\n    fields: ['id', 'name', 'email'],\n\n    proxy: {\n        type: 'rest',\n        url : '/users',\n        reader: 'json'\n    }\n});\n</code></pre>\n\n<p>Defining Proxies directly on a Model is a new approach in version 4—its main benefit is that we can easily load and save Model data without creating a Store, as we did in <a href=\"#/api/Ext\" rel=\"Ext\" class=\"docClass\">Ext</a> JS 3.</p>\n\n<p>Above we set up the User Model to use a RestProxy. This Proxy uses AJAX calls to load and save its data from a server. It’s smart enough that it knows how to build RESTful urls automatically given a base url (‘/users’ in this case). We also set up our Proxy with a JsonReader—this takes the server response and decodes it into User Model instances. In this case, we know that our server will respond with JSON, so a JsonReader makes sense.</p>\n\n<p>Let’s load up a User model now:</p>\n\n<pre><code>// GET /users/123\nUser.load(123, {\n    success: function(user) {\n        console.log(user.get('name'));\n    }\n});\n</code></pre>\n\n<p>The code above loads data from the URL /users/123, which was constructed from the ID we passed to User.load. It then uses the JsonReader to decode the response into a User object. Our server sends back a response like this, which would make the code above log “Aaron Conran”:</p>\n\n<pre><code>// Response from GET /users/123\n{\n    \"id\": 123,\n    \"name\": \"Aaron Conran\",\n    \"email\": \"aaron@sencha.com\"\n}\n</code></pre>\n\n<p>Easy enough, but let’s take it to the next level with associations.</p>\n\n<h3>Associations</h3>\n\n<p>We usually have many Models in an application, but none of them exist in a vacuum. There are relationships between models—each User in our system makes Orders, and each Order is composed of Order Items. <a href=\"#/api/Ext\" rel=\"Ext\" class=\"docClass\">Ext</a> JS 4 gives us the ability to represent these relationships on the client side with a simple and intuitive Associations API. Let’s see how we’d set those three Models up:</p>\n\n<pre><code>// each User hasMany Orders\nExt.define('User', {\n    extend: 'Ext.data.Model',\n    fields: ['id', 'name', 'email'],\n    proxy : {\n        type: 'rest',\n        url : '/users',\n        reader: 'json'\n    },\n\n    hasMany: 'Orders'\n});\n\n// each Order belongsTo a User, and hasMany OrderItems\nExt.define('Order', {\n    extend: 'Ext.data.Model',\n    fields: ['id', 'user_id', 'status'],\n    belongsTo: 'User',\n    hasMany: 'OrderItems'\n});\n\n// each OrderItem belongsTo an Order\nExt.define('OrderItem', {\n    extend: 'Ext.data.Model',\n    fields: ['id', 'order_id', 'name', 'description', 'price', 'quantity'],\n    belongsTo: 'Order'\n});\n</code></pre>\n\n<p>Now that our application knows about its Models and their associations, let’s put them to use. Expanding on the example from the Proxy section above, let’s make our server response look like this:</p>\n\n<pre><code>{\n    \"id\": 123,\n    \"name\": \"Aaron Conran\",\n    \"email\": \"aaron@sencha.com\",\n    \"orders\": [\n        {\n            \"id\": 1,\n            \"status\": \"shipped\",\n            \"orderItems\": [\n                {\n                    \"id\": 2,\n                    \"name\": \"Sencha Touch\",\n                    \"description\": \"The first HTML5 mobile framework\",\n                    \"price\": 0,\n                    \"quantity\": 1\n                }\n            ]\n        }\n    ]\n}\n</code></pre>\n\n<p>When we load our User data now, the associated Orders and OrderItems are loaded along with it, enabling us to interact with them in our code:</p>\n\n<pre><code>User.load(123, {\n    success: function(user) {\n        console.log(user.get('name')); //\"Aaron Conran\"\n        console.log(user.orders().getCount()); //\"1\" -- there is only 1 order in the response above\n\n        //we can iterate over the orders easily using the Associations API\n        user.orders().each(function(order) {\n            console.log(order.get('status')); //\"shipped\"\n\n            //we can even iterate over each Order's OrderItems:\n            order.orderItems().each(function(orderItem) {\n                console.log(orderItem.get('title')); //\"Sencha Touch\"\n            });\n        });\n    }\n});\n</code></pre>\n\n<p>The Associations API is one of the most powerful new capabilities of <a href=\"#/api/Ext\" rel=\"Ext\" class=\"docClass\">Ext</a> JS 4, and because the data package is shared with Sencha Touch, you can get a sneak peek at it by downloading Sencha Touch today or checking out the HasMany and BelongsTo API docs.</p>\n\n<h3>Validations</h3>\n\n<p>The last new feature we’re going to look at today is Validation. In <a href=\"#/api/Ext\" rel=\"Ext\" class=\"docClass\">Ext</a> JS 3, the only place that validations could be performed was inside a form. In all of the examples above, we were manipulating data without a form, so we need to be able to validate at the Model level. Thankfully, Ext JS 4 has support for just that—let’s take a look:</p>\n\n<pre><code>Ext.define('User', {\n    extend: 'Ext.data.Model',\n    fields: ['id', 'name', 'email', 'height'],\n\n    validations: [\n        {type: 'presence', field: 'id'},\n        {type: 'length', field: 'name', min: 2},\n        {type: 'format', field: 'email', matcher: /[a-z]@[a-z].com/}\n    ]\n});\n</code></pre>\n\n<p>We set up three simple validations on our User model. First, we say that the ‘id’ field must be present, then we say that the ‘name’ field must be at least 2 characters long, and finally we use a very simple regular expression to match the format of the ‘email’ field.</p>\n\n<p>We can use these validations to ensure that the data in our model is correct. Here we create a new User without a valid name and missing an ID, and ask it to validate itself:</p>\n\n<pre><code>var ed = new User({\n    email: \"ed@sencha.com\"\n});\n\nvar validation = ed.validate();\n\nconsole.log(validation.isValid()); //false\n\n//we can easily output or manipulate any validation errors\nvalidation.each(function(error) {\n    console.log(error.field + \" \" + error.message);\n});\n\n//in this case, the output looks like this:\n\"id must be present\"\n\"name is the wrong length\"\n</code></pre>\n"
});