<?php

/**
 * This class is used to handle statistics
 *
 */
class EfrontStats
{
    /**
     * Get user seen content in lesson
     *
     * This function calulates the content done by the specified student(s) in the
     * specified lesson. If $users is not specified, then information on all users and/or lessons
     * is calculated.
     * <br/>Example:
     * <code>
     * EfrontStats :: getStudentsSeenContent(3, 'jdoe');                            //Get statistics for user jdoe in lesson 3
     * EfrontStats :: getStudentsSeenContent(3, array('jdoe', 'george'));           //Get statistics for users george and jdoe in lesson 3
     * EfrontStats :: getStudentsSeenContent(3);                                    //Get statistics for all users in lesson 3
     * </code>
     *
     * @param int $lessonId The lesson id
     * @param mixed $users One or more optional user logins
     * @return array The seen content per user login
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function getStudentsSeenContent($lessonId, $users = false) {
        if (!eF_checkParameter($lessonId, 'id')) {
            throw new EfrontLessonException(_INVALIDID, EfrontLessonException :: INVALID_ID);
        }
        if ($users !== false) {
            !is_array($users) ? $users = array($users) : null;                //Convert single login to array
        }


        $doneTests = array();
        $result    = eF_getTableData("done_tests as dt,tests as t, content as c", "c.id, dt.users_LOGIN, dt.score", "t.content_ID = c.id AND t.id=dt.tests_ID AND c.lessons_ID = ".$lessonId);
        foreach ($result as $value) {
            $doneTests[$value['users_LOGIN']][$value['id']] = $value['score'];
        }

        $doneContent      = array();
        $doneProjects = array();
        $usersDoneContent = eF_getTableDataFlat("users u, users_to_lessons ul", "u.login, ul.done_content", "u.user_type = 'student' and u.login = ul.users_LOGIN and ul.lessons_ID = $lessonId");
        sizeof($usersDoneContent) > 0 ? $usersDoneContent = array_combine($usersDoneContent['login'], $usersDoneContent['done_content']) : $usersDoneContent = array();
        foreach ($usersDoneContent as $login => $content) {
            if (!$users || in_array($login, $users)) {                                    //If $users is false, then all users data were asked; otherwise, return only results specified in the $users array. This way we also check $users values integrity (since in order for a value to be returned it must be present in both arrays, $users AND $doneContent)
                unserialize($content) ? $userSeenContent = array_fill_keys(array_keys(unserialize($content)), false) : $userSeenContent = array();        //Using array_fill_keys, we set all the array values to ''. This way, only the array keys contain the content id, while array values contain test scores (when the unit is a test). This way we may use array_sum to calculate the mean score at once)
                $doneTests[$login]    ? $userDoneTests   = $doneTests[$login]                                        : $userDoneTests   = array();
                $doneContent[$login] = $userSeenContent + $userDoneTests;
            }
        }

        return $doneContent;
    }



    /**
     * Get user done tests in lesson
     *
     * This function finds the done tests of the specified users.
     * If $users is not specified, then information on all users and/or lessons
     * is calculated.
     * <br/>Example:
     * <code>
     * EfrontStats :: getStudentsDoneTests(3, 'jdoe');                          //Get statistics for user jdoe in lesson 3
     * EfrontStats :: getStudentsDoneTests(3, array('jdoe', 'george'));         //Get statistics for users george and jdoe in lesson 3
     * EfrontStats :: getStudentsDoneTests(3);                                  //Get statistics for all users in lesson 3
     * </code>
     *
     * @param int $lessonId The lesson id
     * @param mixed $users One or more optional user logins
     * @return array The done tests per user login
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function getStudentsDoneTests($lessonId, $users = false){
        if (!(is_array($users))){
            $users = array($users);
        }
        $usersDoneTests = eF_getTableData("tests t, content c, done_tests dt", "c.name, dt.tests_ID, dt.score, dt.comments, dt.users_LOGIN as login", "dt.tests_ID = t.id and t.content_ID = c.id and c.lessons_ID = $lessonId");
        $doneTests = array();
        foreach ($usersDoneTests as $donetest){
            $login = $donetest['login'];
            if (!$users || in_array($login, $users)){
                $doneTests[$login][$donetest['tests_ID']] = array('name' => $donetest['name'], 'score' => $donetest['score'], 'comments' => $donetest['comments']);
            }
        }
        return $doneTests;
    }


    
    /**
     * Get user login time in a lesson
     *
     * This function finds calculates the total time each student spend in the lesson
     * If $users is not specified, then information on all users and/or lessons
     * is calculated.
     * <br/>Example:
     * <code>
     * EfrontStats :: getUsersTime(3, 'jdoe');                          //Get statistics for user jdoe in lesson 3
     * EfrontStats :: getUsersTime(3, array('jdoe', 'george'));         //Get statistics for users george and jdoe in lesson 3
     * EfrontStats :: getUsersTime(3);                                  //Get statistics for all users in lesson 3
     * </code>
     *
     * @param int $lessonId The lesson id
     * @param mixed $users One or more optional user logins
     * @return array The total time per login (time is an array itself ('hours', 'minutes', 'seconds', 'total_seconds'))
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function getUsersTime($lessonId, $users = false, $fromTimestamp = false, $toTimestamp = false){
        $userTimes = array();
        $lesson = new EfrontLesson($lessonId);
        if (!$users){
            $users = $lesson->getUsers();
        }
        else{
            if (!is_array($users)){
                $users = array($users);
            }
        }
        if (!$fromTimestamp){
            $fromTimestamp = "0000000";
        }
        if (!$toTimestamp){
            $toTimestamp = time();
        }
        foreach ($users as $login){
            $sql = "SELECT id, timestamp, lessons_id FROM logs WHERE users_LOGIN='$login' and timestamp between $fromTimestamp and $toTimestamp order by timestamp";
            $res = eF_execute($sql);
            $total_time = array();
            $total_time['minutes'] = 0;
            $total_time['seconds'] = 0;
            $total_time['hours']   = 0;
            $total_time['total_seconds'] = 0;
            $l_start = 0;
            $inlesson = 0;
            while ($k=mysql_fetch_row($res)){
                $c_id     = $k[0];
                $c_time   = $k[1];
                $c_lesson = $k[2];
                if ($inlesson){
                    if ($c_lesson != $lessonId){
                        $inlesson = 0;
                        $interval = ef_convertIntervalToTime($c_time - $l_start);
                        if ($interval['hours'] == 0 && $interval['minutes'] <= 30){
                            $total_time['minutes'] += $interval['minutes'];
                            $total_time['seconds'] += $interval['seconds'];
                        }
                    }
                    else{
                        $interval = ef_convertIntervalToTime($c_time - $l_start);
                        if ($interval['hours'] == 0 && $interval['minutes'] <= 30){
                            $total_time['minutes'] += $interval['minutes'];
                            $total_time['seconds'] += $interval['seconds'];
                        }
                        $l_start = $c_time;
                    }
                }
                else{
                    if ($c_lesson == $lessonId){
                        $inlesson = 1;
                        $l_start  = $c_time;
                    }

                }
            }

            $sec = $total_time['seconds'];

            if ($sec >=60){
                $nm = floor($sec / 60);
                $ns = $sec %60;
                $total_time['seconds']  = $ns;
                $total_time['minutes'] += $nm;
            }
            if ($total_time['minutes'] >= 60){
                $nh = floor($total_time['minutes']/60);
                $nm = $total_time['minutes'] % 60;
                $total_time['hours']   = $nh;
                $total_time['minutes'] = $nm;
            }

            $total_time['total_seconds'] = $total_time['hours'] * 3600 + $total_time['minutes']*60 + $total_time['seconds'];
            $userTimes[$login] = $total_time;
        }
        return $userTimes;
    }


    /**
     * Get user done projects in lesson
     *
     * This function finds the done projects of the specified users.
     * If $users is not specified, then information on all users and/or lessons
     * is calculated.
     * <br/>Example:
     * <code>
     * EfrontStats :: getStudentsDoneProjects(3, 'jdoe');                           //Get statistics for user jdoe in lesson 3
     * EfrontStats :: getStudentsDoneProjects(3, array('jdoe', 'george'));          //Get statistics for users george and jdoe in lesson 3
     * EfrontStats :: getStudentsDoneProjects(3);                                   //Get statistics for all users in lesson 3
     * </code>
     *
     * @param int $lessonId The lesson id
     * @param mixed $users One or more optional user logins
     * @return array The done projects per user login
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function getStudentsDoneProjects($lessonId, $users = false){
        if (!(is_array($users))){
            $users = array($users);
        }
        $usersDoneProjects = eF_getTableData("projects p, users_to_projects up", "up.projects_ID, p.title, up.grade, up.upload_timestamp, up.users_LOGIN as login", "(up.status = 1 or up.grade!='') and up.projects_ID = p.id and p.lessons_ID = $lessonId");
        $doneProjects = array();
        foreach ($usersDoneProjects as $doneproject){
            $login = $doneproject['login'];
            if (!$users || in_array($login, $users)){
                $doneProjects[$login][$doneproject['projects_ID']]['grade'] = $doneproject['grade'];
                $doneProjects[$login][$doneproject['projects_ID']]['name']  = $doneproject['title'];
                $doneProjects[$login][$doneproject['projects_ID']]['timestamp']  = $doneproject['upload_timestamp'];
            }
        }
        return $doneProjects;
    }



    /**
     * Get user assigned projects in lesson
     *
     * This function finds the assigned projects to the specified users.
     * If $users is not specified, then information on all users and/or lessons
     * is calculated.
     * <br/>Example:
     * <code>
     * EfrontStats :: getStudentsAssignedProjects(3, 'jdoe');                           //Get statistics for user jdoe in lesson 3
     * EfrontStats :: getStudentsAssignedProjects(3, array('jdoe', 'george'));          //Get statistics for users george and jdoe in lesson 3
     * EfrontStats :: getStudentsAssignedProjects(3);                                   //Get statistics for all users in lesson 3
     * </code>
     *
     * @param int $lessonId The lesson id
     * @param mixed $users One or more optional user logins
     * @return array The assigned projects per user login
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function getStudentsAssignedProjects($lessonId, $users = false){
        if (!(is_array($users))){
            $users = array($users);
        }
        $usersAssignedProjects = eF_getTableData("projects p, users_to_projects up", "p.title, up.projects_ID, up.grade, up.upload_timestamp, up.users_LOGIN as login", "up.projects_ID = p.id and p.lessons_ID = $lessonId");
        $asignedProjects = array();
        foreach ($usersAssignedProjects as $project){
            $login = $project['login'];
            if (!$users || in_array($login, $users)){
                $asignedProjects[$login][$project['projects_ID']] = array('title' => $project['title'], 'grade' => $project['grade'], 'timestamp' => $project['upload_timestamp']);
            }
        }
        return $asignedProjects;
    }


    public static function getUsersForumPosts($lessonId, $users = false){
        $total_posts = array();
        $result = eF_getTableData("f_messages fm, f_topics ft, f_forums ff", "fm.users_LOGIN as login, count(*) as cnt", "fm.f_topics_ID = ft.id and ft.f_forums_ID = ff.id and ff.lessons_ID = ".$lesson -> lesson['id'].$sql. " group by fm.users_LOGIN");
        foreach ($result as $data){
            $total_posts[$data['login']] = $data['cnt'];
        }
        foreach ($users as $login){
            if (!isset($total_posts[$login])){
                $total_posts[$login] = 0;
            }
        }
        return $total_posts;
    }

    public static function getUsersComments($lessonId, $users = false){
        $total_comments = array();
        $result = eF_getTableData("comments cm, content c", "cm.users_LOGIN as login, count(*) as cnt", "cm.content_id = c.id and c.lessons_ID = ".$lesson -> lesson['id'].$sql. " group by cm.users_LOGIN");
        foreach ($result as $data){
            $total_comments[$data['login']] = $data['cnt'];
        }
        foreach ($users as $login){
            if (!isset($total_comments[$login])){
                $total_comments[$login] = 0;
            }
        }
        return $total_comments;
    }
    

    /**
     * Get lesson status for students
     *
     * This function checks the status for the designated users in the specified lesson
     * The $lesson parameter may be either an EfrontLesson object or a lesson id (in which
     * case a new EfrontLesson object will be instantiated within the class).
     * If $users is not specified, then information on all user lesson is returned.
     * <br/>Example:
     * <code>
     * $lessonStatus = EfrontStats :: getStudentsLessonStatus(3, 'jdoe');                   //Get information for user jdoe in lesson 3
     * $lessonStatus = EfrontStats :: getStudentsLessonStatus(3, array('jdoe', 'george'));  //Get information for users jdoe, george in lesson 3
     * $lessonStatus = EfrontStats :: getStudentsLessonStatus(3);                           //Get information for all users having lesson 3
     * $lessonStatus = EfrontStats :: getStudentsLessonStatus(new EfrontLesson(3));         //Get information for all users having lesson 3, using the lesson object
     * </code>
     *
     * @param mixed $lesson Either a lesson id or a lesson object
     * @param mixed $users A single user login or an array of users
     * @return array the users' lesson status
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function getStudentsLessonStatus($lesson, $users = false) {
        if (!($lesson instanceof EfrontLesson)) {                            //This means that $lesson is a single id
            $lesson = new EfrontLesson($lesson);
        }
        if ($users) {
            !is_array($users) ? $users = array($users) : null;
            $sql = ' and users_LOGIN in ("'.implode('","', $users).'")';
        }

        $lessonConditions      = $lesson -> getConditions();                            //Get the lesson conditions
        $lessonContent         = new EfrontContentTree($lesson -> lesson['id']);        //Get the lesson content tree
        $usersDoneContent      = EfrontStats :: getStudentsSeenContent($lesson -> lesson['id'], $users);    //Calculate the done content for users in this lesson
        $usersDoneProjects     = EfrontStats :: getStudentsDoneProjects($lesson -> lesson['id'], $users);
        $usersAssignedProjects = EfrontStats :: getStudentsAssignedProjects($lesson -> lesson['id'], $users);
        $visitableContentIds   = array();
        foreach ($iterator = new EfrontNodeFilterIterator(new RecursiveIteratorIterator(new RecursiveArrayIterator($lessonContent -> tree), RecursiveIteratorIterator :: SELF_FIRST), array('ctg_type' => 'theory', 'active' => 1)) as $key => $value) {
            $visitableContentIds[$key] = $key;                                                    //Get the not-test unit ids for this content
        }
        foreach ($iterator = new EfrontNodeFilterIterator(new RecursiveIteratorIterator(new RecursiveArrayIterator($lessonContent -> tree), RecursiveIteratorIterator :: SELF_FIRST), array('ctg_type' => 'scorm', 'active' => 1)) as $key => $value) {
            $visitableContentIds[$key] = $key;                                                    //Get the not-test unit ids for this content
        }
        
        $visitableExampleIds = array();
        foreach ($iterator = new EfrontNodeFilterIterator(new RecursiveIteratorIterator(new RecursiveArrayIterator($lessonContent -> tree), RecursiveIteratorIterator :: SELF_FIRST), array('ctg_type' => 'examples', 'active' => 1)) as $key => $value) {
            $visitableExampleIds[$key] = $key;                                                    //Get the not-test unit ids for this content
        }

        $visitableTestIds = array();
        foreach ($iterator = new EfrontNodeFilterIterator(new RecursiveIteratorIterator(new RecursiveArrayIterator($lessonContent -> tree), RecursiveIteratorIterator :: SELF_FIRST), array('ctg_type' => 'tests', 'active' => 1)) as $key => $value) {
            $visitableTestIds[$key] = $key;                                                    //Get the test unit ids for this content
        }

        $result = eF_getTableData("users_to_lessons ul, users u", "ul.*, u.name, u.surname, u.user_type as basic_user_type", "u.login = ul.users_LOGIN and ul.lessons_ID = ".$lesson -> lesson['id'].$sql);

        foreach ($result as $value) {
            $lessonUsers[$value['users_LOGIN']] = $value;
        }
        
        $lessonStatus = array();
        
        $total_comments = array();
        $result = eF_getTableData("comments cm, content c", "cm.users_LOGIN as login, count(*) as cnt", "cm.content_id = c.id and c.lessons_ID = ".$lesson -> lesson['id'].$sql. " group by cm.users_LOGIN");
        foreach ($result as $data){
            $total_comments[$data['login']] = $data['cnt'];
        }
        
        $total_posts = array();
        $result = eF_getTableData("f_messages fm, f_topics ft, f_forums ff", "fm.users_LOGIN as login, count(*) as cnt", "fm.f_topics_ID = ft.id and ft.f_forums_ID = ff.id and ff.lessons_ID = ".$lesson -> lesson['id'].$sql. " group by fm.users_LOGIN");
        foreach ($result as $data){
            $total_posts[$data['login']] = $data['cnt'];
        }
        
        foreach ($lessonUsers as $login => $value) {
            $doneTests            = array();
            $doneContentAndTests  = array();
            $doneProjects         = array();
            $assignedProjects     = array();
            foreach ($usersDoneContent[$login] as $unitId => $score) {
                is_numeric($score) ? $doneTests[$unitId] = $score : null;
                $doneContentAndTests[$unitId] = $unitId;
            }
            foreach ($usersDoneProjects[$login] as $projectId => $project){
                $doneProjects[$projectId] = $project;
            }
            $avg_grade = 0;
            foreach ($usersAssignedProjects[$login] as $projectId => $project){
                $assignedProjects[$projectId] = $project;
                $avg_grade = $avg_grade + $project['grade'] ;
            }
            if (sizeof($assignedProjects) > 0){
                $avg_grade = round($avg_grade/sizeof($assignedProjects), 2);    
            }
            else{
                $avg_grade = 0.00;
            }
            
            $avg_score = 0.0;
            $res = ef_getTableData("done_tests dt, tests t, content c", "c.name, dt.timestamp, round(dt.score, 4) as score, c.id as unit_id", 
                                "dt.tests_ID = t.id and t.content_ID = c.id and dt.users_LOGIN='".$login."' and c.lessons_ID=".$lesson -> lesson['id']);

            for ($i = 0; $i < sizeof($res); $i++){
                $unit = new EfrontUnit($res[$i]['unit_id']);
                $doneTestsInfo[$res['unit_id']]['timestamp'] = $res[$i]['timestamp'];
                $doneTestsInfo[$res['unit_id']]['name']      = $res[$i]['name'];  
                $doneTestsInfo[$res['unit_id']]['score']     = $res[$i]['score'];  
                $avg_score += $res[$i]['score'];
            }
            if (sizeof($res) > 0){
                $avg_score = round(($avg_score*100)/sizeof($res));    
            }
            else{
                $avg_score = 0;
            }
            
            if ($doneContentUnits = unserialize($lessonUsers[$login]['done_content'])) {         
                $percentageDone = round(100 * (sizeof($doneContentUnits) - sizeof(array_diff($doneContentUnits, $visitableContentIds))) / sizeof($visitableContentIds), 2);
            } else {
                $percentageDone = 0;
            }
            
            $conditionsMet[$login] = self :: checkConditions($usersDoneContent[$login], $lessonConditions, $visitableContentIds);
            $lessonStatus[$login]  = array('login'              => $lessonUsers[$login]['users_LOGIN'],
                                           'name'               => $lessonUsers[$login]['name'],
                                           'surname'            => $lessonUsers[$login]['surname'],
                                           'basic_user_type'    => $lessonUsers[$login]['basic_user_type'],
                                           'user_type'          => $lessonUsers[$login]['user_type'],
                                           'done_content'       => $doneContentAndTests,
                                           'done_tests'         => $doneTests,
                                           'done_tests_info'    => $doneTestsInfo, 
                                           'done_projects'      => $doneProjects,
                                           'assigned_projects'  => $assignedProjects,
                                           'total_content'      => sizeof($visitableContentIds) + sizeof($visitableExampleIds),
                                           'total_tests'        => sizeof($visitableTestIds),
                                           'total_projects'     => sizeof($assignedProjects),
                                           'percentage_done'    => $percentageDone,
                                           'percentage_projects'=> sizeof($assignedProjects) ? round(100 * sizeof($doneProjects) / sizeof($assignedProjects), 2) :0,
                                           'percentage_tests'   => sizeof($visitableTestIds) ? round(100 * sizeof($doneTests) / sizeof($visitableTestIds), 2) :0,
                                           'avg_grade_projects' => $avg_grade,
                                           'avg_score_tests'    => $avg_score,
                                           'current_unit'       => $lessonUsers[$login]['current_unit'],
                                           'completed'          => $lessonUsers[$login]['completed'],
                                           'to_timestamp'       => $lessonUsers[$login]['to_timestamp'],
                                           'score'              => $lessonUsers[$login]['score'],
                                           'comments'           => $lessonUsers[$login]['comments'],
                                           'total_comments'     => isset($total_comments[$login]) ? $total_comments[$login] : 0 ,
                                           'total_posts'        => isset($total_posts[$login]) ? $total_posts[$login] : 0,
                                           'issued_certificate' => $lessonUsers[$login]['issued_certificate'],
                                           'total_conditions'   => sizeof($lessonConditions),
                                           'conditions_passed'  => array_sum($conditionsMet[$login]),
                                           'lesson_passed'      => array_product($conditionsMet[$login]),
                                           'lesson_name'        => $lesson -> lesson['name']);
        }

        
        return $lessonStatus;
    }

    /**
     * Get course status
     *
     * This function is used to get the course status for the user. It
     * also calcluates the course lessons status. It is the equivalent
     * of getStudentsLessonStatus() for courses
     * <br/>Example:
     * <code>
     * $courseStatus = EfrontStats :: getStudentsCourseStatus(3, 'jdoe');       //Get the status for course with id 3
     * </code>
     *
     * @param mixed $course Either a course id or a course object
     * @param mixed $users A single user login or an array of users
     * @return array the users' course status
     * @since 3.5.0
     * @access public
     * @see EfrontStats :: getStudentsLessonStatus()
     * @static
     */
    public static function getStudentsCourseStatus($course, $users = false) {
        if (!($course instanceof EfrontCourse)) {                            //This means that $lesson is a single id
            $course = new EfrontCourse($course);
        }

        if ($users) {
            !is_array($users) ? $users = array($users) : null;
            $sql = ' and users_LOGIN in ("'.implode('","', $users).'")';
        }
        $courseLessons = $course -> getLessons();

        $courseUsers = array();
        $result = eF_getTableData("users_to_courses uc, users u", "uc.*, u.name, u.surname, u.user_type as basic_user_type", "u.login = uc.users_LOGIN and courses_ID=".$course -> course['id'].$sql);
        foreach ($result as $value) {
            $courseUsers[$value['users_LOGIN']] = $value;
        }
        $courseStatus = array();        
        foreach ($courseUsers as $login => $value) {
            $courseStatus[$login] = array('login'              => $courseUsers[$login]['users_LOGIN'],
                                          'name'               => $courseUsers[$login]['name'],
                                          'surname'            => $courseUsers[$login]['surname'],
                                          'basic_user_type'    => $courseUsers[$login]['basic_user_type'],
                                          'user_type'          => $courseUsers[$login]['user_type'],
                                          'total_lessons'      => sizeof($courseLessons),
                                          'completed_lessons'  => 0,
                                          'completed'          => $courseUsers[$login]['completed'],
                                          'completed_str'      => $courseUsers[$login]['completed'] == 1? _YES : _NO, 
                                          'score'              => $courseUsers[$login]['score'],
                                          'issued_certificate' => $courseUsers[$login]['issued_certificate'],
                                          'comments'           => $courseUsers[$login]['comments']);
        }
        foreach ($courseLessons as $lesson) {
            $lessonStatus = self :: getStudentsLessonStatus($lesson['id'], $users);
            foreach ($lessonStatus as $login => $user) {
                $courseStatus[$login]['lessonsStatus'][$lesson['id']] = $user;
                $user['completed'] ? $courseStatus[$login]['completed_lessons']++ : null;
            }
        }
//pr($courseStatus);        
        return $courseStatus;
    }

    /**
     * Check if student meets conditions
     *
     * This function is used to check user progress against lesson condtions.
     * Every condition is checked and an array is returned, each with the condition
     * id and whether it is met or not.
     * <br/>Example:
     * <code>
     * $seenUnits = EfrontStats::getStudentsSeenContent(3, 'jdoe');     //Calculate seen content for user jdoe in lesson with id 3
     * $conditions = $currentLesson -> getConditions();                 //Get conditions for current lesson
     * $conditionsMet = self :: checkConditions(seenUnits, $conditions, $visitableContentIds);  //visitableContentIds is a list of units that are "visitable", that is they are active and non-empty
     * </code>
     *
     * @param array $seenUnits The units that the user has seen
     * @param array $conditions The conditions to check against
     * @param array $visitableContentIds The visitable content ids that make up the lesson
     * @return array An array with condition ids and true/false values depending on whether they are met
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function checkConditions($seenUnits, $conditions, $visitableContentIds) {
        !$seenUnits ? $seenUnits = array() : null;
    	$notSeenUnits  = array_diff_key($visitableContentIds, $seenUnits);            //The units that the user has yet to see
        $conditionsMet = array();
        foreach ($conditions as $conditionId => $condition) {
            switch ($condition['type']) {
                case 'all_units':
                    sizeof($notSeenUnits) == 0 ? $passed = true : $passed = false;
                    break;
                case 'percentage_units':
                    $percentageSeen = round(100 * (sizeof($visitableContentIds) - sizeof($notSeenUnits)) / sizeof($visitableContentIds));
                    $percentageSeen >= $condition['options'][0] ? $passed = true : $passed = false;
                    break;
                case 'specific_unit':
                    in_array($condition['options'][0], array_keys($seenUnits)) ? $passed = true : $passed = false;
                    break;
                case 'all_tests':
                    $passed = true;
                    foreach ($seenUnits as $id => $score) {
                        is_numeric($score) && $score < $condition['options'][0] / 100 ? $passed = false : null;
                    }
                    break;
                case 'mean_all_tests':
                    $meanScore = array_sum($seenUnits) / array_sum(array_count_values($seenUnits));        //array_count_values does not take into account false entries. So, in this expression, the denominator equals the number of tests the user has done
                    $meanScore >= $condition['options'][0] / 100 ? $passed = true : $passed = false;
                    break;
                case 'specific_test':
                    in_array($condition['options'][0], array_keys($seenUnits)) && $seenUnits[$condition['options'][0]] >= $condition['options'][1] / 100 ? $passed = true : $passed = false;
                    break;
                default:
                    break;
            }
            $conditionsMet[$conditionId] = $passed;
        }
        return $conditionsMet;
    }

    /**
     * Get user communication info
     *
     * This returns cimmmunication info for a user
     * <br/>Example:
     * <code>
     * $info = EfrontStats :: getUserCommunicationInfo('jdoe');                   //Get information for user jdoe 
     *
     * @param mixed $user Either a user login or a EfrontLessonUser object
     * @return array the users' basic info
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function getUserCommunicationInfo($user) {   
        if (! ($user instanceof EfrontLessonUser)){
            $user = EfrontUserFactory :: factory($user);
        }
        $info           = array();
        
        $forum_info     = ef_getTableData("f_messages", "*", "users_LOGIN='".$user -> login."'", "timestamp desc");
        $forum_messages = array();
        foreach ($forum_info as $message){
            $forum_messages[$message['id']] = $message;
        }
        
        $personal_messages_info = ef_getTableData("f_personal_messages", "*", "users_LOGIN='".$user -> login."'", "timestamp desc");
        $personal_messages      = array();
        foreach ($personal_messages_info as $message){
            $personal_messages[$message['id']] = $message;
        }
        
        $personal_folders_info = ef_getTableData("f_folders", "*", "users_LOGIN='".$user -> login."'");
        $personal_folders      = array();
        foreach ($personal_folders_info as $folder){
            $personal_folders[$folder['id']] = $folder;
        }
        
        $file_info = ef_getTableData("files", "*", "users_LOGIN='".$user -> login."' and type='file'");
        $files = array();
        $size = 0;
        foreach ($file_info as $file){
            try{
               $files[$file['id']] = new eF_File($file['id']);
                $t_info = $files[$file['id']] -> getInfo();
                $size += $t_info['size']; 
            }
            catch (Exception $e){
                //do nothing
            }
            
        }
        
        $folder_info = ef_getTableData("files", "*", "users_LOGIN='".$user -> login."' and type='folder'");
        $folders = array();
        foreach ($folder_info as $folder){
            $folders[$folder['id']] = new eF_Directory($folder['id']);
        }
        
        $chat_messages_info = ef_getTableData("chatmessages", "*", "users_LOGIN='".$user -> login."'", "timestamp desc");
        $chat_messages      = array();
        foreach ($chat_messages_info as $message){
            $chat_messages[$message['id']] = $message;
        }
        
        $comments_info = ef_getTableData("comments", "*", "users_LOGIN='".$user -> login."'", "timestamp desc");
        $comments      = array();
        foreach ($comments_info as $comment){
            $comments[$comment['id']] = $comment;
        }
        
        $info['forum_messages']            = $forum_messages;
        if (sizeof($info['forum_messages']) > 0){
            $info['last_message']          = current($info['forum_messages']);
        }
        $info['personal_messages']         = $personal_messages;
        $info['personal_folders']          = $personal_folders; 
        $info['files']                     = $files;
        $info['folders']                   = $folders;
        $info['total_size']                = $size;
        $info['chat_messages']             = $chat_messages;
        if (sizeof($info['chat_messages']) > 0){
            $info['last_chat']             = current($info['chat_messages']);
        }
        $info['comments']                  = $comments;

        return $info;
    }
    
    
    /**
     * Get user usage info
     *
     * This returns usage info for a user
     * <br/>Example:
     * <code>
     * $info = EfrontStats :: getUserUsageInfo('jdoe');                   //Get usage information for user jdoe 
     *
     * @param mixed $user Either a user login or a EfrontLessonUser object
     * @return array the users' basic info
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function getUserUsageInfo($user) {   
        if (! ($user instanceof EfrontLessonUser)){
            $user = EfrontUserFactory :: factory($user);
        }
        $info           = array();
        $login_info = ef_getTableData("logs", "*", "users_LOGIN='".$user -> login."' and action = 'login'", "timestamp desc");
        $logins = array();
        foreach ($login_info as $login){
            $logins[$login['id']] = $login;
        }

        $month_login_info = ef_getTableData("logs", "*", "users_LOGIN='".$user -> login."' and action = 'login' and timestamp > ".(time() - 2592000)."");
        $month_logins = array();
        foreach ($month_login_info as $login){
            $month_logins[$login['id']] = $login;
        }
        
        $week_login_info = ef_getTableData("logs", "*", "users_LOGIN='".$user -> login."' and action = 'login' and timestamp > ".(time() - 604800)."");
        $week_logins = array();
        foreach ($week_login_info as $login){
            $week_logins[$login['id']] = $login;
        }
        
        $temp = eF_getUserTimes($user -> login);
        sizeof($temp['duration']) > 0 ? $mean_duration = ceil((array_sum($temp['duration']) / sizeof($temp['duration'])) / 60) : $mean_duration = 0;
        $temp = eF_getUserTimes($user -> login, array('from' => time() - 2592000, 'to' => time()));
        sizeof($temp['duration']) > 0 ? $month_mean_duration = ceil((array_sum($temp['duration']) / sizeof($temp['duration']) / 60)) : $month_mean_duration = 0;

        $temp = eF_getUserTimes($user -> login, array('from' => time() - 604800, 'to' => time()));
        sizeof($temp['duration']) > 0 ? $week_mean_duration = ceil((array_sum($temp['duration']) / sizeof($temp['duration']) / 60)) : $week_mean_duration = 0;

        
        $info['logins']               = $logins;
        if (sizeof($info['logins']) > 0){
            $info['last_login']       = current($info['logins']);
        }
        $info['month_logins']         = $month_logins;
        $info['week_logins']          = $week_logins;
        $info['mean_duration']        = $mean_duration;
        $info['month_mean_duration']  = $month_mean_duration;
        $info['week_mean_duration']   = $week_mean_duration;

        return $info;
    }
    
    
   /**
     * Get statistic information about tests
     *
     * This returns statistic info for a test
     * <br/>Example:
     * <code>
     * $tests = array(2, 4);
     * $info = EfrontStats :: getTestInfo($tests);                   //Get information for tests 2,4
     *
     * @param mixed $tests Either an array of tests id or false (request information for all existing tests)
     * @return array the tests' statistinc info
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function getTestInfo($tests = false){
        $tests_info = array();
        if ($tests == false){
            $tests = ef_getTableData("tests","id");
        }
        
        foreach ($tests as $test_id){
            $test_info                  = array();
            $test                       = new EfrontTest($test_id);
            $unit                       = $test -> getUnit();
            $test_info['general']['id']              = $test_id;
            $test_info['general']['name']            = $unit -> offsetGet('name');
            $test_info['general']['content_ID']      = $unit -> offsetGet('id');
            $test_info['general']['duration']        = $test -> test['duration'];
            $test_info['general']['redoable']        = $test -> test['redoable'];
            $test_info['general']['redoable_str']    = $test -> test['redoable'] == 1 ? _YES : _NO;
            $test_info['general']['onebyone']        = $test -> test['onebyone'];
            $test_info['general']['onebyone_str']    = $test -> test['onebyone'] == 1 ? _YES : _NO;
            $test_info['general']['answers']         = $test -> test['answers'];
            $test_info['general']['answers_str']     = $test -> test['answers'] == 1  ? _YES : _NO;
            $test_info['general']['description']     = $test -> test['description'];
            $test_info['general']['timestamp']       = $unit -> offsetGet('timestamp');
            $test_info['general']['timestamp_str']   = strftime('%d-%m-%Y, %H:%M:%S', $test_info['general']['timestamp']); 
            
            $test_info['questions']['total']         = 0;
            $test_info['questions']['raw_text']      = 0;
            $test_info['questions']['multiple_one']  = 0;
            $test_info['questions']['multiple_many'] = 0;
            $test_info['questions']['true_false']    = 0;
            $test_info['questions']['match']         = 0;
            $test_info['questions']['empty_spaces']  = 0;
            $test_info['questions']['low']           = 0;
            $test_info['questions']['medium']        = 0;
            $test_info['questions']['high']          = 0;
            
            $questions = $test -> getQuestions(true);
            foreach ($questions as $question){
                $test_info['questions']['total']++;
                $test_info['questions'][$question -> question['type']]++;
                $test_info['questions'][$question -> question['difficulty']]++;
            }
            
            $test_info['done'] = array();
            $done_info = ef_getTableData("done_tests d, users u, tests t", "d.users_LOGIN, u.name, u.surname, d.score, d.timestamp",
                "d.tests_ID = t.id and d.users_LOGIN = u.LOGIN and t.id = $test_id");
            foreach ($done_info as $done){
                $done_test = array();
                $done_test['users_LOGIN'] = $done['users_LOGIN'];
                $done_test['name']        = $done['name'];
                $done_test['surname']     = $done['surname'];
                $done_test['score']       = $done['score'] * 100;
                $done_test['timestamp']   = $done['timestamp'];
        
                $test_info['done'][]      = $done_test;
            }
            
            $tests_info[$test_id] = $test_info;
        }
        return $tests_info;
    }
    
    /**
     * Get statistic information about questions
     *
     * This returns statistic info for a set of questions
     * <br/>Example:
     * <code>
     * $questions = array(2, 4);
     * $info = EfrontStats :: getQuestionInfo($questions);                   //Get information for questions 2,4
     *
     * @param mixed $questions Either an array of question id or false (request information for all existing questions)
     * @return array the questions' statistic info
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function getQuestionInfo($questions = false){
        $questions_info = array();
        if ($questions == false){
            $questions = ef_getTableData("questions","id");
        }
        
        foreach ($questions as $question_id){
            $question_info                     = array();
            $question                          = QuestionFactory :: factory($question_id);
            $question_info['general']['id']    = $question_id;
            $question_info['general']['text']  = $question -> question['text'];
            if (mb_strlen(strip_tags($question -> question['text'])) > 50){
                $question_info['general']['reduced_text']  = mb_substr(strip_tags($question -> question['text']), 0, 47).'...';
            }
            else{
                $question_info['general']['reduced_text'] = strip_tags($question -> question['text']);
            }
            $question_info['general']['type']        = $question -> question['type'];
            $question_info['general']['difficulty']  = $question -> question['difficulty'];
            $question_info['general']['content_ID']  = $question -> question['content_ID'];
            $question_info['general']['explanation'] = $question -> question['explanation'];
            $question_info['general']['options']     = $question -> question['options'];
            $question_info['general']['answer']      = $question -> question['answer'];
            $question_info['general']['timestamp']   = $question -> question['timestamp'];
            
            $times_done = 0;
            $total_score = 0;
            $done_info = ef_getTableData("done_questions", "*", "questions_ID =".$question_id);
            foreach ($done_info as $done){
                $total_score += $done['score'];
                $times_done++;
            }
            if ($times_done > 0){
                $question_info['done']['times_done'] = $times_done;
                $question_info['done']['avg_score'] = $total_score / $times_done;
            }
            else{
                $question_info['done']['times_done'] = 0;
                $question_info['done']['avg_score']  = 0;
            }
            
            $questions_info[$question_id] = $question_info;
        }
        return $questions_info;
    }
    
    
    /**
     * Get statistic information about projects
     *
     * This returns statistic info for a set of projects
     * <br/>Example:
     * <code>
     * $projects = array(2, 4);
     * $info = EfrontStats :: getProjectInfo($questions);                   //Get information for projects 2,4
     *
     * @param mixed $projects Either an array of project ids or false (request information for all existing projects)
     * @return array the project' statistic info
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function getProjectInfo($projects = false){
        $projects_info = array();
        if ($projects == false){
            $projects = ef_getTableData("projects","id");
        }
        
        foreach ($projects as $project_id){
            $project_info                           = array();
            $project                                = new EfrontProject($project_id);
            $project_info['general']['id']          = $project_id;
            $project_info['general']['title']       = $project -> project['title'];
            $project_info['general']['data']        = $project -> project['data'];
            $project_info['general']['deadline']    = $project -> project['deadline'];
            $project_info['general']['auto_assign'] = $project -> project['auto_assign'];
            
            $project_info['done'] = array();
            $assigned_data = ef_getTableData("users u, users_to_projects up", "u.LOGIN, u.name, u.surname, up.grade, up.upload_timestamp, up.status", "u.LOGIN = up.users_LOGIN and up.projects_ID=".$project_id);
            foreach ($assigned_data as $data){
                $done_project = array();
                $done_project['users_LOGIN']      = $data['LOGIN'];
                $done_project['name']             = $data['name'];
                $done_project['surname']          = $data['surname'];
                $done_project['grade']            = $data['grade'];
                $done_project['upload_timestamp'] = $data['upload_timestamp'];    
                $done_project['status']           = $data['status'];    
                $project_info['done'][]           = $done_project;
            }
            $projects_info[$project_id] = $project_info;
        }
        return $projects_info;
    }
    
    
}


?>