<?php

/**
 *
 */
class EfrontDirection extends ArrayObject
{
    /**
     *
     */
    const MAXIMUM_NAME_LENGTH = 50;

    /**
     * Instantiate direction
     *
     * This function is the class constructor, which instantiates the
     * EfrontDirection object, based on the direction values
     * <br/>Example:
     * <code>
     * $direction_array = eF_getTableData("directions", "*", "id=4");
     * $direction = new EfrontDirection($direction_array[0]);
     * </code>
     *
     * @param array $array The direction values
     * @since 3.5.0
     * @access public
     */
    function __construct($direction) {
        if (!is_array($direction)) {
            if (!eF_checkParameter($direction, 'id')) {
                throw new EfrontLessonException(_INVALIDID.': '.$direction, EfrontLessonException :: INVALID_ID);
            }
            $result = eF_getTableData("directions", "*", "id=".$direction);
            if (sizeof($result) == 0) {
                throw new EfrontLessonException(_DIRECTIONDOESNOTEXIST.': '.$direction, EfrontLessonException :: DIRECTION_NOT_EXISTS);
            }
            $direction = $result[0];
        }
        parent :: __construct($direction);
    }

    /**
     * Persist changed values
     *
     * This function is used to persist the direction values
     * <br/>Example:
     * <code>
     * $direction_array = eF_getTableData("directions", "*", "id=4");
     * $direction = new EfrontDirection($direction_array[0]);
     * $direction['name'] = 'new name';
     * $direction -> persist();
     * </code>
     *
     * @since 3.5.0
     * @access public
     */
    function persist() {
        foreach (new EfrontAttributesOnlyFilterIterator($this -> getIterator()) as $key => $value) {
            $fields[$key] = $value;
        }
        eF_updateTableData("directions", $fields, "id=".$fields['id']);
    }

    /**
     * Delete a direction
     * This function is used to delete the current direction
     * <br/>Example:
     * <code>
     * $direction_array = eF_getTableData("directions", "*", "id=4");
     * $direction = new EfrontDirection($direction_array[0]);
     * $direction -> delete();
     * </code>
     *
     * @since 3.5.0
     * @access public
     */
    function delete() {
        foreach (new EfrontAttributeFilterIterator(new RecursiveIteratorIterator(new RecursiveArrayIterator($this)), 'id') as $key => $value) {
            eF_deleteTableData("directions", "id=".$value);                                               //Delete Units from database
        }
    }

    /**
     * Get direction's lessons
     *
     * This function is used to get the lessons that belong
     * to this direction.
     * <br/>Example:
     * <code>
     * $lessons = $direction -> getLessons();
     * </code>
     *
     * @param boolean $returnObjects Whether to return EfrontLesson objects or a simple array
     * @param boolean $subDirections Whether to return subDirections lessons as well
     * @return array An array of lesson ids/names pairs or EfrontLesson objects
     * @since 3.5.0
     * @access public
     */
    function getLessons($returnObjects = false, $subDirections = false) {
        if (!$subDirections) {
            $result = eF_getTableData("lessons", "id, name", "directions_ID=".$this['id']);
        } else {
            $directionsTree = new EfrontDirectionsTree();
            $children       = $directionsTree -> getNodeChildren($this['id']);
            foreach (new EfrontAttributeFilterIterator(new RecursiveIteratorIterator(new RecursiveArrayIterator($children)), array('id')) as $value) {
                $siblings[] = $value;
            }
            $result = eF_getTableData("lessons", "id, name", "directions_ID in (".implode(",", $siblings).")");
        }
        $lessons = array();

        foreach ($result as $value) {
            $returnObjects ? $lessons[$value['id']] = new EfrontLesson($value['id']) : $lessons[$value['id']] = $value['name'];
        }

        return $lessons;
    }

    /**
     * Get direction's courses
     *
     * This function is used to get the courses that belong
     * to this direction.
     * <br/>Example:
     * <code>
     * $courses = $direction -> getCourses();
     * </code>
     *
     * @param boolean $returnObjects Whether to return EfrontCourse objects or a simple array
     * @param boolean $subDirections Whether to return subDirections lessons as well
     * @return array An array of course ids/names pairs or EfrontCourse objects
     * @since 3.5.0
     * @access public
     */
    function getCourses($returnObjects = false, $subDirections = false) {
        if (!$subDirections) {
            $result = eF_getTableData("courses", "id, name", "directions_ID=".$this['id']);
        } else {
            $directionsTree = new EfrontDirectionsTree();
            $children       = $directionsTree -> getNodeChildren($this['id']);
            foreach (new EfrontAttributeFilterIterator(new RecursiveIteratorIterator(new RecursiveArrayIterator($children)), array('id')) as $value) {
                $siblings[] = $value;
            }
            $result = eF_getTableData("courses", "id, name", "directions_ID in (".implode(",", $siblings).")");
        }
        $courses = array();

        foreach ($result as $value) {
            $returnObjects ? $courses[$value['id']] = new EfrontCourse($value['id']) : $courses[$value['id']] = $value['name'];
        }

        return $courses;
    }

    /**
     * Create direction
     *
     * This function is used to create a new direction
     * <br/>Example:
     * <code>
     * $fields = array('name' => 'new direction');
     * EfrontDirection :: createDirection($fields);
     * </code>
     *
     * @param array $fields The new direction's fields
     * @return EfrontDirection The new direction
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function createDirection($fields = array()) {
        !isset($fields['name']) ? $fields['name'] = 'Default direction' : null;

        $newId     = eF_insertTableData("directions", $fields);
        $result    = eF_getTableData("directions", "*", "id=".$newId);                                            //We perform an extra step/query for retrieving data, sinve this way we make sure that the array fields will be in correct order (forst id, then name, etc)
        $direction = new EfrontDirection($result[0]);

        return $direction;
    }
}


/**
 * Efront directions tree
 *
 */
class EfrontDirectionsTree extends EfrontTree
{
    /**
     * Initialize tree
     *
     * This function is used to initialize the directions tree
     * <br/>Example:
     * <code>
     * $directionsTree = new EfrontDirectionsTree();
     * </code>
     *
     * @since 3.5.0
     * @access public
     */
    function __construct() {
        $this -> reset();
    }

    /**
     * Reset/initialize directions tree
     *
     * This function is used to initialize or reset the directions tree
     * <br/>Example:
     * <code>
     * $directionsTree = new EfrontDirectionsTree();
     * $directionsTree -> reset();
     * </code>
     *
     * @since 3.5.0
     * @access public
     */
    public function reset() {
        $directions = eF_getTableData("directions", "*");

        if (sizeof($directions) == 0) {
            $this -> tree = new RecursiveArrayIterator(array());
            return;
        }

        foreach ($directions as $node) {                //Assign previous direction ids as keys to the previousNodes array, which will be used for sorting afterwards
            $nodes[$node['id']] = new EfrontDirection($node);        //We convert arrays to array objects, which is best for manipulating data through iterators
        }

        $rejected = array();
        $tree     = $nodes;
        $count    = 0;                                                                          //$count is used to prevent infinite loops
        while (sizeof($tree) > 1 && $count++ < 1000) {                                       //We will merge all branches under the main tree branch, the 0 node, so its size will become 1
            foreach ($nodes as $key => $value) {
                if ($value['parent_direction_ID'] == 0 || in_array($value['parent_direction_ID'], array_keys($nodes))) {        //If the unit parent is in the $nodes array keys - which are the unit ids- or it is 0, then it is  valid
                    $parentNodes[$value['parent_direction_ID']][]      = $value;               //Find which nodes have children and assign them to $parentNodes
                    $tree[$value['parent_direction_ID']][$value['id']] = array();              //We create the "slots" where the node's children will be inserted. This way, the ordering will not be lost
                } else {
                    $rejected = $rejected + array($value['id'] => $value);                   //Append units with invalid parents to $rejected list
                    unset($nodes[$key]);                                                     //Remove the invalid unit from the units array, as well as from the parentUnits, in case a n entry for it was created earlier
                    unset($parentNodes[$value['parent_direction_ID']]);
                }
            }
            if (isset($parentNodes)) {                                                       //If the unit was rejected, there won't be a $parentNodes array
                $leafNodes = array_diff(array_keys($nodes), array_keys($parentNodes));       //Now, it's easy to see which nodes are leaf nodes, just by subtracting $parentNodes from the whole set
                foreach ($leafNodes as $leaf) {
                    $parent_id = $nodes[$leaf]['parent_direction_ID'];                         //Get the leaf's parent
                    $tree[$parent_id][$leaf] = $tree[$leaf];                                 //Append the leaf to its parent's tree branch
                    unset($tree[$leaf]);                                                     //Remove the leaf from the main tree branch
                    unset($nodes[$leaf]);                                                    //Remove the leaf from the nodes set
                }
                unset($parentNodes);                                                         //Reset $parentNodes; new ones will be calculated at the next loop
            }
        }
        if (sizeof($tree) > 0 && !isset($tree[0])) {                                         //This is a special case, where only one node exists in the tree
            $tree = array($tree);
        }

        if (sizeof($rejected) > 0) {                                            //Append rejected nodes to the end of the tree array, updating their parent/previous information
            foreach ($rejected as $key => $value) {
                eF_updateTableData("directions", array("parent_direction_ID" => 0), "id=".$key);
                $value['parent_direction_ID'] = 0;
                $tree[0][] = $value;
            }
        }

        $this -> tree = new RecursiveArrayIterator($tree[0]);
    }

    /**
     * Insert node to the tree
     *
     * @param mixed $node
     * @param mixed $parentNode
     * @param mixed $previousNode
     * @since 3.5.0
     * @access public
     */
    public function insertNode($node, $parentNode = false, $previousNode = false) {}

    /**
     * Remove node from tree
     *
     * @param mixed $node
     * @since 3.5.0
     * @access public
     */
    public function removeNode($node) {



    }

    /**
     * Print an HTML representation of the directions tree
     *
     * This function is used to print an HTML representation of the HTML tree
     * <br/>Example:
     * <code>
     * $directionsTree -> toHTML();                         //Print directions tree
     * </code>
     *
     * @param RecursiveIteratorIterator $iterator An optional custom iterator
     * @param array $lessons An array of EfrontLesson Objects
     * @return string The HTML version of the tree
     * @since 3.5.0
     * @access public
     */
    public function toHTML($iterator = false, $lessons = false, $courses = false) {
        if (!$iterator) {
            $iterator = new EfrontNodeFilterIterator(new RecursiveIteratorIterator(new RecursiveArrayIterator($this -> tree), RecursiveIteratorIterator :: SELF_FIRST));
        }

        if ($lessons === false) {                                                    //If a lessons list is not specified, get all lessons
            $result = eF_getTableData("lessons", "*");                               //Get all lessons at once, thus avoiding looping queries
            foreach ($result as $value) {
                $lessons[$value['id']] = new EfrontLesson($value);                   //Create an array of EfrontLesson objects
            }
        }
        $directionsLessons = array();
        foreach ($lessons as $id => $lesson) {
            if (!$lesson -> lesson['course_only']) {
                $directionsLessons[$lesson -> lesson['directions_ID']][] = $id;     //Create an intermediate array that maps lessons to directions
            }
        }
        if ($courses === false) {                                                   //If a courses list is not specified, get all courses
            $result = eF_getTableData("courses", "*");                              //Get all courses at once, thus avoiding looping queries
            foreach ($result as $value) {
                $courses[$value['id']] = new EfrontCourse($value);                  //Create an array of EfrontCourse objects
            }
        }
        $directionsCourses = array();
        foreach ($courses as $id => $course) {
            $directionsCourses[$course -> course['directions_ID']][] = $id;        //Create an intermediate array that maps courses to directions
        }
        $systemRoles = EfrontLessonUser :: getSystemRoles();

        //We need to calculate which directions will be displayed. We will keep only directions that have lessons or courses and their parents. In order to do so, we traverse the directions tree and set the 'hasNodes' attribute to the nodes that will be kept
        foreach ($iterator as $key => $value) {
            if (isset($directionsLessons[$value['id']]) || isset($directionsCourses[$value['id']])) {
                $count = $iterator -> getDepth();
                $value['hasNodes'] = true;
                isset($directionsLessons[$value['id']]) ? $value['lessons'] = $directionsLessons[$value['id']] : null;        //Assign lessons ids to the direction
                isset($directionsCourses[$value['id']]) ? $value['courses'] = $directionsCourses[$value['id']] : null;        //Assign courses ids to the direction
                while ($count) {
                    $node = $iterator -> getSubIterator($count--);
                    $node['hasNodes'] = true;                        //Mark "keep" all the parents of the node
                }
            }
        }

        $iterator = new EfrontNodeFilterIterator($iterator, array('hasNodes' => true));    //Filter in only tree nodes that have the 'hasNodes' attribute

        $iterator   -> rewind();
        $current    = $iterator -> current();
        $treeString = '            			
        				<a href = "javascript:void(0)" onclick = "showAll()">'._SHOWALL.'</a> / <a href = "javascript:void(0)" onclick = "hideAll()">'._HIDEALL.'</a>';
        while ($iterator -> valid()) {
            $children = array();
            foreach (new EfrontNodeFilterIterator(new ArrayIterator($this -> getNodeChildren($current), RecursiveIteratorIterator :: SELF_FIRST)) as $key => $value) {
                $children[] = $key;
            }
            $treeString .= '
                        <table id = "direction_'.$current['id'].'" style = "margin-top:10px;margin-left:'.(20 * $iterator -> getDepth()).'px">
                            <tr><td class = "lessonsList" width = "1%">
                                    <img name = "default_visible_image" id = "subtree_img'.$current['id'].'" src = "images/others/minus.png" style = "vertical-align:middle" align = "center" onclick = "showHideDirections(this, \''.implode(",", $children).'\', \''.$current['id'].'\', (this.getAttribute(\'src\')).match(/plus/) ? \'show\' : \'hide\')">
                                    <span style = "display:none" id = "subtree_children_'.$current['id'].'">'.implode(",", $children).'</span>
                                </td>
                                <td class = "lessonsList">
                                    <table>
                                        <tr><td><img src = "images/24x24/directions.png" alt= "Categories" title="Categories" /></td>
                                            <td class = "lessonsList_title">'.$current['name'].'</td>
                                        </tr>
                                    </table>
                                </td>
                            </tr>';

            if (sizeof($current['lessons']) > 0) {
                $treeString .= '
                            <tr id = "subtree'.$current['id'].'" name = "default_visible">
                                <td class = "lessonsList_nocolor">&nbsp;</td>
                                <td>';

                    $treeString .= '
                                    <table width = "100%">';
                foreach ($foo = $current['lessons'] as $lessonId) {
                    $treeString .= '
                                    	<tr>';
                    if ($systemRoles[$lessons[$lessonId] -> userStatus['user_type']] == 'student') {
                        if ($lessons[$lessonId] -> userStatus['completed']) {
                            $treeString .= '
                        					<td style = "width:5px;text-align:center">
                            					<img src = "images/16x16/check.png" alt = "'._LESSONCOMPLETE.'" title = "'._LESSONCOMPLETE.'" style = "margin-left:10px;vertical-align:middle" />
                            				</td>';                            
                        } else {
                            $treeString .= '
                                            <td style = "width:50px;">
                                                <span style = "position:absolute;text-align:center;width:50px;border:1px solid #d3d3d3;vertical-align:middle;z-index:2">'.$lessons[$lessonId] -> userStatus['percentage_done'].'%</span>
                                                <span style = "background-color:#A0BDEF;width:'.($lessons[$lessonId] -> userStatus['percentage_done']/2).'px;border:1px dotted #d3d3d3;position:absolute">&nbsp;</span>
                                                &nbsp;&nbsp;
                                            </td>';
                            
                        }
                    } else {
                            $treeString .= '
                        					<td></td>';
                    }
                    if ($systemRoles[$lessons[$lessonId] -> userStatus['user_type']]) {            //If there is a userStatus in the lesson, the user has taken it; otherwise, he doesn't have it
                        $treeString .= '
                                            <td class = "lessonsList_lessons">&nbsp;
                                                '.$lessons[$lessonId] -> toHTMLTooltipLink($systemRoles[$lessons[$lessonId] -> userStatus['user_type']].'.php?ctg=control_panel&lessons_ID='.$lessonId).'
                                                '.($lessons[$lessonId] -> userStatus['user_type'] != $lessons[$lessonId] -> userStatus['basic_user_type'] ? '&nbsp;<span style = "color:green; font-size: 9px; display:inline">('.$lessons[$lessonId] -> userStatus['user_type'].')</span>' : null).'
                                            </td>';
                    } else {
                        $treeString .= '
                                                <td class = "lessonsList_lessons" >&nbsp;
                                                '.$lessons[$lessonId] -> toHTMLTooltipLink('javascript: void(0)').'&nbsp;<span style = "color:green; font-size: 9px; display:inline">(
                                                '.($lessons[$lessonId] -> lesson['price'] ? $lessons[$lessonId] -> lesson['price'].'&nbsp;'.$GLOBALS['CURRENCYSYMBOLS'][$GLOBALS['configuration']['currency']] : _FREELESSON).
                                                ' )</span>&nbsp;<a href = ""><img style = "vertical-align:middle" src = "images/16x16/money.png" title = "'._BUYLESSON.'" alt = "'._BUYLESSON.'" border = "0"></a>
                                            </td>';
                    }
                        $treeString .= '
                                    	</tr>';
                }
                    $treeString .= '
                                    </table>';
                $treeString .= '
                                </td>
                            </tr>';
            }
            if (sizeof($current['courses']) > 0) {
                $treeString .= '
                            <tr id = "subtree'.$current['id'].'" name = "default_visible">
                                <td class = "lessonsList_nocolor">&nbsp;</td>
                                <td>';
                foreach ($current['courses'] as $courseId) {
                    $treeString .= $courses[$courseId] -> toHTML();
                }
                $treeString .= '
                                </td>
                            </tr>';
            }
            $treeString .= '
                        </table>';

            $iterator -> next();
            $current = $iterator -> current();
        }
        
        $treeString .= "
                        <script>
                        	function showAll() {
                        		$$('tr').each(function (tr) 	  {tr.id.match(/subtree/) ? tr.show() : null;});
                           		$$('table').each(function (table) {table.id.match(/direction_/) ? table.show() : null;});
                           		$$('img').each(function (img) {img.src.match(/plus/) ? img.setAttribute('src', 'images/others/minus.png') : null;});
                        	}
                        	function hideAll() {
                        		$$('tr').each(function (tr) 	  {tr.id.match(/subtree/) ? tr.hide() : null;});
                           		//$$('table').each(function (table) {table.id.match(/direction_/) ? table.hide() : null;});
                           		$$('img').each(function (img) {img.src.match(/minus/) ? img.setAttribute('src', 'images/others/plus.png') : null;});
                        	}
                        	
                            function showHideDirections(el, ids, id, mode) {
                            	Element.up(el);		//IE intialization... for some reason, this is needed in many situations, where a simple call like el.up() just doesn't work for IE
                                                                
                                if (mode == 'show') {
                            		el.up().up().nextSiblings().each(function(s) {s.show()});
                                    if (ids) {
                                        ids.split(',').each(function (s) { showHideDirections($('subtree_img'+id), $('subtree_children_'+s).innerHTML, s, 'show') });
                                        ids.split(',').each(function (s) { obj = $('direction_'+s); obj.show()});
    								}
    								$('subtree_img'+id).setAttribute('src', 'images/others/minus.png');
    							} else {
                            		el.up().up().nextSiblings().each(function(s) {s.hide()});
                                    if (ids) {
                                        ids.split(',').each(function (s) { showHideDirections($('subtree_img'+id), $('subtree_children_'+s).innerHTML, s, 'hide') });
                                        ids.split(',').each(function (s) { obj = $('direction_'+s); obj.hide()});
    								}
                            		$('subtree_img'+id).setAttribute('src', 'images/others/plus.png');
    							}
                            }
                        </script>";

        return $treeString;
    }
                                    //show_hide($(\'subtree_img'.$current['id'].'\'), \'subtree'.$current['id'].'\');

    /**
     * Print paths string
     *
     * This function is used to print the paths to the each direction
     * based on its ancestors.
     * <br/>Example:
     * <code>
     * $paths = $directionsTree -> toPathString();  //$paths is an array with direction ids as keys, and paths as values, for example 'Direction 1 -> Directions 1.1 -> Direction 1.1.1'
     * </code>
     *
     * @param boolean $$includeLeaf Whether leaf direction will be included to the path string
     * @return array The direction paths
     * @since 3.5.0
     * @access public
     */
    public function toPathString($includeLeaf = true) {
        $iterator = new EfrontNodeFilterIterator(new RecursiveIteratorIterator(new RecursiveArrayIterator($this -> tree), RecursiveIteratorIterator :: SELF_FIRST));
        foreach ($iterator as $id => $value) {
            $values = array();
            foreach ($this -> getNodeAncestors($id) as $direction) {
                $values[] = $direction['name'];
            }
            if (!$includeLeaf) {
                unset($values[0]);
            }
            $parentsString[$id] = implode('&nbsp;&rarr;&nbsp;', array_reverse($values));
        }

        return $parentsString;
    }

}


?>