<?php

/**
* 
* Content related functions
*
* This file contains all the content related functions, and especially Content Tree 
* creation and manipulation
*
* @package efront
*/

/**
* Checks if a unit is current
*
* The function input is a unit id. It checks if the corresponding unit is assigned to an active period, or not assigned. 
* If so, then it means that it is an active unit, so it returns true; otherwise false is returned.
* <br/>Example:
* <code>
* //Unit 3 is not scheduled to any period; Unit 5 is scheduled to a period which starts at the future; Unit 10 is scheduled to a period currently active
* $is_current = eF_isCurrentContent(3);                     //is_current: true;
* $is_current = eF_isCurrentContent(5);                     //is_current: false;
* $is_current = eF_isCurrentContent(10);                    //is_current: true;
* </code> 
*
* @param int $id The unit id
* @return bool true if the unit is current
* @version 1.2
* Changes from version 1.1 to 1.2:
* - Changed back the behavior, and now it returns true instead of false if a unit does not belong to period
* Changes From version 1.0 to 1.1:
* - If a unit does not belong to a period, it returns false instead of true
* basiki diafora.... to erotima den einai an to unit den anikei se kammia periodo, alla an to gia to mathima exoyn oristei periodoi
* an px yparxoyn 3 periodoi kai to unit den anikei se kammia, prepei  na epistrefete false  makriria 26/1/2007
*/
function eF_isCurrentContent($id)
{
    $time   = time();
    $offset = eF_getOffset();

    $res    = eF_getTableData("current_content,periods,content", "content_ID", "content_ID=$id AND current_content.periods_ID=periods.id AND periods.from_timestamp+$offset<=$time AND periods.to_timestamp+$offset>$time AND content.id=current_content.content_ID AND content.active=1");
    if (isset($res[0]['content_ID'])) {
        return true;
    } else {
        $res1 = eF_getTableData("periods,content", "count(*)", "periods.lessons_ID=content.lessons_ID AND content.id=$id"); 
        if ($res1[0]['count(*)'] == 0) {
            return true;
        } else {
            return false;
        }
      
    }
}


/**
* Return number of current units 
*
* This function calculates the number of units of the specified type that are current.
* <br/> Example:
* <code>
* echo eF_numCurrentContent('theory');      //returns 5
* </code>
*
* @param string $ctg_type The content type, can be 'theory', 'examples', 'projects', 'tests'
* @return int The number of units of this type
* @version 1.0
* 22/12/2005
*/
function eF_numCurrentContent($ctg_type, $lesson = false)      // makriria. add 2nd parameter
{
    if($lesson == false) {
        $lesson = $_SESSION['s_lessons_ID'];
    } 

    if($ctg_type == "projects"){ // new way of counting new type of exercises 2007/08/06
        $result = eF_getTableData("projects,users_to_projects", "id", "lessons_ID=".$lesson." AND projects_ID=id AND users_LOGIN='".$_SESSION['s_login']."'");
    }else{
        $time = time();
        $offset = eF_getOffset();
        $result = eF_getTableData("periods", "id", "lessons_ID=".$lesson);
        if (sizeof($result) > 0) {
            $result = eF_getTableData("current_content,periods,content", "distinct content.id", "ctg_type='$ctg_type' AND periods.lessons_ID = ".$lesson." AND current_content.periods_ID=periods.id AND periods.from_timestamp+$offset<=$time AND periods.to_timestamp+$offset>$time AND content.id=current_content.content_ID AND content.active=1");
        } else {
            $result = eF_getTableData("content", "distinct content.id", "ctg_type='$ctg_type' AND lessons_ID=".$lesson." AND active=1");
        }
    }
    return sizeof($result);
}

/**
* Checks if a unit is current and active
*
* This function will check if a unit is both active and current. This is how it works: If a lesson does not have any periods defined, then 
* all units are considered active and current, so this function returns true for every unit. If a lesson has periods defined, then the funtion 
* returns true if the specified unit is part of at least one of the *current* periods; otherwise, it returns false, meaning that this unit does 
* not belong to any current period (even if it belongs to a period that is not current)
* Note: as of eFront 3.0 this function is deprecated and you should use eF_checkCurrentContentInTree() instead
*
* @param int $id The unit id
* @return bool true if the unit is currently active 
* @version 1.0
* @see eF_checkCurrentContentInTree()
*/
function eF_isDoneContent($id)
{

    $time   = time();
    $offset = eF_getOffset();
    $res    = eF_getTableData("current_content,periods,content", "content_ID", "content_ID=$id AND current_content.periods_ID=periods.id AND periods.from_timestamp+$offset<=$time AND content.id=current_content.content_ID AND content.active=1");
    if (isset($res[0]['content_ID'])) {
        return true;
    } else { 
        $res1 = eF_getTableData("periods,content", "count(*)", "periods.lessons_ID=content.lessons_ID AND content.id=$id"); 
        if ($res1[0]['count(*)'] == 0) {
            return true;
        } else {
            return false;
        }
    }
}

/**
* Check each unit if it is current / done
*
* This function will check each tree unit if it is both current or done, 
* adding 2 extra fields in the tree array, isDoneContent and isCurrentContent. 
* This is how it works: If a lesson does not have any periods defined, then 
* all units are considered current and done, so this function sets both 
* fields to true for every unit. 
* If a lesson has periods defined, then the function:
*     - sets isCurrentContent field to:
*        TRUE if the specified unit is part of at least one of the *current* periods, 
*            meaning that the period started before now and ends after now; 
*        FALSE if the unit does not belong to any current period (even if it 
*            belongs to some other period)
*     - sets isDoneContent field to:
*        TRUE if the specified unit is part of at least one of *current or past* periods, 
*            meaning that the period started before now, no matter when it ended (or ends)
*        FALSE if the unit does not belong to any period that started in the past
* 
* For each unit, the function sets the fields isDoneContent and isCurrentContent in the tree to either true or false
*
* Note: This function is a replacement to eF_isDoneContent and eF_isCurrentContent
*
* @param array $tree The content tree
* @return array tree The new tree array
* @version 1.0
*/
function eF_checkContentInTree($tree)
{
    $time   = time();
    $offset = eF_getOffset();       //Get the user offset, if such exists
//pr($tree);    
    if (sizeof($tree) == 0) {
        $ids[] = 0;
    } else {
        foreach ($tree as $branch) {
            if ($branch['id']) {
                $ids[] = $branch['id'];     //Make a list of ids, that will be used in the query
            }
        }
    }
    
//pr($ids);
    $result_done    = eF_getTableDataFlat("current_content,periods,content", "current_content.content_ID", "current_content.content_ID in (".implode(",",$ids).") AND current_content.periods_ID=periods.id AND periods.from_timestamp+$offset<=$time AND content.id=current_content.content_ID AND content.active=1");
    $result_current = eF_getTableDataFlat("current_content,periods,content", "current_content.content_ID", "current_content.content_ID in (".implode(",",$ids).") AND current_content.periods_ID=periods.id AND periods.from_timestamp+$offset<=$time AND periods.to_timestamp+$offset>$time AND content.id=current_content.content_ID AND content.active=1");

    $done_ids       = array_combine($result_done['content_ID'], $result_done['content_ID']);                //Copy the array values to keys
    $current_ids    = array_combine($result_current['content_ID'], $result_current['content_ID']);          //Copy the array values to keys

    if (sizeof($tree) > 0) {
        $lesson_periods = eF_getTableData("periods,content", "count(*)", "periods.lessons_ID=content.lessons_ID AND content.id=".$tree[0]['id']);       //check if this lesson has periods
    }
    $lesson_periods[0]['count(*)'] == 0 ? $lesson_periods = false : $lesson_periods = true;           //Make $lesson_periods a boolean value, based on whether this lesson has periods or not
    
    for ($i = 0; $i < sizeof($tree); $i++) {
        if (isset($done_ids[$tree[$i]['id']])) {                    //If the unit is in a current or past period, set the isDoneContent field in the $tree array to true
            $tree[$i]['isDoneContent'] = true;
        } else {
            if ($lesson_periods) {
                $tree[$i]['isDoneContent'] = false;                 //If the lesson has periods and the unit is not part of a current or past one, set isDoneContent to false
            } else {
                $tree[$i]['isDoneContent'] = true;                  //If the lesson does not have any periods, set isDoneContent to true
            }
        }

        if (isset($current_ids[$tree[$i]['id']])) {                 //If the unit is in a current period, set the isCurrentContent field in the $tree array to true
            $tree[$i]['isCurrentContent'] = true;
        } else {
            if ($lesson_periods) {
                $tree[$i]['isCurrentContent'] = false;              //If the lesson has periods and the unit is not part of a current one of them, set isCurrentContent to false
            } else {
                $tree[$i]['isCurrentContent'] = true;               //If the lesson does not have any periods, set isCurrentContent to true
            }
        }
    }
//pr($tree);
    return $tree;
}
/**
* Gets dates offset for a student
*
* This functions is used to add an offset to all dates concerning a student that has 
* enrolled late to the system. The amount of time he enrolled after the first period 
* started is added to the current time, in order to normalize the content flow per period
*
* @return int The date offset is an integer to be added to a timestamp
* @version 1.0.1
* Changes from version 1.0 to 1.0.1:
* - Fixed bug: it always reported offset.
*/
function eF_getOffset()
{
    if ($_SESSION['s_lesson_options']['dynamic_periods']) {
        $res           = eF_getTableData("users_to_lessons", "from_timestamp", "lessons_ID=".$_SESSION['s_lessons_ID']." AND users_LOGIN='".$_SESSION['s_login']."'");
        $new_timestamp = $res[0]['from_timestamp'];
        if ($new_timestamp > 0) {
            $periods       = eF_getTableData("periods", "id, name, from_timestamp, to_timestamp", "lessons_ID=".$_SESSION['s_lessons_ID'], "from_timestamp ASC");
            if (sizeof ($periods) > 0) {
                $old_timestamp = $periods[0]['from_timestamp'];
                $temp          = $new_timestamp - $old_timestamp;
                if ($new_timestamp > $old_timestamp) {
                    $offset = $new_timestamp - $old_timestamp;
                }
            }
        }
    }
    if (!isset($offset)) {
        $offset = 0;
    }
    return $offset;
}

/**
* Children of a unit
*
* This function returns an array containing the children of a unit. If a lesson id is specified *and*
* the unit content id is 0, then all the units which are part of that lesson are returned.
* <br/>Example:
* <code>
* $unit_children = eF_getChildren(3);           //returns an array with the ids of all the children units of unit with id 3
* $lesson_units  = eF_getChildren(0, 1);        //returns an array with the ids of all the units contained in the lesson with id 1
* </code>
* 
* @param int $content_ID The id of the unit
* @param int $lessons_ID The id of the lesson. It is concidered only if $content_ID is 0
* @return array The unit's children ids
* @version 1.2
* Changes from 1.1 to 1.2 (2007/10/01 - venakis):
* - Fixed bug in children calculation (it returned fewer than normal)
* Changes from 1.0 to 1.1 (18/9/05):
* - Added the unset command before the return 
*/
function eF_getChildren($content_ID, $lessons_ID = false, $tree = null, $tree_indexes = null) 
{

    if ($tree) {
        if (!$tree_indexes) {
            for ($i = 0; $i < sizeof($tree); $i++) {
                $tree_indexes[$tree[$i]['id']] = $i;
            }
        }

        if ($content_ID == 0) {
            $children = array_keys($tree_indexes);
        } else {        
            $children[] = $content_ID;
            $size       = sizeof($tree);
            $found      = true;
            $count      = 0;
            
            //for ($i = 0; $size > 0 && $found == true; $i++) {
            while($size > 0 && $found == true) {
                foreach ($tree as $key => $node) {
                    $i++;
                    if (in_array($node['parent_id'], $children)) {
                        $children[] = $node['id'];
                        unset($tree[$key]);
                        $found = true;
                        $i = 0;
                    } 
                    $size = sizeof($tree);
                    if ($i == $size) {
                        $found = false;
                    }
                }
            }
            
            unset($children[0]);                                                 //This is the unit itself.
        }
    } else {        
        $now            = 0;
        $count          = 0;
        $children[0] = $content_ID;

        while ($now <= $count) {
            if ($content_ID == 0 AND $lessons_ID) {
                $add_sql = "AND lessons_ID = $lessons_ID";
            } else {
                $add_sql = "";
            }
            $parent = $children[$now];
            $res    = eF_getTableData("content", "id", "parent_content_ID=$parent $add_sql", "previous_content_ID");

            for ($i = 0; $i < sizeof($res); $i++) {
                $count++;
                $children[$count] = $res[$i]['id'];
            }
            $now++;
        }

        unset($children[0]);                                                 //This is the unit itself.
    }
    
    return $children;
}


/**
* Parents of a unit
*
* This function returns an array containing the parents of a unit. 
* For example, for the unit 3.2.1, the array will hold the ids of the units 3.2.1,
* 3.2, 3 and the 0.
* <br/>Example:
* <code>
* $parents = eF_getParents(213);
* print_r($parents);
* //Returns:
* //Array
* //(
* //    [0] => 213
* //    [1] => 211
* //    [2] => 206
* //    [3] => 0
* //)
* </code>
* 
* @param int $content_ID The id of the unit
* @param array $tree The content tree
* @return array The unit's parents ids
* @version 2.0
* Changes from version 1.0 to version 2.0 (2007/05/15 - venakis):
* - Rewritten so that if $tree is specified, it doesn't perform any database queries
*/

function eF_getParents($content_ID, $tree = null, $tree_indexes = null) 
{
    if ($tree) {
        if (!$tree_indexes) {
            for ($i = 0; $i < sizeof($tree); $i++) {
                $tree_indexes[$tree[$i]['id']] = $i;
            }
        }

        $parents['id'][]   = $content_ID;
        $parents['name'][] = $tree[$tree_indexes[$content_ID]]['name'];
        
        while ($tree[$tree_indexes[$content_ID]]['parent_id'] != 0) {
            $content_ID = $tree[$tree_indexes[$content_ID]]['parent_id'];
            $parents['id'][]   = $content_ID;
            $parents['name'][] = $tree[$tree_indexes[$content_ID]]['name'];
        }
        
    } else {
        $now            = 0;
        $count          = 0;
        $parents['id'][0] = $content_ID;

        while ($now <= $count) {
            $child = $parents['id'][$now];
            $res   = eF_getTableData("content", "parent_content_ID, name", "id=$child");
            for ($i = 0; $i < sizeof($res) && $res[$i]['parent_content_ID'] != 0; $i++) {
                $parents['name'][$count] = $res[$i]['name'];
                $count++;
                $parents['id'][$count] = $res[$i]['parent_content_ID'];
            }
            $parents['name'][$count] = $res[$i]['name'];
            $now++;
        }
    }
    
    return $parents;
}
/*
function eF_getParents($content_ID) 
{
    $now            = 0;
    $count          = 0;
    $content_IDs[0] = $content_ID;
    
    while ($now <= $count) {
        $child = $content_IDs[$now];
        $res   = eF_getTableData("content", "parent_content_ID", "id=$child");
        for ($i = 0; $i < sizeof($res); $i++) {
            $count++;
            $content_IDs[$count] = $res[$i]['parent_content_ID'];
        }
        $now++;
    }
    return $content_IDs;;
}
*/

/**
* Calculates the lesson tree
*
* This function is used to create an array with all the lesson's units, sorted the right way, so that 
* parent-children and previous-next relationships are preserved. Each element of the array is another array,
* whose elements are holding unit information: Specifically, these elements are:
* - 'id'        : the unit id
* - 'name'      : the name of the unit
* - 'ctg_type'  : the unit type
* - 'parent_id' : the id of the unit's parent
* - 'level'     : The level of the unit
* - 'active'    : whether it is an active unit
* - 'data'      : Whether the unit has data.
* If a content id is specified, then only the tree under and including the specified unit is returned. content id
* defaults to 0, which means the entire lesson tree. We may also specify the content type, using the $ctg parameter. 
* This will result in returning a tree containing only units of the specified type. The $ctg_in_tree variable is 
* only used when $only_current or $only_done variables are set to true. It is an array that, after the function
* returns, will contain the unit types present in the list. So, if only theory and exercise units are present in
* the tree, then this variable will be equivalent to: array('theory'=>'theory', 'exercise'=>'exercise');
* <br/>Example:
* <code>
* // We name the first variable $nouse because we will not be using it
* // This call returns the full lesson tree, only active and done units.
* eF_getContentTree($nouse, $lessons_ID, 0, false, true, false, true, false);
* </code>
* 
* @param array $ctg_in_tree Lists the unit types present in the list.
* @param int $lessons_ID The lesson id
* @param int $content_ID The content id
* @param string $ctg The content type
* @param bool $only_active Shows only active units
* @param bool $only_current Shows only current units
* @param bool $only_done Shows only units done so far
* @param mixed $only_period If true, then it shows only units that belong to a currently active period. Otherwise, 
*                            if set to a number, then it shows only units assigned to the period with this id
* @param string $only_unseen Gets only units that the student with the specified login hasn't seen
* @return array The content tree
* @version 1.2.1
* Changes from version 1.2 to 1.2.1 (20/11/2005):
* - Replaced > with >= in the $only_period part, since, without it, it was displaying the first unit as clickable in the student tree, even if it wasn't eligible (i.e. out of period)
* Changes from version 1.1 to 1.2 (18/11/2005):
* - Moved the part that examines the $ctg at the end of the function body, in order to exclude units from the student theory / examples etc tree. 
*   Before that, parent units were appearing, of units not in the current period
* - Replaced > with >= in the $ctg check part
* Changes from version 1.0 to 1.1 (3/11/2005):
* - Replaced >= with > in many places
*
*/
function eF_getContentTree(&$ctg_in_tree, $lessons_ID = false, $content_ID = 0, $ctg = false, $only_active = true, $only_current = false, $only_done = false, $only_period = false, $only_unseen = false)
{
    if (!$lessons_ID) {
        return "";
    }
    if ($content_ID == "" OR $content_ID === false) {                                                   //defaults to 0
        $content_ID = 0;
    } 
    $counter = 0;
    $level   = 0;

    $tree = eF_createContentStructure($content_ID, $lessons_ID, $only_active);           //New way of getting the tree, at least 5 times faster (and many lines of code less)!

    $tree = eF_checkContentInTree($tree);                                                //Get done and current units (see function definition for more information on what "done" and "current" means)

    if ($_SESSION['s_type'] == 'student') {
        $seen_content = eF_getSeenContent($_SESSION['s_login'], $lessons_ID);
        //print_r($seen_content);
        foreach ($tree as $key => $value)
        {
            if (in_array($value['id'], $seen_content)) {
                $tree[$key]['seen']='yes';
            }
        }
    }    
/*
    foreach ($tree as $key => $value)
    {
        foreach ($seen_content as $key1 => $value1)
        {
            if ($value['id']==$value1['id'])
                $tree[$key]['seen']='yes';
        }
    }
*/
    $keep = array();

    if ($only_current) {
        $counter = 0;
        for ($i = sizeof($tree) - 1; $i >= 0; $i--) {
            //if (!eF_isCurrentContent($tree[$i]['id'])) {
            if (!$tree[$i]['isCurrentContent']) {
                $parent_keep = false;
                for ($k = 0; $k < sizeof($keep) AND !$parent_keep; $k++) {
                    if ($tree[$i]['id'] == $keep[$k]) {
                        $parent_keep    = true;
                        $keep[$counter] = $tree[$i]['parent_id'];
                        $counter++;
                    }
                }
                if (!$parent_keep) {
                    $tree = array_merge(array_slice($tree, 0, $i), array_slice($tree, $i + 1, sizeof($tree) - 1));
                }
            } else {
                $ctg_in_tree[$tree[$i]['ctg_type']] = $tree[$i]['ctg_type'];
                $keep[$counter] = $tree[$i]['parent_id'];
                $counter++;
            }
        }
    }
//print_r($tree);    
    $keep = array();

    if ($only_done) {
        $counter = 0;
        for ($i = sizeof($tree) - 1; $i >= 0; $i--) {
            //if (!eF_isDoneContent($tree[$i]['id'])) {
            if (!($tree[$i]['isDoneContent'])) {
                $parent_keep = false;
                for($k = 0; $k < sizeof($keep) AND !$parent_keep; $k++) {
                    if($tree[$i]['id'] == $keep[$k]) {
                        $parent_keep    = true;
                        $keep[$counter] = $tree[$i]['parent_id'];
                        $counter++;
                    }
                }
                if(!$parent_keep)
                    $tree = array_merge(array_slice($tree, 0, $i), array_slice($tree, $i + 1, sizeof($tree) - 1));
            } else {
                $ctg_in_tree[$tree[$i]['ctg_type']] = $tree[$i]['ctg_type'];
                $keep[$counter] = $tree[$i]['parent_id'];
                $counter++;
            }
        }
    }
    
    $keep = array();
    
    if ($only_active) {
        $counter = 0;
        for ($i = sizeof($tree) - 1; $i >= 0; $i--) {
            //if (!eF_isDoneContent($tree[$i]['id'])) {
            if (!$tree[$i]['isDoneContent']) {
                $parent_keep = false;
                for ($k = 0; $k < sizeof($keep) AND !$parent_keep; $k++) {
                    if ($tree[$i]['id'] == $keep[$k]) {
                        $parent_keep    = true;
                        $keep[$counter] = $tree[$i]['parent_id'];
                        $counter++;
                    }
                }
                if (!$parent_keep) {
                    $tree[$i]['active'] = false;
                }
            } else {
                $keep[$counter] = $tree[$i]['parent_id'];
                $counter++;
            }
        }
    }
    
    $keep = array();
    
    if ($only_period AND $only_period === true) {                                     
        $result = eF_getTableData("current_content", "content_ID");
        for ($i = 0; $i < sizeof($result); $i++) {
            $currentContent[$result[$i]['content_ID']] = $result[$i]['content_ID'];
        }

        $counter = 0;
        for ($i = sizeof($tree) - 1; $i >= 0; $i--) {
            //$res = eF_getTableData("current_content", "content_ID", "content_ID=".$tree[$i]['id']);
            //if (sizeof($res) == 0) {
            if (!isset($currentContent[$tree[$i]['id']])) {
                $parent_keep = false;
                for ($k = 0; $k < sizeof($keep) AND !$parent_keep; $k++) {
                    if ($tree[$i]['id'] == $keep[$k]) {
                        $parent_keep    = true;
                        $keep[$counter] = $tree[$i]['parent_id'];
                        $counter++;
                    }
                }
                if (!$parent_keep) {
                    $tree = array_merge(array_slice($tree, 0, $i), array_slice($tree, $i + 1, sizeof($tree) - 1));
                } else {
                    $tree[$i]['active'] = false;
                }
            } else {
                $keep[$counter] = $tree[$i]['parent_id'];
                $counter++;
            }
        }
    } elseif ($only_period) {
        $counter = 0;
        for ($i = sizeof($tree) - 1; $i >= 0; $i--) {
            $res = eF_getTableData("current_content", "content_ID", "content_ID=".$tree[$i]['id']." AND periods_ID=$only_period");
            if (sizeof($res) == 0) {
                $parent_keep = false;
                $keep        = array();
                for ($k = 0; $k < sizeof($keep) AND !$parent_keep; $k++) {
                    if ($tree[$i]['id'] == $keep[$k]) {
                        $parent_keep    = true;
                        $keep[$counter] = $tree[$i]['parent_id'];
                        $counter++;
                    }
                }
                if (!$parent_keep) {
                    $tree = array_merge(array_slice($tree, 0, $i), array_slice($tree, $i + 1, sizeof($tree) - 1));
                } else {
                    $tree[$i]['active'] = false;
                }
            } else {
                $keep[$counter] = $tree[$i]['parent_id'];
                $counter++;
            }
        }
    }
    
    $keep = array();
/*
    if ($only_unseen) {
        $counter = 0;
        for ($i = sizeof($tree) - 1; $i >= 0; $i--) {
            $res = eF_getTableData("logs","count(*) as times","(action='content' OR action='lessons' OR action = 'tests') AND users_LOGIN='".$only_unseen."' AND comments=".$tree[$i]['id']);
            if ($res[0]['times'] > 0 OR ($tree[$i]['data'] == false && $tree[$i]['ctg_type'] != 'tests')) {
                $parent_keep = false;
                for ($k = 0; $k < sizeof($keep) AND !$parent_keep; $k++) {
                    if ($tree[$i]['id'] == $keep[$k]) {
                        $parent_keep    = true;
                        $keep[$counter] = $tree[$i]['parent_id'];
                        $counter++;
                    }
                }
                if (!isset($tree[$i]['seen']) && $tree[$i]['ctg_type'] == 'tests') {            //This line checks against 'seen' attribute for tests, so that a test is considered 'unseen' not if it hasn't been visited, but if it hasn't been completed
                    //do nothing
                } else if (!$parent_keep) {
                    $tree = array_merge(array_slice($tree, 0, $i), array_slice($tree, $i + 1, sizeof($tree) - 1));
                } else {
                    $tree[$i]['active'] = false;
                }
            } else {
                $keep[$counter] = $tree[$i]['parent_id'];
                $counter++;
            }
        }
    }

    $keep = array();
*/
    if ($only_unseen) {
        $counter = 0;
        for ($i = sizeof($tree) - 1; $i >= 0; $i--) {
            if ($tree[$i]['seen'] == 'yes') {
                $parent_keep = false;
                for ($k = 0; $k < sizeof($keep) AND !$parent_keep; $k++) {
                    if ($tree[$i]['id'] == $keep[$k]) {
                        $parent_keep    = true;
                        $keep[$counter] = $tree[$i]['parent_id'];
                        $counter++;
                    }
                }
                if (!isset($tree[$i]['seen']) && $tree[$i]['ctg_type'] == 'tests') {            //This line checks against 'seen' attribute for tests, so that a test is considered 'unseen' not if it hasn't been visited, but if it hasn't been completed
                    //do nothing
                } else if (!$parent_keep) {
                    $tree = array_merge(array_slice($tree, 0, $i), array_slice($tree, $i + 1, sizeof($tree) - 1));
                } else {
                    $tree[$i]['active'] = false;
                }
            } else {
                $keep[$counter] = $tree[$i]['parent_id'];
                $counter++;
            }
        }
    }

    $keep = array();


    if ($ctg) {                                                                                         // ????? ???? ????????? ??????? ?? ???? ?? ctg
        $counter = 0;
        for ($i = sizeof($tree) - 1; $i >= 0; $i--) {
            if ($tree[$i]['ctg_type'] != $ctg) {
                $parent_keep = false;
                if (!isset($keep)) {
                    echo $keep = null;
                }
                for ($k = 0; $k < sizeof($keep) AND !$parent_keep; $k++) {
                    if ($tree[$i]['id'] == $keep[$k]) {
                        $parent_keep    = true;
                        $keep[$counter] = $tree[$i]['parent_id'];
                        $counter++;
                    }
                }
                if (!$parent_keep) {
                    $tree = array_merge(array_slice($tree, 0, $i), array_slice($tree, $i + 1, sizeof($tree) - 1));
                }
            } else {
                $keep[$counter] = $tree[$i]['parent_id'];
                $counter++;
            }
        }        
    }    
//    echo "<br>";print_r($tree);

    return $tree;
}


/**
* Creates linear order of a tree
*
* This function gets the initial tree, created by eF_createTree() and puts 
* units in a linear order, so that they are sorted by children-parent and
* previous-next relations.
*
* @param int $lessons_ID The lesson id
* @param array $tree The tree created by eF_createTree() ?
* @param bool $only_active List only active units
* @param mixed $breaks ?
* @return array An array with keys set to unit ids and values to the order in the tree of this unit
* @see eF_createTree()
* @version 1.0
* @todo needs about 0.5s for a 41-unit lesson, which is a long time for such a widely used function
* @deprecated
*/
function eF_createLinearOrder($lessons_ID, &$tree, $only_active)
{
    if ($only_active) {
        $str = " AND active=1";
    } else {
        $str = '';
    }
    
    $start = eF_getTableData("content", "id", "(lessons_ID=$lessons_ID $str) AND (parent_content_ID=0 OR previous_content_ID=0)");
    $i = 0;

    for ($k = 0; $k < sizeof($start); $k++) {
        $content_ID = $start[$k]['id'];
        if (!isset($order[$content_ID])) {
            $i++;
            while ($content_ID) {
                $order[$content_ID] = $i;
                //$node = eF_getNextContent($content_ID, $period_check = false);
                if ($node) {
                    $content_ID = $node['id'];
                    $i++;
                } else {
                    $breaks[$content_ID] = true;
                    $content_ID          = false;
                }
            }
        }
    }
    
    $minimum = min($order);
    foreach ($order AS $key => $value) {
        $order[$key] = $value - $minimum + 1;
    }
    asort ($order);
    
    return $order;
}


/**
* Correct tree order
*
* This function gets a content tree as an argument, and updates content database so that unit succession 
* corresponds to this tree structure. The content tree has a similar structure to the one returned by
* eF_getContentTree().
*
* @param array $tree The content tree
* @param int $lessons_id the lesson id for which the content will be updated
* @return bool true if everything is ok
* @see eF_getContentTree()
* @version 1.0
*/
function eF_fixOrder($tree, $lessons_ID)
{
    $result = eF_getTableData("content", "COUNT(id) AS num", "previous_content_ID=0 AND lessons_ID=$lessons_ID");
    if ($result[0]['num'] > 1) {
        for ($i = 0; $i < sizeof($tree) - 1; $i++) {
            eF_updateTableData("content", array('previous_content_ID' => $tree[$i]['id']), "id=".$tree[$i + 1]['id']);
        }
    }
    return true;
}


/**
* Returns unit levels
*
* This function accepts a content tree as an argument and returns unit levels in an array.
* Specifically, for a tree structure, similar to the one returned by eF_getContentTree(),
* creates an array entry where the key is the unit id and the value is the corresponding level
* This function is used from within eF_getContentTree()
*
* @param $tree The content tree
* @return array An array where each key is a unit and each value is its level inside the tree
* @see eF_getContentTree()
* @version 1.0
*/
function eF_getLevels($tree)
{
    for ($i = 0; $i < sizeof($tree); $i++) {
        $levels[$tree[$i]['id']] = $tree[$i]['level'];
    }
    return $levels;
}


/**
* Reshuffle tree
* 
* This function accepts a content tree, usually created by eF_createTree() and a content order,
* usually created by eF_createLinearOrder(). It uses the latter to reform the former so that
* units appear with the correct order. Both argumnets are passed by reference.
* This function is used from within eF_getContentTree()
*
* @param array $tree The content tree
* @param array $order The content order
* @see eF_createTree()
* @see eF_createLinearOrder()
* @see eF_getContentTree()
* @version 1.0
* @todo Needs about 0.5s for a 41-unit lesson, a long time for a much used function as this one.
*/
function eF_reshuffleTree(&$tree, &$order) 
{
    for ($j = 0; $j < sizeof($tree); $j++) {
        for ($i = 0; $i < sizeof($tree); $i++) {
            $found = false;
            $level = $tree[$i]['level'];
            for ($k = $i + 1; $k < sizeof($tree) AND !$found; $k++) {
                if ($level == $tree[$k]['level'] AND $order[$tree[$i]['id']] > $order[$tree[$k]['id']] AND $tree[$k]['parent_id'] == $tree[$i]['parent_id']) {
                    $l = 1;
                    while (isset($tree[$k + $l]) && $tree[$k + $l]['level'] > $level) {
                        $l++;
                    }
                    if ($k + $l == sizeof($tree)) {
                        $l = sizeof($tree) - $k;
                    }
                        
                    $tree  = array_merge(array_slice($tree,  0,  $i),  array_slice($tree,  $k,  $l),  array_slice($tree,  $i,  $k-$i),  array_slice($tree,  $k+$l,  sizeof($tree)-$k-$l));
                    $found = true;
                }
            }
        }
    }
}


/**
* Create the content tree
*
* This function is used to create the content tree for a given lesson or unit.
* It recurses through units and builds an array, where each element corresponds to 
* a unit and contains the following information about it:
* - 'id'        : the unit id
* - 'name'      : the name of the unit
* - 'ctg_type'  : the unit type
* - 'parent_id' : the id of the unit's parent
* - 'level'     : The level of the unit
* - 'active'    : whether it is an active unit
* - 'data'      : Whether the unit has data.
* This function is used from within eF_getContentTree()
*
* @param int $lessons_ID the id of the lesson
* @param int $content_ID the content unit id
* @param int $counter used in recursion
* @param int $level the unit level
* @param array $tree the content tree
* @param bool $only_active if set, only active units are included in the tree
* @see eF_getContentTree()
* @version 1.0
*/
function eF_createTree($lessons_ID, $content_ID, &$counter, &$level, &$tree, $only_active)
{
    $parent_id = $content_ID;
    if ($only_active) {
        $str = " AND active=1";
    } else {
        $str = '';
    }
        
    $res = eF_getTableData("content", "id,  name,  ctg_type,  parent_content_ID,  active,  data,  previous_content_ID",  "parent_content_ID=$content_ID AND lessons_ID=$lessons_ID $str", "previous_content_ID");
    
    for ($i = 0; $i < sizeof($res); $i++) {
        $content_ID = $res[$i]['id'];
        $name       = $res[$i]['name'];
        $ctg_type   = $res[$i]['ctg_type'];
        $parent_id  = $res[$i]['parent_content_ID'];
        $active     = $res[$i]['active'];
        $data       = $res[$i]['data'];
        
        if (eF_isLinked($data)) {
            $res1 = eF_getTableData("content", "data", "id=".mb_substr($data, 8));
            $data = $res1[0]['data'];
        }
        
        $tree[$counter]['id']        = $content_ID;
        $tree[$counter]['name']      = $name;
        $tree[$counter]['ctg_type']  = $ctg_type;
        $tree[$counter]['parent_id'] = $parent_id;
        $tree[$counter]['level']     = $level;
        $tree[$counter]['active']    = $active;
        
        if ($data == "") {
            $tree[$counter]['data'] = false;
        } else {
            $tree[$counter]['data'] = true;
        }
        
        $counter++;
        $level++;
        eF_createTree($lessons_ID, $content_ID, $counter, $level, $tree, $only_active);
        $level--;
    }
}

/**
* Check if a unit is linked form another lesson
*
* When we copy units from one lesson to another, then a special strin is appended to the data, '<:link:>' plus the content_ID.
* This function checks the data to see if it is linked to another unit.
*
* @param string $data the unit content
* @return bool True if the unit is linked to another.
* @version 1.0
*/
function eF_isLinked($data)
{
    if(mb_substr($data, 0, 8) == "<:link:>") {
        return true;
    } else {
        return false;
    }
}

/**
* Checks if the designated unit belongs to a period
*
* This function accepts a content id and checks to see if it is part of a period, which holds if it
* is listed in the current_content table.
* If so, ti returns true; otherwise it returns false.
*
* @param int $content_ID The content id
* @return bool Whether this unit belongs to the vurrent content
* @version 1.0
*/
function eF_inPeriod($content_ID) {
    //$result = eF_getTableData("content, current_content, periods", "content.id", "content.previous_content_ID=$content_ID AND current_content.periods_ID=periods.id AND periods.from_timestamp+$offset<=$time AND content.id=current_content.content_ID AND content.active=1");
    $result = eF_getTableData("current_content", "content_ID", "content_ID=$content_ID");
    if (sizeof($result) == 0) {
        return false;
    } else {
        return true;
    }
}

/**
* Get a unit's previous unit 
*
* This function gets a unit id and returns its previous unit. If the parameter 
* $period_check is set, then the previous unit in a current period is returned. 
* If the parameter $active_check is set, then the next active unit is returned.
* The return value is an array that consists of 3 values:
* - 'id'       : The unit id
* - 'name'     : The unit name
* - 'ctg_type' : The unit type
* 
* @param int $content_ID The unit id
* @param bool $period_check If set, the previous unit in a current period is returned
* @param bool $active_check  If set, checks if unit is active
* @param int $lessons_ID The lesson id. If false (default),it uses session lesson
* @param bool $empty_check  If set, it returns only units with data.(not empty units) 
* @return array The previous unit
* @version 1.3
* Changes from version 1.2 to 1.3 (18/11/05):
* - Fixed bug of not reporting the previous unit is some special cases
* Changes from version 1.1 to 1.2 (23/9/05):
* - Added $active_check parameter
* Changes from version 1.0 to 1.1:
* - Fixed bug for period_check malfunctioning
* Changes from version 1.1 to 1.2(15/3/2007):
* - Added parameter empty_check
*/
function eF_getPreviousContent($content_ID, $period_check = false, $active_check = true, $lessons_ID = false, $empty_check = false)
{
    if (!$lessons_ID) {
        $lessons_ID = $_SESSION['s_lessons_ID'];
    }
    if(!$empty_check) { 
        if ($period_check == "true") {
            $time   = time();
            $offset = eF_getOffset();
            $res = eF_getTableData("content,current_content", "DISTINCT content.previous_content_ID,content.active", "content.id=$content_ID and lessons_ID = $lessons_ID");

            while (sizeof($res) > 0 && $res[0]['active'] == 1 && !eF_isCurrentContent($res[0]['previous_content_ID'])) {
                $res = eF_getTableData("content,current_content", "DISTINCT content.previous_content_ID,content.active", "lessons_ID = $lessons_ID and content.id=".$res[0]['previous_content_ID']);
            }
        } else {
            $res = eF_getTableData("content", "DISTINCT previous_content_ID,active", "lessons_ID = $lessons_ID and content.id=$content_ID");
        }
    } else {
        if ($period_check == "true") {
            $time   = time();
            $offset = eF_getOffset();
            $res = eF_getTableData("content,current_content", "DISTINCT content.previous_content_ID,content.active, content.data", "content.id=$content_ID and lessons_ID = $lessons_ID");
             if($res[0]['previous_content_ID'] != 0){ 
                $res2 = eF_getTableData("content","data,ctg_type","id=".$res[0]['previous_content_ID']);
            } 
            while ((sizeof($res) > 0 && $res[0]['active'] == 1) && (!eF_isCurrentContent($res[0]['previous_content_ID']) || ($res2[0]['data']== "" && $res2[0]['ctg_type'] !="tests") )) {
                $res = eF_getTableData("content,current_content", "DISTINCT content.previous_content_ID,content.active, content.data", "lessons_ID = $lessons_ID and content.id=".$res[0]['previous_content_ID']);
                if($res[0]['previous_content_ID'] != 0){
                    $res2 = eF_getTableData("content","data,ctg_type","id=".$res[0]['previous_content_ID']);
                }
            }
        }else {
            $res = eF_getTableData("content", "DISTINCT previous_content_ID,active,data", "lessons_ID = $lessons_ID and content.id=$content_ID");
            if($res[0]['previous_content_ID'] != 0){ 
                $res2 = eF_getTableData("content","data,ctg_type","id=".$res[0]['previous_content_ID']);
            } 
            while ((sizeof($res) > 0 && $res[0]['active'] == 1) && ($res2[0]['data']== "" && $res2[0]['ctg_type'] !="tests")) {
              $res = eF_getTableData("content", "DISTINCT previous_content_ID,active,data", "lessons_ID = $lessons_ID and content.id=".$res[0]['previous_content_ID']);
                if($res[0]['previous_content_ID'] != 0){ 
                    $res2 = eF_getTableData("content","data,ctg_type","id=".$res[0]['previous_content_ID']);
                }    
            }   
        }
    }
    if (sizeof($res) > 0) {
        $res = eF_getTableData("content", "id,name,ctg_type,active", "lessons_ID = $lessons_ID and id=" .$res[0]['previous_content_ID']);
        $node['id']       = $res[0]['id'];
        $node['name']     = $res[0]['name'];
        $node['ctg_type'] = $res[0]['ctg_type'];
        $active           = $res[0]['active'];
        
        if ($node['id'] != 0) {
            if ($active == 0 && $active_check) {
                $node = eF_getPreviousContent($node['id'], $period_check, $active_check);
            }
        } else {
            $res[0]['name']     = '';
            $res[0]['ctg_type'] = '';
        } 

        return $node;
    }
    
    return false;
}

/**
* Get a unit's next unit 
*
* This function gets a unit id and returns its next unit. If the parameter 
* $period_check is set, then the next unit in a current period is returned. 
* If the parameter $active_check is set, then the next active unit is returned.
* The return value is an array that consists of 3 values:
* - 'id'       : The unit id
* - 'name'     : The unit name
* - 'ctg_type' : The unit type
* 
* @param int $content_ID The unit id
* @param bool $period_check If set, the next unit in a current period is returned
* @param bool $active_check If set, then it returns the next active unit
* @param int $lessons_ID The lesson id. If false (default),it uses session lesson
* @param bool $empty_check  If set, it returns only units with data.(not empty units) 
* @return array The next unit
* @version 1.2
* Changes from version 1.1 to 1.2 23/9/05):
* - Added $active_check parameter
* Changes from version 1.0 to 1.1:
* - Fixed bug for period_check malfunctioning
* Changes from version 1.1 to 1.2(15/3/2007):
* - Added parameter empty_check
*/
function eF_getNextContent($content_ID, $period_check = false, $active_check = true, $lessons_ID = false, $empty_check = false)
{
    if (!$lessons_ID) {
        $lessons_ID = $_SESSION['s_lessons_ID'];
    }
    if(!$empty_check) {  
        if ($period_check == "true") {
            $time   = time();
            $offset = eF_getOffset();
            $res = eF_getTableData("content, current_content", "DISTINCT content.id, content.active", "content.previous_content_ID=$content_ID and lessons_ID = $lessons_ID");
            while (sizeof($res) > 0 && $res[0]['active'] == 1 && !eF_isCurrentContent($res[0]['id'])) {
                $res = eF_getTableData("content, current_content", "DISTINCT content.id, content.active", "lessons_ID = $lessons_ID and content.previous_content_ID=".$res[0]['id']);
            }
        } else {
            $res = eF_getTableData("content", "DISTINCT id, active", "previous_content_ID=$content_ID and lessons_ID = $lessons_ID");
        }
    }else {
        if ($period_check == "true") {
            $time   = time();
            $offset = eF_getOffset();
            $res = eF_getTableData("content, current_content", "DISTINCT content.id, content.active, content.data, content.ctg_type", "content.previous_content_ID=$content_ID and lessons_ID = $lessons_ID");
//print_r($res);
            while ( (sizeof($res) > 0 && $res[0]['active'] == 1) && (!eF_isCurrentContent($res[0]['id']) || ($res[0]['data']== "" && $res[0]['ctg_type'] !="tests" ))) {
                $res = eF_getTableData("content, current_content", "DISTINCT content.id, content.active, content.data, content.ctg_type", "lessons_ID = $lessons_ID and content.previous_content_ID=".$res[0]['id']);
            }
        } else {
            $res = eF_getTableData("content", "DISTINCT id, active,data,ctg_type", "previous_content_ID=$content_ID and lessons_ID = $lessons_ID");
            while ( (sizeof($res) > 0 && $res[0]['active'] == 1) && ( $res[0]['data']== "" && $res[0]['ctg_type'] !="tests")) {
                $res = eF_getTableData("content", "DISTINCT id, active, data, ctg_type", "lessons_ID = $lessons_ID and content.previous_content_ID=".$res[0]['id']);
            }
    
        }
    
    }    

        if (sizeof($res) > 0) {
        $node['id'] = $res[0]['id'];
        $active     = $res[0]['active'];

        if ($node['id']) {
            if ($active == 0 && $active_check) {
                $node = eF_getNextContent($node['id'], $period_check, $active_check);
            }
            if (!$node) {
                return false;
            }
            $res = eF_getTableData("content", "name, ctg_type", "lessons_ID = $lessons_ID and id=".$node['id']);
            if (!$res) {
                return false;
            }
            $node['name']     = $res[0]['name'];
            $node['ctg_type'] = $res[0]['ctg_type'];
            return $node;
        }
    }
    return false;
}


/**
* Synchronizes period assignment through units
*
* When the professor changes the parent of a unit, then this function is called so that
* the unit is included to the same period as the parent.
*
* @param int $id The unit id
* @param int parent_ID The new parent id
* @return bool True if everything is ok
*/
function eF_fixPeriod($id, $parent_ID)
{
    if ($parent_ID != 0) {
        $res_p      = eF_getTableData("current_content", "periods_ID", "content_ID=$parent_ID");
        $res_c      = eF_getTableData("current_content", "periods_ID", "content_ID=$id");
        $old_parent = eF_getTableData("content", "parent_content_ID", "id=$id");
        
        if ($old_parent[0]['parent_content_ID'] != $parent_ID) {
            for ($i = 0; $i < sizeof($res_c); $i++) {
                $already[] = $res_c[$i]['periods_ID'];
            }
            for ($i = 0; $i < sizeof($res_p); $i++) {
                if (!in_array($res_p[$i]['periods_ID'],$already)) {
                    $fields['periods_ID'] = $res_p[$i]['periods_ID'];
                    $fields['content_ID'] = $id;
                    eF_insertTableData("current_content", $fields);
                }
            }
        }
    }
        
    return true;
}


/**
* Create content structure
*
* This function creates the content structure in its primitive form. This form consists
* of a multidimensional array, where each value represents a unit. If a unit has children,
* these children exist as sub-arrays of the unit. This way, the array created closely matches 
* the tree structure
* 
* @param int $content_ID The content id to build the tree for
* @param int $lessons_ID The lesson to build the tree for, defaults to session value
* @param bool $only_active Whether it should return only active values, defaults to true
*
*/
function eF_createContentStructure($content_ID = 0, $lessons_ID = '', $only_active = true) {

    if (!$lessons_ID || !eF_checkParameter($lessons_ID, 'id')) {
        $lessons_ID = $_SESSION['s_lessons_ID'];
    }

    if ($GLOBALS['configuration']['caching'] && $only_active) {                                                             //Cache this copy of the tree, but only if active units are requested (which is usualy the case)
        $result = eF_getTableData("cache", "*", "name='".$_SESSION['s_lessons_ID']."_contentTree'");                                 //Get cached copy
        if (sizeof($result) > 0 && $result[0]['expired'] == 0 && time() - $result[0]['timestamp'] < $GLOBALS['configuration']['cache_timeout']) {               //Check if cached copy is valid, and if so return it
            $tree = unserialize($result[0]['value']);
            return $tree;
        } else {
            eF_deleteTableData("cache", "name='".$_SESSION['s_lessons_ID']."_contentTree'");
        }
    }

    $units = eF_getTableData("content", "id, name, parent_content_ID, ctg_type, active, previous_content_ID, data", "lessons_ID=".$lessons_ID, "parent_content_ID, previous_content_ID");        //Get all the lesson units

    for ($i = 0; $i < sizeof($units); $i++) {                                                       //Initialize the nodes array
        $units[$i]['data'] == '' ? $units[$i]['data'] = false : $units[$i]['data'] = true;          //Set data bit, if the node has data.        
        
        $nodes[$units[$i]['id']]             = $units[$i];
        $nodes[$units[$i]['id']]['children'] = array();
    }

    $q = 0;
    while (sizeof($units) > 0 && $q++ < 1000 && !isset($wanted)) {                                   //$q is put here to prevent an infinite loop
        $leaves = eF_findLeaves($units);                                                            //Get the leaf nodes of the tree
        foreach ($leaves as $leaf) {
            $nodes[$leaf['parent_content_ID']]['children'][$nodes[$leaf['id']]['id']] = $nodes[$leaf['id']];

            if ($leaf['id'] == $content_ID) {                                                       //If the user asked for the tree below a specified content id, then put it in $wanted
                $wanted = $nodes[$leaf['id']];
            }
            unset($nodes[$leaf['id']]);
        }
    }
    isset($wanted) ? $nodes = array(0 => $wanted) : $nodes = $nodes[0]['children'];                 //$nodes now has either the whole tree (which was put in $nodes[0]['children'] from the loop above), or the branch under the specified content_ID (which was put in $wanted)

    $tree = array();
    $tree = eF_makeCompatibleTree($tree, $nodes, 0);                                                //Convert the tree from its multidimensional form to its flat form, which is used throughout eFront

    if (sizeof($tree) > 0) {
        $tree = eF_putInCorrectOrder($tree, $only_active);                                          //Reorder the units to match the content series
    }

    if ($GLOBALS['configuration']['caching'] && $only_active) {                                                           //If we reached here, it means that the tree was not cached, so cache it
        $cache_fields = array('name'      => $_SESSION['s_lessons_ID'].'_contentTree',
                              'value'     => serialize($tree),
                              'timestamp' => time(),
                              'expired'   => 0);
        eF_insertTableData("cache", $cache_fields);
    }
    
    return $tree;    
}

/**
* Reorder tree according to content sequence
*
* This function is used to reorder the tree array, so that the array values succession 
* match the content sucession. The reordering is done based on array indexes, as returned
* by eF_makeCompatibleTree
*
* @param array $tree The content tree, as returned by eF_makeCompatibleTree
* @param bool $only_active Whether the tree should contain only active units
* @return array The correct content tree
* @version 1.1
* Changes from version 1.0 to 1.1 (2007/06/18 - venakis):
* - Added the check if the first unit is active
*/
function eF_putInCorrectOrder($tree, $only_active) {
    $current_unit    = $tree[0];
    if ($current_unit['active'] || !$only_active) {
        $correct_tree[0] = $tree[0];
    }
    unset($tree[0]);
    foreach ($tree as $node) {
        $current_unit = $tree[$current_unit['id']];
        if ($current_unit['active'] == 1 || !$only_active) {
            $correct_tree[] = $current_unit;
        }
    }
    return $correct_tree;
}

/**
* Get the leaf nodes of the tree
*
* Ths function is used to traverse through the primitive content tree array,
* to find all units that do not have children (leaf units). It also removes
* these leaves from the array.
*
* @param array $units The array of units
* @return array The tree leaves
*/
function eF_findLeaves(&$units) {
    
    for ($i = 0; $i < sizeof($units); $i++) {
        $ok = true;
        for ($j = 0; $j < sizeof($units); $j++) {
            if ($units[$j]['parent_content_ID'] == $units[$i]['id']) {
                $ok = false;
            }
        }
        if ($ok) {
            $leaves[]      = $units[$i];
            $unset_array[] = $i;
        }
    }
    
    for ($i = 0; $i < sizeof($unset_array); $i++) {
        unset($units[$unset_array[$i]]);
    }
    
    $units = array_values($units);
    return $leaves;
}

/**
* Convert primitive content tree to eFront-compatible tree
*
* This recursive function will convert the given (multidimensional) units array to the corresponding (flat) 
* efront content tree. It starts with an empty array as input.
*
* @param array $tree The efront-compatible content tree array (empty on initial call)
* @param array $nodes The primitive content array
* @param int $level The current tree level (depth) to consider
* @return array The new, efront-compatible content tree
*/
function eF_makeCompatibleTree(&$tree, $nodes, $level = 0) {
    foreach($nodes as $node) {
        $new_node = array('id'        => $node['id'], 
                          'name'      => $node['name'],
                          'ctg_type'  => $node['ctg_type'],
                          'parent_id' => $node['parent_content_ID'],
                          'level'     => $level,
                          'active'    => $node['active'],
                          'data'      => $node['data']);

        if (!isset($tree[$node['previous_content_ID']])) {
            $tree[$node['previous_content_ID']] = $new_node;
        } 

        eF_makeCompatibleTree($tree, $node['children'], $level + 1);
    }
    
    return $tree;
}


/**
* Gets the student's recently viewed content
*
* This function builds an array with units that a student has recently visited, visible at the student's main page.
* 
* @param string $login The user login
* @param int $lessons_ID The lesson id
* @return array The seen content list
* @version 2.3
* Changes from 2.2 to 2.3 (2007/05/21):
* - Rewritten test check so that it does not perform queries inside the loop
* Changes from 2.1 to 2.2 (2007/01/29):
* - Changed so that for a test, it is considered done only if it is completed, not just visited.
* Changes from 2.0 to 2.1 (17/11/2005):
* - Added the eF_isDone Content function to provide the ability to exclude units not part of the current period
* Changes from 1.0 to 2.0 (16/11/2005):
* - Changed the query and return model. Now, it correctly returns the unit order
*/
function eF_getLastSeenContent($login, $lessons_ID)
{
    $result     = eF_getTableData("logs l, content c", "max(l.timestamp) as time_stamp, l.users_LOGIN, c.id, c.lessons_ID, c.name, c.ctg_type", "l.comments = c.id and l.users_LOGIN='$login' and c.lessons_ID=$lessons_ID group by l.users_LOGIN, c.id, c.lessons_ID", "time_stamp desc");
    $done_tests = eF_getTableDataFlat("done_tests as dt,tests as t, content as c", "c.id", "dt.users_LOGIN='".$login."' AND t.content_ID = c.id AND c.lessons_ID = ".$lessons_ID." AND t.id=dt.tests_ID"); 

    foreach ($result as $key => $value) {
        if (!eF_isDoneContent($value['id'])) {
            unset($result[$key]);
        }
        if ($value['ctg_type'] == "tests" && !in_array($value['id'], $done_tests['id'])) {
            unset($result[$key]);   
        }
//        if ($value['ctg_type'] == "tests") {
//            $res = eF_getTableData("done_tests as dt,tests as t", "count(*)", "dt.users_LOGIN='".$login."' AND t.content_ID=".$value['id']." AND t.id=dt.tests_ID");
//            if($res[0]['count(*)'] == 0 ){
//            echo $value['id']."!<br>";
//                unset($result[$key]);   
//            }
//        }
    }

    $result = array_values($result);
    
    return $result;
}

/**
* Get the student seen content
*
* This function gets the units that the student has set as seen
* A unit is considered seen when the student clicks the corresponding icon
* below the unit content. The same way he may deignate a unit as unseen (except
* for test units).
*
* @param string $login The student login
* @param int $lessons_ID The current lesson id
* @return array An array holding the seen unit ids (including tests)
* @version 1.0
*/
function eF_getSeenContent($login, $lessons_ID)
{
    $done_content = eF_getTableDataFlat("users_to_lessons", "done_content", "users_LOGIN='".$login."' and lessons_ID=".$lessons_ID);
    $done_tests   = eF_getTableDataFlat("done_tests as dt,tests as t, content as c", "c.id", "dt.users_LOGIN='".$login."' AND t.content_ID = c.id AND c.lessons_ID = ".$lessons_ID." AND t.id=dt.tests_ID"); 
    (sizeof($done_content['done_content'][0]) > 0 && $done_content['done_content'][0]) ? $done_content = unserialize($done_content['done_content'][0])       : $done_content = array();
    sizeof($done_tests['id'][0])             > 0 ? $done_tests   = array_combine($done_tests['id'], $done_tests['id']) : $done_tests   = array();

    return ($done_content + $done_tests);
}

/**
* Get new content
* 
* @param string $student The student to calculate new stuff for
* @param int The lesson id
* @param bool current_period
* @version 1.0 (28/2/2005 - makriria)
*/
function eF_getNewContent($student, $lesson_ID, $type = 'content', $current_period = false)
{
    $login_limit     = 5;                                                                   //The limit of login times for the user, beyond which the content is not considered new anymore
    $timestamp_limit = 7 * 24 * 60 * 60;                                                    //The limit of time that passed since the last user login, set to a week later.
    $new_content     = array();                                                             
    
    if ($type == 'content') {

        if (!$current_period) {
            $res_content = eF_getTableData("content c, lessons l", "c.id, c.name, c.ctg_type, c.timestamp", "c.lessons_ID=l.id AND c.timestamp > ".(time() - $timestamp_limit)." AND l.id=".$lesson_ID, "c.timestamp asc");       //???? ??? ?? ??????????? ??? ?????????
        } else {
            $res_content = eF_getTableData("content c, lessons l, periods p, current_content cc", "c.id, c.name, c.ctg_type, c.timestamp", "c.lessons_ID=l.id AND c.timestamp > ".(time() - $timestamp_limit)." AND l.id=".$lesson_ID ." AND p.id=cc.periods_ID AND cc.content_ID=c.id AND p.lessons_ID=l.id AND p.lessons_ID=c.lessons_ID AND p.from_timestamp <" .time()." AND p.to_timestamp >" .time() , "c.timestamp asc");
            //$res_content = eF_getTableData("content,lessons", "content.id,timestamp", "content.lessons_ID=lessons.id AND lessons.id=". $lesson_ID,"timestamp asc");       
        }

        for ($i = 0; $i < sizeof($res_content); $i++) {                                                 
                $new_content[] = array('id'        => $res_content[$i]['id'],
                                       'timestamp' => $res_content[$i]['timestamp'],
                                       'name'      => $res_content[$i]['name'],
                                       'ctg_type'  => $res_content[$i]['ctg_type']);
                                                                                                       
        }
    }
    elseif ($type == 'news') {                                                                        
        $new_content = eF_getTableData("news","*", "(lessons_ID = 0 OR lessons_ID=".$lesson_ID .") AND timestamp > ".(time() - $timestamp_limit),"lessons_ID, timestamp asc");       

    }
    return $new_content;
}

/**
* Repair content tree
* 
* This function is used to repair a damaged content tree
* It can repair the parent information (if, for example, a unit's parent unit does not exist), or the 
* unit succession information (if, for some case, the next/previous unit information is invalid)
* In the latter case, the content tree will be altered, no matter if a damage is present or not, so
* use with extreme caution!
*
* @param int $lessons_ID the lesson to repair its content tree
* @param bool $repair_parents Whether to repair parent information
* @param bool $repair_order Whether to repair order information
* @return bool on success
*/
function eF_repairTree($lessons_ID, $repair_parents = true, $repair_order = false) {
    if ($repair_parents) {
        $units = eF_getTableData("content", "name, id, parent_content_ID, previous_content_ID, timestamp", "lessons_ID=$lessons_ID", "parent_content_ID,previous_content_ID,timestamp");

        foreach ($units as $unit) {
            $ids[]                = $unit['id'];                                //Get an array with the unit ids
            $parents[$unit['id']] = $unit['parent_content_ID'];                 //Get an array with the unit parent ids
        }

        $wrong_parents = array_diff($parents, $ids);                            //See which parents do not exist, by comparing the parents array with the ids array (this way, all parents that do not have a corresponding id will be returned)
        foreach ($wrong_parents as $id => $parent_id) {
            if ($parent_id != 0) {                                              //We ommit the case where the parent is 0, since we do not have such a unit id (and the case where the parent is 0 does not cause any problem)
                eF_updateTableData("content", array('parent_content_ID' => 0), "id=$id");   //Set the units with non-existent parents to have 0 as parent unit
            }
        }
    }
    
    if ($repair_order) {
        $units = eF_getTableData("content", "name, id, parent_content_ID, previous_content_ID, timestamp", "lessons_ID=$lessons_ID", "parent_content_ID,previous_content_ID,timestamp");

        foreach ($units as $unit) {
            $nodes[$unit['id']] = array('id' => $unit['id']);
        }

        $loop = 0;
        while (sizeof($units) > 0 && $loop++ < 1000) {                                   //$loop is put here to prevent an infinite loop
            $leaves = eF_findLeaves($units);                                                            //Get the leaf nodes of the tree
            foreach ($leaves as $leaf) {
                $nodes[$leaf['parent_content_ID']][$leaf['id']] = $nodes[$leaf['id']];

                unset($nodes[$leaf['id']]);
            }
        }

        $row = array(0);
        array_walk_recursive($nodes, create_function('$a, $b, &$row', 'if ($b == "id") $row[] = $a;'), $row);

        for ($i = 1; $i < sizeof($row); $i++) {
            eF_updateTableData("content", array('previous_content_ID' => $row[$i - 1]), "id=".$row[$i]);
        }
    }
    
    if (G_ENABLE_CACHE && sizeof(eF_getTableData("cache", "*", "name='".$_SESSION['s_lessons_ID']."_contentTree'"))) {
        eF_updateTableData("cache", array('expired' => 1), "name='".$_SESSION['s_lessons_ID']."_contentTree'");
    } 
    
    return true;
    
}

Abstract class Plugin
{
	function __construct() {
		$this -> setLanguage();
	}
	
	protected function setLanguage() {
		//Code for language inclusion here...
	}
	protected function parseXML() {
		//Code for XML parsing here...
	}
	
	abstract public function display();
	abstract public function install();
	abstract public function uninstall();
	abstract public function upgrade();
}
?>