/*
Copyright (c) 2001-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/


/**	@file	yalc.h	Include file for the yalc module.
*/

#ifndef YALC_INC
#define YALC_INC 1


#include <dk.h>
#include <dktypes.h>
#include <dkstr.h>

#include <yalcmsg.h>

/**	Request type.
*/
typedef struct {
  int cmd;		/**< Command, what to do. */
  char *printer_name;	/**< Printer name. */
  char *server_name;	/**< Server name. */
  char *user_name;	/**< User name. */
  char *snmp_host;	/**< Host to contact via SNMP. */
  char *snmp_comm;	/**< SNMP community name. */
  char *snmp_vers;	/**< SNMP protocol version. */
  char *snmp_type;	/**< SNMP type. */
  char *private_snmp;	/**< SNMP options. */
  dk_app_t *app;	/**< Application. */
  int argc;		/**< Command line arguments (number). */
  char **argv;		/**< Command line arguments (array). */
  union {
    struct {
      int max_verbosity;	/**< Flag: As verbose as possible. */
      int verbosity;		/**< Flag: Verbose. */
      int short_status;		/**< Flag: Short status only. */
      int periodically;		/**< Flag: Run periodically. */
      int period_time;		/**< Number of seconds when running periodicaly. */
      int remove_filter_status;	/**< Flag: Remove filter status. */
      int remove_done_jobs;	/**< Flag: Remove done jobs from list. */
    } lpq;			/**< LPQ request. */
    struct {
      int all;			/**< Flag: All accessable jobs. */
    } lprm;			/**< LPRM request. */
    struct {
      int all;			/**< Flag: all queues. */
    } lpc;			/**< LPC request. */
    struct {
      int type;			/**< Type:  YALC_TYPE_xx. */
      char *class_name;		/**< -C option. */
      int suppress_header;	/**< Flag:  print without header file. */
      int indent_cols;		/**< -i option. */
      char *job_name;		/**< -J option. */
      int number_of_copies;	/* number of copies to print */
      char *mail_recipient;	/**< E-mail of user, for notification. */
      int keep_spool_queue;	/**< Flag: Keep name of spool queue in control file. */
      char *remote_account;	/**< Account on server. */
      char *job_title;		/**< Job title, normally file name. */
      int with;			/**< Line width. */
      char *other_options;	/**< -Z/-o options. */
    } lpr;			/**< LPR/LPD request. */
    struct {
      int force_output;		/**< Flag: Force output. */
      int output_found;		/**< Flag: Output found. */
      int run_verbose;		/**< Flag: Run verbose. */
      int keep_logfile;		/**< Flag: Keep log file. */
    } snmp;			/**< SNMP request. */
  } args;			/**< Request type specific data. */
  void *cfg_file;		/**< Configuration file. */
#if DK_HAVE_WINREG_H
  struct {
    int what;			/**< Which keys open (1=hklm, 2=hkcu, 3=both)? */
    HKEY hklm;			/**< Subkey in HKLM. */
    HKEY hkcu;			/**< Subkey in HKCU. */
  } reg;			/**< Windows registry access. */
#endif
  int read_cfg_res;		/**< Result from reading configuration file(s). */
  char *queue;			/**< Queue name. */
  char *host;			/**< Host name. */
  char *dev;			/**< Device. */
  char *user;			/**< user name. */
  char *printer_from_env;	/**< Printer name from environment. */
  int  type;			/**< Print job type. */
  int  have_timeout;		/**< Flag: Have timout. */
  double to;			/**< Timeout in seconds. */
  unsigned short portno;	/**< Port number. */
  unsigned short o_min;		/**< Minimum local port number. */
  unsigned short o_max;		/**< Maximum local port number. */
  int xuser_allowed;		/**< Flag: -U allowed. */
  char *control_file_name;	/**< Name of control file. */
  unsigned short job_number;	/**< Job number. */
  int  data_file_first;		/**< Flag: Data file first. */
} yalc_rq_t ;

/**	No operation */
#define YALC_OP_NOP	0

/**	Lpr request */
#define YALC_OP_LPR	1

/**	Lprm request */
#define YALC_OP_LPRM	2

/**	Lpq request */
#define YALC_OP_LPQ	3

/**	Lpc request */
#define YALC_OP_LPC	4

/**	Status request via SNMP */
#define YALC_OP_SNMP_STATUS	5

/**	Short simple SNMP request */
#define YALC_OP_SNMP_SHORT	6

/*
  Do not change order without
  appropriate changes to
  pstype_keywords in yalc.c
*/

/**	Transmission method: Not (yet) defined.
*/
#define YALC_METHOD_UNKNOWN	0

/**	Transmission method: LPRng protocol.
*/
#define YALC_METHOD_LPRNG	1

/**	Transmission method: BSD LPD protocol.
*/
#define YALC_METHOD_BSD		2

/**	Transmission method: Raw TCP/IP.
*/
#define YALC_METHOD_RAWTCPIP	3

/**	Transmission method: JetDirect.
*/
#define YALC_METHOD_JETDIRECT	4



/**	Print job type: Not (yet) known.
*/
#define YALC_TYPE_UNKNOWN	0

/**	Print job type: Binary (literal).
*/
#define YALC_TYPE_BINARY	1

/**	Print job type: Cifplot.
*/
#define YALC_TYPE_CIFPLOT	2

/**	Print job type: DVI.
*/
#define YALC_TYPE_DVI		3

/**	Print job type: Plot.
*/
#define YALC_TYPE_PLOT		4

/**	Print job type: troff -n.
*/
#define YALC_TYPE_TROFF_N	5

/**	Print job type: Filter.
*/
#define YALC_TYPE_FILTER	6

/**	Print job type: troff -t.
*/
#define YALC_TYPE_TROFF_T	7

/**	Print job type: PR.
*/
#define YALC_TYPE_PR		8

/**	Print job type: Raster.
*/
#define YALC_TYPE_RASTER	9

#if defined(EXTERN)
#undef EXTERN
#endif
#ifndef DK_YALC_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	Create request.
 * 	@return	Pointer to new request on success, NULL on error.
 */
EXTERN yalc_rq_t *yalc_new DK_PR((void));

/**	Delete request.
 * 	@param	rq	YALC request.
 */
EXTERN void yalc_delete DK_PR((yalc_rq_t *rq));

/**	Initialize request data.
 * 	@param	rq	YALC request.
 */
EXTERN void yalc_init DK_PR((yalc_rq_t *rq));

/**	Clean up request data.
 * 	@param	rq	YALC request.
 */
EXTERN void yalc_cleanup DK_PR((yalc_rq_t *rq));

/**	Seet request type to LPQ.
 * 	@param	rq	YALC request.
 * 	@param	max_verbosity	Maximum verbosity.
 * 	@param	verbosity_level	Level of verbosity.
 * 	@param	short_status_line	Flag: Print short status line.
 * 	@param	periodically:		Flag: Run periodically.
 * 	@param	sleep_time		Time interval.
 * 	@param	rdj			Flag: Remove done jobs.
 */
EXTERN void yalc_set_lpq DK_PR((yalc_rq_t *rq, int max_verbosity, int verbosity_level, int short_status_line, int periodically, int sleep_time, int rdj));

/**	Set request type to LPRM.
 * 	@param	rq	YALC request.
 * 	@param	all	Flag: Remove all accessable jobs.
 */
EXTERN void yalc_set_lprm DK_PR((yalc_rq_t *rq, int all));

/**	Set request type to LPC
 * 	@param	rq	YALC request.
 * 	@param	all	Flag: Deal with all accessable jobs.
 */
EXTERN void yalc_set_lpc DK_PR((yalc_rq_t *rq, int all));

/**	Set printer for request.
 * 	@param	rq	YALC request.
 * 	@param	name	Printer name.
 */
EXTERN void yalc_set_printer DK_PR((yalc_rq_t *rq, char *name));

/**	Set server for request.
 * 	@param	rq	YALC request.
 * 	@param	name	Host name.
 */
EXTERN void yalc_set_server DK_PR((yalc_rq_t *rq, char *name));

/**	Set user name for request.
 * 	@param	rq	YALC request.
 * 	@param	name	User name.
 */
EXTERN void yalc_set_user DK_PR((yalc_rq_t *rq, char *name));

/**	Set application for request.
 * 	@param	rq	YALC request.
 * 	@param	app	Application.
 */
EXTERN void yalc_set_app DK_PR((yalc_rq_t *rq, dk_app_t *app));

/**	Set command line arguments for request.
 * 	@param	rq	YALC request.
 * 	@param	sz	Number of command line arguments.
 * 	@param	args	Command line arguments array.
 */
EXTERN void yalc_set_args DK_PR((yalc_rq_t *rq, int sz, char **args));

/**	Run request.
 * 	@param	rq	YALC request.
 * 	@return	Success flag.
 */
EXTERN int  yalc_run DK_PR((yalc_rq_t *rq));

/**	Get text to print version.
 * 	@return	Pointer to string arrays containing
 * 	version information.
 */
EXTERN char **yalc_get_version DK_PR((void));

/**	Set file type for LPR request.
 * 	@param	rq	YALC request.
 * 	@param	t	file type.
 */
EXTERN void yalc_set_lpr_filetype DK_PR((yalc_rq_t *rq, int t));

/**	Set flag to suppress header page for LPR request.
 * 	@param	rq	YALC request.
 * 	@param	t	Flag value.
*/
EXTERN void yalc_set_lpr_suppress_header DK_PR((yalc_rq_t *rq, int t));

/**	Set number of indent columns for LPR request.
 * 	@param	rq	YALC request.
 * 	@param	t	Indent columns.
 */
EXTERN void yalc_set_lpr_indent_cols DK_PR((yalc_rq_t *rq, int t));

/**	Set number of copies for LPR request.
 * 	@param	rq	YALC request.
 * 	@param	t	Number of copies.
 */
EXTERN void yalc_set_lpr_number_of_copies DK_PR((yalc_rq_t *rq, int t));

/**	Set LPR keep spool queue.
 * 	@param	rq	YALC request.
 * 	@param	t	Flag: Keep spool queue.
 */
EXTERN void yalc_set_lpr_keep_spool_queue DK_PR((yalc_rq_t *rq, int t));

/**	Set LPR line width.
 * 	@param	rq	YALC request.
 * 	@param	t	Text width.
 */
EXTERN void yalc_set_lpr_width DK_PR((yalc_rq_t *rq, int t));

/**	Set LPR class name.
 * 	@param	rq	YALC request.
 * 	@param	cln	Class name.
 */
EXTERN void yalc_set_lpr_class_name DK_PR((yalc_rq_t *rq, char *cln));

/**	Set LPR job name.
 * 	@param	rq	YALC request.
 *	@param	cln	Job name.
 */
EXTERN void yalc_set_lpr_job_name DK_PR((yalc_rq_t *rq, char *cln));

/**	Set e-mail address for user notification.
 * 	@param	rq	YALC request.
 * 	@param	cln	E-Mail address.
 */
EXTERN void yalc_set_lpr_mail_addr DK_PR((yalc_rq_t *rq, char *cln));

/**	Set LPR remote account.
 * 	@param	rq	YALC request.
 * 	@param	cln	Remote account name.
 */
EXTERN void yalc_set_lpr_remote_account DK_PR((yalc_rq_t *rq, char *cln));

/**	Set LPR job title.
 * 	@param	rq	YALC request.
 * 	@param	cln	Job title.
 */
EXTERN void yalc_set_lpr_job_title DK_PR((yalc_rq_t *rq, char *cln));

/**	Set other LPR options.
 * 	@param	rq	YALC request.
 * 	@param	cln	Other options.
 */
EXTERN void yalc_set_lpr_other_options DK_PR((yalc_rq_t *rq, char *cln));

/**	Set request type LPR.
 * 	@param	rq	YALC request.
 */
EXTERN void yalc_set_lpr DK_PR((yalc_rq_t *rq));

/**	Set SNMP status only.
 * 	@param	rq	YALC request.
 */
EXTERN void yalc_set_snmp_status DK_PR((yalc_rq_t *rq));

/**	Set short SNMP response.
 * 	@param	rq	YALC request.
 */
EXTERN void yalc_set_snmp_short DK_PR((yalc_rq_t *rq));

/**	Set verbose SNMP operations.
 * 	@param	rq	YALC request type.
 */
EXTERN void yalc_set_snmp_verbose DK_PR((yalc_rq_t *rq));

/**	Set SNMP host.
 * 	@param	rq	YALC request.
 * 	@param	hn	Host name.
 */
EXTERN void yalc_set_snmp_host DK_PR((yalc_rq_t *rq, char *hn));

/**	Set SNMP community.
 * 	@param	rq	YALC request.
 * 	@param	co	Community name.
 */
EXTERN void yalc_set_snmp_comm DK_PR((yalc_rq_t *rq, char *co));

/**	Set SNMP version.
 * 	@param	rq	YALC request.
 * 	@param	co	SNMP version.
 */
EXTERN void yalc_set_snmp_vers DK_PR((yalc_rq_t *rq, char *co));

/**	Set SNMP type.
 * 	@param	rq	YALC request.
 * 	@param	co	SNMP type.
 */
EXTERN void yalc_set_snmp_type DK_PR((yalc_rq_t *rq, char *co));

/**	Set force flag for SNMP operations.
 * 	@param	rq	YALC request.
 * 	@param	val	New force flag value.
 */
EXTERN void yalc_set_snmp_force DK_PR((yalc_rq_t *rq,int val));

#if defined(__cplusplus)
}
#endif

#endif

